package Jacode4e;
$VERSION = '2.13.6.19';
# 如果您可以阅读此字符，则可以通过选择所有内容并将其保存为文件名“Jacode4e.pm”来将其用作模块。
# 如果您可以閱讀此字符，則可以通過選擇所有內容並將其保存為文件名“Jacode4e.pm”來將其用作模塊。
# この文字が読める場合は、内容を全て選択してファイル名を "Jacode4e.pm" にして保存すればそのままモジュールとして利用することができます。
# 이 문자를 읽을 수 있는 경우는, 내용을 모두 선택해 파일명을 "Jacode4e.pm" 로 해 보존하면 그대로 모듈로서 이용할 수가 있습니다.
# But that is very far into the future isn't it?
####################################################################
#
# Jacode4e - jacode.pl-like program for enterprise
#
# Copyright (c) 2018, 2019, 2021, 2022 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################
#
# ** ATTENTION **
# ---------------
#
# This software IS NOT "jacode.pl"!
# If you want "jacode.pl", search it on CPAN again.
#
# And if you need round-trip conversion, you had better use
# Jacode4e::RoundTrip module.
#
# SYNOPSIS
# --------
#
#   use FindBin;
#   use lib "$FindBin::Bin/lib";
#   use Jacode4e;
#
#   $return =
#   Jacode4e::convert(\$line, $OUTPUT_encoding, $INPUT_encoding [, { %option }]);
#
#     $return
#       Number of characters in $line
#
#     $line
#       String variable to convert
#       After conversion, this variable is overwritten
#
#     $OUTPUT_encoding, and $INPUT_encoding
#       To convert, you must specify both $OUTPUT_encoding and $INPUT_encoding.
#       The encodings you can specify are as follows:
#
#       mnemonic      means
#       -----------------------------------------------------------------------
#       cp932x        CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
#       cp932         Microsoft CP932, IANA Windows-31J
#       cp932ibm      IBM CP932
#       cp932nec      NEC CP932
#       sjis2004      JISC Shift_JIS-2004
#       cp00930       IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
#       keis78        HITACHI KEIS78
#       keis83        HITACHI KEIS83
#       keis90        HITACHI KEIS90
#       jef           FUJITSU JEF (12 point size for printing with option OUTPUT_SHIFTING)
#       jef9p         FUJITSU JEF ( 9 point size for printing with option OUTPUT_SHIFTING)
#       jipsj         NEC JIPS(J)
#       jipse         NEC JIPS(E)
#       letsj         UNISYS LetsJ
#       utf8          UTF-8.0 (aka UTF-8)
#       utf8.1        UTF-8.1
#       utf8jp        UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
#       -----------------------------------------------------------------------
#
#     %option
#       The options you can specify are as follows:
#
#       key mnemonic      value means
#       -----------------------------------------------------------------------
#       INPUT_LAYOUT      input record layout by 'S' and 'D' sequence
#                         'S' means one char as SBCS, 'D' means one char as DBCS
#                         default is 'S'
#                         each letter may optionally be followed by a number
#                         indicating the repeat count, see samples
#       OUTPUT_SHIFTING   true means use output shift code, false means not use
#                         default is false
#       SPACE             output space code in DBCS/MBCS
#                         default is U+3000 in Unicode
#       GETA              output geta code in DBCS/MBCS
#                         default is U+3013 in Unicode
#       OVERRIDE_MAPPING  hash reference of FROM => TO override mapping
#                         { "\x12\x34"=>"\x56\x78", "\x9A\xBC"=>"\xDE\xFE", }
#                         (CAUTION! override also SPACE option)
#       -----------------------------------------------------------------------
#
# SAMPLES
# -------
#
#   use FindBin;
#   use lib "$FindBin::Bin/lib";
#   use Jacode4e;
#   Jacode4e::VERSION('2.13.6.19');
#
#   while (<>) {
#       $return =
#       Jacode4e::convert(\$_, 'cp932x', 'cp00930', {
#           'INPUT_LAYOUT'     => 'SSSDDDSSDDSDSD',
#           'OUTPUT_SHIFTING'  => 0,
#           'SPACE'            => "\x81\xA2",
#           'GETA'             => "\x81\xA1",
#           'OVERRIDE_MAPPING' => { "\x44\x5A" => "\x81\x7C", },
#       });
#       print $_;
#   }
#
#   while (<>) {
#       $return =
#       Jacode4e::convert(\$_, 'cp932x', 'cp00930', {
#           'INPUT_LAYOUT'     => 'S3D3S2D2SDSD',
#           'OUTPUT_SHIFTING'  => 0,
#           'SPACE'            => "\x81\xA2",
#           'GETA'             => "\x81\xA1",
#           'OVERRIDE_MAPPING' => { "\x44\x5A" => "\x81\x7C", },
#       });
#       print $_;
#   }
#
#   while (<>) {
#       Jacode4e::convert(\$_, 'cp932ibm', 'utf8.1', {});
#       print $_;
#   }
#
#   while (<>) {
#       $return =
#       Jacode4e::convert(\$_, 'utf8.1', 'utf8', {
#                                 # FULLWIDTH TILDE => WAVE DASH
#           'OVERRIDE_MAPPING' => { "\xEF\xBD\x9E"  => "\xE3\x80\x9C", },
#       });
#       print $_;
#   }
#
######################################################################
$VERSION = $VERSION;

use 5.00503;
use strict;
BEGIN { $INC{'warnings.pm'} = '' if $] < 5.006 }; use warnings; $^W=1;

my %tr = ();
my @encodings = qw(
    cp932x
    cp932
    cp932ibm
    cp932nec
    sjis2004
    cp00930
    keis78
    keis83
    keis90
    jef
    jipsj
    jipse
    letsj
    unicode
    utf8
    utf8.1
    utf8jp
);
my @io_encodings = grep( ! /^(?:unicode)$/, @encodings);

if ($0 eq __FILE__) {
    if (not @ARGV) {
        die <<END;

usage:
    perl $0 --dumptable

END
    }
}

while (<DATA>) {
    next if /^#/;
    chomp;

    my %hex = ();
    @hex{@encodings} = split(/ +/,$_);

    my %bin = ();
    for my $encoding (@io_encodings) {
        $bin{$encoding} = (($hex{$encoding} =~ /^[0123456789ABCDEF]+$/) ? pack('H*', $hex{$encoding}) : '');
    }

    for my $encoding (@io_encodings) {
        if (($bin{$encoding} ne '') and ($bin{'utf8jp'} ne '')) {
            if ($encoding eq 'utf8jp') {
            }
            elsif (defined($tr{'utf8jp'}{$encoding}{$bin{$encoding}}) and ($tr{'utf8jp'}{$encoding}{$bin{$encoding}} ne '')) {
                die qq{@{[__FILE__]} duplicate definitions \$tr{'utf8jp'}{'$encoding'}{'$hex{$encoding}'} = "$hex{utf8}($bin{utf8})" and "} . uc unpack('H*',$tr{'utf8jp'}{$encoding}{$bin{$encoding}}) . qq{"($tr{'utf8jp'}{$encoding}{$bin{$encoding}})\n};
            }
            elsif (defined($tr{$encoding}{'utf8jp'}{$bin{'utf8jp' }}) and ($tr{$encoding}{'utf8jp'}{$bin{'utf8jp' }} ne '')) {
                die qq{@{[__FILE__]} duplicate definitions \$tr{'$encoding'}{'utf8jp'}{'$hex{utf8}'} = "$hex{$encoding}" and "} . uc unpack('H*',$tr{$encoding}{'utf8jp'}{$bin{'utf8jp'}})    . qq{"\n};
            }
            if ($bin{$encoding} ne '') {
                $tr{'utf8jp'}{$encoding}{$bin{$encoding}} = $bin{'utf8jp'};
                $tr{$encoding}{'utf8jp'}{$bin{'utf8jp'} } = $bin{$encoding};
            }
        }
    }
}

my $data_count = scalar(keys %{$tr{'utf8jp'}{'utf8jp'}});
if ($data_count != 11578) {
    die qq{@{[__FILE__]} is probably broken(data_count=$data_count).\n};
}

my %Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode = ();
for (split /\n/, <<'END') {
# https://support.microsoft.com/ja-jp/help/170559/prb-conversion-problem-between-shift-jis-and-unicode
# CodePage 932 : 398 non-round-trip mappings
0x8790   -> U+2252   -> 0x81e0   Approximately Equal To Or The Image Of
0x8791   -> U+2261   -> 0x81df   Identical To
0x8792   -> U+222b   -> 0x81e7   Integral
0x8795   -> U+221a   -> 0x81e3   Square Root
0x8796   -> U+22a5   -> 0x81db   Up Tack
0x8797   -> U+2220   -> 0x81da   Angle
0x879a   -> U+2235   -> 0x81e6   Because
0x879b   -> U+2229   -> 0x81bf   Intersection
0x879c   -> U+222a   -> 0x81be   Union
0xed40   -> U+7e8a   -> 0xfa5c   CJK Unified Ideograph
0xed41   -> U+891c   -> 0xfa5d   CJK Unified Ideograph
0xed42   -> U+9348   -> 0xfa5e   CJK Unified Ideograph
0xed43   -> U+9288   -> 0xfa5f   CJK Unified Ideograph
0xed44   -> U+84dc   -> 0xfa60   CJK Unified Ideograph
0xed45   -> U+4fc9   -> 0xfa61   CJK Unified Ideograph
0xed46   -> U+70bb   -> 0xfa62   CJK Unified Ideograph
0xed47   -> U+6631   -> 0xfa63   CJK Unified Ideograph
0xed48   -> U+68c8   -> 0xfa64   CJK Unified Ideograph
0xed49   -> U+92f9   -> 0xfa65   CJK Unified Ideograph
0xed4a   -> U+66fb   -> 0xfa66   CJK Unified Ideograph
0xed4b   -> U+5f45   -> 0xfa67   CJK Unified Ideograph
0xed4c   -> U+4e28   -> 0xfa68   CJK Unified Ideograph
0xed4d   -> U+4ee1   -> 0xfa69   CJK Unified Ideograph
0xed4e   -> U+4efc   -> 0xfa6a   CJK Unified Ideograph
0xed4f   -> U+4f00   -> 0xfa6b   CJK Unified Ideograph
0xed50   -> U+4f03   -> 0xfa6c   CJK Unified Ideograph
0xed51   -> U+4f39   -> 0xfa6d   CJK Unified Ideograph
0xed52   -> U+4f56   -> 0xfa6e   CJK Unified Ideograph
0xed53   -> U+4f92   -> 0xfa6f   CJK Unified Ideograph
0xed54   -> U+4f8a   -> 0xfa70   CJK Unified Ideograph
0xed55   -> U+4f9a   -> 0xfa71   CJK Unified Ideograph
0xed56   -> U+4f94   -> 0xfa72   CJK Unified Ideograph
0xed57   -> U+4fcd   -> 0xfa73   CJK Unified Ideograph
0xed58   -> U+5040   -> 0xfa74   CJK Unified Ideograph
0xed59   -> U+5022   -> 0xfa75   CJK Unified Ideograph
0xed5a   -> U+4fff   -> 0xfa76   CJK Unified Ideograph
0xed5b   -> U+501e   -> 0xfa77   CJK Unified Ideograph
0xed5c   -> U+5046   -> 0xfa78   CJK Unified Ideograph
0xed5d   -> U+5070   -> 0xfa79   CJK Unified Ideograph
0xed5e   -> U+5042   -> 0xfa7a   CJK Unified Ideograph
0xed5f   -> U+5094   -> 0xfa7b   CJK Unified Ideograph
0xed60   -> U+50f4   -> 0xfa7c   CJK Unified Ideograph
0xed61   -> U+50d8   -> 0xfa7d   CJK Unified Ideograph
0xed62   -> U+514a   -> 0xfa7e   CJK Unified Ideograph
0xed63   -> U+5164   -> 0xfa80   CJK Unified Ideograph
0xed64   -> U+519d   -> 0xfa81   CJK Unified Ideograph
0xed65   -> U+51be   -> 0xfa82   CJK Unified Ideograph
0xed66   -> U+51ec   -> 0xfa83   CJK Unified Ideograph
0xed67   -> U+5215   -> 0xfa84   CJK Unified Ideograph
0xed68   -> U+529c   -> 0xfa85   CJK Unified Ideograph
0xed69   -> U+52a6   -> 0xfa86   CJK Unified Ideograph
0xed6a   -> U+52c0   -> 0xfa87   CJK Unified Ideograph
0xed6b   -> U+52db   -> 0xfa88   CJK Unified Ideograph
0xed6c   -> U+5300   -> 0xfa89   CJK Unified Ideograph
0xed6d   -> U+5307   -> 0xfa8a   CJK Unified Ideograph
0xed6e   -> U+5324   -> 0xfa8b   CJK Unified Ideograph
0xed6f   -> U+5372   -> 0xfa8c   CJK Unified Ideograph
0xed70   -> U+5393   -> 0xfa8d   CJK Unified Ideograph
0xed71   -> U+53b2   -> 0xfa8e   CJK Unified Ideograph
0xed72   -> U+53dd   -> 0xfa8f   CJK Unified Ideograph
0xed73   -> U+fa0e   -> 0xfa90   CJK compatibility Ideograph
0xed74   -> U+549c   -> 0xfa91   CJK Unified Ideograph
0xed75   -> U+548a   -> 0xfa92   CJK Unified Ideograph
0xed76   -> U+54a9   -> 0xfa93   CJK Unified Ideograph
0xed77   -> U+54ff   -> 0xfa94   CJK Unified Ideograph
0xed78   -> U+5586   -> 0xfa95   CJK Unified Ideograph
0xed79   -> U+5759   -> 0xfa96   CJK Unified Ideograph
0xed7a   -> U+5765   -> 0xfa97   CJK Unified Ideograph
0xed7b   -> U+57ac   -> 0xfa98   CJK Unified Ideograph
0xed7c   -> U+57c8   -> 0xfa99   CJK Unified Ideograph
0xed7d   -> U+57c7   -> 0xfa9a   CJK Unified Ideograph
0xed7e   -> U+fa0f   -> 0xfa9b   CJK compatibility Ideograph
0xed80   -> U+fa10   -> 0xfa9c   CJK compatibility Ideograph
0xed81   -> U+589e   -> 0xfa9d   CJK Unified Ideograph
0xed82   -> U+58b2   -> 0xfa9e   CJK Unified Ideograph
0xed83   -> U+590b   -> 0xfa9f   CJK Unified Ideograph
0xed84   -> U+5953   -> 0xfaa0   CJK Unified Ideograph
0xed85   -> U+595b   -> 0xfaa1   CJK Unified Ideograph
0xed86   -> U+595d   -> 0xfaa2   CJK Unified Ideograph
0xed87   -> U+5963   -> 0xfaa3   CJK Unified Ideograph
0xed88   -> U+59a4   -> 0xfaa4   CJK Unified Ideograph
0xed89   -> U+59ba   -> 0xfaa5   CJK Unified Ideograph
0xed8a   -> U+5b56   -> 0xfaa6   CJK Unified Ideograph
0xed8b   -> U+5bc0   -> 0xfaa7   CJK Unified Ideograph
0xed8c   -> U+752f   -> 0xfaa8   CJK Unified Ideograph
0xed8d   -> U+5bd8   -> 0xfaa9   CJK Unified Ideograph
0xed8e   -> U+5bec   -> 0xfaaa   CJK Unified Ideograph
0xed8f   -> U+5c1e   -> 0xfaab   CJK Unified Ideograph
0xed90   -> U+5ca6   -> 0xfaac   CJK Unified Ideograph
0xed91   -> U+5cba   -> 0xfaad   CJK Unified Ideograph
0xed92   -> U+5cf5   -> 0xfaae   CJK Unified Ideograph
0xed93   -> U+5d27   -> 0xfaaf   CJK Unified Ideograph
0xed94   -> U+5d53   -> 0xfab0   CJK Unified Ideograph
0xed95   -> U+fa11   -> 0xfab1   CJK compatibility Ideograph
0xed96   -> U+5d42   -> 0xfab2   CJK Unified Ideograph
0xed97   -> U+5d6d   -> 0xfab3   CJK Unified Ideograph
0xed98   -> U+5db8   -> 0xfab4   CJK Unified Ideograph
0xed99   -> U+5db9   -> 0xfab5   CJK Unified Ideograph
0xed9a   -> U+5dd0   -> 0xfab6   CJK Unified Ideograph
0xed9b   -> U+5f21   -> 0xfab7   CJK Unified Ideograph
0xed9c   -> U+5f34   -> 0xfab8   CJK Unified Ideograph
0xed9d   -> U+5f67   -> 0xfab9   CJK Unified Ideograph
0xed9e   -> U+5fb7   -> 0xfaba   CJK Unified Ideograph
0xed9f   -> U+5fde   -> 0xfabb   CJK Unified Ideograph
0xeda0   -> U+605d   -> 0xfabc   CJK Unified Ideograph
0xeda1   -> U+6085   -> 0xfabd   CJK Unified Ideograph
0xeda2   -> U+608a   -> 0xfabe   CJK Unified Ideograph
0xeda3   -> U+60de   -> 0xfabf   CJK Unified Ideograph
0xeda4   -> U+60d5   -> 0xfac0   CJK Unified Ideograph
0xeda5   -> U+6120   -> 0xfac1   CJK Unified Ideograph
0xeda6   -> U+60f2   -> 0xfac2   CJK Unified Ideograph
0xeda7   -> U+6111   -> 0xfac3   CJK Unified Ideograph
0xeda8   -> U+6137   -> 0xfac4   CJK Unified Ideograph
0xeda9   -> U+6130   -> 0xfac5   CJK Unified Ideograph
0xedaa   -> U+6198   -> 0xfac6   CJK Unified Ideograph
0xedab   -> U+6213   -> 0xfac7   CJK Unified Ideograph
0xedac   -> U+62a6   -> 0xfac8   CJK Unified Ideograph
0xedad   -> U+63f5   -> 0xfac9   CJK Unified Ideograph
0xedae   -> U+6460   -> 0xfaca   CJK Unified Ideograph
0xedaf   -> U+649d   -> 0xfacb   CJK Unified Ideograph
0xedb0   -> U+64ce   -> 0xfacc   CJK Unified Ideograph
0xedb1   -> U+654e   -> 0xfacd   CJK Unified Ideograph
0xedb2   -> U+6600   -> 0xface   CJK Unified Ideograph
0xedb3   -> U+6615   -> 0xfacf   CJK Unified Ideograph
0xedb4   -> U+663b   -> 0xfad0   CJK Unified Ideograph
0xedb5   -> U+6609   -> 0xfad1   CJK Unified Ideograph
0xedb6   -> U+662e   -> 0xfad2   CJK Unified Ideograph
0xedb7   -> U+661e   -> 0xfad3   CJK Unified Ideograph
0xedb8   -> U+6624   -> 0xfad4   CJK Unified Ideograph
0xedb9   -> U+6665   -> 0xfad5   CJK Unified Ideograph
0xedba   -> U+6657   -> 0xfad6   CJK Unified Ideograph
0xedbb   -> U+6659   -> 0xfad7   CJK Unified Ideograph
0xedbc   -> U+fa12   -> 0xfad8   CJK compatibility Ideograph
0xedbd   -> U+6673   -> 0xfad9   CJK Unified Ideograph
0xedbe   -> U+6699   -> 0xfada   CJK Unified Ideograph
0xedbf   -> U+66a0   -> 0xfadb   CJK Unified Ideograph
0xedc0   -> U+66b2   -> 0xfadc   CJK Unified Ideograph
0xedc1   -> U+66bf   -> 0xfadd   CJK Unified Ideograph
0xedc2   -> U+66fa   -> 0xfade   CJK Unified Ideograph
0xedc3   -> U+670e   -> 0xfadf   CJK Unified Ideograph
0xedc4   -> U+f929   -> 0xfae0   CJK compatibility Ideograph
0xedc5   -> U+6766   -> 0xfae1   CJK Unified Ideograph
0xedc6   -> U+67bb   -> 0xfae2   CJK Unified Ideograph
0xedc7   -> U+6852   -> 0xfae3   CJK Unified Ideograph
0xedc8   -> U+67c0   -> 0xfae4   CJK Unified Ideograph
0xedc9   -> U+6801   -> 0xfae5   CJK Unified Ideograph
0xedca   -> U+6844   -> 0xfae6   CJK Unified Ideograph
0xedcb   -> U+68cf   -> 0xfae7   CJK Unified Ideograph
0xedcc   -> U+fa13   -> 0xfae8   CJK compatibility Ideograph
0xedcd   -> U+6968   -> 0xfae9   CJK Unified Ideograph
0xedce   -> U+fa14   -> 0xfaea   CJK compatibility Ideograph
0xedcf   -> U+6998   -> 0xfaeb   CJK Unified Ideograph
0xedd0   -> U+69e2   -> 0xfaec   CJK Unified Ideograph
0xedd1   -> U+6a30   -> 0xfaed   CJK Unified Ideograph
0xedd2   -> U+6a6b   -> 0xfaee   CJK Unified Ideograph
0xedd3   -> U+6a46   -> 0xfaef   CJK Unified Ideograph
0xedd4   -> U+6a73   -> 0xfaf0   CJK Unified Ideograph
0xedd5   -> U+6a7e   -> 0xfaf1   CJK Unified Ideograph
0xedd6   -> U+6ae2   -> 0xfaf2   CJK Unified Ideograph
0xedd7   -> U+6ae4   -> 0xfaf3   CJK Unified Ideograph
0xedd8   -> U+6bd6   -> 0xfaf4   CJK Unified Ideograph
0xedd9   -> U+6c3f   -> 0xfaf5   CJK Unified Ideograph
0xedda   -> U+6c5c   -> 0xfaf6   CJK Unified Ideograph
0xeddb   -> U+6c86   -> 0xfaf7   CJK Unified Ideograph
0xeddc   -> U+6c6f   -> 0xfaf8   CJK Unified Ideograph
0xeddd   -> U+6cda   -> 0xfaf9   CJK Unified Ideograph
0xedde   -> U+6d04   -> 0xfafa   CJK Unified Ideograph
0xeddf   -> U+6d87   -> 0xfafb   CJK Unified Ideograph
0xede0   -> U+6d6f   -> 0xfafc   CJK Unified Ideograph
0xede1   -> U+6d96   -> 0xfb40   CJK Unified Ideograph
0xede2   -> U+6dac   -> 0xfb41   CJK Unified Ideograph
0xede3   -> U+6dcf   -> 0xfb42   CJK Unified Ideograph
0xede4   -> U+6df8   -> 0xfb43   CJK Unified Ideograph
0xede5   -> U+6df2   -> 0xfb44   CJK Unified Ideograph
0xede6   -> U+6dfc   -> 0xfb45   CJK Unified Ideograph
0xede7   -> U+6e39   -> 0xfb46   CJK Unified Ideograph
0xede8   -> U+6e5c   -> 0xfb47   CJK Unified Ideograph
0xede9   -> U+6e27   -> 0xfb48   CJK Unified Ideograph
0xedea   -> U+6e3c   -> 0xfb49   CJK Unified Ideograph
0xedeb   -> U+6ebf   -> 0xfb4a   CJK Unified Ideograph
0xedec   -> U+6f88   -> 0xfb4b   CJK Unified Ideograph
0xeded   -> U+6fb5   -> 0xfb4c   CJK Unified Ideograph
0xedee   -> U+6ff5   -> 0xfb4d   CJK Unified Ideograph
0xedef   -> U+7005   -> 0xfb4e   CJK Unified Ideograph
0xedf0   -> U+7007   -> 0xfb4f   CJK Unified Ideograph
0xedf1   -> U+7028   -> 0xfb50   CJK Unified Ideograph
0xedf2   -> U+7085   -> 0xfb51   CJK Unified Ideograph
0xedf3   -> U+70ab   -> 0xfb52   CJK Unified Ideograph
0xedf4   -> U+710f   -> 0xfb53   CJK Unified Ideograph
0xedf5   -> U+7104   -> 0xfb54   CJK Unified Ideograph
0xedf6   -> U+715c   -> 0xfb55   CJK Unified Ideograph
0xedf7   -> U+7146   -> 0xfb56   CJK Unified Ideograph
0xedf8   -> U+7147   -> 0xfb57   CJK Unified Ideograph
0xedf9   -> U+fa15   -> 0xfb58   CJK compatibility Ideograph
0xedfa   -> U+71c1   -> 0xfb59   CJK Unified Ideograph
0xedfb   -> U+71fe   -> 0xfb5a   CJK Unified Ideograph
0xedfc   -> U+72b1   -> 0xfb5b   CJK Unified Ideograph
0xee40   -> U+72be   -> 0xfb5c   CJK Unified Ideograph
0xee41   -> U+7324   -> 0xfb5d   CJK Unified Ideograph
0xee42   -> U+fa16   -> 0xfb5e   CJK compatibility Ideograph
0xee43   -> U+7377   -> 0xfb5f   CJK Unified Ideograph
0xee44   -> U+73bd   -> 0xfb60   CJK Unified Ideograph
0xee45   -> U+73c9   -> 0xfb61   CJK Unified Ideograph
0xee46   -> U+73d6   -> 0xfb62   CJK Unified Ideograph
0xee47   -> U+73e3   -> 0xfb63   CJK Unified Ideograph
0xee48   -> U+73d2   -> 0xfb64   CJK Unified Ideograph
0xee49   -> U+7407   -> 0xfb65   CJK Unified Ideograph
0xee4a   -> U+73f5   -> 0xfb66   CJK Unified Ideograph
0xee4b   -> U+7426   -> 0xfb67   CJK Unified Ideograph
0xee4c   -> U+742a   -> 0xfb68   CJK Unified Ideograph
0xee4d   -> U+7429   -> 0xfb69   CJK Unified Ideograph
0xee4e   -> U+742e   -> 0xfb6a   CJK Unified Ideograph
0xee4f   -> U+7462   -> 0xfb6b   CJK Unified Ideograph
0xee50   -> U+7489   -> 0xfb6c   CJK Unified Ideograph
0xee51   -> U+749f   -> 0xfb6d   CJK Unified Ideograph
0xee52   -> U+7501   -> 0xfb6e   CJK Unified Ideograph
0xee53   -> U+756f   -> 0xfb6f   CJK Unified Ideograph
0xee54   -> U+7682   -> 0xfb70   CJK Unified Ideograph
0xee55   -> U+769c   -> 0xfb71   CJK Unified Ideograph
0xee56   -> U+769e   -> 0xfb72   CJK Unified Ideograph
0xee57   -> U+769b   -> 0xfb73   CJK Unified Ideograph
0xee58   -> U+76a6   -> 0xfb74   CJK Unified Ideograph
0xee59   -> U+fa17   -> 0xfb75   CJK compatibility Ideograph
0xee5a   -> U+7746   -> 0xfb76   CJK Unified Ideograph
0xee5b   -> U+52af   -> 0xfb77   CJK Unified Ideograph
0xee5c   -> U+7821   -> 0xfb78   CJK Unified Ideograph
0xee5d   -> U+784e   -> 0xfb79   CJK Unified Ideograph
0xee5e   -> U+7864   -> 0xfb7a   CJK Unified Ideograph
0xee5f   -> U+787a   -> 0xfb7b   CJK Unified Ideograph
0xee60   -> U+7930   -> 0xfb7c   CJK Unified Ideograph
0xee61   -> U+fa18   -> 0xfb7d   CJK compatibility Ideograph
0xee62   -> U+fa19   -> 0xfb7e   CJK compatibility Ideograph
0xee63   -> U+fa1a   -> 0xfb80   CJK compatibility Ideograph
0xee64   -> U+7994   -> 0xfb81   CJK Unified Ideograph
0xee65   -> U+fa1b   -> 0xfb82   CJK compatibility Ideograph
0xee66   -> U+799b   -> 0xfb83   CJK Unified Ideograph
0xee67   -> U+7ad1   -> 0xfb84   CJK Unified Ideograph
0xee68   -> U+7ae7   -> 0xfb85   CJK Unified Ideograph
0xee69   -> U+fa1c   -> 0xfb86   CJK compatibility Ideograph
0xee6a   -> U+7aeb   -> 0xfb87   CJK Unified Ideograph
0xee6b   -> U+7b9e   -> 0xfb88   CJK Unified Ideograph
0xee6c   -> U+fa1d   -> 0xfb89   CJK compatibility Ideograph
0xee6d   -> U+7d48   -> 0xfb8a   CJK Unified Ideograph
0xee6e   -> U+7d5c   -> 0xfb8b   CJK Unified Ideograph
0xee6f   -> U+7db7   -> 0xfb8c   CJK Unified Ideograph
0xee70   -> U+7da0   -> 0xfb8d   CJK Unified Ideograph
0xee71   -> U+7dd6   -> 0xfb8e   CJK Unified Ideograph
0xee72   -> U+7e52   -> 0xfb8f   CJK Unified Ideograph
0xee73   -> U+7f47   -> 0xfb90   CJK Unified Ideograph
0xee74   -> U+7fa1   -> 0xfb91   CJK Unified Ideograph
0xee75   -> U+fa1e   -> 0xfb92   CJK compatibility Ideograph
0xee76   -> U+8301   -> 0xfb93   CJK Unified Ideograph
0xee77   -> U+8362   -> 0xfb94   CJK Unified Ideograph
0xee78   -> U+837f   -> 0xfb95   CJK Unified Ideograph
0xee79   -> U+83c7   -> 0xfb96   CJK Unified Ideograph
0xee7a   -> U+83f6   -> 0xfb97   CJK Unified Ideograph
0xee7b   -> U+8448   -> 0xfb98   CJK Unified Ideograph
0xee7c   -> U+84b4   -> 0xfb99   CJK Unified Ideograph
0xee7d   -> U+8553   -> 0xfb9a   CJK Unified Ideograph
0xee7e   -> U+8559   -> 0xfb9b   CJK Unified Ideograph
0xee80   -> U+856b   -> 0xfb9c   CJK Unified Ideograph
0xee81   -> U+fa1f   -> 0xfb9d   CJK compatibility Ideograph
0xee82   -> U+85b0   -> 0xfb9e   CJK Unified Ideograph
0xee83   -> U+fa20   -> 0xfb9f   CJK compatibility Ideograph
0xee84   -> U+fa21   -> 0xfba0   CJK compatibility Ideograph
0xee85   -> U+8807   -> 0xfba1   CJK Unified Ideograph
0xee86   -> U+88f5   -> 0xfba2   CJK Unified Ideograph
0xee87   -> U+8a12   -> 0xfba3   CJK Unified Ideograph
0xee88   -> U+8a37   -> 0xfba4   CJK Unified Ideograph
0xee89   -> U+8a79   -> 0xfba5   CJK Unified Ideograph
0xee8a   -> U+8aa7   -> 0xfba6   CJK Unified Ideograph
0xee8b   -> U+8abe   -> 0xfba7   CJK Unified Ideograph
0xee8c   -> U+8adf   -> 0xfba8   CJK Unified Ideograph
0xee8d   -> U+fa22   -> 0xfba9   CJK compatibility Ideograph
0xee8e   -> U+8af6   -> 0xfbaa   CJK Unified Ideograph
0xee8f   -> U+8b53   -> 0xfbab   CJK Unified Ideograph
0xee90   -> U+8b7f   -> 0xfbac   CJK Unified Ideograph
0xee91   -> U+8cf0   -> 0xfbad   CJK Unified Ideograph
0xee92   -> U+8cf4   -> 0xfbae   CJK Unified Ideograph
0xee93   -> U+8d12   -> 0xfbaf   CJK Unified Ideograph
0xee94   -> U+8d76   -> 0xfbb0   CJK Unified Ideograph
0xee95   -> U+fa23   -> 0xfbb1   CJK compatibility Ideograph
0xee96   -> U+8ecf   -> 0xfbb2   CJK Unified Ideograph
0xee97   -> U+fa24   -> 0xfbb3   CJK compatibility Ideograph
0xee98   -> U+fa25   -> 0xfbb4   CJK compatibility Ideograph
0xee99   -> U+9067   -> 0xfbb5   CJK Unified Ideograph
0xee9a   -> U+90de   -> 0xfbb6   CJK Unified Ideograph
0xee9b   -> U+fa26   -> 0xfbb7   CJK compatibility Ideograph
0xee9c   -> U+9115   -> 0xfbb8   CJK Unified Ideograph
0xee9d   -> U+9127   -> 0xfbb9   CJK Unified Ideograph
0xee9e   -> U+91da   -> 0xfbba   CJK Unified Ideograph
0xee9f   -> U+91d7   -> 0xfbbb   CJK Unified Ideograph
0xeea0   -> U+91de   -> 0xfbbc   CJK Unified Ideograph
0xeea1   -> U+91ed   -> 0xfbbd   CJK Unified Ideograph
0xeea2   -> U+91ee   -> 0xfbbe   CJK Unified Ideograph
0xeea3   -> U+91e4   -> 0xfbbf   CJK Unified Ideograph
0xeea4   -> U+91e5   -> 0xfbc0   CJK Unified Ideograph
0xeea5   -> U+9206   -> 0xfbc1   CJK Unified Ideograph
0xeea6   -> U+9210   -> 0xfbc2   CJK Unified Ideograph
0xeea7   -> U+920a   -> 0xfbc3   CJK Unified Ideograph
0xeea8   -> U+923a   -> 0xfbc4   CJK Unified Ideograph
0xeea9   -> U+9240   -> 0xfbc5   CJK Unified Ideograph
0xeeaa   -> U+923c   -> 0xfbc6   CJK Unified Ideograph
0xeeab   -> U+924e   -> 0xfbc7   CJK Unified Ideograph
0xeeac   -> U+9259   -> 0xfbc8   CJK Unified Ideograph
0xeead   -> U+9251   -> 0xfbc9   CJK Unified Ideograph
0xeeae   -> U+9239   -> 0xfbca   CJK Unified Ideograph
0xeeaf   -> U+9267   -> 0xfbcb   CJK Unified Ideograph
0xeeb0   -> U+92a7   -> 0xfbcc   CJK Unified Ideograph
0xeeb1   -> U+9277   -> 0xfbcd   CJK Unified Ideograph
0xeeb2   -> U+9278   -> 0xfbce   CJK Unified Ideograph
0xeeb3   -> U+92e7   -> 0xfbcf   CJK Unified Ideograph
0xeeb4   -> U+92d7   -> 0xfbd0   CJK Unified Ideograph
0xeeb5   -> U+92d9   -> 0xfbd1   CJK Unified Ideograph
0xeeb6   -> U+92d0   -> 0xfbd2   CJK Unified Ideograph
0xeeb7   -> U+fa27   -> 0xfbd3   CJK compatibility Ideograph
0xeeb8   -> U+92d5   -> 0xfbd4   CJK Unified Ideograph
0xeeb9   -> U+92e0   -> 0xfbd5   CJK Unified Ideograph
0xeeba   -> U+92d3   -> 0xfbd6   CJK Unified Ideograph
0xeebb   -> U+9325   -> 0xfbd7   CJK Unified Ideograph
0xeebc   -> U+9321   -> 0xfbd8   CJK Unified Ideograph
0xeebd   -> U+92fb   -> 0xfbd9   CJK Unified Ideograph
0xeebe   -> U+fa28   -> 0xfbda   CJK compatibility Ideograph
0xeebf   -> U+931e   -> 0xfbdb   CJK Unified Ideograph
0xeec0   -> U+92ff   -> 0xfbdc   CJK Unified Ideograph
0xeec1   -> U+931d   -> 0xfbdd   CJK Unified Ideograph
0xeec2   -> U+9302   -> 0xfbde   CJK Unified Ideograph
0xeec3   -> U+9370   -> 0xfbdf   CJK Unified Ideograph
0xeec4   -> U+9357   -> 0xfbe0   CJK Unified Ideograph
0xeec5   -> U+93a4   -> 0xfbe1   CJK Unified Ideograph
0xeec6   -> U+93c6   -> 0xfbe2   CJK Unified Ideograph
0xeec7   -> U+93de   -> 0xfbe3   CJK Unified Ideograph
0xeec8   -> U+93f8   -> 0xfbe4   CJK Unified Ideograph
0xeec9   -> U+9431   -> 0xfbe5   CJK Unified Ideograph
0xeeca   -> U+9445   -> 0xfbe6   CJK Unified Ideograph
0xeecb   -> U+9448   -> 0xfbe7   CJK Unified Ideograph
0xeecc   -> U+9592   -> 0xfbe8   CJK Unified Ideograph
0xeecd   -> U+f9dc   -> 0xfbe9   CJK compatibility Ideograph
0xeece   -> U+fa29   -> 0xfbea   CJK compatibility Ideograph
0xeecf   -> U+969d   -> 0xfbeb   CJK Unified Ideograph
0xeed0   -> U+96af   -> 0xfbec   CJK Unified Ideograph
0xeed1   -> U+9733   -> 0xfbed   CJK Unified Ideograph
0xeed2   -> U+973b   -> 0xfbee   CJK Unified Ideograph
0xeed3   -> U+9743   -> 0xfbef   CJK Unified Ideograph
0xeed4   -> U+974d   -> 0xfbf0   CJK Unified Ideograph
0xeed5   -> U+974f   -> 0xfbf1   CJK Unified Ideograph
0xeed6   -> U+9751   -> 0xfbf2   CJK Unified Ideograph
0xeed7   -> U+9755   -> 0xfbf3   CJK Unified Ideograph
0xeed8   -> U+9857   -> 0xfbf4   CJK Unified Ideograph
0xeed9   -> U+9865   -> 0xfbf5   CJK Unified Ideograph
0xeeda   -> U+fa2a   -> 0xfbf6   CJK compatibility Ideograph
0xeedb   -> U+fa2b   -> 0xfbf7   CJK compatibility Ideograph
0xeedc   -> U+9927   -> 0xfbf8   CJK Unified Ideograph
0xeedd   -> U+fa2c   -> 0xfbf9   CJK compatibility Ideograph
0xeede   -> U+999e   -> 0xfbfa   CJK Unified Ideograph
0xeedf   -> U+9a4e   -> 0xfbfb   CJK Unified Ideograph
0xeee0   -> U+9ad9   -> 0xfbfc   CJK Unified Ideograph
0xeee1   -> U+9adc   -> 0xfc40   CJK Unified Ideograph
0xeee2   -> U+9b75   -> 0xfc41   CJK Unified Ideograph
0xeee3   -> U+9b72   -> 0xfc42   CJK Unified Ideograph
0xeee4   -> U+9b8f   -> 0xfc43   CJK Unified Ideograph
0xeee5   -> U+9bb1   -> 0xfc44   CJK Unified Ideograph
0xeee6   -> U+9bbb   -> 0xfc45   CJK Unified Ideograph
0xeee7   -> U+9c00   -> 0xfc46   CJK Unified Ideograph
0xeee8   -> U+9d70   -> 0xfc47   CJK Unified Ideograph
0xeee9   -> U+9d6b   -> 0xfc48   CJK Unified Ideograph
0xeeea   -> U+fa2d   -> 0xfc49   CJK compatibility Ideograph
0xeeeb   -> U+9e19   -> 0xfc4a   CJK Unified Ideograph
0xeeec   -> U+9ed1   -> 0xfc4b   CJK Unified Ideograph
0xeeef   -> U+2170   -> 0xfa40   Small Roman Numeral One
0xeef0   -> U+2171   -> 0xfa41   Small Roman Numeral Two
0xeef1   -> U+2172   -> 0xfa42   Small Roman Numeral Three
0xeef2   -> U+2173   -> 0xfa43   Small Roman Numeral Four
0xeef3   -> U+2174   -> 0xfa44   Small Roman Numeral Five
0xeef4   -> U+2175   -> 0xfa45   Small Roman Numeral Six
0xeef5   -> U+2176   -> 0xfa46   Small Roman Numeral Seven
0xeef6   -> U+2177   -> 0xfa47   Small Roman Numeral Eight
0xeef7   -> U+2178   -> 0xfa48   Small Roman Numeral Nine
0xeef8   -> U+2179   -> 0xfa49   Small Roman Numeral Ten
0xeef9   -> U+ffe2   -> 0x81ca   Fullwidth Not Sign
0xeefa   -> U+ffe4   -> 0xfa55   Fullwidth Broken Bar
0xeefb   -> U+ff07   -> 0xfa56   Fullwidth Apostrophe
0xeefc   -> U+ff02   -> 0xfa57   Fullwidth Quotation Mark
0xfa4a   -> U+2160   -> 0x8754   Roman Numeral One
0xfa4b   -> U+2161   -> 0x8755   Roman Numeral Two
0xfa4c   -> U+2162   -> 0x8756   Roman Numeral Three
0xfa4d   -> U+2163   -> 0x8757   Roman Numeral Four
0xfa4e   -> U+2164   -> 0x8758   Roman Numeral Five
0xfa4f   -> U+2165   -> 0x8759   Roman Numeral Six
0xfa50   -> U+2166   -> 0x875a   Roman Numeral Seven
0xfa51   -> U+2167   -> 0x875b   Roman Numeral Eight
0xfa52   -> U+2168   -> 0x875c   Roman Numeral Nine
0xfa53   -> U+2169   -> 0x875d   Roman Numeral Ten
0xfa54   -> U+ffe2   -> 0x81ca   Fullwidth Not Sign
0xfa58   -> U+3231   -> 0x878a   Parenthesized Ideograph Stock
0xfa59   -> U+2116   -> 0x8782   Numero Sign
0xfa5a   -> U+2121   -> 0x8784   Telephone Sign
0xfa5b   -> U+2235   -> 0x81e6   Because
END
    next if /^#/;
    if (my($cp932a, $Unicode, $cp932b) = / 0x([0123456789abcdef]{4}) .+? U\+([0123456789abcdef]{4}) .+? 0x([0123456789abcdef]{4}) /x) {
        $Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode{ pack('H*',uc($cp932a)) } = pack('H*',uc($cp932b));
    }
}

my %x = (

    # utf8jp(UTF-8-SPUA-JP) is best choice as internal encoding, because it
    # makes one character by one code point on fixed length without grapheme
    # clustering.
    # Other all are not so.

    'cp932x' => {
        'get_ctype' => sub { m!^[^\x81-\x9F\xE0-\xFC]! ? 'SBCS' : m!^[\x81-\x9F\xE0-\xFC]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^((?:\x9C\x5A)?[\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932x'}{$Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode{$_[0]}||$_[0]} },
        'putc'      => sub { local $^W; $tr{'cp932x'}{'utf8jp'}{$_[0]} },
    },
    'cp932' => {
        'get_ctype' => sub { m!^[^\x81-\x9F\xE0-\xFC]! ? 'SBCS' : m!^[\x81-\x9F\xE0-\xFC]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932'}{$Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode{$_[0]}||$_[0]} },
        'putc'      => sub { local $^W; $tr{'cp932'}{'utf8jp'}{$_[0]} },
    },
    'cp932ibm' => {
        'get_ctype' => sub { m!^[^\x81-\x9F\xE0-\xFC]! ? 'SBCS' : m!^[\x81-\x9F\xE0-\xFC]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
#       'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932ibm'}{$_[0]} },
#                                                      VVVVVVVV--- 'cp932' with KB170559 is better than only 'cp932ibm' on read
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932'   }{$Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode{$_[0]}||$_[0]} },
        'putc'      => sub { local $^W; $tr{'cp932ibm'}{'utf8jp'}{$_[0]} },
    },
    'cp932nec' => {
        'get_ctype' => sub { m!^[^\x81-\x9F\xE0-\xFC]! ? 'SBCS' : m!^[\x81-\x9F\xE0-\xFC]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
#       'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932nec'}{$_[0]} },
#                                                      VVVVVVVV--- 'cp932' with KB170559 is better than only 'cp932nec' on read
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp932'   }{$Knowledge_Base_Article_ID_170559_prb_conversion_problem_between_shift_jis_and_unicode{$_[0]}||$_[0]} },
        'putc'      => sub { local $^W; $tr{'cp932nec'}{'utf8jp'}{$_[0]} },
    },
    'sjis2004' => {
        'get_ctype' => sub { m!^[^\x81-\x9F\xE0-\xFC]! ? 'SBCS' : m!^[\x81-\x9F\xE0-\xFC]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'sjis2004'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'sjis2004'}{'utf8jp'}{$_[0]} },
    },
    'cp00930' => {
        'get_ctype' => sub { s!^\x0F!! ? 'SBCS' : s!^\x0E!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x0F", 'DBCS'=>"\x0E", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'cp00930'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'cp00930'}{'utf8jp'}{$_[0]} },
    },
    'keis78' => {
        'get_ctype' => sub { s!^\x0A\x41!! ? 'SBCS' : s!^\x0A\x42!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x0A\x41", 'DBCS'=>"\x0A\x42", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'keis78'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'keis78'}{'utf8jp'}{$_[0]} },
    },
    'keis83' => {
        'get_ctype' => sub { s!^\x0A\x41!! ? 'SBCS' : s!^\x0A\x42!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x0A\x41", 'DBCS'=>"\x0A\x42", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'keis83'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'keis83'}{'utf8jp'}{$_[0]} },
    },
    'keis90' => {
        'get_ctype' => sub { s!^\x0A\x41!! ? 'SBCS' : s!^\x0A\x42!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x0A\x41", 'DBCS'=>"\x0A\x42", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'keis90'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'keis90'}{'utf8jp'}{$_[0]} },
    },
    'jef' => {
        'get_ctype' => sub { s!^\x29!! ? 'SBCS' : s!^[\x28\x38]!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x29", 'DBCS'=>"\x28", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'jef'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'jef'}{'utf8jp'}{$_[0]} },
    },
    'jef9p' => {
        'get_ctype' => sub { s!^\x29!! ? 'SBCS' : s!^[\x28\x38]!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x29", 'DBCS'=>"\x38", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'jef'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'jef'}{'utf8jp'}{$_[0]} },
    },
    'jipsj' => {
        'get_ctype' => sub { s!^\x1A\x71!! ? 'SBCS' : s!^\x1A\x70!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x1A\x71", 'DBCS'=>"\x1A\x70", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'jipsj'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'jipsj'}{'utf8jp'}{$_[0]} },
    },
    'jipse' => {
        'get_ctype' => sub { s!^\x3F\x76!! ? 'SBCS' : s!^\x3F\x75!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x3F\x76", 'DBCS'=>"\x3F\x75", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'jipse'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'jipse'}{'utf8jp'}{$_[0]} },
    },
    'letsj' => {
        'get_ctype' => sub { s!^\x93\xF1!! ? 'SBCS' : s!^\x93\x70!! ? 'DBCS' : undef },
        'set_ctype' => sub { {'SBCS'=>"\x93\xF1", 'DBCS'=>"\x93\x70", }->{$_[0]} },
        'getoct'    => sub { $_[0] eq 'SBCS' ? s!^([\x00-\xFF])!! : s!^([\x00-\xFF]{1,2})!!; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'letsj'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'letsj'}{'utf8jp'}{$_[0]} },
    },
    'utf8' => {
        'get_ctype' => sub { m!^[\x00-\x7F\xFE\xFF]! ? 'SBCS' : m!^[^\x00-\x7F\xFE\xFF]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { s!^(
            [\x00-\x7F\x80-\xBF\xC0-\xC1\xF5-\xFF] |
            [\xE0-\xE2\xE4-\xEF][\x80-\xBF]{2}     |
            \xE3(?:
                \x81\x8B\xE3\x82\x9A               | # U+304B+309A
                \x81\x8D\xE3\x82\x9A               | # U+304D+309A
                \x81\x8F\xE3\x82\x9A               | # U+304F+309A
                \x81\x91\xE3\x82\x9A               | # U+3051+309A
                \x81\x93\xE3\x82\x9A               | # U+3053+309A
                \x82\xAB\xE3\x82\x9A               | # U+30AB+309A
                \x82\xAD\xE3\x82\x9A               | # U+30AD+309A
                \x82\xAF\xE3\x82\x9A               | # U+30AF+309A
                \x82\xB1\xE3\x82\x9A               | # U+30B1+309A
                \x82\xB3\xE3\x82\x9A               | # U+30B3+309A
                \x82\xBB\xE3\x82\x9A               | # U+30BB+309A
                \x83\x84\xE3\x82\x9A               | # U+30C4+309A
                \x83\x88\xE3\x82\x9A               | # U+30C8+309A
                \x87\xB7\xE3\x82\x9A               | # U+31F7+309A
                [\x80-\xBF]{2}
            )                                      |
            [\xC2\xC4-\xC8\xCC-\xDF][\x80-\xBF]    |
            \xC3(?:
                \xA6\xCC\x80                       | # U+00E6+0300
                [\x80-\xBF]
            )                                      |
            \xC9(?:
                \x94\xCC\x80                       | # U+0254+0300
                \x94\xCC\x81                       | # U+0254+0301
                \x99\xCC\x80                       | # U+0259+0300
                \x99\xCC\x81                       | # U+0259+0301
                \x9A\xCC\x80                       | # U+025A+0300
                \x9A\xCC\x81                       | # U+025A+0301
                [\x80-\xBF]
            )                                      |
            \xCA(?:
                \x8C\xCC\x80                       | # U+028C+0300
                \x8C\xCC\x81                       | # U+028C+0301
                [\x80-\xBF]
            )                                      |
            \xCB(?:
                \xA5\xCB\xA9                       | # U+02E5+02E9
                \xA9\xCB\xA5                       | # U+02E9+02E5
                [\x80-\xBF]
            )                                      |
            [\xF0-\xF4][\x80-\xBF]{3}              |
            [\x00-\xFF]
        )!!xs; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'utf8'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'utf8'}{'utf8jp'}{$_[0]} },
    },
    'utf8.1' => {
        'get_ctype' => sub { m!^[\x00-\x7F\xFE\xFF]! ? 'SBCS' : m!^[^\x00-\x7F\xFE\xFF]! ? 'DBCS' : undef },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { s!^(
            [\x00-\x7F\x80-\xBF\xC0-\xC1\xF5-\xFF] |
            [\xE0-\xE2\xE4-\xEF][\x80-\xBF]{2}     |
            \xE3(?:
                \x81\x8B\xE3\x82\x9A               | # U+304B+309A
                \x81\x8D\xE3\x82\x9A               | # U+304D+309A
                \x81\x8F\xE3\x82\x9A               | # U+304F+309A
                \x81\x91\xE3\x82\x9A               | # U+3051+309A
                \x81\x93\xE3\x82\x9A               | # U+3053+309A
                \x82\xAB\xE3\x82\x9A               | # U+30AB+309A
                \x82\xAD\xE3\x82\x9A               | # U+30AD+309A
                \x82\xAF\xE3\x82\x9A               | # U+30AF+309A
                \x82\xB1\xE3\x82\x9A               | # U+30B1+309A
                \x82\xB3\xE3\x82\x9A               | # U+30B3+309A
                \x82\xBB\xE3\x82\x9A               | # U+30BB+309A
                \x83\x84\xE3\x82\x9A               | # U+30C4+309A
                \x83\x88\xE3\x82\x9A               | # U+30C8+309A
                \x87\xB7\xE3\x82\x9A               | # U+31F7+309A
                [\x80-\xBF]{2}
            )                                      |
            [\xC2\xC4-\xC8\xCC-\xDF][\x80-\xBF]    |
            \xC3(?:
                \xA6\xCC\x80                       | # U+00E6+0300
                [\x80-\xBF]
            )                                      |
            \xC9(?:
                \x94\xCC\x80                       | # U+0254+0300
                \x94\xCC\x81                       | # U+0254+0301
                \x99\xCC\x80                       | # U+0259+0300
                \x99\xCC\x81                       | # U+0259+0301
                \x9A\xCC\x80                       | # U+025A+0300
                \x9A\xCC\x81                       | # U+025A+0301
                [\x80-\xBF]
            )                                      |
            \xCA(?:
                \x8C\xCC\x80                       | # U+028C+0300
                \x8C\xCC\x81                       | # U+028C+0301
                [\x80-\xBF]
            )                                      |
            \xCB(?:
                \xA5\xCB\xA9                       | # U+02E5+02E9
                \xA9\xCB\xA5                       | # U+02E9+02E5
                [\x80-\xBF]
            )                                      |
            [\xF0-\xF4][\x80-\xBF]{3}              |
            [\x00-\xFF]
        )!!xs; $1 },
        'getc'      => sub { local $^W; $tr{'utf8jp'}{'utf8.1'}{$_[0]} },
        'putc'      => sub { local $^W; $tr{'utf8.1'}{'utf8jp'}{$_[0]} },
    },
    'utf8jp' => {
        'get_ctype' => sub { m!^\xF3\xB0(?:[\x80-\x82][\x80-\xBF]|\x83[\x80-\xBE])! ? 'SBCS' : 'DBCS' },
        'set_ctype' => sub { q!! },
        'getoct'    => sub { s!^(\xF3[\xB0-\xB5][\x80-\xBF][\x80-\xBF]|[\x00-\xFF])!!; $1; },
        'getc'      => sub { $_[0] },
        'putc'      => sub { $_[0] },
    },
);

#---------------------------------------------------------------------
# convert encoding to OUTPUT_encoding from INPUT_encoding
#---------------------------------------------------------------------
sub convert {
    local $_            = ${$_[0]};
    my $OUTPUT_encoding = $_[1];
    my $INPUT_encoding  = $_[2];
    my $option          = ($_[3] || {});
    my $last_ctype      = undef;
    my $output          = '';
    my $count           = 0;
    $option->{'OVERRIDE_MAPPING'} ||= {};

    if (ref($_[0]) ne 'SCALAR') {
        die "@{[__FILE__]} \$_[0] isn't scalar reference\n";
    }
    if (not exists $x{$OUTPUT_encoding}) {
        die "@{[__FILE__]} unknown OUTPUT encoding '$OUTPUT_encoding'\n";
    }
    if (not exists $x{$INPUT_encoding}) {
        die "@{[__FILE__]} unknown INPUT encoding '$INPUT_encoding'\n";
    }

    my $INPUT_LAYOUT = undef;
    my @ctype = ();
    if ($INPUT_encoding =~ /^(?:cp932x|cp932|cp932ibm|cp932nec|sjis2004|cp00930|keis78|keis83|keis90|jef|jef9p|jipsj|jipse|letsj)$/) {
        if (defined $option->{'INPUT_LAYOUT'}) {
            $INPUT_LAYOUT = $option->{'INPUT_LAYOUT'};
            $INPUT_LAYOUT =~ s/([SD])([0-9]+)/$1 x $2/ge;
            if ($INPUT_LAYOUT =~ /^[SD]*$/) {
                @ctype = map {{'S'=>'SBCS', 'D'=>'DBCS',}->{$_}} split(//,$INPUT_LAYOUT);
            }
            else {
                die "@{[__FILE__]} INPUT_LAYOUT isn't 'S' or 'D' sequence '$INPUT_LAYOUT'";
            }
        }
    }

    while ($_ ne '') {
        my $ctype = '';
        if (defined $INPUT_LAYOUT) {
            $ctype = (shift(@ctype) || 'SBCS');
        }
        else {
            $ctype = ($x{$INPUT_encoding}{'get_ctype'}->() || $last_ctype || 'SBCS');
        }

        if (not defined($last_ctype) or ($ctype ne $last_ctype)) {
            if ($option->{'OUTPUT_SHIFTING'}) {
                $output .= $x{$OUTPUT_encoding}{'set_ctype'}->($ctype);
            }
            $last_ctype = $ctype;
        }

        my $input_octets = $x{$INPUT_encoding}{'getoct'}->($ctype);
        if (defined $input_octets) {
            if (defined $option->{'OVERRIDE_MAPPING'}{$input_octets}) {
                $output .= $option->{'OVERRIDE_MAPPING'}{$input_octets};
            }
            else {
                my $char = $x{$INPUT_encoding}{'getc'}->($input_octets);
                if (not defined $char) {
                    if (defined $option->{'GETA'}) {
                        $output .= $option->{'GETA'};
                    }
                    else {
                        $output .= $x{$OUTPUT_encoding}{'putc'}->("\xF3\xB0\x85\xAB");
                    }
                }
                elsif ($char eq "\xF3\xB0\x84\x80") {
                    if (defined $option->{'SPACE'}) {
                        $output .= $option->{'SPACE'};
                    }
                    else {
                        $output .= $x{$OUTPUT_encoding}{'putc'}->($char);
                    }
                }
                else {
                    my $output_octets = $x{$OUTPUT_encoding}{'putc'}->($char);
                    if (not defined ($output_octets) or ($output_octets eq '')) {
                        if (defined $option->{'GETA'}) {
                            $output .= $option->{'GETA'};
                        }
                        else {
                            $output .= $x{$OUTPUT_encoding}{'putc'}->("\xF3\xB0\x85\xAB");
                        }
                    }
                    else {
                        $output .= $output_octets;
                    }
                }
            }
        }

        $count++;
    }

    ${$_[0]} = $output;
    return $count;
}

#---------------------------------------------------------------------
# confirm version
#---------------------------------------------------------------------
sub VERSION {
    my($version) = @_;
    if ($version ne $Jacode4e::VERSION) {
        die "@{[__FILE__]} $Jacode4e::VERSION isn't $version";
    }
}

#---------------------------------------------------------------------
# dump encoding tables
#---------------------------------------------------------------------
END {
    if (
        ($0 eq __FILE__)  and
        defined($ARGV[0]) and
        ($ARGV[0] eq '--dumptable')
    ) {

        # dump DBCS tables
        for my $encoding (qw(
            cp932 cp932ibm cp932nec sjis2004 cp00930 keis78 keis83 keis90 jef jef9p jipsj jipse letsj
        )) {
            open(FILE,">$0-$Jacode4e::VERSION.TABLE.\U$encoding\E.txt") || die;
            binmode(FILE);
            for (my $octet1=0x00; $octet1<=0xFF; $octet1+=0x01) {
                for (my $octet2=0x00; $octet2<=0xF0; $octet2+=0x10) {
                    my @line = ();
                    for (my $column=0x00; $column<=0x0F; $column+=0x01) {
                        my $octets = pack('CC', $octet1, $octet2 + $column);
                        Jacode4e::convert(\$octets, 'utf8', $encoding, { 'INPUT_LAYOUT'=>'D', 'GETA'=>"　" });
                        push @line, $octets;
                    }
                    if (grep(!/　/,@line) >= 1) {
                        printf FILE ('%s %02X%02X: ', uc $encoding, $octet1, $octet2);
                        printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 0], $line[ 1], $line[ 2], $line[ 3]);
                        printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 4], $line[ 5], $line[ 6], $line[ 7]);
                        printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 8], $line[ 9], $line[10], $line[11]);
                        printf FILE ('%-2s%-2s%-2s%-2s ', $line[12], $line[13], $line[14], $line[15]);
                        print  FILE "\n";
                    }
                }
            }
            close(FILE);
        }

        # dump CP932X table
        open(FILE,">$0-$Jacode4e::VERSION.TABLE.CP932X.txt") || die;
        binmode(FILE);
        for (my $octet1=0x00; $octet1<=0xFF; $octet1+=0x01) {
            for (my $octet2=0x00; $octet2<=0xF0; $octet2+=0x10) {
                my @line = ();
                for (my $column=0x00; $column<=0x0F; $column+=0x01) {
                    my $octets = pack('CC', $octet1, $octet2 + $column);
                    Jacode4e::convert(\$octets, 'utf8', 'cp932x', { 'INPUT_LAYOUT'=>'D', 'GETA'=>"　" });
                    push @line, $octets;
                }
                for (my $column=0x00; $column<=0x0F; $column+=0x01) {
                    my $octets = pack('CCCC', 0x9C, 0x5A, $octet1, $octet2 + $column);
                    Jacode4e::convert(\$octets, 'utf8', 'cp932x', { 'INPUT_LAYOUT'=>'D', 'GETA'=>"　" });
                    push @line, $octets;
                }
                if (grep(!/　/,@line) >= 1) {
                    printf FILE ('CP932X %02X%02X: ', $octet1, $octet2);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 0], $line[ 1], $line[ 2], $line[ 3]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 4], $line[ 5], $line[ 6], $line[ 7]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[ 8], $line[ 9], $line[10], $line[11]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[12], $line[13], $line[14], $line[15]);
                    print  FILE '  ';
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[16], $line[17], $line[18], $line[19]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[20], $line[21], $line[22], $line[23]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[24], $line[25], $line[26], $line[27]);
                    printf FILE ('%-2s%-2s%-2s%-2s ', $line[28], $line[29], $line[30], $line[31]);
                    print  FILE "\n";
                }
            }
        }
        close(FILE);

        exit;
    }
}

#---------------------------------------------------------------------
# document
#---------------------------------------------------------------------

=pod

=encoding utf8

=head1 NAME

Jacode4e - jacode.pl-like program for enterprise

=head1 SYNOPSIS

  use FindBin;
  use lib "$FindBin::Bin/lib";
  use Jacode4e;
 
  $return =
  Jacode4e::convert(\$line, $OUTPUT_encoding, $INPUT_encoding [, { %option }]);
 
    $return
      Number of characters in $line
 
    $line
      String variable to convert
      After conversion, this variable is overwritten
 
    $OUTPUT_encoding, and $INPUT_encoding
      To convert, you must specify both $OUTPUT_encoding and $INPUT_encoding.
      The encodings you can specify are as follows:
 
      mnemonic      means
      -----------------------------------------------------------------------
      cp932x        CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
      cp932         Microsoft CP932, IANA Windows-31J
      cp932ibm      IBM CP932
      cp932nec      NEC CP932
      sjis2004      JISC Shift_JIS-2004
      cp00930       IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
      keis78        HITACHI KEIS78
      keis83        HITACHI KEIS83
      keis90        HITACHI KEIS90
      jef           FUJITSU JEF (12 point size for printing with option OUTPUT_SHIFTING)
      jef9p         FUJITSU JEF ( 9 point size for printing with option OUTPUT_SHIFTING)
      jipsj         NEC JIPS(J)
      jipse         NEC JIPS(E)
      letsj         UNISYS LetsJ
      utf8          UTF-8.0 (aka UTF-8)
      utf8.1        UTF-8.1
      utf8jp        UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
      -----------------------------------------------------------------------
 
    %option
      The options you can specify are as follows:
 
      key mnemonic      value means
      -----------------------------------------------------------------------
      INPUT_LAYOUT      input record layout by 'S' and 'D' sequence
                        'S' means one char as SBCS, 'D' means one char as DBCS
                        each letter may optionally be followed by a number
                        indicating the repeat count, see samples
      OUTPUT_SHIFTING   true means use output shift code, false means not use
                        default is false
      SPACE             output space code in DBCS/MBCS
      GETA              output geta code in DBCS/MBCS
      OVERRIDE_MAPPING  hash reference of FROM => TO override mapping
                        { "\x12\x34"=>"\x56\x78", "\x9A\xBC"=>"\xDE\xFE", }
                        (CAUTION! override also SPACE option)
      -----------------------------------------------------------------------

    If you need round-trip conversion, you had better use Jacode4e::RoundTrip
    module.

=head1 SAMPLES

  use FindBin;
  use lib "$FindBin::Bin/lib";
  use Jacode4e;
  Jacode4e::VERSION('2.13.6.19');
  
  while (<>) {
      $return =
      Jacode4e::convert(\$_, 'cp932x', 'cp00930', {
          'INPUT_LAYOUT'     => 'SSSDDDSSDDSDSD',
          'OUTPUT_SHIFTING'  => 0,
          'SPACE'            => "\x81\xA2",
          'GETA'             => "\x81\xA1",
          'OVERRIDE_MAPPING' => { "\x44\x5A" => "\x81\x7C", },
      });
      print $_;
  }
 
  while (<>) {
      $return =
      Jacode4e::convert(\$_, 'cp932x', 'cp00930', {
          'INPUT_LAYOUT'     => 'S3D3S2D2SDSD',
          'OUTPUT_SHIFTING'  => 0,
          'SPACE'            => "\x81\xA2",
          'GETA'             => "\x81\xA1",
          'OVERRIDE_MAPPING' => { "\x44\x5A" => "\x81\x7C", },
      });
      print $_;
  }
 
  while (<>) {
      Jacode4e::convert(\$_, 'cp932ibm', 'utf8.1', {});
      print $_;
  }
 
  while (<>) {
      $return =
      Jacode4e::convert(\$_, 'utf8.1', 'utf8', {
                                # FULLWIDTH TILDE => WAVE DASH
          'OVERRIDE_MAPPING' => { "\xEF\xBD\x9E"  => "\xE3\x80\x9C", },
      });
      print $_;
  }

=head1 INPUT SI/SO code

  Wikipedia tells us Kanji shift code of each encoding of vendors.
  Jacode4e::convert() handle SI/SO(Shift In and Shift Out) code in $line
  automatically. If $line has no SI/SO code, we can use option INPUT_LAYOUT
  instead of SI/SO code.
  
  Actually saying, we have to use option INPUT_LAYOUT almost always, if
  $INPUT_encoding is any of enterprise encodings.
  
  If $INPUT_encoding is any of cp932x, cp932, cp932ibm, cp932nec, and sjis2004,
  the INPUT_LAYOUT option is unnecessary. This is because SBCS and DBCS can be
  judged from the character string itself. However, you can also use the
  INPUT_LAYOUT option to force conversion as SBCS.
  
  ---------------------------------------------------------------------------
                     SO(Shift Out)       SI(Shift In)
  $INPUT_encoding    KI(KANJI In)        KO(KANJI Out)
  mnemonic           switch to DBCS      switch to SBCS    %option
  ---------------------------------------------------------------------------
  'cp932x'           (nothing)           (nothing)         also 'INPUT_LAYOUT' => ...
  'cp932'            (nothing)           (nothing)         also 'INPUT_LAYOUT' => ...
  'cp932ibm'         (nothing)           (nothing)         also 'INPUT_LAYOUT' => ...
  'cp932nec'         (nothing)           (nothing)         also 'INPUT_LAYOUT' => ...
  'sjis2004'         (nothing)           (nothing)         also 'INPUT_LAYOUT' => ...
  'cp00930'          "\x0E"              "\x0F"            xor 'INPUT_LAYOUT' => ...
  'keis78'           "\x0A\x42"          "\x0A\x41"        xor 'INPUT_LAYOUT' => ...
  'keis83'           "\x0A\x42"          "\x0A\x41"        xor 'INPUT_LAYOUT' => ...
  'keis90'           "\x0A\x42"          "\x0A\x41"        xor 'INPUT_LAYOUT' => ...
  'jef'              "\x28" or "\x38"    "\x29"            xor 'INPUT_LAYOUT' => ...
  'jef9p'            "\x28" or "\x38"    "\x29"            xor 'INPUT_LAYOUT' => ...
  'jipsj'            "\x1A\x70"          "\x1A\x71"        xor 'INPUT_LAYOUT' => ...
  'jipse'            "\x3F\x75"          "\x3F\x76"        xor 'INPUT_LAYOUT' => ...
  'letsj'            "\x93\x70"          "\x93\xF1"        xor 'INPUT_LAYOUT' => ...
  'utf8'             (nothing)           (nothing)         
  'utf8.1'           (nothing)           (nothing)         
  'utf8jp'           (nothing)           (nothing)         
  ---------------------------------------------------------------------------

=head1 OUTPUT SI/SO code

  Jacode4e::convert() doesn't output SI/SO code on default. Thus, if you
  need SI/SO code then you have to use option 'OUTPUT_SHIFTING' => 1.
  
  ---------------------------------------------------------------------------
                     SO(Shift Out)       SI(Shift In)
  $OUTPUT_encoding   KI(KANJI In)        KO(KANJI Out)
  mnemonic           switch to DBCS      switch to SBCS    %option
  ---------------------------------------------------------------------------
  'cp932x'           (nothing)           (nothing)         
  'cp932'            (nothing)           (nothing)         
  'cp932ibm'         (nothing)           (nothing)         
  'cp932nec'         (nothing)           (nothing)         
  'sjis2004'         (nothing)           (nothing)         
  'cp00930'          "\x0E"              "\x0F"            'OUTPUT_SHIFTING' => 1
  'keis78'           "\x0A\x42"          "\x0A\x41"        'OUTPUT_SHIFTING' => 1
  'keis83'           "\x0A\x42"          "\x0A\x41"        'OUTPUT_SHIFTING' => 1
  'keis90'           "\x0A\x42"          "\x0A\x41"        'OUTPUT_SHIFTING' => 1
  'jef'              "\x28"              "\x29"            'OUTPUT_SHIFTING' => 1
  'jef9p'            "\x38"              "\x29"            'OUTPUT_SHIFTING' => 1
  'jipsj'            "\x1A\x70"          "\x1A\x71"        'OUTPUT_SHIFTING' => 1
  'jipse'            "\x3F\x75"          "\x3F\x76"        'OUTPUT_SHIFTING' => 1
  'letsj'            "\x93\x70"          "\x93\xF1"        'OUTPUT_SHIFTING' => 1
  'utf8'             (nothing)           (nothing)         
  'utf8.1'           (nothing)           (nothing)         
  'utf8jp'           (nothing)           (nothing)         
  ---------------------------------------------------------------------------

=head1 OUTPUT DBCS/MBCS SPACE code

  The default space code is as follows.
  You can change the space code using the option 'SPACE' if you want.
  
  ---------------------------------------------------------------------------
  $OUTPUT_encoding
  mnemonic           default code        %option
  ---------------------------------------------------------------------------
  'cp932x'           "\x81\x40"          
  'cp932'            "\x81\x40"          
  'cp932ibm'         "\x81\x40"          
  'cp932nec'         "\x81\x40"          
  'sjis2004'         "\x81\x40"          'SPACE' => "\x20\x20" for CP/M-86 compatible
  'cp00930'          "\x40\x40"          
  'keis78'           "\xA1\xA1"          
  'keis83'           "\xA1\xA1"          
  'keis90'           "\xA1\xA1"          
  'jef'              "\xA1\xA1"          'SPACE' => "\x40\x40" for 99FR-0012-2 and 99FR-0012-3 compatible
  'jef9p'            "\xA1\xA1"          'SPACE' => "\x40\x40" for 99FR-0012-2 and 99FR-0012-3 compatible
  'jipsj'            "\x21\x21"          
  'jipse'            "\x4F\x4F"          
  'letsj'            "\x20\x20"          'SPACE' => "\xA1\xA1" for EUC-JP like space
  'utf8'             "\xE3\x80\x80"      
  'utf8.1'           "\xE3\x80\x80"      
  'utf8jp'           "\xF3\xB0\x84\x80"  
  ---------------------------------------------------------------------------

=head1 OUTPUT DBCS/MBCS GETA code

  If a character isn't included in $OUTPUT_encoding set, GETA code will be
  used instead of converted code.
  
  The default GETA code is as follows.
  You can change GETA code using option 'GETA' if you want.
  
  "GETA" doesn't mean "GETA", but means "GETA-MARK".
  
  GETA is Japanese wooden shoes that made for walk on paddy field. One GETA
  has two teeth, and they make GETA-MARK on the ground by bite the earth
  twice. Thus, GETA code is double byte code, or often multibyte code.
  
  ---------------------------------------------------------------------------
  $OUTPUT_encoding
  mnemonic           default code        %option sample
  ---------------------------------------------------------------------------
  'cp932x'           "\x81\xAC"          'GETA' => "\x81\xA1"
  'cp932'            "\x81\xAC"          'GETA' => "\x81\x9C"
  'cp932ibm'         "\x81\xAC"          'GETA' => "\x81\x9C"
  'cp932nec'         "\x81\xAC"          'GETA' => "\x81\x9C"
  'sjis2004'         "\x81\xAC"          'GETA' => "\x81\xFC"
  'cp00930'          "\x44\x7D"          
  'keis78'           "\xA2\xAE"          
  'keis83'           "\xA2\xAE"          
  'keis90'           "\xA2\xAE"          
  'jef'              "\xA2\xAE"          
  'jef9p'            "\xA2\xAE"          
  'jipsj'            "\x22\x2E"          
  'jipse'            "\x7F\x4B"          
  'letsj'            "\xA2\xAE"          
  'utf8'             "\xE3\x80\x93"      
  'utf8.1'           "\xE3\x80\x93"      
  'utf8jp'           "\xF3\xB0\x85\xAB"  
  ---------------------------------------------------------------------------

=head1 RAISON D'ETRE

 This software has been developed for use promotion of JIS X 0213.
 
 JIS X 0213 contains almost all of the characters used in everyday life in
 Japan. In addition, JIS X 0213 was designed with consideration for use
 in Shift_JIS environment. However, because it did not consider CP932 and
 GAIJI, it is not very popular.
 
 "JIS X 0213 bridge" needs last one meter.
 
 To solve the problem, Jacode4e proposes CP932X [shi:pi:kju:san'nikai]. #'
 
 CP932X implements JIS X 0213 over CP932.
 
 We can continue to use CP932 data, application programs, databases, and
 networks using CP932X.
 
 If you need round-trip conversion, you had better use Jacode4e::RoundTrip
 module.

=head1 WHAT IS "CP932X"?

=over 4

=item * "cp932x" as mnemonic

=item * CP932X is CP932

=item * Pronounce [si: pi: nain thri: tu: kai] in English

=item * Pronounce [shi: pi: kju: san' ni kai] in Japanese #'

=item * [si: pi: nain thri: tu: iks] is reserved for Microsoft Corporation ;-P

=item * CP932 upper compatible

=item * Supports JIS X 0213 character set

=item * Used ghost character "\x9C\x5A" as single shift code

=item * Used "\x9C\x5A\x9C\x5A" for single "\x9C\x5A"

=item * You can use private use characters you made

=item * You can use your operating system, network, and database.

=item * In most cases, application programs can be used as it is.

=back

=head1 WHAT IS "UTF-8-SPUA-JP"?

=over 4

=item * "utf8jp" as mnemonic

=item * UTF-8-SPUA-JP is UTF-8

=item * Internal character encoding of Jacode4e and Jacode4e::RoundTrip, universally

=item * Implements JIS X 0213 character set on to Unicode Supplementary Private Use Area-A

=item * Code point ordered by JIS level, plane, row, cell

=item * Uniformly length encoding

=item * No grapheme clustering, one character by uniquely code point

=back

=head1 CP932 vs. CP932IBM

 A merit of the CP932IBM is that all of the extended characters are outside of
 the JIS row number 1 to 94. CP932IBM is useful when you transfer data to
 enterprise system.
 
 The following is the difference between CP932 and CP932IBM.
 
 #+++++++----------------------------------------------------------------------------------------------------------- CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
 #||||||| ++++------------------------------------------------------------------------------------------------------ Microsoft CP932, IANA Windows-31J
 #||||||| |||| ++++------------------------------------------------------------------------------------------------- IBM CP932
 #||||||| |||| |||| ++++-------------------------------------------------------------------------------------------- NEC CP932
 #||||||| |||| |||| |||| ++++--------------------------------------------------------------------------------------- JISC Shift_JIS-2004
 #||||||| |||| |||| |||| |||| ++++---------------------------------------------------------------------------------- IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
 #||||||| |||| |||| |||| |||| |||| ++++----------------------------------------------------------------------------- HITACHI KEIS78
 #||||||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------------ HITACHI KEIS83
 #||||||| |||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------- HITACHI KEIS90
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| ++++-------------------------------------------------------------- FUJITSU JEF
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++--------------------------------------------------------- NEC JIPS(J)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++---------------------------------------------------- NEC JIPS(E)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++----------------------------------------------- UNISYS LetsJ
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| +++++++++------------------------------------- Unicode
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| ++++++++++++------------------------ UTF-8.0 (aka UTF-8)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| ++++++++++++----------- UTF-8.1
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| |||||||||||| ++++++++-- UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
 #2345678 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 123456789 123456789012 123456789012 12345678
 #VVVVVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVVVVVVV VVVVVVVVVVVV VVVVVVVVVVVV VVVVVVVV
 8754     8754 FA4A 8754 8754 41F1 74B1 74B1 74B1 77DE 2D35 60F5  --  2160      E285A0       E285A0       F3B095BC
 8755     8755 FA4B 8755 8755 41F2 74B2 74B2 74B2 77DF 2D36 60F6  --  2161      E285A1       E285A1       F3B095BD
 8756     8756 FA4C 8756 8756 41F3 74B3 74B3 74B3 77E0 2D37 60F7  --  2162      E285A2       E285A2       F3B095BE
 8757     8757 FA4D 8757 8757 41F4 74B4 74B4 74B4 77E1 2D38 60F8  --  2163      E285A3       E285A3       F3B095BF
 8758     8758 FA4E 8758 8758 41F5 74B5 74B5 74B5 77E2 2D39 60F9  --  2164      E285A4       E285A4       F3B09680
 8759     8759 FA4F 8759 8759 41F6 74B6 74B6 74B6 77E3 2D3A 607A  --  2165      E285A5       E285A5       F3B09681
 875A     875A FA50 875A 875A 41F7 74B7 74B7 74B7 77E4 2D3B 605E  --  2166      E285A6       E285A6       F3B09682
 875B     875B FA51 875B 875B 41F8 74B8 74B8 74B8 77E5 2D3C 604C  --  2167      E285A7       E285A7       F3B09683
 875C     875C FA52 875C 875C 41F9 74B9 74B9 74B9 77E6 2D3D 607E  --  2168      E285A8       E285A8       F3B09684
 875D     875D FA53 875D 875D 41FA 74BA 74BA 74BA 77E7 2D3E 606E  --  2169      E285A9       E285A9       F3B09685
 8782     8782 FA59 8782 8782 446E 72BE 72BE 72BE 76FA 2D62 6059 A271 2116      E28496       E28496       F3B096A9
 8784     8784 FA5A 8784 8784 446F 72B8 72B8 72B8 76F6 2D64 6063  --  2121      E284A1       E284A1       F3B096AB
 878A     878A FA58 878A 878A 446D 73D0 73D0 73D0 78D5 2D6A 6069  --  3231      E388B1       E388B1       F3B096B1

=head1 CP932 vs. CP932NEC

 A merit of the CP932NEC is that all of the extended characters are inside the
 JIS row number 1 to 94. For example, CP932NEC is useful when you convert data
 to EUC-JP encoding.
 
 The following is the difference between CP932 and CP932NEC.
 
 #+++++++----------------------------------------------------------------------------------------------------------- CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
 #||||||| ++++------------------------------------------------------------------------------------------------------ Microsoft CP932, IANA Windows-31J
 #||||||| |||| ++++------------------------------------------------------------------------------------------------- IBM CP932
 #||||||| |||| |||| ++++-------------------------------------------------------------------------------------------- NEC CP932
 #||||||| |||| |||| |||| ++++--------------------------------------------------------------------------------------- JISC Shift_JIS-2004
 #||||||| |||| |||| |||| |||| ++++---------------------------------------------------------------------------------- IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
 #||||||| |||| |||| |||| |||| |||| ++++----------------------------------------------------------------------------- HITACHI KEIS78
 #||||||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------------ HITACHI KEIS83
 #||||||| |||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------- HITACHI KEIS90
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| ++++-------------------------------------------------------------- FUJITSU JEF
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++--------------------------------------------------------- NEC JIPS(J)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++---------------------------------------------------- NEC JIPS(E)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++----------------------------------------------- UNISYS LetsJ
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| +++++++++------------------------------------- Unicode
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| ++++++++++++------------------------ UTF-8.0 (aka UTF-8)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| ++++++++++++----------- UTF-8.1
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| |||||||||||| ++++++++-- UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
 #2345678 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 123456789 123456789012 123456789012 12345678
 #VVVVVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVVVVVVV VVVVVVVVVVVV VVVVVVVVVVVV VVVVVVVV
 FA40     FA40 FA40 EEEF 86B3 41B1 74C1 74C1 74C1 77EB A2BD 428E  --  2170      E285B0       E285B0       F3B2AB9C
 FA41     FA41 FA41 EEF0 86B4 41B2 74C2 74C2 74C2 77EC A2BE 428F  --  2171      E285B1       E285B1       F3B2AB9D
 FA42     FA42 FA42 EEF1 86B5 41B3 74C3 74C3 74C3 77ED A2BF 4290  --  2172      E285B2       E285B2       F3B2AB9E
 FA43     FA43 FA43 EEF2 86B6 41B4 74C4 74C4 74C4 77EE A2C0 4291  --  2173      E285B3       E285B3       F3B2AB9F
 FA44     FA44 FA44 EEF3 86B7 41B5 74C5 74C5 74C5 77EF A2C1 4292  --  2174      E285B4       E285B4       F3B2ABA0
 FA45     FA45 FA45 EEF4 86B8 41B6 74C6 74C6 74C6 77F0 A2C2 4293  --  2175      E285B5       E285B5       F3B2ABA1
 FA46     FA46 FA46 EEF5 86B9 41B7 74C7 74C7 74C7 77F1 A2C3 4294  --  2176      E285B6       E285B6       F3B2ABA2
 FA47     FA47 FA47 EEF6 86BA 41B8 74C8 74C8 74C8 77F2 A2C4 4295  --  2177      E285B7       E285B7       F3B2ABA3
 FA48     FA48 FA48 EEF7 86BB 41B9 74C9 74C9 74C9 77F3 A2C5 4296  --  2178      E285B8       E285B8       F3B2ABA4
 FA49     FA49 FA49 EEF8 86BC 41BA 74CA 74CA 74CA 77F4 A2C6 4297  --  2179      E285B9       E285B9       F3B2ABA5
 FA55     FA55 FA55 EEFA  --  E9F5  --   --   --  76A1 A5FB 45FB  --  FFE4      EFBFA4       EFBFA4       F3B2ABB1
 FA56     FA56 FA56 EEFB 81AD 427D  --   --   --  76A2 A3F9 43EF  --  FF07      EFBC87       EFBC87       F3B2ABB2
 FA57     FA57 FA57 EEFC 81AE 427F  --   --   --  76A3 A3FA 43FA  --  FF02      EFBC82       EFBC82       F3B2ABB3
 FA5C     FA5C FA5C ED40 EDB5 51BF 67FE 67FE 67FE 5FDC C9D8 9AAE D463 7E8A      E7BA8A       E7BA8A       F3B2ABB8
 FA5D     FA5D FA5D ED41 EE8E 51DA 69E8 69E8 69E8 64B5 CEA9 A349 DC5F 891C      E8A49C       E8A49C       F3B2ABB9
 FA5E     FA5E FA5E ED42 EF58 51F7 6CC7 6CC7 6CC7 6AD9 D2B9 A789 E469 9348      E98D88       E98D88       F3B2ABBA
 FA5F     FA5F FA5F ED43 EF4D 524B 6CA3 6CA3 6CA3 69FE D1ED A6CF E378 9288      E98A88       E98A88       F3B2ABBB
 FA60     FA60 FA60 ED44 F8D3 527B 69AE 69AE 69AE 62C4 CCCA 9F9D D921 84DC      E8939C       E8939C       F3B2ABBC
 FA61     FA61 FA61 ED45 87B7 5294 59E4 59E4 59E4 42C7 B2F6 82EC B13B 4FC9      E4BF89       E4BF89       F3B2ABBD
 FA62     FA62 FA62 ED46 F580 529E 63FE 63FE 63FE 70E1 C2FE 93FE  --  70BB      E782BB       E782BB       F3B2ABBE
 FA63     FA63 FA63 ED47 EB54 52EB 60F8 60F8 60F8 51A3 BDBF 8E90 C22D 6631      E698B1       E698B1       F3B2ABBF
 FA64     FA64 FA64 ED48 EB89 5386 61DC 61DC 61DC 70CD BFB7 9087 C37C 68C8      E6A388       E6A388       F3B2AC80
 FA65     FA65 FA65 ED49 FB42 5394 6CBB 6CBB 6CBB 6AC9 D2B6 A786 E450 92F9      E98BB9       E98BB9       F3B2AC81
 FA66     FA66 FA66 ED4A EB56 5397 60D3 60D3 60D3 51FC BDD0 8EA5 C23F 66FB      E69BBB       E69BBB       F3B2AC82
 FA67     FA67 FA67 ED4B EAB8 53C7 5ED8 5ED8 5ED8 4CF6 BAC7 8A98 BC74 5F45      E5BD85       E5BD85       F3B2AC83
 FA68     FA68 FA68 ED4C 87A2 5644 59AC 59AC 59AC 41AC B2AB 8252 B029 4E28      E4B8A8       E4B8A8       F3B2AC84
 FA69     FA69 FA69 ED4D 87A6 565D 59C1 59C1 59C1 41E8 B2C2 8293 B048 4EE1      E4BBA1       E4BBA1       F3B2AC85
 FA6A     FA6A FA6A ED4E  --  5662 59C8 59C8 59C8 43CF B2D3 82A8  --  4EFC      E4BBBC       E4BBBC       F3B2AC86
 FA6B     FA6B FA6B ED4F F05E 5665 59C7 59C7 59C7 41EF B2CD 82A2 B052 4F00      E4BC80       E4BC80       F3B2AC87
 FA6C     FA6C FA6C ED50 87A9 5666 9FA1 9FA1 9FA1 41F0 B2CE 82A3 B054 4F03      E4BC83       E4BC83       F3B2AC88
 FA6D     FA6D FA6D ED51  --  566A 59CE 59CE 59CE 41FB B2DB 82BB B063 4F39      E4BCB9       E4BCB9       F3B2AC89
 FA6E     FA6E FA6E ED52 87AE 566E 59CB 59CB 59CB 42A8 B2D6 82AC B06E 4F56      E4BD96       E4BD96       F3B2AC8A
 FA6F     FA6F FA6F ED53 F06C 5670 59DB 59DB 59DB 42B5 B2EA 82CC B127 4F92      E4BE92       E4BE92       F3B2AC8B
 FA70     FA70 FA70 ED54 87B3 5671 59E0 59E0 59E0 42B3 B2E7 82B9 B123 4F8A      E4BE8A       E4BE8A       F3B2AC8C
 FA71     FA71 FA71 ED55 F06D 5674 59DA 59DA 59DA 42B9 BADA 8ABA B12C 4F9A      E4BE9A       E4BE9A       F3B2AC8D
 FA72     FA72 FA72 ED56 87B4 567B 59D9 59D9 59D9 42B7 B2E1 82B3 B129 4F94      E4BE94       E4BE94       F3B2AC8E
 FA73     FA73 FA73 ED57  --  5686 59E6 59E6 59E6 42CA B2F5 82EB B13E 4FCD      E4BF8D       E4BF8D       F3B2AC8F
 FA74     FA74 FA74 ED58 87BF 568A 5AA1 5AA1 5AA1 42E7 B3B2 8382 B15F 5040      E58180       E58180       F3B2AC90
 FA75     FA75 FA75 ED59 F07C 5691 59F1 59F1 59F1 42DF B3A1 8341 B158 5022      E580A2       E580A2       F3B2AC91
 FA76     FA76 FA76 ED5A  --  5694 59FA 59FA 59FA 43D0 B3B3 8383 B148 4FFF      E4BFBF       E4BFBF       F3B2AC92
 FA77     FA77 FA77 ED5B F07B 5698 59F3 59F3 59F3 42DE B3A6 8346 B157 501E      E5809E       E5809E       F3B2AC93
 FA78     FA78 FA78 ED5C F080 56A2 5AA7 5AA7 5AA7 42E9 B3C0 8391 B163 5046      E58186       E58186       F3B2AC94
 FA79     FA79 FA79 ED5D F087 56A3 5AAC 5AAC 5AAC 42F5 B3C3 8394 B174 5070      E581B0       E581B0       F3B2AC95
 FA7A     FA7A FA7A ED5E F07E 56A4 5AA6 5AA6 5AA6 42E8 B3C4 8395 B161 5042      E58182       E58182       F3B2AC96
 FA7B     FA7B FA7B ED5F 87C2 56A8 5AB0 5AB0 5AB0 42FC B3C6 8397 B223 5094      E58294       E58294       F3B2AC97
 FA7C     FA7C FA7C ED60  --  56B2 5ABE 5ABE 5ABE 43B9 B3D2 83A7  --  50F4      E583B4       E583B4       F3B2AC98
 FA7D     FA7D FA7D ED61  --  56B5 5AC1 5AC1 5AC1 43B2 B3D5 83AA B23B 50D8      E58398       E58398       F3B2AC99
 FA7E     FA7E FA7E ED62 87CD 56C9 5ACB 5ACB 5ACB 43D3 B3E8 83CA B266 514A      E5858A       E5858A       F3B2AC9A
 FA80     FA80 FA80 ED63  --  56CE 5AD3 5AD3 5AD3 43D9 B3F0 83DC B26D 5164      E585A4       E585A4       F3B2AC9B
 FA81     FA81 FA81 ED64 87D2 56DC 5AE1 5AE1 5AE1 43F2 B3F7 83ED B275 519D      E5869D       E5869D       F3B2AC9C
 FA82     FA82 FA82 ED65  --  56E8 5AE4 5AE4 5AE4 43F7 B4A2 8442 B27C 51BE      E586BE       E586BE       F3B2AC9D
 FA83     FA83 FA83 ED66  --  56EF 6DC3 6DC3 6DC3 44AC D4A8 A948  --  51EC      E587AC       E587AC       F3B2AC9E
 FA84     FA84 FA84 ED67 87DB 56F7 9FA3 9FA3 9FA3 44B4 B4B4 8484 B335 5215      E58895       E58895       F3B2AC9F
 FA85     FA85 FA85 ED68  --  574F 5AF0 5AF0 5AF0 44F7 B4C1 8492 B358 529C      E58A9C       E58A9C       F3B2ACA0
 FA86     FA86 FA86 ED69 F162 5750 5AF1 5AF1 5AF1 44D2 B4C3 8494 B35B 52A6      E58AA6       E58AA6       F3B2ACA1
 FA87     FA87 FA87 ED6A  --  5756 5AF7 5AF7 5AF7 44FA B4D0 84A5 B365 52C0      E58B80       E58B80       F3B2ACA2
 FA88     FA88 FA88 ED6B 87E5 5758 5BA1 5BA1 5BA1 44ED B4D8 84AE B36E 52DB      E58B9B       E58B9B       F3B2ACA3
 FA89     FA89 FA89 ED6C 87E9 5761 5BA9 5BA9 5BA9 45A3 B4E8 84CA B37B 5300      E58C80       E58C80       F3B2ACA4
 FA8A     FA8A FA8A ED6D 87EA 5763 5BAB 5BAB 5BAB 45A8 B4EC 84CE  --  5307      E58C87       E58C87       F3B2ACA5
 FA8B     FA8B FA8B ED6E F16D 576B 5BAF 5BAF 5BAF 45B3 B4EF 84DB  --  5324      E58CA4       E58CA4       F3B2ACA6
 FA8C     FA8C FA8C ED6F  --  577C 5BBC 5BBC 5BBC 45D2 B5A7 8547 B440 5372      E58DB2       E58DB2       F3B2ACA7
 FA8D     FA8D FA8D ED70 87F0 5780 5BC4 5BC4 5BC4 45DB B5AD 8554 B447 5393      E58E93       E58E93       F3B2ACA8
 FA8E     FA8E FA8E ED71 87F2 5787 5BC9 5BC9 5BC9 45E2 B5B1 8581 B450 53B2      E58EB2       E58EB2       F3B2ACA9
 FA8F     FA8F FA8F ED72  --  578B 5BCE 5BCE 5BCE 45F3 B5BC 858D B45E 53DD      E58F9D       E58F9D       F3B2ACAA
 FA90     FA90 FA90 ED73  --  578E 5BD3 5BD3 5BD3 45F2 B5BE 858F  --  FA0E      EFA88E       EFA88E       F3B2ACAB
 FA91     FA91 FA91 ED74 87F6 57A9 5BDB 5BDB 5BDB 46C4 B5D3 85A8 B52A 549C      E5929C       E5929C       F3B2ACAC
 FA92     FA92 FA92 ED75 F18C 57AE 5BDA 5BDA 5BDA 46C1 B5CF 85A4  --  548A      E5928A       E5928A       F3B2ACAD
 FA93     FA93 FA93 ED76 8840 57BD 5BE1 5BE1 5BE1 46C9 B5DC 85BC B52F 54A9      E592A9       E592A9       F3B2ACAE
 FA94     FA94 FA94 ED77 8842 57C0 5BE3 5BE3 5BE3 45FA B5E0 85B2 B544 54FF      E593BF       E593BF       F3B2ACAF
 FA95     FA95 FA95 ED78 8849 57DE 5BED 5BED 5BED 45F6 B5EE 85DA B568 5586      E59686       E59686       F3B2ACB0
 FA96     FA96 FA96 ED79  --  5860 5CB2 5CB2 5CB2 48C9 B6D6 86AC  --  5759      E59D99       E59D99       F3B2ACB1
 FA97     FA97 FA97 ED7A  --  5863 5CB6 5CB6 5CB6 48CA B6DE 86BE B742 5765      E59DA5       E59DA5       F3B2ACB2
 FA98     FA98 FA98 ED7B F1EA 586C 5CB9 5CB9 5CB9 48CB B6E3 86B5 B764 57AC      E59EAC       E59EAC       F3B2ACB3
 FA99     FA99 FA99 ED7C 886E 5870 5CBD 5CBD 5CBD 47F7 B6E5 86B7 B768 57C8      E59F88       E59F88       F3B2ACB4
 FA9A     FA9A FA9A ED7D 886D 5873 9FA6 9FA6 9FA6 47F6 B6E6 86B8 B767 57C7      E59F87       E59F87       F3B2ACB5
 FA9B     FA9B FA9B ED7E 886A 5877 5CBB 5CBB 5CBB 48CC B6E7 86B9  --  FA0F      EFA88F       EFA88F       F3B2ACB6
 FA9C     FA9C FA9C ED80 8876 5882 C4CD 5CC7 5CC7 C4CD 444D C4D4  --  FA10      EFA890       EFA890       F3B2ACB7
 FA9D     FA9D FA9D ED81 887C 588A 5CD1 5CD1 5CD1 48BB B7A8 8748  --  589E      E5A29E       E5A29E       F3B2ACB8
 FA9E     FA9E FA9E ED82  --  588C 5CD3 5CD3 5CD3 48CD B7AE 8755 B84E 58B2      E5A2B2       E5A2B2       F3B2ACB9
 FA9F     FA9F FA9F ED83 F259 58A1 5BCD 5BCD 5BCD 48D0 B7B8 8788 B861 590B      E5A48B       E5A48B       F3B2ACBA
 FAA0     FAA0 FAA0 ED84  --  58AD 5CE4 5CE4 5CE4 48E3 B7CB 879E B875 5953      E5A593       E5A593       F3B2ACBB
 FAA1     FAA1 FAA1 ED85 F264 58B0 5CE6 5CE6 5CE6 48E4 B7CF 87A4 B877 595B      E5A59B       E5A59B       F3B2ACBC
 FAA2     FAA2 FAA2 ED86 8889 58B1 5CE8 5CE8 5CE8 48E5 B7CE 87A3 B878 595D      E5A59D       E5A59D       F3B2ACBD
 FAA3     FAA3 FAA3 ED87  --  58B2 5CE9 5CE9 5CE9 48EB B7D3 87A8 B87C 5963      E5A5A3       E5A5A3       F3B2ACBE
 FAA4     FAA4 FAA4 ED88 888D 58BE 5CF4 5CF4 5CF4 49A3 B7DB 87BB B92F 59A4      E5A6A4       E5A6A4       F3B2ACBF
 FAA5     FAA5 FAA5 ED89  --  58C0 5CF7 5CF7 5CF7 49F0 B7E4 87B6 B937 59BA      E5A6BA       E5A6BA       F3B2AD80
 FAA6     FAA6 FAA6 ED8A 9875 58EC 5DB3 5DB3 5DB3 49F2 B8B4 8884 BA3E 5B56      E5AD96       E5AD96       F3B2AD81
 FAA7     FAA7 FAA7 ED8B F0A2 58FC 5DC2 5DC2 5DC2 4AC9 B8CA 889D BA5B 5BC0      E5AF80       E5AF80       F3B2AD82
 FAA8     FAA8 FAA8 ED8C ECC7 58FE 5DC5 5DC5 5DC5 4ABD B8CC 889F CD2A 752F      E794AF       E794AF       F3B2AD83
 FAA9     FAA9 FAA9 ED8D 9878 5942 5DC6 5DC6 5DC6 4AC0 B8CD 88A2 BA61 5BD8      E5AF98       E5AF98       F3B2AD84
 FAAA     FAAA FAAA ED8E 9879 5944 5DC7 5DC7 5DC7 4AC6 B8D4 88A9  --  5BEC      E5AFAC       E5AFAC       F3B2AD85
 FAAB     FAAB FAAB ED8F 987B 5951 5DD2 5DD2 5DD2 4AD4 B8E3 88B5 BA6B 5C1E      E5B09E       E5B09E       F3B2AD86
 FAAC     FAAC FAAC ED90 F0C1 5967 5DDE 5DDE 5DDE 4AF1 B8FB 88FB BB33 5CA6      E5B2A6       E5B2A6       F3B2AD87
 FAAD     FAAD FAAD ED91 9889 5968 5DDD 5DDD 5DDD 4AF3 B8F9 88EF BB38 5CBA      E5B2BA       E5B2BA       F3B2AD88
 FAAE     FAAE FAAE ED92  --  5970 5DE6 5DE6 5DE6 4BE4 B9A9 8949  --  5CF5      E5B3B5       E5B3B5       F3B2AD89
 FAAF     FAAF FAAF ED93 9891 5978 5DE8 5DE8 5DE8 4AFB B9B1 8981 BB4A 5D27      E5B4A7       E5B4A7       F3B2AD8A
 FAB0     FAB0 FAB0 ED94 9895 597F 5DF0 5DF0 5DF0 4BA3 C6B7 9787  --  5D53      E5B593       E5B593       F3B2AD8B
 FAB1     FAB1 FAB1 ED95 9892 5980 5DEC 5DEC 5DEC 4BD0 B9BA 898A  --  FA11      EFA891       EFA891       F3B2AD8C
 FAB2     FAB2 FAB2 ED96 F0D3 5984 5DEF 5DEF 5DEF 4BA2 B9BD 898E BB50 5D42      E5B582       E5B582       F3B2AD8D
 FAB3     FAB3 FAB3 ED97 9897 5987 9FA9 9FA9 9FA9 4BD3 B9C0 8991 BB5E 5D6D      E5B5AD       E5B5AD       F3B2AD8E
 FAB4     FAB4 FAB4 ED98 989C 5993 5EA3 5EA3 5EA3 4BDF B9D3 89A8 BB74 5DB8      E5B6B8       E5B6B8       F3B2AD8F
 FAB5     FAB5 FAB5 ED99 F0E5 5995 5EA5 5EA5 5EA5 4BE0 B9D5 89AA BB75 5DB9      E5B6B9       E5B6B9       F3B2AD90
 FAB6     FAB6 FAB6 ED9A  --  5997 5EA7 5EA7 5EA7 4BE5 B9DA 89BA BB79 5DD0      E5B790       E5B790       F3B2AD91
 FAB7     FAB7 FAB7 ED9B  --  59C7 5ED2 5ED2 5ED2 4CEB BABF 8A90 BC64 5F21      E5BCA1       E5BCA1       F3B2AD92
 FAB8     FAB8 FAB8 ED9C EAB4 59CB 5ED3 5ED3 5ED3 4CEE BAC1 8A92 BC6D 5F34      E5BCB4       E5BCB4       F3B2AD93
 FAB9     FAB9 FAB9 ED9D EABC 59D8 5FCA 5FCA 5FCA 4DA2 BAD1 8AA6 BC7E 5F67      E5BDA7       E5BDA7       F3B2AD94
 FABA     FABA FABA ED9E EAC3 59E8 5EE6 5EE6 5EE6 4DC0 BAEC 8ACE  --  5FB7      E5BEB7       E5BEB7       F3B2AD95
 FABB     FABB FABB ED9F EAC5 59F1 5EEE 5EEE 5EEE 4DC7 BAFE 8AFE BD42 5FDE      E5BF9E       E5BF9E       F3B2AD96
 FABC     FABC FABC EDA0  --  5A44 5EF8 5EF8 5EF8 4DCC BBA9 8C49 BD67 605D      E6819D       E6819D       F3B2AD97
 FABD     FABD FABD EDA1  --  5A51 5EF9 5EF9 5EF9 4EA8 BBB1 8C81  --  6085      E68285       E68285       F3B2AD98
 FABE     FABE FABE EDA2 F2CB 5A57 5EF6 5EF6 5EF6 4DD0 B5E5 85B7 BD70 608A      E6828A       E6828A       F3B2AD99
 FABF     FABF FABF EDA3  --  5A5F 5FA5 5FA5 5FA5 4EC1 BBC4 8C95 BE30 60DE      E6839E       E6839E       F3B2AD9A
 FAC0     FAC0 FAC0 EDA4 EAD3 5A65 5FA3 5FA3 5FA3 4EBD BBBD 8C8E BE2C 60D5      E68395       E68395       F3B2AD9B
 FAC1     FAC1 FAC1 EDA5  --  5A67 5FAA 5FAA 5FAA 4EED BBC9 8C9A  --  6120      E684A0       E684A0       F3B2AD9C
 FAC2     FAC2 FAC2 EDA6 F2D6 5A68 5FAD 5FAD 5FAD 4EC2 BBCE 8CA3 BE33 60F2      E683B2       E683B2       F3B2AD9D
 FAC3     FAC3 FAC3 EDA7  --  5A76 5FAF 5FAF 5FAF 4EEE BBD4 8CA9 BE3D 6111      E68491       E68491       F3B2AD9E
 FAC4     FAC4 FAC4 EDA8 EAD9 5A79 5FB1 5FB1 5FB1 4ED0 BBD6 8CAC BE4D 6137      E684B7       E684B7       F3B2AD9F
 FAC5     FAC5 FAC5 EDA9 EAD8 5A7E 5FB3 5FB3 5FB3 4ECE BBD8 8CAE BE49 6130      E684B0       E684B0       F3B2ADA0
 FAC6     FAC6 FAC6 EDAA F2E7 5A94 5FBE 5FBE 5FBE 4EDB B6A8 8648 BE64 6198      E68698       E68698       F3B2ADA1
 FAC7     FAC7 FAC7 EDAB  --  5AB5 5BD7 5BD7 5BD7 4EF7 BBF9 8CEF BF28 6213      E68893       E68893       F3B2ADA2
 FAC8     FAC8 FAC8 EDAC F341 5ADA 5FD6 5FD6 5FD6 4FB5 BCB1 8D81 BF49 62A6      E68AA6       E68AA6       F3B2ADA3
 FAC9     FAC9 FAC9 EDAD EAF2 5AFB 5FE2 5FE2 5FE2 4FED BCCC 8D9F C044 63F5      E68FB5       E68FB5       F3B2ADA4
 FACA     FACA FACA EDAE EAF8 5B50 5FEB 5FEB 5FEB 4FFA BCD9 8DAF C064 6460      E691A0       E691A0       F3B2ADA5
 FACB     FACB FACB EDAF F36F 5B53 5FEE 5FEE 5FEE 50A2 BCDB 8DBB C074 649D      E6929D       E6929D       F3B2ADA6
 FACC     FACC FACC EDB0 EAFA 5B54 5FF2 5FF2 5FF2 4FA7 BCDC 8DBC C126 64CE      E6938E       E6938E       F3B2ADA7
 FACD     FACD FACD EDB1  --  5B7E 5FF8 5FF8 5FF8 50C9 BCF0 8DDC  --  654E      E6958E       E6958E       F3B2ADA8
 FACE     FACE FACE EDB2 EB4B 5B97 60B3 60B3 60B3 51BD BDBD 8E8E C175 6600      E69880       E69880       F3B2ADA9
 FACF     FACF FACF EDB3 EB4D 5B98 60B1 60B1 60B1 51C4 BDB5 8E85 C17C 6615      E69895       E69895       F3B2ADAA
 FAD0     FAD0 FAD0 EDB4  --  4F9F B9B7 60B6 60B6 B9B7 3937 F9F7  --  663B      E698BB       E698BB       F3B2ADAB
 FAD1     FAD1 FAD1 EDB5 EB4C 5B9D 60AF 60AF 60AF 51C0 BDB7 8E87 C178 6609      E69889       E69889       F3B2ADAC
 FAD2     FAD2 FAD2 EDB6  --  5BA0 60BE 60BE 60BE 51A1 BDC2 8E93 C22B 662E      E698AE       E698AE       F3B2ADAD
 FAD3     FAD3 FAD3 EDB7 EB4E 5BA1 60C4 60C4 60C4 51C7 BDC4 8E95 C221 661E      E6989E       E6989E       F3B2ADAE
 FAD4     FAD4 FAD4 EDB8 EB51 5BA4 60B9 60B9 60B9 51CA BDC7 8E98 C225 6624      E698A4       E698A4       F3B2ADAF
 FAD5     FAD5 FAD5 EDB9 EB5D 5BA7 60D1 60D1 60D1 51D9 BDD6 8EAC  --  6665      E699A5       E699A5       F3B2ADB0
 FAD6     FAD6 FAD6 EDBA F39E 5BA8 60D6 60D6 60D6 52A1 BDE0 8EB2 C238 6657      E69997       E69997       F3B2ADB1
 FAD7     FAD7 FAD7 EDBB EB5A 5BAB 60D7 60D7 60D7 51D4 BDDE 8EBE C23A 6659      E69999       E69999       F3B2ADB2
 FAD8     FAD8 FAD8 EDBC  --  5BAE 60DD 60DD 60DD 51DE BDE4 8EB6  --  FA12      EFA892       EFA892       F3B2ADB3
 FAD9     FAD9 FAD9 EDBD EB5E 5BAF 60E1 60E1 60E1 51B3 BDE6 8EB8 C244 6673      E699B3       E699B3       F3B2ADB4
 FADA     FADA FADA EDBE F3AB 5BB4 60E6 60E6 60E6 51E5 BDF7 8EED C252 6699      E69A99       E69A99       F3B2ADB5
 FADB     FADB FADB EDBF EB63 5BB6 60EA 60EA 60EA 51AB C5D8 96AE C257 66A0      E69AA0       E69AA0       F3B2ADB6
 FADC     FADC FADC EDC0 EB64 5BB8 60EE 60EE 60EE 51EB BEA5 8F45 C25B 66B2      E69AB2       E69AB2       F3B2ADB7
 FADD     FADD FADD EDC1 F3B1 5BBB 60EF 60EF 60EF 51F1 BEAC 8F53 C25E 66BF      E69ABF       E69ABF       F3B2ADB8
 FADE     FADE FADE EDC2 EB6B 5BC7 60C7 60C7 60C7 52A2 BDD1 8EA6 C26D 66FA      E69BBA       E69BBA       F3B2ADB9
 FADF     FADF FADF EDC3  --  5BC8 60FC 60FC 60FC 52BC BEBC 8F8D C270 670E      E69C8E       E69C8E       F3B2ADBA
 FAE0     FAE0 FAE0 EDC4 EB6D 5BC9 61A7 61A7 61A7 52AD BEBE 8F8F  --  F929      EFA4A9       EFA4A9       F3B2ADBB
 FAE1     FAE1 FAE1 EDC5 EB6F 5BD8 61B2 61B2 61B2 70CE BEC8 8F99 C321 6766      E69DA6       E69DA6       F3B2ADBC
 FAE2     FAE2 FAE2 EDC6 EB77 5BE3 61BB 61BB 61BB 53B2 BED9 8FAF C335 67BB      E69EBB       E69EBB       F3B2ADBD
 FAE3     FAE3 FAE3 EDC7 EB7D 5BEC 61BE 61BE 61BE 52EE BEE5 8FB7 C349 6852      E6A192       E6A192       F3B2ADBE
 FAE4     FAE4 FAE4 EDC8 EB79 5BF0 9FAC 9FAC 9FAC 53B4 BEE8 8FCA C339 67C0      E69F80       E69F80       F3B2ADBF
 FAE5     FAE5 FAE5 EDC9 EB7C 5BF3 61CC 61CC 61CC 53C0 BEE2 8FB4  --  6801      E6A081       E6A081       F3B2AE80
 FAE6     FAE6 FAE6 EDCA F3DD 5BF9 61C6 61C6 61C6 53C7 BEF0 8FDC C358 6844      E6A184       E6A184       F3B2AE81
 FAE7     FAE7 FAE7 EDCB F3FB 5C5F 61DF 61DF 61DF 70CF BFBA 908A C37E 68CF      E6A38F       E6A38F       F3B2AE82
 FAE8     FAE8 FAE8 EDCC F3F7 5C66 61E0 61E0 61E0 70D0 BFBC 908D  --  FA13      EFA893       EFA893       F3B2AE83
 FAE9     FAE9 FAE9 EDCD EB98 5C6E 61E9 61E9 61E9 53ED BFC0 9091 C44C 6968      E6A5A8       E6A5A8       F3B2AE84
 FAEA     FAEA FAEA EDCE EB9A 5C78 61F8 61F8 61F8 70D1 BFD7 90AD  --  FA14      EFA894       EFA894       F3B2AE85
 FAEB     FAEB FAEB EDCF F455 5C79 65F4 65F4 65F4 52F4 C6AA 9751 C459 6998      E6A698       E6A698       F3B2AE86
 FAEC     FAEC FAEC EDD0 EBA0 5C90 61EF 61EF 61EF 70D2 BFE4 90B6 C46A 69E2      E6A7A2       E6A7A2       F3B2AE87
 FAED     FAED FAED EDD1 F460 5C9A 62A3 62A3 62A3 70D3 BFE5 90B7 C47D 6A30      E6A8B0       E6A8B0       F3B2AE88
 FAEE     FAEE FAEE EDD2 EBAE 5C9B 62A9 62A9 62A9 54AD BFE6 90B8  --  6A6B      E6A9AB       E6A9AB       F3B2AE89
 FAEF     FAEF FAEF EDD3 F472 5CA6 62AE 62AE 62AE 52E8 BFEC 90CE C527 6A46      E6A986       E6A986       F3B2AE8A
 FAF0     FAF0 FAF0 EDD4 EBAF 5CA9 62B2 62B2 62B2 54AF BFF0 90DC C535 6A73      E6A9B3       E6A9B3       F3B2AE8B
 FAF1     FAF1 FAF1 EDD5 F479 5CAF 62B3 62B3 62B3 54B0 BFF8 90EE C536 6A7E      E6A9BE       E6A9BE       F3B2AE8C
 FAF2     FAF2 FAF2 EDD6  --  5CB9 62B9 62B9 62B9 54BC C0A8 9148  --  6AE2      E6ABA2       E6ABA2       F3B2AE8D
 FAF3     FAF3 FAF3 EDD7 EBB6 5CBC 62B8 62B8 62B8 54BD C0A5 9145 C555 6AE4      E6ABA4       E6ABA4       F3B2AE8E
 FAF4     FAF4 FAF4 EDD8 EBC9 5CEA 62D0 62D0 62D0 54F8 C0CF 91A4 C638 6BD6      E6AF96       E6AF96       F3B2AE8F
 FAF5     FAF5 FAF5 EDD9 F4B3 5CF6 62D5 62D5 62D5 55B5 C0D9 91AF C657 6C3F      E6B0BF       E6B0BF       F3B2AE90
 FAF6     FAF6 FAF6 EDDA EBD0 5CF9 62D6 62D6 62D6 55BD C0DE 91BE C660 6C5C      E6B19C       E6B19C       F3B2AE91
 FAF7     FAF7 FAF7 EDDB EBD5 5CFD 62D8 62D8 62D8 55C3 C0E5 91B7 C66A 6C86      E6B286       E6B286       F3B2AE92
 FAF8     FAF8 FAF8 EDDC  --  5CFE 62D9 62D9 62D9 55BF C0EB 91CD C663 6C6F      E6B1AF       E6B1AF       F3B2AE93
 FAF9     FAF9 FAF9 EDDD F4C2 5D4E 62E2 62E2 62E2 55D3 C0F4 91EA C721 6CDA      E6B39A       E6B39A       F3B2AE94
 FAFA     FAFA FAFA EDDE EBDF 5D58 62E4 62E4 62E4 55DE C1A7 9247 C72B 6D04      E6B484       E6B484       F3B2AE95
 FAFB     FAFB FAFB EDDF EBE9 5D60 62EB 62EB 62EB 56A3 C1B6 9286 C747 6D87      E6B687       E6B687       F3B2AE96
 FAFC     FAFC FAFC EDE0  --  5D62 62F2 62F2 62F2 70DE C1C7 9298 C743 6D6F      E6B5AF       E6B5AF       F3B2AE97
 FB40     FB40 FB40 EDE1  --  5D6B 62EE 62EE 62EE 56A7 C1BF 9290 C74B 6D96      E6B696       E6B696       F3B2AE98
 FB41     FB41 FB41 EDE2 EBED 5D72 62FC 62FC 62FC 56AD C1CA 929D C74F 6DAC      E6B6AC       E6B6AC       F3B2AE99
 FB42     FB42 FB42 EDE3 F4E6 5D74 63A4 63A4 63A4 56B2 C1D9 92AF C759 6DCF      E6B78F       E6B78F       F3B2AE9A
 FB43     FB43 FB43 EDE4  --  5D7A 62FD 62FD 62FD 56B8 C1CD 92A2  --  6DF8      E6B7B8       E6B7B8       F3B2AE9B
 FB44     FB44 FB44 EDE5  --  5D7F 63A6 63A6 63A6 70DF C1DF 92BF  --  6DF2      E6B7B2       E6B7B2       F3B2AE9C
 FB45     FB45 FB45 EDE6 EBF4 5D80 63BD 63BD 63BD 55B1 C1D1 92A6 C766 6DFC      E6B7BC       E6B7BC       F3B2AE9D
 FB46     FB46 FB46 EDE7  --  5D8A 63B7 63B7 63B7 56C2 C1FB 92FB C76E 6E39      E6B8B9       E6B8B9       F3B2AE9E
 FB47     FB47 FB47 EDE8 EBF8 5D8E 63B4 63B4 63B4 56CF C1F3 92DF C77C 6E5C      E6B99C       E6B99C       F3B2AE9F
 FB48     FB48 FB48 EDE9 F4EE 5D93 63AE 63AE 63AE 56BE C1F4 92EA C76B 6E27      E6B8A7       E6B8A7       F3B2AEA0
 FB49     FB49 FB49 EDEA F4F1 5D95 63B8 63B8 63B8 56C3 C1F0 92DC C770 6E3C      E6B8BC       E6B8BC       F3B2AEA1
 FB4A     FB4A FB4A EDEB F543 5DA6 9FB6 9FB6 9FB6 56DF C2A8 9348 C831 6EBF      E6BABF       E6BABF       F3B2AEA2
 FB4B     FB4B FB4B EDEC EC4E 5DC1 63D8 63D8 63D8 56FE C2CE 93A3 C865 6F88      E6BE88       E6BE88       F3B2AEA3
 FB4C     FB4C FB4C EDED F560 5DCA 63DF 63DF 63DF 57AE C2DF 93BF C878 6FB5      E6BEB5       E6BEB5       F3B2AEA4
 FB4D     FB4D FB4D EDEE EC5A 5DD5 63E4 63E4 63E4 57B8 C2E4 93B6 C926 6FF5      E6BFB5       E6BFB5       F3B2AEA5
 FB4E     FB4E FB4E EDEF EC5B 5DD7 63EB 63EB 63EB 57B9 C2EF 93DB C92B 7005      E78085       E78085       F3B2AEA6
 FB4F     FB4F FB4F EDF0 F569 5DD8 63E8 63E8 63E8 57BA C2EE 93DA C92D 7007      E78087       E78087       F3B2AEA7
 FB50     FB50 FB50 EDF1 EC5D 5DE3 63F1 63F1 63F1 57BC C2F1 93DD  --  7028      E780A8       E780A8       F3B2AEA8
 FB51     FB51 FB51 EDF2 EC64 5DEA 63F8 63F8 63F8 57D1 C2FB 93FB C94A 7085      E78285       E78285       F3B2AEA9
 FB52     FB52 FB52 EDF3 EC66 5DED 63FD 63FD 63FD 57DE C2FC 93FC C953 70AB      E782AB       E782AB       F3B2AEAA
 FB53     FB53 FB53 EDF4 EC6B 5DF5 9FB9 9FB9 9FB9 58A6 C3A8 9448 C969 710F      E7848F       E7848F       F3B2AEAB
 FB54     FB54 FB54 EDF5 F58B 5DF6 64A5 64A5 64A5 58A5 C3A6 9446 C963 7104      E78484       E78484       F3B2AEAC
 FB55     FB55 FB55 EDF6 EC76 5DFB 64AD 64AD 64AD 57F1 C3B8 9488 C97C 715C      E7859C       E7859C       F3B2AEAD
 FB56     FB56 FB56 EDF7 EC71 5DFC 64AE 64AE 64AE 57EB C3B9 9489 C974 7146      E78586       E78586       F3B2AEAE
 FB57     FB57 FB57 EDF8 EC72 5DFD 64AF 64AF 64AF 57EC C3BC 948D C975 7147      E78587       E78587       F3B2AEAF
 FB58     FB58 FB58 EDF9 EC79 5E49 64B9 64B9 64B9 70E3 C3CE 94A3  --  FA15      EFA895       EFA895       F3B2AEB0
 FB59     FB59 FB59 EDFA EC7D 5E4B 64B4 64B4 64B4 57F6 C3D0 94A5 CA33 71C1      E78781       E78781       F3B2AEB1
 FB5A     FB5A FB5A EDFB EC81 5E5A 64C2 64C2 64C2 58A9 C3D5 94AA CA3D 71FE      E787BE       E787BE       F3B2AEB2
 FB5B     FB5B FB5B EDFC F5BB 5E73 64CC 64CC 64CC 58FA C3EB 94CD CA6F 72B1      E78AB1       E78AB1       F3B2AEB3
 FB5C     FB5C FB5C EE40 EC89 5E75 9FBA 9FBA 9FBA 58D6 C3EE 94DA CA71 72BE      E78ABE       E78ABE       F3B2AEB4
 FB5D     FB5D FB5D EE41  --  5E86 64D0 64D0 64D0 58ED C3FB 94FB CB2E 7324      E78CA4       E78CA4       F3B2AEB5
 FB5E     FB5E FB5E EE42 EC8F 5E8A 64CE 64CE 64CE 58EF C3F8 94EE  --  FA16      EFA896       EFA896       F3B2AEB6
 FB5F     FB5F FB5F EE43 F5D5 5E94 64D2 64D2 64D2 58F6 C4A5 9545 CB4A 7377      E78DB7       E78DB7       F3B2AEB7
 FB60     FB60 FB60 EE44  --  5E99 64DF 64DF 64DF 59BA C4BA 958A CB66 73BD      E78EBD       E78EBD       F3B2AEB8
 FB61     FB61 FB61 EE45 EC99 5E9E 64DB 64DB 64DB 59BC C4B8 9588 CB6A 73C9      E78F89       E78F89       F3B2AEB9
 FB62     FB62 FB62 EE46 EC9B 5E9F 64E4 64E4 64E4 59C0 C4C1 9592 CB70 73D6      E78F96       E78F96       F3B2AEBA
 FB63     FB63 FB63 EE47 EC9D 5EA1 64E5 64E5 64E5 59C3 C4C2 9593 CB74 73E3      E78FA3       E78FA3       F3B2AEBB
 FB64     FB64 FB64 EE48  --  5EA2 64E8 64E8 64E8 5AAC C4C6 9597 CB6E 73D2      E78F92       E78F92       F3B2AEBC
 FB65     FB65 FB65 EE49 EC9F 5EA7 64EB 64EB 64EB 59D0 C4C7 9598 CC25 7407      E79087       E79087       F3B2AEBD
 FB66     FB66 FB66 EE4A  --  5EA8 64EC 64EC 64EC 59C7 C4CB 959E CB79 73F5      E78FB5       E78FB5       F3B2AEBE
 FB67     FB67 FB67 EE4B ECA4 5EAA 64F5 64F5 64F5 59D7 C4D2 95A7 CC2B 7426      E790A6       E790A6       F3B2AEBF
 FB68     FB68 FB68 EE4C ECA6 5EAB 64F6 64F6 64F6 59DA C4D3 95A8 CC2E 742A      E790AA       E790AA       F3B2AF80
 FB69     FB69 FB69 EE4D  --  5EAD 64FC 64FC 64FC 59D9 C4DB 95BB CC2D 7429      E790A9       E790A9       F3B2AF81
 FB6A     FB6A FB6A EE4E ECA9 5EAE 64F8 64F8 64F8 59DD C4D7 95AD CC32 742E      E790AE       E790AE       F3B2AF82
 FB6B     FB6B FB6B EE4F ECB1 5EBC 65A8 65A8 65A8 5AAD C4ED 95CF CC42 7462      E791A2       E791A2       F3B2AF83
 FB6C     FB6C FB6C EE50 ECB6 5EBD 65AE 65AE 65AE 59F4 C4F3 95DF CC50 7489      E79289       E79289       F3B2AF84
 FB6D     FB6D FB6D EE51 ECB9 5EC0 65B2 65B2 65B2 59FB C4FB 95FB CC59 749F      E7929F       E7929F       F3B2AF85
 FB6E     FB6E FB6E EE52 ECC5 5ED1 65B9 65B9 65B9 5AB8 C5AC 9653  --  7501      E79481       E79481       F3B2AF86
 FB6F     FB6F FB6F EE53 ECC8 5EE7 65C3 65C3 65C3 5AD0 C5B4 9684 CD3B 756F      E795AF       E795AF       F3B2AF87
 FB70     FB70 FB70 EE54  --  5F69 65CA 65CA 65CA 5AFE C5D3 96A8  --  7682      E79A82       E79A82       F3B2AF88
 FB71     FB71 FB71 EE55 F68F 5F70 65CE 65CE 65CE 5BA4 C5D9 96AF CE3B 769C      E79A9C       E79A9C       F3B2AF89
 FB72     FB72 FB72 EE56 ECE1 5F71 65CF 65CF 65CF 5BA5 C5DA 96BA  --  769E      E79A9E       E79A9E       F3B2AF8A
 FB73     FB73 FB73 EE57 ECDF 5F72 65D1 65D1 65D1 5AFD C5DB 96BB CE3A 769B      E79A9B       E79A9B       F3B2AF8B
 FB74     FB74 FB74 EE58 ECE2 5F73 65D2 65D2 65D2 5BA9 C5DF 96BF CE43 76A6      E79AA6       E79AA6       F3B2AF8C
 FB75     FB75 FB75 EE59  --  5F7A 65D8 65D8 65D8 5BB7 C5E5 96B7  --  FA17      EFA897       EFA897       F3B2AF8D
 FB76     FB76 FB76 EE5A ECF0 5F8E 65E6 65E6 65E6 5BCC C5F2 96DE CE72 7746      E79D86       E79D86       F3B2AF8E
 FB77     FB77 FB77 EE5B F163 5FAA 65F6 65F6 65F6 44DC B4C7 8498 B35D 52AF      E58AAF       E58AAF       F3B2AF8F
 FB78     FB78 FB78 EE5C F6C1 5FAE 65F7 65F7 65F7 5BFC C6B1 9781 CF55 7821      E7A0A1       E7A0A1       F3B2AF90
 FB79     FB79 FB79 EE5D ED41 5FB2 65FA 65FA 65FA 5CA7 C6B5 9785 CF62 784E      E7A18E       E7A18E       F3B2AF91
 FB7A     FB7A FB7A EE5E F6C4 5FB3 66A1 66A1 66A1 5CAE C6BD 978E CF69 7864      E7A1A4       E7A1A4       F3B2AF92
 FB7B     FB7B FB7B EE5F  --  5FB6 66A5 66A5 66A5 5CB2 C6C4 9795 CF6D 787A      E7A1BA       E7A1BA       F3B2AF93
 FB7C     FB7C FB7C EE60 F6DC 5FD1 66AE 66AE 66AE 5CCE C6DA 97BA  --  7930      E7A4B0       E7A4B0       F3B2AF94
 FB7D     FB7D FB7D EE61  --  5FD2 66B0 66B0 66B0 70E5 C6DD 97BD  --  FA18      EFA898       EFA898       F3B2AF95
 FB7E     FB7E FB7E EE62 ED5B 5FD5 66BB 66BB 66BB 5CDA C6EB 97CD  --  FA19      EFA899       EFA899       F3B2AF96
 FB80     FB80 FB80 EE63 ED5C 5FD9 66C2 66C2 66C2 5CDC C6F7 97ED  --  FA1A      EFA89A       EFA89A       F3B2AF97
 FB81     FB81 FB81 EE64 F6E5 5FDC 66D0 66D0 66D0 5CE6 C7A3 9843 D065 7994      E7A694       E7A694       F3B2AF98
 FB82     FB82 FB82 EE65 ED60 5FDD 66D3 66D3 66D3 5CE4 C7A6 9846  --  FA1B      EFA89B       EFA89B       F3B2AF99
 FB83     FB83 FB83 EE66 F6E8 5FDF 66D4 66D4 66D4 5CE8 C7AA 9851 D069 799B      E7A69B       E7A69B       F3B2AF9A
 FB84     FB84 FB84 EE67 F758 604F 66FE 66FE 66FE 5DDD C7F2 98DE D168 7AD1      E7AB91       E7AB91       F3B2AF9B
 FB85     FB85 FB85 EE68 F759 6058 67A2 67A2 67A2 5DE1 C7F5 98EB  --  7AE7      E7ABA7       E7ABA7       F3B2AF9C
 FB86     FB86 FB86 EE69  --  6059 67A6 67A6 67A6 5DE5 C7F7 98ED  --  FA1C      EFA89C       EFA89C       F3B2AF9D
 FB87     FB87 FB87 EE6A ED77 605A 67A4 67A4 67A4 5DE4 C7F8 98EE D16C 7AEB      E7ABAB       E7ABAB       F3B2AF9E
 FB88     FB88 FB88 EE6B ED83 6082 67B7 67B7 67B7 5EA3 C8BB 998C D23B 7B9E      E7AE9E       E7AE9E       F3B2AF9F
 FB89     FB89 FB89 EE6C  --  60AD 67CC 67CC 67CC 5ED3 C8E8 99CA  --  FA1D      EFA89D       EFA89D       F3B2AFA0
 FB8A     FB8A FB8A EE6D EDA0 60C4 67DD 67DD 67DD 5EFD C9A7 9A47 D361 7D48      E7B588       E7B588       F3B2AFA1
 FB8B     FB8B FB8B EE6E EDA2 60C6 67E1 67E1 67E1 5EE0 C9AC 9A53 D368 7D5C      E7B59C       E7B59C       F3B2AFA2
 FB8C     FB8C FB8C EE6F F7C5 60D4 67EC 67EC 67EC 5FE4 C9B9 9A89 D427 7DB7      E7B6B7       E7B6B7       F3B2AFA3
 FB8D     FB8D FB8D EE70 EDA6 60DA 67EB 67EB 67EB 5FB6 C9B6 9A86  --  7DA0      E7B6A0       E7B6A0       F3B2AFA4
 FB8E     FB8E FB8E EE71 EDAA 60E1 67F1 67F1 67F1 5FBE C9BD 9A8E  --  7DD6      E7B796       E7B796       F3B2AFA5
 FB8F     FB8F FB8F EE72 EDB3 60FD 67FB 67FB 67FB 5FD6 C9D5 9AAA D454 7E52      E7B992       E7B992       F3B2AFA6
 FB90     FB90 FB90 EE73 EDB7 6157 68A5 68A5 68A5 5FE8 BFFB 90FB D472 7F47      E7BD87       E7BD87       F3B2AFA7
 FB91     FB91 FB91 EE74  --  616E 68AE 68AE 68AE 5FF7 C9F2 9ADE D52E 7FA1      E7BEA1       E7BEA1       F3B2AFA8
 FB92     FB92 FB92 EE75  --  6177 68B2 68B2 68B2 60A1 C9F6 9AEC  --  FA1E      EFA89E       EFA89E       F3B2AFA9
 FB93     FB93 FB93 EE76 EDEA 61F8 68DA 68DA 68DA 61D9 CBB4 9E84 D75E 8301      E88C81       E88C81       F3B2AFAA
 FB94     FB94 FB94 EE77 EDEB 6248 68DD 68DD 68DD 70E7 CBD3 9EA8  --  8362      E88DA2       E88DA2       F3B2AFAB
 FB95     FB95 FB95 EE78 F8B1 625B 68F2 68F2 68F2 61F0 CBD4 9EA9 D822 837F      E88DBF       E88DBF       F3B2AFAC
 FB96     FB96 FB96 EE79 EDFB 6267 68FC 68FC 68FC 61FE CCA1 9F41 D837 83C7      E88F87       E88F87       F3B2AFAD
 FB97     FB97 FB97 EE7A  --  6274 68F8 68F8 68F8 62AA CBF7 9EED D841 83F6      E88FB6       E88FB6       F3B2AFAE
 FB98     FB98 FB98 EE7B EE47 627E 69A3 69A3 69A3 70E8 CCB2 9F82 D851 8448      E89188       E89188       F3B2AFAF
 FB99     FB99 FB99 EE7C EE4E 628A 69AF 69AF 69AF 70E9 CCC3 9F94 D874 84B4      E892B4       E892B4       F3B2AFB0
 FB9A     FB9A FB9A EE7D F8EF 629E 69BE 69BE 69BE 62D1 CCEE 9FDA D946 8553      E89593       E89593       F3B2AFB1
 FB9B     FB9B FB9B EE7E EE57 62A1 69BF 69BF 69BF 62D4 CCEA 9FCC D948 8559      E89599       E89599       F3B2AFB2
 FB9C     FB9C FB9C EE80  --  62A8 69BA 69BA 69BA 62D9 CCED 9FCF D951 856B      E895AB       E895AB       F3B2AFB3
 FB9D     FB9D FB9D EE81 EE59 62B9 69C1 69C1 69C1 62E4 CCFD 9FFD  --  FA1F      EFA89F       EFA89F       F3B2AFB4
 FB9E     FB9E FB9E EE82 EE5F 62BB 69C9 69C9 69C9 62E5 CDA9 A249  --  85B0      E896B0       E896B0       F3B2AFB5
 FB9F     FB9F FB9F EE83 F957 62CE 69D0 69D0 69D0 70EA CDC2 A293  --  FA20      EFA8A0       EFA8A0       F3B2AFB6
 FBA0     FBA0 FBA0 EE84 F964 62E2 69D7 69D7 69D7 63DF CDD3 A2A8  --  FA21      EFA8A1       EFA8A1       F3B2AFB7
 FBA1     FBA1 FBA1 EE85  --  635E 69DA 69DA 69DA 63E0 CDE2 A2B4  --  8807      E8A087       E8A087       F3B2AFB8
 FBA2     FBA2 FBA2 EE86 EE8D 6390 69E6 69E6 69E6 63F9 CEA4 A344 DC53 88F5      E8A3B5       E8A3B5       F3B2AFB9
 FBA3     FBA3 FBA3 EE87 EE9D 63C5 69FE 69FE 69FE 64E4 CECA A39D DD48 8A12      E8A892       E8A892       F3B2AFBA
 FBA4     FBA4 FBA4 EE88 EEA0 63CB 6AA5 6AA5 6AA5 64EC CEDA A3BA DD54 8A37      E8A8B7       E8A8B7       F3B2AFBB
 FBA5     FBA5 FBA5 EE89 EEA6 63D4 6AAA 6AAA 6AAA 64DA CEDF A3BF DD6A 8A79      E8A9B9       E8A9B9       F3B2AFBC
 FBA6     FBA6 FBA6 EE8A EEA7 63DF 6AAB 6AAB 6AAB 65A7 CEE2 A3B4 DD7A 8AA7      E8AAA7       E8AAA7       F3B2AFBD
 FBA7     FBA7 FBA7 EE8B F9DE 63E1 6AAD 6AAD 6AAD 6BEA CEE3 A3B5 DE24 8ABE      E8AABE       E8AABE       F3B2AFBE
 FBA8     FBA8 FBA8 EE8C EEA9 63ED 6AB3 6AB3 6AB3 65B3 CEEC A3CE DE30 8ADF      E8AB9F       E8AB9F       F3B2AFBF
 FBA9     FBA9 FBA9 EE8D EEAC 63EE 6AB2 6AB2 6AB2 65B7 CEEA A3CC  --  FA22      EFA8A2       EFA8A2       F3B2B080
 FBAA     FBAA FBAA EE8E EEAB 63EF 6AB4 6AB4 6AB4 65B6 CEED A3CF DE35 8AF6      E8ABB6       E8ABB6       F3B2B081
 FBAB     FBAB FBAB EE8F  --  6445 6ABE 6ABE 6ABE 65C7 CEF8 A3EE DE4B 8B53      E8AD93       E8AD93       F3B2B082
 FBAC     FBAC FBAC EE90 F9ED 6450 6AC3 6AC3 6AC3 65D1 CEFE A3FE  --  8B7F      E8ADBF       E8ADBF       F3B2B083
 FBAD     FBAD FBAD EE91 FA4F 647C 6AD0 6AD0 6AD0 66AB CFBC A48D DF39 8CF0      E8B3B0       E8B3B0       F3B2B084
 FBAE     FBAE FBAE EE92 EEB8 647D 6ACF 6ACF 6ACF 65F9 CFBB A48C  --  8CF4      E8B3B4       E8B3B4       F3B2B085
 FBAF     FBAF FBAF EE93 EEBC 6485 6AD5 6AD5 6AD5 65F8 CFC2 A493 DF43 8D12      E8B492       E8B492       F3B2B086
 FBB0     FBB0 FBB0 EE94  --  648D 6AD8 6AD8 6AD8 66B9 CFC5 A496  --  8D76      E8B5B6       E8B5B6       F3B2B087
 FBB1     FBB1 FBB1 EE95  --  648E 6AD9 6AD9 6AD9 66C3 CFC7 A498  --  FA23      EFA8A3       EFA8A3       F3B2B088
 FBB2     FBB2 FBB2 EE96  --  64C9 6AE1 6AE1 6AE1 66ED CFDE A4BE E059 8ECF      E8BB8F       E8BB8F       F3B2B089
 FBB3     FBB3 FBB3 EE97 FA8E 64F1 6AEB 6AEB 6AEB 70F9 CFFD A4FD  --  FA24      EFA8A4       EFA8A4       F3B2B08A
 FBB4     FBB4 FBB4 EE98  --  6549 6AFA 6AFA 6AFA 70FB D0C2 A593  --  FA25      EFA8A5       EFA8A5       F3B2B08B
 FBB5     FBB5 FBB5 EE99  --  6553 6BA3 6BA3 6BA3 70FA D0CD A5A2 E162 9067      E981A7       E981A7       F3B2B08C
 FBB6     FBB6 FBB6 EE9A EEE5 6564 6BC7 6BC7 6BC7 68A6 D0EC A5CE  --  90DE      E9839E       E9839E       F3B2B08D
 FBB7     FBB7 FBB7 EE9B EEE8 6566 6BCA 6BCA 6BCA 68A8 D0F2 A5DE  --  FA26      EFA8A6       EFA8A6       F3B2B08E
 FBB8     FBB8 FBB8 EE9C EEEA 6567 6BCD 6BCD 6BCD 68DF D0F3 A5DF  --  9115      E98495       E98495       F3B2B08F
 FBB9     FBB9 FBB9 EE9D EEEE 656B 6BD1 6BD1 6BD1 68B0 D0F9 A5EF E247 9127      E984A7       E984A7       F3B2B090
 FBBA     FBBA FBBA EE9E FACC 6583 6BD9 6BD9 6BD9 69BE D1B8 A688 E328 91DA      E9879A       E9879A       F3B2B091
 FBBB     FBBB FBBB EE9F EEFA 6585 6BD8 6BD8 6BD8 69BC D1B7 A687 E326 91D7      E98797       E98797       F3B2B092
 FBBC     FBBC FBBC EEA0  --  6586 6BDA 6BDA 6BDA 69BF D1B9 A689 E329 91DE      E9879E       E9879E       F3B2B093
 FBBD     FBBD FBBD EEA1 EF41 658A 6BDC 6BDC 6BDC 69C5 D1BB A68C E32F 91ED      E987AD       E987AD       F3B2B094
 FBBE     FBBE FBBE EEA2 FAD0 658C 6BE1 6BE1 6BE1 69C6 D1C1 A692 E330 91EE      E987AE       E987AE       F3B2B095
 FBBF     FBBF FBBF EEA3 EEFC 658D 6BE0 6BE0 6BE0 69C0 D1BE A68F E32A 91E4      E987A4       E987A4       F3B2B096
 FBC0     FBC0 FBC0 EEA4 EF40 658F 6BDF 6BDF 6BDF 69C1 D1BD A68E E32B 91E5      E987A5       E987A5       F3B2B097
 FBC1     FBC1 FBC1 EEA5  --  6590 6BE5 6BE5 6BE5 69CD D1C6 A697 E33C 9206      E98886       E98886       F3B2B098
 FBC2     FBC2 FBC2 EEA6 EF44 6593 6BE9 6BE9 6BE9 69D1 D1C4 A695 E341 9210      E98890       E98890       F3B2B099
 FBC3     FBC3 FBC3 EEA7 FAD2 6595 6BE7 6BE7 6BE7 69D0 D1CB A69E E33F 920A      E9888A       E9888A       F3B2B09A
 FBC4     FBC4 FBC4 EEA8 EF47 659B 6BF2 6BF2 6BF2 69E1 D1DC A6BC E355 923A      E988BA       E988BA       F3B2B09B
 FBC5     FBC5 FBC5 EEA9 EF49 659E 6BEE 6BEE 6BEE 69E4 D1D6 A6AC E358 9240      E98980       E98980       F3B2B09C
 FBC6     FBC6 FBC6 EEAA EF48 659F 6BF3 6BF3 6BF3 69E2 D1D9 A6AF E356 923C      E988BC       E988BC       F3B2B09D
 FBC7     FBC7 FBC7 EEAB FADA 65A1 6BEF 6BEF 6BEF 69E8 D1DD A6BD E35F 924E      E9898E       E9898E       F3B2B09E
 FBC8     FBC8 FBC8 EEAC FADD 65A3 6BF1 6BF1 6BF1 69EE D1DB A6BB E363 9259      E98999       E98999       F3B2B09F
 FBC9     FBC9 FBC9 EEAD FADB 65A5 6BEC 6BEC 6BEC 69EA D1D1 A6A6 E361 9251      E98991       E98991       F3B2B0A0
 FBCA     FBCA FBCA EEAE EF46 65A6 6BED 6BED 6BED 69E0 D1D3 A6A8 E354 9239      E988B9       E988B9       F3B2B0A1
 FBCB     FBCB FBCB EEAF FAE1 65A8 6BF6 6BF6 6BF6 6BD2 D1E4 A6B6 E369 9267      E989A7       E989A7       F3B2B0A2
 FBCC     FBCC FBCC EEB0 FAED 65AB 6CA6 6CA6 6CA6 6AA7 D1EE A6DA E426 92A7      E98AA7       E98AA7       F3B2B0A3
 FBCD     FBCD FBCD EEB1  --  65AC 6CA8 6CA8 6CA8 69F7 D1EF A6DB E371 9277      E989B7       E989B7       F3B2B0A4
 FBCE     FBCE FBCE EEB2 EF4C 65AD 6CA5 6CA5 6CA5 69F8 D1E6 A6B8 E372 9278      E989B8       E989B8       F3B2B0A5
 FBCF     FBCF FBCF EEB3 FAFA 65B4 6CB1 6CB1 6CB1 6AC4 D1F9 A6EF E44B 92E7      E98BA7       E98BA7       F3B2B0A6
 FBD0     FBD0 FBD0 EEB4 FAF7 65B5 6CAC 6CAC 6CAC 6ABC D1F0 A6DC E441 92D7      E98B97       E98B97       F3B2B0A7
 FBD1     FBD1 FBD1 EEB5 FAF8 65B6 6CB8 6CB8 6CB8 6ABE D1F7 A6ED E443 92D9      E98B99       E98B99       F3B2B0A8
 FBD2     FBD2 FBD2 EEB6 FAF6 65B7 6CB3 6CB3 6CB3 6AB8 D2A4 A744 E43E 92D0      E98B90       E98B90       F3B2B0A9
 FBD3     FBD3 FBD3 EEB7  --  65B8 6CB7 6CB7 6CB7 6BD3 D2A6 A746  --  FA27      EFA8A7       EFA8A7       F3B2B0AA
 FBD4     FBD4 FBD4 EEB8  --  65B9 6CB5 6CB5 6CB5 6ABB D2A3 A743 E440 92D5      E98B95       E98B95       F3B2B0AB
 FBD5     FBD5 FBD5 EEB9 EF52 65BC 6CB0 6CB0 6CB0 6AC2 D1FE A6FE E447 92E0      E98BA0       E98BA0       F3B2B0AC
 FBD6     FBD6 FBD6 EEBA EF51 65BD 9FCA 9FCA 9FCA 6ABA D1F4 A6EA E43F 92D3      E98B93       E98B93       F3B2B0AD
 FBD7     FBD7 FBD7 EEBB EF57 65C1 6CC3 6CC3 6CC3 6AD5 D2B4 A784 E460 9325      E98CA5       E98CA5       F3B2B0AE
 FBD8     FBD8 FBD8 EEBC EF56 65C3 6CBE 6CBE 6CBE 6AD3 D2A9 A749 E45E 9321      E98CA1       E98CA1       F3B2B0AF
 FBD9     FBD9 FBD9 EEBD FB43 65C5 6CCD 6CCD 6CCD 69B3 D2AF A756 E451 92FB      E98BBB       E98BBB       F3B2B0B0
 FBDA     FBDA FBDA EEBE  --  65C6 6CC2 6CC2 6CC2 6BD4 D2B7 A787  --  FA28      EFA8A8       EFA8A8       F3B2B0B1
 FBDB     FBDB FBDB EEBF FB48 65C8 6CBF 6CBF 6CBF 6AD1 D2AB A752 E45C 931E      E98C9E       E98C9E       F3B2B0B2
 FBDC     FBDC FBDC EEC0 EF53 65C9 6CC1 6CC1 6CC1 6ACA D2B5 A785 E452 92FF      E98BBF       E98BBF       F3B2B0B3
 FBDD     FBDD FBDD EEC1 FB47 65CC 6CC4 6CC4 6CC4 6AD0 D2B3 A783 E45B 931D      E98C9D       E98C9D       F3B2B0B4
 FBDE     FBDE FBDE EEC2 FB44 65CE 6CC5 6CC5 6CC5 6ACC D2B2 A782 E454 9302      E98C82       E98C82       F3B2B0B5
 FBDF     FBDF FBDF EEC3 EF5E 65D0 6CCA 6CCA 6CCA 6AE4 D2BB A78C E47A 9370      E98DB0       E98DB0       F3B2B0B6
 FBE0     FBE0 FBE0 EEC4 FB4F 65D5 6CCC 6CCC 6CCC 6ADE D2C3 A794 E46F 9357      E98D97       E98D97       F3B2B0B7
 FBE1     FBE1 FBE1 EEC5  --  65D8 6CD8 6CD8 6CD8 6AF2 D2D3 A7A8 E533 93A4      E98EA4       E98EA4       F3B2B0B8
 FBE2     FBE2 FBE2 EEC6 EF62 65DD 6CE0 6CE0 6CE0 6AF9 D2DF A7BF E53F 93C6      E98F86       E98F86       F3B2B0B9
 FBE3     FBE3 FBE3 EEC7 EF63 65E4 6CDF 6CDF 6CDF 6BA3 D2DA A7BA E549 93DE      E98F9E       E98F9E       F3B2B0BA
 FBE4     FBE4 FBE4 EEC8  --  65E8 6CE6 6CE6 6CE6 6BAE D2E7 A7B9 E550 93F8      E98FB8       E98FB8       F3B2B0BB
 FBE5     FBE5 FBE5 EEC9  --  65F0 6CEF 6CEF 6CEF 6BBD D2F2 A7DE E562 9431      E990B1       E990B1       F3B2B0BC
 FBE6     FBE6 FBE6 EECA FB6C 65F4 6CF0 6CF0 6CF0 6BC5 D2F3 A7DF E56A 9445      E99185       E99185       F3B2B0BD
 FBE7     FBE7 FBE7 EECB  --  65F6 6CF1 6CF1 6CF1 6BC6 D2F4 A7EA E56B 9448      E99188       E99188       F3B2B0BE
 FBE8     FBE8 FBE8 EECC  --  664A 6CF8 6CF8 6CF8 6BE1 D2FC A7FC  --  9592      E99692       E99692       F3B2B0BF
 FBE9     FBE9 FBE9 EECD EF7C 6674 6DA2 6DA2 6DA2 68CC D3B9 A889  --  F9DC      EFA79C       EFA79C       F3B2B180
 FBEA     FBEA FBEA EECE  --  6678 6DA4 6DA4 6DA4 68E0 D3C0 A891  --  FA29      EFA8A9       EFA8A9       F3B2B181
 FBEB     FBEB FBEB EECF EF7D 6679 6DA5 6DA5 6DA5 68D4 D3C1 A892 E668 969D      E99A9D       E99A9D       F3B2B182
 FBEC     FBEC FBEC EED0 FB8C 667F 6DA6 6DA6 6DA6 68DB D3C4 A895 E66F 96AF      E99AAF       E99AAF       F3B2B183
 FBED     FBED FBED EED1 EF86 6697 9FCD 9FCD 9FCD 6CAC D3D4 A8A9 E72C 9733      E99CB3       E99CB3       F3B2B184
 FBEE     FBEE FBEE EED2 EF87 6699 9FCE 9FCE 9FCE 6CAD D3D5 A8AA  --  973B      E99CBB       E99CBB       F3B2B185
 FBEF     FBEF FBEF EED3  --  669E 6DAE 6DAE 6DAE 6CB0 D3D7 A8AD E72E 9743      E99D83       E99D83       F3B2B186
 FBF0     FBF0 FBF0 EED4 EF88 66A2 6DB0 6DB0 6DB0 6CB1 D3D9 A8AF  --  974D      E99D8D       E99D8D       F3B2B187
 FBF1     FBF1 FBF1 EED5 EF8A 66A3 9FCF 9FCF 9FCF 6CAE D3DB A8BB E731 974F      E99D8F       E99D8F       F3B2B188
 FBF2     FBF2 FBF2 EED6  --  66A4 6DB1 6DB1 6DB1 6CB3 D3DC A8BC  --  9751      E99D91       E99D91       F3B2B189
 FBF3     FBF3 FBF3 EED7 FB9C 66A5 6DB2 6DB2 6DB2 6CB5 D3DE A8BE E732 9755      E99D95       E99D95       F3B2B18A
 FBF4     FBF4 FBF4 EED8 EF9F 66CB 6DC1 6DC1 6DC1 6CF3 D4A1 A941 E831 9857      E9A197       E9A197       F3B2B18B
 FBF5     FBF5 FBF5 EED9 EFA3 66CC 6DC2 6DC2 6DC2 6CF7 D4A4 A944 E836 9865      E9A1A5       E9A1A5       F3B2B18C
 FBF6     FBF6 FBF6 EEDA  --  66DD 6DC5 6DC5 6DC5 6DB3 D4B3 A983  --  FA2A      EFA8AA       EFA8AA       F3B2B18D
 FBF7     FBF7 FBF7 EEDB  --  66DF 6DC6 6DC6 6DC6 6DB4 D4B4 A984  --  FA2B      EFA8AB       EFA8AB       F3B2B18E
 FBF8     FBF8 FBF8 EEDC  --  66E5 6DC8 6DC8 6DC8 6DBD D4B7 A987 E85D 9927      E9A4A7       E9A4A7       F3B2B18F
 FBF9     FBF9 FBF9 EEDD  --  66E6 6DC9 6DC9 6DC9 6DBE D4B8 A988  --  FA2C      EFA8AC       EFA8AC       F3B2B190
 FBFA     FBFA FBFA EEDE EFA9 66F9 6DCC 6DCC 6DCC 5CF3 D4C3 A994  --  999E      E9A69E       E9A69E       F3B2B191
 FBFB     FBFB FBFB EEDF EFB1 6759 6DD9 6DD9 6DD9 6DF6 D4D7 A9AD E951 9A4E      E9A98E       E9A98E       F3B2B192
 FBFC     FBFC FBFC EEE0  --  676B 6DD4 6DD4 6DD4 6EA4 D4DC A9BC  --  9AD9      E9AB99       E9AB99       F3B2B193
 FC40     FC40 FC40 EEE1 FC51 676C 6DDC 6DDC 6DDC 6EA7 D4DE A9BE E96D 9ADC      E9AB9C       E9AB9C       F3B2B194
 FC41     FC41 FC41 EEE2 EFC2 678F 9FD2 9FD2 9FD2 6ED0 D4F0 A9DC EA4D 9B75      E9ADB5       E9ADB5       F3B2B195
 FC42     FC42 FC42 EEE3 EFC1 6791 6DE0 6DE0 6DE0 6EEE D4EF A9DB  --  9B72      E9ADB2       E9ADB2       F3B2B196
 FC43     FC43 FC43 EEE4 EFC5 6793 6DE2 6DE2 6DE2 6ED8 D4F3 A9DF EA5B 9B8F      E9AE8F       E9AE8F       F3B2B197
 FC44     FC44 FC44 EEE5 FC69 679B 6DE5 6DE5 6DE5 6EDB D4FB A9FB EA66 9BB1      E9AEB1       E9AEB1       F3B2B198
 FC45     FC45 FC45 EEE6  --  679F 6DE7 6DE7 6DE7 6EEF D5A8 AA48 EA6A 9BBB      E9AEBB       E9AEBB       F3B2B199
 FC46     FC46 FC46 EEE7 EFCD 67B1 9FD3 9FD3 9FD3 6EE3 D5AF AA56 EB25 9C00      E9B080       E9B080       F3B2B19A
 FC47     FC47 FC47 EEE8 EFDC 67E2 9FD8 9FD8 9FD8 6FA3 D3C9 A89A EB7B 9D70      E9B5B0       E9B5B0       F3B2B19B
 FC48     FC48 FC48 EEE9 FCB2 67E4 6DF2 6DF2 6DF2 6FA1 D5D4 AAA9 EB7A 9D6B      E9B5AB       E9B5AB       F3B2B19C
 FC49     FC49 FC49 EEEA  --  67F4 6DF4 6DF4 6DF4 6FB2 D5DA AABA  --  FA2D      EFA8AD       EFA8AD       F3B2B19D
 FC4A     FC4A FC4A EEEB FCCD 67FE 6EAC 6EAC 6EAC 6FAE D5DC AABC EC56 9E19      E9B899       E9B899       F3B2B19E
 FC4B     FC4B FC4B EEEC EFF0 6856 6EA1 6EA1 6EA1 6FD6 D5EB AACD  --  9ED1      E9BB91       E9BB91       F3B2B19F

=head1 UTF-8.0 vs. UTF-8.1

 The large character sets has always some problems. One of problem of UTF-8 is
 difference between Microsoft's definition and general definition. Already, #'
 in this software, "utf8" means Microsoft's definition. So I named the general #'
 UTF-8 "utf8.1" to distinguish them.

 Terminology and definitions
 -------------------------------------------------------------------------------
 UTF-8   [ju:ti:ef eit]         could mean sometimes UTF-8.0, sometimes UTF-8.x
 UTF-8.0 [ju:ti:ef hatten-zero] is Microsoft's definition of the UTF-8 #'
 UTF-8.1 [ju:ti:ef hatten-ichi] is general definition of the UTF-8
 UTF-8.x [ju:ti:ef hatten-ekks] means both UTF-8.0 and UTF-8.1
 -------------------------------------------------------------------------------
 
 The following is the difference between UTF-8.0 and UTF-8.1.
 
 #+++++++----------------------------------------------------------------------------------------------------------- CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
 #||||||| ++++------------------------------------------------------------------------------------------------------ Microsoft CP932, IANA Windows-31J
 #||||||| |||| ++++------------------------------------------------------------------------------------------------- IBM CP932
 #||||||| |||| |||| ++++-------------------------------------------------------------------------------------------- NEC CP932
 #||||||| |||| |||| |||| ++++--------------------------------------------------------------------------------------- JISC Shift_JIS-2004
 #||||||| |||| |||| |||| |||| ++++---------------------------------------------------------------------------------- IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
 #||||||| |||| |||| |||| |||| |||| ++++----------------------------------------------------------------------------- HITACHI KEIS78
 #||||||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------------ HITACHI KEIS83
 #||||||| |||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------- HITACHI KEIS90
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| ++++-------------------------------------------------------------- FUJITSU JEF
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++--------------------------------------------------------- NEC JIPS(J)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++---------------------------------------------------- NEC JIPS(E)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++----------------------------------------------- UNISYS LetsJ
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| +++++++++------------------------------------- Unicode
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| ++++++++++++------------------------ UTF-8.0 (aka UTF-8)
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| ++++++++++++----------- UTF-8.1
 #||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| |||||||||||| ++++++++-- UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
 #2345678 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 123456789 123456789012 123456789012 12345678
 #VVVVVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVVVVVVV VVVVVVVVVVVV VVVVVVVVVVVV VVVVVVVV
 815C     815C 815C 815C  --  DDB7 A1BD A1BD A1BD A1BD 213D 4F7E A1BD 2015      E28095       E28094       F3B0849C
 8161     8161 8161 8161 81D2 DFE5 A1C2 A1C2 A1C2 A1C2 2142 4FC2  --  2225      E288A5       E28096       F3B084A1
 817C     817C 817C 817C 81AF E9F3 A1DD A1DD A1DD A1DD 215D 4F5A  --  FF0D      EFBC8D       E28892       F3B084BC
 9C5A815C  --   --   --  815C 444A  --   --   --   --   --   --   --  2014      E28094       E28095       F3B2B4AC
 9C5A8161  --   --   --  8161 447C  --   --   --   --   --   --  A1C2 2016      E28096       E288A5       F3B2B4B1
 9C5A817C  --   --   --  817C 4260  --   --   --   --   --   --  A1DD 2212      E28892       EFBC8D       F3B2B58C

=head1 FIXED ERRATAS OF MAPPINGS

The mapping of Jacode4e version 2.13.6.17 or earlier has the following erattas:

=head2 KEIS78, KEIS83, and KEIS90

Mapping of 'keis78', 'keis83', and 'keis90' had erattas as following:

  ----------------------------------------------
  Unicode      2.13.6.18             2.13.6.17
               or later              or older
  ----------------------------------------------
  芺 U+82BA    KEIS:(no mapping)     KEIS:68CF
  覀 U+8980    KEIS:(no mapping)     KEIS:59A9
  麽 U+9EBD    KEIS:6DFB             KEIS:5EC3
  ----------------------------------------------

=head2 JEF and JEF9P

Mapping of 'jef' and 'jef9p' had erattas as following:

  ----------------------------------------------
  Unicode      2.13.6.18             2.13.6.17
               or later              or older
  ----------------------------------------------
  漼 U+6F3C    JEF:(no mapping)      JEF:56F3
  臽 U+81FD    JEF:(no mapping)      JEF:52E1
  海 U+FA45    JEF:55FB              JEF:53FA
  渚 U+FA46    JEF:BDED              JEF:55FB
  漢 U+FA47    JEF:56E6              JEF:BDED
  煮 U+FA48    JEF:58A7              JEF:56E6
  爫 U+FA49    JEF:(no mapping)      JEF:58A7
  社 U+FA4C    JEF:5CD3              JEF:5CB3
  祉 U+FA4D    JEF:5CD5              JEF:5CD3
  祈 U+FA4E    JEF:5CD4              JEF:5CD5
  祐 U+FA4F    JEF:5CD6              JEF:5CD4
  祖 U+FA50    JEF:5CD7              JEF:5CD6
  祝 U+FA51    JEF:5CD9              JEF:5CD7
  禍 U+FA52    JEF:5CE2              JEF:5CD9
  禎 U+FA53    JEF:5CE3              JEF:5CE2
  穀 U+FA54    JEF:5CF4              JEF:5CE3
  突 U+FA55    JEF:5DCD              JEF:5CF4
  縉 U+FA58    JEF:(no mapping)      JEF:5FC8
  署 U+FA5A    JEF:5FF0              JEF:5EE3
  者 U+FA5B    JEF:60B5              JEF:5FF0
  臭 U+FA5C    JEF:60E3              JEF:60B5
  艹 U+FA5D    JEF:(no mapping)      JEF:60E3
  ～ U+FF5E    JEF:(no mapping)      JEF:A1C1
  ----------------------------------------------

=head2 JIPS(J)

Mapping of 'jipsj' had erattas as following:

  ----------------------------------------------
  Unicode      2.13.6.18             2.13.6.17
               or later              or older
  ----------------------------------------------
  Ý  U+00DD    JIPSJ:(no mapping)    JIPSJ:A5B1
  耰 U+8030    JIPSJ:(no mapping)    JIPSJ:CAAB
  ----------------------------------------------

=head2 JIPS(E)

Mapping of 'jipse' had erattas as following:

  ----------------------------------------------
  Unicode      2.13.6.18             2.13.6.17
               or later              or older
  ----------------------------------------------
  Ý  U+00DD    JIPSE:(no mapping)    JIPSE:4581
  耰 U+8030    JIPSE:(no mapping)    JIPSE:9D52
  ----------------------------------------------

=head1 DEPENDENCIES

This software requires perl version 5.00503 or later to run.
(All of Perl4 users in the world, pardon me!)

=head1 SOFTWARE LIFE CYCLE

                                         Jacode.pm
                    jcode.pl  Encode.pm  jacode.pl  Jacode4e  Jacode4e::RoundTrip
  --------------------------------------------------------------------------------
  1993 Perl4.036       |                     |                                    
    :     :            :                     :                                    
  1999 Perl5.00503     |                     |         |               |          
  2000 Perl5.6         |                     |         |               |          
  2002 Perl5.8         |         Born        |         |               |          
  2007 Perl5.10        V          |          |         |               |          
  2010 Perl5.12       EOL         |         Born       |               |          
  2011 Perl5.14                   |          |         |               |          
  2012 Perl5.16                   |          |         |               |          
  2013 Perl5.18                   |          |         |               |          
  2014 Perl5.20                   |          |         |               |          
  2015 Perl5.22                   |          |         |               |          
  2016 Perl5.24                   |          |         |               |          
  2017 Perl5.26                   |          |         |               |          
  2018 Perl5.28                   |          |        Born            Born        
  2019 Perl5.30                   |          |         |               |          
  2020 Perl5.32                   :          :         :               :          
  2030 Perl5.52                   :          :         :               :          
  2040 Perl5.72                   :          :         :               :          
  2050 Perl5.92                   :          :         :               :          
  2060 Perl5.112                  :          :         :               :          
  2070 Perl5.132                  :          :         :               :          
  2080 Perl5.152                  :          :         :               :          
  2090 Perl5.172                  :          :         :               :          
  2100 Perl5.192                  :          :         :               :          
  2110 Perl5.212                  :          :         :               :          
  2120 Perl5.232                  :          :         :               :          
    :     :                       V          V         V               V          
  --------------------------------------------------------------------------------

=head1 SOFTWARE COVERAGE

When you lost your way, you can see this matrix and find your way.

  Skill/Use  Amateur    Semipro    Pro        Enterprise  Enterprise(round-trip)
  -------------------------------------------------------------------------------
  Expert     jacode.pl  Encode.pm  Encode.pm  Jacode4e    Jacode4e::RoundTrip
  -------------------------------------------------------------------------------
  Middle     jacode.pl  jacode.pl  Encode.pm  Jacode4e    Jacode4e::RoundTrip
  -------------------------------------------------------------------------------
  Beginner   jacode.pl  jacode.pl  jacode.pl  Jacode4e    Jacode4e::RoundTrip
  -------------------------------------------------------------------------------

=head1 Why CP932X Born?

In order to know why CP932X exists the way it is(or isn't), one must first know why CP932X born. #'

  Q1) Is CCS of JIS X 0208 enough?
  A1) No. Often we require GAIJI.
  
  Q2) Is CCS of JIS X 0213 enough?
  A2) It's not perfect, but enough for many people.
  
  Q3) Is CES by UTF-8 good?
  A3) No. In Japanese information processing, it's unstable and not popular still now.
  
  Q4) Is CES by Shift_JIS-2004 good?
  A4) No. Because Shift_JIS-2004 cannot support very popular CP932 and your GAIJI. We need a realistic solution to solving real problem.
  
  Q5) Is escape sequence good idea to support CCS of JIS X 0213?
  A5) No. Because the programming is so hard.
  
  Q6) Which character is best as single shift code to support CCS of JIS X 0213?
    -- The single shift code must be a DBCS code, because DBCS field cannot store SBCS code in some cases
    -- Moreover, all GAIJI code points must be yours
    -- The impact of this solution must be minimum
  A6) I select 1-55-27 as single shift code. It is ghost character and not used by nobody.

=head1 AUTHOR

INABA Hitoshi E<lt>ina@cpan.orgE<gt> in a CPAN

This project was originated by INABA Hitoshi.

=head1 LICENSE AND COPYRIGHT

This software is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.

This software is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

=head1 SEE ALSO

 Windows-31J information
 http://www2d.biglobe.ne.jp/%7Emsyk/charcode/cp932/index.html

 Windows-31 J Character Set
 http://www2d.biglobe.ne.jp/%7Emsyk/charcode/cp932/Windows-31J-charset.html

 Windows-31 J Duplicate Coded Characters and Unicode
 http://www2d.biglobe.ne.jp/%7Emsyk/charcode/cp932/uni2sjis-Windows-31J.html

 Data transformation function provided by WebSphere MQ
 Chapter 11. Problems with converting certain characters (Japanese language related)
 https://www.ibm.com/developerworks/jp/websphere/library/wmq/mq_transform/11.html
 https://www.ibm.com/developerworks/jp/websphere/library/wmq/mq_transform/11-pdf.pdf

 CPGID 00290
 https://www-01.ibm.com/software/globalization/cdra/
 https://www-01.ibm.com/software/globalization/cp/cp00290.html
 ftp://ftp.software.ibm.com/software/globalization/gcoc/attachments/CP00290.pdf
 ftp://ftp.software.ibm.com/software/globalization/gcoc/attachments/CP00290.txt

 IBM Japanese Graphic Character Set, Kanji DBCS Host and DBCS - PC
 https://www-01.ibm.com/software/globalization/cdra/
 ftp://ftp.software.ibm.com/software/globalization/gcoc/attachments/CP00300.pdf

 IBM Kanji code list
 N:GC18-2040-3

 HiRDB Datareplicator Version 8 manuals, Hitachi, Ltd.
 http://itdoc.hitachi.co.jp/manuals/3020/3020636050/W3600001.HTM
 http://itdoc.hitachi.co.jp/manuals/3020/3020636050/W3600166.HTM
 http://itdoc.hitachi.co.jp/manuals/3020/30203J3820/ISUS0268.HTM
 http://itdoc.hitachi.co.jp/manuals/3000/30003D5820/CLNT0235.HTM

 KEIS code book, Culti Co.,Ltd.
 http://www.culti.co.jp/2016/02/01/%e3%82%ab%e3%83%ab%e3%83%81%e7%99%ba%e8%a1%8c%e6%9b%b8%e7%b1%8d/

 Appendix B.2 Character code differences
 http://itdoc.hitachi.co.jp/manuals/3020/3020759580/G5950334.HTM

 Appendix E Handling of character codes in PDE - Form Designer (applies only to distributed type PDE)
 http://itdoc.hitachi.co.jp/manuals/3020/30203p0360/PDEF0203.HTM

 HITAC Character code table (KEIS83)
 Document number 8080-2-30
 Document number 8080-2-074-10
 Document number 8080-2-100-10
 Document number 8080-2-109

 Linkexpress, FUJITSU LIMITED
 http://software.fujitsu.com/jp/manual/manualfiles/M080093/J2X15930/03Z200/index.html
 http://software.fujitsu.com/jp/manual/manualfiles/M080093/J2X15930/03Z200/unyo05/unyo0413.html
 http://software.fujitsu.com/jp/manual/manualfiles/m130010/b1fw5992/01z200/b5992-c-00-00.html

 JEF code book, Culti Co.,Ltd.
 http://www.culti.co.jp/2016/02/01/jef%e3%82%b3%e3%83%bc%e3%83%89%e3%83%96%e3%83%83%e3%82%af/

 Linkexpress operation manual J2X1-5930-03Z2(00) FUJITSU LIMITED
 http://software.fujitsu.com/jp/manual/manualfiles/M080093/J2X15930/03Z200/index.html
 http://software.fujitsu.com/jp/manual/manualfiles/M070086/J2X15930/01Z200/unyo05/unyo0416.html
 http://software.fujitsu.com/jp/manual/manualfiles/M070086/J2X15930/01Z200/unyo05/unyo0420.html
 http://software.fujitsu.com/jp/manual/manualfiles/M070086/J2X15930/01Z200/unyo05/unyo0421.html
 http://software.fujitsu.com/jp/manual/manualfiles/m120010/b1fw5691/05z200/index.html
 http://software.fujitsu.com/jp/manual/manualfiles/m120010/b1fw5691/05z200/b5691-g-00-00.html
 https://software.fujitsu.com/jp/manual/manualfiles/m200002/b1wd0741/14z200/b0741-c-01-00.html
 https://software.fujitsu.com/jp/manual/manualfiles/m200002/b1wd0741/14z200/b0741-c-07-00.html
 https://software.fujitsu.com/jp/manual/manualfiles/m200002/b1wd0741/14z200/b0741-c-08-00.html
 https://software.fujitsu.com/jp/manual/manualfiles/m200002/b1wd0741/14z200/b0741-c-09-00.html

 hidekatsu-izuno/jef4j
 https://github.com/hidekatsu-izuno/jef4j

 JHTc(JHT command edition)
 http://www.vector.co.jp/soft/winnt/util/se094205.html

 FACOM JEF Character code index dictionary
 Manual code 99FR-0012-3
 Manual code 99FR-8010-1

 JIPS code book, Culti Co.,Ltd.
 http://www.culti.co.jp/2016/02/01/jips%e3%82%b3%e3%83%bc%e3%83%89%e3%83%96%e3%83%83%e3%82%af/

 NEC Corporation Standard character set dictionary <BASIC>
 ZBB10-3

 NEC Corporation Standard character set dictionary <EXTENSION>
 ZBB11-2

 ClearPath Enterprise Servers MultiLingual System Administration, Operations, and Programming Guide ClearPath MCP 15.0 April 2013 8600 0288-308
 https://public.support.unisys.com/aseries/docs/ClearPath-MCP-16.0/PDF/86000288-308.pdf

 Heterogeneous database cooperation among heterogeneous OS environments
 http://www.unisys.co.jp/tec_info/tr56/5605.htm

 UNISYS_SJIS KANJI CONVERT(DOS ver)
 https://www.vector.co.jp/soft/dos/util/se238607.html

 ANSI X3.4-1968 (US-ASCII) with 0x60/0x27 as left/right single quotation mark to Unicode
 http://www.unicode.org/Public/MAPPINGS/VENDORS/MISC/US-ASCII-QUOTES.TXT

 JIS X 0201 (1976) to Unicode 1.1 Table
 http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/JIS/JIS0201.TXT

 JIS X 0208 (1990) to Unicode
 http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/JIS/JIS0208.TXT

 JIS X 0212 (1990) to Unicode
 http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/JIS/JIS0212.TXT

 cp932 to Unicode table
 ftp://ftp.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP932.TXT
 https://support.microsoft.com/ja-jp/help/170559/prb-conversion-problem-between-shift-jis-and-unicode

 Shift_JIS-2004 to Unicode table
 http://x0213.org/codetable/sjis-0213-2004-std.txt

 UTF-8, a transformation format of ISO 10646
 https://www.rfc-editor.org/rfc/rfc3629.txt

 iDIVO Ver.1.4.0
 https://www.hulft.com/shukka/files/iDIVO/SP-DV1-CC-02-01.pdf

 Kanji shift code
 https://ja.wikipedia.org/wiki/%E6%BC%A2%E5%AD%97%E3%82%B7%E3%83%95%E3%83%88%E3%82%B3%E3%83%BC%E3%83%89

 U+32FF SQUARE ERA NAME REIWA
 https://unicode.org/versions/Unicode12.1.0/
 http://en.glyphwiki.org/wiki/u32ff
 https://www.ibm.com/support/pages/zos%E3%81%AB%E3%81%8A%E3%81%91%E3%82%8B%E6%96%B0%E5%85%83%E5%8F%B7%E5%AF%BE%E5%BF%9C%E3%81%AB%E3%81%A4%E3%81%84%E3%81%A6
 http://www.hitachi-support.com/alert/ss/HWS17-007/list.pdf#page=29

 CJKV Information Processing
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/78-vs-83-1.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/78-vs-83-2.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/78-vs-83-3.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/78-vs-83-4.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/83-vs-90-1.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/83-vs-90-2.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/TJ2.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/TJ3.sjs
 https://resources.oreilly.com/examples/9781565922242/tree/master/AppQ/TJ4.sjs

 Very old fj.kanji discussion
 http://www.ie.u-ryukyu.ac.jp/~kono/fj/fj.kanji/index.html

 jcode.pl: Perl library for Japanese character code conversion, Kazumasa Utashiro
 ftp://ftp.iij.ad.jp/pub/IIJ/dist/utashiro/perl/
 http://web.archive.org/web/20090608090304/http://srekcah.org/jcode/
 ftp://ftp.oreilly.co.jp/pcjp98/utashiro/
 http://mail.pm.org/pipermail/tokyo-pm/2002-March/001319.html
 https://twitter.com/uta46/status/11578906320

 jacode - Perl program for Japanese character code conversion
 https://metacpan.org/search?q=jacode.pl

 Jacode4e - jacode.pl-like program for enterprise
 https://metacpan.org/pod/Jacode4e

 Jacode4e::RoundTrip - Jacode4e for round-trip conversion in JIS X 0213
 https://metacpan.org/pod/Jacode4e::RoundTrip

 Modern::Open - Autovivification, Autodie, and 3-args open support
 https://metacpan.org/pod/Modern::Open

 BackPAN
 http://backpan.perl.org/authors/id/I/IN/INA/

=head1 ACKNOWLEDGEMENTS

 I could make this software by good luck. I thank all stakeholders.

 I received character code table of KEIS, JEF, and JIPS by electronic data
 from Culti Co.,Ltd. Moreover, Culti Co.,Ltd. has allowed me to use it to
 make open source software.

 I thank Culti Co.,Ltd. once again.

=head1 HELLO WORLD

 To support JIS X 0213:2004,
 
     Using ghost character 1-55-27(it's me!), #'
 
 Found by JIS X 0208:1997,
 
     Was born in JIS C 6226-1978.
 
 Hello world,
 
     What do we hack, today?
 
                -- 1-55-27, 2018-01-27

=cut

1;

__DATA__
###################################################################################################################
# Jacode4e table
###################################################################################################################
#+++++++----------------------------------------------------------------------------------------------------------- CP932X, Extended CP932 to JIS X 0213 using 0x9C5A as single shift
#||||||| ++++------------------------------------------------------------------------------------------------------ Microsoft CP932, IANA Windows-31J
#||||||| |||| ++++------------------------------------------------------------------------------------------------- IBM CP932
#||||||| |||| |||| ++++-------------------------------------------------------------------------------------------- NEC CP932
#||||||| |||| |||| |||| ++++--------------------------------------------------------------------------------------- JISC Shift_JIS-2004
#||||||| |||| |||| |||| |||| ++++---------------------------------------------------------------------------------- IBM CP00930(CP00290+CP00300), CCSID 5026 katakana
#||||||| |||| |||| |||| |||| |||| ++++----------------------------------------------------------------------------- HITACHI KEIS78
#||||||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------------ HITACHI KEIS83
#||||||| |||| |||| |||| |||| |||| |||| |||| ++++------------------------------------------------------------------- HITACHI KEIS90
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| ++++-------------------------------------------------------------- FUJITSU JEF
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++--------------------------------------------------------- NEC JIPS(J)
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++---------------------------------------------------- NEC JIPS(E)
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ++++----------------------------------------------- UNISYS LetsJ
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| +++++++++------------------------------------- Unicode
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| ++++++++++++------------------------ UTF-8.0 (aka UTF-8)
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| ++++++++++++----------- UTF-8.1
#||||||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||||||| |||||||||||| |||||||||||| ++++++++-- UTF-8-SPUA-JP, JIS X 0213 on SPUA ordered by JIS level, plane, row, cell
#2345678 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 1234 123456789 123456789012 123456789012 12345678
#VVVVVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVV VVVVVVVVV VVVVVVVVVVVV VVVVVVVVVVVV VVVVVVVV
00       00   00   00   00   00   00   00   00   00   00   00   00   0000      00           00           F3B08080
01       01   01   01   01   01   01   01   01   01   01   01   01   0001      01           01           F3B08081
02       02   02   02   02   02   02   02   02   02   02   02   02   0002      02           02           F3B08082
03       03   03   03   03   03   03   03   03   03   03   03   03   0003      03           03           F3B08083
04       04   04   04   04   37   37   37   37   37   04   37   04   0004      04           04           F3B08084
05       05   05   05   05   2D   2D   2D   2D   2D   05   2D   05   0005      05           05           F3B08085
06       06   06   06   06   2E   2E   2E   2E   2E   06   2E   06   0006      06           06           F3B08086
07       07   07   07   07   2F   2F   2F   2F   2F   07   2F   07   0007      07           07           F3B08087
08       08   08   08   08   16   16   16   16   16   08   16   08   0008      08           08           F3B08088
09       09   09   09   09   05   05   05   05   05   09   05   09   0009      09           09           F3B08089
0A       0A   0A   0A   0A   15   15   15   15   15   0A   15   0A   000A      0A           0A           F3B0808A
0B       0B   0B   0B   0B   0B   0B   0B   0B   0B   0B   0B   0B   000B      0B           0B           F3B0808B
0C       0C   0C   0C   0C   0C   0C   0C   0C   0C   0C   0C   0C   000C      0C           0C           F3B0808C
0D       0D   0D   0D   0D   0D   0D   0D   0D   0D   0D   0D   0D   000D      0D           0D           F3B0808D
0E       0E   0E   0E   0E   0E   0E   0E   0E   0E   0E   0E   0E   000E      0E           0E           F3B0808E
0F       0F   0F   0F   0F   0F   0F   0F   0F   0F   0F   0F   0F   000F      0F           0F           F3B0808F
10       10   10   10   10   10   10   10   10   10   10   10   10   0010      10           10           F3B08090
11       11   11   11   11   11   11   11   11   11   11   11   11   0011      11           11           F3B08091
12       12   12   12   12   12   12   12   12   12   12   12   12   0012      12           12           F3B08092
13       13   13   13   13   13   13   13   13   13   13   13   13   0013      13           13           F3B08093
14       14   14   14   14   3C   3C   3C   3C   3C   14   3C   14   0014      14           14           F3B08094
15       15   15   15   15   3D   3D   3D   3D   3D   15   3D   15   0015      15           15           F3B08095
16       16   16   16   16   32   32   32   32   32   16   32   16   0016      16           16           F3B08096
17       17   17   17   17   26   26   26   26   26   17   26   17   0017      17           17           F3B08097
18       18   18   18   18   18   18   18   18   18   18   18   18   0018      18           18           F3B08098
19       19   19   19   19   19   19   19   19   19   19   19   19   0019      19           19           F3B08099
1A       1A   1A   1A   1A   3F   3F   3F   3F   3F   1A   3F   1A   001A      1A           1A           F3B0809A
1B       1B   1B   1B   1B   27   27   27   27   27   1B   27   1B   001B      1B           1B           F3B0809B
1C       1C   1C   1C   1C   1C   1C   1C   1C   1C   1C   1C   1C   001C      1C           1C           F3B0809C
1D       1D   1D   1D   1D   1D   1D   1D   1D   1D   1D   1D   1D   001D      1D           1D           F3B0809D
1E       1E   1E   1E   1E   1E   1E   1E   1E   1E   1E   1E   1E   001E      1E           1E           F3B0809E
1F       1F   1F   1F   1F   1F   1F   1F   1F   1F   1F   1F   1F   001F      1F           1F           F3B0809F
20       20   20   20   20   40   40   40   40   40   20   40   20   0020      20           20           F3B080A0
21       21   21   21   21   5A   4F   4F   4F   4F   21   4F   21   0021      21           21           F3B080A1
22       22   22   22   22   7F   7F   7F   7F   7F   22   7F   22   0022      22           22           F3B080A2
23       23   23   23   23   7B   7B   7B   7B   7B   23   7B   23   0023      23           23           F3B080A3
24       24   24   24   24   E0   E0   E0   E0   E0   24   E0   24   0024      24           24           F3B080A4
25       25   25   25   25   6C   6C   6C   6C   6C   25   6C   25   0025      25           25           F3B080A5
26       26   26   26   26   50   50   50   50   50   26   50   26   0026      26           26           F3B080A6
27       27   27   27   27   7D   7D   7D   7D   7D   27   7D   27   0027      27           27           F3B080A7
28       28   28   28   28   4D   4D   4D   4D   4D   28   4D   28   0028      28           28           F3B080A8
29       29   29   29   29   5D   5D   5D   5D   5D   29   5D   29   0029      29           29           F3B080A9
2A       2A   2A   2A   2A   5C   5C   5C   5C   5C   2A   5C   2A   002A      2A           2A           F3B080AA
2B       2B   2B   2B   2B   4E   4E   4E   4E   4E   2B   4E   2B   002B      2B           2B           F3B080AB
2C       2C   2C   2C   2C   6B   6B   6B   6B   6B   2C   6B   2C   002C      2C           2C           F3B080AC
2D       2D   2D   2D   2D   60   60   60   60   60   2D   60   2D   002D      2D           2D           F3B080AD
2E       2E   2E   2E   2E   4B   4B   4B   4B   4B   2E   4B   2E   002E      2E           2E           F3B080AE
2F       2F   2F   2F   2F   61   61   61   61   61   2F   61   2F   002F      2F           2F           F3B080AF
30       30   30   30   30   F0   F0   F0   F0   F0   30   F0   30   0030      30           30           F3B080B0
31       31   31   31   31   F1   F1   F1   F1   F1   31   F1   31   0031      31           31           F3B080B1
32       32   32   32   32   F2   F2   F2   F2   F2   32   F2   32   0032      32           32           F3B080B2
33       33   33   33   33   F3   F3   F3   F3   F3   33   F3   33   0033      33           33           F3B080B3
34       34   34   34   34   F4   F4   F4   F4   F4   34   F4   34   0034      34           34           F3B080B4
35       35   35   35   35   F5   F5   F5   F5   F5   35   F5   35   0035      35           35           F3B080B5
36       36   36   36   36   F6   F6   F6   F6   F6   36   F6   36   0036      36           36           F3B080B6
37       37   37   37   37   F7   F7   F7   F7   F7   37   F7   37   0037      37           37           F3B080B7
38       38   38   38   38   F8   F8   F8   F8   F8   38   F8   38   0038      38           38           F3B080B8
39       39   39   39   39   F9   F9   F9   F9   F9   39   F9   39   0039      39           39           F3B080B9
3A       3A   3A   3A   3A   7A   7A   7A   7A   7A   3A   7A   3A   003A      3A           3A           F3B080BA
3B       3B   3B   3B   3B   5E   5E   5E   5E   5E   3B   5E   3B   003B      3B           3B           F3B080BB
3C       3C   3C   3C   3C   4C   4C   4C   4C   4C   3C   4C   3C   003C      3C           3C           F3B080BC
3D       3D   3D   3D   3D   7E   7E   7E   7E   7E   3D   7E   3D   003D      3D           3D           F3B080BD
3E       3E   3E   3E   3E   6E   6E   6E   6E   6E   3E   6E   3E   003E      3E           3E           F3B080BE
3F       3F   3F   3F   3F   6F   6F   6F   6F   6F   3F   6F   3F   003F      3F           3F           F3B080BF
40       40   40   40   40   7C   7C   7C   7C   7C   40   7C   40   0040      40           40           F3B08180
41       41   41   41   41   C1   C1   C1   C1   C1   41   C1   41   0041      41           41           F3B08181
42       42   42   42   42   C2   C2   C2   C2   C2   42   C2   42   0042      42           42           F3B08182
43       43   43   43   43   C3   C3   C3   C3   C3   43   C3   43   0043      43           43           F3B08183
44       44   44   44   44   C4   C4   C4   C4   C4   44   C4   44   0044      44           44           F3B08184
45       45   45   45   45   C5   C5   C5   C5   C5   45   C5   45   0045      45           45           F3B08185
46       46   46   46   46   C6   C6   C6   C6   C6   46   C6   46   0046      46           46           F3B08186
47       47   47   47   47   C7   C7   C7   C7   C7   47   C7   47   0047      47           47           F3B08187
48       48   48   48   48   C8   C8   C8   C8   C8   48   C8   48   0048      48           48           F3B08188
49       49   49   49   49   C9   C9   C9   C9   C9   49   C9   49   0049      49           49           F3B08189
4A       4A   4A   4A   4A   D1   D1   D1   D1   D1   4A   D1   4A   004A      4A           4A           F3B0818A
4B       4B   4B   4B   4B   D2   D2   D2   D2   D2   4B   D2   4B   004B      4B           4B           F3B0818B
4C       4C   4C   4C   4C   D3   D3   D3   D3   D3   4C   D3   4C   004C      4C           4C           F3B0818C
4D       4D   4D   4D   4D   D4   D4   D4   D4   D4   4D   D4   4D   004D      4D           4D           F3B0818D
4E       4E   4E   4E   4E   D5   D5   D5   D5   D5   4E   D5   4E   004E      4E           4E           F3B0818E
4F       4F   4F   4F   4F   D6   D6   D6   D6   D6   4F   D6   4F   004F      4F           4F           F3B0818F
50       50   50   50   50   D7   D7   D7   D7   D7   50   D7   50   0050      50           50           F3B08190
51       51   51   51   51   D8   D8   D8   D8   D8   51   D8   51   0051      51           51           F3B08191
52       52   52   52   52   D9   D9   D9   D9   D9   52   D9   52   0052      52           52           F3B08192
53       53   53   53   53   E2   E2   E2   E2   E2   53   E2   53   0053      53           53           F3B08193
54       54   54   54   54   E3   E3   E3   E3   E3   54   E3   54   0054      54           54           F3B08194
55       55   55   55   55   E4   E4   E4   E4   E4   55   E4   55   0055      55           55           F3B08195
56       56   56   56   56   E5   E5   E5   E5   E5   56   E5   56   0056      56           56           F3B08196
57       57   57   57   57   E6   E6   E6   E6   E6   57   E6   57   0057      57           57           F3B08197
58       58   58   58   58   E7   E7   E7   E7   E7   58   E7   58   0058      58           58           F3B08198
59       59   59   59   59   E8   E8   E8   E8   E8   59   E8   59   0059      59           59           F3B08199
5A       5A   5A   5A   5A   E9   E9   E9   E9   E9   5A   E9   5A   005A      5A           5A           F3B0819A
5B       5B   5B   5B   5B   70   4A   4A   4A   4A   5B   4A   5B   005B      5B           5B           F3B0819B
5C       5C   5C   5C   5C   5B   5B   5B   5B   5B   5C   5B   5C   005C      5C           5C           F3B0819C
5D       5D   5D   5D   5D   80   5A   5A   5A   5A   5D   5A   5D   005D      5D           5D           F3B0819D
5E       5E   5E   5E   5E   B0   5F   5F   5F   5F   5E   5F   5E   005E      5E           5E           F3B0819E
5F       5F   5F   5F   5F   6D   6D   6D   6D   6D   5F   6D   5F   005F      5F           5F           F3B0819F
60       60   60   60   60   79   79   79   79   79   60   79   60   0060      60           60           F3B081A0
61       61   61   61   61   62   59   59   59   59   61   57   61   0061      61           61           F3B081A1
62       62   62   62   62   63   62   62   62   62   62   59   62   0062      62           62           F3B081A2
63       63   63   63   63   64   63   63   63   63   63   62   63   0063      63           63           F3B081A3
64       64   64   64   64   65   64   64   64   64   64   63   64   0064      64           64           F3B081A4
65       65   65   65   65   66   65   65   65   65   65   64   65   0065      65           65           F3B081A5
66       66   66   66   66   67   66   66   66   66   66   65   66   0066      66           66           F3B081A6
67       67   67   67   67   68   67   67   67   67   67   66   67   0067      67           67           F3B081A7
68       68   68   68   68   69   68   68   68   68   68   67   68   0068      68           68           F3B081A8
69       69   69   69   69   71   69   69   69   69   69   68   69   0069      69           69           F3B081A9
6A       6A   6A   6A   6A   72   70   70   70   70   6A   69   6A   006A      6A           6A           F3B081AA
6B       6B   6B   6B   6B   73   71   71   71   71   6B   70   6B   006B      6B           6B           F3B081AB
6C       6C   6C   6C   6C   74   72   72   72   72   6C   71   6C   006C      6C           6C           F3B081AC
6D       6D   6D   6D   6D   75   73   73   73   73   6D   72   6D   006D      6D           6D           F3B081AD
6E       6E   6E   6E   6E   76   74   74   74   74   6E   73   6E   006E      6E           6E           F3B081AE
6F       6F   6F   6F   6F   77   75   75   75   75   6F   74   6F   006F      6F           6F           F3B081AF
70       70   70   70   70   78   76   76   76   76   70   75   70   0070      70           70           F3B081B0
71       71   71   71   71   8B   77   77   77   77   71   76   71   0071      71           71           F3B081B1
72       72   72   72   72   9B   78   78   78   78   72   77   72   0072      72           72           F3B081B2
73       73   73   73   73   AB   80   80   80   80   73   78   73   0073      73           73           F3B081B3
74       74   74   74   74   B3   8B   8B   8B   8B   74   80   74   0074      74           74           F3B081B4
75       75   75   75   75   B4   9B   9B   9B   9B   75   8B   75   0075      75           75           F3B081B5
76       76   76   76   76   B5   9C   9C   9C   9C   76   9B   76   0076      76           76           F3B081B6
77       77   77   77   77   B6   A0   A0   A0   A0   77   9C   77   0077      77           77           F3B081B7
78       78   78   78   78   B7   AB   AB   AB   AB   78   A0   78   0078      78           78           F3B081B8
79       79   79   79   79   B8   B0   B0   B0   B0   79   AB   79   0079      79           79           F3B081B9
7A       7A   7A   7A   7A   B9   B1   B1   B1   B1   7A   B0   7A   007A      7A           7A           F3B081BA
7B       7B   7B   7B   7B   C0   C0   C0   C0   C0   7B   C0   7B   007B      7B           7B           F3B081BB
7C       7C   7C   7C   7C   4F   6A   6A   6A   6A   7C   6A   7C   007C      7C           7C           F3B081BC
7D       7D   7D   7D   7D   D0   D0   D0   D0   D0   7D   D0   7D   007D      7D           7D           F3B081BD
7E       7E   7E   7E   7E   A0   A1   A1   A1   A1   7E   A1   7E   007E      7E           7E           F3B081BE
7F       7F   7F   7F   7F   07   07   07   07   07   7F   07   7F   007F      7F           7F           F3B081BF
80       80   80   80   80   20   20   20   20   20   80   20   80    ---        --           --         F3B08280
81       81   81   81   81   21   21   21   21   21   81   21   81    ---        --           --         F3B08281
82       82   82   82   82   22   22   22   22   22   82   22   82    ---        --           --         F3B08282
83       83   83   83   83   23   23   23   23   23   83   23   83    ---        --           --         F3B08283
84       84   84   84   84   24   24   24   24   24   84   24   84    ---        --           --         F3B08284
85       85   85   85   85   25   25   25   25   25   85   25   85    ---        --           --         F3B08285
86       86   86   86   86   06   06   06   06   06   86   06   86    ---        --           --         F3B08286
87       87   87   87   87   17   17   17   17   17   87   17   87    ---        --           --         F3B08287
88       88   88   88   88   28   28   28   28   28   88   28   88    ---        --           --         F3B08288
89       89   89   89   89   29   29   29   29   29   89   29   89    ---        --           --         F3B08289
8A       8A   8A   8A   8A   2A   2A   2A   2A   2A   8A   2A   8A    ---        --           --         F3B0828A
8B       8B   8B   8B   8B   2B   2B   2B   2B   2B   8B   2B   8B    ---        --           --         F3B0828B
8C       8C   8C   8C   8C   2C   2C   2C   2C   2C   8C   2C   8C    ---        --           --         F3B0828C
8D       8D   8D   8D   8D   09   09   09   09   09   8D   09   8D    ---        --           --         F3B0828D
8E       8E   8E   8E   8E   0A   0A   0A   0A   0A   8E   0A   8E    ---        --           --         F3B0828E
8F       8F   8F   8F   8F   1B   1B   1B   1B   1B   8F   1B   8F    ---        --           --         F3B0828F
90       90   90   90   90   30   30   30   30   30   90   30   90    ---        --           --         F3B08290
91       91   91   91   91   31   31   31   31   31   91   31   91    ---        --           --         F3B08291
92       92   92   92   92   1A   1A   1A   1A   1A   92   1A   92    ---        --           --         F3B08292
93       93   93   93   93   33   33   33   33   33   93   33   93    ---        --           --         F3B08293
94       94   94   94   94   34   34   34   34   34   94   34   94    ---        --           --         F3B08294
95       95   95   95   95   35   35   35   35   35   95   35   95    ---        --           --         F3B08295
96       96   96   96   96   36   36   36   36   36   96   36   96    ---        --           --         F3B08296
97       97   97   97   97   08   08   08   08   08   97   08   97    ---        --           --         F3B08297
98       98   98   98   98   38   38   38   38   38   98   38   98    ---        --           --         F3B08298
99       99   99   99   99   39   39   39   39   39   99   39   99    ---        --           --         F3B08299
9A       9A   9A   9A   9A   3A   3A   3A   3A   3A   9A   3A   9A    ---        --           --         F3B0829A
9B       9B   9B   9B   9B   3B   3B   3B   3B   3B   9B   3B   9B    ---        --           --         F3B0829B
9C       9C   9C   9C   9C   04   04   04   04   04   9C   04   9C    ---        --           --         F3B0829C
9D       9D   9D   9D   9D   14   14   14   14   14   9D   14   9D    ---        --           --         F3B0829D
9E       9E   9E   9E   9E   3E   3E   3E   3E   3E   9E   3E   9E    ---        --           --         F3B0829E
9F       9F   9F   9F   9F   4A   E1   E1   E1   E1   9F   E1   9F    ---        --           --         F3B0829F
A0       A0   A0   A0   A0   6A   57   57   57   57   A0   B1   A0    ---        --           --         F3B082A0
A1       A1   A1   A1   A1   41   41   41   41   41   A1   41   A1   FF61      EFBDA1       EFBDA1       F3B082A1
A2       A2   A2   A2   A2   42   42   42   42   42   A2   42   A2   FF62      EFBDA2       EFBDA2       F3B082A2
A3       A3   A3   A3   A3   43   43   43   43   43   A3   43   A3   FF63      EFBDA3       EFBDA3       F3B082A3
A4       A4   A4   A4   A4   44   44   44   44   44   A4   44   A4   FF64      EFBDA4       EFBDA4       F3B082A4
A5       A5   A5   A5   A5   45   45   45   45   45   A5   45   A5   FF65      EFBDA5       EFBDA5       F3B082A5
A6       A6   A6   A6   A6   46   46   46   46   46   A6   46   A6   FF66      EFBDA6       EFBDA6       F3B082A6
A7       A7   A7   A7   A7   47   47   47   47   47   A7   47   A7   FF67      EFBDA7       EFBDA7       F3B082A7
A8       A8   A8   A8   A8   48   48   48   48   48   A8   48   A8   FF68      EFBDA8       EFBDA8       F3B082A8
A9       A9   A9   A9   A9   49   49   49   49   49   A9   49   A9   FF69      EFBDA9       EFBDA9       F3B082A9
AA       AA   AA   AA   AA   51   51   51   51   51   AA   51   AA   FF6A      EFBDAA       EFBDAA       F3B082AA
AB       AB   AB   AB   AB   52   52   52   52   52   AB   52   AB   FF6B      EFBDAB       EFBDAB       F3B082AB
AC       AC   AC   AC   AC   53   53   53   53   53   AC   53   AC   FF6C      EFBDAC       EFBDAC       F3B082AC
AD       AD   AD   AD   AD   54   54   54   54   54   AD   54   AD   FF6D      EFBDAD       EFBDAD       F3B082AD
AE       AE   AE   AE   AE   55   55   55   55   55   AE   55   AE   FF6E      EFBDAE       EFBDAE       F3B082AE
AF       AF   AF   AF   AF   56   56   56   56   56   AF   56   AF   FF6F      EFBDAF       EFBDAF       F3B082AF
B0       B0   B0   B0   B0   58   58   58   58   58   B0   58   B0   FF70      EFBDB0       EFBDB0       F3B082B0
B1       B1   B1   B1   B1   81   81   81   81   81   B1   81   B1   FF71      EFBDB1       EFBDB1       F3B082B1
B2       B2   B2   B2   B2   82   82   82   82   82   B2   82   B2   FF72      EFBDB2       EFBDB2       F3B082B2
B3       B3   B3   B3   B3   83   83   83   83   83   B3   83   B3   FF73      EFBDB3       EFBDB3       F3B082B3
B4       B4   B4   B4   B4   84   84   84   84   84   B4   84   B4   FF74      EFBDB4       EFBDB4       F3B082B4
B5       B5   B5   B5   B5   85   85   85   85   85   B5   85   B5   FF75      EFBDB5       EFBDB5       F3B082B5
B6       B6   B6   B6   B6   86   86   86   86   86   B6   86   B6   FF76      EFBDB6       EFBDB6       F3B082B6
B7       B7   B7   B7   B7   87   87   87   87   87   B7   87   B7   FF77      EFBDB7       EFBDB7       F3B082B7
B8       B8   B8   B8   B8   88   88   88   88   88   B8   88   B8   FF78      EFBDB8       EFBDB8       F3B082B8
B9       B9   B9   B9   B9   89   89   89   89   89   B9   89   B9   FF79      EFBDB9       EFBDB9       F3B082B9
BA       BA   BA   BA   BA   8A   8A   8A   8A   8A   BA   8A   BA   FF7A      EFBDBA       EFBDBA       F3B082BA
BB       BB   BB   BB   BB   8C   8C   8C   8C   8C   BB   8C   BB   FF7B      EFBDBB       EFBDBB       F3B082BB
BC       BC   BC   BC   BC   8D   8D   8D   8D   8D   BC   8D   BC   FF7C      EFBDBC       EFBDBC       F3B082BC
BD       BD   BD   BD   BD   8E   8E   8E   8E   8E   BD   8E   BD   FF7D      EFBDBD       EFBDBD       F3B082BD
BE       BE   BE   BE   BE   8F   8F   8F   8F   8F   BE   8F   BE   FF7E      EFBDBE       EFBDBE       F3B082BE
BF       BF   BF   BF   BF   90   90   90   90   90   BF   90   BF   FF7F      EFBDBF       EFBDBF       F3B082BF
C0       C0   C0   C0   C0   91   91   91   91   91   C0   91   C0   FF80      EFBE80       EFBE80       F3B08380
C1       C1   C1   C1   C1   92   92   92   92   92   C1   92   C1   FF81      EFBE81       EFBE81       F3B08381
C2       C2   C2   C2   C2   93   93   93   93   93   C2   93   C2   FF82      EFBE82       EFBE82       F3B08382
C3       C3   C3   C3   C3   94   94   94   94   94   C3   94   C3   FF83      EFBE83       EFBE83       F3B08383
C4       C4   C4   C4   C4   95   95   95   95   95   C4   95   C4   FF84      EFBE84       EFBE84       F3B08384
C5       C5   C5   C5   C5   96   96   96   96   96   C5   96   C5   FF85      EFBE85       EFBE85       F3B08385
C6       C6   C6   C6   C6   97   97   97   97   97   C6   97   C6   FF86      EFBE86       EFBE86       F3B08386
C7       C7   C7   C7   C7   98   98   98   98   98   C7   98   C7   FF87      EFBE87       EFBE87       F3B08387
C8       C8   C8   C8   C8   99   99   99   99   99   C8   99   C8   FF88      EFBE88       EFBE88       F3B08388
C9       C9   C9   C9   C9   9A   9A   9A   9A   9A   C9   9A   C9   FF89      EFBE89       EFBE89       F3B08389
CA       CA   CA   CA   CA   9D   9D   9D   9D   9D   CA   9D   CA   FF8A      EFBE8A       EFBE8A       F3B0838A
CB       CB   CB   CB   CB   9E   9E   9E   9E   9E   CB   9E   CB   FF8B      EFBE8B       EFBE8B       F3B0838B
CC       CC   CC   CC   CC   9F   9F   9F   9F   9F   CC   9F   CC   FF8C      EFBE8C       EFBE8C       F3B0838C
CD       CD   CD   CD   CD   A2   A2   A2   A2   A2   CD   A2   CD   FF8D      EFBE8D       EFBE8D       F3B0838D
CE       CE   CE   CE   CE   A3   A3   A3   A3   A3   CE   A3   CE   FF8E      EFBE8E       EFBE8E       F3B0838E
CF       CF   CF   CF   CF   A4   A4   A4   A4   A4   CF   A4   CF   FF8F      EFBE8F       EFBE8F       F3B0838F
D0       D0   D0   D0   D0   A5   A5   A5   A5   A5   D0   A5   D0   FF90      EFBE90       EFBE90       F3B08390
D1       D1   D1   D1   D1   A6   A6   A6   A6   A6   D1   A6   D1   FF91      EFBE91       EFBE91       F3B08391
D2       D2   D2   D2   D2   A7   A7   A7   A7   A7   D2   A7   D2   FF92      EFBE92       EFBE92       F3B08392
D3       D3   D3   D3   D3   A8   A8   A8   A8   A8   D3   A8   D3   FF93      EFBE93       EFBE93       F3B08393
D4       D4   D4   D4   D4   A9   A9   A9   A9   A9   D4   A9   D4   FF94      EFBE94       EFBE94       F3B08394
D5       D5   D5   D5   D5   AA   AA   AA   AA   AA   D5   AA   D5   FF95      EFBE95       EFBE95       F3B08395
D6       D6   D6   D6   D6   AC   AC   AC   AC   AC   D6   AC   D6   FF96      EFBE96       EFBE96       F3B08396
D7       D7   D7   D7   D7   AD   AD   AD   AD   AD   D7   AD   D7   FF97      EFBE97       EFBE97       F3B08397
D8       D8   D8   D8   D8   AE   AE   AE   AE   AE   D8   AE   D8   FF98      EFBE98       EFBE98       F3B08398
D9       D9   D9   D9   D9   AF   AF   AF   AF   AF   D9   AF   D9   FF99      EFBE99       EFBE99       F3B08399
DA       DA   DA   DA   DA   BA   BA   BA   BA   BA   DA   BA   DA   FF9A      EFBE9A       EFBE9A       F3B0839A
DB       DB   DB   DB   DB   BB   BB   BB   BB   BB   DB   BB   DB   FF9B      EFBE9B       EFBE9B       F3B0839B
DC       DC   DC   DC   DC   BC   BC   BC   BC   BC   DC   BC   DC   FF9C      EFBE9C       EFBE9C       F3B0839C
DD       DD   DD   DD   DD   BD   BD   BD   BD   BD   DD   BD   DD   FF9D      EFBE9D       EFBE9D       F3B0839D
DE       DE   DE   DE   DE   BE   BE   BE   BE   BE   DE   BE   DE   FF9E      EFBE9E       EFBE9E       F3B0839E
DF       DF   DF   DF   DF   BF   BF   BF   BF   BF   DF   BF   DF   FF9F      EFBE9F       EFBE9F       F3B0839F
E0       E0   E0   E0   E0   57   B2   B2   B2   B2   E0   B2   E0    ---        --           --         F3B083A0
E1       E1   E1   E1   E1   59   B3   B3   B3   B3   E1   B3   E1    ---        --           --         F3B083A1
E2       E2   E2   E2   E2   5F   B4   B4   B4   B4   E2   B4   E2    ---        --           --         F3B083A2
E3       E3   E3   E3   E3   9C   B5   B5   B5   B5   E3   B5   E3    ---        --           --         F3B083A3
E4       E4   E4   E4   E4   A1   B6   B6   B6   B6   E4   B6   E4    ---        --           --         F3B083A4
E5       E5   E5   E5   E5   B1   B7   B7   B7   B7   E5   B7   E5    ---        --           --         F3B083A5
E6       E6   E6   E6   E6   B2   B8   B8   B8   B8   E6   B8   E6    ---        --           --         F3B083A6
E7       E7   E7   E7   E7   E1   B9   B9   B9   B9   E7   B9   E7    ---        --           --         F3B083A7
E8       E8   E8   E8   E8   CA   CA   CA   CA   CA   E8   CA   E8    ---        --           --         F3B083A8
E9       E9   E9   E9   E9   CB   CB   CB   CB   CB   E9   CB   E9    ---        --           --         F3B083A9
EA       EA   EA   EA   EA   CC   CC   CC   CC   CC   EA   CC   EA    ---        --           --         F3B083AA
EB       EB   EB   EB   EB   CD   CD   CD   CD   CD   EB   CD   EB    ---        --           --         F3B083AB
EC       EC   EC   EC   EC   CE   CE   CE   CE   CE   EC   CE   EC    ---        --           --         F3B083AC
ED       ED   ED   ED   ED   CF   CF   CF   CF   CF   ED   CF   ED    ---        --           --         F3B083AD
EE       EE   EE   EE   EE   DA   DA   DA   DA   DA   EE   DA   EE    ---        --           --         F3B083AE
EF       EF   EF   EF   EF   DB   DB   DB   DB   DB   EF   DB   EF    ---        --           --         F3B083AF
F0       F0   F0   F0   F0   DC   DC   DC   DC   DC   F0   DC   F0    ---        --           --         F3B083B0
F1       F1   F1   F1   F1   DD   DD   DD   DD   DD   F1   DD   F1    ---        --           --         F3B083B1
F2       F2   F2   F2   F2   DE   DE   DE   DE   DE   F2   DE   F2    ---        --           --         F3B083B2
F3       F3   F3   F3   F3   DF   DF   DF   DF   DF   F3   DF   F3    ---        --           --         F3B083B3
F4       F4   F4   F4   F4   EA   EA   EA   EA   EA   F4   EA   F4    ---        --           --         F3B083B4
F5       F5   F5   F5   F5   EB   EB   EB   EB   EB   F5   EB   F5    ---        --           --         F3B083B5
F6       F6   F6   F6   F6   EC   EC   EC   EC   EC   F6   EC   F6    ---        --           --         F3B083B6
F7       F7   F7   F7   F7   ED   ED   ED   ED   ED   F7   ED   F7    ---        --           --         F3B083B7
F8       F8   F8   F8   F8   EE   EE   EE   EE   EE   F8   EE   F8    ---        --           --         F3B083B8
F9       F9   F9   F9   F9   EF   EF   EF   EF   EF   F9   EF   F9    ---        --           --         F3B083B9
FA       FA   FA   FA   FA   FA   FA   FA   FA   FA   FA   FA   FA    ---        --           --         F3B083BA
FB       FB   FB   FB   FB   FB   FB   FB   FB   FB   FB   FB   FB    ---        --           --         F3B083BB
FC       FC   FC   FC   FC   FC   FC   FC   FC   FC   FC   FC   FC    ---        --           --         F3B083BC
FD       FD   FD   FD   FD   FD   FD   FD   FD   FD   FD   FD   FD    ---        --           --         F3B083BD
FE       FE   FE   FE   FE   FE   FE   FE   FE   FE   FE   FE   FE    ---        --           --         F3B083BE
FF       FF   FF   FF   FF   FF   FF   FF   FF   FF   FF   FF   FF    ---        --           --         F3B083BF
#
# FACOM JEF Character code index dictionary, 99FR-0012-2 and 99FR-0012-3
# show us that DBCS SPACE code is "\x40\x40", but nobody use so.
#
8140     8140 8140 8140 8140 4040 A1A1 A1A1 A1A1 A1A1 2121 4F4F 2020 3000      E38080       E38080       F3B08480
8141     8141 8141 8141 8141 4344 A1A2 A1A2 A1A2 A1A2 2122 4F7F A1A2 3001      E38081       E38081       F3B08481
8142     8142 8142 8142 8142 4341 A1A3 A1A3 A1A3 A1A3 2123 4F7B A1A3 3002      E38082       E38082       F3B08482
8143     8143 8143 8143 8143 426B A1A4 A1A4 A1A4 A1A4 2124 4FE0 A1A4 FF0C      EFBC8C       EFBC8C       F3B08483
8144     8144 8144 8144 8144 424B A1A5 A1A5 A1A5 A1A5 2125 4F6C A1A5 FF0E      EFBC8E       EFBC8E       F3B08484
8145     8145 8145 8145 8145 4345 A1A6 A1A6 A1A6 A1A6 2126 4F50 A1A6 30FB      E383BB       E383BB       F3B08485
8146     8146 8146 8146 8146 427A A1A7 A1A7 A1A7 A1A7 2127 4F7D A1A7 FF1A      EFBC9A       EFBC9A       F3B08486
8147     8147 8147 8147 8147 425E A1A8 A1A8 A1A8 A1A8 2128 4F4D A1A8 FF1B      EFBC9B       EFBC9B       F3B08487
8148     8148 8148 8148 8148 426F A1A9 A1A9 A1A9 A1A9 2129 4F5D A1A9 FF1F      EFBC9F       EFBC9F       F3B08488
8149     8149 8149 8149 8149 425A A1AA A1AA A1AA A1AA 212A 4F5C A1AA FF01      EFBC81       EFBC81       F3B08489
814A     814A 814A 814A 814A 43BE A1AB A1AB A1AB A1AB 212B 4F4E A1AB 309B      E3829B       E3829B       F3B0848A
814B     814B 814B 814B 814B 43BF A1AC A1AC A1AC A1AC 212C 4F6B A1AC 309C      E3829C       E3829C       F3B0848B
814C     814C 814C 814C 814C 4450 A1AD A1AD A1AD A1AD 212D 4F60 A1AD 00B4      C2B4         C2B4         F3B0848C
814D     814D 814D 814D 814D 4279 A1AE A1AE A1AE A1AE 212E 4F4B A1AE FF40      EFBD80       EFBD80       F3B0848D
814E     814E 814E 814E 814E 4460 A1AF A1AF A1AF A1AF 212F 4F61 A1AF 00A8      C2A8         C2A8         F3B0848E
814F     814F 814F 814F 814F 4470 A1B0 A1B0 A1B0 A1B0 2130 4FF0 A1B0 FF3E      EFBCBE       EFBCBE       F3B0848F
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
8150     8150 8150 8150 8150 42A1 A1B1 A1B1 A1B1 A1B1 2131 4FF1 A1B1 FFE3      EFBFA3       EFBFA3       F3B08490
8151     8151 8151 8151 8151 426D A1B2 A1B2 A1B2 A1B2 2132 4FF2 A1B2 FF3F      EFBCBF       EFBCBF       F3B08491
8152     8152 8152 8152 8152 43DC A1B3 A1B3 A1B3 A1B3 2133 4FF3 A1B3 30FD      E383BD       E383BD       F3B08492
8153     8153 8153 8153 8153 43DD A1B4 A1B4 A1B4 A1B4 2134 4FF4 A1B4 30FE      E383BE       E383BE       F3B08493
8154     8154 8154 8154 8154 44DC A1B5 A1B5 A1B5 A1B5 2135 4FF5 A1B5 309D      E3829D       E3829D       F3B08494
8155     8155 8155 8155 8155 44DD A1B6 A1B6 A1B6 A1B6 2136 4FF6 A1B6 309E      E3829E       E3829E       F3B08495
8156     8156 8156 8156 8156 445B A1B7 A1B7 A1B7 A1B7 2137 4FF7 A1B7 3003      E38083       E38083       F3B08496
8157     8157 8157 8157 8157 445C A1B8 A1B8 A1B8 A1B8 2138 4FF8 A1B8 4EDD      E4BB9D       E4BB9D       F3B08497
8158     8158 8158 8158 8158 445D A1B9 A1B9 A1B9 A1B9 2139 4FF9 A1B9 3005      E38085       E38085       F3B08498
8159     8159 8159 8159 8159 445E A1BA A1BA A1BA A1BA 213A 4F7A A1BA 3006      E38086       E38086       F3B08499
815A     815A 815A 815A 815A 445F A1BB A1BB A1BB A1BB 213B 4F5E A1BB 3007      E38087       E38087       F3B0849A
815B     815B 815B 815B 815B 4358 A1BC A1BC A1BC A1BC 213C 4F4C A1BC 30FC      E383BC       E383BC       F3B0849B
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
815C     815C 815C 815C  --  DDB7 A1BD A1BD A1BD A1BD 213D 4F7E A1BD 2015      E28095       E28094       F3B0849C
815D     815D 815D 815D 815D 445A A1BE A1BE A1BE A1BE 213E 4F6E A1BE 2010      E28090       E28090       F3B0849D
815E     815E 815E 815E 815E 4261 A1BF A1BF A1BF A1BF 213F 4F6F A1BF FF0F      EFBC8F       EFBC8F       F3B0849E
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
815F     815F 815F 815F  --  43E0 A1C0 A1C0 A1C0 A1C0 2140 4F7C  --  FF3C      EFBCBC       EFBCBC       F3B0849F
#
# Wave dash
# https://en.wikipedia.org/wiki/Wave_dash
#
# Standard     Release  Code-Point        Glyph     Note
# -----------------------------------------------------------------------------
# Unicode 1.0  1991     U+301C WAVE DASH  \/\ (VA)  The glyph was different from the original JIS C 6226 or JIS X 0208.
# Unicode 8.0  2015     U+301C WAVE DASH  /\/ (AV)  The glyph was fixed in Errata fixed in Unicode 8.0.0, The Unicode Consortium, 6 Oct 2014
# JIS C 6226   1978     01-33             /\/ (AV)  The wave was not stressed this much.
# JIS X 0208   1990     01-33             /\/ (AV)  
# JIS X 0213   2000     1-01-33           /\/ (AV)  
# -----------------------------------------------------------------------------
#
# Errata Fixed in Unicode 8.0.0
# http://www.unicode.org/versions/Unicode8.0.0/erratafixed.html
#
# 2014-October-6
# The character U+301C WAVE DASH was encoded to represent JIS C 6226-1978
# 1-33. However, the representative glyph is inverted relative to the
# original source. The glyph will be modified in future editions to match
# the JIS source. The glyph shown below on the left is the incorrect glyph.
# The corrected glyph is shown on the right. (See document L2/14-198 for
# further context for this change.) 
#
# (Japanese) Wave dash
# https://ja.wikipedia.org/wiki/%E6%B3%A2%E3%83%80%E3%83%83%E3%82%B7%E3%83%A5
#
# (Japanese) I'm just a programmer, cannot fix Unicode bug -- Dan Kogai-san 2006.05.10 11:00
# http://blog.livedoor.jp/dankogai/archives/50488765.html
#
# (Japanese) About WAVE DASH problem -- yasuoka-san 2006.05.10 18:29
# https://srad.jp/~yasuoka/journal/357074/
#
# (Japanese) Reason why Unicode's WAVE DASH example glyph was modified for the first time in 25 years
# https://internet.watch.impress.co.jp/docs/special/691658.html
#
8160     8160 8160 8160 8160 43A1 A1C1 A1C1 A1C1 A1C1 2141 4FC1 A1C1 301C      E3809C       E3809C       F3B084A0
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
8161     8161 8161 8161 81D2 DFE5 A1C2 A1C2 A1C2 A1C2 2142 4FC2  --  2225      E288A5       E28096       F3B084A1
8162     8162 8162 8162 8162 424F A1C3 A1C3 A1C3 A1C3 2143 4FC3 A1C3 FF5C      EFBD9C       EFBD9C       F3B084A2
8163     8163 8163 8163 8163 447F A1C4 A1C4 A1C4 A1C4 2144 4FC4 A1C4 2026      E280A6       E280A6       F3B084A3
8164     8164 8164 8164 8164 447E A1C5 A1C5 A1C5 A1C5 2145 4FC5 A1C5 2025      E280A5       E280A5       F3B084A4
8165     8165 8165 8165 8165 4461 A1C6 A1C6 A1C6 A1C6 2146 4FC6 A1C6 2018      E28098       E28098       F3B084A5
8166     8166 8166 8166 8166 4471 A1C7 A1C7 A1C7 A1C7 2147 4FC7 A1C7 2019      E28099       E28099       F3B084A6
8167     8167 8167 8167 8167 4462 A1C8 A1C8 A1C8 A1C8 2148 4FC8 A1C8 201C      E2809C       E2809C       F3B084A7
8168     8168 8168 8168 8168 4472 A1C9 A1C9 A1C9 A1C9 2149 4FC9 A1C9 201D      E2809D       E2809D       F3B084A8
8169     8169 8169 8169 8169 424D A1CA A1CA A1CA A1CA 214A 4FD1 A1CA FF08      EFBC88       EFBC88       F3B084A9
816A     816A 816A 816A 816A 425D A1CB A1CB A1CB A1CB 214B 4FD2 A1CB FF09      EFBC89       EFBC89       F3B084AA
816B     816B 816B 816B 816B 4463 A1CC A1CC A1CC A1CC 214C 4FD3 A1CC 3014      E38094       E38094       F3B084AB
816C     816C 816C 816C 816C 4473 A1CD A1CD A1CD A1CD 214D 4FD4 A1CD 3015      E38095       E38095       F3B084AC
816D     816D 816D 816D 816D 4444 A1CE A1CE A1CE A1CE 214E 4FD5 A1CE FF3B      EFBCBB       EFBCBB       F3B084AD
816E     816E 816E 816E 816E 4445 A1CF A1CF A1CF A1CF 214F 4FD6 A1CF FF3D      EFBCBD       EFBCBD       F3B084AE
816F     816F 816F 816F 816F 42C0 A1D0 A1D0 A1D0 A1D0 2150 4FD7 A1D0 FF5B      EFBD9B       EFBD9B       F3B084AF
8170     8170 8170 8170 8170 42D0 A1D1 A1D1 A1D1 A1D1 2151 4FD8 A1D1 FF5D      EFBD9D       EFBD9D       F3B084B0
8171     8171 8171 8171 8171 4464 A1D2 A1D2 A1D2 A1D2 2152 4FD9 A1D2 3008      E38088       E38088       F3B084B1
8172     8172 8172 8172 8172 4474 A1D3 A1D3 A1D3 A1D3 2153 4FE2 A1D3 3009      E38089       E38089       F3B084B2
8173     8173 8173 8173 8173 4465 A1D4 A1D4 A1D4 A1D4 2154 4FE3 A1D4 300A      E3808A       E3808A       F3B084B3
8174     8174 8174 8174 8174 4475 A1D5 A1D5 A1D5 A1D5 2155 4FE4 A1D5 300B      E3808B       E3808B       F3B084B4
8175     8175 8175 8175 8175 4342 A1D6 A1D6 A1D6 A1D6 2156 4FE5 A1D6 300C      E3808C       E3808C       F3B084B5
8176     8176 8176 8176 8176 4343 A1D7 A1D7 A1D7 A1D7 2157 4FE6 A1D7 300D      E3808D       E3808D       F3B084B6
8177     8177 8177 8177 8177 4442 A1D8 A1D8 A1D8 A1D8 2158 4FE7 A1D8 300E      E3808E       E3808E       F3B084B7
8178     8178 8178 8178 8178 4443 A1D9 A1D9 A1D9 A1D9 2159 4FE8 A1D9 300F      E3808F       E3808F       F3B084B8
8179     8179 8179 8179 8179 4466 A1DA A1DA A1DA A1DA 215A 4FE9 A1DA 3010      E38090       E38090       F3B084B9
817A     817A 817A 817A 817A 4476 A1DB A1DB A1DB A1DB 215B 4F4A A1DB 3011      E38091       E38091       F3B084BA
817B     817B 817B 817B 817B 424E A1DC A1DC A1DC A1DC 215C 4F5B A1DC FF0B      EFBC8B       EFBC8B       F3B084BB
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
817C     817C 817C 817C 81AF E9F3 A1DD A1DD A1DD A1DD 215D 4F5A  --  FF0D      EFBC8D       E28892       F3B084BC
817D     817D 817D 817D 817D 444B A1DE A1DE A1DE A1DE 215E 4F5F A1DE 00B1      C2B1         C2B1         F3B084BD
817E     817E 817E 817E 817E 447A A1DF A1DF A1DF A1DF 215F 4F6D A1DF 00D7      C397         C397         F3B084BE
8180     8180 8180 8180 8180 447B A1E0 A1E0 A1E0 A1E0 2160 4F79 A1E0 00F7      C3B7         C3B7         F3B084BF
8181     8181 8181 8181 8181 427E A1E1 A1E1 A1E1 A1E1 2161 4F57 A1E1 FF1D      EFBC9D       EFBC9D       F3B08580
8182     8182 8182 8182 8182 444C A1E2 A1E2 A1E2 A1E2 2162 4F59 A1E2 2260      E289A0       E289A0       F3B08581
8183     8183 8183 8183 8183 424C A1E3 A1E3 A1E3 A1E3 2163 4F62 A1E3 FF1C      EFBC9C       EFBC9C       F3B08582
8184     8184 8184 8184 8184 426E A1E4 A1E4 A1E4 A1E4 2164 4F63 A1E4 FF1E      EFBC9E       EFBC9E       F3B08583
8185     8185 8185 8185 8185 4467 A1E5 A1E5 A1E5 A1E5 2165 4F64 A1E5 2266      E289A6       E289A6       F3B08584
8186     8186 8186 8186 8186 4477 A1E6 A1E6 A1E6 A1E6 2166 4F65 A1E6 2267      E289A7       E289A7       F3B08585
8187     8187 8187 8187 8187 444D A1E7 A1E7 A1E7 A1E7 2167 4F66 A1E7 221E      E2889E       E2889E       F3B08586
8188     8188 8188 8188 8188 4468 A1E8 A1E8 A1E8 A1E8 2168 4F67 A1E8 2234      E288B4       E288B4       F3B08587
8189     8189 8189 8189 8189 4469 A1E9 A1E9 A1E9 A1E9 2169 4F68 A1E9 2642      E29982       E29982       F3B08588
818A     818A 818A 818A 818A 4479 A1EA A1EA A1EA A1EA 216A 4F69 A1EA 2640      E29980       E29980       F3B08589
818B     818B 818B 818B 818B 44ED A1EB A1EB A1EB A1EB 216B 4F70 A1EB 00B0      C2B0         C2B0         F3B0858A
818C     818C 818C 818C 818C 44EE A1EC A1EC A1EC A1EC 216C 4F71 A1EC 2032      E280B2       E280B2       F3B0858B
818D     818D 818D 818D 818D 44EF A1ED A1ED A1ED A1ED 216D 4F72 A1ED 2033      E280B3       E280B3       F3B0858C
818E     818E 818E 818E 818E 444E A1EE A1EE A1EE A1EE 216E 4F73 A1EE 2103      E28483       E28483       F3B0858D
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
818F     818F 818F 818F 818F 425B A1EF A1EF A1EF A1EF 216F 4F74 A1EF FFE5      EFBFA5       EFBFA5       F3B0858E
8190     8190 8190 8190 8190 42E0 A1F0 A1F0 A1F0 A1F0 2170 4F75 A1F0 FF04      EFBC84       EFBC84       F3B0858F
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
8191     8191 8191 8191  --  434A A1F1 A1F1 A1F1 A1F1 2171 4F76  --  FFE0      EFBFA0       EFBFA0       F3B08590
8192     8192 8192 8192  --  424A A1F2 A1F2 A1F2 A1F2 2172 4F77  --  FFE1      EFBFA1       EFBFA1       F3B08591
8193     8193 8193 8193 8193 426C A1F3 A1F3 A1F3 A1F3 2173 4F78 A1F3 FF05      EFBC85       EFBC85       F3B08592
8194     8194 8194 8194 8194 427B A1F4 A1F4 A1F4 A1F4 2174 4F80 A1F4 FF03      EFBC83       EFBC83       F3B08593
8195     8195 8195 8195 8195 4250 A1F5 A1F5 A1F5 A1F5 2175 4F8B A1F5 FF06      EFBC86       EFBC86       F3B08594
8196     8196 8196 8196 8196 425C A1F6 A1F6 A1F6 A1F6 2176 4F9B A1F6 FF0A      EFBC8A       EFBC8A       F3B08595
8197     8197 8197 8197 8197 427C A1F7 A1F7 A1F7 A1F7 2177 4F9C A1F7 FF20      EFBCA0       EFBCA0       F3B08596
8198     8198 8198 8198 8198 446A A1F8 A1F8 A1F8 A1F8 2178 4FA0 A1F8 00A7      C2A7         C2A7         F3B08597
8199     8199 8199 8199 8199 44E5 A1F9 A1F9 A1F9 A1F9 2179 4FAB A1F9 2606      E29886       E29886       F3B08598
819A     819A 819A 819A 819A 44E6 A1FA A1FA A1FA A1FA 217A 4FB0 A1FA 2605      E29885       E29885       F3B08599
819B     819B 819B 819B 819B 44E0 A1FB A1FB A1FB A1FB 217B 4FC0 A1FB 25CB      E2978B       E2978B       F3B0859A
819C     819C 819C 819C 819C 44E1 A1FC A1FC A1FC A1FC 217C 4F6A A1FC 25CF      E2978F       E2978F       F3B0859B
819D     819D 819D 819D 819D 44E4 A1FD A1FD A1FD A1FD 217D 4FD0 A1FD 25CE      E2978E       E2978E       F3B0859C
819E     819E 819E 819E 819E 44E7 A1FE A1FE A1FE A1FE 217E 4FA1 A1FE 25C7      E29787       E29787       F3B0859D
819F     819F 819F 819F 819F 44E8 A2A1 A2A1 A2A1 A2A1 2221 7F4F A2A1 25C6      E29786       E29786       F3B0859E
81A0     81A0 81A0 81A0 81A0 44E9 A2A2 A2A2 A2A2 A2A2 2222 7F7F A2A2 25A1      E296A1       E296A1       F3B0859F
81A1     81A1 81A1 81A1 81A1 44EA A2A3 A2A3 A2A3 A2A3 2223 7F7B A2A3 25A0      E296A0       E296A0       F3B085A0
81A2     81A2 81A2 81A2 81A2 44E2 A2A4 A2A4 A2A4 A2A4 2224 7FE0 A2A4 25B3      E296B3       E296B3       F3B085A1
81A3     81A3 81A3 81A3 81A3 44E3 A2A5 A2A5 A2A5 A2A5 2225 7F6C A2A5 25B2      E296B2       E296B2       F3B085A2
81A4     81A4 81A4 81A4 81A4 44EB A2A6 A2A6 A2A6 A2A6 2226 7F50 A2A6 25BD      E296BD       E296BD       F3B085A3
81A5     81A5 81A5 81A5 81A5 44EC A2A7 A2A7 A2A7 A2A7 2227 7F7D A2A7 25BC      E296BC       E296BC       F3B085A4
81A6     81A6 81A6 81A6 81A6 446B A2A8 A2A8 A2A8 A2A8 2228 7F4D A2A8 203B      E280BB       E280BB       F3B085A5
81A7     81A7 81A7 81A7 81A7 446C A2A9 A2A9 A2A9 A2A9 2229 7F5D A2A9 3012      E38092       E38092       F3B085A6
81A8     81A8 81A8 81A8 81A8 44F0 A2AA A2AA A2AA A2AA 222A 7F5C A2AA 2192      E28692       E28692       F3B085A7
81A9     81A9 81A9 81A9 81A9 44F1 A2AB A2AB A2AB A2AB 222B 7F4E A2AB 2190      E28690       E28690       F3B085A8
81AA     81AA 81AA 81AA 81AA 44F2 A2AC A2AC A2AC A2AC 222C 7F6B A2AC 2191      E28691       E28691       F3B085A9
81AB     81AB 81AB 81AB 81AB 44F3 A2AD A2AD A2AD A2AD 222D 7F60 A2AD 2193      E28693       E28693       F3B085AA
81AC     81AC 81AC 81AC 81AC 447D A2AE A2AE A2AE A2AE 222E 7F4B A2AE 3013      E38093       E38093       F3B085AB
# Category 1(1 of 2) JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
81B8     81B8 81B8 81B8 81B8 4364 A2BA A2BA A2BA 7FD0 A5D4 45A9 A2BA 2208      E28888       E28888       F3B085B7
81B9     81B9 81B9 81B9 81B9 4365 A2BB A2BB A2BB 7FD1 A5D5 45AA A2BB 220B      E2888B       E2888B       F3B085B8
81BA     81BA 81BA 81BA 81BA 4366 A2BC A2BC A2BC 7FD2 A6F4 46EA A2BC 2286      E28A86       E28A86       F3B085B9
81BB     81BB 81BB 81BB 81BB 4367 A2BD A2BD A2BD 7FD3 A6F5 46EB A2BD 2287      E28A87       E28A87       F3B085BA
81BC     81BC 81BC 81BC 81BC 4368 A2BF A2BE A2BE 7FD4 A5D2 45A7 A2BE 2282      E28A82       E28A82       F3B085BB
81BD     81BD 81BD 81BD 81BD 4369 A2BE A2BF A2BF 7FD5 A5D3 45A8 A2BF 2283      E28A83       E28A83       F3B085BC
81BE     81BE 81BE 81BE 81BE 436A A2C0 A2C0 A2C0 7FD6 2D7C 606A A2C0 222A      E288AA       E288AA       F3B085BD
81BF     81BF 81BF 81BF 81BF 436B A2C1 A2C1 A2C1 7FD7 2D7B 60C0 A2C1 2229      E288A9       E288A9       F3B085BE
81C8     81C8 81C8 81C8 81C8 436C A2CA A2CA A2CA 7FD8 A5D0 45A5 A2CA 2227      E288A7       E288A7       F3B08687
81C9     81C9 81C9 81C9 81C9 436D A2CB A2CB A2CB 7FD9 A5D1 45A6 A2CB 2228      E288A8       E288A8       F3B08688
#
# Conversion tables differ between venders (2002-04-04)
# http://www8.plala.or.jp/tkubota1/unicode-symbols-map2.html
#
# JIS      0208   SJIS   CP932  APPLE  0213   IBMGLY IBMIRV G-EUC  G-SJIS
# -----------------------------------------------------------------------
# 0x005C   ------ U+00A5 U+005C U+00A5 ------ U+00A5 U+005C U+005C U+00A5
# 0x007E   ------ U+203E U+007E U+007E ------ U+203E U+007E U+007E U+203E
# 0x2131   U+FFE3 U+FFE3 U+FFE3 U+FFE3 U+203E U+FFE3 U+FFE3 U+FFE3 U+FFE3
# 0x213D   U+2015 U+2015 U+2015 U+2014 U+2014 U+2014 U+2014 U+2015 U+2015
# 0x2140   U+005C U+005C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C U+FF3C
# 0x2141   U+301C U+301C U+FF5E U+301C U+301C U+301C U+301C U+301C U+301C
# 0x2142   U+2016 U+2016 U+2225 U+2016 U+2016 U+2016 U+2016 U+2016 U+2016
# 0x215D   U+2212 U+2212 U+FF0D U+2212 U+2212 U+2212 U+2212 U+2212 U+2212
# 0x216F   U+FFE5 U+FFE5 U+FFE5 U+FFE5 U+00A5 U+FFE5 U+FFE5 U+FFE5 U+FFE5
# 0x2171   U+00A2 U+00A2 U+FFE0 U+00A2 U+00A2 U+FFE0 U+FFE0 U+00A2 U+00A2
# 0x2172   U+00A3 U+00A3 U+FFE1 U+00A3 U+00A3 U+FFE1 U+FFE1 U+00A3 U+00A3
# 0x224C   U+00AC U+00AC U+FFE2 U+00AC U+00AC U+FFE2 U+FFE2 U+00AC U+00AC
#
81CA     81CA 81CA 81CA  --  425F A2CC A2CC A2CC 76A7  --   --   --  FFE2      EFBFA2       EFBFA2       F3B08689
81CB     81CB 81CB 81CB 81CB 436E A2CD A2CD A2CD 7FDA A6F6 46EC A2CD 21D2      E28792       E28792       F3B0868A
81CC     81CC 81CC 81CC 81CC 436F A2CE A2CE A2CE 7FDB A6F7 46ED A2CE 21D4      E28794       E28794       F3B0868B
81CD     81CD 81CD 81CD 81CD 4370 A2CF A2CF A2CF 7FDC A5E7 45B9 A2CF 2200      E28880       E28880       F3B0868C
81CE     81CE 81CE 81CE 81CE 4371 A2D0 A2D0 A2D0 7FDD A5E8 45CA A2D0 2203      E28883       E28883       F3B0868D
81DA     81DA 81DA 81DA 81DA 434B A2DC A2DC A2DC 7FDE 2D77 609C A2DC 2220      E288A0       E288A0       F3B08699
81DB     81DB 81DB 81DB 81DB 434C A2DD A2DD A2DD 7FDF 2D76 609B A2DD 22A5      E28AA5       E28AA5       F3B0869A
81DC     81DC 81DC 81DC 81DC 434D A2DE A2DE A2DE 7FE0 A6F8 46EE A2DE 2312      E28C92       E28C92       F3B0869B
81DD     81DD 81DD 81DD 81DD 434E A2DF A2DF A2DF 7FE1 A5CA 459D A2DF 2202      E28882       E28882       F3B0869C
81DE     81DE 81DE 81DE 81DE 434F A2E0 A2E0 A2E0 7FE2 A6F9 46EF A2E0 2207      E28887       E28887       F3B0869D
81DF     81DF 81DF 81DF 81DF 435B A2E1 A2E1 A2E1 76AA 2D71 6076 A2E1 2261      E289A1       E289A1       F3B0869E
81E0     81E0 81E0 81E0 81E0 435C A2E2 A2E2 A2E2 76A9 2D70 6075 A2E2 2252      E28992       E28992       F3B0869F
81E1     81E1 81E1 81E1 81E1 435D A2E3 A2E3 A2E3 7FE3 A5E1 45B3 A2E3 226A      E289AA       E289AA       F3B086A0
81E2     81E2 81E2 81E2 81E2 435E A2E4 A2E4 A2E4 7FE4 A5E2 45B4 A2E4 226B      E289AB       E289AB       F3B086A1
81E3     81E3 81E3 81E3 81E3 435F A2E5 A2E5 A2E5 7FE5 2D75 608B A2E5 221A      E2889A       E2889A       F3B086A2
81E4     81E4 81E4 81E4 81E4 4360 A2E6 A2E6 A2E6 7FE6 A5DD 45BD A2E6 223D      E288BD       E288BD       F3B086A3
81E5     81E5 81E5 81E5 81E5 4361 A2E7 A2E7 A2E7 7FE7 A5DC 45BC A2E7 221D      E2889D       E2889D       F3B086A4
81E6     81E6 81E6 81E6 81E6 4478 A2E8 A2E8 A2E8 76A8 2D7A 60B0 A2E8 2235      E288B5       E288B5       F3B086A5
81E7     81E7 81E7 81E7 81E7 4362 A2E9 A2E9 A2E9 7FE8 2D72 6077 A2E9 222B      E288AB       E288AB       F3B086A6
81E8     81E8 81E8 81E8 81E8 4363 A2EA A2EA A2EA 7FE9 A6FA 46FA A2EA 222C      E288AC       E288AC       F3B086A7
81F0     81F0 81F0 81F0 81F0 4372 A2F2 A2F2 A2F2 76D1 A4DF 44BF A2F2 212B      E284AB       E284AB       F3B086AF
81F1     81F1 81F1 81F1 81F1 4373 A2F3 A2F3 A2F3 76D3 A4FE 44FE A2F3 2030      E280B0       E280B0       F3B086B0
81F2     81F2 81F2 81F2 81F2 4374 A2F4 A2F4 A2F4 7FEA A6FB 46FB A2F4 266F      E299AF       E299AF       F3B086B1
81F3     81F3 81F3 81F3 81F3 4375 A2F5 A2F5 A2F5 7FEB A6FC 46FC A2F5 266D      E299AD       E299AD       F3B086B2
81F4     81F4 81F4 81F4 81F4 4376 A2F6 A2F6 A2F6 7FEC A6FD 46FD A2F6 266A      E299AA       E299AA       F3B086B3
81F5     81F5 81F5 81F5 81F5 4377 A2F7 A2F7 A2F7 7FED A5EE 45DA A2F7 2020      E280A0       E280A0       F3B086B4
81F6     81F6 81F6 81F6 81F6 4378 A2F8 A2F8 A2F8 7FEE A5EF 45DB A2F8 2021      E280A1       E280A1       F3B086B5
81F7     81F7 81F7 81F7 81F7 4379 A2F9 A2F9 A2F9 7FEF A5F0 45DC A2F9 00B6      C2B6         C2B6         F3B086B6
81FC     81FC 81FC 81FC 81FC 437A A2FE A2FE A2FE 7FF0 A6FE 46FE A2FE 25EF      E297AF       E297AF       F3B086BB
# End of Category 1(1 of 2) JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
824F     824F 824F 824F 824F 42F0 A3B0 A3B0 A3B0 A3B0 2330 7BF0 A3B0 FF10      EFBC90       EFBC90       F3B0878B
8250     8250 8250 8250 8250 42F1 A3B1 A3B1 A3B1 A3B1 2331 7BF1 A3B1 FF11      EFBC91       EFBC91       F3B0878C
8251     8251 8251 8251 8251 42F2 A3B2 A3B2 A3B2 A3B2 2332 7BF2 A3B2 FF12      EFBC92       EFBC92       F3B0878D
8252     8252 8252 8252 8252 42F3 A3B3 A3B3 A3B3 A3B3 2333 7BF3 A3B3 FF13      EFBC93       EFBC93       F3B0878E
8253     8253 8253 8253 8253 42F4 A3B4 A3B4 A3B4 A3B4 2334 7BF4 A3B4 FF14      EFBC94       EFBC94       F3B0878F
8254     8254 8254 8254 8254 42F5 A3B5 A3B5 A3B5 A3B5 2335 7BF5 A3B5 FF15      EFBC95       EFBC95       F3B08790
8255     8255 8255 8255 8255 42F6 A3B6 A3B6 A3B6 A3B6 2336 7BF6 A3B6 FF16      EFBC96       EFBC96       F3B08791
8256     8256 8256 8256 8256 42F7 A3B7 A3B7 A3B7 A3B7 2337 7BF7 A3B7 FF17      EFBC97       EFBC97       F3B08792
8257     8257 8257 8257 8257 42F8 A3B8 A3B8 A3B8 A3B8 2338 7BF8 A3B8 FF18      EFBC98       EFBC98       F3B08793
8258     8258 8258 8258 8258 42F9 A3B9 A3B9 A3B9 A3B9 2339 7BF9 A3B9 FF19      EFBC99       EFBC99       F3B08794
8260     8260 8260 8260 8260 42C1 A3C1 A3C1 A3C1 A3C1 2341 7BC1 A3C1 FF21      EFBCA1       EFBCA1       F3B0879C
8261     8261 8261 8261 8261 42C2 A3C2 A3C2 A3C2 A3C2 2342 7BC2 A3C2 FF22      EFBCA2       EFBCA2       F3B0879D
8262     8262 8262 8262 8262 42C3 A3C3 A3C3 A3C3 A3C3 2343 7BC3 A3C3 FF23      EFBCA3       EFBCA3       F3B0879E
8263     8263 8263 8263 8263 42C4 A3C4 A3C4 A3C4 A3C4 2344 7BC4 A3C4 FF24      EFBCA4       EFBCA4       F3B0879F
8264     8264 8264 8264 8264 42C5 A3C5 A3C5 A3C5 A3C5 2345 7BC5 A3C5 FF25      EFBCA5       EFBCA5       F3B087A0
8265     8265 8265 8265 8265 42C6 A3C6 A3C6 A3C6 A3C6 2346 7BC6 A3C6 FF26      EFBCA6       EFBCA6       F3B087A1
8266     8266 8266 8266 8266 42C7 A3C7 A3C7 A3C7 A3C7 2347 7BC7 A3C7 FF27      EFBCA7       EFBCA7       F3B087A2
8267     8267 8267 8267 8267 42C8 A3C8 A3C8 A3C8 A3C8 2348 7BC8 A3C8 FF28      EFBCA8       EFBCA8       F3B087A3
8268     8268 8268 8268 8268 42C9 A3C9 A3C9 A3C9 A3C9 2349 7BC9 A3C9 FF29      EFBCA9       EFBCA9       F3B087A4
8269     8269 8269 8269 8269 42D1 A3CA A3CA A3CA A3CA 234A 7BD1 A3CA FF2A      EFBCAA       EFBCAA       F3B087A5
826A     826A 826A 826A 826A 42D2 A3CB A3CB A3CB A3CB 234B 7BD2 A3CB FF2B      EFBCAB       EFBCAB       F3B087A6
826B     826B 826B 826B 826B 42D3 A3CC A3CC A3CC A3CC 234C 7BD3 A3CC FF2C      EFBCAC       EFBCAC       F3B087A7
826C     826C 826C 826C 826C 42D4 A3CD A3CD A3CD A3CD 234D 7BD4 A3CD FF2D      EFBCAD       EFBCAD       F3B087A8
826D     826D 826D 826D 826D 42D5 A3CE A3CE A3CE A3CE 234E 7BD5 A3CE FF2E      EFBCAE       EFBCAE       F3B087A9
826E     826E 826E 826E 826E 42D6 A3CF A3CF A3CF A3CF 234F 7BD6 A3CF FF2F      EFBCAF       EFBCAF       F3B087AA
826F     826F 826F 826F 826F 42D7 A3D0 A3D0 A3D0 A3D0 2350 7BD7 A3D0 FF30      EFBCB0       EFBCB0       F3B087AB
8270     8270 8270 8270 8270 42D8 A3D1 A3D1 A3D1 A3D1 2351 7BD8 A3D1 FF31      EFBCB1       EFBCB1       F3B087AC
8271     8271 8271 8271 8271 42D9 A3D2 A3D2 A3D2 A3D2 2352 7BD9 A3D2 FF32      EFBCB2       EFBCB2       F3B087AD
8272     8272 8272 8272 8272 42E2 A3D3 A3D3 A3D3 A3D3 2353 7BE2 A3D3 FF33      EFBCB3       EFBCB3       F3B087AE
8273     8273 8273 8273 8273 42E3 A3D4 A3D4 A3D4 A3D4 2354 7BE3 A3D4 FF34      EFBCB4       EFBCB4       F3B087AF
8274     8274 8274 8274 8274 42E4 A3D5 A3D5 A3D5 A3D5 2355 7BE4 A3D5 FF35      EFBCB5       EFBCB5       F3B087B0
8275     8275 8275 8275 8275 42E5 A3D6 A3D6 A3D6 A3D6 2356 7BE5 A3D6 FF36      EFBCB6       EFBCB6       F3B087B1
8276     8276 8276 8276 8276 42E6 A3D7 A3D7 A3D7 A3D7 2357 7BE6 A3D7 FF37      EFBCB7       EFBCB7       F3B087B2
8277     8277 8277 8277 8277 42E7 A3D8 A3D8 A3D8 A3D8 2358 7BE7 A3D8 FF38      EFBCB8       EFBCB8       F3B087B3
8278     8278 8278 8278 8278 42E8 A3D9 A3D9 A3D9 A3D9 2359 7BE8 A3D9 FF39      EFBCB9       EFBCB9       F3B087B4
8279     8279 8279 8279 8279 42E9 A3DA A3DA A3DA A3DA 235A 7BE9 A3DA FF3A      EFBCBA       EFBCBA       F3B087B5
8281     8281 8281 8281 8281 4281 A3E1 A3E1 A3E1 A3E1 2361 7B57 A3E1 FF41      EFBD81       EFBD81       F3B087BC
8282     8282 8282 8282 8282 4282 A3E2 A3E2 A3E2 A3E2 2362 7B59 A3E2 FF42      EFBD82       EFBD82       F3B087BD
8283     8283 8283 8283 8283 4283 A3E3 A3E3 A3E3 A3E3 2363 7B62 A3E3 FF43      EFBD83       EFBD83       F3B087BE
8284     8284 8284 8284 8284 4284 A3E4 A3E4 A3E4 A3E4 2364 7B63 A3E4 FF44      EFBD84       EFBD84       F3B087BF
8285     8285 8285 8285 8285 4285 A3E5 A3E5 A3E5 A3E5 2365 7B64 A3E5 FF45      EFBD85       EFBD85       F3B08880
8286     8286 8286 8286 8286 4286 A3E6 A3E6 A3E6 A3E6 2366 7B65 A3E6 FF46      EFBD86       EFBD86       F3B08881
8287     8287 8287 8287 8287 4287 A3E7 A3E7 A3E7 A3E7 2367 7B66 A3E7 FF47      EFBD87       EFBD87       F3B08882
8288     8288 8288 8288 8288 4288 A3E8 A3E8 A3E8 A3E8 2368 7B67 A3E8 FF48      EFBD88       EFBD88       F3B08883
8289     8289 8289 8289 8289 4289 A3E9 A3E9 A3E9 A3E9 2369 7B68 A3E9 FF49      EFBD89       EFBD89       F3B08884
828A     828A 828A 828A 828A 4291 A3EA A3EA A3EA A3EA 236A 7B69 A3EA FF4A      EFBD8A       EFBD8A       F3B08885
828B     828B 828B 828B 828B 4292 A3EB A3EB A3EB A3EB 236B 7B70 A3EB FF4B      EFBD8B       EFBD8B       F3B08886
828C     828C 828C 828C 828C 4293 A3EC A3EC A3EC A3EC 236C 7B71 A3EC FF4C      EFBD8C       EFBD8C       F3B08887
828D     828D 828D 828D 828D 4294 A3ED A3ED A3ED A3ED 236D 7B72 A3ED FF4D      EFBD8D       EFBD8D       F3B08888
828E     828E 828E 828E 828E 4295 A3EE A3EE A3EE A3EE 236E 7B73 A3EE FF4E      EFBD8E       EFBD8E       F3B08889
828F     828F 828F 828F 828F 4296 A3EF A3EF A3EF A3EF 236F 7B74 A3EF FF4F      EFBD8F       EFBD8F       F3B0888A
8290     8290 8290 8290 8290 4297 A3F0 A3F0 A3F0 A3F0 2370 7B75 A3F0 FF50      EFBD90       EFBD90       F3B0888B
8291     8291 8291 8291 8291 4298 A3F1 A3F1 A3F1 A3F1 2371 7B76 A3F1 FF51      EFBD91       EFBD91       F3B0888C
8292     8292 8292 8292 8292 4299 A3F2 A3F2 A3F2 A3F2 2372 7B77 A3F2 FF52      EFBD92       EFBD92       F3B0888D
8293     8293 8293 8293 8293 42A2 A3F3 A3F3 A3F3 A3F3 2373 7B78 A3F3 FF53      EFBD93       EFBD93       F3B0888E
8294     8294 8294 8294 8294 42A3 A3F4 A3F4 A3F4 A3F4 2374 7B80 A3F4 FF54      EFBD94       EFBD94       F3B0888F
8295     8295 8295 8295 8295 42A4 A3F5 A3F5 A3F5 A3F5 2375 7B8B A3F5 FF55      EFBD95       EFBD95       F3B08890
8296     8296 8296 8296 8296 42A5 A3F6 A3F6 A3F6 A3F6 2376 7B9B A3F6 FF56      EFBD96       EFBD96       F3B08891
8297     8297 8297 8297 8297 42A6 A3F7 A3F7 A3F7 A3F7 2377 7B9C A3F7 FF57      EFBD97       EFBD97       F3B08892
8298     8298 8298 8298 8298 42A7 A3F8 A3F8 A3F8 A3F8 2378 7BA0 A3F8 FF58      EFBD98       EFBD98       F3B08893
8299     8299 8299 8299 8299 42A8 A3F9 A3F9 A3F9 A3F9 2379 7BAB A3F9 FF59      EFBD99       EFBD99       F3B08894
829A     829A 829A 829A 829A 42A9 A3FA A3FA A3FA A3FA 237A 7BB0 A3FA FF5A      EFBD9A       EFBD9A       F3B08895
829F     829F 829F 829F 829F 4447 A4A1 A4A1 A4A1 A4A1 2421 E04F A4A1 3041      E38181       E38181       F3B0889A
82A0     82A0 82A0 82A0 82A0 4481 A4A2 A4A2 A4A2 A4A2 2422 E07F A4A2 3042      E38182       E38182       F3B0889B
82A1     82A1 82A1 82A1 82A1 4448 A4A3 A4A3 A4A3 A4A3 2423 E07B A4A3 3043      E38183       E38183       F3B0889C
82A2     82A2 82A2 82A2 82A2 4482 A4A4 A4A4 A4A4 A4A4 2424 E0E0 A4A4 3044      E38184       E38184       F3B0889D
82A3     82A3 82A3 82A3 82A3 4449 A4A5 A4A5 A4A5 A4A5 2425 E06C A4A5 3045      E38185       E38185       F3B0889E
82A4     82A4 82A4 82A4 82A4 4483 A4A6 A4A6 A4A6 A4A6 2426 E050 A4A6 3046      E38186       E38186       F3B0889F
82A5     82A5 82A5 82A5 82A5 4451 A4A7 A4A7 A4A7 A4A7 2427 E07D A4A7 3047      E38187       E38187       F3B088A0
82A6     82A6 82A6 82A6 82A6 4484 A4A8 A4A8 A4A8 A4A8 2428 E04D A4A8 3048      E38188       E38188       F3B088A1
82A7     82A7 82A7 82A7 82A7 4452 A4A9 A4A9 A4A9 A4A9 2429 E05D A4A9 3049      E38189       E38189       F3B088A2
82A8     82A8 82A8 82A8 82A8 4485 A4AA A4AA A4AA A4AA 242A E05C A4AA 304A      E3818A       E3818A       F3B088A3
82A9     82A9 82A9 82A9 82A9 4486 A4AB A4AB A4AB A4AB 242B E04E A4AB 304B      E3818B       E3818B       F3B088A4
82AA     82AA 82AA 82AA 82AA 44C0 A4AC A4AC A4AC A4AC 242C E06B A4AC 304C      E3818C       E3818C       F3B088A5
82AB     82AB 82AB 82AB 82AB 4487 A4AD A4AD A4AD A4AD 242D E060 A4AD 304D      E3818D       E3818D       F3B088A6
82AC     82AC 82AC 82AC 82AC 44C1 A4AE A4AE A4AE A4AE 242E E04B A4AE 304E      E3818E       E3818E       F3B088A7
82AD     82AD 82AD 82AD 82AD 4488 A4AF A4AF A4AF A4AF 242F E061 A4AF 304F      E3818F       E3818F       F3B088A8
82AE     82AE 82AE 82AE 82AE 44C2 A4B0 A4B0 A4B0 A4B0 2430 E0F0 A4B0 3050      E38190       E38190       F3B088A9
82AF     82AF 82AF 82AF 82AF 4489 A4B1 A4B1 A4B1 A4B1 2431 E0F1 A4B1 3051      E38191       E38191       F3B088AA
82B0     82B0 82B0 82B0 82B0 44C3 A4B2 A4B2 A4B2 A4B2 2432 E0F2 A4B2 3052      E38192       E38192       F3B088AB
82B1     82B1 82B1 82B1 82B1 448A A4B3 A4B3 A4B3 A4B3 2433 E0F3 A4B3 3053      E38193       E38193       F3B088AC
82B2     82B2 82B2 82B2 82B2 44C4 A4B4 A4B4 A4B4 A4B4 2434 E0F4 A4B4 3054      E38194       E38194       F3B088AD
82B3     82B3 82B3 82B3 82B3 448C A4B5 A4B5 A4B5 A4B5 2435 E0F5 A4B5 3055      E38195       E38195       F3B088AE
82B4     82B4 82B4 82B4 82B4 44C5 A4B6 A4B6 A4B6 A4B6 2436 E0F6 A4B6 3056      E38196       E38196       F3B088AF
82B5     82B5 82B5 82B5 82B5 448D A4B7 A4B7 A4B7 A4B7 2437 E0F7 A4B7 3057      E38197       E38197       F3B088B0
82B6     82B6 82B6 82B6 82B6 44C6 A4B8 A4B8 A4B8 A4B8 2438 E0F8 A4B8 3058      E38198       E38198       F3B088B1
82B7     82B7 82B7 82B7 82B7 448E A4B9 A4B9 A4B9 A4B9 2439 E0F9 A4B9 3059      E38199       E38199       F3B088B2
82B8     82B8 82B8 82B8 82B8 44C7 A4BA A4BA A4BA A4BA 243A E07A A4BA 305A      E3819A       E3819A       F3B088B3
82B9     82B9 82B9 82B9 82B9 448F A4BB A4BB A4BB A4BB 243B E05E A4BB 305B      E3819B       E3819B       F3B088B4
82BA     82BA 82BA 82BA 82BA 44C8 A4BC A4BC A4BC A4BC 243C E04C A4BC 305C      E3819C       E3819C       F3B088B5
82BB     82BB 82BB 82BB 82BB 4490 A4BD A4BD A4BD A4BD 243D E07E A4BD 305D      E3819D       E3819D       F3B088B6
82BC     82BC 82BC 82BC 82BC 44C9 A4BE A4BE A4BE A4BE 243E E06E A4BE 305E      E3819E       E3819E       F3B088B7
82BD     82BD 82BD 82BD 82BD 4491 A4BF A4BF A4BF A4BF 243F E06F A4BF 305F      E3819F       E3819F       F3B088B8
82BE     82BE 82BE 82BE 82BE 44CA A4C0 A4C0 A4C0 A4C0 2440 E07C A4C0 3060      E381A0       E381A0       F3B088B9
82BF     82BF 82BF 82BF 82BF 4492 A4C1 A4C1 A4C1 A4C1 2441 E0C1 A4C1 3061      E381A1       E381A1       F3B088BA
82C0     82C0 82C0 82C0 82C0 44CB A4C2 A4C2 A4C2 A4C2 2442 E0C2 A4C2 3062      E381A2       E381A2       F3B088BB
82C1     82C1 82C1 82C1 82C1 4456 A4C3 A4C3 A4C3 A4C3 2443 E0C3 A4C3 3063      E381A3       E381A3       F3B088BC
82C2     82C2 82C2 82C2 82C2 4493 A4C4 A4C4 A4C4 A4C4 2444 E0C4 A4C4 3064      E381A4       E381A4       F3B088BD
82C3     82C3 82C3 82C3 82C3 44CC A4C5 A4C5 A4C5 A4C5 2445 E0C5 A4C5 3065      E381A5       E381A5       F3B088BE
82C4     82C4 82C4 82C4 82C4 4494 A4C6 A4C6 A4C6 A4C6 2446 E0C6 A4C6 3066      E381A6       E381A6       F3B088BF
82C5     82C5 82C5 82C5 82C5 44CD A4C7 A4C7 A4C7 A4C7 2447 E0C7 A4C7 3067      E381A7       E381A7       F3B08980
82C6     82C6 82C6 82C6 82C6 4495 A4C8 A4C8 A4C8 A4C8 2448 E0C8 A4C8 3068      E381A8       E381A8       F3B08981
82C7     82C7 82C7 82C7 82C7 44CE A4C9 A4C9 A4C9 A4C9 2449 E0C9 A4C9 3069      E381A9       E381A9       F3B08982
82C8     82C8 82C8 82C8 82C8 4496 A4CA A4CA A4CA A4CA 244A E0D1 A4CA 306A      E381AA       E381AA       F3B08983
82C9     82C9 82C9 82C9 82C9 4497 A4CB A4CB A4CB A4CB 244B E0D2 A4CB 306B      E381AB       E381AB       F3B08984
82CA     82CA 82CA 82CA 82CA 4498 A4CC A4CC A4CC A4CC 244C E0D3 A4CC 306C      E381AC       E381AC       F3B08985
82CB     82CB 82CB 82CB 82CB 4499 A4CD A4CD A4CD A4CD 244D E0D4 A4CD 306D      E381AD       E381AD       F3B08986
82CC     82CC 82CC 82CC 82CC 449A A4CE A4CE A4CE A4CE 244E E0D5 A4CE 306E      E381AE       E381AE       F3B08987
82CD     82CD 82CD 82CD 82CD 449D A4CF A4CF A4CF A4CF 244F E0D6 A4CF 306F      E381AF       E381AF       F3B08988
82CE     82CE 82CE 82CE 82CE 44CF A4D0 A4D0 A4D0 A4D0 2450 E0D7 A4D0 3070      E381B0       E381B0       F3B08989
82CF     82CF 82CF 82CF 82CF 44D5 A4D1 A4D1 A4D1 A4D1 2451 E0D8 A4D1 3071      E381B1       E381B1       F3B0898A
82D0     82D0 82D0 82D0 82D0 449E A4D2 A4D2 A4D2 A4D2 2452 E0D9 A4D2 3072      E381B2       E381B2       F3B0898B
82D1     82D1 82D1 82D1 82D1 44D0 A4D3 A4D3 A4D3 A4D3 2453 E0E2 A4D3 3073      E381B3       E381B3       F3B0898C
82D2     82D2 82D2 82D2 82D2 44D6 A4D4 A4D4 A4D4 A4D4 2454 E0E3 A4D4 3074      E381B4       E381B4       F3B0898D
82D3     82D3 82D3 82D3 82D3 449F A4D5 A4D5 A4D5 A4D5 2455 E0E4 A4D5 3075      E381B5       E381B5       F3B0898E
82D4     82D4 82D4 82D4 82D4 44D1 A4D6 A4D6 A4D6 A4D6 2456 E0E5 A4D6 3076      E381B6       E381B6       F3B0898F
82D5     82D5 82D5 82D5 82D5 44D7 A4D7 A4D7 A4D7 A4D7 2457 E0E6 A4D7 3077      E381B7       E381B7       F3B08990
82D6     82D6 82D6 82D6 82D6 44A2 A4D8 A4D8 A4D8 A4D8 2458 E0E7 A4D8 3078      E381B8       E381B8       F3B08991
82D7     82D7 82D7 82D7 82D7 44D2 A4D9 A4D9 A4D9 A4D9 2459 E0E8 A4D9 3079      E381B9       E381B9       F3B08992
82D8     82D8 82D8 82D8 82D8 44D8 A4DA A4DA A4DA A4DA 245A E0E9 A4DA 307A      E381BA       E381BA       F3B08993
82D9     82D9 82D9 82D9 82D9 44A3 A4DB A4DB A4DB A4DB 245B E04A A4DB 307B      E381BB       E381BB       F3B08994
82DA     82DA 82DA 82DA 82DA 44D3 A4DC A4DC A4DC A4DC 245C E05B A4DC 307C      E381BC       E381BC       F3B08995
82DB     82DB 82DB 82DB 82DB 44D9 A4DD A4DD A4DD A4DD 245D E05A A4DD 307D      E381BD       E381BD       F3B08996
82DC     82DC 82DC 82DC 82DC 44A4 A4DE A4DE A4DE A4DE 245E E05F A4DE 307E      E381BE       E381BE       F3B08997
82DD     82DD 82DD 82DD 82DD 44A5 A4DF A4DF A4DF A4DF 245F E06D A4DF 307F      E381BF       E381BF       F3B08998
82DE     82DE 82DE 82DE 82DE 44A6 A4E0 A4E0 A4E0 A4E0 2460 E079 A4E0 3080      E38280       E38280       F3B08999
82DF     82DF 82DF 82DF 82DF 44A7 A4E1 A4E1 A4E1 A4E1 2461 E057 A4E1 3081      E38281       E38281       F3B0899A
82E0     82E0 82E0 82E0 82E0 44A8 A4E2 A4E2 A4E2 A4E2 2462 E059 A4E2 3082      E38282       E38282       F3B0899B
82E1     82E1 82E1 82E1 82E1 4453 A4E3 A4E3 A4E3 A4E3 2463 E062 A4E3 3083      E38283       E38283       F3B0899C
82E2     82E2 82E2 82E2 82E2 44A9 A4E4 A4E4 A4E4 A4E4 2464 E063 A4E4 3084      E38284       E38284       F3B0899D
82E3     82E3 82E3 82E3 82E3 4454 A4E5 A4E5 A4E5 A4E5 2465 E064 A4E5 3085      E38285       E38285       F3B0899E
82E4     82E4 82E4 82E4 82E4 44AA A4E6 A4E6 A4E6 A4E6 2466 E065 A4E6 3086      E38286       E38286       F3B0899F
82E5     82E5 82E5 82E5 82E5 4455 A4E7 A4E7 A4E7 A4E7 2467 E066 A4E7 3087      E38287       E38287       F3B089A0
82E6     82E6 82E6 82E6 82E6 44AC A4E8 A4E8 A4E8 A4E8 2468 E067 A4E8 3088      E38288       E38288       F3B089A1
82E7     82E7 82E7 82E7 82E7 44AD A4E9 A4E9 A4E9 A4E9 2469 E068 A4E9 3089      E38289       E38289       F3B089A2
82E8     82E8 82E8 82E8 82E8 44AE A4EA A4EA A4EA A4EA 246A E069 A4EA 308A      E3828A       E3828A       F3B089A3
82E9     82E9 82E9 82E9 82E9 44AF A4EB A4EB A4EB A4EB 246B E070 A4EB 308B      E3828B       E3828B       F3B089A4
82EA     82EA 82EA 82EA 82EA 44BA A4EC A4EC A4EC A4EC 246C E071 A4EC 308C      E3828C       E3828C       F3B089A5
82EB     82EB 82EB 82EB 82EB 44BB A4ED A4ED A4ED A4ED 246D E072 A4ED 308D      E3828D       E3828D       F3B089A6
82EC     82EC 82EC 82EC 82EC 4457 A4EE A4EE A4EE A4EE 246E E073 A4EE 308E      E3828E       E3828E       F3B089A7
82ED     82ED 82ED 82ED 82ED 44BC A4EF A4EF A4EF A4EF 246F E074 A4EF 308F      E3828F       E3828F       F3B089A8
82EE     82EE 82EE 82EE 82EE 44DA A4F0 A4F0 A4F0 A4F0 2470 E075 A4F0 3090      E38290       E38290       F3B089A9
82EF     82EF 82EF 82EF 82EF 44DB A4F1 A4F1 A4F1 A4F1 2471 E076 A4F1 3091      E38291       E38291       F3B089AA
82F0     82F0 82F0 82F0 82F0 4446 A4F2 A4F2 A4F2 A4F2 2472 E077 A4F2 3092      E38292       E38292       F3B089AB
82F1     82F1 82F1 82F1 82F1 44BD A4F3 A4F3 A4F3 A4F3 2473 E078 A4F3 3093      E38293       E38293       F3B089AC
8340     8340 8340 8340 8340 4347 A5A1 A5A1 A5A1 A5A1 2521 6C4F A5A1 30A1      E382A1       E382A1       F3B089B8
8341     8341 8341 8341 8341 4381 A5A2 A5A2 A5A2 A5A2 2522 6C7F A5A2 30A2      E382A2       E382A2       F3B089B9
8342     8342 8342 8342 8342 4348 A5A3 A5A3 A5A3 A5A3 2523 6C7B A5A3 30A3      E382A3       E382A3       F3B089BA
8343     8343 8343 8343 8343 4382 A5A4 A5A4 A5A4 A5A4 2524 6CE0 A5A4 30A4      E382A4       E382A4       F3B089BB
8344     8344 8344 8344 8344 4349 A5A5 A5A5 A5A5 A5A5 2525 6C6C A5A5 30A5      E382A5       E382A5       F3B089BC
8345     8345 8345 8345 8345 4383 A5A6 A5A6 A5A6 A5A6 2526 6C50 A5A6 30A6      E382A6       E382A6       F3B089BD
8346     8346 8346 8346 8346 4351 A5A7 A5A7 A5A7 A5A7 2527 6C7D A5A7 30A7      E382A7       E382A7       F3B089BE
8347     8347 8347 8347 8347 4384 A5A8 A5A8 A5A8 A5A8 2528 6C4D A5A8 30A8      E382A8       E382A8       F3B089BF
8348     8348 8348 8348 8348 4352 A5A9 A5A9 A5A9 A5A9 2529 6C5D A5A9 30A9      E382A9       E382A9       F3B08A80
8349     8349 8349 8349 8349 4385 A5AA A5AA A5AA A5AA 252A 6C5C A5AA 30AA      E382AA       E382AA       F3B08A81
834A     834A 834A 834A 834A 4386 A5AB A5AB A5AB A5AB 252B 6C4E A5AB 30AB      E382AB       E382AB       F3B08A82
834B     834B 834B 834B 834B 43C0 A5AC A5AC A5AC A5AC 252C 6C6B A5AC 30AC      E382AC       E382AC       F3B08A83
834C     834C 834C 834C 834C 4387 A5AD A5AD A5AD A5AD 252D 6C60 A5AD 30AD      E382AD       E382AD       F3B08A84
834D     834D 834D 834D 834D 43C1 A5AE A5AE A5AE A5AE 252E 6C4B A5AE 30AE      E382AE       E382AE       F3B08A85
834E     834E 834E 834E 834E 4388 A5AF A5AF A5AF A5AF 252F 6C61 A5AF 30AF      E382AF       E382AF       F3B08A86
834F     834F 834F 834F 834F 43C2 A5B0 A5B0 A5B0 A5B0 2530 6CF0 A5B0 30B0      E382B0       E382B0       F3B08A87
8350     8350 8350 8350 8350 4389 A5B1 A5B1 A5B1 A5B1 2531 6CF1 A5B1 30B1      E382B1       E382B1       F3B08A88
8351     8351 8351 8351 8351 43C3 A5B2 A5B2 A5B2 A5B2 2532 6CF2 A5B2 30B2      E382B2       E382B2       F3B08A89
8352     8352 8352 8352 8352 438A A5B3 A5B3 A5B3 A5B3 2533 6CF3 A5B3 30B3      E382B3       E382B3       F3B08A8A
8353     8353 8353 8353 8353 43C4 A5B4 A5B4 A5B4 A5B4 2534 6CF4 A5B4 30B4      E382B4       E382B4       F3B08A8B
8354     8354 8354 8354 8354 438C A5B5 A5B5 A5B5 A5B5 2535 6CF5 A5B5 30B5      E382B5       E382B5       F3B08A8C
8355     8355 8355 8355 8355 43C5 A5B6 A5B6 A5B6 A5B6 2536 6CF6 A5B6 30B6      E382B6       E382B6       F3B08A8D
8356     8356 8356 8356 8356 438D A5B7 A5B7 A5B7 A5B7 2537 6CF7 A5B7 30B7      E382B7       E382B7       F3B08A8E
8357     8357 8357 8357 8357 43C6 A5B8 A5B8 A5B8 A5B8 2538 6CF8 A5B8 30B8      E382B8       E382B8       F3B08A8F
8358     8358 8358 8358 8358 438E A5B9 A5B9 A5B9 A5B9 2539 6CF9 A5B9 30B9      E382B9       E382B9       F3B08A90
8359     8359 8359 8359 8359 43C7 A5BA A5BA A5BA A5BA 253A 6C7A A5BA 30BA      E382BA       E382BA       F3B08A91
835A     835A 835A 835A 835A 438F A5BB A5BB A5BB A5BB 253B 6C5E A5BB 30BB      E382BB       E382BB       F3B08A92
835B     835B 835B 835B 835B 43C8 A5BC A5BC A5BC A5BC 253C 6C4C A5BC 30BC      E382BC       E382BC       F3B08A93
835C     835C 835C 835C 835C 4390 A5BD A5BD A5BD A5BD 253D 6C7E A5BD 30BD      E382BD       E382BD       F3B08A94
835D     835D 835D 835D 835D 43C9 A5BE A5BE A5BE A5BE 253E 6C6E A5BE 30BE      E382BE       E382BE       F3B08A95
835E     835E 835E 835E 835E 4391 A5BF A5BF A5BF A5BF 253F 6C6F A5BF 30BF      E382BF       E382BF       F3B08A96
835F     835F 835F 835F 835F 43CA A5C0 A5C0 A5C0 A5C0 2540 6C7C A5C0 30C0      E38380       E38380       F3B08A97
8360     8360 8360 8360 8360 4392 A5C1 A5C1 A5C1 A5C1 2541 6CC1 A5C1 30C1      E38381       E38381       F3B08A98
8361     8361 8361 8361 8361 43CB A5C2 A5C2 A5C2 A5C2 2542 6CC2 A5C2 30C2      E38382       E38382       F3B08A99
8362     8362 8362 8362 8362 4356 A5C3 A5C3 A5C3 A5C3 2543 6CC3 A5C3 30C3      E38383       E38383       F3B08A9A
8363     8363 8363 8363 8363 4393 A5C4 A5C4 A5C4 A5C4 2544 6CC4 A5C4 30C4      E38384       E38384       F3B08A9B
8364     8364 8364 8364 8364 43CC A5C5 A5C5 A5C5 A5C5 2545 6CC5 A5C5 30C5      E38385       E38385       F3B08A9C
8365     8365 8365 8365 8365 4394 A5C6 A5C6 A5C6 A5C6 2546 6CC6 A5C6 30C6      E38386       E38386       F3B08A9D
8366     8366 8366 8366 8366 43CD A5C7 A5C7 A5C7 A5C7 2547 6CC7 A5C7 30C7      E38387       E38387       F3B08A9E
8367     8367 8367 8367 8367 4395 A5C8 A5C8 A5C8 A5C8 2548 6CC8 A5C8 30C8      E38388       E38388       F3B08A9F
8368     8368 8368 8368 8368 43CE A5C9 A5C9 A5C9 A5C9 2549 6CC9 A5C9 30C9      E38389       E38389       F3B08AA0
8369     8369 8369 8369 8369 4396 A5CA A5CA A5CA A5CA 254A 6CD1 A5CA 30CA      E3838A       E3838A       F3B08AA1
836A     836A 836A 836A 836A 4397 A5CB A5CB A5CB A5CB 254B 6CD2 A5CB 30CB      E3838B       E3838B       F3B08AA2
836B     836B 836B 836B 836B 4398 A5CC A5CC A5CC A5CC 254C 6CD3 A5CC 30CC      E3838C       E3838C       F3B08AA3
836C     836C 836C 836C 836C 4399 A5CD A5CD A5CD A5CD 254D 6CD4 A5CD 30CD      E3838D       E3838D       F3B08AA4
836D     836D 836D 836D 836D 439A A5CE A5CE A5CE A5CE 254E 6CD5 A5CE 30CE      E3838E       E3838E       F3B08AA5
836E     836E 836E 836E 836E 439D A5CF A5CF A5CF A5CF 254F 6CD6 A5CF 30CF      E3838F       E3838F       F3B08AA6
836F     836F 836F 836F 836F 43CF A5D0 A5D0 A5D0 A5D0 2550 6CD7 A5D0 30D0      E38390       E38390       F3B08AA7
8370     8370 8370 8370 8370 43D5 A5D1 A5D1 A5D1 A5D1 2551 6CD8 A5D1 30D1      E38391       E38391       F3B08AA8
8371     8371 8371 8371 8371 439E A5D2 A5D2 A5D2 A5D2 2552 6CD9 A5D2 30D2      E38392       E38392       F3B08AA9
8372     8372 8372 8372 8372 43D0 A5D3 A5D3 A5D3 A5D3 2553 6CE2 A5D3 30D3      E38393       E38393       F3B08AAA
8373     8373 8373 8373 8373 43D6 A5D4 A5D4 A5D4 A5D4 2554 6CE3 A5D4 30D4      E38394       E38394       F3B08AAB
8374     8374 8374 8374 8374 439F A5D5 A5D5 A5D5 A5D5 2555 6CE4 A5D5 30D5      E38395       E38395       F3B08AAC
8375     8375 8375 8375 8375 43D1 A5D6 A5D6 A5D6 A5D6 2556 6CE5 A5D6 30D6      E38396       E38396       F3B08AAD
8376     8376 8376 8376 8376 43D7 A5D7 A5D7 A5D7 A5D7 2557 6CE6 A5D7 30D7      E38397       E38397       F3B08AAE
8377     8377 8377 8377 8377 43A2 A5D8 A5D8 A5D8 A5D8 2558 6CE7 A5D8 30D8      E38398       E38398       F3B08AAF
8378     8378 8378 8378 8378 43D2 A5D9 A5D9 A5D9 A5D9 2559 6CE8 A5D9 30D9      E38399       E38399       F3B08AB0
8379     8379 8379 8379 8379 43D8 A5DA A5DA A5DA A5DA 255A 6CE9 A5DA 30DA      E3839A       E3839A       F3B08AB1
837A     837A 837A 837A 837A 43A3 A5DB A5DB A5DB A5DB 255B 6C4A A5DB 30DB      E3839B       E3839B       F3B08AB2
837B     837B 837B 837B 837B 43D3 A5DC A5DC A5DC A5DC 255C 6C5B A5DC 30DC      E3839C       E3839C       F3B08AB3
837C     837C 837C 837C 837C 43D9 A5DD A5DD A5DD A5DD 255D 6C5A A5DD 30DD      E3839D       E3839D       F3B08AB4
837D     837D 837D 837D 837D 43A4 A5DE A5DE A5DE A5DE 255E 6C5F A5DE 30DE      E3839E       E3839E       F3B08AB5
837E     837E 837E 837E 837E 43A5 A5DF A5DF A5DF A5DF 255F 6C6D A5DF 30DF      E3839F       E3839F       F3B08AB6
8380     8380 8380 8380 8380 43A6 A5E0 A5E0 A5E0 A5E0 2560 6C79 A5E0 30E0      E383A0       E383A0       F3B08AB7
8381     8381 8381 8381 8381 43A7 A5E1 A5E1 A5E1 A5E1 2561 6C57 A5E1 30E1      E383A1       E383A1       F3B08AB8
8382     8382 8382 8382 8382 43A8 A5E2 A5E2 A5E2 A5E2 2562 6C59 A5E2 30E2      E383A2       E383A2       F3B08AB9
8383     8383 8383 8383 8383 4353 A5E3 A5E3 A5E3 A5E3 2563 6C62 A5E3 30E3      E383A3       E383A3       F3B08ABA
8384     8384 8384 8384 8384 43A9 A5E4 A5E4 A5E4 A5E4 2564 6C63 A5E4 30E4      E383A4       E383A4       F3B08ABB
8385     8385 8385 8385 8385 4354 A5E5 A5E5 A5E5 A5E5 2565 6C64 A5E5 30E5      E383A5       E383A5       F3B08ABC
8386     8386 8386 8386 8386 43AA A5E6 A5E6 A5E6 A5E6 2566 6C65 A5E6 30E6      E383A6       E383A6       F3B08ABD
8387     8387 8387 8387 8387 4355 A5E7 A5E7 A5E7 A5E7 2567 6C66 A5E7 30E7      E383A7       E383A7       F3B08ABE
8388     8388 8388 8388 8388 43AC A5E8 A5E8 A5E8 A5E8 2568 6C67 A5E8 30E8      E383A8       E383A8       F3B08ABF
8389     8389 8389 8389 8389 43AD A5E9 A5E9 A5E9 A5E9 2569 6C68 A5E9 30E9      E383A9       E383A9       F3B08B80
838A     838A 838A 838A 838A 43AE A5EA A5EA A5EA A5EA 256A 6C69 A5EA 30EA      E383AA       E383AA       F3B08B81
838B     838B 838B 838B 838B 43AF A5EB A5EB A5EB A5EB 256B 6C70 A5EB 30EB      E383AB       E383AB       F3B08B82
838C     838C 838C 838C 838C 43BA A5EC A5EC A5EC A5EC 256C 6C71 A5EC 30EC      E383AC       E383AC       F3B08B83
838D     838D 838D 838D 838D 43BB A5ED A5ED A5ED A5ED 256D 6C72 A5ED 30ED      E383AD       E383AD       F3B08B84
838E     838E 838E 838E 838E 4357 A5EE A5EE A5EE A5EE 256E 6C73 A5EE 30EE      E383AE       E383AE       F3B08B85
838F     838F 838F 838F 838F 43BC A5EF A5EF A5EF A5EF 256F 6C74 A5EF 30EF      E383AF       E383AF       F3B08B86
8390     8390 8390 8390 8390 43DA A5F0 A5F0 A5F0 A5F0 2570 6C75 A5F0 30F0      E383B0       E383B0       F3B08B87
8391     8391 8391 8391 8391 43DB A5F1 A5F1 A5F1 A5F1 2571 6C76 A5F1 30F1      E383B1       E383B1       F3B08B88
8392     8392 8392 8392 8392 4346 A5F2 A5F2 A5F2 A5F2 2572 6C77 A5F2 30F2      E383B2       E383B2       F3B08B89
8393     8393 8393 8393 8393 43BD A5F3 A5F3 A5F3 A5F3 2573 6C78 A5F3 30F3      E383B3       E383B3       F3B08B8A
8394     8394 8394 8394 8394 43D4 A5F4 A5F4 A5F4 A5F4 2574 6C80 A5F4 30F4      E383B4       E383B4       F3B08B8B
8395     8395 8395 8395 8395 4359 A5F5 A5F5 A5F5 A5F5 2575 6C8B A5F5 30F5      E383B5       E383B5       F3B08B8C
8396     8396 8396 8396 8396 435A A5F6 A5F6 A5F6 A5F6 2576 6C9B A5F6 30F6      E383B6       E383B6       F3B08B8D
839F     839F 839F 839F 839F 4161 A6A1 A6A1 A6A1 A6A1 2621 504F A6A1 0391      CE91         CE91         F3B08B96
83A0     83A0 83A0 83A0 83A0 4162 A6A2 A6A2 A6A2 A6A2 2622 507F A6A2 0392      CE92         CE92         F3B08B97
83A1     83A1 83A1 83A1 83A1 4163 A6A3 A6A3 A6A3 A6A3 2623 507B A6A3 0393      CE93         CE93         F3B08B98
83A2     83A2 83A2 83A2 83A2 4164 A6A4 A6A4 A6A4 A6A4 2624 50E0 A6A4 0394      CE94         CE94         F3B08B99
83A3     83A3 83A3 83A3 83A3 4165 A6A5 A6A5 A6A5 A6A5 2625 506C A6A5 0395      CE95         CE95         F3B08B9A
83A4     83A4 83A4 83A4 83A4 4166 A6A6 A6A6 A6A6 A6A6 2626 5050 A6A6 0396      CE96         CE96         F3B08B9B
83A5     83A5 83A5 83A5 83A5 4167 A6A7 A6A7 A6A7 A6A7 2627 507D A6A7 0397      CE97         CE97         F3B08B9C
83A6     83A6 83A6 83A6 83A6 4168 A6A8 A6A8 A6A8 A6A8 2628 504D A6A8 0398      CE98         CE98         F3B08B9D
83A7     83A7 83A7 83A7 83A7 4169 A6A9 A6A9 A6A9 A6A9 2629 505D A6A9 0399      CE99         CE99         F3B08B9E
83A8     83A8 83A8 83A8 83A8 416A A6AA A6AA A6AA A6AA 262A 505C A6AA 039A      CE9A         CE9A         F3B08B9F
83A9     83A9 83A9 83A9 83A9 416B A6AB A6AB A6AB A6AB 262B 504E A6AB 039B      CE9B         CE9B         F3B08BA0
83AA     83AA 83AA 83AA 83AA 416C A6AC A6AC A6AC A6AC 262C 506B A6AC 039C      CE9C         CE9C         F3B08BA1
83AB     83AB 83AB 83AB 83AB 416D A6AD A6AD A6AD A6AD 262D 5060 A6AD 039D      CE9D         CE9D         F3B08BA2
83AC     83AC 83AC 83AC 83AC 416E A6AE A6AE A6AE A6AE 262E 504B A6AE 039E      CE9E         CE9E         F3B08BA3
83AD     83AD 83AD 83AD 83AD 416F A6AF A6AF A6AF A6AF 262F 5061 A6AF 039F      CE9F         CE9F         F3B08BA4
83AE     83AE 83AE 83AE 83AE 4170 A6B0 A6B0 A6B0 A6B0 2630 50F0 A6B0 03A0      CEA0         CEA0         F3B08BA5
83AF     83AF 83AF 83AF 83AF 4171 A6B1 A6B1 A6B1 A6B1 2631 50F1 A6B1 03A1      CEA1         CEA1         F3B08BA6
83B0     83B0 83B0 83B0 83B0 4172 A6B2 A6B2 A6B2 A6B2 2632 50F2 A6B2 03A3      CEA3         CEA3         F3B08BA7
83B1     83B1 83B1 83B1 83B1 4173 A6B3 A6B3 A6B3 A6B3 2633 50F3 A6B3 03A4      CEA4         CEA4         F3B08BA8
83B2     83B2 83B2 83B2 83B2 4174 A6B4 A6B4 A6B4 A6B4 2634 50F4 A6B4 03A5      CEA5         CEA5         F3B08BA9
83B3     83B3 83B3 83B3 83B3 4175 A6B5 A6B5 A6B5 A6B5 2635 50F5 A6B5 03A6      CEA6         CEA6         F3B08BAA
83B4     83B4 83B4 83B4 83B4 4176 A6B6 A6B6 A6B6 A6B6 2636 50F6 A6B6 03A7      CEA7         CEA7         F3B08BAB
83B5     83B5 83B5 83B5 83B5 4177 A6B7 A6B7 A6B7 A6B7 2637 50F7 A6B7 03A8      CEA8         CEA8         F3B08BAC
83B6     83B6 83B6 83B6 83B6 4178 A6B8 A6B8 A6B8 A6B8 2638 50F8 A6B8 03A9      CEA9         CEA9         F3B08BAD
83BF     83BF 83BF 83BF 83BF 4141 A6C1 A6C1 A6C1 A6C1 2641 50C1 A6C1 03B1      CEB1         CEB1         F3B08BB6
83C0     83C0 83C0 83C0 83C0 4142 A6C2 A6C2 A6C2 A6C2 2642 50C2 A6C2 03B2      CEB2         CEB2         F3B08BB7
83C1     83C1 83C1 83C1 83C1 4143 A6C3 A6C3 A6C3 A6C3 2643 50C3 A6C3 03B3      CEB3         CEB3         F3B08BB8
83C2     83C2 83C2 83C2 83C2 4144 A6C4 A6C4 A6C4 A6C4 2644 50C4 A6C4 03B4      CEB4         CEB4         F3B08BB9
83C3     83C3 83C3 83C3 83C3 4145 A6C5 A6C5 A6C5 A6C5 2645 50C5 A6C5 03B5      CEB5         CEB5         F3B08BBA
83C4     83C4 83C4 83C4 83C4 4146 A6C6 A6C6 A6C6 A6C6 2646 50C6 A6C6 03B6      CEB6         CEB6         F3B08BBB
83C5     83C5 83C5 83C5 83C5 4147 A6C7 A6C7 A6C7 A6C7 2647 50C7 A6C7 03B7      CEB7         CEB7         F3B08BBC
83C6     83C6 83C6 83C6 83C6 4148 A6C8 A6C8 A6C8 A6C8 2648 50C8 A6C8 03B8      CEB8         CEB8         F3B08BBD
83C7     83C7 83C7 83C7 83C7 4149 A6C9 A6C9 A6C9 A6C9 2649 50C9 A6C9 03B9      CEB9         CEB9         F3B08BBE
83C8     83C8 83C8 83C8 83C8 414A A6CA A6CA A6CA A6CA 264A 50D1 A6CA 03BA      CEBA         CEBA         F3B08BBF
83C9     83C9 83C9 83C9 83C9 414B A6CB A6CB A6CB A6CB 264B 50D2 A6CB 03BB      CEBB         CEBB         F3B08C80
83CA     83CA 83CA 83CA 83CA 414C A6CC A6CC A6CC A6CC 264C 50D3 A6CC 03BC      CEBC         CEBC         F3B08C81
83CB     83CB 83CB 83CB 83CB 414D A6CD A6CD A6CD A6CD 264D 50D4 A6CD 03BD      CEBD         CEBD         F3B08C82
83CC     83CC 83CC 83CC 83CC 414E A6CE A6CE A6CE A6CE 264E 50D5 A6CE 03BE      CEBE         CEBE         F3B08C83
83CD     83CD 83CD 83CD 83CD 414F A6CF A6CF A6CF A6CF 264F 50D6 A6CF 03BF      CEBF         CEBF         F3B08C84
83CE     83CE 83CE 83CE 83CE 4150 A6D0 A6D0 A6D0 A6D0 2650 50D7 A6D0 03C0      CF80         CF80         F3B08C85
83CF     83CF 83CF 83CF 83CF 4151 A6D1 A6D1 A6D1 A6D1 2651 50D8 A6D1 03C1      CF81         CF81         F3B08C86
83D0     83D0 83D0 83D0 83D0 4152 A6D2 A6D2 A6D2 A6D2 2652 50D9 A6D2 03C3      CF83         CF83         F3B08C87
83D1     83D1 83D1 83D1 83D1 4153 A6D3 A6D3 A6D3 A6D3 2653 50E2 A6D3 03C4      CF84         CF84         F3B08C88
83D2     83D2 83D2 83D2 83D2 4154 A6D4 A6D4 A6D4 A6D4 2654 50E3 A6D4 03C5      CF85         CF85         F3B08C89
83D3     83D3 83D3 83D3 83D3 4155 A6D5 A6D5 A6D5 A6D5 2655 50E4 A6D5 03C6      CF86         CF86         F3B08C8A
83D4     83D4 83D4 83D4 83D4 4156 A6D6 A6D6 A6D6 A6D6 2656 50E5 A6D6 03C7      CF87         CF87         F3B08C8B
83D5     83D5 83D5 83D5 83D5 4157 A6D7 A6D7 A6D7 A6D7 2657 50E6 A6D7 03C8      CF88         CF88         F3B08C8C
83D6     83D6 83D6 83D6 83D6 4158 A6D8 A6D8 A6D8 A6D8 2658 50E7 A6D8 03C9      CF89         CF89         F3B08C8D
8440     8440 8440 8440 8440 41C0 A7A1 A7A1 A7A1 A7A1 2721 7D4F A7A1 0410      D090         D090         F3B08CB4
8441     8441 8441 8441 8441 41C1 A7A2 A7A2 A7A2 A7A2 2722 7D7F A7A2 0411      D091         D091         F3B08CB5
8442     8442 8442 8442 8442 41C2 A7A3 A7A3 A7A3 A7A3 2723 7D7B A7A3 0412      D092         D092         F3B08CB6
8443     8443 8443 8443 8443 41C3 A7A4 A7A4 A7A4 A7A4 2724 7DE0 A7A4 0413      D093         D093         F3B08CB7
8444     8444 8444 8444 8444 41C4 A7A5 A7A5 A7A5 A7A5 2725 7D6C A7A5 0414      D094         D094         F3B08CB8
8445     8445 8445 8445 8445 41C5 A7A6 A7A6 A7A6 A7A6 2726 7D50 A7A6 0415      D095         D095         F3B08CB9
8446     8446 8446 8446 8446 41C6 A7A7 A7A7 A7A7 A7A7 2727 7D7D A7A7 0401      D081         D081         F3B08CBA
8447     8447 8447 8447 8447 41C7 A7A8 A7A8 A7A8 A7A8 2728 7D4D A7A8 0416      D096         D096         F3B08CBB
8448     8448 8448 8448 8448 41C8 A7A9 A7A9 A7A9 A7A9 2729 7D5D A7A9 0417      D097         D097         F3B08CBC
8449     8449 8449 8449 8449 41C9 A7AA A7AA A7AA A7AA 272A 7D5C A7AA 0418      D098         D098         F3B08CBD
844A     844A 844A 844A 844A 41CA A7AB A7AB A7AB A7AB 272B 7D4E A7AB 0419      D099         D099         F3B08CBE
844B     844B 844B 844B 844B 41CB A7AC A7AC A7AC A7AC 272C 7D6B A7AC 041A      D09A         D09A         F3B08CBF
844C     844C 844C 844C 844C 41CC A7AD A7AD A7AD A7AD 272D 7D60 A7AD 041B      D09B         D09B         F3B08D80
844D     844D 844D 844D 844D 41CD A7AE A7AE A7AE A7AE 272E 7D4B A7AE 041C      D09C         D09C         F3B08D81
844E     844E 844E 844E 844E 41CE A7AF A7AF A7AF A7AF 272F 7D61 A7AF 041D      D09D         D09D         F3B08D82
844F     844F 844F 844F 844F 41CF A7B0 A7B0 A7B0 A7B0 2730 7DF0 A7B0 041E      D09E         D09E         F3B08D83
8450     8450 8450 8450 8450 41D0 A7B1 A7B1 A7B1 A7B1 2731 7DF1 A7B1 041F      D09F         D09F         F3B08D84
8451     8451 8451 8451 8451 41D1 A7B2 A7B2 A7B2 A7B2 2732 7DF2 A7B2 0420      D0A0         D0A0         F3B08D85
8452     8452 8452 8452 8452 41D2 A7B3 A7B3 A7B3 A7B3 2733 7DF3 A7B3 0421      D0A1         D0A1         F3B08D86
8453     8453 8453 8453 8453 41D3 A7B4 A7B4 A7B4 A7B4 2734 7DF4 A7B4 0422      D0A2         D0A2         F3B08D87
8454     8454 8454 8454 8454 41D4 A7B5 A7B5 A7B5 A7B5 2735 7DF5 A7B5 0423      D0A3         D0A3         F3B08D88
8455     8455 8455 8455 8455 41D5 A7B6 A7B6 A7B6 A7B6 2736 7DF6 A7B6 0424      D0A4         D0A4         F3B08D89
8456     8456 8456 8456 8456 41D6 A7B7 A7B7 A7B7 A7B7 2737 7DF7 A7B7 0425      D0A5         D0A5         F3B08D8A
8457     8457 8457 8457 8457 41D7 A7B8 A7B8 A7B8 A7B8 2738 7DF8 A7B8 0426      D0A6         D0A6         F3B08D8B
8458     8458 8458 8458 8458 41D8 A7B9 A7B9 A7B9 A7B9 2739 7DF9 A7B9 0427      D0A7         D0A7         F3B08D8C
8459     8459 8459 8459 8459 41D9 A7BA A7BA A7BA A7BA 273A 7D7A A7BA 0428      D0A8         D0A8         F3B08D8D
845A     845A 845A 845A 845A 41DA A7BB A7BB A7BB A7BB 273B 7D5E A7BB 0429      D0A9         D0A9         F3B08D8E
845B     845B 845B 845B 845B 41DB A7BC A7BC A7BC A7BC 273C 7D4C A7BC 042A      D0AA         D0AA         F3B08D8F
845C     845C 845C 845C 845C 41DC A7BD A7BD A7BD A7BD 273D 7D7E A7BD 042B      D0AB         D0AB         F3B08D90
845D     845D 845D 845D 845D 41DD A7BE A7BE A7BE A7BE 273E 7D6E A7BE 042C      D0AC         D0AC         F3B08D91
845E     845E 845E 845E 845E 41DE A7BF A7BF A7BF A7BF 273F 7D6F A7BF 042D      D0AD         D0AD         F3B08D92
845F     845F 845F 845F 845F 41DF A7C0 A7C0 A7C0 A7C0 2740 7D7C A7C0 042E      D0AE         D0AE         F3B08D93
8460     8460 8460 8460 8460 41E0 A7C1 A7C1 A7C1 A7C1 2741 7DC1 A7C1 042F      D0AF         D0AF         F3B08D94
8470     8470 8470 8470 8470 4180 A7D1 A7D1 A7D1 A7D1 2751 7DD8 A7D1 0430      D0B0         D0B0         F3B08DA4
8471     8471 8471 8471 8471 4181 A7D2 A7D2 A7D2 A7D2 2752 7DD9 A7D2 0431      D0B1         D0B1         F3B08DA5
8472     8472 8472 8472 8472 4182 A7D3 A7D3 A7D3 A7D3 2753 7DE2 A7D3 0432      D0B2         D0B2         F3B08DA6
8473     8473 8473 8473 8473 4183 A7D4 A7D4 A7D4 A7D4 2754 7DE3 A7D4 0433      D0B3         D0B3         F3B08DA7
8474     8474 8474 8474 8474 4184 A7D5 A7D5 A7D5 A7D5 2755 7DE4 A7D5 0434      D0B4         D0B4         F3B08DA8
8475     8475 8475 8475 8475 4185 A7D6 A7D6 A7D6 A7D6 2756 7DE5 A7D6 0435      D0B5         D0B5         F3B08DA9
8476     8476 8476 8476 8476 4186 A7D7 A7D7 A7D7 A7D7 2757 7DE6 A7D7 0451      D191         D191         F3B08DAA
8477     8477 8477 8477 8477 4187 A7D8 A7D8 A7D8 A7D8 2758 7DE7 A7D8 0436      D0B6         D0B6         F3B08DAB
8478     8478 8478 8478 8478 4188 A7D9 A7D9 A7D9 A7D9 2759 7DE8 A7D9 0437      D0B7         D0B7         F3B08DAC
8479     8479 8479 8479 8479 4189 A7DA A7DA A7DA A7DA 275A 7DE9 A7DA 0438      D0B8         D0B8         F3B08DAD
847A     847A 847A 847A 847A 418A A7DB A7DB A7DB A7DB 275B 7D4A A7DB 0439      D0B9         D0B9         F3B08DAE
847B     847B 847B 847B 847B 418B A7DC A7DC A7DC A7DC 275C 7D5B A7DC 043A      D0BA         D0BA         F3B08DAF
847C     847C 847C 847C 847C 418C A7DD A7DD A7DD A7DD 275D 7D5A A7DD 043B      D0BB         D0BB         F3B08DB0
847D     847D 847D 847D 847D 418D A7DE A7DE A7DE A7DE 275E 7D5F A7DE 043C      D0BC         D0BC         F3B08DB1
847E     847E 847E 847E 847E 418E A7DF A7DF A7DF A7DF 275F 7D6D A7DF 043D      D0BD         D0BD         F3B08DB2
8480     8480 8480 8480 8480 418F A7E0 A7E0 A7E0 A7E0 2760 7D79 A7E0 043E      D0BE         D0BE         F3B08DB3
8481     8481 8481 8481 8481 4190 A7E1 A7E1 A7E1 A7E1 2761 7D57 A7E1 043F      D0BF         D0BF         F3B08DB4
8482     8482 8482 8482 8482 4191 A7E2 A7E2 A7E2 A7E2 2762 7D59 A7E2 0440      D180         D180         F3B08DB5
8483     8483 8483 8483 8483 4192 A7E3 A7E3 A7E3 A7E3 2763 7D62 A7E3 0441      D181         D181         F3B08DB6
8484     8484 8484 8484 8484 4193 A7E4 A7E4 A7E4 A7E4 2764 7D63 A7E4 0442      D182         D182         F3B08DB7
8485     8485 8485 8485 8485 4194 A7E5 A7E5 A7E5 A7E5 2765 7D64 A7E5 0443      D183         D183         F3B08DB8
8486     8486 8486 8486 8486 4195 A7E6 A7E6 A7E6 A7E6 2766 7D65 A7E6 0444      D184         D184         F3B08DB9
8487     8487 8487 8487 8487 4196 A7E7 A7E7 A7E7 A7E7 2767 7D66 A7E7 0445      D185         D185         F3B08DBA
8488     8488 8488 8488 8488 4197 A7E8 A7E8 A7E8 A7E8 2768 7D67 A7E8 0446      D186         D186         F3B08DBB
8489     8489 8489 8489 8489 4198 A7E9 A7E9 A7E9 A7E9 2769 7D68 A7E9 0447      D187         D187         F3B08DBC
848A     848A 848A 848A 848A 4199 A7EA A7EA A7EA A7EA 276A 7D69 A7EA 0448      D188         D188         F3B08DBD
848B     848B 848B 848B 848B 419A A7EB A7EB A7EB A7EB 276B 7D70 A7EB 0449      D189         D189         F3B08DBE
848C     848C 848C 848C 848C 419B A7EC A7EC A7EC A7EC 276C 7D71 A7EC 044A      D18A         D18A         F3B08DBF
848D     848D 848D 848D 848D 419C A7ED A7ED A7ED A7ED 276D 7D72 A7ED 044B      D18B         D18B         F3B08E80
848E     848E 848E 848E 848E 419D A7EE A7EE A7EE A7EE 276E 7D73 A7EE 044C      D18C         D18C         F3B08E81
848F     848F 848F 848F 848F 419E A7EF A7EF A7EF A7EF 276F 7D74 A7EF 044D      D18D         D18D         F3B08E82
8490     8490 8490 8490 8490 419F A7F0 A7F0 A7F0 A7F0 2770 7D75 A7F0 044E      D18E         D18E         F3B08E83
8491     8491 8491 8491 8491 41A0 A7F1 A7F1 A7F1 A7F1 2771 7D76 A7F1 044F      D18F         D18F         F3B08E84
# Category 1(2 of 2) JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
849F     849F 849F 849F 849F 437C AFBC A8A1 A8A1 7CD1 2C24 6BE0 A8A1 2500      E29480       E29480       F3B08E92
84A0     84A0 84A0 84A0 84A0 437D AFBF A8A2 A8A2 7CD2 2C26 6B50 A8A2 2502      E29482       E29482       F3B08E93
84A1     84A1 84A1 84A1 84A1 437E AFA1 A8A3 A8A3 7CC1 2C30 6BF0 A8A3 250C      E2948C       E2948C       F3B08E94
84A2     84A2 84A2 84A2 84A2 437F AFA4 A8A4 A8A4 7CC2 2C34 6BF4 A8A4 2510      E29490       E29490       F3B08E95
84A3     84A3 84A3 84A3 84A3 43B0 AFAA A8A5 A8A5 7CC3 2C3C 6B4C A8A5 2518      E29498       E29498       F3B08E96
84A4     84A4 84A4 84A4 84A4 43B1 AFA7 A8A6 A8A6 7CC4 2C38 6BF8 A8A6 2514      E29494       E29494       F3B08E97
84A5     84A5 84A5 84A5 84A5 43B2 AFAD A8A7 A8A7 7CC7 2C40 6B7C A8A7 251C      E2949C       E2949C       F3B08E98
84A6     84A6 84A6 84A6 84A6 43B3 AFB3 A8A8 A8A8 7CC5 2C50 6BD7 A8A8 252C      E294AC       E294AC       F3B08E99
84A7     84A7 84A7 84A7 84A7 43B4 AFB0 A8A9 A8A9 7CC8 2C48 6BC8 A8A9 2524      E294A4       E294A4       F3B08E9A
84A8     84A8 84A8 84A8 84A8 43B5 AFB6 A8AA A8AA 7CC6 2C58 6BE7 A8AA 2534      E294B4       E294B4       F3B08E9B
84A9     84A9 84A9 84A9 84A9 43B6 AFB9 A8AB A8AB 7CC9 2C60 6B79 A8AB 253C      E294BC       E294BC       F3B08E9C
84AA     84AA 84AA 84AA 84AA 43B7 AFBD A8AC A8AC 7CF6 2C25 6B6C A8AC 2501      E29481       E29481       F3B08E9D
84AB     84AB 84AB 84AB 84AB 43B8 AFC0 A8AD A8AD 7CF7 2C27 6B7D A8AD 2503      E29483       E29483       F3B08E9E
84AC     84AC 84AC 84AC 84AC 43B9 AFA2 A8AE A8AE 7CE6 2C33 6BF3 A8AE 250F      E2948F       E2948F       F3B08E9F
84AD     84AD 84AD 84AD 84AD 43E1 AFA5 A8AF A8AF 7CE7 2C37 6BF7 A8AF 2513      E29493       E29493       F3B08EA0
84AE     84AE 84AE 84AE 84AE 43E2 AFAB A8B0 A8B0 7CE8 2C3F 6B6F A8B0 251B      E2949B       E2949B       F3B08EA1
84AF     84AF 84AF 84AF 84AF 43E3 AFA8 A8B1 A8B1 7CE9 2C3B 6B5E A8B1 2517      E29497       E29497       F3B08EA2
84B0     84B0 84B0 84B0 84B0 43E4 AFAE A8B2 A8B2 7CF3 2C47 6BC7 A8B2 2523      E294A3       E294A3       F3B08EA3
84B1     84B1 84B1 84B1 84B1 43E5 AFB4 A8B3 A8B3 7CF1 2C57 6BE6 A8B3 2533      E294B3       E294B3       F3B08EA4
84B2     84B2 84B2 84B2 84B2 43E6 AFB1 A8B4 A8B4 7CF4 2C4F 6BD6 A8B4 252B      E294AB       E294AB       F3B08EA5
84B3     84B3 84B3 84B3 84B3 43E7 AFB7 A8B5 A8B5 7CF2 2C5F 6B6D A8B5 253B      E294BB       E294BB       F3B08EA6
84B4     84B4 84B4 84B4 84B4 43E8 AFBA A8B6 A8B6 7CF5 2C6F 6B74 A8B6 254B      E2958B       E2958B       F3B08EA7
84B5     84B5 84B5 84B5 84B5 43E9 A8B7 A8B7 A8B7 7CCA 2C44 6BC4 A8B7 2520      E294A0       E294A0       F3B08EA8
84B6     84B6 84B6 84B6 84B6 43EA A8B8 A8B8 A8B8 7CCB 2C53 6BE2 A8B8 252F      E294AF       E294AF       F3B08EA9
84B7     84B7 84B7 84B7 84B7 43EB A8B9 A8B9 A8B9 7CCC 2C4C 6BD3 A8B9 2528      E294A8       E294A8       F3B08EAA
84B8     84B8 84B8 84B8 84B8 43EC A8BA A8BA A8BA 7CC0 2C5B 6B4A A8BA 2537      E294B7       E294B7       F3B08EAB
84B9     84B9 84B9 84B9 84B9 43ED A8BB A8BB A8BB 7CCE 2C63 6B62 A8BB 253F      E294BF       E294BF       F3B08EAC
84BA     84BA 84BA 84BA 84BA 43EE A8BC A8BC A8BC 7FA2 2C41 6BC1 A8BC 251D      E2949D       E2949D       F3B08EAD
84BB     84BB 84BB 84BB 84BB 43EF A8BD A8BD A8BD 7FA3 2C54 6BE3 A8BD 2530      E294B0       E294B0       F3B08EAE
84BC     84BC 84BC 84BC 84BC 43F0 A8BE A8BE A8BE 7FA4 2C49 6BC9 A8BE 2525      E294A5       E294A5       F3B08EAF
84BD     84BD 84BD 84BD 84BD 43F1 A8BF A8BF A8BF 7FA1 2C5C 6B5B A8BF 2538      E294B8       E294B8       F3B08EB0
84BE     84BE 84BE 84BE 84BE 43F2 A8C0 A8C0 A8C0 7CCD 2C66 6B65 A8C0 2542      E29582       E29582       F3B08EB1
# End of Category 1(2 of 2) JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
# NEC Kanji Row 13, Appendix C, CJKV Information Processing by Ken Lunde 1999
8740     8740 8740 8740 8740 E270 76B1 76B1 76B1 77C9 2D21 604F  --  2460      E291A0       E291A0       F3B095A8
8741     8741 8741 8741 8741 E271 76B2 76B2 76B2 77CA 2D22 607F  --  2461      E291A1       E291A1       F3B095A9
8742     8742 8742 8742 8742 E272 76B3 76B3 76B3 77CB 2D23 607B  --  2462      E291A2       E291A2       F3B095AA
8743     8743 8743 8743 8743 E273 76B4 76B4 76B4 77CC 2D24 60E0  --  2463      E291A3       E291A3       F3B095AB
8744     8744 8744 8744 8744 E274 76B5 76B5 76B5 77CD 2D25 606C  --  2464      E291A4       E291A4       F3B095AC
8745     8745 8745 8745 8745 E275 76B6 76B6 76B6 77CE 2D26 6050  --  2465      E291A5       E291A5       F3B095AD
8746     8746 8746 8746 8746 E276 76B7 76B7 76B7 77CF 2D27 607D  --  2466      E291A6       E291A6       F3B095AE
8747     8747 8747 8747 8747 E277 76B8 76B8 76B8 77D0 2D28 604D  --  2467      E291A7       E291A7       F3B095AF
8748     8748 8748 8748 8748 E278 76B9 76B9 76B9 77D1 2D29 605D  --  2468      E291A8       E291A8       F3B095B0
8749     8749 8749 8749 8749 E279 76BA 76BA 76BA 77D2 2D2A 605C  --  2469      E291A9       E291A9       F3B095B1
874A     874A 874A 874A 874A E27A 77A1 77A1 77A1 77D3 2D2B 604E  --  246A      E291AA       E291AA       F3B095B2
874B     874B 874B 874B 874B E27B 77A2 77A2 77A2 77D4 2D2C 606B  --  246B      E291AB       E291AB       F3B095B3
874C     874C 874C 874C 874C E27C 77A3 77A3 77A3 77D5 2D2D 6060  --  246C      E291AC       E291AC       F3B095B4
874D     874D 874D 874D 874D E27D 77A4 77A4 77A4 77D6 2D2E 604B  --  246D      E291AD       E291AD       F3B095B5
874E     874E 874E 874E 874E E27E 77A5 77A5 77A5 77D7 2D2F 6061  --  246E      E291AE       E291AE       F3B095B6
874F     874F 874F 874F 874F E27F 77A6 77A6 77A6 77D8 2D30 60F0  --  246F      E291AF       E291AF       F3B095B7
8750     8750 8750 8750 8750 E280 77A7 77A7 77A7 77D9 2D31 60F1  --  2470      E291B0       E291B0       F3B095B8
8751     8751 8751 8751 8751 E281 77A8 77A8 77A8 77DA 2D32 60F2  --  2471      E291B1       E291B1       F3B095B9
8752     8752 8752 8752 8752 E282 77A9 77A9 77A9 77DB 2D33 60F3  --  2472      E291B2       E291B2       F3B095BA
8753     8753 8753 8753 8753 E283 77AA 77AA 77AA 77DC 2D34 60F4  --  2473      E291B3       E291B3       F3B095BB
8754     8754 FA4A 8754 8754 41F1 74B1 74B1 74B1 77DE 2D35 60F5  --  2160      E285A0       E285A0       F3B095BC
8755     8755 FA4B 8755 8755 41F2 74B2 74B2 74B2 77DF 2D36 60F6  --  2161      E285A1       E285A1       F3B095BD
8756     8756 FA4C 8756 8756 41F3 74B3 74B3 74B3 77E0 2D37 60F7  --  2162      E285A2       E285A2       F3B095BE
8757     8757 FA4D 8757 8757 41F4 74B4 74B4 74B4 77E1 2D38 60F8  --  2163      E285A3       E285A3       F3B095BF
8758     8758 FA4E 8758 8758 41F5 74B5 74B5 74B5 77E2 2D39 60F9  --  2164      E285A4       E285A4       F3B09680
8759     8759 FA4F 8759 8759 41F6 74B6 74B6 74B6 77E3 2D3A 607A  --  2165      E285A5       E285A5       F3B09681
875A     875A FA50 875A 875A 41F7 74B7 74B7 74B7 77E4 2D3B 605E  --  2166      E285A6       E285A6       F3B09682
875B     875B FA51 875B 875B 41F8 74B8 74B8 74B8 77E5 2D3C 604C  --  2167      E285A7       E285A7       F3B09683
875C     875C FA52 875C 875C 41F9 74B9 74B9 74B9 77E6 2D3D 607E  --  2168      E285A8       E285A8       F3B09684
875D     875D FA53 875D 875D 41FA 74BA 74BA 74BA 77E7 2D3E 606E  --  2169      E285A9       E285A9       F3B09685
875F     875F 875F 875F 875F E8AA 70CF 70CF 70CF 76DA 2D40 607C  --  3349      E38D89       E38D89       F3B09687
8760     8760 8760 8760 8760 E875 70D1 70D1 70D1 76DC 2D41 60C1  --  3314      E38C94       E38C94       F3B09688
8761     8761 8761 8761 8761 E883 70D0 70D0 70D0 76DB 2D42 60C2  --  3322      E38CA2       E38CA2       F3B09689
8762     8762 8762 8762 8762 E8AE 70D3 70D3 70D3 76D9 2D43 60C3  --  334D      E38D8D       E38D8D       F3B0968A
8763     8763 8763 8763 8763 E879 70D5 70D5 70D5 76DE 2D44 60C4  --  3318      E38C98       E38C98       F3B0968B
8764     8764 8764 8764 8764 E888 70D7 70D7 70D7 76E0 2D45 60C5  --  3327      E38CA7       E38CA7       F3B0968C
8765     8765 8765 8765 8765 E864 70D8 70D8 70D8 76E1 2D46 60C6  --  3303      E38C83       E38C83       F3B0968D
8766     8766 8766 8766 8766 E897 70D9 70D9 70D9 76E2 2D47 60C7  --  3336      E38CB6       E38CB6       F3B0968E
8767     8767 8767 8767 8767 E8B2  --   --   --   --  2D48 60C8  --  3351      E38D91       E38D91       F3B0968F
8768     8768 8768 8768 8768 E8B8 70DB 70DB 70DB 76E7 2D49 60C9  --  3357      E38D97       E38D97       F3B09690
8769     8769 8769 8769 8769 E86E 70DC 70DC 70DC 76E6 2D4A 60D1  --  330D      E38C8D       E38C8D       F3B09691
876A     876A 876A 876A 876A E887 70E2 70E2 70E2 76E3 2D4B 60D2  --  3326      E38CA6       E38CA6       F3B09692
876B     876B 876B 876B 876B E884 70E1 70E1 70E1 76E4 2D4C 60D3  --  3323      E38CA3       E38CA3       F3B09693
876C     876C 876C 876C 876C E88C 70E0 70E0 70E0 76E5 2D4D 60D4  --  332B      E38CAB       E38CAB       F3B09694
876D     876D 876D 876D 876D E8AB 70DA 70DA 70DA 76E9 2D4E 60D5  --  334A      E38D8A       E38D8A       F3B09695
876E     876E 876E 876E 876E E89C 70DF 70DF 70DF 76EB 2D4F 60D6  --  333B      E38CBB       E38CBB       F3B09696
876F     876F 876F 876F 876F E8F9 6FCC 6FCC 6FCC 76BC 2D50 60D7  --  339C      E38E9C       E38E9C       F3B09697
8770     8770 8770 8770 8770 E8FA 6FCB 6FCB 6FCB 76BD 2D51 60D8  --  339D      E38E9D       E38E9D       F3B09698
8771     8771 8771 8771 8771 E8FB 6FCA 6FCA 6FCA 76BE 2D52 60D9  --  339E      E38E9E       E38E9E       F3B09699
8772     8772 8772 8772 8772 E8EB 6FDA 6FDA 6FDA 76C4 2D53 60E2  --  338E      E38E8E       E38E8E       F3B0969A
8773     8773 8773 8773 8773 E8EC 6FD9 6FD9 6FD9 76C5 2D54 60E3  --  338F      E38E8F       E38E8F       F3B0969B
8774     8774 8774 8774 8774 E963 6FFA 6FFA 6FFA 76C6 2D55 60E4  --  33C4      E38F84       E38F84       F3B0969C
8775     8775 8775 8775 8775 E8FE 6FD0 6FD0 6FD0 76C0 2D56 60E5  --  33A1      E38EA1       E38EA1       F3B0969D
877E     877E 877E 877E 877E E8D8 73F8 73F8 73F8  --  2D5F 606D  --  337B      E38DBB       E38DBB       F3B096A6
8780     8780 8780 8780 8780 E6AC  --   --   --   --  2D60 6079  --  301D      E3809D       E3809D       F3B096A7
8781     8781 8781 8781 8781 E6AE  --   --   --   --  2D61 6057  --  301F      E3809F       E3809F       F3B096A8
8782     8782 FA59 8782 8782 446E 72BE 72BE 72BE 76FA 2D62 6059 A271 2116      E28496       E28496       F3B096A9
8783     8783 8783 8783 8783 E96C 72B6 72B6 72B6 76F4 2D63 6062  --  33CD      E38F8D       E38F8D       F3B096AA
8784     8784 FA5A 8784 8784 446F 72B8 72B8 72B8 76F6 2D64 6063  --  2121      E284A1       E284A1       F3B096AB
8785     8785 8785 8785 8785 E767 73EA 73EA 73EA  --  2D65 6064  --  32A4      E38AA4       E38AA4       F3B096AC
8786     8786 8786 8786 8786 E768 73EB 73EB 73EB  --  2D66 6065  --  32A5      E38AA5       E38AA5       F3B096AD
8787     8787 8787 8787 8787 E769 73EC 73EC 73EC  --  2D67 6066  --  32A6      E38AA6       E38AA6       F3B096AE
8788     8788 8788 8788 8788 E76A 73ED 73ED 73ED  --  2D68 6067  --  32A7      E38AA7       E38AA7       F3B096AF
8789     8789 8789 8789 8789 E76B 73EE 73EE 73EE  --  2D69 6068  --  32A8      E38AA8       E38AA8       F3B096B0
878A     878A FA58 878A 878A 446D 73D0 73D0 73D0 78D5 2D6A 6069  --  3231      E388B1       E388B1       F3B096B1
878B     878B 878B 878B 878B E6EF 73B1 73B1 73B1 78CF 2D6B 6070  --  3232      E388B2       E388B2       F3B096B2
878C     878C 878C 878C 878C E6F6 73BD 73BD 73BD 78CB 2D6C 6071  --  3239      E388B9       E388B9       F3B096B3
878D     878D 878D 878D 878D E8DB 72F5 72F5 72F5  --  2D6D 6072  --  337E      E38DBE       E38DBE       F3B096B4
878E     878E 878E 878E 878E E8DA 72F6 72F6 72F6  --  2D6E 6073  --  337D      E38DBD       E38DBD       F3B096B5
878F     878F 878F 878F 878F E8D9 72F7 72F7 72F7  --  2D6F 6074  --  337C      E38DBC       E38DBC       F3B096B6
8793     8793 8793 8793 8793 DFE8 6FB8 6FB8 6FB8  --  2D73 6078  --  222E      E288AE       E288AE       F3B096BA
8794     8794 8794 8794  --  DFD6 6FA2 6FA2 6FA2  --  2D74 6080  --  2211      E28891       E28891       F3B096BB
8798     8798 8798 8798 8798 DFE0 6FB7 6FB7 6FB7  --  2D78 60A0  --  221F      E2889F       E2889F       F3B096BF
8799     8799 8799 8799 8799 E0AC 6FBA 6FBA 6FBA  --  2D79 60AB  --  22BF      E28ABF       E28ABF       F3B09780
# End of NEC Kanji Row 13, Appendix C, CJKV Information Processing by Ken Lunde 1999
889F     889F 889F 889F 889F 4867 B0A1 B0A1 B0A1 B0A1 3021 F04F B0A1 4E9C      E4BA9C       E4BA9C       F3B09A82
88A0     88A0 88A0 88A0 88A0 54D4 B0A2 B0A2 B0A2 47B9 D6A1 AC41 B0A2 5516      E59496       E59496       F3B09A83
88A1     88A1 88A1 88A1 88A1 557A B0A3 B0A3 B0A3 B0A3 3023 F07B B0A3 5A03      E5A883       E5A883       F3B09A84
88A2     88A2 88A2 88A2 88A2 46AE B0A4 B0A4 B0A4 B0A4 3024 F0E0 B0A4 963F      E998BF       E998BF       F3B09A85
88A3     88A3 88A3 88A3 88A3 4EF9 B0A5 B0A5 B0A5 B0A5 3025 F06C B0A5 54C0      E59380       E59380       F3B09A86
88A4     88A4 88A4 88A4 88A4 45C6 B0A6 B0A6 B0A6 B0A6 3026 F050 B0A6 611B      E6849B       E6849B       F3B09A87
88A5     88A5 88A5 88A5 88A5 4F81 B0A7 B0A7 B0A7 B0A7 3027 F07D B0A7 6328      E68CA8       E68CA8       F3B09A88
88A6     88A6 88A6 88A6 88A6 5179 B0A8 B0A8 B0A8 B0A8 3028 F04D B0A8 59F6      E5A7B6       E5A7B6       F3B09A89
88A7     88A7 88A7 88A7 88A7 4DCE B0A9 B0A9 B0A9 B0A9 3029 F05D B0A9 9022      E980A2       E980A2       F3B09A8A
88A8     88A8 88A8 88A8 88A8 50D9 B0AA B0AA B0AA B0AA 302A F05C B0AA 8475      E891B5       E891B5       F3B09A8B
88A9     88A9 88A9 88A9 88A9 50E2 B0AB B0AB B0AB B0AB 302B F04E B0AB 831C      E88C9C       E88C9C       F3B09A8C
88AA     88AA 88AA 88AA 88AA 509B B0AC B0AC B0AC B0AC 302C F06B B0AC 7A50      E7A990       E7A990       F3B09A8D
88AB     88AB 88AB 88AB 88AB 4965 B0AD B0AD B0AD B0AD 302D F060 B0AD 60AA      E682AA       E682AA       F3B09A8E
88AC     88AC 88AC 88AC 88AC 4D60 B0AE B0AE B0AE B0AE 302E F04B B0AE 63E1      E68FA1       E68FA1       F3B09A8F
88AD     88AD 88AD 88AD 88AD 4E43 B0AF B0AF B0AF B0AF 302F F061 B0AF 6E25      E6B8A5       E6B8A5       F3B09A90
88AE     88AE 88AE 88AE 88AE 47EC B0B0 B0B0 B0B0 B0B0 3030 F0F0 B0B0 65ED      E697AD       E697AD       F3B09A91
88AF     88AF 88AF 88AF 88AF 4E72 B0B1 B0B1 B0B1 B0B1 3031 F0F1 B0B1 8466      E891A6       E891A6       F3B09A92
88B0     88B0 88B0 88B0 88B0 4ADD B0B2 B0B2 B0B2 B0B2 3032 F0F2 B0B2 82A6      E88AA6       E88AA6       F3B09A93
88B1     88B1 88B1 88B1 88B1 67AA F2CD B0B3 B0B3 F2CD 724D 77D4 B0B3 9BF5      E9AFB5       E9AFB5       F3B09A94
88B2     88B2 88B2 88B2 88B2 50DA B0B4 B0B4 B0B4 B0B4 3034 F0F4 B0B4 6893      E6A293       E6A293       F3B09A95
88B3     88B3 88B3 88B3 88B3 4782 B0B5 B0B5 B0B5 B0B5 3035 F0F5 B0B5 5727      E59CA7       E59CA7       F3B09A96
88B4     88B4 88B4 88B4 88B4 5581 B0B6 B0B6 B0B6 B0B6 3036 F0F6 B0B6 65A1      E696A1       E696A1       F3B09A97
88B5     88B5 88B5 88B5 88B5 4C69 B0B7 B0B7 B0B7 B0B7 3037 F0F7 B0B7 6271      E689B1       E689B1       F3B09A98
88B6     88B6 88B6 88B6 88B6 50AB B0B8 B0B8 B0B8 B0B8 3038 F0F8 B0B8 5B9B      E5AE9B       E5AE9B       F3B09A99
88B7     88B7 88B7 88B7 88B7 549C B0B9 B0B9 B0B9 B0B9 3039 F0F9 B0B9 59D0      E5A790       E5A790       F3B09A9A
88B8     88B8 88B8 88B8 88B8 5144 B0BA B0BA B0BA B0BA 303A F07A B0BA 867B      E899BB       E899BB       F3B09A9B
88B9     88B9 88B9 88B9 88B9 5245 B0BB B0BB B0BB B0BB 303B F05E B0BB 98F4      E9A3B4       E9A3B4       F3B09A9C
88BA     88BA 88BA 88BA 88BA 4EF7 B0BC B0BC B0BC B0BC 303C F04C B0BC 7D62      E7B5A2       E7B5A2       F3B09A9D
88BB     88BB 88BB 88BB 88BB 49DC B0BD B0BD B0BD B0BD 303D F07E B0BD 7DBE      E7B6BE       E7B6BE       F3B09A9E
88BC     88BC 88BC 88BC 88BC 4F8E B0BE B0BE B0BE B0BE 303E F06E B0BE 9B8E      E9AE8E       E9AE8E       F3B09A9F
88BD     88BD 88BD 88BD 88BD 4D7F B0BF B0BF B0BF B0BF 303F F06F B0BF 6216      E68896       E68896       F3B09AA0
88BE     88BE 88BE 88BE 88BE 4BFB B0C0 B0C0 B0C0 B0C0 3040 F07C B0C0 7C9F      E7B29F       E7B29F       F3B09AA1
88BF     88BF 88BF 88BF 88BF 5448 B0C1 B0C1 B0C1 B0C1 3041 F0C1 B0C1 88B7      E8A2B7       E8A2B7       F3B09AA2
88C0     88C0 88C0 88C0 88C0 45A8 B0C2 B0C2 B0C2 B0C2 3042 F0C2 B0C2 5B89      E5AE89       E5AE89       F3B09AA3
88C1     88C1 88C1 88C1 88C1 4EEB B0C3 B0C3 B0C3 B0C3 3043 F0C3 B0C3 5EB5      E5BAB5       E5BAB5       F3B09AA4
88C2     88C2 88C2 88C2 88C2 5489 B0C4 B0C4 B0C4 B0C4 3044 F0C4 B0C4 6309      E68C89       E68C89       F3B09AA5
88C3     88C3 88C3 88C3 88C3 4BE2 B0C5 B0C5 B0C5 B0C5 3045 F0C5 B0C5 6697      E69A97       E69A97       F3B09AA6
88C4     88C4 88C4 88C4 88C4 48F1 B0C6 B0C6 B0C6 B0C6 3046 F0C6 B0C6 6848      E6A188       E6A188       F3B09AA7
88C5     88C5 88C5 88C5 88C5 5364 B0C7 B0C7 B0C7 B0C7 3047 F0C7 B0C7 95C7      E99787       E99787       F3B09AA8
88C6     88C6 88C6 88C6 88C6 504C B0C8 B0C8 B0C8 B0C8 3048 F0C8 B0C8 978D      E99E8D       E99E8D       F3B09AA9
88C7     88C7 88C7 88C7 88C7 50D0 B0C9 B0C9 B0C9 B0C9 3049 F0C9 B0C9 674F      E69D8F       E69D8F       F3B09AAA
88C8     88C8 88C8 88C8 88C8 46F6 B0CA B0CA B0CA B0CA 304A F0D1 B0CA 4EE5      E4BBA5       E4BBA5       F3B09AAB
88C9     88C9 88C9 88C9 88C9 45B2 B0CB B0CB B0CB B0CB 304B F0D2 B0CB 4F0A      E4BC8A       E4BC8A       F3B09AAC
88CA     88CA 88CA 88CA 88CA 47C8 B0CC B0CC B0CC B0CC 304C F0D3 B0CC 4F4D      E4BD8D       E4BD8D       F3B09AAD
88CB     88CB 88CB 88CB 88CB 4A9E B0CD B0CD B0CD B0CD 304D F0D4 B0CD 4F9D      E4BE9D       E4BE9D       F3B09AAE
88CC     88CC 88CC 88CC 88CC 4DFB B0CE B0CE B0CE B0CE 304E F0D5 B0CE 5049      E58189       E58189       F3B09AAF
88CD     88CD 88CD 88CD 88CD 4AFA B0CF B0CF B0CF B0CF 304F F0D6 B0CF 56F2      E59BB2       E59BB2       F3B09AB0
88CE     88CE 88CE 88CE 88CE 5147 B0D0 B0D0 B0D0 B0D0 3050 F0D7 B0D0 5937      E5A4B7       E5A4B7       F3B09AB1
88CF     88CF 88CF 88CF 88CF 484F B0D1 B0D1 B0D1 B0D1 3051 F0D8 B0D1 59D4      E5A794       E5A794       F3B09AB2
88D0     88D0 88D0 88D0 88D0 4C65 B0D2 B0D2 B0D2 B0D2 3052 F0D9 B0D2 5A01      E5A881       E5A881       F3B09AB3
88D1     88D1 88D1 88D1 88D1 5189 B0D3 B0D3 B0D3 B0D3 3053 F0E2 B0D3 5C09      E5B089       E5B089       F3B09AB4
88D2     88D2 88D2 88D2 88D2 4FE0 B0D4 B0D4 B0D4 B0D4 3054 F0E3 B0D4 60DF      E6839F       E6839F       F3B09AB5
88D3     88D3 88D3 88D3 88D3 4763 B0D5 B0D5 B0D5 B0D5 3055 F0E4 B0D5 610F      E6848F       E6848F       F3B09AB6
88D4     88D4 88D4 88D4 88D4 4FAD B0D6 B0D6 B0D6 B0D6 3056 F0E5 B0D6 6170      E685B0       E685B0       F3B09AB7
88D5     88D5 88D5 88D5 88D5 47E9 B0D7 B0D7 B0D7 B0D7 3057 F0E6 B0D7 6613      E69893       E69893       F3B09AB8
88D6     88D6 88D6 88D6 88D6 4FF7 B0D8 B0D8 B0D8 B0D8 3058 F0E7 B0D8 6905      E6A485       E6A485       F3B09AB9
88D7     88D7 88D7 88D7 88D7 49D7 B0D9 B0D9 B0D9 B0D9 3059 F0E8 B0D9 70BA      E782BA       E782BA       F3B09ABA
88D8     88D8 88D8 88D8 88D8 558F B0DA B0DA B0DA B0DA 305A F0E9 B0DA 754F      E7958F       E7958F       F3B09ABB
88D9     88D9 88D9 88D9 88D9 4AA9 B0DB B0DB B0DB B0DB 305B F04A B0DB 7570      E795B0       E795B0       F3B09ABC
88DA     88DA 88DA 88DA 88DA 4975 B0DC B0DC B0DC B0DC 305C F05B B0DC 79FB      E7A7BB       E7A7BB       F3B09ABD
88DB     88DB 88DB 88DB 88DB 4880 B0DD B0DD B0DD B0DD 305D F05A B0DD 7DAD      E7B6AD       E7B6AD       F3B09ABE
88DC     88DC 88DC 88DC 88DC 4E9A B0DE B0DE B0DE B0DE 305E F05F B0DE 7DEF      E7B7AF       E7B7AF       F3B09ABF
88DD     88DD 88DD 88DD 88DD 4DF6 B0DF B0DF B0DF B0DF 305F F06D B0DF 80C3      E88383       E88383       F3B09B80
88DE     88DE 88DE 88DE 88DE 5482 B0E0 B0E0 B0E0 B0E0 3060 F079 B0E0 840E      E8908E       E8908E       F3B09B81
88DF     88DF 88DF 88DF 88DF 4AEA B0E1 B0E1 B0E1 B0E1 3061 F057 B0E1 8863      E8A1A3       E8A1A3       F3B09B82
88E0     88E0 88E0 88E0 88E0 5573 B0E2 B0E2 B0E2 B0E2 3062 F059 B0E2 8B02      E8AC82       E8AC82       F3B09B83
88E1     88E1 88E1 88E1 88E1 49BC B0E3 B0E3 B0E3 B0E3 3063 F062 B0E3 9055      E98195       E98195       F3B09B84
88E2     88E2 88E2 88E2 88E2 4CDA B0E4 B0E4 B0E4 B0E4 3064 F063 B0E4 907A      E981BA       E981BA       F3B09B85
88E3     88E3 88E3 88E3 88E3 4A87 B0E5 B0E5 B0E5 B0E5 3065 F064 B0E5 533B      E58CBB       E58CBB       F3B09B86
88E4     88E4 88E4 88E4 88E4 456D B0E6 B0E6 B0E6 B0E6 3066 F065 B0E6 4E95      E4BA95       E4BA95       F3B09B87
88E5     88E5 88E5 88E5 88E5 4F7F B0E7 B0E7 B0E7 B0E7 3067 F066 B0E7 4EA5      E4BAA5       E4BAA5       F3B09B88
88E6     88E6 88E6 88E6 88E6 48E9 B0E8 B0E8 B0E8 B0E8 3068 F067 B0E8 57DF      E59F9F       E59F9F       F3B09B89
88E7     88E7 88E7 88E7 88E7 48AC B0E9 B0E9 B0E9 B0E9 3069 F068 B0E9 80B2      E882B2       E882B2       F3B09B8A
88E8     88E8 88E8 88E8 88E8 49B3 B0EA B0EA B0EA B0EA 306A F069 B0EA 90C1      E98381       E98381       F3B09B8B
88E9     88E9 88E9 88E9 88E9 48AA B0EB B0EB B0EB B0EB 306B F070 B0EB 78EF      E7A3AF       E7A3AF       F3B09B8C
88EA     88EA 88EA 88EA 88EA 4541 B0EC B0EC B0EC B0EC 306C F071 B0EC 4E00      E4B880       E4B880       F3B09B8D
88EB     88EB 88EB 88EB 88EB 4EB0 B0ED B0ED B0ED B0ED 306D F072 B0ED 58F1      E5A3B1       E5A3B1       F3B09B8E
88EC     88EC 88EC 88EC 88EC 5271 B0EE B0EE B0EE B0EE 306E F073 B0EE 6EA2      E6BAA2       E6BAA2       F3B09B8F
88ED     88ED 88ED 88ED 88ED 4AD6 B0EF B0EF B0EF B0EF 306F F074 B0EF 9038      E980B8       E980B8       F3B09B90
88EE     88EE 88EE 88EE 88EE 476A B0F0 B0F0 B0F0 B0F0 3070 F075 B0F0 7A32      E7A8B2       E7A8B2       F3B09B91
88EF     88EF 88EF 88EF 88EF 474C B0F1 B0F1 B0F1 B0F1 3071 F076 B0F1 8328      E88CA8       E88CA8       F3B09B92
88F0     88F0 88F0 88F0 88F0 519A B0F2 B0F2 B0F2 B0F2 3072 F077 B0F2 828B      E88A8B       E88A8B       F3B09B93
88F1     88F1 88F1 88F1 88F1 55C8 B0F3 B0F3 B0F3 B0F3 3073 F078 B0F3 9C2F      E9B0AF       E9B0AF       F3B09B94
88F2     88F2 88F2 88F2 88F2 4C91 B0F4 B0F4 B0F4 B0F4 3074 F080 B0F4 5141      E58581       E58581       F3B09B95
88F3     88F3 88F3 88F3 88F3 4799 B0F5 B0F5 B0F5 B0F5 3075 F08B B0F5 5370      E58DB0       E58DB0       F3B09B96
88F4     88F4 88F4 88F4 88F4 52B1 B0F6 B0F6 B0F6 B0F6 3076 F09B B0F6 54BD      E592BD       E592BD       F3B09B97
88F5     88F5 88F5 88F5 88F5 46CB B0F7 B0F7 B0F7 B0F7 3077 F09C B0F7 54E1      E593A1       E593A1       F3B09B98
88F6     88F6 88F6 88F6 88F6 4A49 B0F8 B0F8 B0F8 B0F8 3078 F0A0 B0F8 56E0      E59BA0       E59BA0       F3B09B99
88F7     88F7 88F7 88F7 88F7 52B9 B0F9 B0F9 B0F9 B0F9 3079 F0AB B0F9 59FB      E5A7BB       E5A7BB       F3B09B9A
88F8     88F8 88F8 88F8 88F8 46DA B0FA B0FA B0FA B0FA 307A F0B0 B0FA 5F15      E5BC95       E5BC95       F3B09B9B
88F9     88F9 88F9 88F9 88F9 4BA3 B0FB B0FB B0FB B0FB 307B F0C0 B0FB 98F2      E9A3B2       E9A3B2       F3B09B9C
88FA     88FA 88FA 88FA 88FA 554E B0FC B0FC B0FC B0FC 307C F06A B0FC 6DEB      E6B7AB       E6B7AB       F3B09B9D
88FB     88FB 88FB 88FB 88FB 4EE9 B0FD B0FD B0FD B0FD 307D F0D0 B0FD 80E4      E883A4       E883A4       F3B09B9E
88FC     88FC 88FC 88FC 88FC 4E8B B0FE B0FE B0FE B0FE 307E F0A1 B0FE 852D      E894AD       E894AD       F3B09B9F
8940     8940 8940 8940 8940 49D8 B1A1 B1A1 B1A1 B1A1 3121 F14F B1A1 9662      E999A2       E999A2       F3B09BA0
8941     8941 8941 8941 8941 4CD2 B1A2 B1A2 B1A2 B1A2 3122 F17F B1A2 9670      E999B0       E999B0       F3B09BA1
8942     8942 8942 8942 8942 4EAE B1A3 B1A3 B1A3 B1A3 3123 F17B B1A3 96A0      E99AA0       E99AA0       F3B09BA2
8943     8943 8943 8943 8943 5577 B1A4 B1A4 B1A4 B1A4 3124 F1E0 B1A4 97FB      E99FBB       E99FBB       F3B09BA3
8944     8944 8944 8944 8944 55BC B1A5 B1A5 B1A5 B1A5 3125 F16C B1A5 540B      E5908B       E5908B       F3B09BA4
8945     8945 8945 8945 8945 485A B1A6 B1A6 B1A6 B1A6 3126 F150 B1A6 53F3      E58FB3       E58FB3       F3B09BA5
8946     8946 8946 8946 8946 46D5 B1A7 B1A7 B1A7 B1A7 3127 F17D B1A7 5B87      E5AE87       E5AE87       F3B09BA6
8947     8947 8947 8947 8947 5099 B1A8 B1A8 B1A8 B1A8 3128 F14D B1A8 70CF      E7838F       E7838F       F3B09BA7
8948     8948 8948 8948 8948 478F B1A9 B1A9 B1A9 B1A9 3129 F15D B1A9 7FBD      E7BEBD       E7BEBD       F3B09BA8
8949     8949 8949 8949 8949 55DF B1AA B1AA B1AA B1AA 312A F15C B1AA 8FC2      E8BF82       E8BF82       F3B09BA9
894A     894A 894A 894A 894A 49DD B1AB B1AB B1AB B1AB 312B F14E B1AB 96E8      E99BA8       E99BA8       F3B09BAA
894B     894B 894B 894B 894B 4C82 B1AC B1AC B1AC B1AC 312C F16B B1AC 536F      E58DAF       E58DAF       F3B09BAB
894C     894C 894C 894C 894C 4BE7 B1AD B1AD B1AD B1AD 312D F160 B1AD 9D5C      E9B59C       E9B59C       F3B09BAC
894D     894D 894D 894D 894D 559E B1AE B1AE B1AE B1AE 312E F14B B1AE 7ABA      E7AABA       E7AABA       F3B09BAD
894E     894E 894E 894E 894E 4DEC B1AF B1AF B1AF B1AF 312F F161 B1AF 4E11      E4B891       E4B891       F3B09BAE
894F     894F 894F 894F 894F 4FB1 B1B0 B1B0 B1B0 B1B0 3130 F1F0 B1B0 7893      E7A293       E7A293       F3B09BAF
8950     8950 8950 8950 8950 4A89 B1B1 B1B1 B1B1 B1B1 3131 F1F1 B1B1 81FC      E887BC       E887BC       F3B09BB0
8951     8951 8951 8951 8951 51A0 B1B2 B1B2 B1B2 B1B2 3132 F1F2 B1B2 6E26      E6B8A6       E6B8A6       F3B09BB1
8952     8952 8952 8952 8952 50F3 B1B3 B1B3 B1B3 47BA 3133 F1F3 B1B3 5618      E59898       E59898       F3B09BB2
8953     8953 8953 8953 8953 4CDB B1B4 B1B4 B1B4 B1B4 3134 F1F4 B1B4 5504      E59484       E59484       F3B09BB3
8954     8954 8954 8954 8954 55F7 B1B5 B1B5 B1B5 B1B5 3135 F1F5 B1B5 6B1D      E6AC9D       E6AC9D       F3B09BB4
8955     8955 8955 8955 8955 549A B1B6 B1B6 B1B6 B1B6 3136 F1F6 B1B6 851A      E8949A       E8949A       F3B09BB5
8956     8956 8956 8956 8956 54AB B1B7 B1B7 B1B7 B1B7 3137 F1F7 B1B7 9C3B      E9B0BB       E9B0BB       F3B09BB6
8957     8957 8957 8957 8957 5255 B1B8 B1B8 B1B8 B1B8 3138 F1F8 B1B8 59E5      E5A7A5       E5A7A5       F3B09BB7
8958     8958 8958 8958 8958 5367 B1B9 B1B9 B1B9 B1B9 3139 F1F9 B1B9 53A9      E58EA9       E58EA9       F3B09BB8
8959     8959 8959 8959 8959 466B B1BA B1BA B1BA B1BA 313A F17A B1BA 6D66      E6B5A6       E6B5A6       F3B09BB9
895A     895A 895A 895A 895A 4E64 B1BB B1BB B1BB B1BB 313B F15E B1BB 74DC      E7939C       E7939C       F3B09BBA
895B     895B 895B 895B 895B 545F B1BC B1BC B1BC B1BC 313C F14C B1BC 958F      E9968F       E9968F       F3B09BBB
895C     895C 895C 895C 895C 529B B1BD B1BD B1BD B1BD 313D F17E B1BD 5642      E59982       E59982       F3B09BBC
895D     895D 895D 895D 895D 499D B1BE B1BE B1BE B1BE 313E F16E B1BE 4E91      E4BA91       E4BA91       F3B09BBD
895E     895E 895E 895E 895E 46AB B1BF B1BF B1BF B1BF 313F F16F B1BF 904B      E9818B       E9818B       F3B09BBE
895F     895F 895F 895F 895F 4ABC B1C0 B1C0 B1C0 B1C0 3140 F17C B1C0 96F2      E99BB2       E99BB2       F3B09BBF
8960     8960 8960 8960 8960 515F B1C1 B1C1 B1C1 B1C1 3141 F1C1 B1C1 834F      E88D8F       E88D8F       F3B09C80
8961     8961 8961 8961 8961 50A7 B1C2 B1C2 B1C2 B1C2 3142 F1C2 B1C2 990C      E9A48C       E9A48C       F3B09C81
8962     8962 8962 8962 8962 5194 B1C3 B1C3 B1C3 B1C3 3143 F1C3 B1C3 53E1      E58FA1       E58FA1       F3B09C82
8963     8963 8963 8963 8963 48D3 B1C4 B1C4 B1C4 B1C4 3144 F1C4 B1C4 55B6      E596B6       E596B6       F3B09C83
8964     8964 8964 8964 8964 55E8 B1C5 B1C5 B1C5 B1C5 3145 F1C5 B1C5 5B30      E5ACB0       E5ACB0       F3B09C84
8965     8965 8965 8965 8965 495E B1C6 B1C6 B1C6 B1C6 3146 F1C6 B1C6 5F71      E5BDB1       E5BDB1       F3B09C85
8966     8966 8966 8966 8966 47AC B1C7 B1C7 B1C7 B1C7 3147 F1C7 B1C7 6620      E698A0       E698A0       F3B09C86
8967     8967 8967 8967 8967 4EBC B1C8 B1C8 B1C8 B1C8 3148 F1C8 B1C8 66F3      E69BB3       E69BB3       F3B09C87
8968     8968 8968 8968 8968 45FA B1C9 B1C9 B1C9 B1C9 3149 F1C9 B1C9 6804      E6A084       E6A084       F3B09C88
8969     8969 8969 8969 8969 45FB B1CA B1CA B1CA B1CA 314A F1D1 B1CA 6C38      E6B0B8       E6B0B8       F3B09C89
896A     896A 896A 896A 896A 4F57 B1CB B1CB B1CB B1CB 314B F1D2 B1CB 6CF3      E6B3B3       E6B3B3       F3B09C8A
896B     896B 896B 896B 896B 50B7 B1CC B1CC B1CC B1CC 314C F1D3 B1CC 6D29      E6B4A9       E6B4A9       F3B09C8B
896C     896C 896C 896C 896C 4F52 B1CD B1CD B1CD B1CD 314D F1D4 B1CD 745B      E7919B       E7919B       F3B09C8C
896D     896D 896D 896D 896D 51B5 B1CE B1CE B1CE B1CE 314E F1D5 B1CE 76C8      E79B88       E79B88       F3B09C8D
896E     896E 896E 896E 896E 5254 B1CF B1CF B1CF B1CF 314F F1D6 B1CF 7A4E      E7A98E       E7A98E       F3B09C8E
896F     896F 896F 896F 896F 5576 B1D0 B1D0 B1D0 B1D0 3150 F1D7 B1D0 9834      E9A0B4       E9A0B4       F3B09C8F
8970     8970 8970 8970 8970 45EE B1D1 B1D1 B1D1 B1D1 3151 F1D8 B1D1 82F1      E88BB1       E88BB1       F3B09C90
8971     8971 8971 8971 8971 4780 B1D2 B1D2 B1D2 B1D2 3152 F1D9 B1D2 885B      E8A19B       E8A19B       F3B09C91
8972     8972 8972 8972 8972 51C1 B1D3 B1D3 B1D3 B1D3 3153 F1E2 B1D3 8A60      E8A9A0       E8A9A0       F3B09C92
8973     8973 8973 8973 8973 4D6F B1D4 B1D4 B1D4 B1D4 3154 F1E3 B1D4 92ED      E98BAD       E98BAD       F3B09C93
8974     8974 8974 8974 8974 4A45 B1D5 B1D5 B1D5 B1D5 3155 F1E4 B1D5 6DB2      E6B6B2       E6B6B2       F3B09C94
8975     8975 8975 8975 8975 50CE B1D6 B1D6 B1D6 B1D6 3156 F1E5 B1D6 75AB      E796AB       E796AB       F3B09C95
8976     8976 8976 8976 8976 4774 B1D7 B1D7 B1D7 B1D7 3157 F1E6 B1D7 76CA      E79B8A       E79B8A       F3B09C96
8977     8977 8977 8977 8977 49AC B1D8 B1D8 B1D8 B1D8 3158 F1E7 B1D8 99C5      E9A785       E9A785       F3B09C97
8978     8978 8978 8978 8978 4873 B1D9 B1D9 B1D9 B1D9 3159 F1E8 B1D9 60A6      E682A6       E682A6       F3B09C98
8979     8979 8979 8979 8979 54E5 B1DA B1DA B1DA B1DA 315A F1E9 B1DA 8B01      E8AC81       E8AC81       F3B09C99
897A     897A 897A 897A 897A 46DF B1DB B1DB B1DB B1DB 315B F14A B1DB 8D8A      E8B68A       E8B68A       F3B09C9A
897B     897B 897B 897B 897B 5177 B1DC B1DC B1DC B1DC 315C F15B B1DC 95B2      E996B2       E996B2       F3B09C9B
897C     897C 897C 897C 897C 49EB B1DD B1DD B1DD B1DD 315D F15A B1DD 698E      E6A68E       E6A68E       F3B09C9C
897D     897D 897D 897D 897D 53B7 B1DE B1DE B1DE B1DE 315E F15F B1DE 53AD      E58EAD       E58EAD       F3B09C9D
897E     897E 897E 897E 897E 4645 B1DF B1DF B1DF B1DF 315F F16D B1DF 5186      E58686       E58686       F3B09C9E
8980     8980 8980 8980 8980 47E1 B1E0 B1E0 B1E0 B1E0 3160 F179 B1E0 5712      E59C92       E59C92       F3B09C9F
8981     8981 8981 8981 8981 50A4 B1E1 B1E1 B1E1 B1E1 3161 F157 B1E1 5830      E5A0B0       E5A0B0       F3B09CA0
8982     8982 8982 8982 8982 52F2 B1E2 B1E2 B1E2 B1E2 3162 F159 B1E2 5944      E5A584       E5A584       F3B09CA1
8983     8983 8983 8983 8983 5078 B1E3 B1E3 B1E3 B1E3 3163 F162 B1E3 5BB4      E5AEB4       E5AEB4       F3B09CA2
8984     8984 8984 8984 8984 47D0 B1E4 B1E4 B1E4 B1E4 3164 F163 B1E4 5EF6      E5BBB6       E5BBB6       F3B09CA3
8985     8985 8985 8985 8985 54DE B1E5 B1E5 B1E5 B1E5 3165 F164 B1E5 6028      E680A8       E680A8       F3B09CA4
8986     8986 8986 8986 8986 55D0 B1E6 B1E6 B1E6 B1E6 3166 F165 B1E6 63A9      E68EA9       E68EA9       F3B09CA5
8987     8987 8987 8987 8987 4AD5 B1E7 B1E7 B1E7 B1E7 3167 F166 B1E7 63F4      E68FB4       E68FB4       F3B09CA6
8988     8988 8988 8988 8988 4E6F B1E8 B1E8 B1E8 B1E8 3168 F167 B1E8 6CBF      E6B2BF       E6B2BF       F3B09CA7
8989     8989 8989 8989 8989 49ED B1E9 B1E9 B1E9 B1E9 3169 F168 B1E9 6F14      E6BC94       E6BC94       F3B09CA8
898A     898A 898A 898A 898A 4DAD B1EA B1EA B1EA B1EA 316A F169 B1EA 708E      E7828E       E7828E       F3B09CA9
898B     898B 898B 898B 898B 53EE B1EB B1EB B1EB 70E0 D6A8 AC48 B1EB 7114      E78494       E78494       F3B09CAA
898C     898C 898C 898C 898C 4BDD B1EC B1EC B1EC B1EC 316C F171 B1EC 7159      E78599       E78599       F3B09CAB
898D     898D 898D 898D 898D 5047 B1ED B1ED B1ED B1ED 316D F172 B1ED 71D5      E78795       E78795       F3B09CAC
898E     898E 898E 898E 898E 4C5A B1EE B1EE B1EE B1EE 316E F173 B1EE 733F      E78CBF       E78CBF       F3B09CAD
898F     898F 898F 898F 898F 4BAA B1EF B1EF B1EF B1EF 316F F174 B1EF 7E01      E7B881       E7B881       F3B09CAE
8990     8990 8990 8990 8990 4D54 B1F0 B1F0 B1F0 B1F0 3170 F175 B1F0 8276      E889B6       E889B6       F3B09CAF
8991     8991 8991 8991 8991 504B B1F1 B1F1 B1F1 B1F1 3171 F176 B1F1 82D1      E88B91       E88B91       F3B09CB0
8992     8992 8992 8992 8992 4CEC B1F2 B1F2 B1F2 B1F2 3172 F177 B1F2 8597      E89697       E89697       F3B09CB1
8993     8993 8993 8993 8993 46E3 B1F3 B1F3 B1F3 B1F3 3173 F178 B1F3 9060      E981A0       E981A0       F3B09CB2
8994     8994 8994 8994 8994 49D4 B1F4 B1F4 B1F4 B1F4 3174 F180 B1F4 925B      E9899B       E9899B       F3B09CB3
8995     8995 8995 8995 8995 52C3 B1F5 B1F5 B1F5 B1F5 3175 F18B B1F5 9D1B      E9B49B       E9B49B       F3B09CB4
8996     8996 8996 8996 8996 4743 B1F6 B1F6 B1F6 B1F6 3176 F19B B1F6 5869      E5A1A9       E5A1A9       F3B09CB5
8997     8997 8997 8997 8997 4DA4 B1F7 B1F7 B1F7 B1F7 3177 F19C B1F7 65BC      E696BC       E696BC       F3B09CB6
8998     8998 8998 8998 8998 4E82 B1F8 B1F8 B1F8 B1F8 3178 F1A0 B1F8 6C5A      E6B19A       E6B19A       F3B09CB7
8999     8999 8999 8999 8999 55D7 B1F9 B1F9 B1F9 B1F9 3179 F1AB B1F9 7525      E794A5       E794A5       F3B09CB8
899A     899A 899A 899A 899A 50E6 B1FA B1FA B1FA B1FA 317A F1B0 B1FA 51F9      E587B9       E587B9       F3B09CB9
899B     899B 899B 899B 899B 4792 B1FB B1FB B1FB B1FB 317B F1C0 B1FB 592E      E5A4AE       E5A4AE       F3B09CBA
899C     899C 899C 899C 899C 46E5 B1FC B1FC B1FC B1FC 317C F16A B1FC 5965      E5A5A5       E5A5A5       F3B09CBB
899D     899D 899D 899D 899D 4CEA B1FD B1FD B1FD B1FD 317D F1D0 B1FD 5F80      E5BE80       E5BE80       F3B09CBC
899E     899E 899E 899E 899E 4895 B1FE B1FE B1FE B1FE 317E F1A1 B1FE 5FDC      E5BF9C       E5BF9C       F3B09CBD
899F     899F 899F 899F 899F 49E1 B2A1 B2A1 B2A1 B2A1 3221 F24F B2A1 62BC      E68ABC       E68ABC       F3B09CBE
89A0     89A0 89A0 89A0 89A0 4FA3 B2A2 B2A2 B2A2 B2A2 3222 F27F B2A2 65FA      E697BA       E697BA       F3B09CBF
89A1     89A1 89A1 89A1 89A1 45F2 B2A3 B2A3 B2A3 B2A3 3223 F27B B2A3 6A2A      E6A8AA       E6A8AA       F3B09D80
89A2     89A2 89A2 89A2 89A2 4AA0 B2A4 B2A4 B2A4 B2A4 3224 F2E0 B2A4 6B27      E6ACA7       E6ACA7       F3B09D81
89A3     89A3 89A3 89A3 89A3 5459 B2A5 B2A5 B2A5 B2A5 3225 F26C B2A5 6BB4      E6AEB4       E6AEB4       F3B09D82
89A4     89A4 89A4 89A4 89A4 49B1 B2A6 B2A6 B2A6 B2A6 3226 F250 B2A6 738B      E78E8B       E78E8B       F3B09D83
89A5     89A5 89A5 89A5 89A5 4FFB B2A7 B2A7 B2A7 B2A7 3227 F27D B2A7 7FC1      E7BF81       E7BF81       F3B09D84
89A6     89A6 89A6 89A6 89A6 546C B2A8 B2A8 B2A8 B2A8 3228 F24D B2A8 8956      E8A596       E8A596       F3B09D85
89A7     89A7 89A7 89A7 89A7 67D0 F2F4 B2A9 B2A9 F2F4 7274 7780 B2A9 9D2C      E9B4AC       E9B4AC       F3B09D86
89A8     89A8 89A8 89A8 89A8 5464 B2AA B2AA B2AA 6FB0 D5C3 AA94 B2AA 9D0E      E9B48E       E9B48E       F3B09D87
89A9     89A9 89A9 89A9 89A9 4AC6 B2AB B2AB B2AB B2AB 322B F24E B2AB 9EC4      E9BB84       E9BB84       F3B09D88
89AA     89AA 89AA 89AA 89AA 4572 B2AC B2AC B2AC B2AC 322C F26B B2AC 5CA1      E5B2A1       E5B2A1       F3B09D89
89AB     89AB 89AB 89AB 89AB 47FD B2AD B2AD B2AD B2AD 322D F260 B2AD 6C96      E6B296       E6B296       F3B09D8A
89AC     89AC 89AC 89AC 89AC 49E0 B2AE B2AE B2AE B2AE 322E F24B B2AE 837B      E88DBB       E88DBB       F3B09D8B
89AD     89AD 89AD 89AD 89AD 454E B2AF B2AF B2AF B2AF 322F F261 B2AF 5104      E58484       E58484       F3B09D8C
89AE     89AE 89AE 89AE 89AE 45E9 B2B0 B2B0 B2B0 B2B0 3230 F2F0 B2B0 5C4B      E5B18B       E5B18B       F3B09D8D
89AF     89AF 89AF 89AF 89AF 4E6C B2B1 B2B1 B2B1 B2B1 3231 F2F1 B2B1 61B6      E686B6       E686B6       F3B09D8E
89B0     89B0 89B0 89B0 89B0 5474 B2B2 B2B2 B2B2 B2B2 3232 F2F2 B2B2 81C6      E88786       E88786       F3B09D8F
89B1     89B1 89B1 89B1 89B1 4E77 B2B3 B2B3 B2B3 B2B3 3233 F2F3 B2B3 6876      E6A1B6       E6A1B6       F3B09D90
89B2     89B2 89B2 89B2 89B2 509A B2B4 B2B4 B2B4 B2B4 3234 F2F4 B2B4 7261      E789A1       E789A1       F3B09D91
89B3     89B3 89B3 89B3 89B3 4BBD B2B5 B2B5 B2B5 B2B5 3235 F2F5 B2B5 4E59      E4B999       E4B999       F3B09D92
89B4     89B4 89B4 89B4 89B4 4E67 B2B6 B2B6 B2B6 B2B6 3236 F2F6 B2B6 4FFA      E4BFBA       E4BFBA       F3B09D93
89B5     89B5 89B5 89B5 89B5 4E79 B2B7 B2B7 B2B7 B2B7 3237 F2F7 B2B7 5378      E58DB8       E58DB8       F3B09D94
89B6     89B6 89B6 89B6 89B6 4C81 B2B8 B2B8 B2B8 B2B8 3238 F2F8 B2B8 6069      E681A9       E681A9       F3B09D95
89B7     89B7 89B7 89B7 89B7 4950 B2B9 B2B9 B2B9 B2B9 3239 F2F9 B2B9 6E29      E6B8A9       E6B8A9       F3B09D96
89B8     89B8 89B8 89B8 89B8 50B8 B2BA B2BA B2BA B2BA 323A F27A B2BA 7A4F      E7A98F       E7A98F       F3B09D97
89B9     89B9 89B9 89B9 89B9 484D B2BB B2BB B2BB B2BB 323B F25E B2BB 97F3      E99FB3       E99FB3       F3B09D98
89BA     89BA 89BA 89BA 89BA 455F B2BC B2BC B2BC B2BC 323C F24C B2BC 4E0B      E4B88B       E4B88B       F3B09D99
89BB     89BB 89BB 89BB 89BB 4590 B2BD B2BD B2BD B2BD 323D F27E B2BD 5316      E58C96       E58C96       F3B09D9A
89BC     89BC 89BC 89BC 89BC 4D71 B2BE B2BE B2BE B2BE 323E F26E B2BE 4EEE      E4BBAE       E4BBAE       F3B09D9B
89BD     89BD 89BD 89BD 89BD 4770 B2BF B2BF B2BF B2BF 323F F26F B2BF 4F55      E4BD95       E4BD95       F3B09D9C
89BE     89BE 89BE 89BE 89BE 5461 B2C0 B2C0 B2C0 B2C0 3240 F27C B2C0 4F3D      E4BCBD       E4BCBD       F3B09D9D
89BF     89BF 89BF 89BF 89BF 475A B2C1 B2C1 B2C1 B2C1 3241 F2C1 B2C1 4FA1      E4BEA1       E4BEA1       F3B09D9E
89C0     89C0 89C0 89C0 89C0 4876 B2C2 B2C2 B2C2 B2C2 3242 F2C2 B2C2 4F73      E4BDB3       E4BDB3       F3B09D9F
89C1     89C1 89C1 89C1 89C1 459F B2C3 B2C3 B2C3 B2C3 3243 F2C3 B2C3 52A0      E58AA0       E58AA0       F3B09DA0
89C2     89C2 89C2 89C2 89C2 47C6 B2C4 B2C4 B2C4 B2C4 3244 F2C4 B2C4 53EF      E58FAF       E58FAF       F3B09DA1
89C3     89C3 89C3 89C3 89C3 477E B2C5 B2C5 B2C5 B2C5 3245 F2C5 B2C5 5609      E59889       E59889       F3B09DA2
89C4     89C4 89C4 89C4 89C4 497E B2C6 B2C6 B2C6 B2C6 3246 F2C6 B2C6 590F      E5A48F       E5A48F       F3B09DA3
89C5     89C5 89C5 89C5 89C5 4E89 B2C7 B2C7 B2C7 B2C7 3247 F2C7 B2C7 5AC1      E5AB81       E5AB81       F3B09DA4
89C6     89C6 89C6 89C6 89C6 4686 B2C8 B2C8 B2C8 B2C8 3248 F2C8 B2C8 5BB6      E5AEB6       E5AEB6       F3B09DA5
89C7     89C7 89C7 89C7 89C7 52B8 B2C9 B2C9 B2C9 B2C9 3249 F2C9 B2C9 5BE1      E5AFA1       E5AFA1       F3B09DA6
89C8     89C8 89C8 89C8 89C8 48ED B2CA B2CA B2CA B2CA 324A F2D1 B2CA 79D1      E7A791       E7A791       F3B09DA7
89C9     89C9 89C9 89C9 89C9 4F46 B2CB B2CB B2CB B2CB 324B F2D2 B2CB 6687      E69A87       E69A87       F3B09DA8
89CA     89CA 89CA 89CA 89CA 47B9 B2CC B2CC B2CC B2CC 324C F2D3 B2CC 679C      E69E9C       E69E9C       F3B09DA9
89CB     89CB 89CB 89CB 89CB 4EE8 B2CD B2CD B2CD B2CD 324D F2D4 B2CD 67B6      E69EB6       E69EB6       F3B09DAA
89CC     89CC 89CC 89CC 89CC 46F7 B2CE B2CE B2CE B2CE 324E F2D5 B2CE 6B4C      E6AD8C       E6AD8C       F3B09DAB
89CD     89CD 89CD 89CD 89CD 465A B2CF B2CF B2CF B2CF 324F F2D6 B2CF 6CB3      E6B2B3       E6B2B3       F3B09DAC
89CE     89CE 89CE 89CE 89CE 4785 B2D0 B2D0 B2D0 B2D0 3250 F2D7 B2D0 706B      E781AB       E781AB       F3B09DAD
89CF     89CF 89CF 89CF 89CF 4C9A B2D1 B2D1 B2D1 B2D1 3251 F2D8 B2D1 73C2      E78F82       E78F82       F3B09DAE
89D0     89D0 89D0 89D0 89D0 5286 B2D2 B2D2 B2D2 B2D2 3252 F2D9 B2D2 798D      E7A68D       E7A68D       F3B09DAF
89D1     89D1 89D1 89D1 89D1 5463 B2D3 B2D3 B2D3 B2D3 3253 F2E2 B2D3 79BE      E7A6BE       E7A6BE       F3B09DB0
89D2     89D2 89D2 89D2 89D2 4DBE B2D4 B2D4 B2D4 B2D4 3254 F2E3 B2D4 7A3C      E7A8BC       E7A8BC       F3B09DB1
89D3     89D3 89D3 89D3 89D3 4F51 B2D5 B2D5 B2D5 B2D5 3255 F2E4 B2D5 7B87      E7AE87       E7AE87       F3B09DB2
89D4     89D4 89D4 89D4 89D4 46D9 B2D6 B2D6 B2D6 B2D6 3256 F2E5 B2D6 82B1      E88AB1       E88AB1       F3B09DB3
89D5     89D5 89D5 89D5 89D5 5377 B2D7 B2D7 B2D7 B2D7 3257 F2E6 B2D7 82DB      E88B9B       E88B9B       F3B09DB4
89D6     89D6 89D6 89D6 89D6 547A B2D8 B2D8 B2D8 B2D8 3258 F2E7 B2D8 8304      E88C84       E88C84       F3B09DB5
89D7     89D7 89D7 89D7 89D7 4BA1 B2D9 B2D9 B2D9 B2D9 3259 F2E8 B2D9 8377      E88DB7       E88DB7       F3B09DB6
89D8     89D8 89D8 89D8 89D8 4B6C B2DA B2DA B2DA B2DA 325A F2E9 B2DA 83EF      E88FAF       E88FAF       F3B09DB7
89D9     89D9 89D9 89D9 89D9 4C51 B2DB B2DB B2DB B2DB 325B F24A B2DB 83D3      E88F93       E88F93       F3B09DB8
89DA     89DA 89DA 89DA 89DA 51C3 B2DC B2DC B2DC B2DC 325C F25B B2DC 8766      E89DA6       E89DA6       F3B09DB9
89DB     89DB 89DB 89DB 89DB 49D6 B2DD B2DD B2DD B2DD 325D F25A B2DD 8AB2      E8AAB2       E8AAB2       F3B09DBA
89DC     89DC 89DC 89DC 89DC 52CA B2DE B2DE B2DE B2DE 325E F25F B2DE 5629      E598A9       E598A9       F3B09DBB
89DD     89DD 89DD 89DD 89DD 4AEB B2DF B2DF B2DF B2DF 325F F26D B2DF 8CA8      E8B2A8       E8B2A8       F3B09DBC
89DE     89DE 89DE 89DE 89DE 535D B2E0 B2E0 B2E0 B2E0 3260 F279 B2E0 8FE6      E8BFA6       E8BFA6       F3B09DBD
89DF     89DF 89DF 89DF 89DF 488D B2E1 B2E1 B2E1 B2E1 3261 F257 B2E1 904E      E9818E       E9818E       F3B09DBE
89E0     89E0 89E0 89E0 89E0 4EE4 B2E2 B2E2 B2E2 B2E2 3262 F259 B2E2 971E      E99C9E       E99C9E       F3B09DBF
89E1     89E1 89E1 89E1 89E1 4FAA B2E3 B2E3 B2E3 B2E3 3263 F262 B2E3 868A      E89A8A       E89A8A       F3B09E80
89E2     89E2 89E2 89E2 89E2 54C7 B2E4 B2E4 B2E4 B2E4 3264 F263 B2E4 4FC4      E4BF84       E4BF84       F3B09E81
89E3     89E3 89E3 89E3 89E3 4F9D B2E5 B2E5 B2E5 B2E5 3265 F264 B2E5 5CE8      E5B3A8       E5B3A8       F3B09E82
89E4     89E4 89E4 89E4 89E4 49DE B2E6 B2E6 B2E6 B2E6 3266 F265 B2E6 6211      E68891       E68891       F3B09E83
89E5     89E5 89E5 89E5 89E5 53FC B2E7 B2E7 B2E7 B2E7 3267 F266 B2E7 7259      E78999       E78999       F3B09E84
89E6     89E6 89E6 89E6 89E6 4677 B2E8 B2E8 B2E8 B2E8 3268 F267 B2E8 753B      E794BB       E794BB       F3B09E85
89E7     89E7 89E7 89E7 89E7 538A B2E9 B2E9 B2E9 B2E9 3269 F268 B2E9 81E5      E887A5       E887A5       F3B09E86
89E8     89E8 89E8 89E8 89E8 4EBA B2EA B2EA B2EA B2EA 326A F269 B2EA 82BD      E88ABD       E88ABD       F3B09E87
89E9     89E9 89E9 89E9 89E9 5599 B2EB B2EB B2EB B2EB 326B F270 B2EB 86FE      E89BBE       E89BBE       F3B09E88
89EA     89EA 89EA 89EA 89EA 4641 B2EC B2EC B2EC B2EC 326C F271 B2EC 8CC0      E8B380       E8B380       F3B09E89
89EB     89EB 89EB 89EB 89EB 4745 B2ED B2ED B2ED B2ED 326D F272 B2ED 96C5      E99B85       E99B85       F3B09E8A
89EC     89EC 89EC 89EC 89EC 554B B2EE B2EE B2EE B2EE 326E F273 B2EE 9913      E9A493       E9A493       F3B09E8B
89ED     89ED 89ED 89ED 89ED 53BE B2EF B2EF B2EF B2EF 326F F274 B2EF 99D5      E9A795       E9A795       F3B09E8C
89EE     89EE 89EE 89EE 89EE 484B B2F0 B2F0 B2F0 B2F0 3270 F275 B2F0 4ECB      E4BB8B       E4BB8B       F3B09E8D
89EF     89EF 89EF 89EF 89EF 45BD B2F1 B2F1 B2F1 B2F1 3271 F276 B2F1 4F1A      E4BC9A       E4BC9A       F3B09E8E
89F0     89F0 89F0 89F0 89F0 47B7 B2F2 B2F2 B2F2 B2F2 3272 F277 B2F2 89E3      E8A7A3       E8A7A3       F3B09E8F
89F1     89F1 89F1 89F1 89F1 46A9 B2F3 B2F3 B2F3 B2F3 3273 F278 B2F3 56DE      E59B9E       E59B9E       F3B09E90
89F2     89F2 89F2 89F2 89F2 5053 B2F4 B2F4 B2F4 B2F4 3274 F280 B2F4 584A      E5A18A       E5A18A       F3B09E91
89F3     89F3 89F3 89F3 89F3 4EC8 B2F5 B2F5 B2F5 B2F5 3275 F28B B2F5 58CA      E5A38A       E5A38A       F3B09E92
89F4     89F4 89F4 89F4 89F4 4BAF B2F6 B2F6 B2F6 B2F6 3276 F29B B2F6 5EFB      E5BBBB       E5BBBB       F3B09E93
89F5     89F5 89F5 89F5 89F5 4C60 B2F7 B2F7 B2F7 B2F7 3277 F29C B2F7 5FEB      E5BFAB       E5BFAB       F3B09E94
89F6     89F6 89F6 89F6 89F6 4E71 B2F8 B2F8 B2F8 B2F8 3278 F2A0 B2F8 602A      E680AA       E680AA       F3B09E95
89F7     89F7 89F7 89F7 89F7 5091 B2F9 B2F9 B2F9 B2F9 3279 F2AB B2F9 6094      E68294       E68294       F3B09E96
89F8     89F8 89F8 89F8 89F8 5594 B2FA B2FA B2FA B2FA 327A F2B0 B2FA 6062      E681A2       E681A2       F3B09E97
89F9     89F9 89F9 89F9 89F9 4EF2 B2FB B2FB B2FB B2FB 327B F2C0 B2FB 61D0      E68790       E68790       F3B09E98
89FA     89FA 89FA 89FA 89FA 4DA0 B2FC B2FC B2FC B2FC 327C F26A B2FC 6212      E68892       E68892       F3B09E99
89FB     89FB 89FB 89FB 89FB 53A2 B2FD B2FD B2FD B2FD 327D F2D0 B2FD 62D0      E68B90       E68B90       F3B09E9A
89FC     89FC 89FC 89FC 89FC 488F B2FE B2FE B2FE B2FE 327E F2A1 B2FE 6539      E694B9       E694B9       F3B09E9B
8A40     8A40 8A40 8A40 8A40 52CB B3A1 B3A1 B3A1 B3A1 3321 F34F B3A1 9B41      E9AD81       E9AD81       F3B09E9C
8A41     8A41 8A41 8A41 8A41 556D B3A2 B3A2 B3A2 B3A2 3322 F37F B3A2 6666      E699A6       E699A6       F3B09E9D
8A42     8A42 8A42 8A42 8A42 4682 B3A3 B3A3 B3A3 B3A3 3323 F37B B3A3 68B0      E6A2B0       E6A2B0       F3B09E9E
8A43     8A43 8A43 8A43 8A43 45A7 B3A4 B3A4 B3A4 B3A4 3324 F3E0 B3A4 6D77      E6B5B7       E6B5B7       F3B09E9F
8A44     8A44 8A44 8A44 8A44 4AF5 B3A5 B3A5 B3A5 B3A5 3325 F36C B3A5 7070      E781B0       E781B0       F3B09EA0
8A45     8A45 8A45 8A45 8A45 47AE B3A6 B3A6 B3A6 B3A6 3326 F350 B3A6 754C      E7958C       E7958C       F3B09EA1
8A46     8A46 8A46 8A46 8A46 4AB0 B3A7 B3A7 B3A7 B3A7 3327 F37D B3A7 7686      E79A86       E79A86       F3B09EA2
8A47     8A47 8A47 8A47 8A47 4AE4 B3A8 B3A8 B3A8 B3A8 3328 F34D B3A8 7D75      E7B5B5       E7B5B5       F3B09EA3
8A48     8A48 8A48 8A48 8A48 4F7B B3A9 B3A9 B3A9 B3A9 3329 F35D B3A9 82A5      E88AA5       E88AA5       F3B09EA4
8A49     8A49 8A49 8A49 8A49 5090 B3AA B3AA B3AA B3AA 332A F35C B3AA 87F9      E89FB9       E89FB9       F3B09EA5
8A4A     8A4A 8A4A 8A4A 8A4A 46CD B3AB B3AB B3AB B3AB 332B F34E B3AB 958B      E9968B       E9968B       F3B09EA6
8A4B     8A4B 8A4B 8A4B 8A4B 49A9 B3AC B3AC B3AC B3AC 332C F36B B3AC 968E      E99A8E       E99A8E       F3B09EA7
8A4C     8A4C 8A4C 8A4C 8A4C 49EA B3AD B3AD B3AD B3AD 332D F360 B3AD 8C9D      E8B29D       E8B29D       F3B09EA8
8A4D     8A4D 8A4D 8A4D 8A4D 51AC B3AE B3AE B3AE B3AE 332E F34B B3AE 51F1      E587B1       E587B1       F3B09EA9
8A4E     8A4E 8A4E 8A4E 8A4E 556A B3AF B3AF B3AF B3AF 332F F361 B3AF 52BE      E58ABE       E58ABE       F3B09EAA
8A4F     8A4F 8A4F 8A4F 8A4F 4694 B3B0 B3B0 B3B0 B3B0 3330 F3F0 B3B0 5916      E5A496       E5A496       F3B09EAB
8A50     8A50 8A50 8A50 8A50 53E9 B3B1 B3B1 B3B1 B3B1 3331 F3F1 B3B1 54B3      E592B3       E592B3       F3B09EAC
8A51     8A51 8A51 8A51 8A51 4968 B3B2 B3B2 B3B2 B3B2 3332 F3F2 B3B2 5BB3      E5AEB3       E5AEB3       F3B09EAD
8A52     8A52 8A52 8A52 8A52 5074 B3B3 B3B3 B3B3 B3B3 3333 F3F3 B3B3 5D16      E5B496       E5B496       F3B09EAE
8A53     8A53 8A53 8A53 8A53 528A B3B4 B3B4 B3B4 B3B4 3334 F3F4 B3B4 6168      E685A8       E685A8       F3B09EAF
8A54     8A54 8A54 8A54 8A54 4C42 B3B5 B3B5 B3B5 B3B5 3335 F3F5 B3B5 6982      E6A682       E6A682       F3B09EB0
8A55     8A55 8A55 8A55 8A55 4F91 B3B6 B3B6 B3B6 B3B6 3336 F3F6 B3B6 6DAF      E6B6AF       E6B6AF       F3B09EB1
8A56     8A56 8A56 8A56 8A56 5356 B3B7 B3B7 B3B7 B3B7 3337 F3F7 B3B7 788D      E7A28D       E7A28D       F3B09EB2
8A57     8A57 8A57 8A57 8A57 4F6D B3B8 B3B8 B3B8 B3B8 3338 F3F8 B3B8 84CB      E8938B       E8938B       F3B09EB3
8A58     8A58 8A58 8A58 8A58 4BA5 B3B9 B3B9 B3B9 B3B9 3339 F3F9 B3B9 8857      E8A197       E8A197       F3B09EB4
8A59     8A59 8A59 8A59 8A59 4D8D B3BA B3BA B3BA B3BA 333A F37A B3BA 8A72      E8A9B2       E8A9B2       F3B09EB5
8A5A     8A5A 8A5A 8A5A 8A5A 546D B3BB B3BB B3BB B3BB 333B F35E B3BB 93A7      E98EA7       E98EA7       F3B09EB6
8A5B     8A5B 8A5B 8A5B 8A5B 559C B3BC B3BC B3BC B3BC 333C F34C B3BC 9AB8      E9AAB8       E9AAB8       F3B09EB7
8A5C     8A5C 8A5C 8A5C 8A5C 55CC B3BD B3BD B3BD B3BD 333D F37E B3BD 6D6C      E6B5AC       E6B5AC       F3B09EB8
8A5D     8A5D 8A5D 8A5D 8A5D 4B85 B3BE B3BE B3BE B3BE 333E F36E B3BE 99A8      E9A6A8       E9A6A8       F3B09EB9
8A5E     8A5E 8A5E 8A5E 8A5E 55BB B3BF B3BF B3BF B3BF 333F F36F B3BF 86D9      E89B99       E89B99       F3B09EBA
8A5F     8A5F 8A5F 8A5F 8A5F 47CE B3C0 B3C0 B3C0 B3C0 3340 F37C B3C0 57A3      E59EA3       E59EA3       F3B09EBB
8A60     8A60 8A60 8A60 8A60 4AC1 B3C1 B3C1 B3C1 B3C1 3341 F3C1 B3C1 67FF      E69FBF       E69FBF       F3B09EBC
8A61     8A61 8A61 8A61 8A61 62E1 E9DA B3C2 B3C2 E9DA 695A 68E9 B3C2 86CE      E89B8E       E89B8E       F3B09EBD
8A62     8A62 8A62 8A62 8A62 538F B3C3 B3C3 B3C3 B3C3 3343 F3C3 B3C3 920E      E9888E       E9888E       F3B09EBE
8A63     8A63 8A63 8A63 8A63 55D6 B3C4 B3C4 B3C4 B3C4 3344 F3C4 B3C4 5283      E58A83       E58A83       F3B09EBF
8A64     8A64 8A64 8A64 8A64 55D8 B3C5 B3C5 B3C5 B3C5 3345 F3C5 B3C5 5687      E59A87       E59A87       F3B09F80
8A65     8A65 8A65 8A65 8A65 47BA B3C6 B3C6 B3C6 B3C6 3346 F3C6 B3C6 5404      E59084       E59084       F3B09F81
8A66     8A66 8A66 8A66 8A66 54AD B3C7 B3C7 B3C7 B3C7 3347 F3C7 B3C7 5ED3      E5BB93       E5BB93       F3B09F82
8A67     8A67 8A67 8A67 8A67 48E5 B3C8 B3C8 B3C8 B3C8 3348 F3C8 B3C8 62E1      E68BA1       E68BA1       F3B09F83
8A68     8A68 8A68 8A68 8A68 5B52 D9F8 B3C9 B3C9 D9F8 5978 E8A0 B3C9 64B9      E692B9       E692B9       F3B09F84
8A69     8A69 8A69 8A69 8A69 47DE B3CA B3CA B3CA B3CA 334A F3D1 B3CA 683C      E6A0BC       E6A0BC       F3B09F85
8A6A     8A6A 8A6A 8A6A 8A6A 4B9E B3CB B3CB B3CB B3CB 334B F3D2 B3CB 6838      E6A0B8       E6A0B8       F3B09F86
8A6B     8A6B 8A6B 8A6B 8A6B 50C0 B3CC B3CC B3CC B3CC 334C F3D3 B3CC 6BBB      E6AEBB       E6AEBB       F3B09F87
8A6C     8A6C 8A6C 8A6C 8A6C 4D61 B3CD B3CD B3CD B3CD 334D F3D4 B3CD 7372      E78DB2       E78DB2       F3B09F88
8A6D     8A6D 8A6D 8A6D 8A6D 487A B3CE B3CE B3CE B3CE 334E F3D5 B3CE 78BA      E7A2BA       E7A2BA       F3B09F89
8A6E     8A6E 8A6E 8A6E 8A6E 4F5C B3CF B3CF B3CF B3CF 334F F3D6 B3CF 7A6B      E7A9AB       E7A9AB       F3B09F8A
8A6F     8A6F 8A6F 8A6F 8A6F 4A4C B3D0 B3D0 B3D0 B3D0 3350 F3D7 B3D0 899A      E8A69A       E8A69A       F3B09F8B
8A70     8A70 8A70 8A70 8A70 477D B3D1 B3D1 B3D1 B3D1 3351 F3D8 B3D1 89D2      E8A792       E8A792       F3B09F8C
8A71     8A71 8A71 8A71 8A71 53CC B3D2 B3D2 B3D2 B3D2 3352 F3D9 B3D2 8D6B      E8B5AB       E8B5AB       F3B09F8D
8A72     8A72 8A72 8A72 8A72 4A5A B3D3 B3D3 B3D3 B3D3 3353 F3E2 B3D3 8F03      E8BC83       E8BC83       F3B09F8E
8A73     8A73 8A73 8A73 8A73 5162 B3D4 B3D4 B3D4 B3D4 3354 F3E3 B3D4 90ED      E983AD       E983AD       F3B09F8F
8A74     8A74 8A74 8A74 8A74 4AE9 B3D5 B3D5 B3D5 B3D5 3355 F3E4 B3D5 95A3      E996A3       E996A3       F3B09F90
8A75     8A75 8A75 8A75 8A75 505A B3D6 B3D6 B3D6 B3D6 3356 F3E5 B3D6 9694      E99A94       E99A94       F3B09F91
8A76     8A76 8A76 8A76 8A76 4AA3 B3D7 B3D7 B3D7 B3D7 3357 F3E6 B3D7 9769      E99DA9       E99DA9       F3B09F92
8A77     8A77 8A77 8A77 8A77 45BE B3D8 B3D8 B3D8 B3D8 3358 F3E7 B3D8 5B66      E5ADA6       E5ADA6       F3B09F93
8A78     8A78 8A78 8A78 8A78 4D46 B3D9 B3D9 B3D9 B3D9 3359 F3E8 B3D9 5CB3      E5B2B3       E5B2B3       F3B09F94
8A79     8A79 8A79 8A79 8A79 4761 B3DA B3DA B3DA B3DA 335A F3E9 B3DA 697D      E6A5BD       E6A5BD       F3B09F95
8A7A     8A7A 8A7A 8A7A 8A7A 47F1 B3DB B3DB B3DB B3DB 335B F34A B3DB 984D      E9A18D       E9A18D       F3B09F96
8A7B     8A7B 8A7B 8A7B 8A7B 548E B3DC B3DC B3DC B3DC 335C F35B B3DC 984E      E9A18E       E9A18E       F3B09F97
8A7C     8A7C 8A7C 8A7C 8A7C 4AE3 B3DD B3DD B3DD B3DD 335D F35A B3DD 639B      E68E9B       E68E9B       F3B09F98
8A7D     8A7D 8A7D 8A7D 8A7D 47E0 B3DE B3DE B3DE B3DE 335E F35F B3DE 7B20      E7ACA0       E7ACA0       F3B09F99
8A7E     8A7E 8A7E 8A7E 8A7E 4B6F B3DF B3DF B3DF B3DF 335F F36D B3DF 6A2B      E6A8AB       E6A8AB       F3B09F9A
8A80     8A80 8A80 8A80 8A80 4FB2 B3E0 B3E0 B3E0 B3E0 3360 F379 B3E0 6A7F      E6A9BF       E6A9BF       F3B09F9B
8A81     8A81 8A81 8A81 8A81 49AD B3E1 B3E1 B3E1 B3E1 3361 F357 B3E1 68B6      E6A2B6       E6A2B6       F3B09F9C
8A82     8A82 8A82 8A82 8A82 5571 B3E2 B3E2 B3E2 B3E2 3362 F359 B3E2 9C0D      E9B08D       E9B08D       F3B09F9D
8A83     8A83 8A83 8A83 8A83 46C2 B3E3 B3E3 B3E3 B3E3 3363 F362 B3E3 6F5F      E6BD9F       E6BD9F       F3B09F9E
8A84     8A84 8A84 8A84 8A84 47D6 B3E4 B3E4 B3E4 B3E4 3364 F363 B3E4 5272      E589B2       E589B2       F3B09F9F
8A85     8A85 8A85 8A85 8A85 5475 B3E5 B3E5 B3E5 47B8 B5ED 85CF B3E5 559D      E5969D       E5969D       F3B09FA0
8A86     8A86 8A86 8A86 8A86 5384 B3E6 B3E6 B3E6 B3E6 3366 F365 B3E6 6070      E681B0       E681B0       F3B09FA1
8A87     8A87 8A87 8A87 8A87 4FC5 B3E7 B3E7 B3E7 B3E7 3367 F366 B3E7 62EC      E68BAC       E68BAC       F3B09FA2
8A88     8A88 8A88 8A88 8A88 47B4 B3E8 B3E8 B3E8 B3E8 3368 F367 B3E8 6D3B      E6B4BB       E6B4BB       F3B09FA3
8A89     8A89 8A89 8A89 8A89 5441 B3E9 B3E9 B3E9 B3E9 3369 F368 B3E9 6E07      E6B887       E6B887       F3B09FA4
8A8A     8A8A 8A8A 8A8A 8A8A 4BA2 B3EA B3EA B3EA B3EA 336A F369 B3EA 6ED1      E6BB91       E6BB91       F3B09FA5
8A8B     8A8B 8A8B 8A8B 8A8B 48DF B3EB B3EB B3EB B3EB 336B F370 B3EB 845B      E8919B       E8919B       F3B09FA6
8A8C     8A8C 8A8C 8A8C 8A8C 545E B3EC B3EC B3EC 63EE 336C F371 B3EC 8910      E8A490       E8A490       F3B09FA7
8A8D     8A8D 8A8D 8A8D 8A8D 50FE B3ED B3ED B3ED B3ED 336D F372 B3ED 8F44      E8BD84       E8BD84       F3B09FA8
8A8E     8A8E 8A8E 8A8E 8A8E 4F97 B3EE B3EE B3EE B3EE 336E F373 B3EE 4E14      E4B894       E4B894       F3B09FA9
8A8F     8A8F 8A8F 8A8F 8A8F 53D5 B3EF B3EF B3EF B3EF 336F F374 B3EF 9C39      E9B0B9       E9B0B9       F3B09FAA
8A90     8A90 8A90 8A90 8A90 4F59 B3F0 B3F0 B3F0 B3F0 3370 F375 B3F0 53F6      E58FB6       E58FB6       F3B09FAB
8A91     8A91 8A91 8A91 8A91 5163 B3F1 B3F1 B3F1 B3F1 3371 F376 B3F1 691B      E6A49B       E6A49B       F3B09FAC
8A92     8A92 8A92 8A92 8A92 4E69 B3F2 B3F2 B3F2 B3F2 3372 F377 B3F2 6A3A      E6A8BA       E6A8BA       F3B09FAD
8A93     8A93 8A93 8A93 8A93 5445 B3F3 B3F3 B3F3 B3F3 3373 F378 B3F3 9784      E99E84       E99E84       F3B09FAE
8A94     8A94 8A94 8A94 8A94 4856 B3F4 B3F4 B3F4 B3F4 3374 F380 B3F4 682A      E6A0AA       E6A0AA       F3B09FAF
8A95     8A95 8A95 8A95 8A95 51BE B3F5 B3F5 B3F5 B3F5 3375 F38B B3F5 515C      E5859C       E5859C       F3B09FB0
8A96     8A96 8A96 8A96 8A96 6046 E3DE B3F6 B3F6 E3DE 635E 625F B3F6 7AC3      E7AB83       E7AB83       F3B09FB1
8A97     8A97 8A97 8A97 8A97 49B2 B3F7 B3F7 B3F7 B3F7 3377 F39C B3F7 84B2      E892B2       E892B2       F3B09FB2
8A98     8A98 8A98 8A98 8A98 4ACC B3F8 B3F8 B3F8 B3F8 3378 F3A0 B3F8 91DC      E9879C       E9879C       F3B09FB3
8A99     8A99 8A99 8A99 8A99 4962 B3F9 B3F9 B3F9 B3F9 3379 F3AB B3F9 938C      E98E8C       E98E8C       F3B09FB4
8A9A     8A9A 8A9A 8A9A 8A9A 547D B3FA B3FA B3FA 47BC D6AE AC55 B3FA 565B      E5999B       E5999B       F3B09FB5
8A9B     8A9B 8A9B 8A9B 8A9B 4C54 B3FB B3FB B3FB B3FB 337B F3C0 B3FB 9D28      E9B4A8       E9B4A8       F3B09FB6
8A9C     8A9C 8A9C 8A9C 8A9C 5242 B3FC B3FC B3FC B3FC 337C F36A B3FC 6822      E6A0A2       E6A0A2       F3B09FB7
8A9D     8A9D 8A9D 8A9D 8A9D 4BD3 B3FD B3FD B3FD B3FD 337D F3D0 B3FD 8305      E88C85       E88C85       F3B09FB8
8A9E     8A9E 8A9E 8A9E 8A9E 4F7C B3FE B3FE B3FE B3FE 337E F3A1 B3FE 8431      E890B1       E890B1       F3B09FB9
8A9F     8A9F 8A9F 8A9F 8A9F 52AB B4A1 B4A1 B4A1 B4A1 3421 F44F B4A1 7CA5      E7B2A5       E7B2A5       F3B09FBA
8AA0     8AA0 8AA0 8AA0 8AA0 4B9B B4A2 B4A2 B4A2 B4A2 3422 F47F B4A2 5208      E58888       E58888       F3B09FBB
8AA1     8AA1 8AA1 8AA1 8AA1 4E5D B4A3 B4A3 B4A3 B4A3 3423 F47B B4A3 82C5      E88B85       E88B85       F3B09FBC
8AA2     8AA2 8AA2 8AA2 8AA2 49C2 B4A4 B4A4 B4A4 B4A4 3424 F4E0 B4A4 74E6      E793A6       E793A6       F3B09FBD
8AA3     8AA3 8AA3 8AA3 8AA3 4970 B4A5 B4A5 B4A5 B4A5 3425 F46C B4A5 4E7E      E4B9BE       E4B9BE       F3B09FBE
8AA4     8AA4 8AA4 8AA4 8AA4 4F98 B4A6 B4A6 B4A6 B4A6 3426 F450 B4A6 4F83      E4BE83       E4BE83       F3B09FBF
8AA5     8AA5 8AA5 8AA5 8AA5 4E7B B4A7 B4A7 B4A7 B4A7 3427 F47D B4A7 51A0      E586A0       E586A0       F3B0A080
8AA6     8AA6 8AA6 8AA6 8AA6 4B81 B4A8 B4A8 B4A8 B4A8 3428 F44D B4A8 5BD2      E5AF92       E5AF92       F3B0A081
8AA7     8AA7 8AA7 8AA7 8AA7 4CD7 B4A9 B4A9 B4A9 B4A9 3429 F45D B4A9 520A      E5888A       E5888A       F3B0A082
8AA8     8AA8 8AA8 8AA8 8AA8 4B9A B4AA B4AA B4AA B4AA 342A F45C B4AA 52D8      E58B98       E58B98       F3B0A083
8AA9     8AA9 8AA9 8AA9 8AA9 4C7E B4AB B4AB B4AB B4AB 342B F44E B4AB 52E7      E58BA7       E58BA7       F3B0A084
8AAA     8AAA 8AAA 8AAA 8AAA 4893 B4AC B4AC B4AC B4AC 342C F46B B4AC 5DFB      E5B7BB       E5B7BB       F3B0A085
8AAB     8AAB 8AAB 8AAB 8AAB 52B5 B4AD B4AD B4AD B4AD 342D F460 B4AD 559A      E5969A       E5969A       F3B0A086
8AAC     8AAC 8AAC 8AAC 8AAC 4FAC B4AE B4AE B4AE B4AE 342E F44B B4AE 582A      E5A0AA       E5A0AA       F3B0A087
8AAD     8AAD 8AAD 8AAD 8AAD 55DE B4AF B4AF B4AF B4AF 342F F461 B4AF 59E6      E5A7A6       E5A7A6       F3B0A088
8AAE     8AAE 8AAE 8AAE 8AAE 48C5 B4B0 B4B0 B4B0 B4B0 3430 F4F0 B4B0 5B8C      E5AE8C       E5AE8C       F3B0A089
8AAF     8AAF 8AAF 8AAF 8AAF 49A2 B4B1 B4B1 B4B1 B4B1 3431 F4F1 B4B1 5B98      E5AE98       E5AE98       F3B0A08A
8AB0     8AB0 8AB0 8AB0 8AB0 47F8 B4B2 B4B2 B4B2 B4B2 3432 F4F2 B4B2 5BDB      E5AF9B       E5AF9B       F3B0A08B
8AB1     8AB1 8AB1 8AB1 8AB1 4A6F B4B3 B4B3 B4B3 B4B3 3433 F4F3 B4B3 5E72      E5B9B2       E5B9B2       F3B0A08C
8AB2     8AB2 8AB2 8AB2 8AB2 48DA B4B4 B4B4 B4B4 B4B4 3434 F4F4 B4B4 5E79      E5B9B9       E5B9B9       F3B0A08D
8AB3     8AB3 8AB3 8AB3 8AB3 4DEB B4B5 B4B5 B4B5 B4B5 3435 F4F5 B4B5 60A3      E682A3       E682A3       F3B0A08E
8AB4     8AB4 8AB4 8AB4 8AB4 47F7 B4B6 B4B6 B4B6 B4B6 3436 F4F6 B4B6 611F      E6849F       E6849F       F3B0A08F
8AB5     8AB5 8AB5 8AB5 8AB5 4DD3 B4B7 B4B7 B4B7 B4B7 3437 F4F7 B4B7 6163      E685A3       E685A3       F3B0A090
8AB6     8AB6 8AB6 8AB6 8AB6 51A5 B4B8 B4B8 B4B8 B4B8 3438 F4F8 B4B8 61BE      E686BE       E686BE       F3B0A091
8AB7     8AB7 8AB7 8AB7 8AB7 49AA B4B9 B4B9 B4B9 B4B9 3439 F4F9 B4B9 63DB      E68F9B       E68F9B       F3B0A092
8AB8     8AB8 8AB8 8AB8 8AB8 4FCF B4BA B4BA B4BA B4BA 343A F47A B4BA 6562      E695A2       E695A2       F3B0A093
8AB9     8AB9 8AB9 8AB9 8AB9 4EC1 B4BB B4BB B4BB B4BB 343B F45E B4BB 67D1      E69F91       E69F91       F3B0A094
8ABA     8ABA 8ABA 8ABA 8ABA 536D B4BC B4BC B4BC B4BC 343C F44C B4BC 6853      E6A193       E6A193       F3B0A095
8ABB     8ABB 8ABB 8ABB 8ABB 55B6 B4BD B4BD B4BD B4BD 343D F47E B4BD 68FA      E6A3BA       E6A3BA       F3B0A096
8ABC     8ABC 8ABC 8ABC 8ABC 4EFB B4BE B4BE B4BE B4BE 343E F46E B4BE 6B3E      E6ACBE       E6ACBE       F3B0A097
8ABD     8ABD 8ABD 8ABD 8ABD 4D4E B4BF B4BF B4BF B4BF 343F F46F B4BF 6B53      E6AD93       E6AD93       F3B0A098
8ABE     8ABE 8ABE 8ABE 8ABE 4EC4 B4C0 B4C0 B4C0 B4C0 3440 F47C B4C0 6C57      E6B197       E6B197       F3B0A099
8ABF     8ABF 8ABF 8ABF 8ABF 4F58 B4C1 B4C1 B4C1 B4C1 3441 F4C1 B4C1 6F22      E6BCA2       E6BCA2       F3B0A09A
8AC0     8AC0 8AC0 8AC0 8AC0 54BA B4C2 B4C2 B4C2 B4C2 3442 F4C2 B4C2 6F97      E6BE97       E6BE97       F3B0A09B
8AC1     8AC1 8AC1 8AC1 8AC1 5DAA DEF5 B4C3 B4C3 DEF5 5E75 5F8B B4C3 6F45      E6BD85       E6BD85       F3B0A09C
8AC2     8AC2 8AC2 8AC2 8AC2 4AB9 B4C4 B4C4 B4C4 B4C4 3444 F4C4 B4C4 74B0      E792B0       E792B0       F3B0A09D
8AC3     8AC3 8AC3 8AC3 8AC3 4B55 B4C5 B4C5 B4C5 B4C5 3445 F4C5 B4C5 7518      E79498       E79498       F3B0A09E
8AC4     8AC4 8AC4 8AC4 8AC4 49E3 B4C6 B4C6 B4C6 B4C6 3446 F4C6 B4C6 76E3      E79BA3       E79BA3       F3B0A09F
8AC5     8AC5 8AC5 8AC5 8AC5 4DA2 B4C7 B4C7 B4C7 B4C7 3447 F4C7 B4C7 770B      E79C8B       E79C8B       F3B0A0A0
8AC6     8AC6 8AC6 8AC6 8AC6 5287 B4C8 B4C8 B4C8 B4C8 3448 F4C8 B4C8 7AFF      E7ABBF       E7ABBF       F3B0A0A1
8AC7     8AC7 8AC7 8AC7 8AC7 46F2 B4C9 B4C9 B4C9 B4C9 3449 F4C9 B4C9 7BA1      E7AEA1       E7AEA1       F3B0A0A2
8AC8     8AC8 8AC8 8AC8 8AC8 4BEC B4CA B4CA B4CA B4CA 344A F4D1 B4CA 7C21      E7B0A1       E7B0A1       F3B0A0A3
8AC9     8AC9 8AC9 8AC9 8AC9 4CFC B4CB B4CB B4CB B4CB 344B F4D2 B4CB 7DE9      E7B7A9       E7B7A9       F3B0A0A4
8ACA     8ACA 8ACA 8ACA 8ACA 5084 B4CC B4CC B4CC B4CC 344C F4D3 B4CC 7F36      E7BCB6       E7BCB6       F3B0A0A5
8ACB     8ACB 8ACB 8ACB 8ACB 54B6 B4CD B4CD B4CD B4CD 344D F4D4 B4CD 7FF0      E7BFB0       E7BFB0       F3B0A0A6
8ACC     8ACC 8ACC 8ACC 8ACC 4C8C B4CE B4CE B4CE B4CE 344E F4D5 B4CE 809D      E8829D       E8829D       F3B0A0A7
8ACD     8ACD 8ACD 8ACD 8ACD 4C4A B4CF B4CF B4CF B4CF 344F F4D6 B4CF 8266      E889A6       E889A6       F3B0A0A8
8ACE     8ACE 8ACE 8ACE 8ACE 51EE B4D0 B4D0 B4D0 B4D0 3450 F4D7 B4D0 839E      E88E9E       E88E9E       F3B0A0A9
8ACF     8ACF 8ACF 8ACF 8ACF 487E B4D1 B4D1 B4D1 B4D1 3451 F4D8 B4D1 89B3      E8A6B3       E8A6B3       F3B0A0AA
8AD0     8AD0 8AD0 8AD0 8AD0 63E0 EBDD B4D2 B4D2 EBDD 6B5D 705A B4D2 8ACC      E8AB8C       E8AB8C       F3B0A0AB
8AD1     8AD1 8AD1 8AD1 8AD1 49A1 B4D3 B4D3 B4D3 B4D3 3453 F4E2 B4D3 8CAB      E8B2AB       E8B2AB       F3B0A0AC
8AD2     8AD2 8AD2 8AD2 8AD2 4DA9 B4D4 B4D4 B4D4 B4D4 3454 F4E3 B4D4 9084      E98284       E98284       F3B0A0AD
8AD3     8AD3 8AD3 8AD3 8AD3 4E4E B4D5 B4D5 B4D5 B4D5 3455 F4E4 B4D5 9451      E99191       E99191       F3B0A0AE
8AD4     8AD4 8AD4 8AD4 8AD4 45C1 B4D6 B4D6 B4D6 B4D6 3456 F4E5 B4D6 9593      E99693       E99693       F3B0A0AF
8AD5     8AD5 8AD5 8AD5 8AD5 4DD9 B4D7 B4D7 B4D7 B4D7 3457 F4E6 B4D7 9591      E99691       E99691       F3B0A0B0
8AD6     8AD6 8AD6 8AD6 8AD6 45C5 B4D8 B4D8 B4D8 B4D8 3458 F4E7 B4D8 95A2      E996A2       E996A2       F3B0A0B1
8AD7     8AD7 8AD7 8AD7 8AD7 4FFE B4D9 B4D9 B4D9 B4D9 3459 F4E8 B4D9 9665      E999A5       E999A5       F3B0A0B2
8AD8     8AD8 8AD8 8AD8 8AD8 4E75 B4DA B4DA B4DA B4DA 345A F4E9 B4DA 97D3      E99F93       E99F93       F3B0A0B3
8AD9     8AD9 8AD9 8AD9 8AD9 4862 B4DB B4DB B4DB B4DB 345B F44A B4DB 9928      E9A4A8       E9A4A8       F3B0A0B4
8ADA     8ADA 8ADA 8ADA 8ADA 4CAA B4DC B4DC B4DC B4DC 345C F45B B4DC 8218      E88898       E88898       F3B0A0B5
8ADB     8ADB 8ADB 8ADB 8ADB 467A B4DD B4DD B4DD B4DD 345D F45A B4DD 4E38      E4B8B8       E4B8B8       F3B0A0B6
8ADC     8ADC 8ADC 8ADC 8ADC 49CB B4DE B4DE B4DE B4DE 345E F45F B4DE 542B      E590AB       E590AB       F3B0A0B7
8ADD     8ADD 8ADD 8ADD 8ADD 4748 B4DF B4DF B4DF B4DF 345F F46D B4DF 5CB8      E5B2B8       E5B2B8       F3B0A0B8
8ADE     8ADE 8ADE 8ADE 8ADE 49CF B4E0 B4E0 B4E0 B4E0 3460 F479 B4E0 5DCC      E5B78C       E5B78C       F3B0A0B9
8ADF     8ADF 8ADF 8ADF 8ADF 4E4C B4E1 B4E1 B4E1 B4E1 3461 F457 B4E1 73A9      E78EA9       E78EA9       F3B0A0BA
8AE0     8AE0 8AE0 8AE0 8AE0 5447 B4E2 B4E2 B4E2 B4E2 3462 F459 B4E2 764C      E7998C       E7998C       F3B0A0BB
8AE1     8AE1 8AE1 8AE1 8AE1 49A5 B4E3 B4E3 B4E3 B4E3 3463 F462 B4E3 773C      E79CBC       E79CBC       F3B0A0BC
8AE2     8AE2 8AE2 8AE2 8AE2 45BF B4E4 B4E4 B4E4 B4E4 3464 F463 B4E4 5CA9      E5B2A9       E5B2A9       F3B0A0BD
8AE3     8AE3 8AE3 8AE3 8AE3 558B B4E5 B4E5 B4E5 B4E5 3465 F464 B4E5 7FEB      E7BFAB       E7BFAB       F3B0A0BE
8AE4     8AE4 8AE4 8AE4 8AE4 5584 B4E6 B4E6 B4E6 B4E6 3466 F465 B4E6 8D0B      E8B48B       E8B48B       F3B0A0BF
8AE5     8AE5 8AE5 8AE5 8AE5 51A4 B4E7 B4E7 B4E7 B4E7 3467 F466 B4E7 96C1      E99B81       E99B81       F3B0A180
8AE6     8AE6 8AE6 8AE6 8AE6 539D B4E8 B4E8 B4E8 B4E8 3468 F467 B4E8 9811      E9A091       E9A091       F3B0A181
8AE7     8AE7 8AE7 8AE7 8AE7 48B8 B4E9 B4E9 B4E9 B4E9 3469 F468 B4E9 9854      E9A194       E9A194       F3B0A182
8AE8     8AE8 8AE8 8AE8 8AE8 4C57 B4EA B4EA B4EA B4EA 346A F469 B4EA 9858      E9A198       E9A198       F3B0A183
8AE9     8AE9 8AE9 8AE9 8AE9 47D2 B4EB B4EB B4EB B4EB 346B F470 B4EB 4F01      E4BC81       E4BC81       F3B0A184
8AEA     8AEA 8AEA 8AEA 8AEA 4E9F B4EC B4EC B4EC B4EC 346C F471 B4EC 4F0E      E4BC8E       E4BC8E       F3B0A185
8AEB     8AEB 8AEB 8AEB 8AEB 4B91 B4ED B4ED B4ED B4ED 346D F472 B4ED 5371      E58DB1       E58DB1       F3B0A186
8AEC     8AEC 8AEC 8AEC 8AEC 45D1 B4EE B4EE B4EE B4EE 346E F473 B4EE 559C      E5969C       E5969C       F3B0A187
8AED     8AED 8AED 8AED 8AED 45FC B4EF B4EF B4EF B4EF 346F F474 B4EF 5668      E599A8       E599A8       F3B0A188
8AEE     8AEE 8AEE 8AEE 8AEE 46F1 B4F0 B4F0 B4F0 B4F0 3470 F475 B4F0 57FA      E59FBA       E59FBA       F3B0A189
8AEF     8AEF 8AEF 8AEF 8AEF 4D69 B4F1 B4F1 B4F1 B4F1 3471 F476 B4F1 5947      E5A587       E5A587       F3B0A18A
8AF0     8AF0 8AF0 8AF0 8AF0 4F8B B4F2 B4F2 B4F2 B4F2 3472 F477 B4F2 5B09      E5AC89       E5AC89       F3B0A18B
8AF1     8AF1 8AF1 8AF1 8AF1 49DF B4F3 B4F3 B4F3 B4F3 3473 F478 B4F3 5BC4      E5AF84       E5AF84       F3B0A18C
8AF2     8AF2 8AF2 8AF2 8AF2 474A B4F4 B4F4 B4F4 B4F4 3474 F480 B4F4 5C90      E5B290       E5B290       F3B0A18D
8AF3     8AF3 8AF3 8AF3 8AF3 4AB5 B4F5 B4F5 B4F5 B4F5 3475 F48B B4F5 5E0C      E5B88C       E5B88C       F3B0A18E
8AF4     8AF4 8AF4 8AF4 8AF4 4B57 B4F6 B4F6 B4F6 B4F6 3476 F49B B4F6 5E7E      E5B9BE       E5B9BE       F3B0A18F
8AF5     8AF5 8AF5 8AF5 8AF5 5097 B4F7 B4F7 B4F7 B4F7 3477 F49C B4F7 5FCC      E5BF8C       E5BF8C       F3B0A190
8AF6     8AF6 8AF6 8AF6 8AF6 4BF2 B4F8 B4F8 B4F8 B4F8 3478 F4A0 B4F8 63EE      E68FAE       E68FAE       F3B0A191
8AF7     8AF7 8AF7 8AF7 8AF7 4D96 B4F9 B4F9 B4F9 B4F9 3479 F4AB B4F9 673A      E69CBA       E69CBA       F3B0A192
8AF8     8AF8 8AF8 8AF8 8AF8 4DAB B4FA B4FA B4FA B4FA 347A F4B0 B4FA 65D7      E69797       E69797       F3B0A193
8AF9     8AF9 8AF9 8AF9 8AF9 4B77 B4FB B4FB B4FB B4FB 347B F4C0 B4FB 65E2      E697A2       E697A2       F3B0A194
8AFA     8AFA 8AFA 8AFA 8AFA 468B B4FC B4FC B4FC B4FC 347C F46A B4FC 671F      E69C9F       E69C9F       F3B0A195
8AFB     8AFB 8AFB 8AFB 8AFB 4EF1 B4FD B4FD B4FD B4FD 347D F4D0 B4FD 68CB      E6A38B       E6A38B       F3B0A196
8AFC     8AFC 8AFC 8AFC 8AFC 4EB1 B4FE B4FE B4FE B4FE 347E F4A1 B4FE 68C4      E6A384       E6A384       F3B0A197
8B40     8B40 8B40 8B40 8B40 4579 B5A1 B5A1 B5A1 B5A1 3521 F54F B5A1 6A5F      E6A99F       E6A99F       F3B0A198
8B41     8B41 8B41 8B41 8B41 4982 B5A2 B5A2 B5A2 B5A2 3522 F57F B5A2 5E30      E5B8B0       E5B8B0       F3B0A199
8B42     8B42 8B42 8B42 8B42 4A5B B5A3 B5A3 B5A3 B5A3 3523 F57B B5A3 6BC5      E6AF85       E6AF85       F3B0A19A
8B43     8B43 8B43 8B43 8B43 45B9 B5A4 B5A4 B5A4 B5A4 3524 F5E0 B5A4 6C17      E6B097       E6B097       F3B0A19B
8B44     8B44 8B44 8B44 8B44 494C B5A5 B5A5 B5A5 B5A5 3525 F56C B5A5 6C7D      E6B1BD       E6B1BD       F3B0A19C
8B45     8B45 8B45 8B45 8B45 50D4 B5A6 B5A6 B5A6 B5A6 3526 F550 B5A6 757F      E795BF       E795BF       F3B0A19D
8B46     8B46 8B46 8B46 8B46 4ED8 B5A7 B5A7 B5A7 B5A7 3527 F57D B5A7 7948      E7A588       E7A588       F3B0A19E
8B47     8B47 8B47 8B47 8B47 4AEC B5A8 B5A8 B5A8 B5A8 3528 F54D B5A8 5B63      E5ADA3       E5ADA3       F3B0A19F
8B48     8B48 8B48 8B48 8B48 5264 B5A9 B5A9 B5A9 B5A9 3529 F55D B5A9 7A00      E7A880       E7A880       F3B0A1A0
8B49     8B49 8B49 8B49 8B49 46DB B5AA B5AA B5AA B5AA 352A F55C B5AA 7D00      E7B480       E7B480       F3B0A1A1
8B4A     8B4A 8B4A 8B4A 8B4A 55E0 B5AB B5AB B5AB B5AB 352B F54E B5AB 5FBD      E5BEBD       E5BEBD       F3B0A1A2
8B4B     8B4B 8B4B 8B4B 8B4B 46FB B5AC B5AC B5AC B5AC 352C F56B B5AC 898F      E8A68F       E8A68F       F3B0A1A3
8B4C     8B4C 8B4C 8B4C 8B4C 4794 B5AD B5AD B5AD B5AD 352D F560 B5AD 8A18      E8A898       E8A898       F3B0A1A4
8B4D     8B4D 8B4D 8B4D 8B4D 487F B5AE B5AE B5AE B5AE 352E F54B B5AE 8CB4      E8B2B4       E8B2B4       F3B0A1A5
8B4E     8B4E 8B4E 8B4E 8B4E 48D0 B5AF B5AF B5AF B5AF 352F F561 B5AF 8D77      E8B5B7       E8B5B7       F3B0A1A6
8B4F     8B4F 8B4F 8B4F 8B4F 4DAA B5B0 B5B0 B5B0 B5B0 3530 F5F0 B5B0 8ECC      E8BB8C       E8BB8C       F3B0A1A7
8B50     8B50 8B50 8B50 8B50 4787 B5B1 B5B1 B5B1 B5B1 3531 F5F1 B5B1 8F1D      E8BC9D       E8BC9D       F3B0A1A8
8B51     8B51 8B51 8B51 8B51 53CE B5B2 B5B2 B5B2 B5B2 3532 F5F2 B5B2 98E2      E9A3A2       E9A3A2       F3B0A1A9
8B52     8B52 8B52 8B52 8B52 5058 B5B3 B5B3 B5B3 B5B3 3533 F5F3 B5B3 9A0E      E9A88E       E9A88E       F3B0A1AA
8B53     8B53 8B53 8B53 8B53 4AC8 B5B4 B5B4 B5B4 B5B4 3534 F5F4 B5B4 9B3C      E9ACBC       E9ACBC       F3B0A1AB
8B54     8B54 8B54 8B54 8B54 47C0 B5B5 B5B5 B5B5 B5B5 3535 F5F5 B5B5 4E80      E4BA80       E4BA80       F3B0A1AC
8B55     8B55 8B55 8B55 8B55 5059 B5B6 B5B6 B5B6 B5B6 3536 F5F6 B5B6 507D      E581BD       E581BD       F3B0A1AD
8B56     8B56 8B56 8B56 8B56 49CA B5B7 B5B7 B5B7 B5B7 3537 F5F7 B5B7 5100      E58480       E58480       F3B0A1AE
8B57     8B57 8B57 8B57 8B57 5395 B5B8 B5B8 B5B8 B5B8 3538 F5F8 B5B8 5993      E5A693       E5A693       F3B0A1AF
8B58     8B58 8B58 8B58 8B58 4D8B B5B9 B5B9 B5B9 B5B9 3539 F5F9 B5B9 5B9C      E5AE9C       E5AE9C       F3B0A1B0
8B59     8B59 8B59 8B59 8B59 4F92 B5BA B5BA B5BA B5BA 353A F57A B5BA 622F      E688AF       E688AF       F3B0A1B1
8B5A     8B5A 8B5A 8B5A 8B5A 48F7 B5BB B5BB B5BB B5BB 353B F55E B5BB 6280      E68A80       E68A80       F3B0A1B2
8B5B     8B5B 8B5B 8B5B 8B5B 4FCD B5BC B5BC B5BC B5BC 353C F54C B5BC 64EC      E693AC       E693AC       F3B0A1B3
8B5C     8B5C 8B5C 8B5C 8B5C 517D B5BD B5BD B5BD B5BD 353D F57E B5BD 6B3A      E6ACBA       E6ACBA       F3B0A1B4
8B5D     8B5D 8B5D 8B5D 8B5D 4FDC B5BE B5BE B5BE B5BE 353E F56E B5BE 72A0      E78AA0       E78AA0       F3B0A1B5
8B5E     8B5E 8B5E 8B5E 8B5E 4CB4 B5BF B5BF B5BF B5BF 353F F56F B5BF 7591      E79691       E79691       F3B0A1B6
8B5F     8B5F 8B5F 8B5F 8B5F 5192 B5C0 B5C0 B5C0 B5C0 3540 F57C B5C0 7947      E7A587       E7A587       F3B0A1B7
8B60     8B60 8B60 8B60 8B60 4598 B5C1 B5C1 B5C1 B5C1 3541 F5C1 B5C1 7FA9      E7BEA9       E7BEA9       F3B0A1B8
8B61     8B61 8B61 8B61 8B61 51C6 B5C2 B5C2 B5C2 B5C2 3542 F5C2 B5C2 87FB      E89FBB       E89FBB       F3B0A1B9
8B62     8B62 8B62 8B62 8B62 5265 B5C3 B5C3 B5C3 B5C3 3543 F5C3 B5C3 8ABC      E8AABC       E8AABC       F3B0A1BA
8B63     8B63 8B63 8B63 8B63 4781 B5C4 B5C4 B5C4 B5C4 3544 F5C4 B5C4 8B70      E8ADB0       E8ADB0       F3B0A1BB
8B64     8B64 8B64 8B64 8B64 5541 B5C5 B5C5 B5C5 B5C5 3545 F5C5 B5C5 63AC      E68EAC       E68EAC       F3B0A1BC
8B65     8B65 8B65 8B65 8B65 46D7 B5C6 B5C6 B5C6 B5C6 3546 F5C6 B5C6 83CA      E88F8A       E88F8A       F3B0A1BD
8B66     8B66 8B66 8B66 8B66 51ED B5C7 B5C7 B5C7 B5C7 3547 F5C7 B5C7 97A0      E99EA0       E99EA0       F3B0A1BE
8B67     8B67 8B67 8B67 8B67 457E B5C8 B5C8 B5C8 B5C8 3548 F5C8 B5C8 5409      E59089       E59089       F3B0A1BF
8B68     8B68 8B68 8B68 8B68 55EA B5C9 B5C9 B5C9 B5C9 3549 F5C9 B5C9 5403      E59083       E59083       F3B0A280
8B69     8B69 8B69 8B69 8B69 4D6D B5CA B5CA B5CA B5CA 354A F5D1 B5CA 55AB      E596AB       E596AB       F3B0A281
8B6A     8B6A 8B6A 8B6A 8B6A 534B B5CB B5CB B5CB B5CB 354B F5D2 B5CB 6854      E6A194       E6A194       F3B0A282
8B6B     8B6B 8B6B 8B6B 8B6B 4BC2 B5CC B5CC B5CC B5CC 354C F5D3 B5CC 6A58      E6A998       E6A998       F3B0A283
8B6C     8B6C 8B6C 8B6C 8B6C 4B5F B5CD B5CD B5CD B5CD 354D F5D4 B5CD 8A70      E8A9B0       E8A9B0       F3B0A284
8B6D     8B6D 8B6D 8B6D 8B6D 5248 B5CE B5CE B5CE B5CE 354E F5D5 B5CE 7827      E7A0A7       E7A0A7       F3B0A285
8B6E     8B6E 8B6E 8B6E 8B6E 4AA6 B5CF B5CF B5CF B5CF 354F F5D6 B5CF 6775      E69DB5       E69DB5       F3B0A286
8B6F     8B6F 8B6F 8B6F 8B6F 55AE B5D0 B5D0 B5D0 B5D0 3550 F5D7 B5D0 9ECD      E9BB8D       E9BB8D       F3B0A287
8B70     8B70 8B70 8B70 8B70 4AA4 B5D1 B5D1 B5D1 B5D1 3551 F5D8 B5D1 5374      E58DB4       E58DB4       F3B0A288
8B71     8B71 8B71 8B71 8B71 4ADE B5D2 B5D2 B5D2 B5D2 3552 F5D9 B5D2 5BA2      E5AEA2       E5AEA2       F3B0A289
8B72     8B72 8B72 8B72 8B72 4D5C B5D3 B5D3 B5D3 B5D3 3553 F5E2 B5D3 811A      E8849A       E8849A       F3B0A28A
8B73     8B73 8B73 8B73 8B73 5260 B5D4 B5D4 B5D4 B5D4 3554 F5E3 B5D4 8650      E89990       E89990       F3B0A28B
8B74     8B74 8B74 8B74 8B74 4BFC B5D5 B5D5 B5D5 B5D5 3555 F5E4 B5D5 9006      E98086       E98086       F3B0A28C
8B75     8B75 8B75 8B75 8B75 4D56 B5D6 B5D6 B5D6 B5D6 3556 F5E5 B5D6 4E18      E4B898       E4B898       F3B0A28D
8B76     8B76 8B76 8B76 8B76 458A B5D7 B5D7 B5D7 B5D7 3557 F5E6 B5D7 4E45      E4B985       E4B985       F3B0A28E
8B77     8B77 8B77 8B77 8B77 5085 B5D8 B5D8 B5D8 B5D8 3558 F5E7 B5D8 4EC7      E4BB87       E4BB87       F3B0A28F
8B78     8B78 8B78 8B78 8B78 4A81 B5D9 B5D9 B5D9 B5D9 3559 F5E8 B5D9 4F11      E4BC91       E4BC91       F3B0A290
8B79     8B79 8B79 8B79 8B79 485C B5DA B5DA B5DA B5DA 355A F5E9 B5DA 53CA      E58F8A       E58F8A       F3B0A291
8B7A     8B7A 8B7A 8B7A 8B7A 4B98 B5DB B5DB B5DB B5DB 355B F54A B5DB 5438      E590B8       E590B8       F3B0A292
8B7B     8B7B 8B7B 8B7B 8B7B 4592 B5DC B5DC B5DC B5DC 355C F55B B5DC 5BAE      E5AEAE       E5AEAE       F3B0A293
8B7C     8B7C 8B7C 8B7C 8B7C 4B63 B5DD B5DD B5DD B5DD 355D F55A B5DD 5F13      E5BC93       E5BC93       F3B0A294
8B7D     8B7D 8B7D 8B7D 8B7D 47CC B5DE B5DE B5DE B5DE 355E F55F B5DE 6025      E680A5       E680A5       F3B0A295
8B7E     8B7E 8B7E 8B7E 8B7E 4C5C B5DF B5DF B5DF B5DF 355F F56D B5DF 6551      E69591       E69591       F3B0A296
8B80     8B80 8B80 8B80 8B80 50F0 B5E0 B5E0 B5E0 B5E0 3560 F579 B5E0 673D      E69CBD       E69CBD       F3B0A297
8B81     8B81 8B81 8B81 8B81 49A0 B5E1 B5E1 B5E1 B5E1 3561 F557 B5E1 6C42      E6B182       E6B182       F3B0A298
8B82     8B82 8B82 8B82 8B82 50C8 B5E2 B5E2 B5E2 B5E2 3562 F559 B5E2 6C72      E6B1B2       E6B1B2       F3B0A299
8B83     8B83 8B83 8B83 8B83 4D4B B5E3 B5E3 B5E3 B5E3 3563 F562 B5E3 6CE3      E6B3A3       E6B3A3       F3B0A29A
8B84     8B84 8B84 8B84 8B84 5473 B5E4 B5E4 B5E4 B5E4 3564 F563 B5E4 7078      E781B8       E781B8       F3B0A29B
8B85     8B85 8B85 8B85 8B85 48DB B5E5 B5E5 B5E5 B5E5 3565 F564 B5E5 7403      E79083       E79083       F3B0A29C
8B86     8B86 8B86 8B86 8B86 4986 B5E6 B5E6 B5E6 B5E6 3566 F565 B5E6 7A76      E7A9B6       E7A9B6       F3B0A29D
8B87     8B87 8B87 8B87 8B87 50AA B5E7 B5E7 B5E7 B5E7 3567 F566 B5E7 7AAE      E7AAAE       E7AAAE       F3B0A29E
8B88     8B88 8B88 8B88 8B88 51CF B5E8 B5E8 B5E8 B5E8 3568 F567 B5E8 7B08      E7AC88       E7AC88       F3B0A29F
8B89     8B89 8B89 8B89 8B89 49F3 B5E9 B5E9 B5E9 B5E9 3569 F568 B5E9 7D1A      E7B49A       E7B49A       F3B0A2A0
8B8A     8B8A 8B8A 8B8A 8B8A 51FE B5EA B5EA B5EA B5EA 356A F569 B5EA 7CFE      E7B3BE       E7B3BE       F3B0A2A1
8B8B     8B8B 8B8B 8B8B 8B8B 4757 B5EB B5EB B5EB B5EB 356B F570 B5EB 7D66      E7B5A6       E7B5A6       F3B0A2A2
8B8C     8B8C 8B8C 8B8C 8B8C 4C7B B5EC B5EC B5EC B5EC 356C F571 B5EC 65E7      E697A7       E697A7       F3B0A2A3
8B8D     8B8D 8B8D 8B8D 8B8D 4973 B5ED B5ED B5ED B5ED 356D F572 B5ED 725B      E7899B       E7899B       F3B0A2A4
8B8E     8B8E 8B8E 8B8E 8B8E 4ACA B5EE B5EE B5EE B5EE 356E F573 B5EE 53BB      E58EBB       E58EBB       F3B0A2A5
8B8F     8B8F 8B8F 8B8F 8B8F 485E B5EF B5EF B5EF B5EF 356F F574 B5EF 5C45      E5B185       E5B185       F3B0A2A6
8B90     8B90 8B90 8B90 8B90 4A52 B5F0 B5F0 B5F0 B5F0 3570 F575 B5F0 5DE8      E5B7A8       E5B7A8       F3B0A2A7
8B91     8B91 8B91 8B91 8B91 4F61 B5F1 B5F1 B5F1 B5F1 3571 F576 B5F1 62D2      E68B92       E68B92       F3B0A2A8
8B92     8B92 8B92 8B92 8B92 4D77 B5F2 B5F2 B5F2 B5F2 3572 F577 B5F2 62E0      E68BA0       E68BA0       F3B0A2A9
8B93     8B93 8B93 8B93 8B93 4B82 B5F3 B5F3 B5F3 B5F3 3573 F578 B5F3 6319      E68C99       E68C99       F3B0A2AA
8B94     8B94 8B94 8B94 8B94 527A B5F4 B5F4 B5F4 B5F4 3574 F580 B5F4 6E20      E6B8A0       E6B8A0       F3B0A2AB
8B95     8B95 8B95 8B95 8B95 4F72 B5F5 B5F5 B5F5 B5F5 3575 F58B B5F5 865A      E8999A       E8999A       F3B0A2AC
8B96     8B96 8B96 8B96 8B96 4A9A B5F6 B5F6 B5F6 B5F6 3576 F59B B5F6 8A31      E8A8B1       E8A8B1       F3B0A2AD
8B97     8B97 8B97 8B97 8B97 4DC2 B5F7 B5F7 B5F7 B5F7 3577 F59C B5F7 8DDD      E8B79D       E8B79D       F3B0A2AE
8B98     8B98 8B98 8B98 8B98 51E7 B5F8 B5F8 B5F8 B5F8 3578 F5A0 B5F8 92F8      E98BB8       E98BB8       F3B0A2AF
8B99     8B99 8B99 8B99 8B99 4B78 B5F9 B5F9 B5F9 B5F9 3579 F5AB B5F9 6F01      E6BC81       E6BC81       F3B0A2B0
8B9A     8B9A 8B9A 8B9A 8B9A 55AA B5FA B5FA B5FA B5FA 357A F5B0 B5FA 79A6      E7A6A6       E7A6A6       F3B0A2B1
8B9B     8B9B 8B9B 8B9B 8B9B 48EE B5FB B5FB B5FB B5FB 357B F5C0 B5FB 9B5A      E9AD9A       E9AD9A       F3B0A2B2
8B9C     8B9C 8B9C 8B9C 8B9C 4BAC B5FC B5FC B5FC B5FC 357C F56A B5FC 4EA8      E4BAA8       E4BAA8       F3B0A2B3
8B9D     8B9D 8B9D 8B9D 8B9D 4B73 B5FD B5FD B5FD B5FD 357D F5D0 B5FD 4EAB      E4BAAB       E4BAAB       F3B0A2B4
8B9E     8B9E 8B9E 8B9E 8B9E 4575 B5FE B5FE B5FE B5FE 357E F5A1 B5FE 4EAC      E4BAAC       E4BAAC       F3B0A2B5
8B9F     8B9F 8B9F 8B9F 8B9F 48BC B6A1 B6A1 B6A1 B6A1 3621 F64F B6A1 4F9B      E4BE9B       E4BE9B       F3B0A2B6
8BA0     8BA0 8BA0 8BA0 8BA0 52EC B6A2 B6A2 B6A2 43CB B2E2 82B4 B6A2 4FA0      E4BEA0       E4BEA0       F3B0A2B7
8BA1     8BA1 8BA1 8BA1 8BA1 55AD B6A3 B6A3 B6A3 B6A3 3623 F67B B6A3 50D1      E58391       E58391       F3B0A2B8
8BA2     8BA2 8BA2 8BA2 8BA2 555A B6A4 B6A4 B6A4 B6A4 3624 F6E0 B6A4 5147      E58587       E58587       F3B0A2B9
8BA3     8BA3 8BA3 8BA3 8BA3 4971 B6A5 B6A5 B6A5 B6A5 3625 F66C B6A5 7AF6      E7ABB6       E7ABB6       F3B0A2BA
8BA4     8BA4 8BA4 8BA4 8BA4 47B3 B6A6 B6A6 B6A6 B6A6 3626 F650 B6A6 5171      E585B1       E585B1       F3B0A2BB
8BA5     8BA5 8BA5 8BA5 8BA5 51EC B6A7 B6A7 B6A7 B6A7 3627 F67D B6A7 51F6      E587B6       E587B6       F3B0A2BC
8BA6     8BA6 8BA6 8BA6 8BA6 47D7 B6A8 B6A8 B6A8 B6A8 3628 F64D B6A8 5354      E58D94       E58D94       F3B0A2BD
8BA7     8BA7 8BA7 8BA7 8BA7 4D4D B6A9 B6A9 B6A9 B6A9 3629 F65D B6A9 5321      E58CA1       E58CA1       F3B0A2BE
8BA8     8BA8 8BA8 8BA8 8BA8 53E2 B6AA B6AA B6AA B6AA 362A F65C B6AA 537F      E58DBF       E58DBF       F3B0A2BF
8BA9     8BA9 8BA9 8BA9 8BA9 4DD5 B6AB B6AB B6AB B6AB 362B F64E B6AB 53EB      E58FAB       E58FAB       F3B0A380
8BAA     8BAA 8BAA 8BAA 8BAA 4C5F B6AC B6AC B6AC B6AC 362C F66B B6AC 55AC      E596AC       E596AC       F3B0A381
8BAB     8BAB 8BAB 8BAB 8BAB 4A95 B6AD B6AD B6AD B6AD 362D F660 B6AD 5883      E5A283       E5A283       F3B0A382
8BAC     8BAC 8BAC 8BAC 8BAC 51B3 B6AE B6AE B6AE B6AE 362E F64B B6AE 5CE1      E5B3A1       E5B3A1       F3B0A383
8BAD     8BAD 8BAD 8BAD 8BAD 46D2 B6AF B6AF B6AF B6AF 362F F661 B6AF 5F37      E5BCB7       E5BCB7       F3B0A384
8BAE     8BAE 8BAE 8BAE 8BAE 5385 B6B0 B6B0 B6B0 B6B0 3630 F6F0 B6B0 5F4A      E5BD8A       E5BD8A       F3B0A385
8BAF     8BAF 8BAF 8BAF 8BAF 5542 B6B1 B6B1 B6B1 B6B1 3631 F6F1 B6B1 602F      E680AF       E680AF       F3B0A386
8BB0     8BB0 8BB0 8BB0 8BB0 4BD5 B6B2 B6B2 B6B2 B6B2 3632 F6F2 B6B2 6050      E68190       E68190       F3B0A387
8BB1     8BB1 8BB1 8BB1 8BB1 494E B6B3 B6B3 B6B3 B6B3 3633 F6F3 B6B3 606D      E681AD       E681AD       F3B0A388
8BB2     8BB2 8BB2 8BB2 8BB2 55B1 B6B4 B6B4 B6B4 B6B4 3634 F6F4 B6B4 631F      E68C9F       E68C9F       F3B0A389
8BB3     8BB3 8BB3 8BB3 8BB3 47A5 B6B5 B6B5 B6B5 B6B5 3635 F6F5 B6B5 6559      E69599       E69599       F3B0A38A
8BB4     8BB4 8BB4 8BB4 8BB4 4589 B6B6 B6B6 B6B6 B6B6 3636 F6F6 B6B6 6A4B      E6A98B       E6A98B       F3B0A38B
8BB5     8BB5 8BB5 8BB5 8BB5 4A58 B6B7 B6B7 B6B7 B6B7 3637 F6F7 B6B7 6CC1      E6B381       E6B381       F3B0A38C
8BB6     8BB6 8BB6 8BB6 8BB6 4E5A B6B8 B6B8 B6B8 B6B8 3638 F6F8 B6B8 72C2      E78B82       E78B82       F3B0A38D
8BB7     8BB7 8BB7 8BB7 8BB7 4C9E B6B9 B6B9 B6B9 B6B9 3639 F6F9 B6B9 72ED      E78BAD       E78BAD       F3B0A38E
8BB8     8BB8 8BB8 8BB8 8BB8 51BB B6BA B6BA B6BA B6BA 363A F67A B6BA 77EF      E79FAF       E79FAF       F3B0A38F
8BB9     8BB9 8BB9 8BB9 8BB9 4BE3 B6BB B6BB B6BB B6BB 363B F65E B6BB 80F8      E883B8       E883B8       F3B0A390
8BBA     8BBA 8BBA 8BBA 8BBA 5051 B6BC B6BC B6BC B6BC 363C F64C B6BC 8105      E88485       E88485       F3B0A391
8BBB     8BBB 8BBB 8BBB 8BBB 466A B6BD B6BD B6BD B6BD 363D F67E B6BD 8208      E88888       E88888       F3B0A392
8BBC     8BBC 8BBC 8BBC 8BBC 5565 B6BE B6BE B6BE B6BE 363E F66E B6BE 854E      E8958E       E8958E       F3B0A393
8BBD     8BBD 8BBD 8BBD 8BBD 4995 B6BF B6BF B6BF B6BF 363F F66F B6BF 90F7      E983B7       E983B7       F3B0A394
8BBE     8BBE 8BBE 8BBE 8BBE 4A8D B6C0 B6C0 B6C0 B6C0 3640 F67C B6C0 93E1      E98FA1       E98FA1       F3B0A395
8BBF     8BBF 8BBF 8BBF 8BBF 4A5C B6C1 B6C1 B6C1 B6C1 3641 F6C1 B6C1 97FF      E99FBF       E99FBF       F3B0A396
8BC0     8BC0 8BC0 8BC0 8BC0 51D5 B6C2 B6C2 B6C2 B6C2 3642 F6C2 B6C2 9957      E9A597       E9A597       F3B0A397
8BC1     8BC1 8BC1 8BC1 8BC1 4CD5 B6C3 B6C3 B6C3 B6C3 3643 F6C3 B6C3 9A5A      E9A99A       E9A99A       F3B0A398
8BC2     8BC2 8BC2 8BC2 8BC2 4DED B6C4 B6C4 B6C4 B6C4 3644 F6C4 B6C4 4EF0      E4BBB0       E4BBB0       F3B0A399
8BC3     8BC3 8BC3 8BC3 8BC3 4ED9 B6C5 B6C5 B6C5 B6C5 3645 F6C5 B6C5 51DD      E5879D       E5879D       F3B0A39A
8BC4     8BC4 8BC4 8BC4 8BC4 6880 B6C6 B6C6 B6C6 47C8 B6DA 86BA B6C6 5C2D      E5B0AD       E5B0AD       F3B0A39B
8BC5     8BC5 8BC5 8BC5 8BC5 4C83 B6C7 B6C7 B6C7 B6C7 3647 F6C7 B6C7 6681      E69A81       E69A81       F3B0A39C
8BC6     8BC6 8BC6 8BC6 8BC6 456B B6C8 B6C8 B6C8 B6C8 3648 F6C8 B6C8 696D      E6A5AD       E6A5AD       F3B0A39D
8BC7     8BC7 8BC7 8BC7 8BC7 488B B6C9 B6C9 B6C9 B6C9 3649 F6C9 B6C9 5C40      E5B180       E5B180       F3B0A39E
8BC8     8BC8 8BC8 8BC8 8BC8 49B6 B6CA B6CA B6CA B6CA 364A F6D1 B6CA 66F2      E69BB2       E69BB2       F3B0A39F
8BC9     8BC9 8BC9 8BC9 8BC9 49B7 B6CB B6CB B6CB B6CB 364B F6D2 B6CB 6975      E6A5B5       E6A5B5       F3B0A3A0
8BCA     8BCA 8BCA 8BCA 8BCA 45FD B6CC B6CC B6CC B6CC 364C F6D3 B6CC 7389      E78E89       E78E89       F3B0A3A1
8BCB     8BCB 8BCB 8BCB 8BCB 49C0 B6CD B6CD B6CD B6CD 364D F6D4 B6CD 6850      E6A190       E6A190       F3B0A3A2
8BCC     8BCC 8BCC 8BCC 8BCC 55E7 B6CE B6CE B6CE B6CE 364E F6D5 B6CE 7C81      E7B281       E7B281       F3B0A3A3
8BCD     8BCD 8BCD 8BCD 8BCD 4FC9 B6CF B6CF B6CF B6CF 364F F6D6 B6CF 50C5      E58385       E58385       F3B0A3A4
8BCE     8BCE 8BCE 8BCE 8BCE 499F B6D0 B6D0 B6D0 B6D0 3650 F6D7 B6D0 52E4      E58BA4       E58BA4       F3B0A3A5
8BCF     8BCF 8BCF 8BCF 8BCF 49AE B6D1 B6D1 B6D1 B6D1 3651 F6D8 B6D1 5747      E59D87       E59D87       F3B0A3A6
8BD0     8BD0 8BD0 8BD0 8BD0 4A7D B6D2 B6D2 B6D2 B6D2 3652 F6D9 B6D2 5DFE      E5B7BE       E5B7BE       F3B0A3A7
8BD1     8BD1 8BD1 8BD1 8BD1 4B8E B6D3 B6D3 B6D3 B6D3 3653 F6E2 B6D3 9326      E98CA6       E98CA6       F3B0A3A8
8BD2     8BD2 8BD2 8BD2 8BD2 54CF B6D4 B6D4 B6D4 B6D4 3654 F6E3 B6D4 65A4      E696A4       E696A4       F3B0A3A9
8BD3     8BD3 8BD3 8BD3 8BD3 4B9D B6D5 B6D5 B6D5 B6D5 3655 F6E4 B6D5 6B23      E6ACA3       E6ACA3       F3B0A3AA
8BD4     8BD4 8BD4 8BD4 8BD4 4CF8 B6D6 B6D6 B6D6 B6D6 3656 F6E5 B6D6 6B3D      E6ACBD       E6ACBD       F3B0A3AB
8BD5     8BD5 8BD5 8BD5 8BD5 4CD8 B6D7 B6D7 B6D7 B6D7 3657 F6E6 B6D7 7434      E790B4       E790B4       F3B0A3AC
8BD6     8BD6 8BD6 8BD6 8BD6 4D75 B6D8 B6D8 B6D8 B6D8 3658 F6E7 B6D8 7981      E7A681       E7A681       F3B0A3AD
8BD7     8BD7 8BD7 8BD7 8BD7 555D B6D9 B6D9 B6D9 B6D9 3659 F6E8 B6D9 79BD      E7A6BD       E7A6BD       F3B0A3AE
8BD8     8BD8 8BD8 8BD8 8BD8 4B8F B6DA B6DA B6DA B6DA 365A F6E9 B6DA 7B4B      E7AD8B       E7AD8B       F3B0A3AF
8BD9     8BD9 8BD9 8BD9 8BD9 4B97 B6DB B6DB B6DB B6DB 365B F64A B6DB 7DCA      E7B78A       E7B78A       F3B0A3B0
8BDA     8BDA 8BDA 8BDA 8BDA 4DD0 B6DC B6DC B6DC B6DC 365C F65B B6DC 82B9      E88AB9       E88AB9       F3B0A3B1
8BDB     8BDB 8BDB 8BDB 8BDB 4C73 B6DD B6DD B6DD B6DD 365D F65A B6DD 83CC      E88F8C       E88F8C       F3B0A3B2
8BDC     8BDC 8BDC 8BDC 8BDC 4AA7 B6DE B6DE B6DE B6DE 365E F65F B6DE 887F      E8A1BF       E8A1BF       F3B0A3B3
8BDD     8BDD 8BDD 8BDD 8BDD 54AF B6DF B6DF B6DF B6DF 365F F66D B6DF 895F      E8A59F       E8A59F       F3B0A3B4
8BDE     8BDE 8BDE 8BDE 8BDE 4D87 B6E0 B6E0 B6E0 B6E0 3660 F679 B6E0 8B39      E8ACB9       E8ACB9       F3B0A3B5
8BDF     8BDF 8BDF 8BDF 8BDF 4673 B6E1 B6E1 B6E1 B6E1 3661 F657 B6E1 8FD1      E8BF91       E8BF91       F3B0A3B6
8BE0     8BE0 8BE0 8BE0 8BE0 4586 B6E2 B6E2 B6E2 B6E2 3662 F659 B6E2 91D1      E98791       E98791       F3B0A3B7
8BE1     8BE1 8BE1 8BE1 8BE1 507A B6E3 B6E3 B6E3 B6E3 3663 F662 B6E3 541F      E5909F       E5909F       F3B0A3B8
8BE2     8BE2 8BE2 8BE2 8BE2 478C B6E4 B6E4 B6E4 B6E4 3664 F663 B6E4 9280      E98A80       E98A80       F3B0A3B9
8BE3     8BE3 8BE3 8BE3 8BE3 4549 B6E5 B6E5 B6E5 B6E5 3665 F664 B6E5 4E5D      E4B99D       E4B99D       F3B0A3BA
8BE4     8BE4 8BE4 8BE4 8BE4 4F63 B6E6 B6E6 B6E6 B6E6 3666 F665 B6E6 5036      E580B6       E580B6       F3B0A3BB
8BE5     8BE5 8BE5 8BE5 8BE5 4C8B B6E7 B6E7 B6E7 B6E7 3667 F666 B6E7 53E5      E58FA5       E58FA5       F3B0A3BC
8BE6     8BE6 8BE6 8BE6 8BE6 4554 B6E8 B6E8 B6E8 B6E8 3668 F667 B6E8 533A      E58CBA       E58CBA       F3B0A3BD
8BE7     8BE7 8BE7 8BE7 8BE7 5274 B6E9 B6E9 B6E9 B6E9 3669 F668 B6E9 72D7      E78B97       E78B97       F3B0A3BE
8BE8     8BE8 8BE8 8BE8 8BE8 4DE9 B6EA B6EA B6EA B6EA 366A F669 B6EA 7396      E78E96       E78E96       F3B0A3BF
8BE9     8BE9 8BE9 8BE9 8BE9 4E4D B6EB B6EB B6EB B6EB 366B F670 B6EB 77E9      E79FA9       E79FA9       F3B0A480
8BEA     8BEA 8BEA 8BEA 8BEA 4A5F B6EC B6EC B6EC B6EC 366C F671 B6EC 82E6      E88BA6       E88BA6       F3B0A481
8BEB     8BEB 8BEB 8BEB 8BEB 5353 B6ED B6ED B6ED 66EA CFDC A4BC B6ED 8EAF      E8BAAF       E8BAAF       F3B0A482
8BEC     8BEC 8BEC 8BEC 8BEC 4CFE B6EE B6EE B6EE B6EE 366E F673 B6EE 99C6      E9A786       E9A786       F3B0A483
8BED     8BED 8BED 8BED 8BED 4F85 B6EF B6EF B6EF B6EF 366F F674 B6EF 99C8      E9A788       E9A788       F3B0A484
8BEE     8BEE 8BEE 8BEE 8BEE 49FC B6F0 B6F0 B6F0 B6F0 3670 F675 B6F0 99D2      E9A792       E9A792       F3B0A485
8BEF     8BEF 8BEF 8BEF 8BEF 47AF B6F1 B6F1 B6F1 B6F1 3671 F676 B6F1 5177      E585B7       E585B7       F3B0A486
8BF0     8BF0 8BF0 8BF0 8BF0 52B2 B6F2 B6F2 B6F2 B6F2 3672 F677 B6F2 611A      E6849A       E6849A       F3B0A487
8BF1     8BF1 8BF1 8BF1 8BF1 5588 B6F3 B6F3 B6F3 B6F3 3673 F678 B6F3 865E      E8999E       E8999E       F3B0A488
8BF2     8BF2 8BF2 8BF2 8BF2 4EB9 B6F4 B6F4 B6F4 B6F4 3674 F680 B6F4 55B0      E596B0       E596B0       F3B0A489
8BF3     8BF3 8BF3 8BF3 8BF3 478D B6F5 B6F5 B6F5 B6F5 3675 F68B B6F5 7A7A      E7A9BA       E7A9BA       F3B0A48A
8BF4     8BF4 8BF4 8BF4 8BF4 50C3 B6F6 B6F6 B6F6 B6F6 3676 F69B B6F6 5076      E581B6       E581B6       F3B0A48B
8BF5     8BF5 8BF5 8BF5 8BF5 55F4 B6F7 B6F7 B6F7 B6F7 3677 F69C B6F7 5BD3      E5AF93       E5AF93       F3B0A48C
8BF6     8BF6 8BF6 8BF6 8BF6 49EC B6F8 B6F8 B6F8 B6F8 3678 F6A0 B6F8 9047      E98187       E98187       F3B0A48D
8BF7     8BF7 8BF7 8BF7 8BF7 4BB9 B6F9 B6F9 B6F9 B6F9 3679 F6AB B6F9 9685      E99A85       E99A85       F3B0A48E
8BF8     8BF8 8BF8 8BF8 8BF8 4CB8 B6FA B6FA B6FA B6FA 367A F6B0 B6FA 4E32      E4B8B2       E4B8B2       F3B0A48F
8BF9     8BF9 8BF9 8BF9 8BF9 4DA6 B6FB B6FB B6FB B6FB 367B F6C0 B6FB 6ADB      E6AB9B       E6AB9B       F3B0A490
8BFA     8BFA 8BFA 8BFA 8BFA 4BF0 B6FC B6FC B6FC B6FC 367C F66A B6FC 91E7      E987A7       E987A7       F3B0A491
8BFB     8BFB 8BFB 8BFB 8BFB 54B2 B6FD B6FD B6FD B6FD 367D F6D0 B6FD 5C51      E5B191       E5B191       F3B0A492
8BFC     8BFC 8BFC 8BFC 8BFC 4D94 B6FE B6FE B6FE B6FE 367E F6A1 B6FE 5C48      E5B188       E5B188       F3B0A493
8C40     8C40 8C40 8C40 8C40 4D44 B7A1 B7A1 B7A1 B7A1 3721 F74F B7A1 6398      E68E98       E68E98       F3B0A494
8C41     8C41 8C41 8C41 8C41 50A6 B7A2 B7A2 B7A2 B7A2 3722 F77F B7A2 7A9F      E7AA9F       E7AA9F       F3B0A495
8C42     8C42 8C42 8C42 8C42 4FF8 B7A3 B7A3 B7A3 B7A3 3723 F77B B7A3 6C93      E6B293       E6B293       F3B0A496
8C43     8C43 8C43 8C43 8C43 4DF1 B7A4 B7A4 B7A4 B7A4 3724 F7E0 B7A4 9774      E99DB4       E99DB4       F3B0A497
8C44     8C44 8C44 8C44 8C44 52E1 B7A5 B7A5 B7A5 B7A5 3725 F76C B7A5 8F61      E8BDA1       E8BDA1       F3B0A498
8C45     8C45 8C45 8C45 8C45 4A65 B7A6 B7A6 B7A6 B7A6 3726 F750 B7A6 7AAA      E7AAAA       E7AAAA       F3B0A499
8C46     8C46 8C46 8C46 8C46 467F B7A7 B7A7 B7A7 B7A7 3727 F77D B7A7 718A      E7868A       E7868A       F3B0A49A
8C47     8C47 8C47 8C47 8C47 4CE5 B7A8 B7A8 B7A8 B7A8 3728 F74D B7A8 9688      E99A88       E99A88       F3B0A49B
8C48     8C48 8C48 8C48 8C48 4E85 B7A9 B7A9 B7A9 B7A9 3729 F75D B7A9 7C82      E7B282       E7B282       F3B0A49C
8C49     8C49 8C49 8C49 8C49 479B B7AA B7AA B7AA B7AA 372A F75C B7AA 6817      E6A097       E6A097       F3B0A49D
8C4A     8C4A 8C4A 8C4A 8C4A 4B69 B7AB B7AB B7AB B7AB 372B F74E B7AB 7E70      E7B9B0       E7B9B0       F3B0A49E
8C4B     8C4B 8C4B 8C4B 8C4B 4879 B7AC B7AC B7AC B7AC 372C F76B B7AC 6851      E6A191       E6A191       F3B0A49F
8C4C     8C4C 8C4C 8C4C 8C4C 4F70 B7AD B7AD B7AD B7AD 372D F760 B7AD 936C      E98DAC       E98DAC       F3B0A4A0
8C4D     8C4D 8C4D 8C4D 8C4D 497A B7AE B7AE B7AE B7AE 372E F74B B7AE 52F2      E58BB2       E58BB2       F3B0A4A1
8C4E     8C4E 8C4E 8C4E 8C4E 47CF B7AF B7AF B7AF B7AF 372F F761 B7AF 541B      E5909B       E5909B       F3B0A4A2
8C4F     8C4F 8C4F 8C4F 8C4F 4998 B7B0 B7B0 B7B0 B7B0 3730 F7F0 B7B0 85AB      E896AB       E896AB       F3B0A4A3
8C50     8C50 8C50 8C50 8C50 4BC8 B7B1 B7B1 B7B1 B7B1 3731 F7F1 B7B1 8A13      E8A893       E8A893       F3B0A4A4
8C51     8C51 8C51 8C51 8C51 47B1 B7B2 B7B2 B7B2 B7B2 3732 F7F2 B7B2 7FA4      E7BEA4       E7BEA4       F3B0A4A5
8C52     8C52 8C52 8C52 8C52 4897 B7B3 B7B3 B7B3 B7B3 3733 F7F3 B7B3 8ECD      E8BB8D       E8BB8D       F3B0A4A6
8C53     8C53 8C53 8C53 8C53 499E B7B4 B7B4 B7B4 B7B4 3734 F7F4 B7B4 90E1      E983A1       E983A1       F3B0A4A7
8C54     8C54 8C54 8C54 8C54 53DB B7B5 B7B5 B7B5 B7B5 3735 F7F5 B7B5 5366      E58DA6       E58DA6       F3B0A4A8
8C55     8C55 8C55 8C55 8C55 4F6A B7B6 B7B6 B7B6 B7B6 3736 F7F6 B7B6 8888      E8A288       E8A288       F3B0A4A9
8C56     8C56 8C56 8C56 8C56 53D2 B7B7 B7B7 B7B7 B7B7 3737 F7F7 B7B7 7941      E7A581       E7A581       F3B0A4AA
8C57     8C57 8C57 8C57 8C57 4851 B7B8 B7B8 B7B8 B7B8 3738 F7F8 B7B8 4FC2      E4BF82       E4BF82       F3B0A4AB
8C58     8C58 8C58 8C58 8C58 4A43 B7B9 B7B9 B7B9 B7B9 3739 F7F9 B7B9 50BE      E582BE       E582BE       F3B0A4AC
8C59     8C59 8C59 8C59 8C59 4C52 B7BA B7BA B7BA B7BA 373A F77A B7BA 5211      E58891       E58891       F3B0A4AD
8C5A     8C5A 8C5A 8C5A 8C5A 4BE8 B7BB B7BB B7BB B7BB 373B F75E B7BB 5144      E58584       E58584       F3B0A4AE
8C5B     8C5B 8C5B 8C5B 8C5B 47E4 B7BC B7BC B7BC B7BC 373C F74C B7BC 5553      E59593       E59593       F3B0A4AF
8C5C     8C5C 8C5C 8C5C 8C5C 49E6 B7BD B7BD B7BD B7BD 373D F77E B7BD 572D      E59CAD       E59CAD       F3B0A4B0
8C5D     8C5D 8C5D 8C5D 8C5D 4CB9 B7BE B7BE B7BE B7BE 373E F76E B7BE 73EA      E78FAA       E78FAA       F3B0A4B1
8C5E     8C5E 8C5E 8C5E 8C5E 46E6 B7BF B7BF B7BF B7BF 373F F76F B7BF 578B      E59E8B       E59E8B       F3B0A4B2
8C5F     8C5F 8C5F 8C5F 8C5F 4C64 B7C0 B7C0 B7C0 B7C0 3740 F77C B7C0 5951      E5A591       E5A591       F3B0A4B3
8C60     8C60 8C60 8C60 8C60 4690 B7C1 B7C1 B7C1 B7C1 3741 F7C1 B7C1 5F62      E5BDA2       E5BDA2       F3B0A4B4
8C61     8C61 8C61 8C61 8C61 4E60 B7C2 B7C2 B7C2 B7C2 3742 F7C2 B7C2 5F84      E5BE84       E5BE84       F3B0A4B5
8C62     8C62 8C62 8C62 8C62 4643 B7C3 B7C3 B7C3 B7C3 3743 F7C3 B7C3 6075      E681B5       E681B5       F3B0A4B6
8C63     8C63 8C63 8C63 8C63 48A8 B7C4 B7C4 B7C4 B7C4 3744 F7C4 B7C4 6176      E685B6       E685B6       F3B0A4B7
8C64     8C64 8C64 8C64 8C64 4EB6 B7C5 B7C5 B7C5 B7C5 3745 F7C5 B7C5 6167      E685A7       E685A7       F3B0A4B8
8C65     8C65 8C65 8C65 8C65 5175 B7C6 B7C6 B7C6 B7C6 3746 F7C6 B7C6 61A9      E686A9       E686A9       F3B0A4B9
8C66     8C66 8C66 8C66 8C66 4DC1 B7C7 B7C7 B7C7 B7C7 3747 F7C7 B7C7 63B2      E68EB2       E68EB2       F3B0A4BA
8C67     8C67 8C67 8C67 8C67 4E49 B7C8 B7C8 B7C8 B7C8 3748 F7C8 B7C8 643A      E690BA       E690BA       F3B0A4BB
8C68     8C68 8C68 8C68 8C68 47B0 B7C9 B7C9 B7C9 B7C9 3749 F7C9 B7C9 656C      E695AC       E695AC       F3B0A4BC
8C69     8C69 8C69 8C69 8C69 4993 B7CA B7CA B7CA B7CA 374A F7D1 B7CA 666F      E699AF       E699AF       F3B0A4BD
8C6A     8C6A 8C6A 8C6A 8C6A 49AB B7CB B7CB B7CB B7CB 374B F7D2 B7CB 6842      E6A182       E6A182       F3B0A4BE
8C6B     8C6B 8C6B 8C6B 8C6B 5198 B7CC B7CC B7CC B7CC 374C F7D3 B7CC 6E13      E6B893       E6B893       F3B0A4BF
8C6C     8C6C 8C6C 8C6C 8C6C 51B4 B7CD B7CD B7CD B7CD 374D F7D4 B7CD 7566      E795A6       E795A6       F3B0A580
8C6D     8C6D 8C6D 8C6D 8C6D 4FC7 B7CE B7CE B7CE B7CE 374E F7D5 B7CE 7A3D      E7A8BD       E7A8BD       F3B0A581
8C6E     8C6E 8C6E 8C6E 8C6E 48FA B7CF B7CF B7CF B7CF 374F F7D6 B7CF 7CFB      E7B3BB       E7B3BB       F3B0A582
8C6F     8C6F 8C6F 8C6F 8C6F 4688 B7D0 B7D0 B7D0 B7D0 3750 F7D7 B7D0 7D4C      E7B58C       E7B58C       F3B0A583
8C70     8C70 8C70 8C70 8C70 4A4A B7D1 B7D1 B7D1 B7D1 3751 F7D8 B7D1 7D99      E7B699       E7B699       F3B0A584
8C71     8C71 8C71 8C71 8C71 5373 B7D2 B7D2 B7D2 5FDF D6B6 AC86 B7D2 7E4B      E7B98B       E7B98B       F3B0A585
8C72     8C72 8C72 8C72 8C72 55EF B7D3 B7D3 B7D3 B7D3 3753 F7E2 B7D3 7F6B      E7BDAB       E7BDAB       F3B0A586
8C73     8C73 8C73 8C73 8C73 4EC9 B7D4 B7D4 B7D4 B7D4 3754 F7E3 B7D4 830E      E88C8E       E88C8E       F3B0A587
8C74     8C74 8C74 8C74 8C74 53B3 B7D5 B7D5 B7D5 B7D5 3755 F7E4 B7D5 834A      E88D8A       E88D8A       F3B0A588
8C75     8C75 8C75 8C75 8C75 4DF9 B7D6 B7D6 B7D6 B7D6 3756 F7E5 B7D6 86CD      E89B8D       E89B8D       F3B0A589
8C76     8C76 8C76 8C76 8C76 45C9 B7D7 B7D7 B7D7 B7D7 3757 F7E6 B7D7 8A08      E8A888       E8A888       F3B0A58A
8C77     8C77 8C77 8C77 8C77 54FB B7D8 B7D8 B7D8 B7D8 3758 F7E7 B7D8 8A63      E8A9A3       E8A9A3       F3B0A58B
8C78     8C78 8C78 8C78 8C78 4AD7 B7D9 B7D9 B7D9 B7D9 3759 F7E8 B7D9 8B66      E8ADA6       E8ADA6       F3B0A58C
8C79     8C79 8C79 8C79 8C79 48A6 B7DA B7DA B7DA B7DA 375A F7E9 B7DA 8EFD      E8BBBD       E8BBBD       F3B0A58D
8C7A     8C7A 8C7A 8C7A 8C7A 66C4 F0F4 B7DB B7DB F0F4 7074 7580 B7DB 981A      E9A09A       E9A09A       F3B0A58E
8C7B     8C7B 8C7B 8C7B 8C7B 4D8E B7DC B7DC B7DC B7DC 375C F75B B7DC 9D8F      E9B68F       E9B68F       F3B0A58F
8C7C     8C7C 8C7C 8C7C 8C7C 4953 B7DD B7DD B7DD B7DD 375D F75A B7DD 82B8      E88AB8       E88AB8       F3B0A590
8C7D     8C7D 8C7D 8C7D 8C7D 4BA9 B7DE B7DE B7DE B7DE 375E F75F B7DE 8FCE      E8BF8E       E8BF8E       F3B0A591
8C7E     8C7E 8C7E 8C7E 8C7E 4D78 B7DF B7DF B7DF B7DF 375F F76D B7DF 9BE8      E9AFA8       E9AFA8       F3B0A592
8C80     8C80 8C80 8C80 8C80 49F0 B7E0 B7E0 B7E0 B7E0 3760 F779 B7E0 5287      E58A87       E58A87       F3B0A593
8C81     8C81 8C81 8C81 8C81 54D9 B7E1 B7E1 B7E1 B7E1 3761 F757 B7E1 621F      E6889F       E6889F       F3B0A594
8C82     8C82 8C82 8C82 8C82 4A74 B7E2 B7E2 B7E2 B7E2 3762 F759 B7E2 6483      E69283       E69283       F3B0A595
8C83     8C83 8C83 8C83 8C83 4BD0 B7E3 B7E3 B7E3 B7E3 3763 F762 B7E3 6FC0      E6BF80       E6BF80       F3B0A596
8C84     8C84 8C84 8C84 8C84 53C3 B7E4 B7E4 B7E4 B7E4 3764 F763 B7E4 9699      E99A99       E99A99       F3B0A597
8C85     8C85 8C85 8C85 8C85 50D6 B7E5 B7E5 B7E5 B7E5 3765 F764 B7E5 6841      E6A181       E6A181       F3B0A598
8C86     8C86 8C86 8C86 8C86 4FC3 B7E6 B7E6 B7E6 B7E6 3766 F765 B7E6 5091      E58291       E58291       F3B0A599
8C87     8C87 8C87 8C87 8C87 4CB5 B7E7 B7E7 B7E7 B7E7 3767 F766 B7E7 6B20      E6ACA0       E6ACA0       F3B0A59A
8C88     8C88 8C88 8C88 8C88 47A4 B7E8 B7E8 B7E8 B7E8 3768 F767 B7E8 6C7A      E6B1BA       E6B1BA       F3B0A59B
8C89     8C89 8C89 8C89 8C89 4AC2 B7E9 B7E9 B7E9 B7E9 3769 F768 B7E9 6F54      E6BD94       E6BD94       F3B0A59C
8C8A     8C8A 8C8A 8C8A 8C8A 4B86 B7EA B7EA B7EA B7EA 376A F769 B7EA 7A74      E7A9B4       E7A9B4       F3B0A59D
8C8B     8C8B 8C8B 8C8B 8C8B 46BF B7EB B7EB B7EB B7EB 376B F770 B7EB 7D50      E7B590       E7B590       F3B0A59E
8C8C     8C8C 8C8C 8C8C 8C8C 4B43 B7EC B7EC B7EC B7EC 376C F771 B7EC 8840      E8A180       E8A180       F3B0A59F
8C8D     8C8D 8C8D 8C8D 8C8D 54CC B7ED B7ED B7ED B7ED 376D F772 B7ED 8A23      E8A8A3       E8A8A3       F3B0A5A0
8C8E     8C8E 8C8E 8C8E 8C8E 4561 B7EE B7EE B7EE B7EE 376E F773 B7EE 6708      E69C88       E69C88       F3B0A5A1
8C8F     8C8F 8C8F 8C8F 8C8F 48B2 B7EF B7EF B7EF B7EF 376F F774 B7EF 4EF6      E4BBB6       E4BBB6       F3B0A5A2
8C90     8C90 8C90 8C90 8C90 52FA B7F0 B7F0 B7F0 B7F0 3770 F775 B7F0 5039      E580B9       E580B9       F3B0A5A3
8C91     8C91 8C91 8C91 8C91 52F1 B7F1 B7F1 B7F1 B7F1 3771 F776 B7F1 5026      E580A6       E580A6       F3B0A5A4
8C92     8C92 8C92 8C92 8C92 4681 B7F2 B7F2 B7F2 B7F2 3772 F777 B7F2 5065      E581A5       E581A5       F3B0A5A5
8C93     8C93 8C93 8C93 8C93 48F4 B7F3 B7F3 B7F3 B7F3 3773 F778 B7F3 517C      E585BC       E585BC       F3B0A5A6
8C94     8C94 8C94 8C94 8C94 4B6B B7F4 B7F4 B7F4 B7F4 3774 F780 B7F4 5238      E588B8       E588B8       F3B0A5A7
8C95     8C95 8C95 8C95 8C95 4CD3 B7F5 B7F5 B7F5 B7F5 3775 F78B B7F5 5263      E589A3       E589A3       F3B0A5A8
8C96     8C96 8C96 8C96 8C96 4FD3 B7F6 B7F6 B7F6 B7F6 3776 F79B B7F6 55A7      E596A7       E596A7       F3B0A5A9
8C97     8C97 8C97 8C97 8C97 4F49 B7F7 B7F7 B7F7 B7F7 3777 F79C B7F7 570F      E59C8F       E59C8F       F3B0A5AA
8C98     8C98 8C98 8C98 8C98 4AE7 B7F8 B7F8 B7F8 B7F8 3778 F7A0 B7F8 5805      E5A085       E5A085       F3B0A5AB
8C99     8C99 8C99 8C99 8C99 4EC3 B7F9 B7F9 B7F9 B7F9 3779 F7AB B7F9 5ACC      E5AB8C       E5AB8C       F3B0A5AC
8C9A     8C9A 8C9A 8C9A 8C9A 4666 B7FA B7FA B7FA B7FA 377A F7B0 B7FA 5EFA      E5BBBA       E5BBBA       F3B0A5AD
8C9B     8C9B 8C9B 8C9B 8C9B 47BE B7FB B7FB B7FB B7FB 377B F7C0 B7FB 61B2      E686B2       E686B2       F3B0A5AE
8C9C     8C9C 8C9C 8C9C 8C9C 4DC6 B7FC B7FC B7FC B7FC 377C F76A B7FC 61F8      E687B8       E687B8       F3B0A5AF
8C9D     8C9D 8C9D 8C9D 8C9D 508D B7FD B7FD B7FD B7FD 377D F7D0 B7FD 62F3      E68BB3       E68BB3       F3B0A5B0
8C9E     8C9E 8C9E 8C9E 8C9E 5493 B7FE B7FE B7FE B7FE 377E F7A1 B7FE 6372      E68DB2       E68DB2       F3B0A5B1
8C9F     8C9F 8C9F 8C9F 8C9F 486D B8A1 B8A1 B8A1 B8A1 3821 F84F B8A1 691C      E6A49C       E6A49C       F3B0A5B2
8CA0     8CA0 8CA0 8CA0 8CA0 48DC B8A2 B8A2 B8A2 B8A2 3822 F87F B8A2 6A29      E6A8A9       E6A8A9       F3B0A5B3
8CA1     8CA1 8CA1 8CA1 8CA1 548A B8A3 B8A3 B8A3 B8A3 3823 F87B B8A3 727D      E789BD       E789BD       F3B0A5B4
8CA2     8CA2 8CA2 8CA2 8CA2 4AC5 B8A4 B8A4 B8A4 B8A4 3824 F8E0 B8A4 72AC      E78AAC       E78AAC       F3B0A5B5
8CA3     8CA3 8CA3 8CA3 8CA3 4F45 B8A5 B8A5 B8A5 B8A5 3825 F86C B8A5 732E      E78CAE       E78CAE       F3B0A5B6
8CA4     8CA4 8CA4 8CA4 8CA4 46D3 B8A6 B8A6 B8A6 B8A6 3826 F850 B8A6 7814      E7A094       E7A094       F3B0A5B7
8CA5     8CA5 8CA5 8CA5 8CA5 5369 B8A7 B8A7 B8A7 B8A7 3827 F87D B8A7 786F      E7A1AF       E7A1AF       F3B0A5B8
8CA6     8CA6 8CA6 8CA6 8CA6 49F6 B8A8 B8A8 B8A8 B8A8 3828 F84D B8A8 7D79      E7B5B9       E7B5B9       F3B0A5B9
8CA7     8CA7 8CA7 8CA7 8CA7 4552 B8A9 B8A9 B8A9 B8A9 3829 F85D B8A9 770C      E79C8C       E79C8C       F3B0A5BA
8CA8     8CA8 8CA8 8CA8 8CA8 4B79 B8AA B8AA B8AA B8AA 382A F85C B8AA 80A9      E882A9       E882A9       F3B0A5BB
8CA9     8CA9 8CA9 8CA9 8CA9 45BB B8AB B8AB B8AB B8AB 382B F84E B8AB 898B      E8A68B       E8A68B       F3B0A5BC
8CAA     8CAA 8CAA 8CAA 8CAA 497C B8AC B8AC B8AC B8AC 382C F86B B8AC 8B19      E8AC99       E8AC99       F3B0A5BD
8CAB     8CAB 8CAB 8CAB 8CAB 47E5 B8AD B8AD B8AD B8AD 382D F860 B8AD 8CE2      E8B3A2       E8B3A2       F3B0A5BE
8CAC     8CAC 8CAC 8CAC 8CAC 4E55 B8AE B8AE B8AE B8AE 382E F84B B8AE 8ED2      E8BB92       E8BB92       F3B0A5BF
8CAD     8CAD 8CAD 8CAD 8CAD 4EC2 B8AF B8AF B8AF B8AF 382F F861 B8AF 9063      E981A3       E981A3       F3B0A680
8CAE     8CAE 8CAE 8CAE 8CAE 4E45 B8B0 B8B0 B8B0 B8B0 3830 F8F0 B8B0 9375      E98DB5       E98DB5       F3B0A681
8CAF     8CAF 8CAF 8CAF 8CAF 4A6C B8B1 B8B1 B8B1 B8B1 3831 F8F1 B8B1 967A      E999BA       E999BA       F3B0A682
8CB0     8CB0 8CB0 8CB0 8CB0 4AE5 B8B2 B8B2 B8B2 B8B2 3832 F8F2 B8B2 9855      E9A195       E9A195       F3B0A683
8CB1     8CB1 8CB1 8CB1 8CB1 48B4 B8B3 B8B3 B8B3 B8B3 3833 F8F3 B8B3 9A13      E9A893       E9A893       F3B0A684
8CB2     8CB2 8CB2 8CB2 8CB2 4C7D B8B4 B8B4 B8B4 6FB5 D5DF AABF B8B4 9E78      E9B9B8       E9B9B8       F3B0A685
8CB3     8CB3 8CB3 8CB3 8CB3 4695 B8B5 B8B5 B8B5 B8B5 3835 F8F5 B8B5 5143      E58583       E58583       F3B0A686
8CB4     8CB4 8CB4 8CB4 8CB4 4574 B8B6 B8B6 B8B6 B8B6 3836 F8F6 B8B6 539F      E58E9F       E58E9F       F3B0A687
8CB5     8CB5 8CB5 8CB5 8CB5 4C8E B8B7 B8B7 B8B7 B8B7 3837 F8F7 B8B7 53B3      E58EB3       E58EB3       F3B0A688
8CB6     8CB6 8CB6 8CB6 8CB6 50A2 B8B8 B8B8 B8B8 B8B8 3838 F8F8 B8B8 5E7B      E5B9BB       E5B9BB       F3B0A689
8CB7     8CB7 8CB7 8CB7 8CB7 4E42 B8B9 B8B9 B8B9 B8B9 3839 F8F9 B8B9 5F26      E5BCA6       E5BCA6       F3B0A68A
8CB8     8CB8 8CB8 8CB8 8CB8 478A B8BA B8BA B8BA B8BA 383A F87A B8BA 6E1B      E6B89B       E6B89B       F3B0A68B
8CB9     8CB9 8CB9 8CB9 8CB9 47FC B8BB B8BB B8BB B8BB 383B F85E B8BB 6E90      E6BA90       E6BA90       F3B0A68C
8CBA     8CBA 8CBA 8CBA 8CBA 4C6E B8BC B8BC B8BC B8BC 383C F84C B8BC 7384      E78E84       E78E84       F3B0A68D
8CBB     8CBB 8CBB 8CBB 8CBB 46B7 B8BD B8BD B8BD B8BD 383D F87E B8BD 73FE      E78FBE       E78FBE       F3B0A68E
8CBC     8CBC 8CBC 8CBC 8CBC 5563 B8BE B8BE B8BE B8BE 383E F86E B8BE 7D43      E7B583       E7B583       F3B0A68F
8CBD     8CBD 8CBD 8CBD 8CBD 53AC B8BF B8BF B8BF B8BF 383F F86F B8BF 8237      E888B7       E888B7       F3B0A690
8CBE     8CBE 8CBE 8CBE 8CBE 479A B8C0 B8C0 B8C0 B8C0 3840 F87C B8C0 8A00      E8A880       E8A880       F3B0A691
8CBF     8CBF 8CBF 8CBF 8CBF 55AB B8C1 B8C1 B8C1 B8C1 3841 F8C1 B8C1 8AFA      E8ABBA       E8ABBA       F3B0A692
8CC0     8CC0 8CC0 8CC0 8CC0 48AD B8C2 B8C2 B8C2 B8C2 3842 F8C2 B8C2 9650      E99990       E99990       F3B0A693
8CC1     8CC1 8CC1 8CC1 8CC1 54A5 B8C3 B8C3 B8C3 B8C3 3843 F8C3 B8C3 4E4E      E4B98E       E4B98E       F3B0A694
8CC2     8CC2 8CC2 8CC2 8CC2 49FA B8C4 B8C4 B8C4 B8C4 3844 F8C4 B8C4 500B      E5808B       E5808B       F3B0A695
8CC3     8CC3 8CC3 8CC3 8CC3 45D4 B8C5 B8C5 B8C5 B8C5 3845 F8C5 B8C5 53E4      E58FA4       E58FA4       F3B0A696
8CC4     8CC4 8CC4 8CC4 8CC4 4AF6 B8C6 B8C6 B8C6 B8C6 3846 F8C6 B8C6 547C      E591BC       E591BC       F3B0A697
8CC5     8CC5 8CC5 8CC5 8CC5 4A94 B8C7 B8C7 B8C7 B8C7 3847 F8C7 B8C7 56FA      E59BBA       E59BBA       F3B0A698
8CC6     8CC6 8CC6 8CC6 8CC6 53F2 B8C8 B8C8 B8C8 B8C8 3848 F8C8 B8C8 59D1      E5A791       E5A791       F3B0A699
8CC7     8CC7 8CC7 8CC7 8CC7 4EAF B8C9 B8C9 B8C9 B8C9 3849 F8C9 B8C9 5B64      E5ADA4       E5ADA4       F3B0A69A
8CC8     8CC8 8CC8 8CC8 8CC8 47AD B8CA B8CA B8CA B8CA 384A F8D1 B8CA 5DF1      E5B7B1       E5B7B1       F3B0A69B
8CC9     8CC9 8CC9 8CC9 8CC9 45F8 B8CB B8CB B8CB B8CB 384B F8D2 B8CB 5EAB      E5BAAB       E5BAAB       F3B0A69C
8CCA     8CCA 8CCA 8CCA 8CCA 536B B8CC B8CC B8CC B8CC 384C F8D3 B8CC 5F27      E5BCA7       E5BCA7       F3B0A69D
8CCB     8CCB 8CCB 8CCB 8CCB 45CF B8CD B8CD B8CD B8CD 384D F8D4 B8CD 6238      E688B8       E688B8       F3B0A69E
8CCC     8CCC 8CCC 8CCC 8CCC 4AF7 B8CE B8CE B8CE B8CE 384E F8D5 B8CE 6545      E69585       E69585       F3B0A69F
8CCD     8CCD 8CCD 8CCD 8CCD 5061 B8CF B8CF B8CF B8CF 384F F8D6 B8CF 67AF      E69EAF       E69EAF       F3B0A6A0
8CCE     8CCE 8CCE 8CCE 8CCE 4D52 B8D0 B8D0 B8D0 B8D0 3850 F8D7 B8D0 6E56      E6B996       E6B996       F3B0A6A1
8CCF     8CCF 8CCF 8CCF 8CCF 51DB B8D1 B8D1 B8D1 B8D1 3851 F8D8 B8D1 72D0      E78B90       E78B90       F3B0A6A2
8CD0     8CD0 8CD0 8CD0 8CD0 5471 B8D2 B8D2 B8D2 B8D2 3852 F8D9 B8D2 7CCA      E7B38A       E7B38A       F3B0A6A3
8CD1     8CD1 8CD1 8CD1 8CD1 4F5A B8D3 B8D3 B8D3 B8D3 3853 F8E2 B8D3 88B4      E8A2B4       E8A2B4       F3B0A6A4
8CD2     8CD2 8CD2 8CD2 8CD2 4CC5 B8D4 B8D4 B8D4 B8D4 3854 F8E3 B8D4 80A1      E882A1       E882A1       F3B0A6A5
8CD3     8CD3 8CD3 8CD3 8CD3 4D63 B8D5 B8D5 B8D5 B8D5 3855 F8E4 B8D5 80E1      E883A1       E883A1       F3B0A6A6
8CD4     8CD4 8CD4 8CD4 8CD4 516D B8D6 B8D6 B8D6 B8D6 3856 F8E5 B8D6 83F0      E88FB0       E88FB0       F3B0A6A7
8CD5     8CD5 8CD5 8CD5 8CD5 4AAB B8D7 B8D7 B8D7 B8D7 3857 F8E6 B8D7 864E      E8998E       E8998E       F3B0A6A8
8CD6     8CD6 8CD6 8CD6 8CD6 4EB4 B8D8 B8D8 B8D8 B8D8 3858 F8E7 B8D8 8A87      E8AA87       E8AA87       F3B0A6A9
8CD7     8CD7 8CD7 8CD7 8CD7 55CB B8D9 B8D9 B8D9 B8D9 3859 F8E8 B8D9 8DE8      E8B7A8       E8B7A8       F3B0A6AA
8CD8     8CD8 8CD8 8CD8 8CD8 55BF B8DA B8DA B8DA B8DA 385A F8E9 B8DA 9237      E988B7       E988B7       F3B0A6AB
8CD9     8CD9 8CD9 8CD9 8CD9 4C4D B8DB B8DB B8DB B8DB 385B F84A B8DB 96C7      E99B87       E99B87       F3B0A6AC
8CDA     8CDA 8CDA 8CDA 8CDA 4EEE B8DC B8DC B8DC B8DC 385C F85B B8DC 9867      E9A1A7       E9A1A7       F3B0A6AD
8CDB     8CDB 8CDB 8CDB 8CDB 4FAF B8DD B8DD B8DD B8DD 385D F85A B8DD 9F13      E9BC93       E9BC93       F3B0A6AE
8CDC     8CDC 8CDC 8CDC 8CDC 4545 B8DE B8DE B8DE B8DE 385E F85F B8DE 4E94      E4BA94       E4BA94       F3B0A6AF
8CDD     8CDD 8CDD 8CDD 8CDD 4B74 B8DF B8DF B8DF B8DF 385F F86D B8DF 4E92      E4BA92       E4BA92       F3B0A6B0
8CDE     8CDE 8CDE 8CDE 8CDE 4D99 B8E0 B8E0 B8E0 B8E0 3860 F879 B8E0 4F0D      E4BC8D       E4BC8D       F3B0A6B1
8CDF     8CDF 8CDF 8CDF 8CDF 45DF B8E1 B8E1 B8E1 B8E1 3861 F857 B8E1 5348      E58D88       E58D88       F3B0A6B2
8CE0     8CE0 8CE0 8CE0 8CE0 4AEE B8E2 B8E2 B8E2 B8E2 3862 F859 B8E2 5449      E59189       E59189       F3B0A6B3
8CE1     8CE1 8CE1 8CE1 8CE1 47F6 B8E3 B8E3 B8E3 B8E3 3863 F862 B8E3 543E      E590BE       E590BE       F3B0A6B4
8CE2     8CE2 8CE2 8CE2 8CE2 50B5 B8E4 B8E4 B8E4 B8E4 3864 F863 B8E4 5A2F      E5A8AF       E5A8AF       F3B0A6B5
8CE3     8CE3 8CE3 8CE3 8CE3 45E5 B8E5 B8E5 B8E5 B8E5 3865 F864 B8E5 5F8C      E5BE8C       E5BE8C       F3B0A6B6
8CE4     8CE4 8CE4 8CE4 8CE4 4698 B8E6 B8E6 B8E6 B8E6 3866 F865 B8E6 5FA1      E5BEA1       E5BEA1       F3B0A6B7
8CE5     8CE5 8CE5 8CE5 8CE5 4AB1 B8E7 B8E7 B8E7 B8E7 3867 F866 B8E7 609F      E6829F       E6829F       F3B0A6B8
8CE6     8CE6 8CE6 8CE6 8CE6 52D3 B8E8 B8E8 B8E8 B8E8 3868 F867 B8E8 68A7      E6A2A7       E6A2A7       F3B0A6B9
8CE7     8CE7 8CE7 8CE7 8CE7 55EE B8E9 B8E9 B8E9 B8E9 3869 F868 B8E9 6A8E      E6AA8E       E6AA8E       F3B0A6BA
8CE8     8CE8 8CE8 8CE8 8CE8 53F1 B8EA B8EA B8EA B8EA 386A F869 B8EA 745A      E7919A       E7919A       F3B0A6BB
8CE9     8CE9 8CE9 8CE9 8CE9 4F4E B8EB B8EB B8EB B8EB 386B F870 B8EB 7881      E7A281       E7A281       F3B0A6BC
8CEA     8CEA 8CEA 8CEA 8CEA 48E7 B8EC B8EC B8EC B8EC 386C F871 B8EC 8A9E      E8AA9E       E8AA9E       F3B0A6BD
8CEB     8CEB 8CEB 8CEB 8CEB 4D91 B8ED B8ED B8ED B8ED 386D F872 B8ED 8AA4      E8AAA4       E8AAA4       F3B0A6BE
8CEC     8CEC 8CEC 8CEC 8CEC 49DB B8EE B8EE B8EE B8EE 386E F873 B8EE 8B77      E8ADB7       E8ADB7       F3B0A6BF
8CED     8CED 8CED 8CED 8CED 5170 B8EF B8EF B8EF B8EF 386F F874 B8EF 9190      E98690       E98690       F3B0A780
8CEE     8CEE 8CEE 8CEE 8CEE 4EB5 B8F0 B8F0 B8F0 B8F0 3870 F875 B8F0 4E5E      E4B99E       E4B99E       F3B0A781
8CEF     8CEF 8CEF 8CEF 8CEF 4FC6 B8F1 B8F1 B8F1 B8F1 3871 F876 B8F1 9BC9      E9AF89       E9AF89       F3B0A782
8CF0     8CF0 8CF0 8CF0 8CF0 478E B8F2 B8F2 B8F2 B8F2 3872 F877 B8F2 4EA4      E4BAA4       E4BAA4       F3B0A783
8CF1     8CF1 8CF1 8CF1 8CF1 5462 B8F3 B8F3 B8F3 B8F3 3873 F878 B8F3 4F7C      E4BDBC       E4BDBC       F3B0A784
8CF2     8CF2 8CF2 8CF2 8CF2 53E3 B8F4 B8F4 B8F4 B8F4 3874 F880 B8F4 4FAF      E4BEAF       E4BEAF       F3B0A785
8CF3     8CF3 8CF3 8CF3 8CF3 4CAE B8F5 B8F5 B8F5 B8F5 3875 F88B B8F5 5019      E58099       E58099       F3B0A786
8CF4     8CF4 8CF4 8CF4 8CF4 5343 B8F6 B8F6 B8F6 B8F6 3876 F89B B8F6 5016      E58096       E58096       F3B0A787
8CF5     8CF5 8CF5 8CF5 8CF5 459E B8F7 B8F7 B8F7 B8F7 3877 F89C B8F7 5149      E58589       E58589       F3B0A788
8CF6     8CF6 8CF6 8CF6 8CF6 46BB B8F8 B8F8 B8F8 B8F8 3878 F8A0 B8F8 516C      E585AC       E585AC       F3B0A789
8CF7     8CF7 8CF7 8CF7 8CF7 4870 B8F9 B8F9 B8F9 B8F9 3879 F8AB B8F9 529F      E58A9F       E58A9F       F3B0A78A
8CF8     8CF8 8CF8 8CF8 8CF8 4978 B8FA B8FA B8FA B8FA 387A F8B0 B8FA 52B9      E58AB9       E58AB9       F3B0A78B
8CF9     8CF9 8CF9 8CF9 8CF9 51E9 B8FB B8FB B8FB B8FB 387B F8C0 B8FB 52FE      E58BBE       E58BBE       F3B0A78C
8CFA     8CFA 8CFA 8CFA 8CFA 4942 B8FC B8FC B8FC B8FC 387C F86A B8FC 539A      E58E9A       E58E9A       F3B0A78D
8CFB     8CFB 8CFB 8CFB 8CFB 4588 B8FD B8FD B8FD B8FD 387D F8D0 B8FD 53E3      E58FA3       E58FA3       F3B0A78E
8CFC     8CFC 8CFC 8CFC 8CFC 46A5 B8FE B8FE B8FE B8FE 387E F8A1 B8FE 5411      E59091       E59091       F3B0A78F
8D40     8D40 8D40 8D40 8D40 51AF B9A1 B9A1 B9A1 B9A1 3921 F94F B9A1 540E      E5908E       E5908E       F3B0A790
8D41     8D41 8D41 8D41 8D41 5253 B9A2 B9A2 B9A2 B9A2 3922 F97F B9A2 5589      E59689       E59689       F3B0A791
8D42     8D42 8D42 8D42 8D42 4EF3 B9A3 B9A3 B9A3 B9A3 3923 F97B B9A3 5751      E59D91       E59D91       F3B0A792
8D43     8D43 8D43 8D43 8D43 5466 B9A4 B9A4 B9A4 B9A4 3924 F9E0 B9A4 57A2      E59EA2       E59EA2       F3B0A793
8D44     8D44 8D44 8D44 8D44 469E B9A5 B9A5 B9A5 B9A5 3925 F96C B9A5 597D      E5A5BD       E5A5BD       F3B0A794
8D45     8D45 8D45 8D45 8D45 4BD9 B9A6 B9A6 B9A6 B9A6 3926 F950 B9A6 5B54      E5AD94       E5AD94       F3B0A795
8D46     8D46 8D46 8D46 8D46 45F4 B9A7 B9A7 B9A7 B9A7 3927 F97D B9A7 5B5D      E5AD9D       E5AD9D       F3B0A796
8D47     8D47 8D47 8D47 8D47 46C8 B9A8 B9A8 B9A8 B9A8 3928 F94D B9A8 5B8F      E5AE8F       E5AE8F       F3B0A797
8D48     8D48 8D48 8D48 8D48 456A B9A9 B9A9 B9A9 B9A9 3929 F95D B9A9 5DE5      E5B7A5       E5B7A5       F3B0A798
8D49     8D49 8D49 8D49 8D49 4CA3 B9AA B9AA B9AA B9AA 392A F95C B9AA 5DE7      E5B7A7       E5B7A7       F3B0A799
8D4A     8D4A 8D4A 8D4A 8D4A 53CB B9AB B9AB B9AB B9AB 392B F94E B9AB 5DF7      E5B7B7       E5B7B7       F3B0A79A
8D4B     8D4B 8D4B 8D4B 8D4B 45A9 B9AC B9AC B9AC B9AC 392C F96B B9AC 5E78      E5B9B8       E5B9B8       F3B0A79B
8D4C     8D4C 8D4C 8D4C 8D4C 45B7 B9AD B9AD B9AD B9AD 392D F960 B9AD 5E83      E5BA83       E5BA83       F3B0A79C
8D4D     8D4D 8D4D 8D4D 8D4D 515A B9AE B9AE B9AE B9AE 392E F94B B9AE 5E9A      E5BA9A       E5BA9A       F3B0A79D
8D4E     8D4E 8D4E 8D4E 8D4E 469A B9AF B9AF B9AF B9AF 392F F961 B9AF 5EB7      E5BAB7       E5BAB7       F3B0A79E
8D4F     8D4F 8D4F 8D4F 8D4F 45D3 B9B0 B9B0 B9B0 B9B0 3930 F9F0 B9B0 5F18      E5BC98       E5BC98       F3B0A79F
8D50     8D50 8D50 8D50 8D50 47C7 B9B1 B9B1 B9B1 B9B1 3931 F9F1 B9B1 6052      E68192       E68192       F3B0A7A0
8D51     8D51 8D51 8D51 8D51 538E B9B2 B9B2 B9B2 B9B2 3932 F9F2 B9B2 614C      E6858C       E6858C       F3B0A7A1
8D52     8D52 8D52 8D52 8D52 4A67 B9B3 B9B3 B9B3 B9B3 3933 F9F3 B9B3 6297      E68A97       E68A97       F3B0A7A2
8D53     8D53 8D53 8D53 8D53 5045 B9B4 B9B4 B9B4 B9B4 3934 F9F4 B9B4 62D8      E68B98       E68B98       F3B0A7A3
8D54     8D54 8D54 8D54 8D54 4CAC B9B5 B9B5 B9B5 B9B5 3935 F9F5 B9B5 63A7      E68EA7       E68EA7       F3B0A7A4
8D55     8D55 8D55 8D55 8D55 4BC5 B9B6 B9B6 B9B6 B9B6 3936 F9F6 B9B6 653B      E694BB       E694BB       F3B0A7A5
# IBM Selected Kanji and Non-Kanji, Appendix Q, CJKV Information Processing by Ken Lunde 1999
#
# U+6602
# https://glyphwiki.org/wiki/u6602
#
8D56     8D56 8D56 8D56 8D56 5B99 60AE B9B7 B9B7 50FC BDB6 8E86 B9B7 6602      E69882       E69882       F3B0A7A6
# End of IBM Selected Kanji and Non-Kanji, Appendix Q, CJKV Information Processing by Ken Lunde 1999
8D57     8D57 8D57 8D57 8D57 48B0 B9B8 B9B8 B9B8 B9B8 3938 F9F8 B9B8 6643      E69983       E69983       F3B0A7A7
8D58     8D58 8D58 8D58 8D58 4A6D B9B9 B9B9 B9B9 B9B9 3939 F9F9 B9B9 66F4      E69BB4       E69BB4       F3B0A7A8
8D59     8D59 8D59 8D59 8D59 508B B9BA B9BA B9BA B9BA 393A F97A B9BA 676D      E69DAD       E69DAD       F3B0A7A9
8D5A     8D5A 8D5A 8D5A 8D5A 48BF B9BB B9BB B9BB B9BB 393B F95E B9BB 6821      E6A0A1       E6A0A1       F3B0A7AA
8D5B     8D5B 8D5B 8D5B 8D5B 52CC B9BC B9BC B9BC B9BC 393C F94C B9BC 6897      E6A297       E6A297       F3B0A7AB
8D5C     8D5C 8D5C 8D5C 8D5C 4796 B9BD B9BD B9BD B9BD 393D F97E B9BD 69CB      E6A78B       E6A78B       F3B0A7AC
8D5D     8D5D 8D5D 8D5D 8D5D 45AD B9BE B9BE B9BE B9BE 393E F96E B9BE 6C5F      E6B19F       E6B19F       F3B0A7AD
8D5E     8D5E 8D5E 8D5E 8D5E 4FD4 B9BF B9BF B9BF B9BF 393F F96F B9BF 6D2A      E6B4AA       E6B4AA       F3B0A7AE
8D5F     8D5F 8D5F 8D5F 8D5F 46EB B9C0 B9C0 B9C0 B9C0 3940 F97C B9C0 6D69      E6B5A9       E6B5A9       F3B0A7AF
8D60     8D60 8D60 8D60 8D60 48C6 B9C1 B9C1 B9C1 B9C1 3941 F9C1 B9C1 6E2F      E6B8AF       E6B8AF       F3B0A7B0
8D61     8D61 8D61 8D61 8D61 4A4D B9C2 B9C2 B9C2 B9C2 3942 F9C2 B9C2 6E9D      E6BA9D       E6BA9D       F3B0A7B1
8D62     8D62 8D62 8D62 8D62 48A0 B9C3 B9C3 B9C3 B9C3 3943 F9C3 B9C3 7532      E794B2       E794B2       F3B0A7B2
8D63     8D63 8D63 8D63 8D63 4CBB B9C4 B9C4 B9C4 B9C4 3944 F9C4 B9C4 7687      E79A87       E79A87       F3B0A7B3
8D64     8D64 8D64 8D64 8D64 4AEF B9C5 B9C5 B9C5 B9C5 3945 F9C5 B9C5 786C      E7A1AC       E7A1AC       F3B0A7B4
8D65     8D65 8D65 8D65 8D65 4FD5 B9C6 B9C6 B9C6 B9C6 3946 F9C6 B9C6 7A3F      E7A8BF       E7A8BF       F3B0A7B5
8D66     8D66 8D66 8D66 8D66 514B B9C7 B9C7 B9C7 B9C7 3947 F9C7 B9C7 7CE0      E7B3A0       E7B3A0       F3B0A7B6
8D67     8D67 8D67 8D67 8D67 4B67 B9C8 B9C8 B9C8 B9C8 3948 F9C8 B9C8 7D05      E7B485       E7B485       F3B0A7B7
8D68     8D68 8D68 8D68 8D68 4C78 B9C9 B9C9 B9C9 B9C9 3949 F9C9 B9C9 7D18      E7B498       E7B498       F3B0A7B8
8D69     8D69 8D69 8D69 8D69 506D B9CA B9CA B9CA B9CA 394A F9D1 B9CA 7D5E      E7B59E       E7B59E       F3B0A7B9
8D6A     8D6A 8D6A 8D6A 8D6A 48CB B9CB B9CB B9CB B9CB 394B F9D2 B9CB 7DB1      E7B6B1       E7B6B1       F3B0A7BA
8D6B     8D6B 8D6B 8D6B 8D6B 499A B9CC B9CC B9CC B9CC 394C F9D3 B9CC 8015      E88095       E88095       F3B0A7BB
8D6C     8D6C 8D6C 8D6C 8D6C 4760 B9CD B9CD B9CD B9CD 394D F9D4 B9CD 8003      E88083       E88083       F3B0A7BC
8D6D     8D6D 8D6D 8D6D 8D6D 50EB B9CE B9CE B9CE B9CE 394E F9D5 B9CE 80AF      E882AF       E882AF       F3B0A7BD
8D6E     8D6E 8D6E 8D6E 8D6E 52D5 B9CF B9CF B9CF B9CF 394F F9D6 B9CF 80B1      E882B1       E882B1       F3B0A7BE
8D6F     8D6F 8D6F 8D6F 8D6F 5247 B9D0 B9D0 B9D0 B9D0 3950 F9D7 B9D0 8154      E88594       E88594       F3B0A7BF
8D70     8D70 8D70 8D70 8D70 51D8 B9D1 B9D1 B9D1 B9D1 3951 F9D8 B9D1 818F      E8868F       E8868F       F3B0A880
8D71     8D71 8D71 8D71 8D71 489F B9D2 B9D2 B9D2 B9D2 3952 F9D9 B9D2 822A      E888AA       E888AA       F3B0A881
8D72     8D72 8D72 8D72 8D72 46DC B9D3 B9D3 B9D3 B9D3 3953 F9E2 B9D3 8352      E88D92       E88D92       F3B0A882
8D73     8D73 8D73 8D73 8D73 45A4 B9D4 B9D4 B9D4 B9D4 3954 F9E3 B9D4 884C      E8A18C       E8A18C       F3B0A883
8D74     8D74 8D74 8D74 8D74 4BED B9D5 B9D5 B9D5 B9D5 3955 F9E4 B9D5 8861      E8A1A1       E8A1A1       F3B0A884
8D75     8D75 8D75 8D75 8D75 4A6E B9D6 B9D6 B9D6 B9D6 3956 F9E5 B9D6 8B1B      E8AC9B       E8AC9B       F3B0A885
8D76     8D76 8D76 8D76 8D76 4AF1 B9D7 B9D7 B9D7 B9D7 3957 F9E6 B9D7 8CA2      E8B2A2       E8B2A2       F3B0A886
8D77     8D77 8D77 8D77 8D77 4E8F B9D8 B9D8 B9D8 B9D8 3958 F9E7 B9D8 8CFC      E8B3BC       E8B3BC       F3B0A887
8D78     8D78 8D78 8D78 8D78 5052 B9D9 B9D9 B9D9 B9D9 3959 F9E8 B9D9 90CA      E9838A       E9838A       F3B0A888
8D79     8D79 8D79 8D79 8D79 4D86 B9DA B9DA B9DA B9DA 395A F9E9 B9DA 9175      E985B5       E985B5       F3B0A889
8D7A     8D7A 8D7A 8D7A 8D7A 4863 B9DB B9DB B9DB B9DB 395B F94A B9DB 9271      E989B1       E989B1       F3B0A88A
8D7B     8D7B 8D7B 8D7B 8D7B 5FAF E2E8 B9DC B9DC E2E8 6268 5967 B9DC 783F      E7A0BF       E7A0BF       F3B0A88B
8D7C     8D7C 8D7C 8D7C 8D7C 46A1 B9DD B9DD B9DD B9DD 395D F95A B9DD 92FC      E98BBC       E98BBC       F3B0A88C
8D7D     8D7D 8D7D 8D7D 8D7D 549B B9DE B9DE B9DE B9DE 395E F95F B9DE 95A4      E996A4       E996A4       F3B0A88D
8D7E     8D7E 8D7E 8D7E 8D7E 49E5 B9DF B9DF B9DF B9DF 395F F96D B9DF 964D      E9998D       E9998D       F3B0A88E
8D80     8D80 8D80 8D80 8D80 4A97 B9E0 B9E0 B9E0 B9E0 3960 F979 B9E0 9805      E9A085       E9A085       F3B0A88F
8D81     8D81 8D81 8D81 8D81 46BD B9E1 B9E1 B9E1 B9E1 3961 F957 B9E1 9999      E9A699       E9A699       F3B0A890
8D82     8D82 8D82 8D82 8D82 4571 B9E2 B9E2 B9E2 B9E2 3962 F959 B9E2 9AD8      E9AB98       E9AB98       F3B0A891
8D83     8D83 8D83 8D83 8D83 4DD6 B9E3 B9E3 B9E3 B9E3 3963 F962 B9E3 9D3B      E9B4BB       E9B4BB       F3B0A892
8D84     8D84 8D84 8D84 8D84 49C7 B9E4 B9E4 B9E4 B9E4 3964 F963 B9E4 525B      E5899B       E5899B       F3B0A893
8D85     8D85 8D85 8D85 8D85 54C6 B9E5 B9E5 B9E5 B9E5 3965 F964 B9E5 52AB      E58AAB       E58AAB       F3B0A894
8D86     8D86 8D86 8D86 8D86 48E2 B9E6 B9E6 B9E6 B9E6 3966 F965 B9E6 53F7      E58FB7       E58FB7       F3B0A895
8D87     8D87 8D87 8D87 8D87 45A0 B9E7 B9E7 B9E7 B9E7 3967 F966 B9E7 5408      E59088       E59088       F3B0A896
8D88     8D88 8D88 8D88 8D88 55B9 B9E8 B9E8 B9E8 B9E8 3968 F967 B9E8 58D5      E5A395       E5A395       F3B0A897
8D89     8D89 8D89 8D89 8D89 55B5 B9E9 B9E9 B9E9 B9E9 3969 F968 B9E9 62F7      E68BB7       E68BB7       F3B0A898
8D8A     8D8A 8D8A 8D8A 8D8A 5380 B9EA B9EA B9EA B9EA 396A F969 B9EA 6FE0      E6BFA0       E6BFA0       F3B0A899
8D8B     8D8B 8D8B 8D8B 8D8B 4C68 B9EB B9EB B9EB B9EB 396B F970 B9EB 8C6A      E8B1AA       E8B1AA       F3B0A89A
8D8C     8D8C 8D8C 8D8C 8D8C 50A9 B9EC B9EC B9EC B9EC 396C F971 B9EC 8F5F      E8BD9F       E8BD9F       F3B0A89B
8D8D     8D8D 8D8D 8D8D 8D8D 5261 B9ED B9ED B9ED 6FC3 D6BB AC8C B9ED 9EB9      E9BAB9       E9BAB9       F3B0A89C
8D8E     8D8E 8D8E 8D8E 8D8E 479E B9EE B9EE B9EE B9EE 396E F973 B9EE 514B      E5858B       E5858B       F3B0A89D
8D8F     8D8F 8D8F 8D8F 8D8F 4BF5 B9EF B9EF B9EF B9EF 396F F974 B9EF 523B      E588BB       E588BB       F3B0A89E
8D90     8D90 8D90 8D90 8D90 4954 B9F0 B9F0 B9F0 B9F0 3970 F975 B9F0 544A      E5918A       E5918A       F3B0A89F
8D91     8D91 8D91 8D91 8D91 458F B9F1 B9F1 B9F1 B9F1 3971 F976 B9F1 56FD      E59BBD       E59BBD       F3B0A8A0
8D92     8D92 8D92 8D92 8D92 4E94 B9F2 B9F2 B9F2 B9F2 3972 F977 B9F2 7A40      E7A980       E7A980       F3B0A8A1
8D93     8D93 8D93 8D93 8D93 51EB B9F3 B9F3 B9F3 B9F3 3973 F978 B9F3 9177      E985B7       E985B7       F3B0A8A2
8D94     8D94 8D94 8D94 8D94 50FD B9F4 B9F4 B9F4 B9F4 3974 F980 B9F4 9D60      E9B5A0       E9B5A0       F3B0A8A3
8D95     8D95 8D95 8D95 8D95 4679 B9F5 B9F5 B9F5 B9F5 3975 F98B B9F5 9ED2      E9BB92       E9BB92       F3B0A8A4
8D96     8D96 8D96 8D96 8D96 4E95 B9F6 B9F6 B9F6 B9F6 3976 F99B B9F6 7344      E78D84       E78D84       F3B0A8A5
8D97     8D97 8D97 8D97 8D97 54D0 B9F7 B9F7 B9F7 B9F7 3977 F99C B9F7 6F09      E6BC89       E6BC89       F3B0A8A6
8D98     8D98 8D98 8D98 8D98 4BA4 B9F8 B9F8 B9F8 B9F8 3978 F9A0 B9F8 8170      E885B0       E885B0       F3B0A8A7
8D99     8D99 8D99 8D99 8D99 5495 B9F9 B9F9 B9F9 B9F9 3979 F9AB B9F9 7511      E79491       E79491       F3B0A8A8
8D9A     8D9A 8D9A 8D9A 8D9A 5056 B9FA B9FA B9FA B9FA 397A F9B0 B9FA 5FFD      E5BFBD       E5BFBD       F3B0A8A9
8D9B     8D9B 8D9B 8D9B 8D9B 549D B9FB B9FB B9FB B9FB 397B F9C0 B9FB 60DA      E6839A       E6839A       F3B0A8AA
8D9C     8D9C 8D9C 8D9C 8D9C 4AA5 B9FC B9FC B9FC B9FC 397C F96A B9FC 9AA8      E9AAA8       E9AAA8       F3B0A8AB
8D9D     8D9D 8D9D 8D9D 8D9D 50DC B9FD B9FD B9FD B9FD 397D F9D0 B9FD 72DB      E78B9B       E78B9B       F3B0A8AC
8D9E     8D9E 8D9E 8D9E 8D9E 4841 B9FE B9FE B9FE B9FE 397E F9A1 B9FE 8FBC      E8BEBC       E8BEBC       F3B0A8AD
8D9F     8D9F 8D9F 8D9F 8D9F 4D4C BAA1 BAA1 BAA1 BAA1 3A21 7A4F BAA1 6B64      E6ADA4       E6ADA4       F3B0A8AE
8DA0     8DA0 8DA0 8DA0 8DA0 4946 BAA2 BAA2 BAA2 BAA2 3A22 7A7F BAA2 9803      E9A083       E9A083       F3B0A8AF
8DA1     8DA1 8DA1 8DA1 8DA1 45E0 BAA3 BAA3 BAA3 BAA3 3A23 7A7B BAA3 4ECA      E4BB8A       E4BB8A       F3B0A8B0
8DA2     8DA2 8DA2 8DA2 8DA2 4B44 BAA4 BAA4 BAA4 BAA4 3A24 7AE0 BAA4 56F0      E59BB0       E59BB0       F3B0A8B1
8DA3     8DA3 8DA3 8DA3 8DA3 53A6 BAA5 BAA5 BAA5 BAA5 3A25 7A6C BAA5 5764      E59DA4       E59DA4       F3B0A8B2
8DA4     8DA4 8DA4 8DA4 8DA4 5488 BAA6 BAA6 BAA6 BAA6 3A26 7A50 BAA6 58BE      E5A2BE       E5A2BE       F3B0A8B3
8DA5     8DA5 8DA5 8DA5 8DA5 4A77 BAA7 BAA7 BAA7 BAA7 3A27 7A7D BAA7 5A5A      E5A99A       E5A99A       F3B0A8B4
8DA6     8DA6 8DA6 8DA6 8DA6 53B8 BAA8 BAA8 BAA8 BAA8 3A28 7A4D BAA8 6068      E681A8       E681A8       F3B0A8B5
8DA7     8DA7 8DA7 8DA7 8DA7 4EA4 BAA9 BAA9 BAA9 BAA9 3A29 7A5D BAA9 61C7      E68787       E68787       F3B0A8B6
8DA8     8DA8 8DA8 8DA8 8DA8 54E8 BAAA BAAA BAAA BAAA 3A2A 7A5C BAAA 660F      E6988F       E6988F       F3B0A8B7
8DA9     8DA9 8DA9 8DA9 8DA9 4FE7 BAAB BAAB BAAB BAAB 3A2B 7A4E BAAB 6606      E69886       E69886       F3B0A8B8
8DAA     8DAA 8DAA 8DAA 8DAA 4667 BAAC BAAC BAAC BAAC 3A2C 7A6B BAAC 6839      E6A0B9       E6A0B9       F3B0A8B9
8DAB     8DAB 8DAB 8DAB 8DAB 5359 BAAD BAAD BAAD BAAD 3A2D 7A60 BAAD 68B1      E6A2B1       E6A2B1       F3B0A8BA
8DAC     8DAC 8DAC 8DAC 8DAC 4B46 BAAE BAAE BAAE BAAE 3A2E 7A4B BAAE 6DF7      E6B7B7       E6B7B7       F3B0A8BB
8DAD     8DAD 8DAD 8DAD 8DAD 53FD BAAF BAAF BAAF BAAF 3A2F 7A61 BAAF 75D5      E79795       E79795       F3B0A8BC
8DAE     8DAE 8DAE 8DAE 8DAE 4CBD BAB0 BAB0 BAB0 BAB0 3A30 7AF0 BAB0 7D3A      E7B4BA       E7B4BA       F3B0A8BD
8DAF     8DAF 8DAF 8DAF 8DAF 55B4 BAB1 BAB1 BAB1 BAB1 3A31 7AF1 BAB1 826E      E889AE       E889AE       F3B0A8BE
8DB0     8DB0 8DB0 8DB0 8DB0 50A8 BAB2 BAB2 BAB2 BAB2 3A32 7AF2 BAB2 9B42      E9AD82       E9AD82       F3B0A8BF
8DB1     8DB1 8DB1 8DB1 8DB1 55DC BAB3 BAB3 BAB3 BAB3 3A33 7AF3 BAB3 4E9B      E4BA9B       E4BA9B       F3B0A980
8DB2     8DB2 8DB2 8DB2 8DB2 4576 BAB4 BAB4 BAB4 BAB4 3A34 7AF4 BAB4 4F50      E4BD90       E4BD90       F3B0A981
8DB3     8DB3 8DB3 8DB3 8DB3 50A5 BAB5 BAB5 BAB5 BAB5 3A35 7AF5 BAB5 53C9      E58F89       E58F89       F3B0A982
8DB4     8DB4 8DB4 8DB4 8DB4 51B8 BAB6 BAB6 BAB6 BAB6 3A36 7AF6 BAB6 5506      E59486       E59486       F3B0A983
8DB5     8DB5 8DB5 8DB5 8DB5 4ED3 BAB7 BAB7 BAB7 BAB7 3A37 7AF7 BAB7 5D6F      E5B5AF       E5B5AF       F3B0A984
8DB6     8DB6 8DB6 8DB6 8DB6 485B BAB8 BAB8 BAB8 BAB8 3A38 7AF8 BAB8 5DE6      E5B7A6       E5B7A6       F3B0A985
8DB7     8DB7 8DB7 8DB7 8DB7 498B BAB9 BAB9 BAB9 BAB9 3A39 7AF9 BAB9 5DEE      E5B7AE       E5B7AE       F3B0A986
8DB8     8DB8 8DB8 8DB8 8DB8 4773 BABA BABA BABA BABA 3A3A 7A7A BABA 67FB      E69FBB       E69FBB       F3B0A987
8DB9     8DB9 8DB9 8DB9 8DB9 4EAD BABB BABB BABB BABB 3A3B 7A5E BABB 6C99      E6B299       E6B299       F3B0A988
8DBA     8DBA 8DBA 8DBA 8DBA 5188 BABC BABC BABC BABC 3A3C 7A4C BABC 7473      E791B3       E791B3       F3B0A989
8DBB     8DBB 8DBB 8DBB 8DBB 495B BABD BABD BABD BABD 3A3D 7A7E BABD 7802      E7A082       E7A082       F3B0A98A
8DBC     8DBC 8DBC 8DBC 8DBC 5176 BABE BABE BABE BABE 3A3E 7A6E BABE 8A50      E8A990       E8A990       F3B0A98B
8DBD     8DBD 8DBD 8DBD 8DBD 4D70 BABF BABF BABF BABF 3A3F 7A6F BABF 9396      E98E96       E98E96       F3B0A98C
8DBE     8DBE 8DBE 8DBE 8DBE 4F8A BAC0 BAC0 BAC0 BAC0 3A40 7A7C BAC0 88DF      E8A39F       E8A39F       F3B0A98D
8DBF     8DBF 8DBF 8DBF 8DBF 4E8D BAC1 BAC1 BAC1 BAC1 3A41 7AC1 BAC1 5750      E59D90       E59D90       F3B0A98E
8DC0     8DC0 8DC0 8DC0 8DC0 48B1 BAC2 BAC2 BAC2 BAC2 3A42 7AC2 BAC2 5EA7      E5BAA7       E5BAA7       F3B0A98F
8DC1     8DC1 8DC1 8DC1 8DC1 548F BAC3 BAC3 BAC3 BAC3 3A43 7AC3 BAC3 632B      E68CAB       E68CAB       F3B0A990
8DC2     8DC2 8DC2 8DC2 8DC2 4B58 BAC4 BAC4 BAC4 BAC4 3A44 7AC4 BAC4 50B5      E582B5       E582B5       F3B0A991
8DC3     8DC3 8DC3 8DC3 8DC3 4C43 BAC5 BAC5 BAC5 BAC5 3A45 7AC5 BAC5 50AC      E582AC       E582AC       F3B0A992
8DC4     8DC4 8DC4 8DC4 8DC4 48EB BAC6 BAC6 BAC6 BAC6 3A46 7AC6 BAC6 518D      E5868D       E5868D       F3B0A993
8DC5     8DC5 8DC5 8DC5 8DC5 46F5 BAC7 BAC7 BAC7 BAC7 3A47 7AC7 BAC7 6700      E69C80       E69C80       F3B0A994
8DC6     8DC6 8DC6 8DC6 8DC6 4A6B BAC8 BAC8 BAC8 BAC8 3A48 7AC8 BAC8 54C9      E59389       E59389       F3B0A995
8DC7     8DC7 8DC7 8DC7 8DC7 53C1 BAC9 BAC9 BAC9 BAC9 3A49 7AC9 BAC9 585E      E5A19E       E5A19E       F3B0A996
8DC8     8DC8 8DC8 8DC8 8DC8 49A3 BACA BACA BACA BACA 3A4A 7AD1 BACA 59BB      E5A6BB       E5A6BB       F3B0A997
8DC9     8DC9 8DC9 8DC9 8DC9 4ED4 BACB BACB BACB BACB 3A4B 7AD2 BACB 5BB0      E5AEB0       E5AEB0       F3B0A998
8DCA     8DCA 8DCA 8DCA 8DCA 4DCD BACC BACC BACC BACC 3A4C 7AD3 BACC 5F69      E5BDA9       E5BDA9       F3B0A999
8DCB     8DCB 8DCB 8DCB 8DCB 49F1 BACD BACD BACD BACD 3A4D 7AD4 BACD 624D      E6898D       E6898D       F3B0A99A
8DCC     8DCC 8DCC 8DCC 8DCC 4B4F BACE BACE BACE BACE 3A4E 7AD5 BACE 63A1      E68EA1       E68EA1       F3B0A99B
8DCD     8DCD 8DCD 8DCD 8DCD 4DE0 BACF BACF BACF BACF 3A4F 7AD6 BACF 683D      E6A0BD       E6A0BD       F3B0A99C
8DCE     8DCE 8DCE 8DCE 8DCE 48E4 BAD0 BAD0 BAD0 BAD0 3A50 7AD7 BAD0 6B73      E6ADB3       E6ADB3       F3B0A99D
8DCF     8DCF 8DCF 8DCF 8DCF 4769 BAD1 BAD1 BAD1 BAD1 3A51 7AD8 BAD1 6E08      E6B888       E6B888       F3B0A99E
8DD0     8DD0 8DD0 8DD0 8DD0 4CBC BAD2 BAD2 BAD2 BAD2 3A52 7AD9 BAD2 707D      E781BD       E781BD       F3B0A99F
8DD1     8DD1 8DD1 8DD1 8DD1 53BF BAD3 BAD3 BAD3 BAD3 3A53 7AE2 BAD3 91C7      E98787       E98787       F3B0A9A0
8DD2     8DD2 8DD2 8DD2 8DD2 5354 BAD4 BAD4 BAD4 BAD4 3A54 7AE3 BAD4 7280      E78A80       E78A80       F3B0A9A1
8DD3     8DD3 8DD3 8DD3 8DD3 4CC3 BAD5 BAD5 BAD5 BAD5 3A55 7AE4 BAD5 7815      E7A095       E7A095       F3B0A9A2
8DD4     8DD4 8DD4 8DD4 8DD4 54B4 BAD6 BAD6 BAD6 BAD6 3A56 7AE5 BAD6 7826      E7A0A6       E7A0A6       F3B0A9A3
8DD5     8DD5 8DD5 8DD5 8DD5 4C71 BAD7 BAD7 BAD7 BAD7 3A57 7AE6 BAD7 796D      E7A5AD       E7A5AD       F3B0A9A4
8DD6     8DD6 8DD6 8DD6 8DD6 47DB BAD8 BAD8 BAD8 BAD8 3A58 7AE7 BAD8 658E      E6968E       E6968E       F3B0A9A5
8DD7     8DD7 8DD7 8DD7 8DD7 4768 BAD9 BAD9 BAD9 BAD9 3A59 7AE8 BAD9 7D30      E7B4B0       E7B4B0       F3B0A9A6
8DD8     8DD8 8DD8 8DD8 8DD8 4BA8 BADA BADA BADA BADA 3A5A 7AE9 BADA 83DC      E88F9C       E88F9C       F3B0A9A7
8DD9     8DD9 8DD9 8DD9 8DD9 49BD BADB BADB BADB BADB 3A5B 7A4A BADB 88C1      E8A381       E8A381       F3B0A9A8
8DDA     8DDA 8DDA 8DDA 8DDA 4D6E BADC BADC BADC BADC 3A5C 7A5B BADC 8F09      E8BC89       E8BC89       F3B0A9A9
8DDB     8DDB 8DDB 8DDB 8DDB 4797 BADD BADD BADD BADD 3A5D 7A5A BADD 969B      E99A9B       E99A9B       F3B0A9AA
8DDC     8DDC 8DDC 8DDC 8DDC 4766 BADE BADE BADE BADE 3A5E 7A5F BADE 5264      E589A4       E589A4       F3B0A9AB
8DDD     8DDD 8DDD 8DDD 8DDD 47CD BADF BADF BADF BADF 3A5F 7A6D BADF 5728      E59CA8       E59CA8       F3B0A9AC
8DDE     8DDE 8DDE 8DDE 8DDE 46BC BAE0 BAE0 BAE0 BAE0 3A60 7A79 BAE0 6750      E69D90       E69D90       F3B0A9AD
8DDF     8DDF 8DDF 8DDF 8DDF 4C4E BAE1 BAE1 BAE1 BAE1 3A61 7A57 BAE1 7F6A      E7BDAA       E7BDAA       F3B0A9AE
8DE0     8DE0 8DE0 8DE0 8DE0 49AF BAE2 BAE2 BAE2 BAE2 3A62 7A59 BAE2 8CA1      E8B2A1       E8B2A1       F3B0A9AF
8DE1     8DE1 8DE1 8DE1 8DE1 50C9 BAE3 BAE3 BAE3 BAE3 3A63 7A62 BAE3 51B4      E586B4       E586B4       F3B0A9B0
8DE2     8DE2 8DE2 8DE2 8DE2 45EC BAE4 BAE4 BAE4 BAE4 3A64 7A63 BAE4 5742      E59D82       E59D82       F3B0A9B1
8DE3     8DE3 8DE3 8DE3 8DE3 4591 BAE5 BAE5 BAE5 BAE5 3A65 7A64 BAE5 962A      E998AA       E998AA       F3B0A9B2
8DE4     8DE4 8DE4 8DE4 8DE4 4A7C BAE6 BAE6 BAE6 BAE6 3A66 7A65 BAE6 583A      E5A0BA       E5A0BA       F3B0A9B3
8DE5     8DE5 8DE5 8DE5 8DE5 4B52 BAE7 BAE7 BAE7 BAE7 3A67 7A66 BAE7 698A      E6A68A       E6A68A       F3B0A9B4
8DE6     8DE6 8DE6 8DE6 8DE6 5451 BAE8 BAE8 BAE8 BAE8 3A68 7A67 BAE8 80B4      E882B4       E882B4       F3B0A9B5
8DE7     8DE7 8DE7 8DE7 8DE7 4CDE BAE9 BAE9 BAE9 BAE9 3A69 7A68 BAE9 54B2      E592B2       E592B2       F3B0A9B6
8DE8     8DE8 8DE8 8DE8 8DE8 457F BAEA BAEA BAEA BAEA 3A6A 7A69 BAEA 5D0E      E5B48E       E5B48E       F3B0A9B7
8DE9     8DE9 8DE9 8DE9 8DE9 46A3 BAEB BAEB BAEB BAEB 3A6B 7A70 BAEB 57FC      E59FBC       E59FBC       F3B0A9B8
8DEA     8DEA 8DEA 8DEA 8DEA 52D2 BAEC BAEC BAEC BAEC 3A6C 7A71 BAEC 7895      E7A295       E7A295       F3B0A9B9
8DEB     8DEB 8DEB 8DEB 8DEB 50AF BAED BAED BAED BAED 3A6D 7A72 BAED 9DFA      E9B7BA       E9B7BA       F3B0A9BA
8DEC     8DEC 8DEC 8DEC 8DEC 4594 BAEE BAEE BAEE BAEE 3A6E 7A73 BAEE 4F5C      E4BD9C       E4BD9C       F3B0A9BB
8DED     8DED 8DED 8DED 8DED 4A90 BAEF BAEF BAEF BAEF 3A6F 7A74 BAEF 524A      E5898A       E5898A       F3B0A9BC
8DEE     8DEE 8DEE 8DEE 8DEE 4F50 BAF0 BAF0 BAF0 BAF0 3A70 7A75 BAF0 548B      E5928B       E5928B       F3B0A9BD
8DEF     8DEF 8DEF 8DEF 8DEF 50A3 BAF1 BAF1 BAF1 BAF1 3A71 7A76 BAF1 643E      E690BE       E690BE       F3B0A9BE
8DF0     8DF0 8DF0 8DF0 8DF0 48E8 BAF2 BAF2 BAF2 BAF2 3A72 7A77 BAF2 6628      E698A8       E698A8       F3B0A9BF
8DF1     8DF1 8DF1 8DF1 8DF1 4FA4 BAF3 BAF3 BAF3 BAF3 3A73 7A78 BAF3 6714      E69C94       E69C94       F3B0AA80
8DF2     8DF2 8DF2 8DF2 8DF2 51F1 BAF4 BAF4 BAF4 BAF4 3A74 7A80 BAF4 67F5      E69FB5       E69FB5       F3B0AA81
8DF3     8DF3 8DF3 8DF3 8DF3 5499 BAF5 BAF5 BAF5 BAF5 3A75 7A8B BAF5 7A84      E7AA84       E7AA84       F3B0AA82
8DF4     8DF4 8DF4 8DF4 8DF4 46C9 BAF6 BAF6 BAF6 BAF6 3A76 7A9B BAF6 7B56      E7AD96       E7AD96       F3B0AA83
8DF5     8DF5 8DF5 8DF5 8DF5 4CF4 BAF7 BAF7 BAF7 BAF7 3A77 7A9C BAF7 7D22      E7B4A2       E7B4A2       F3B0AA84
8DF6     8DF6 8DF6 8DF6 8DF6 50B1 BAF8 BAF8 BAF8 BAF8 3A78 7AA0 BAF8 932F      E98CAF       E98CAF       F3B0AA85
8DF7     8DF7 8DF7 8DF7 8DF7 4771 BAF9 BAF9 BAF9 BAF9 3A79 7AAB BAF9 685C      E6A19C       E6A19C       F3B0AA86
8DF8     8DF8 8DF8 8DF8 8DF8 5391 BAFA BAFA BAFA BAFA 3A7A 7AB0 BAFA 9BAD      E9AEAD       E9AEAD       F3B0AA87
8DF9     8DF9 8DF9 8DF9 8DF9 496E BAFB BAFB BAFB BAFB 3A7B 7AC0 BAFB 7B39      E7ACB9       E7ACB9       F3B0AA88
8DFA     8DFA 8DFA 8DFA 8DFA 5457 BAFC BAFC BAFC BAFC 3A7C 7A6A BAFC 5319      E58C99       E58C99       F3B0AA89
8DFB     8DFB 8DFB 8DFB 8DFB 4EA1 BAFD BAFD BAFD BAFD 3A7D 7AD0 BAFD 518A      E5868A       E5868A       F3B0AA8A
8DFC     8DFC 8DFC 8DFC 8DFC 48CF BAFE BAFE BAFE BAFE 3A7E 7AA1 BAFE 5237      E588B7       E588B7       F3B0AA8B
8E40     8E40 8E40 8E40 8E40 4B93 BBA1 BBA1 BBA1 BBA1 3B21 5E4F BBA1 5BDF      E5AF9F       E5AF9F       F3B0AA8C
8E41     8E41 8E41 8E41 8E41 4F9E BBA2 BBA2 BBA2 BBA2 3B22 5E7F BBA2 62F6      E68BB6       E68BB6       F3B0AA8D
8E42     8E42 8E42 8E42 8E42 4C77 BBA3 BBA3 BBA3 BBA3 3B23 5E7B BBA3 64AE      E692AE       E692AE       F3B0AA8E
8E43     8E43 8E43 8E43 8E43 4FD6 BBA4 BBA4 BBA4 BBA4 3B24 5EE0 BBA4 64E6      E693A6       E693A6       F3B0AA8F
8E44     8E44 8E44 8E44 8E44 48F9 BBA5 BBA5 BBA5 BBA5 3B25 5E6C BBA5 672D      E69CAD       E69CAD       F3B0AA90
8E45     8E45 8E45 8E45 8E45 49E9 BBA6 BBA6 BBA6 BBA6 3B26 5E50 BBA6 6BBA      E6AEBA       E6AEBA       F3B0AA91
8E46     8E46 8E46 8E46 8E46 4CF1 BBA7 BBA7 BBA7 BBA7 3B27 5E7D BBA7 85A9      E896A9       E896A9       F3B0AA92
8E47     8E47 8E47 8E47 8E47 4A8A BBA8 BBA8 BBA8 BBA8 3B28 5E4D BBA8 96D1      E99B91       E99B91       F3B0AA93
8E48     8E48 8E48 8E48 8E48 52A4 BBA9 BBA9 BBA9 BBA9 3B29 5E5D BBA9 7690      E79A90       E79A90       F3B0AA94
8E49     8E49 8E49 8E49 8E49 4FE4 BBAA BBAA BBAA BBAA 3B2A 5E5C BBAA 9BD6      E9AF96       E9AF96       F3B0AA95
8E4A     8E4A 8E4A 8E4A 8E4A 557D BBAB BBAB BBAB BBAB 3B2B 5E4E BBAB 634C      E68D8C       E68D8C       F3B0AA96
8E4B     8E4B 8E4B 8E4B 8E4B 55D2 BBAC BBAC BBAC BBAC 3B2C 5E6B BBAC 9306      E98C86       E98C86       F3B0AA97
8E4C     8E4C 8E4C 8E4C 8E4C 4EB7 BBAD BBAD BBAD BBAD 3B2D 5E60 BBAD 9BAB      E9AEAB       E9AEAB       F3B0AA98
8E4D     8E4D 8E4D 8E4D 8E4D 4DC7 BBAE BBAE BBAE BBAE 3B2E 5E4B BBAE 76BF      E79ABF       E79ABF       F3B0AA99
8E4E     8E4E 8E4E 8E4E 8E4E 54C3 BBAF BBAF BBAF BBAF 3B2F 5E61 BBAF 6652      E69992       E69992       F3B0AA9A
8E4F     8E4F 8E4F 8E4F 8E4F 4543 BBB0 BBB0 BBB0 BBB0 3B30 5EF0 BBB0 4E09      E4B889       E4B889       F3B0AA9B
8E50     8E50 8E50 8E50 8E50 509C BBB1 BBB1 BBB1 BBB1 3B31 5EF1 BBB1 5098      E58298       E58298       F3B0AA9C
8E51     8E51 8E51 8E51 8E51 48D2 BBB2 BBB2 BBB2 BBB2 3B32 5EF2 BBB2 53C2      E58F82       E58F82       F3B0AA9D
8E52     8E52 8E52 8E52 8E52 4565 BBB3 BBB3 BBB3 BBB3 3B33 5EF3 BBB3 5C71      E5B1B1       E5B1B1       F3B0AA9E
8E53     8E53 8E53 8E53 8E53 506E BBB4 BBB4 BBB4 BBB4 3B34 5EF4 BBB4 60E8      E683A8       E683A8       F3B0AA9F
8E54     8E54 8E54 8E54 8E54 53DE BBB5 BBB5 BBB5 BBB5 3B35 5EF5 BBB5 6492      E69292       E69292       F3B0AAA0
8E55     8E55 8E55 8E55 8E55 4AC9 BBB6 BBB6 BBB6 BBB6 3B36 5EF6 BBB6 6563      E695A3       E695A3       F3B0AAA1
8E56     8E56 8E56 8E56 8E56 51F3 BBB7 BBB7 BBB7 BBB7 3B37 5EF7 BBB7 685F      E6A19F       E6A19F       F3B0AAA2
8E57     8E57 8E57 8E57 8E57 54CE BBB8 BBB8 BBB8 BBB8 3B38 5EF8 BBB8 71E6      E787A6       E787A6       F3B0AAA3
8E58     8E58 8E58 8E58 8E58 5399 BBB9 BBB9 BBB9 BBB9 3B39 5EF9 BBB9 73CA      E78F8A       E78F8A       F3B0AAA4
8E59     8E59 8E59 8E59 8E59 45C0 BBBA BBBA BBBA BBBA 3B3A 5E7A BBBA 7523      E794A3       E794A3       F3B0AAA5
8E5A     8E5A 8E5A 8E5A 8E5A 4843 BBBB BBBB BBBB BBBB 3B3B 5E5E BBBB 7B97      E7AE97       E7AE97       F3B0AAA6
8E5B     8E5B 8E5B 8E5B 8E5B 55EB BBBC BBBC BBBC BBBC 3B3C 5E4C BBBC 7E82      E7BA82       E7BA82       F3B0AAA7
8E5C     8E5C 8E5C 8E5C 8E5C 4F5D BBBD BBBD BBBD BBBD 3B3D 5E7E BBBD 8695      E89A95       E89A95       F3B0AAA8
8E5D     8E5D 8E5D 8E5D 8E5D 516B BBBE BBBE BBBE BBBE 3B3E 5E6E BBBE 8B83      E8AE83       E8AE83       F3B0AAA9
8E5E     8E5E 8E5E 8E5E 8E5E 4DEE BBBF BBBF BBBF BBBF 3B3F 5E6F BBBF 8CDB      E8B39B       E8B39B       F3B0AAAA
8E5F     8E5F 8E5F 8E5F 8E5F 4899 BBC0 BBC0 BBC0 BBC0 3B40 5E7C BBC0 9178      E985B8       E985B8       F3B0AAAB
8E60     8E60 8E60 8E60 8E60 55B3 BBC1 BBC1 BBC1 BBC1 3B41 5EC1 BBC1 9910      E9A490       E9A490       F3B0AAAC
8E61     8E61 8E61 8E61 8E61 4EFC BBC2 BBC2 BBC2 BBC2 3B42 5EC2 BBC2 65AC      E696AC       E696AC       F3B0AAAD
8E62     8E62 8E62 8E62 8E62 5093 BBC3 BBC3 BBC3 BBC3 3B43 5EC3 BBC3 66AB      E69AAB       E69AAB       F3B0AAAE
8E63     8E63 8E63 8E63 8E63 4992 BBC4 BBC4 BBC4 BBC4 3B44 5EC4 BBC4 6B8B      E6AE8B       E6AE8B       F3B0AAAF
8E64     8E64 8E64 8E64 8E64 48B5 BBC5 BBC5 BBC5 BBC5 3B45 5EC5 BBC5 4ED5      E4BB95       E4BB95       F3B0AAB0
8E65     8E65 8E65 8E65 8E65 5465 BBC6 BBC6 BBC6 BBC6 3B46 5EC6 BBC6 4ED4      E4BB94       E4BB94       F3B0AAB1
8E66     8E66 8E66 8E66 8E66 508A BBC7 BBC7 BBC7 BBC7 3B47 5EC7 BBC7 4F3A      E4BCBA       E4BCBA       F3B0AAB2
8E67     8E67 8E67 8E67 8E67 48B6 BBC8 BBC8 BBC8 BBC8 3B48 5EC8 BBC8 4F7F      E4BDBF       E4BDBF       F3B0AAB3
8E68     8E68 8E68 8E68 8E68 4BC0 BBC9 BBC9 BBC9 BBC9 3B49 5EC9 BBC9 523A      E588BA       E588BA       F3B0AAB4
8E69     8E69 8E69 8E69 8E69 45F0 BBCA BBCA BBCA BBCA 3B4A 5ED1 BBCA 53F8      E58FB8       E58FB8       F3B0AAB5
8E6A     8E6A 8E6A 8E6A 8E6A 47D9 BBCB BBCB BBCB BBCB 3B4B 5ED2 BBCB 53F2      E58FB2       E58FB2       F3B0AAB6
8E6B     8E6B 8E6B 8E6B 8E6B 4B5E BBCC BBCC BBCC BBCC 3B4C 5ED3 BBCC 55E3      E597A3       E597A3       F3B0AAB7
8E6C     8E6C 8E6C 8E6C 8E6C 4544 BBCD BBCD BBCD BBCD 3B4D 5ED4 BBCD 56DB      E59B9B       E59B9B       F3B0AAB8
8E6D     8E6D 8E6D 8E6D 8E6D 467D BBCE BBCE BBCE BBCE 3B4E 5ED5 BBCE 58EB      E5A3AB       E5A3AB       F3B0AAB9
8E6E     8E6E 8E6E 8E6E 8E6E 4990 BBCF BBCF BBCF BBCF 3B4F 5ED6 BBCF 59CB      E5A78B       E5A78B       F3B0AABA
8E6F     8E6F 8E6F 8E6F 8E6F 4CC1 BBD0 BBD0 BBD0 BBD0 3B50 5ED7 BBD0 59C9      E5A789       E5A789       F3B0AABB
8E70     8E70 8E70 8E70 8E70 4B42 BBD1 BBD1 BBD1 BBD1 3B51 5ED8 BBD1 59FF      E5A7BF       E5A7BF       F3B0AABC
8E71     8E71 8E71 8E71 8E71 4564 BBD2 BBD2 BBD2 BBD2 3B52 5ED9 BBD2 5B50      E5AD90       E5AD90       F3B0AABD
8E72     8E72 8E72 8E72 8E72 55BA BBD3 BBD3 BBD3 BBD3 3B53 5EE2 BBD3 5C4D      E5B18D       E5B18D       F3B0AABE
8E73     8E73 8E73 8E73 8E73 4553 BBD4 BBD4 BBD4 BBD4 3B54 5EE3 BBD4 5E02      E5B882       E5B882       F3B0AABF
8E74     8E74 8E74 8E74 8E74 4A44 BBD5 BBD5 BBD5 BBD5 3B55 5EE4 BBD5 5E2B      E5B8AB       E5B8AB       F3B0AB80
8E75     8E75 8E75 8E75 8E75 4661 BBD6 BBD6 BBD6 BBD6 3B56 5EE5 BBD6 5FD7      E5BF97       E5BF97       F3B0AB81
8E76     8E76 8E76 8E76 8E76 46AD BBD7 BBD7 BBD7 BBD7 3B57 5EE6 BBD7 601D      E6809D       E6809D       F3B0AB82
8E77     8E77 8E77 8E77 8E77 47B6 BBD8 BBD8 BBD8 BBD8 3B58 5EE7 BBD8 6307      E68C87       E68C87       F3B0AB83
8E78     8E78 8E78 8E78 8E78 477B BBD9 BBD9 BBD9 BBD9 3B59 5EE8 BBD9 652F      E694AF       E694AF       F3B0AB84
8E79     8E79 8E79 8E79 8E79 4FF2 BBDA BBDA BBDA BBDA 3B5A 5EE9 BBDA 5B5C      E5AD9C       E5AD9C       F3B0AB85
8E7A     8E7A 8E7A 8E7A 8E7A 4EA5 BBDB BBDB BBDB BBDB 3B5B 5E4A BBDB 65AF      E696AF       E696AF       F3B0AB86
8E7B     8E7B 8E7B 8E7B 8E7B 4888 BBDC BBDC BBDC BBDC 3B5C 5E5B BBDC 65BD      E696BD       E696BD       F3B0AB87
8E7C     8E7C 8E7C 8E7C 8E7C 4D6A BBDD BBDD BBDD BBDD 3B5D 5E5A BBDD 65E8      E697A8       E697A8       F3B0AB88
8E7D     8E7D 8E7D 8E7D 8E7D 4672 BBDE BBDE BBDE BBDE 3B5E 5E5F BBDE 679D      E69E9D       E69E9D       F3B0AB89
8E7E     8E7E 8E7E 8E7E 8E7E 48D8 BBDF BBDF BBDF BBDF 3B5F 5E6D BBDF 6B62      E6ADA2       E6ADA2       F3B0AB8A
8E80     8E80 8E80 8E80 8E80 4999 BBE0 BBE0 BBE0 BBE0 3B60 5E79 BBE0 6B7B      E6ADBB       E6ADBB       F3B0AB8B
8E81     8E81 8E81 8E81 8E81 488E BBE1 BBE1 BBE1 BBE1 3B61 5E57 BBE1 6C0F      E6B08F       E6B08F       F3B0AB8C
8E82     8E82 8E82 8E82 8E82 519F BBE2 BBE2 BBE2 BBE2 3B62 5E59 BBE2 7345      E78D85       E78D85       F3B0AB8D
8E83     8E83 8E83 8E83 8E83 4FEB BBE3 BBE3 BBE3 BBE3 3B63 5E62 BBE3 7949      E7A589       E7A589       F3B0AB8E
8E84     8E84 8E84 8E84 8E84 46C3 BBE4 BBE4 BBE4 BBE4 3B64 5E63 BBE4 79C1      E7A781       E7A781       F3B0AB8F
8E85     8E85 8E85 8E85 8E85 46C7 BBE5 BBE5 BBE5 BBE5 3B65 5E64 BBE5 7CF8      E7B3B8       E7B3B8       F3B0AB90
8E86     8E86 8E86 8E86 8E86 4680 BBE6 BBE6 BBE6 BBE6 3B66 5E65 BBE6 7D19      E7B499       E7B499       F3B0AB91
8E87     8E87 8E87 8E87 8E87 4AF0 BBE7 BBE7 BBE7 BBE7 3B67 5E66 BBE7 7D2B      E7B4AB       E7B4AB       F3B0AB92
8E88     8E88 8E88 8E88 8E88 5382 BBE8 BBE8 BBE8 BBE8 3B68 5E67 BBE8 80A2      E882A2       E882A2       F3B0AB93
8E89     8E89 8E89 8E89 8E89 4874 BBE9 BBE9 BBE9 BBE9 3B69 5E68 BBE9 8102      E88482       E88482       F3B0AB94
8E8A     8E8A 8E8A 8E8A 8E8A 4A48 BBEA BBEA BBEA BBEA 3B6A 5E69 BBEA 81F3      E887B3       E887B3       F3B0AB95
8E8B     8E8B 8E8B 8E8B 8E8B 4A50 BBEB BBEB BBEB BBEB 3B6B 5E70 BBEB 8996      E8A696       E8A696       F3B0AB96
8E8C     8E8C 8E8C 8E8C 8E8C 4E90 BBEC BBEC BBEC BBEC 3B6C 5E71 BBEC 8A5E      E8A99E       E8A99E       F3B0AB97
8E8D     8E8D 8E8D 8E8D 8E8D 4CB0 BBED BBED BBED BBED 3B6D 5E72 BBED 8A69      E8A9A9       E8A9A9       F3B0AB98
8E8E     8E8E 8E8E 8E8E 8E8E 4948 BBEE BBEE BBEE BBEE 3B6E 5E73 BBEE 8A66      E8A9A6       E8A9A6       F3B0AB99
8E8F     8E8F 8E8F 8E8F 8E8F 4B76 BBEF BBEF BBEF BBEF 3B6F 5E74 BBEF 8A8C      E8AA8C       E8AA8C       F3B0AB9A
8E90     8E90 8E90 8E90 8E90 4F9C BBF0 BBF0 BBF0 BBF0 3B70 5E75 BBF0 8AEE      E8ABAE       E8ABAE       F3B0AB9B
8E91     8E91 8E91 8E91 8E91 4650 BBF1 BBF1 BBF1 BBF1 3B71 5E76 BBF1 8CC7      E8B387       E8B387       F3B0AB9C
8E92     8E92 8E92 8E92 8E92 4F94 BBF2 BBF2 BBF2 BBF2 3B72 5E77 BBF2 8CDC      E8B39C       E8B39C       F3B0AB9D
8E93     8E93 8E93 8E93 8E93 52B7 BBF3 BBF3 BBF3 BBF3 3B73 5E78 BBF3 96CC      E99B8C       E99B8C       F3B0AB9E
8E94     8E94 8E94 8E94 8E94 4A8F BBF4 BBF4 BBF4 BBF4 3B74 5E80 BBF4 98FC      E9A3BC       E9A3BC       F3B0AB9F
8E95     8E95 8E95 8E95 8E95 4A9F BBF5 BBF5 BBF5 BBF5 3B75 5E8B BBF5 6B6F      E6ADAF       E6ADAF       F3B0ABA0
8E96     8E96 8E96 8E96 8E96 45CE BBF6 BBF6 BBF6 BBF6 3B76 5E9B BBF6 4E8B      E4BA8B       E4BA8B       F3B0ABA1
8E97     8E97 8E97 8E97 8E97 4BBC BBF7 BBF7 BBF7 BBF7 3B77 5E9C BBF7 4F3C      E4BCBC       E4BCBC       F3B0ABA2
8E98     8E98 8E98 8E98 8E98 4FC2 BBF8 BBF8 BBF8 BBF8 3B78 5EA0 BBF8 4F8D      E4BE8D       E4BE8D       F3B0ABA3
8E99     8E99 8E99 8E99 8E99 469F BBF9 BBF9 BBF9 BBF9 3B79 5EAB BBF9 5150      E58590       E58590       F3B0ABA4
8E9A     8E9A 8E9A 8E9A 8E9A 48F2 BBFA BBFA BBFA BBFA 3B7A 5EB0 BBFA 5B57      E5AD97       E5AD97       F3B0ABA5
8E9B     8E9B 8E9B 8E9B 8E9B 46A0 BBFB BBFB BBFB BBFB 3B7B 5EC0 BBFB 5BFA      E5AFBA       E5AFBA       F3B0ABA6
8E9C     8E9C 8E9C 8E9C 8E9C 4CBE BBFC BBFC BBFC BBFC 3B7C 5E6A BBFC 6148      E68588       E68588       F3B0ABA7
8E9D     8E9D 8E9D 8E9D 8E9D 46D8 BBFD BBFD BBFD BBFD 3B7D 5ED0 BBFD 6301      E68C81       E68C81       F3B0ABA8
8E9E     8E9E 8E9E 8E9E 8E9E 4651 BBFE BBFE BBFE BBFE 3B7E 5EA1 BBFE 6642      E69982       E69982       F3B0ABA9
8E9F     8E9F 8E9F 8E9F 8E9F 4597 BCA1 BCA1 BCA1 BCA1 3C21 4C4F BCA1 6B21      E6ACA1       E6ACA1       F3B0ABAA
8EA0     8EA0 8EA0 8EA0 8EA0 4857 BCA2 BCA2 BCA2 BCA2 3C22 4C7F BCA2 6ECB      E6BB8B       E6BB8B       F3B0ABAB
8EA1     8EA1 8EA1 8EA1 8EA1 4584 BCA3 BCA3 BCA3 BCA3 3C23 4C7B BCA3 6CBB      E6B2BB       E6B2BB       F3B0ABAC
8EA2     8EA2 8EA2 8EA2 8EA2 4C79 BCA4 BCA4 BCA4 BCA4 3C24 4CE0 BCA4 723E      E788BE       E788BE       F3B0ABAD
8EA3     8EA3 8EA3 8EA3 8EA3 5249 BCA5 BCA5 BCA5 BCA5 3C25 4C6C BCA5 74BD      E792BD       E792BD       F3B0ABAE
8EA4     8EA4 8EA4 8EA4 8EA4 5182 BCA6 BCA6 BCA6 BCA6 3C26 4C50 BCA6 75D4      E79794       E79794       F3B0ABAF
8EA5     8EA5 8EA5 8EA5 8EA5 4A62 BCA7 BCA7 BCA7 BCA7 3C27 4C7D BCA7 78C1      E7A381       E7A381       F3B0ABB0
8EA6     8EA6 8EA6 8EA6 8EA6 4853 BCA8 BCA8 BCA8 BCA8 3C28 4C4D BCA8 793A      E7A4BA       E7A4BA       F3B0ABB1
8EA7     8EA7 8EA7 8EA7 8EA7 51E6 BCA9 BCA9 BCA9 BCA9 3C29 4C5D BCA9 800C      E8808C       E8808C       F3B0ABB2
8EA8     8EA8 8EA8 8EA8 8EA8 4CAB BCAA BCAA BCAA BCAA 3C2A 4C5C BCAA 8033      E880B3       E880B3       F3B0ABB3
8EA9     8EA9 8EA9 8EA9 8EA9 45C3 BCAB BCAB BCAB BCAB 3C2B 4C4E BCAB 81EA      E887AA       E887AA       F3B0ABB4
8EAA     8EAA 8EAA 8EAA 8EAA 4FE3 BCAC BCAC BCAC BCAC 3C2C 4C6B BCAC 8494      E89294       E89294       F3B0ABB5
8EAB     8EAB 8EAB 8EAB 8EAB 4D74 BCAD BCAD BCAD BCAD 3C2D 4C60 BCAD 8F9E      E8BE9E       E8BE9E       F3B0ABB6
8EAC     8EAC 8EAC 8EAC 8EAC 506A BCAE BCAE BCAE BCAE 3C2E 4C4B BCAE 6C50      E6B190       E6B190       F3B0ABB7
8EAD     8EAD 8EAD 8EAD 8EAD 4697 BCAF BCAF BCAF BCAF 3C2F 4C61 BCAF 9E7F      E9B9BF       E9B9BF       F3B0ABB8
8EAE     8EAE 8EAE 8EAE 8EAE 468F BCB0 BCB0 BCB0 BCB0 3C30 4CF0 BCB0 5F0F      E5BC8F       E5BC8F       F3B0ABB9
8EAF     8EAF 8EAF 8EAF 8EAF 4A75 BCB1 BCB1 BCB1 BCB1 3C31 4CF1 BCB1 8B58      E8AD98       E8AD98       F3B0ABBA
8EB0     8EB0 8EB0 8EB0 8EB0 50BA BCB2 BCB2 BCB2 BCB2 3C32 4CF2 BCB2 9D2B      E9B4AB       E9B4AB       F3B0ABBB
8EB1     8EB1 8EB1 8EB1 8EB1 545C BCB3 BCB3 BCB3 BCB3 3C33 4CF3 BCB3 7AFA      E7ABBA       E7ABBA       F3B0ABBC
8EB2     8EB2 8EB2 8EB2 8EB2 4BA6 BCB4 BCB4 BCB4 BCB4 3C34 4CF4 BCB4 8EF8      E8BBB8       E8BBB8       F3B0ABBD
8EB3     8EB3 8EB3 8EB3 8EB3 4C75 BCB5 BCB5 BCB5 BCB5 3C35 4CF5 BCB5 5B8D      E5AE8D       E5AE8D       F3B0ABBE
8EB4     8EB4 8EB4 8EB4 8EB4 52A5 BCB6 BCB6 BCB6 BCB6 3C36 4CF6 BCB6 96EB      E99BAB       E99BAB       F3B0ABBF
8EB5     8EB5 8EB5 8EB5 8EB5 4547 BCB7 BCB7 BCB7 BCB7 3C37 4CF7 BCB7 4E03      E4B883       E4B883       F3B0AC80
8EB6     8EB6 8EB6 8EB6 8EB6 5042 BCB8 BCB8 BCB8 BCB8 3C38 4CF8 BCB8 53F1      E58FB1       E58FB1       F3B0AC81
8EB7     8EB7 8EB7 8EB7 8EB7 4D57 BCB9 BCB9 BCB9 BCB9 3C39 4CF9 BCB9 57F7      E59FB7       E59FB7       F3B0AC82
8EB8     8EB8 8EB8 8EB8 8EB8 4941 BCBA BCBA BCBA BCBA 3C3A 4C7A BCBA 5931      E5A4B1       E5A4B1       F3B0AC83
8EB9     8EB9 8EB9 8EB9 8EB9 53E0 BCBB BCBB BCBB BCBB 3C3B 4C5E BCBB 5AC9      E5AB89       E5AB89       F3B0AC84
8EBA     8EBA 8EBA 8EBA 8EBA 47B8 BCBC BCBC BCBC BCBC 3C3C 4C4C BCBC 5BA4      E5AEA4       E5AEA4       F3B0AC85
8EBB     8EBB 8EBB 8EBB 8EBB 5469 BCBD BCBD BCBD BCBD 3C3D 4C7E BCBD 6089      E68289       E68289       F3B0AC86
8EBC     8EBC 8EBC 8EBC 8EBC 4C90 BCBE BCBE BCBE BCBE 3C3E 4C6E BCBE 6E7F      E6B9BF       E6B9BF       F3B0AC87
8EBD     8EBD 8EBD 8EBD 8EBD 4DAE BCBF BCBF BCBF BCBF 3C3F 4C6F BCBF 6F06      E6BC86       E6BC86       F3B0AC88
8EBE     8EBE 8EBE 8EBE 8EBE 5142 BCC0 BCC0 BCC0 BCC0 3C40 4C7C BCC0 75BE      E796BE       E796BE       F3B0AC89
8EBF     8EBF 8EBF 8EBF 8EBF 46EC BCC1 BCC1 BCC1 BCC1 3C41 4CC1 BCC1 8CEA      E8B3AA       E8B3AA       F3B0AC8A
8EC0     8EC0 8EC0 8EC0 8EC0 45ED BCC2 BCC2 BCC2 BCC2 3C42 4CC2 BCC2 5B9F      E5AE9F       E5AE9F       F3B0AC8B
8EC1     8EC1 8EC1 8EC1 8EC1 52BD BCC3 BCC3 BCC3 BCC3 3C43 4CC3 BCC3 8500      E89480       E89480       F3B0AC8C
8EC2     8EC2 8EC2 8EC2 8EC2 4875 BCC4 BCC4 BCC4 BCC4 3C44 4CC4 BCC4 7BE0      E7AFA0       E7AFA0       F3B0AC8D
8EC3     8EC3 8EC3 8EC3 8EC3 54D2 BCC5 BCC5 BCC5 BCC5 3C45 4CC5 BCC5 5072      E581B2       E581B2       F3B0AC8E
8EC4     8EC4 8EC4 8EC4 8EC4 47A7 BCC6 BCC6 BCC6 BCC6 3C46 4CC6 BCC6 67F4      E69FB4       E69FB4       F3B0AC8F
8EC5     8EC5 8EC5 8EC5 8EC5 4972 BCC7 BCC7 BCC7 BCC7 3C47 4CC7 BCC7 829D      E88A9D       E88A9D       F3B0AC90
8EC6     8EC6 8EC6 8EC6 8EC6 55C0 BCC8 BCC8 BCC8 4AE5 B8E8 88CA BCC8 5C61      E5B1A1       E5B1A1       F3B0AC91
8EC7     8EC7 8EC7 8EC7 8EC7 62A6 E9A2 BCC9 BCC9 E9A2 6922 687F BCC9 854A      E8958A       E8958A       F3B0AC92
8EC8     8EC8 8EC8 8EC8 8EC8 4F6B BCCA BCCA BCCA BCCA 3C4A 4CD1 BCCA 7E1E      E7B89E       E7B89E       F3B0AC93
8EC9     8EC9 8EC9 8EC9 8EC9 4C58 BCCB BCCB BCCB BCCB 3C4B 4CD2 BCCB 820E      E8888E       E8888E       F3B0AC94
8ECA     8ECA 8ECA 8ECA 8ECA 48C4 BCCC BCCC BCCC BCCC 3C4C 4CD3 BCCC 5199      E58699       E58699       F3B0AC95
8ECB     8ECB 8ECB 8ECB 8ECB 498A BCCD BCCD BCCD BCCD 3C4D 4CD4 BCCD 5C04      E5B084       E5B084       F3B0AC96
8ECC     8ECC 8ECC 8ECC 8ECC 4BB3 BCCE BCCE BCCE BCCE 3C4E 4CD5 BCCE 6368      E68DA8       E68DA8       F3B0AC97
8ECD     8ECD 8ECD 8ECD 8ECD 5151 BCCF BCCF BCCF BCCF 3C4F 4CD6 BCCF 8D66      E8B5A6       E8B5A6       F3B0AC98
8ECE     8ECE 8ECE 8ECE 8ECE 4BD1 BCD0 BCD0 BCD0 BCD0 3C50 4CD7 BCD0 659C      E6969C       E6969C       F3B0AC99
8ECF     8ECF 8ECF 8ECF 8ECF 4C4C BCD1 BCD1 BCD1 BCD1 3C51 4CD8 BCD1 716E      E785AE       E785AE       F3B0AC9A
8ED0     8ED0 8ED0 8ED0 8ED0 4663 BCD2 BCD2 BCD2 BCD2 3C52 4CD9 BCD2 793E      E7A4BE       E7A4BE       F3B0AC9B
8ED1     8ED1 8ED1 8ED1 8ED1 4F47 BCD3 BCD3 BCD3 BCD3 3C53 4CE2 BCD3 7D17      E7B497       E7B497       F3B0AC9C
8ED2     8ED2 8ED2 8ED2 8ED2 4670 BCD4 BCD4 BCD4 BCD4 3C54 4CE3 BCD4 8005      E88085       E88085       F3B0AC9D
8ED3     8ED3 8ED3 8ED3 8ED3 4DBB BCD5 BCD5 BCD5 BCD5 3C55 4CE4 BCD5 8B1D      E8AC9D       E8AC9D       F3B0AC9E
8ED4     8ED4 8ED4 8ED4 8ED4 4653 BCD6 BCD6 BCD6 BCD6 3C56 4CE5 BCD6 8ECA      E8BB8A       E8BB8A       F3B0AC9F
8ED5     8ED5 8ED5 8ED5 8ED5 5458 BCD7 BCD7 BCD7 BCD7 3C57 4CE6 BCD7 906E      E981AE       E981AE       F3B0ACA0
8ED6     8ED6 8ED6 8ED6 8ED6 4FEA BCD8 BCD8 BCD8 BCD8 3C58 4CE7 BCD8 86C7      E89B87       E89B87       F3B0ACA1
8ED7     8ED7 8ED7 8ED7 8ED7 4EB8 BCD9 BCD9 BCD9 BCD9 3C59 4CE8 BCD9 90AA      E982AA       E982AA       F3B0ACA2
8ED8     8ED8 8ED8 8ED8 8ED8 4AD0 BCDA BCDA BCDA BCDA 3C5A 4CE9 BCDA 501F      E5809F       E5809F       F3B0ACA3
8ED9     8ED9 8ED9 8ED9 8ED9 5470 BCDB BCDB BCDB BCDB 3C5B 4C4A BCDB 52FA      E58BBA       E58BBA       F3B0ACA4
8EDA     8EDA 8EDA 8EDA 8EDA 4C6B BCDC BCDC BCDC BCDC 3C5C 4C5B BCDC 5C3A      E5B0BA       E5B0BA       F3B0ACA5
8EDB     8EDB 8EDB 8EDB 8EDB 54C1 BCDD BCDD BCDD BCDD 3C5D 4C5A BCDD 6753      E69D93       E69D93       F3B0ACA6
8EDC     8EDC 8EDC 8EDC 8EDC 546A BCDE BCDE BCDE BCDE 3C5E 4C5F BCDE 707C      E781BC       E781BC       F3B0ACA7
8EDD     8EDD 8EDD 8EDD 8EDD 53A9 BCDF BCDF BCDF BCDF 3C5F 4C6D BCDF 7235      E788B5       E788B5       F3B0ACA8
8EDE     8EDE 8EDE 8EDE 8EDE 52BB BCE0 BCE0 BCE0 BCE0 3C60 4C79 BCE0 914C      E9858C       E9858C       F3B0ACA9
8EDF     8EDF 8EDF 8EDF 8EDF 4D79 BCE1 BCE1 BCE1 BCE1 3C61 4C57 BCE1 91C8      E98788       E98788       F3B0ACAA
8EE0     8EE0 8EE0 8EE0 8EE0 5257 BCE2 BCE2 BCE2 BCE2 3C62 4C59 BCE2 932B      E98CAB       E98CAB       F3B0ACAB
8EE1     8EE1 8EE1 8EE1 8EE1 46D6 BCE3 BCE3 BCE3 BCE3 3C63 4C62 BCE3 82E5      E88BA5       E88BA5       F3B0ACAC
8EE2     8EE2 8EE2 8EE2 8EE2 4FE9 BCE4 BCE4 BCE4 BCE4 3C64 4C63 BCE4 5BC2      E5AF82       E5AF82       F3B0ACAD
8EE3     8EE3 8EE3 8EE3 8EE3 4B6E BCE5 BCE5 BCE5 BCE5 3C65 4C64 BCE5 5F31      E5BCB1       E5BCB1       F3B0ACAE
8EE4     8EE4 8EE4 8EE4 8EE4 5472 BCE6 BCE6 BCE6 BCE6 3C66 4C65 BCE6 60F9      E683B9       E683B9       F3B0ACAF
8EE5     8EE5 8EE5 8EE5 8EE5 4762 BCE7 BCE7 BCE7 BCE7 3C67 4C66 BCE7 4E3B      E4B8BB       E4B8BB       F3B0ACB0
8EE6     8EE6 8EE6 8EE6 8EE6 46A6 BCE8 BCE8 BCE8 BCE8 3C68 4C67 BCE8 53D6      E58F96       E58F96       F3B0ACB1
8EE7     8EE7 8EE7 8EE7 8EE7 46F0 BCE9 BCE9 BCE9 BCE9 3C69 4C68 BCE9 5B88      E5AE88       E5AE88       F3B0ACB2
8EE8     8EE8 8EE8 8EE8 8EE8 45CB BCEA BCEA BCEA BCEA 3C6A 4C69 BCEA 624B      E6898B       E6898B       F3B0ACB3
8EE9     8EE9 8EE9 8EE9 8EE9 4D42 BCEB BCEB BCEB BCEB 3C6B 4C70 BCEB 6731      E69CB1       E69CB1       F3B0ACB4
8EEA     8EEA 8EEA 8EEA 8EEA 4865 BCEC BCEC BCEC BCEC 3C6C 4C71 BCEC 6B8A      E6AE8A       E6AE8A       F3B0ACB5
8EEB     8EEB 8EEB 8EEB 8EEB 4A7B BCED BCED BCED BCED 3C6D 4C72 BCED 72E9      E78BA9       E78BA9       F3B0ACB6
8EEC     8EEC 8EEC 8EEC 8EEC 4B99 BCEE BCEE BCEE BCEE 3C6E 4C73 BCEE 73E0      E78FA0       E78FA0       F3B0ACB7
8EED     8EED 8EED 8EED 8EED 4692 BCEF BCEF BCEF BCEF 3C6F 4C74 BCEF 7A2E      E7A8AE       E7A8AE       F3B0ACB8
8EEE     8EEE 8EEE 8EEE 8EEE 51CE BCF0 BCF0 BCF0 BCF0 3C70 4C75 BCF0 816B      E885AB       E885AB       F3B0ACB9
8EEF     8EEF 8EEF 8EEF 8EEF 4CED BCF1 BCF1 BCF1 BCF1 3C71 4C76 BCF1 8DA3      E8B6A3       E8B6A3       F3B0ACBA
8EF0     8EF0 8EF0 8EF0 8EF0 4758 BCF2 BCF2 BCF2 BCF2 3C72 4C77 BCF2 9152      E98592       E98592       F3B0ACBB
8EF1     8EF1 8EF1 8EF1 8EF1 49BE BCF3 BCF3 BCF3 BCF3 3C73 4C78 BCF3 9996      E9A696       E9A696       F3B0ACBC
8EF2     8EF2 8EF2 8EF2 8EF2 546F BCF4 BCF4 BCF4 BCF4 3C74 4C80 BCF4 5112      E58492       E58492       F3B0ACBD
8EF3     8EF3 8EF3 8EF3 8EF3 484C BCF5 BCF5 BCF5 BCF5 3C75 4C8B BCF5 53D7      E58F97       E58F97       F3B0ACBE
8EF4     8EF4 8EF4 8EF4 8EF4 548D BCF6 BCF6 BCF6 BCF6 3C76 4C9B BCF6 546A      E591AA       E591AA       F3B0ACBF
8EF5     8EF5 8EF5 8EF5 8EF5 46B2 BCF7 BCF7 BCF7 BCF7 3C77 4C9C BCF7 5BFF      E5AFBF       E5AFBF       F3B0AD80
8EF6     8EF6 8EF6 8EF6 8EF6 4C61 BCF8 BCF8 BCF8 BCF8 3C78 4CA0 BCF8 6388      E68E88       E68E88       F3B0AD81
8EF7     8EF7 8EF7 8EF7 8EF7 46CF BCF9 BCF9 BCF9 BCF9 3C79 4CAB BCF9 6A39      E6A8B9       E6A8B9       F3B0AD82
8EF8     8EF8 8EF8 8EF8 8EF8 51A3 BCFA BCFA BCFA BCFA 3C7A 4CB0 BCFA 7DAC      E7B6AC       E7B6AC       F3B0AD83
8EF9     8EF9 8EF9 8EF9 8EF9 47FB BCFB BCFB BCFB BCFB 3C7B 4CC0 BCFB 9700      E99C80       E99C80       F3B0AD84
8EFA     8EFA 8EFA 8EFA 8EFA 5065 BCFC BCFC BCFC BCFC 3C7C 4C6A BCFC 56DA      E59B9A       E59B9A       F3B0AD85
8EFB     8EFB 8EFB 8EFB 8EFB 4791 BCFD BCFD BCFD BCFD 3C7D 4CD0 BCFD 53CE      E58F8E       E58F8E       F3B0AD86
8EFC     8EFC 8EFC 8EFC 8EFC 48A1 BCFE BCFE BCFE BCFE 3C7E 4CA1 BCFE 5468      E591A8       E591A8       F3B0AD87
8F40     8F40 8F40 8F40 8F40 47DD BDA1 BDA1 BDA1 BDA1 3D21 7E4F BDA1 5B97      E5AE97       E5AE97       F3B0AD88
8F41     8F41 8F41 8F41 8F41 4BEE BDA2 BDA2 BDA2 BDA2 3D22 7E7F BDA2 5C31      E5B0B1       E5B0B1       F3B0AD89
8F42     8F42 8F42 8F42 8F42 4741 BDA3 BDA3 BDA3 BDA3 3D23 7E7B BDA3 5DDE      E5B79E       E5B79E       F3B0AD8A
8F43     8F43 8F43 8F43 8F43 46DE BDA4 BDA4 BDA4 BDA4 3D24 7EE0 BDA4 4FEE      E4BFAE       E4BFAE       F3B0AD8B
8F44     8F44 8F44 8F44 8F44 5282 BDA5 BDA5 BDA5 BDA5 3D25 7E6C BDA5 6101      E68481       E68481       F3B0AD8C
8F45     8F45 8F45 8F45 8F45 4DD7 BDA6 BDA6 BDA6 BDA6 3D26 7E50 BDA6 62FE      E68BBE       E68BBE       F3B0AD8D
8F46     8F46 8F46 8F46 8F46 4B72 BDA7 BDA7 BDA7 BDA7 3D27 7E7D BDA7 6D32      E6B4B2       E6B4B2       F3B0AD8E
8F47     8F47 8F47 8F47 8F47 45F1 BDA8 BDA8 BDA8 BDA8 3D28 7E4D BDA8 79C0      E7A780       E7A780       F3B0AD8F
8F48     8F48 8F48 8F48 8F48 4658 BDA9 BDA9 BDA9 BDA9 3D29 7E5D BDA9 79CB      E7A78B       E7A78B       F3B0AD90
8F49     8F49 8F49 8F49 8F49 49F4 BDAA BDAA BDAA BDAA 3D2A 7E5C BDAA 7D42      E7B582       E7B582       F3B0AD91
8F4A     8F4A 8F4A 8F4A 8F4A 52DA BDAB BDAB BDAB 5FE0 C9D3 9AA8 BDAB 7E4D      E7B98D       E7B98D       F3B0AD92
8F4B     8F4B 8F4B 8F4B 8F4B 4A69 BDAC BDAC BDAC BDAC 3D2C 7E6B BDAC 7FD2      E7BF92       E7BF92       F3B0AD93
8F4C     8F4C 8F4C 8F4C 8F4C 4DB3 BDAD BDAD BDAD BDAD 3D2D 7E60 BDAD 81ED      E887AD       E887AD       F3B0AD94
8F4D     8F4D 8F4D 8F4D 8F4D 4BB8 BDAE BDAE BDAE BDAE 3D2E 7E4B BDAE 821F      E8889F       E8889F       F3B0AD95
8F4E     8F4E 8F4E 8F4E 8F4E 4F93 BDAF BDAF BDAF BDAF 3D2F 7E61 BDAF 8490      E89290       E89290       F3B0AD96
8F4F     8F4F 8F4F 8F4F 8F4F 4B88 BDB0 BDB0 BDB0 BDB0 3D30 7EF0 BDB0 8846      E8A186       E8A186       F3B0AD97
8F50     8F50 8F50 8F50 8F50 4DDF BDB1 BDB1 BDB1 BDB1 3D31 7EF1 BDB1 8972      E8A5B2       E8A5B2       F3B0AD98
8F51     8F51 8F51 8F51 8F51 53FE BDB2 BDB2 BDB2 BDB2 3D32 7EF2 BDB2 8B90      E8AE90       E8AE90       F3B0AD99
8F52     8F52 8F52 8F52 8F52 52F6 BDB3 BDB3 BDB3 BDB3 3D33 7EF3 BDB3 8E74      E8B9B4       E8B9B4       F3B0AD9A
8F53     8F53 8F53 8F53 8F53 55A6 BDB4 BDB4 BDB4 BDB4 3D34 7EF4 BDB4 8F2F      E8BCAF       E8BCAF       F3B0AD9B
8F54     8F54 8F54 8F54 8F54 4AD3 BDB5 BDB5 BDB5 BDB5 3D35 7EF5 BDB5 9031      E980B1       E980B1       F3B0AD9C
8F55     8F55 8F55 8F55 8F55 5548 BDB6 BDB6 BDB6 BDB6 3D36 7EF6 BDB6 914B      E9858B       E9858B       F3B0AD9D
8F56     8F56 8F56 8F56 8F56 5149 BDB7 BDB7 BDB7 BDB7 3D37 7EF7 BDB7 916C      E985AC       E985AC       F3B0AD9E
8F57     8F57 8F57 8F57 8F57 47A0 BDB8 BDB8 BDB8 BDB8 3D38 7EF8 BDB8 96C6      E99B86       E99B86       F3B0AD9F
8F58     8F58 8F58 8F58 8F58 536F BDB9 BDB9 BDB9 BDB9 3D39 7EF9 BDB9 919C      E9869C       E9869C       F3B0ADA0
8F59     8F59 8F59 8F59 8F59 53D9 BDBA BDBA BDBA BDBA 3D3A 7E7A BDBA 4EC0      E4BB80       E4BB80       F3B0ADA1
8F5A     8F5A 8F5A 8F5A 8F5A 46E0 BDBB BDBB BDBB BDBB 3D3B 7E5E BDBB 4F4F      E4BD8F       E4BD8F       F3B0ADA2
8F5B     8F5B 8F5B 8F5B 8F5B 48C9 BDBC BDBC BDBC BDBC 3D3C 7E4C BDBC 5145      E58585       E58585       F3B0ADA3
8F5C     8F5C 8F5C 8F5C 8F5C 454A BDBD BDBD BDBD BDBD 3D3D 7E7E BDBD 5341      E58D81       E58D81       F3B0ADA4
8F5D     8F5D 8F5D 8F5D 8F5D 48FB BDBE BDBE BDBE BDBE 3D3E 7E6E BDBE 5F93      E5BE93       E5BE93       F3B0ADA5
8F5E     8F5E 8F5E 8F5E 8F5E 517F BDBF BDBF BDBF BDBF 3D3F 7E6F BDBF 620E      E6888E       E6888E       F3B0ADA6
8F5F     8F5F 8F5F 8F5F 8F5F 4DEA BDC0 BDC0 BDC0 BDC0 3D40 7E7C BDC0 67D4      E69F94       E69F94       F3B0ADA7
8F60     8F60 8F60 8F60 8F60 4D84 BDC1 BDC1 BDC1 BDC1 3D41 7EC1 BDC1 6C41      E6B181       E6B181       F3B0ADA8
8F61     8F61 8F61 8F61 8F61 4892 BDC2 BDC2 BDC2 BDC2 3D42 7EC2 BDC2 6E0B      E6B88B       E6B88B       F3B0ADA9
8F62     8F62 8F62 8F62 8F62 4F65 BDC3 BDC3 BDC3 BDC3 3D43 7EC3 BDC3 7363      E78DA3       E78DA3       F3B0ADAA
8F63     8F63 8F63 8F63 8F63 4E93 BDC4 BDC4 BDC4 BDC4 3D44 7EC4 BDC4 7E26      E7B8A6       E7B8A6       F3B0ADAB
8F64     8F64 8F64 8F64 8F64 45A3 BDC5 BDC5 BDC5 BDC5 3D45 7EC5 BDC5 91CD      E9878D       E9878D       F3B0ADAC
8F65     8F65 8F65 8F65 8F65 4CE2 BDC6 BDC6 BDC6 BDC6 3D46 7EC6 BDC6 9283      E98A83       E98A83       F3B0ADAD
8F66     8F66 8F66 8F66 8F66 4FA5 BDC7 BDC7 BDC7 BDC7 3D47 7EC7 BDC7 53D4      E58F94       E58F94       F3B0ADAE
8F67     8F67 8F67 8F67 8F67 54DD BDC8 BDC8 BDC8 BDC8 3D48 7EC8 BDC8 5919      E5A499       E5A499       F3B0ADAF
8F68     8F68 8F68 8F68 8F68 4882 BDC9 BDC9 BDC9 BDC9 3D49 7EC9 BDC9 5BBF      E5AEBF       E5AEBF       F3B0ADB0
8F69     8F69 8F69 8F69 8F69 4ABF BDCA BDCA BDCA BDCA 3D4A 7ED1 BDCA 6DD1      E6B791       E6B791       F3B0ADB1
8F6A     8F6A 8F6A 8F6A 8F6A 4CEB BDCB BDCB BDCB BDCB 3D4B 7ED2 BDCB 795D      E7A59D       E7A59D       F3B0ADB2
8F6B     8F6B 8F6B 8F6B 8F6B 49F7 BDCC BDCC BDCC BDCC 3D4C 7ED3 BDCC 7E2E      E7B8AE       E7B8AE       F3B0ADB3
8F6C     8F6C 8F6C 8F6C 8F6C 4FC8 BDCD BDCD BDCD BDCD 3D4D 7ED4 BDCD 7C9B      E7B29B       E7B29B       F3B0ADB4
8F6D     8F6D 8F6D 8F6D 8F6D 5279 BDCE BDCE BDCE BDCE 3D4E 7ED5 BDCE 587E      E5A1BE       E5A1BE       F3B0ADB5
8F6E     8F6E 8F6E 8F6E 8F6E 4EBF BDCF BDCF BDCF BDCF 3D4F 7ED6 BDCF 719F      E7869F       E7869F       F3B0ADB6
8F6F     8F6F 8F6F 8F6F 8F6F 459B BDD0 BDD0 BDD0 BDD0 3D50 7ED7 BDD0 51FA      E587BA       E587BA       F3B0ADB7
8F70     8F70 8F70 8F70 8F70 486C BDD1 BDD1 BDD1 BDD1 3D51 7ED8 BDD1 8853      E8A193       E8A193       F3B0ADB8
8F71     8F71 8F71 8F71 8F71 4A9C BDD2 BDD2 BDD2 BDD2 3D52 7ED9 BDD2 8FF0      E8BFB0       E8BFB0       F3B0ADB9
8F72     8F72 8F72 8F72 8F72 4660 BDD3 BDD3 BDD3 BDD3 3D53 7EE2 BDD3 4FCA      E4BF8A       E4BF8A       F3B0ADBA
8F73     8F73 8F73 8F73 8F73 5072 BDD4 BDD4 BDD4 BDD4 3D54 7EE3 BDD4 5CFB      E5B3BB       E5B3BB       F3B0ADBB
8F74     8F74 8F74 8F74 8F74 4662 BDD5 BDD5 BDD5 BDD5 3D55 7EE4 BDD5 6625      E698A5       E698A5       F3B0ADBC
8F75     8F75 8F75 8F75 8F75 4DB0 BDD6 BDD6 BDD6 BDD6 3D56 7EE5 BDD6 77AC      E79EAC       E79EAC       F3B0ADBD
8F76     8F76 8F76 8F76 8F76 546B BDD7 BDD7 BDD7 BDD7 3D57 7EE6 BDD7 7AE3      E7ABA3       E7ABA3       F3B0ADBE
8F77     8F77 8F77 8F77 8F77 4FB8 BDD8 BDD8 BDD8 BDD8 3D58 7EE7 BDD8 821C      E8889C       E8889C       F3B0ADBF
8F78     8F78 8F78 8F78 8F78 4CC0 BDD9 BDD9 BDD9 BDD9 3D59 7EE8 BDD9 99FF      E9A7BF       E9A7BF       F3B0AE80
8F79     8F79 8F79 8F79 8F79 4F80 BDDA BDDA BDDA BDDA 3D5A 7EE9 BDDA 51C6      E58786       E58786       F3B0AE81
8F7A     8F7A 8F7A 8F7A 8F7A 4EF5 BDDB BDDB BDDB BDDB 3D5B 7E4A BDDB 5FAA      E5BEAA       E5BEAA       F3B0AE82
8F7B     8F7B 8F7B 8F7B 8F7B 4D67 BDDC BDDC BDDC BDDC 3D5C 7E5B BDDC 65EC      E697AC       E697AC       F3B0AE83
8F7C     8F7C 8F7C 8F7C 8F7C 4FAE BDDD BDDD BDDD BDDD 3D5D 7E5A BDDD 696F      E6A5AF       E6A5AF       F3B0AE84
8F7D     8F7D 8F7D 8F7D 8F7D 53A4 BDDE BDDE BDDE BDDE 3D5E 7E5F BDDE 6B89      E6AE89       E6AE89       F3B0AE85
8F7E     8F7E 8F7E 8F7E 8F7E 4894 BDDF BDDF BDDF BDDF 3D5F 7E6D BDDF 6DF3      E6B7B3       E6B7B3       F3B0AE86
8F80     8F80 8F80 8F80 8F80 46CE BDE0 BDE0 BDE0 BDE0 3D60 7E79 BDE0 6E96      E6BA96       E6BA96       F3B0AE87
8F81     8F81 8F81 8F81 8F81 4AD2 BDE1 BDE1 BDE1 BDE1 3D61 7E57 BDE1 6F64      E6BDA4       E6BDA4       F3B0AE88
8F82     8F82 8F82 8F82 8F82 5082 BDE2 BDE2 BDE2 BDE2 3D62 7E59 BDE2 76FE      E79BBE       E79BBE       F3B0AE89
8F83     8F83 8F83 8F83 8F83 47A6 BDE3 BDE3 BDE3 BDE3 3D63 7E62 BDE3 7D14      E7B494       E7B494       F3B0AE8A
8F84     8F84 8F84 8F84 8F84 4D47 BDE4 BDE4 BDE4 BDE4 3D64 7E63 BDE4 5DE1      E5B7A1       E5B7A1       F3B0AE8B
8F85     8F85 8F85 8F85 8F85 526E BDE5 BDE5 BDE5 BDE5 3D65 7E64 BDE5 9075      E981B5       E981B5       F3B0AE8C
8F86     8F86 8F86 8F86 8F86 51A9 BDE6 BDE6 BDE6 BDE6 3D66 7E65 BDE6 9187      E98687       E98687       F3B0AE8D
8F87     8F87 8F87 8F87 8F87 47B2 BDE7 BDE7 BDE7 BDE7 3D67 7E66 BDE7 9806      E9A086       E9A086       F3B0AE8E
8F88     8F88 8F88 8F88 8F88 48E6 BDE8 BDE8 BDE8 BDE8 3D68 7E67 BDE8 51E6      E587A6       E587A6       F3B0AE8F
8F89     8F89 8F89 8F89 8F89 474B BDE9 BDE9 BDE9 BDE9 3D69 7E68 BDE9 521D      E5889D       E5889D       F3B0AE90
8F8A     8F8A 8F8A 8F8A 8F8A 458C BDEA BDEA BDEA BDEA 3D6A 7E69 BDEA 6240      E68980       E68980       F3B0AE91
8F8B     8F8B 8F8B 8F8B 8F8B 4FBE BDEB BDEB BDEB BDEB 3D6B 7E70 BDEB 6691      E69A91       E69A91       F3B0AE92
8F8C     8F8C 8F8C 8F8C 8F8C 5152 BDEC BDEC BDEC BDEC 3D6C 7E71 BDEC 66D9      E69B99       E69B99       F3B0AE93
8F8D     8F8D 8F8D 8F8D 8F8D 50E4 BDED BDED BDED 55AF C1E3 92B5 BDED 6E1A      E6B89A       E6B89A       F3B0AE94
8F8E     8F8E 8F8E 8F8E 8F8E 50C1 BDEE BDEE BDEE BDEE 3D6E 7E73 BDEE 5EB6      E5BAB6       E5BAB6       F3B0AE95
8F8F     8F8F 8F8F 8F8F 8F8F 4A9B BDEF BDEF BDEF BDEF 3D6F 7E74 BDEF 7DD2      E7B792       E7B792       F3B0AE96
8F90     8F90 8F90 8F90 8F90 4C99 BDF0 BDF0 BDF0 BDF0 3D70 7E75 BDF0 7F72      E7BDB2       E7BDB2       F3B0AE97
8F91     8F91 8F91 8F91 8F91 479D BDF1 BDF1 BDF1 BDF1 3D71 7E76 BDF1 66F8      E69BB8       E69BB8       F3B0AE98
8F92     8F92 8F92 8F92 8F92 53F9 BDF2 BDF2 BDF2 BDF2 3D72 7E77 BDF2 85AF      E896AF       E896AF       F3B0AE99
8F93     8F93 8F93 8F93 8F93 55E2 BDF3 BDF3 BDF3 BDF3 3D73 7E78 BDF3 85F7      E897B7       E897B7       F3B0AE9A
8F94     8F94 8F94 8F94 8F94 47E2 BDF4 BDF4 BDF4 BDF4 3D74 7E80 BDF4 8AF8      E8ABB8       E8ABB8       F3B0AE9B
8F95     8F95 8F95 8F95 8F95 4671 BDF5 BDF5 BDF5 BDF5 3D75 7E8B BDF5 52A9      E58AA9       E58AA9       F3B0AE9C
8F96     8F96 8F96 8F96 8F96 50F5 BDF6 BDF6 BDF6 BDF6 3D76 7E9B BDF6 53D9      E58F99       E58F99       F3B0AE9D
8F97     8F97 8F97 8F97 8F97 4699 BDF7 BDF7 BDF7 BDF7 3D77 7E9C BDF7 5973      E5A5B3       E5A5B3       F3B0AE9E
8F98     8F98 8F98 8F98 8F98 4D4F BDF8 BDF8 BDF8 BDF8 3D78 7EA0 BDF8 5E8F      E5BA8F       E5BA8F       F3B0AE9F
8F99     8F99 8F99 8F99 8F99 50B3 BDF9 BDF9 BDF9 BDF9 3D79 7EAB BDF9 5F90      E5BE90       E5BE90       F3B0AEA0
8F9A     8F9A 8F9A 8F9A 8F9A 5298 BDFA BDFA BDFA BDFA 3D7A 7EB0 BDFA 6055      E68195       E68195       F3B0AEA1
8F9B     8F9B 8F9B 8F9B 8F9B 52C0 BDFB BDFB BDFB BDFB 3D7B 7EC0 BDFB 92E4      E98BA4       E98BA4       F3B0AEA2
8F9C     8F9C 8F9C 8F9C 8F9C 496D BDFC BDFC BDFC BDFC 3D7C 7E6A BDFC 9664      E999A4       E999A4       F3B0AEA3
8F9D     8F9D 8F9D 8F9D 8F9D 4C5B BDFD BDFD BDFD BDFD 3D7D 7ED0 BDFD 50B7      E582B7       E582B7       F3B0AEA4
8F9E     8F9E 8F9E 8F9E 8F9E 4AA2 BDFE BDFE BDFE BDFE 3D7E 7EA1 BDFE 511F      E5849F       E5849F       F3B0AEA5
8F9F     8F9F 8F9F 8F9F 8F9F 45B5 BEA1 BEA1 BEA1 BEA1 3E21 6E4F BEA1 52DD      E58B9D       E58B9D       F3B0AEA6
8FA0     8FA0 8FA0 8FA0 8FA0 4E6B BEA2 BEA2 BEA2 BEA2 3E22 6E7F BEA2 5320      E58CA0       E58CA0       F3B0AEA7
8FA1     8FA1 8FA1 8FA1 8FA1 4E47 BEA3 BEA3 BEA3 BEA3 3E23 6E7B BEA3 5347      E58D87       E58D87       F3B0AEA8
8FA2     8FA2 8FA2 8FA2 8FA2 4F86 BEA4 BEA4 BEA4 BEA4 3E24 6EE0 BEA4 53EC      E58FAC       E58FAC       F3B0AEA9
8FA3     8FA3 8FA3 8FA3 8FA3 5568 BEA5 BEA5 BEA5 BEA5 3E25 6E6C BEA5 54E8      E593A8       E593A8       F3B0AEAA
8FA4     8FA4 8FA4 8FA4 8FA4 46BE BEA6 BEA6 BEA6 BEA6 3E26 6E50 BEA6 5546      E59586       E59586       F3B0AEAB
8FA5     8FA5 8FA5 8FA5 8FA5 4D43 BEA7 BEA7 BEA7 BEA7 3E27 6E7D BEA7 5531      E594B1       E594B1       F3B0AEAC
8FA6     8FA6 8FA6 8FA6 8FA6 54EC BEA8 BEA8 BEA8 BEA8 3E28 6E4D BEA8 5617      E59897       E59897       F3B0AEAD
8FA7     8FA7 8FA7 8FA7 8FA7 4ECA BEA9 BEA9 BEA9 BEA9 3E29 6E5D BEA9 5968      E5A5A8       E5A5A8       F3B0AEAE
8FA8     8FA8 8FA8 8FA8 8FA8 53AF BEAA BEAA BEAA BEAA 3E2A 6E5C BEAA 59BE      E5A6BE       E5A6BE       F3B0AEAF
8FA9     8FA9 8FA9 8FA9 8FA9 5498 BEAB BEAB BEAB BEAB 3E2B 6E4E BEAB 5A3C      E5A8BC       E5A8BC       F3B0AEB0
8FAA     8FAA 8FAA 8FAA 8FAA 52EF BEAC BEAC BEAC BEAC 3E2C 6E6B BEAC 5BB5      E5AEB5       E5AEB5       F3B0AEB1
8FAB     8FAB 8FAB 8FAB 8FAB 4959 BEAD BEAD BEAD BEAD 3E2D 6E60 BEAD 5C06      E5B086       E5B086       F3B0AEB2
8FAC     8FAC 8FAC 8FAC 8FAC 455D BEAE BEAE BEAE BEAE 3E2E 6E4B BEAE 5C0F      E5B08F       E5B08F       F3B0AEB3
8FAD     8FAD 8FAD 8FAD 8FAD 484A BEAF BEAF BEAF BEAF 3E2F 6E61 BEAF 5C11      E5B091       E5B091       F3B0AEB4
8FAE     8FAE 8FAE 8FAE 8FAE 4871 BEB0 BEB0 BEB0 BEB0 3E30 6EF0 BEB0 5C1A      E5B09A       E5B09A       F3B0AEB5
8FAF     8FAF 8FAF 8FAF 8FAF 4850 BEB1 BEB1 BEB1 BEB1 3E31 6EF1 BEB1 5E84      E5BA84       E5BA84       F3B0AEB6
8FB0     8FB0 8FB0 8FB0 8FB0 4AED BEB2 BEB2 BEB2 BEB2 3E32 6EF2 BEB2 5E8A      E5BA8A       E5BA8A       F3B0AEB7
8FB1     8FB1 8FB1 8FB1 8FB1 5562 BEB3 BEB3 BEB3 BEB3 3E33 6EF3 BEB3 5EE0      E5BBA0       E5BBA0       F3B0AEB8
8FB2     8FB2 8FB2 8FB2 8FB2 4964 BEB4 BEB4 BEB4 BEB4 3E34 6EF4 BEB4 5F70      E5BDB0       E5BDB0       F3B0AEB9
8FB3     8FB3 8FB3 8FB3 8FB3 4C48 BEB5 BEB5 BEB5 BEB5 3E35 6EF5 BEB5 627F      E689BF       E689BF       F3B0AEBA
8FB4     8FB4 8FB4 8FB4 8FB4 4EC0 BEB6 BEB6 BEB6 BEB6 3E36 6EF6 BEB6 6284      E68A84       E68A84       F3B0AEBB
8FB5     8FB5 8FB5 8FB5 8FB5 4BFD BEB7 BEB7 BEB7 BEB7 3E37 6EF7 BEB7 62DB      E68B9B       E68B9B       F3B0AEBC
8FB6     8FB6 8FB6 8FB6 8FB6 4EEC BEB8 BEB8 BEB8 BEB8 3E38 6EF8 BEB8 638C      E68E8C       E68E8C       F3B0AEBD
8FB7     8FB7 8FB7 8FB7 8FB7 4D8F BEB9 BEB9 BEB9 BEB9 3E39 6EF9 BEB9 6377      E68DB7       E68DB7       F3B0AEBE
8FB8     8FB8 8FB8 8FB8 8FB8 46FE BEBA BEBA BEBA BEBA 3E3A 6E7A BEBA 6607      E69887       E69887       F3B0AEBF
8FB9     8FB9 8FB9 8FB9 8FB9 46D1 BEBB BEBB BEBB BEBB 3E3B 6E5E BEBB 660C      E6988C       E6988C       F3B0AF80
8FBA     8FBA 8FBA 8FBA 8FBA 45B3 BEBC BEBC BEBC BEBC 3E3C 6E4C BEBC 662D      E698AD       E698AD       F3B0AF81
8FBB     8FBB 8FBB 8FBB 8FBB 4AF8 BEBD BEBD BEBD BEBD 3E3D 6E7E BEBD 6676      E699B6       E699B6       F3B0AF82
8FBC     8FBC 8FBC 8FBC 8FBC 4578 BEBE BEBE BEBE BEBE 3E3E 6E6E BEBE 677E      E69DBE       E69DBE       F3B0AF83
8FBD     8FBD 8FBD 8FBD 8FBD 50AC BEBF BEBF BEBF BEBF 3E3F 6E6F BEBF 68A2      E6A2A2       E6A2A2       F3B0AF84
8FBE     8FBE 8FBE 8FBE 8FBE 539B BEC0 BEC0 BEC0 BEC0 3E40 6E7C BEC0 6A1F      E6A89F       E6A89F       F3B0AF85
8FBF     8FBF 8FBF 8FBF 8FBF 5585 BEC1 BEC1 BEC1 BEC1 3E41 6EC1 BEC1 6A35      E6A8B5       E6A8B5       F3B0AF86
8FC0     8FC0 8FC0 8FC0 8FC0 46C6 BEC2 BEC2 BEC2 BEC2 3E42 6EC2 BEC2 6CBC      E6B2BC       E6B2BC       F3B0AF87
8FC1     8FC1 8FC1 8FC1 8FC1 486B BEC3 BEC3 BEC3 BEC3 3E43 6EC3 BEC3 6D88      E6B688       E6B688       F3B0AF88
8FC2     8FC2 8FC2 8FC2 8FC2 4AC0 BEC4 BEC4 BEC4 BEC4 3E44 6EC4 BEC4 6E09      E6B889       E6B889       F3B0AF89
8FC3     8FC3 8FC3 8FC3 8FC3 5155 BEC5 BEC5 BEC5 BEC5 3E45 6EC5 BEC5 6E58      E6B998       E6B998       F3B0AF8A
8FC4     8FC4 8FC4 8FC4 8FC4 4983 BEC6 BEC6 BEC6 BEC6 3E46 6EC6 BEC6 713C      E784BC       E784BC       F3B0AF8B
8FC5     8FC5 8FC5 8FC5 8FC5 4DE6 BEC7 BEC7 BEC7 BEC7 3E47 6EC7 BEC7 7126      E784A6       E784A6       F3B0AF8C
8FC6     8FC6 8FC6 8FC6 8FC6 4742 BEC8 BEC8 BEC8 BEC8 3E48 6EC8 BEC8 7167      E785A7       E785A7       F3B0AF8D
8FC7     8FC7 8FC7 8FC7 8FC7 4E4A BEC9 BEC9 BEC9 BEC9 3E49 6EC9 BEC9 75C7      E79787       E79787       F3B0AF8E
8FC8     8FC8 8FC8 8FC8 8FC8 4847 BECA BECA BECA BECA 3E4A 6ED1 BECA 7701      E79C81       E79C81       F3B0AF8F
8FC9     8FC9 8FC9 8FC9 8FC9 4976 BECB BECB BECB BECB 3E4B 6ED2 BECB 785D      E7A19D       E7A19D       F3B0AF90
8FCA     8FCA 8FCA 8FCA 8FCA 51B0 BECC BECC BECC BECC 3E4C 6ED3 BECC 7901      E7A481       E7A481       F3B0AF91
8FCB     8FCB 8FCB 8FCB 8FCB 49F9 BECD BECD BECD BECD 3E4D 6ED4 BECD 7965      E7A5A5       E7A5A5       F3B0AF92
8FCC     8FCC 8FCC 8FCC 8FCC 4BDF BECE BECE BECE BECE 3E4E 6ED5 BECE 79F0      E7A7B0       E7A7B0       F3B0AF93
8FCD     8FCD 8FCD 8FCD 8FCD 476E BECF BECF BECF BECF 3E4F 6ED6 BECF 7AE0      E7ABA0       E7ABA0       F3B0AF94
8FCE     8FCE 8FCE 8FCE 8FCE 49B5 BED0 BED0 BED0 BED0 3E50 6ED7 BED0 7B11      E7AC91       E7AC91       F3B0AF95
8FCF     8FCF 8FCF 8FCF 8FCF 4BD4 BED1 BED1 BED1 BED1 3E51 6ED8 BED1 7CA7      E7B2A7       E7B2A7       F3B0AF96
8FD0     8FD0 8FD0 8FD0 8FD0 4CE4 BED2 BED2 BED2 BED2 3E52 6ED9 BED2 7D39      E7B4B9       E7B4B9       F3B0AF97
8FD1     8FD1 8FD1 8FD1 8FD1 5267 BED3 BED3 BED3 BED3 3E53 6EE2 BED3 8096      E88296       E88296       F3B0AF98
8FD2     8FD2 8FD2 8FD2 8FD2 5180 BED4 BED4 BED4 BED4 3E54 6EE3 BED4 83D6      E88F96       E88F96       F3B0AF99
8FD3     8FD3 8FD3 8FD3 8FD3 53F8 BED5 BED5 BED5 61B9 CCC7 9F98 BED5 848B      E8928B       E8928B       F3B0AF9A
8FD4     8FD4 8FD4 8FD4 8FD4 53A8 BED6 BED6 BED6 BED6 3E56 6EE5 BED6 8549      E89589       E89589       F3B0AF9B
8FD5     8FD5 8FD5 8FD5 8FD5 4DA5 BED7 BED7 BED7 BED7 3E57 6EE6 BED7 885D      E8A19D       E8A19D       F3B0AF9C
8FD6     8FD6 8FD6 8FD6 8FD6 4FA8 BED8 BED8 BED8 BED8 3E58 6EE7 BED8 88F3      E8A3B3       E8A3B3       F3B0AF9D
8FD7     8FD7 8FD7 8FD7 8FD7 50EF BED9 BED9 BED9 BED9 3E59 6EE8 BED9 8A1F      E8A89F       E8A89F       F3B0AF9E
8FD8     8FD8 8FD8 8FD8 8FD8 49BA BEDA BEDA BEDA BEDA 3E5A 6EE9 BEDA 8A3C      E8A8BC       E8A8BC       F3B0AF9F
8FD9     8FD9 8FD9 8FD9 8FD9 4F88 BEDB BEDB BEDB BEDB 3E5B 6E4A BEDB 8A54      E8A994       E8A994       F3B0AFA0
8FDA     8FDA 8FDA 8FDA 8FDA 4DE7 BEDC BEDC BEDC BEDC 3E5C 6E5B BEDC 8A73      E8A9B3       E8A9B3       F3B0AFA1
8FDB     8FDB 8FDB 8FDB 8FDB 4994 BEDD BEDD BEDD BEDD 3E5D 6E5A BEDD 8C61      E8B1A1       E8B1A1       F3B0AFA2
8FDC     8FDC 8FDC 8FDC 8FDC 4AAD BEDE BEDE BEDE BEDE 3E5E 6E5F BEDE 8CDE      E8B39E       E8B39E       F3B0AFA3
8FDD     8FDD 8FDD 8FDD 8FDD 507F BEDF BEDF BEDF 69A1 D6C8 AC99 BEDF 91A4      E986A4       E986A4       F3B0AFA4
8FDE     8FDE 8FDE 8FDE 8FDE 4FB4 BEE0 BEE0 BEE0 BEE0 3E60 6E79 BEE0 9266      E989A6       E989A6       F3B0AFA5
8FDF     8FDF 8FDF 8FDF 8FDF 53BD BEE1 BEE1 BEE1 BEE1 3E61 6E57 BEE1 937E      E98DBE       E98DBE       F3B0AFA6
8FE0     8FE0 8FE0 8FE0 8FE0 4CC8 BEE2 BEE2 BEE2 BEE2 3E62 6E59 BEE2 9418      E99098       E99098       F3B0AFA7
8FE1     8FE1 8FE1 8FE1 8FE1 4BF9 BEE3 BEE3 BEE3 BEE3 3E63 6E62 BEE3 969C      E99A9C       E99A9C       F3B0AFA8
8FE2     8FE2 8FE2 8FE2 8FE2 52E7 BEE4 BEE4 BEE4 BEE4 3E64 6E63 BEE4 9798      E99E98       E99E98       F3B0AFA9
8FE3     8FE3 8FE3 8FE3 8FE3 455E BEE5 BEE5 BEE5 BEE5 3E65 6E64 BEE5 4E0A      E4B88A       E4B88A       F3B0AFAA
8FE4     8FE4 8FE4 8FE4 8FE4 49D3 BEE6 BEE6 BEE6 BEE6 3E66 6E65 BEE6 4E08      E4B888       E4B888       F3B0AFAB
8FE5     8FE5 8FE5 8FE5 8FE5 4F62 BEE7 BEE7 BEE7 BEE7 3E67 6E66 BEE7 4E1E      E4B89E       E4B89E       F3B0AFAC
8FE6     8FE6 8FE6 8FE6 8FE6 4947 BEE8 BEE8 BEE8 BEE8 3E68 6E67 BEE8 4E57      E4B997       E4B997       F3B0AFAD
8FE7     8FE7 8FE7 8FE7 8FE7 527E BEE9 BEE9 BEE9 BEE9 3E69 6E68 BEE9 5197      E58697       E58697       F3B0AFAE
8FE8     8FE8 8FE8 8FE8 8FE8 4D85 BEEA BEEA BEEA BEEA 3E6A 6E69 BEEA 5270      E589B0       E589B0       F3B0AFAF
8FE9     8FE9 8FE9 8FE9 8FE9 45DB BEEB BEEB BEEB BEEB 3E6B 6E70 BEEB 57CE      E59F8E       E59F8E       F3B0AFB0
8FEA     8FEA 8FEA 8FEA 8FEA 45D9 BEEC BEEC BEEC BEEC 3E6C 6E71 BEEC 5834      E5A0B4       E5A0B4       F3B0AFB1
8FEB     8FEB 8FEB 8FEB 8FEB 5196 BEED BEED BEED BEED 3E6D 6E72 BEED 58CC      E5A38C       E5A38C       F3B0AFB2
8FEC     8FEC 8FEC 8FEC 8FEC 4DE4 BEEE BEEE BEEE BEEE 3E6E 6E73 BEEE 5B22      E5ACA2       E5ACA2       F3B0AFB3
8FED     8FED 8FED 8FED 8FED 46EF BEEF BEEF BEEF BEEF 3E6F 6E74 BEEF 5E38      E5B8B8       E5B8B8       F3B0AFB4
8FEE     8FEE 8FEE 8FEE 8FEE 485F BEF0 BEF0 BEF0 BEF0 3E70 6E75 BEF0 60C5      E68385       E68385       F3B0AFB5
8FEF     8FEF 8FEF 8FEF 8FEF 55F6 BEF1 BEF1 BEF1 BEF1 3E71 6E76 BEF1 64FE      E693BE       E693BE       F3B0AFB6
8FF0     8FF0 8FF0 8FF0 8FF0 475E BEF2 BEF2 BEF2 BEF2 3E72 6E77 BEF2 6761      E69DA1       E69DA1       F3B0AFB7
8FF1     8FF1 8FF1 8FF1 8FF1 50F4 BEF3 BEF3 BEF3 BEF3 3E73 6E78 BEF3 6756      E69D96       E69D96       F3B0AFB8
8FF2     8FF2 8FF2 8FF2 8FF2 4CC9 BEF4 BEF4 BEF4 BEF4 3E74 6E80 BEF4 6D44      E6B584       E6B584       F3B0AFB9
8FF3     8FF3 8FF3 8FF3 8FF3 4852 BEF5 BEF5 BEF5 BEF5 3E75 6E8B BEF5 72B6      E78AB6       E78AB6       F3B0AFBA
8FF4     8FF4 8FF4 8FF4 8FF4 4E54 BEF6 BEF6 BEF6 BEF6 3E76 6E9B BEF6 7573      E795B3       E795B3       F3B0AFBB
8FF5     8FF5 8FF5 8FF5 8FF5 4E98 BEF7 BEF7 BEF7 BEF7 3E77 6E9C BEF7 7A63      E7A9A3       E7A9A3       F3B0AFBC
8FF6     8FF6 8FF6 8FF6 8FF6 4ABA BEF8 BEF8 BEF8 BEF8 3E78 6EA0 BEF8 84B8      E892B8       E892B8       F3B0AFBD
8FF7     8FF7 8FF7 8FF7 8FF7 4B7B BEF9 BEF9 BEF9 BEF9 3E79 6EAB BEF9 8B72      E8ADB2       E8ADB2       F3B0AFBE
8FF8     8FF8 8FF8 8FF8 8FF8 50C2 BEFA BEFA BEFA BEFA 3E7A 6EB0 BEFA 91B8      E986B8       E986B8       F3B0AFBF
8FF9     8FF9 8FF9 8FF9 8FF9 4E57 BEFB BEFB BEFB BEFB 3E7B 6EC0 BEFB 9320      E98CA0       E98CA0       F3B0B080
8FFA     8FFA 8FFA 8FFA 8FFA 50CA BEFC BEFC BEFC BEFC 3E7C 6E6A BEFC 5631      E598B1       E598B1       F3B0B081
8FFB     8FFB 8FFB 8FFB 8FFB 4F48 BEFD BEFD BEFD BEFD 3E7D 6ED0 BEFD 57F4      E59FB4       E59FB4       F3B0B082
8FFC     8FFC 8FFC 8FFC 8FFC 4957 BEFE BEFE BEFE BEFE 3E7E 6EA1 BEFE 98FE      E9A3BE       E9A3BE       F3B0B083
9040     9040 9040 9040 9040 4EE5 BFA1 BFA1 BFA1 BFA1 3F21 6F4F BFA1 62ED      E68BAD       E68BAD       F3B0B084
9041     9041 9041 9041 9041 47C5 BFA2 BFA2 BFA2 BFA2 3F22 6F7F BFA2 690D      E6A48D       E6A48D       F3B0B085
9042     9042 9042 9042 9042 4E73 BFA3 BFA3 BFA3 BFA3 3F23 6F7B BFA3 6B96      E6AE96       E6AE96       F3B0B086
9043     9043 9043 9043 9043 54E6 BFA4 BFA4 BFA4 BFA4 3F24 6FE0 BFA4 71ED      E787AD       E787AD       F3B0B087
9044     9044 9044 9044 9044 489A BFA5 BFA5 BFA5 BFA5 3F25 6F6C BFA5 7E54      E7B994       E7B994       F3B0B088
9045     9045 9045 9045 9045 48D7 BFA6 BFA6 BFA6 BFA6 3F26 6F50 BFA6 8077      E881B7       E881B7       F3B0B089
9046     9046 9046 9046 9046 47C3 BFA7 BFA7 BFA7 BFA7 3F27 6F7D BFA7 8272      E889B2       E889B2       F3B0B08A
9047     9047 9047 9047 9047 4CD1 BFA8 BFA8 BFA8 BFA8 3F28 6F4D BFA8 89E6      E8A7A6       E8A7A6       F3B0B08B
9048     9048 9048 9048 9048 46B8 BFA9 BFA9 BFA9 BFA9 3F29 6F5D BFA9 98DF      E9A39F       E9A39F       F3B0B08C
9049     9049 9049 9049 9049 5347 BFAA BFAA BFAA BFAA 3F2A 6F5C BFAA 8755      E89D95       E89D95       F3B0B08D
904A     904A 904A 904A 904A 52E5 BFAB BFAB BFAB BFAB 3F2B 6F4E BFAB 8FB1      E8BEB1       E8BEB1       F3B0B08E
904B     904B 904B 904B 904B 49CE BFAC BFAC BFAC BFAC 3F2C 6F6B BFAC 5C3B      E5B0BB       E5B0BB       F3B0B08F
904C     904C 904C 904C 904C 46F4 BFAD BFAD BFAD BFAD 3F2D 6F60 BFAD 4F38      E4BCB8       E4BCB8       F3B0B090
904D     904D 904D 904D 904D 45A5 BFAE BFAE BFAE BFAE 3F2E 6F4B BFAE 4FE1      E4BFA1       E4BFA1       F3B0B091
904E     904E 904E 904E 904E 4F8C BFAF BFAF BFAF BFAF 3F2F 6F61 BFAF 4FB5      E4BEB5       E4BEB5       F3B0B092
904F     904F 904F 904F 904F 4F76 BFB0 BFB0 BFB0 BFB0 3F30 6FF0 BFB0 5507      E59487       E59487       F3B0B093
9050     9050 9050 9050 9050 524C BFB1 BFB1 BFB1 BFB1 3F31 6FF1 BFB1 5A20      E5A8A0       E5A8A0       F3B0B094
9051     9051 9051 9051 9051 4B59 BFB2 BFB2 BFB2 BFB2 3F32 6FF2 BFB2 5BDD      E5AF9D       E5AF9D       F3B0B095
9052     9052 9052 9052 9052 4987 BFB3 BFB3 BFB3 BFB3 3F33 6FF3 BFB3 5BE9      E5AFA9       E5AFA9       F3B0B096
9053     9053 9053 9053 9053 46E7 BFB4 BFB4 BFB4 BFB4 3F34 6FF4 BFB4 5FC3      E5BF83       E5BF83       F3B0B097
9054     9054 9054 9054 9054 4A76 BFB5 BFB5 BFB5 BFB5 3F35 6FF5 BFB5 614E      E6858E       E6858E       F3B0B098
9055     9055 9055 9055 9055 48F6 BFB6 BFB6 BFB6 BFB6 3F36 6FF6 BFB6 632F      E68CAF       E68CAF       F3B0B099
9056     9056 9056 9056 9056 4587 BFB7 BFB7 BFB7 BFB7 3F37 6FF7 BFB7 65B0      E696B0       E696B0       F3B0B09A
9057     9057 9057 9057 9057 4BB2 BFB8 BFB8 BFB8 BFB8 3F38 6FF8 BFB8 664B      E6998B       E6998B       F3B0B09B
9058     9058 9058 9058 9058 459D BFB9 BFB9 BFB9 BFB9 3F39 6FF9 BFB9 68EE      E6A3AE       E6A3AE       F3B0B09C
9059     9059 9059 9059 9059 4D7E BFBA BFBA BFBA BFBA 3F3A 6F7A BFBA 699B      E6A69B       E6A69B       F3B0B09D
905A     905A 905A 905A 905A 4D41 BFBB BFBB BFBB BFBB 3F3B 6F5E BFBB 6D78      E6B5B8       E6B5B8       F3B0B09E
905B     905B 905B 905B 905B 4777 BFBC BFBC BFBC BFBC 3F3C 6F4C BFBC 6DF1      E6B7B1       E6B7B1       F3B0B09F
905C     905C 905C 905C 905C 4967 BFBD BFBD BFBD BFBD 3F3D 6F7E BFBD 7533      E794B3       E794B3       F3B0B0A0
905D     905D 905D 905D 905D 54E1 BFBE BFBE BFBE BFBE 3F3E 6F6E BFBE 75B9      E796B9       E796B9       F3B0B0A1
905E     905E 905E 905E 905E 4642 BFBF BFBF BFBF BFBF 3F3F 6F6F BFBF 771F      E79C9F       E79C9F       F3B0B0A2
905F     905F 905F 905F 905F 45A1 BFC0 BFC0 BFC0 BFC0 3F40 6F7C BFC0 795E      E7A59E       E7A59E       F3B0B0A3
9060     9060 9060 9060 9060 4C8A BFC1 BFC1 BFC1 BFC1 3F41 6FC1 BFC1 79E6      E7A7A6       E7A7A6       F3B0B0A4
9061     9061 9061 9061 9061 4E59 BFC2 BFC2 BFC2 BFC2 3F42 6FC2 BFC2 7D33      E7B4B3       E7B4B3       F3B0B0A5
9062     9062 9062 9062 9062 4A47 BFC3 BFC3 BFC3 BFC3 3F43 6FC3 BFC3 81E3      E887A3       E887A3       F3B0B0A6
9063     9063 9063 9063 9063 4FE5 BFC4 BFC4 BFC4 BFC4 3F44 6FC4 BFC4 82AF      E88AAF       E88AAF       F3B0B0A7
9064     9064 9064 9064 9064 545A BFC5 BFC5 BFC5 BFC5 3F45 6FC5 BFC5 85AA      E896AA       E896AA       F3B0B0A8
9065     9065 9065 9065 9065 4885 BFC6 BFC6 BFC6 BFC6 3F46 6FC6 BFC6 89AA      E8A6AA       E8A6AA       F3B0B0A9
9066     9066 9066 9066 9066 4D68 BFC7 BFC7 BFC7 BFC7 3F47 6FC7 BFC7 8A3A      E8A8BA       E8A8BA       F3B0B0AA
9067     9067 9067 9067 9067 479C BFC8 BFC8 BFC8 BFC8 3F48 6FC8 BFC8 8EAB      E8BAAB       E8BAAB       F3B0B0AB
9068     9068 9068 9068 9068 4D5E BFC9 BFC9 BFC9 BFC9 3F49 6FC9 BFC9 8F9B      E8BE9B       E8BE9B       F3B0B0AC
9069     9069 9069 9069 9069 4669 BFCA BFCA BFCA BFCA 3F4A 6FD1 BFCA 9032      E980B2       E980B2       F3B0B0AD
906A     906A 906A 906A 906A 4988 BFCB BFCB BFCB BFCB 3F4B 6FD2 BFCB 91DD      E9879D       E9879D       F3B0B0AE
906B     906B 906B 906B 906B 4DDE BFCC BFCC BFCC BFCC 3F4C 6FD3 BFCC 9707      E99C87       E99C87       F3B0B0AF
906C     906C 906C 906C 906C 4593 BFCD BFCD BFCD BFCD 3F4D 6FD4 BFCD 4EBA      E4BABA       E4BABA       F3B0B0B0
906D     906D 906D 906D 906D 4776 BFCE BFCE BFCE BFCE 3F4E 6FD5 BFCE 4EC1      E4BB81       E4BB81       F3B0B0B1
906E     906E 906E 906E 906E 4F44 BFCF BFCF BFCF BFCF 3F4F 6FD6 BFCF 5203      E58883       E58883       F3B0B0B2
906F     906F 906F 906F 906F 4D89 BFD0 BFD0 BFD0 BFD0 3F50 6FD7 BFD0 5875      E5A1B5       E5A1B5       F3B0B0B3
9070     9070 9070 9070 9070 514F BFD1 BFD1 BFD1 BFD1 3F51 6FD8 BFD1 58EC      E5A3AC       E5A3AC       F3B0B0B4
9071     9071 9071 9071 9071 4D5F BFD2 BFD2 BFD2 BFD2 3F52 6FD9 BFD2 5C0B      E5B08B       E5B08B       F3B0B0B5
9072     9072 9072 9072 9072 4B66 BFD3 BFD3 BFD3 BFD3 3F53 6FE2 BFD3 751A      E7949A       E7949A       F3B0B0B6
9073     9073 9073 9073 9073 4FB9 BFD4 BFD4 BFD4 BFD4 3F54 6FE3 BFD4 5C3D      E5B0BD       E5B0BD       F3B0B0B7
9074     9074 9074 9074 9074 52C4 BFD5 BFD5 BFD5 BFD5 3F55 6FE4 BFD5 814E      E8858E       E8858E       F3B0B0B8
9075     9075 9075 9075 9075 50F2 BFD6 BFD6 BFD6 BFD6 3F56 6FE5 BFD6 8A0A      E8A88A       E8A88A       F3B0B0B9
9076     9076 9076 9076 9076 5068 BFD7 BFD7 BFD7 BFD7 3F57 6FE6 BFD7 8FC5      E8BF85       E8BF85       F3B0B0BA
9077     9077 9077 9077 9077 4C84 BFD8 BFD8 BFD8 BFD8 3F58 6FE7 BFD8 9663      E999A3       E999A3       F3B0B0BB
9078     9078 9078 9078 9078 66AB F0D7 BFD9 BFD9 F0D7 7057 75E6 BFD9 976D      E99DAD       E99DAD       F3B0B0BC
9079     9079 9079 9079 9079 52F4 BFDA BFDA BFDA BFDA 3F5A 6FE9 BFDA 7B25      E7ACA5       E7ACA5       F3B0B0BD
907A     907A 907A 907A 907A 4BC1 BFDB BFDB BFDB BFDB 3F5B 6F4A BFDB 8ACF      E8AB8F       E8AB8F       F3B0B0BE
907B     907B 907B 907B 907B 46B0 BFDC BFDC BFDC BFDC 3F5C 6F5B BFDC 9808      E9A088       E9A088       F3B0B0BF
907C     907C 907C 907C 907C 4BAE BFDD BFDD BFDD BFDD 3F5D 6F5A BFDD 9162      E985A2       E985A2       F3B0B180
907D     907D 907D 907D 907D 47C1 BFDE BFDE BFDE BFDE 3F5E 6F5F BFDE 56F3      E59BB3       E59BB3       F3B0B181
907E     907E 907E 907E 907E 5178 BFDF BFDF BFDF BFDF 3F5F 6F6D BFDF 53A8      E58EA8       E58EA8       F3B0B182
9080     9080 9080 9080 9080 4FB3 BFE0 BFE0 BFE0 BFE0 3F60 6F79 BFE0 9017      E98097       E98097       F3B0B183
9081     9081 9081 9081 9081 49C4 BFE1 BFE1 BFE1 BFE1 3F61 6F57 BFE1 5439      E590B9       E590B9       F3B0B184
9082     9082 9082 9082 9082 4BC9 BFE2 BFE2 BFE2 BFE2 3F62 6F59 BFE2 5782      E59E82       E59E82       F3B0B185
9083     9083 9083 9083 9083 52AF BFE3 BFE3 BFE3 BFE3 3F63 6F62 BFE3 5E25      E5B8A5       E5B8A5       F3B0B186
9084     9084 9084 9084 9084 48D5 BFE4 BFE4 BFE4 BFE4 3F64 6F63 BFE4 63A8      E68EA8       E68EA8       F3B0B187
9085     9085 9085 9085 9085 459C BFE5 BFE5 BFE5 BFE5 3F65 6F64 BFE5 6C34      E6B0B4       E6B0B4       F3B0B188
9086     9086 9086 9086 9086 4FCB BFE6 BFE6 BFE6 BFE6 3F66 6F65 BFE6 708A      E7828A       E7828A       F3B0B189
9087     9087 9087 9087 9087 526C BFE7 BFE7 BFE7 BFE7 3F67 6F66 BFE7 7761      E79DA1       E79DA1       F3B0B18A
9088     9088 9088 9088 9088 4FCC BFE8 BFE8 BFE8 BFE8 3F68 6F67 BFE8 7C8B      E7B28B       E7B28B       F3B0B18B
9089     9089 9089 9089 9089 4F69 BFE9 BFE9 BFE9 BFE9 3F69 6F68 BFE9 7FE0      E7BFA0       E7BFA0       F3B0B18C
908A     908A 908A 908A 908A 4F54 BFEA BFEA BFEA BFEA 3F6A 6F69 BFEA 8870      E8A1B0       E8A1B0       F3B0B18D
908B     908B 908B 908B 908B 4CF7 BFEB BFEB BFEB BFEB 3F6B 6F70 BFEB 9042      E98182       E98182       F3B0B18E
908C     908C 908C 908C 908C 4E7C BFEC BFEC BFEC BFEC 3F6C 6F71 BFEC 9154      E98594       E98594       F3B0B18F
908D     908D 908D 908D 908D 5390 BFED BFED BFED BFED 3F6D 6F72 BFED 9310      E98C90       E98C90       F3B0B190
908E     908E 908E 908E 908E 54EF BFEE BFEE BFEE BFEE 3F6E 6F73 BFEE 9318      E98C98       E98C98       F3B0B191
908F     908F 908F 908F 908F 4E76 BFEF BFEF BFEF BFEF 3F6F 6F74 BFEF 968F      E99A8F       E99A8F       F3B0B192
9090     9090 9090 9090 9090 4B5B BFF0 BFF0 BFF0 BFF0 3F70 6F75 BFF0 745E      E7919E       E7919E       F3B0B193
9091     9091 9091 9091 9091 52C5 BFF1 BFF1 BFF1 BFF1 3F71 6F76 BFF1 9AC4      E9AB84       E9AB84       F3B0B194
9092     9092 9092 9092 9092 4C7F BFF2 BFF2 BFF2 BFF2 3F72 6F77 BFF2 5D07      E5B487       E5B487       F3B0B195
9093     9093 9093 9093 9093 509E BFF3 BFF3 BFF3 BFF3 3F73 6F78 BFF3 5D69      E5B5A9       E5B5A9       F3B0B196
9094     9094 9094 9094 9094 46CC BFF4 BFF4 BFF4 BFF4 3F74 6F80 BFF4 6570      E695B0       E695B0       F3B0B197
9095     9095 9095 9095 9095 52D0 BFF5 BFF5 BFF5 BFF5 3F75 6F8B BFF5 67A2      E69EA2       E69EA2       F3B0B198
9096     9096 9096 9096 9096 5561 BFF6 BFF6 BFF6 BFF6 3F76 6F9B BFF6 8DA8      E8B6A8       E8B6A8       F3B0B199
9097     9097 9097 9097 9097 51FB BFF7 BFF7 BFF7 BFF7 3F77 6F9C BFF7 96DB      E99B9B       E99B9B       F3B0B19A
9098     9098 9098 9098 9098 4ED6 BFF8 BFF8 BFF8 BFF8 3F78 6FA0 BFF8 636E      E68DAE       E68DAE       F3B0B19B
9099     9099 9099 9099 9099 4668 BFF9 BFF9 BFF9 BFF9 3F79 6FAB BFF9 6749      E69D89       E69D89       F3B0B19C
909A     909A 909A 909A 909A 4FED BFFA BFFA BFFA BFFA 3F7A 6FB0 BFFA 6919      E6A499       E6A499       F3B0B19D
909B     909B 909B 909B 909B 4753 BFFB BFFB BFFB BFFB 3F7B 6FC0 BFFB 83C5      E88F85       E88F85       F3B0B19E
909C     909C 909C 909C 909C 55E4 BFFC BFFC BFFC BFFC 3F7C 6F6A BFFC 9817      E9A097       E9A097       F3B0B19F
909D     909D 909D 909D 909D 4F64 BFFD BFFD BFFD BFFD 3F7D 6FD0 BFFD 96C0      E99B80       E99B80       F3B0B1A0
909E     909E 909E 909E 909E 4C8D BFFE BFFE BFFE BFFE 3F7E 6FA1 BFFE 88FE      E8A3BE       E8A3BE       F3B0B1A1
909F     909F 909F 909F 909F 4887 C0A1 C0A1 C0A1 C0A1 4021 7C4F C0A1 6F84      E6BE84       E6BE84       F3B0B1A2
90A0     90A0 90A0 90A0 90A0 51EF C0A2 C0A2 C0A2 C0A2 4022 7C7F C0A2 647A      E691BA       E691BA       F3B0B1A3
90A1     90A1 90A1 90A1 90A1 4B60 C0A3 C0A3 C0A3 C0A3 4023 7C7B C0A3 5BF8      E5AFB8       E5AFB8       F3B0B1A4
90A2     90A2 90A2 90A2 90A2 469B C0A4 C0A4 C0A4 C0A4 4024 7CE0 C0A4 4E16      E4B896       E4B896       F3B0B1A5
90A3     90A3 90A3 90A3 90A3 465B C0A5 C0A5 C0A5 C0A5 4025 7C6C C0A5 702C      E780AC       E780AC       F3B0B1A6
90A4     90A4 90A4 90A4 90A4 5086 C0A6 C0A6 C0A6 C0A6 4026 7C50 C0A6 755D      E7959D       E7959D       F3B0B1A7
90A5     90A5 90A5 90A5 90A5 4BBB C0A7 C0A7 C0A7 C0A7 4027 7C7D C0A7 662F      E698AF       E698AF       F3B0B1A8
90A6     90A6 90A6 90A6 90A6 52DC C0A8 C0A8 C0A8 C0A8 4028 7C4D C0A8 51C4      E58784       E58784       F3B0B1A9
90A7     90A7 90A7 90A7 90A7 46C4 C0A9 C0A9 C0A9 C0A9 4029 7C5D C0A9 5236      E588B6       E588B6       F3B0B1AA
90A8     90A8 90A8 90A8 90A8 47EE C0AA C0AA C0AA C0AA 402A 7C5C C0AA 52E2      E58BA2       E58BA2       F3B0B1AB
90A9     90A9 90A9 90A9 90A9 4FF1 C0AB C0AB C0AB C0AB 402B 7C4E C0AB 59D3      E5A793       E5A793       F3B0B1AC
90AA     90AA 90AA 90AA 90AA 4A61 C0AC C0AC C0AC C0AC 402C 7C6B C0AC 5F81      E5BE81       E5BE81       F3B0B1AD
90AB     90AB 90AB 90AB 90AB 466E C0AD C0AD C0AD C0AD 402D 7C60 C0AD 6027      E680A7       E680A7       F3B0B1AE
90AC     90AC 90AC 90AC 90AC 45BA C0AE C0AE C0AE C0AE 402E 7C4B C0AE 6210      E68890       E68890       F3B0B1AF
90AD     90AD 90AD 90AD 90AD 45C7 C0AF C0AF C0AF C0AF 402F 7C61 C0AF 653F      E694BF       E694BF       F3B0B1B0
90AE     90AE 90AE 90AE 90AE 477C C0B0 C0B0 C0B0 C0B0 4030 7CF0 C0B0 6574      E695B4       E695B4       F3B0B1B1
90AF     90AF 90AF 90AF 90AF 47A9 C0B1 C0B1 C0B1 C0B1 4031 7CF1 C0B1 661F      E6989F       E6989F       F3B0B1B2
90B0     90B0 90B0 90B0 90B0 4795 C0B2 C0B2 C0B2 C0B2 4032 7CF2 C0B2 6674      E699B4       E699B4       F3B0B1B3
90B1     90B1 90B1 90B1 90B1 5375 C0B3 C0B3 C0B3 C0B3 4033 7CF3 C0B3 68F2      E6A3B2       E6A3B2       F3B0B1B4
90B2     90B2 90B2 90B2 90B2 4DCC C0B4 C0B4 C0B4 C0B4 4034 7CF4 C0B4 6816      E6A096       E6A096       F3B0B1B5
90B3     90B3 90B3 90B3 90B3 4577 C0B5 C0B5 C0B5 C0B5 4035 7CF5 C0B5 6B63      E6ADA3       E6ADA3       F3B0B1B6
90B4     90B4 90B4 90B4 90B4 4596 C0B6 C0B6 C0B6 C0B6 4036 7CF6 C0B6 6E05      E6B885       E6B885       F3B0B1B7
90B5     90B5 90B5 90B5 90B5 5158 C0B7 C0B7 C0B7 C0B7 4037 7CF7 C0B7 7272      E789B2       E789B2       F3B0B1B8
90B6     90B6 90B6 90B6 90B6 4599 C0B8 C0B8 C0B8 C0B8 4038 7CF8 C0B8 751F      E7949F       E7949F       F3B0B1B9
90B7     90B7 90B7 90B7 90B7 4844 C0B9 C0B9 C0B9 C0B9 4039 7CF9 C0B9 76DB      E79B9B       E79B9B       F3B0B1BA
90B8     90B8 90B8 90B8 90B8 466C C0BA C0BA C0BA C0BA 403A 7C7A C0BA 7CBE      E7B2BE       E7B2BE       F3B0B1BB
90B9     90B9 90B9 90B9 90B9 4BB5 C0BB C0BB C0BB C0BB 403B 7C5E C0BB 8056      E88196       E88196       F3B0B1BC
90BA     90BA 90BA 90BA 90BA 496C C0BC C0BC C0BC C0BC 403C 7C4C C0BC 58F0      E5A3B0       E5A3B0       F3B0B1BD
90BB     90BB 90BB 90BB 90BB 457B C0BD C0BD C0BD C0BD 403D 7C7E C0BD 88FD      E8A3BD       E8A3BD       F3B0B1BE
90BC     90BC 90BC 90BC 90BC 4558 C0BE C0BE C0BE C0BE 403E 7C6E C0BE 897F      E8A5BF       E8A5BF       F3B0B1BF
90BD     90BD 90BD 90BD 90BD 46FD C0BF C0BF C0BF C0BF 403F 7C6F C0BF 8AA0      E8AAA0       E8AAA0       F3B0B280
90BE     90BE 90BE 90BE 90BE 4FA0 C0C0 C0C0 C0C0 C0C0 4040 7C7C C0C0 8A93      E8AA93       E8AA93       F3B0B281
90BF     90BF 90BF 90BF 90BF 4AE8 C0C1 C0C1 C0C1 C0C1 4041 7CC1 C0C1 8ACB      E8AB8B       E8AB8B       F3B0B282
90C0     90C0 90C0 90C0 90C0 53B1 C0C2 C0C2 C0C2 C0C2 4042 7CC2 C0C2 901D      E9809D       E9809D       F3B0B283
90C1     90C1 90C1 90C1 90C1 5378 C0C3 C0C3 C0C3 C0C3 4043 7CC3 C0C3 9192      E98692       E98692       F3B0B284
90C2     90C2 90C2 90C2 90C2 464C C0C4 C0C4 C0C4 C0C4 4044 7CC4 C0C4 9752      E99D92       E99D92       F3B0B285
90C3     90C3 90C3 90C3 90C3 4644 C0C5 C0C5 C0C5 C0C5 4045 7CC5 C0C5 9759      E99D99       E99D99       F3B0B286
90C4     90C4 90C4 90C4 90C4 468E C0C6 C0C6 C0C6 C0C6 4046 7CC6 C0C6 6589      E69689       E69689       F3B0B287
90C5     90C5 90C5 90C5 90C5 47C4 C0C7 C0C7 C0C7 C0C7 4047 7CC7 C0C7 7A0E      E7A88E       E7A88E       F3B0B288
90C6     90C6 90C6 90C6 90C6 55EC C0C8 C0C8 C0C8 C0C8 4048 7CC8 C0C8 8106      E88486       E88486       F3B0B289
90C7     90C7 90C7 90C7 90C7 4FDA C0C9 C0C9 C0C9 C0C9 4049 7CC9 C0C9 96BB      E99ABB       E99ABB       F3B0B28A
90C8     90C8 90C8 90C8 90C8 4B53 C0CA C0CA C0CA C0CA 404A 7CD1 C0CA 5E2D      E5B8AD       E5B8AD       F3B0B28B
90C9     90C9 90C9 90C9 90C9 4ED7 C0CB C0CB C0CB C0CB 404B 7CD2 C0CB 60DC      E6839C       E6839C       F3B0B28C
90CA     90CA 90CA 90CA 90CA 5595 C0CC C0CC C0CC C0CC 404C 7CD3 C0CC 621A      E6889A       E6889A       F3B0B28D
90CB     90CB 90CB 90CB 90CB 5141 C0CD C0CD C0CD C0CD 404D 7CD4 C0CD 65A5      E696A5       E696A5       F3B0B28E
90CC     90CC 90CC 90CC 90CC 4C59 C0CE C0CE C0CE C0CE 404E 7CD5 C0CE 6614      E69894       E69894       F3B0B28F
90CD     90CD 90CD 90CD 90CD 4AE2 C0CF C0CF C0CF C0CF 404F 7CD6 C0CF 6790      E69E90       E69E90       F3B0B290
90CE     90CE 90CE 90CE 90CE 4580 C0D0 C0D0 C0D0 C0D0 4050 7CD7 C0D0 77F3      E79FB3       E79FB3       F3B0B291
90CF     90CF 90CF 90CF 90CF 4891 C0D1 C0D1 C0D1 C0D1 4051 7CD8 C0D1 7A4D      E7A98D       E7A98D       F3B0B292
90D0     90D0 90D0 90D0 90D0 4DFE C0D2 C0D2 C0D2 C0D2 4052 7CD9 C0D2 7C4D      E7B18D       E7B18D       F3B0B293
90D1     90D1 90D1 90D1 90D1 4866 C0D3 C0D3 C0D3 C0D3 4053 7CE2 C0D3 7E3E      E7B8BE       E7B8BE       F3B0B294
90D2     90D2 90D2 90D2 90D2 535E C0D4 C0D4 C0D4 C0D4 4054 7CE3 C0D4 810A      E8848A       E8848A       F3B0B295
90D3     90D3 90D3 90D3 90D3 4BD2 C0D5 C0D5 C0D5 C0D5 4055 7CE4 C0D5 8CAC      E8B2AC       E8B2AC       F3B0B296
90D4     90D4 90D4 90D4 90D4 46F9 C0D6 C0D6 C0D6 C0D6 4056 7CE5 C0D6 8D64      E8B5A4       E8B5A4       F3B0B297
90D5     90D5 90D5 90D5 90D5 4CCD C0D7 C0D7 C0D7 C0D7 4057 7CE6 C0D7 8DE1      E8B7A1       E8B7A1       F3B0B298
90D6     90D6 90D6 90D6 90D6 5368 C0D8 C0D8 C0D8 C0D8 4058 7CE7 C0D8 8E5F      E8B99F       E8B99F       F3B0B299
90D7     90D7 90D7 90D7 90D7 507D C0D9 C0D9 C0D9 C0D9 4059 7CE8 C0D9 78A9      E7A2A9       E7A2A9       F3B0B29A
90D8     90D8 90D8 90D8 90D8 4765 C0DA C0DA C0DA C0DA 405A 7CE9 C0DA 5207      E58887       E58887       F3B0B29B
90D9     90D9 90D9 90D9 90D9 525C C0DB C0DB C0DB C0DB 405B 7C4A C0DB 62D9      E68B99       E68B99       F3B0B29C
90DA     90DA 90DA 90DA 90DA 47A8 C0DC C0DC C0DC C0DC 405C 7C5B C0DC 63A5      E68EA5       E68EA5       F3B0B29D
90DB     90DB 90DB 90DB 90DB 4E8C C0DD C0DD C0DD C0DD 405D 7C5A C0DD 6442      E69182       E69182       F3B0B29E
90DC     90DC 90DC 90DC 90DC 498D C0DE C0DE C0DE C0DE 405E 7C5F C0DE 6298      E68A98       E68A98       F3B0B29F
90DD     90DD 90DD 90DD 90DD 45E2 C0DF C0DF C0DF C0DF 405F 7C6D C0DF 8A2D      E8A8AD       E8A8AD       F3B0B2A0
90DE     90DE 90DE 90DE 90DE 5169 C0E0 C0E0 C0E0 C0E0 4060 7C79 C0E0 7A83      E7AA83       E7AA83       F3B0B2A1
90DF     90DF 90DF 90DF 90DF 47A1 C0E1 C0E1 C0E1 C0E1 4061 7C57 C0E1 7BC0      E7AF80       E7AF80       F3B0B2A2
90E0     90E0 90E0 90E0 90E0 4952 C0E2 C0E2 C0E2 C0E2 4062 7C59 C0E2 8AAC      E8AAAC       E8AAAC       F3B0B2A3
90E1     90E1 90E1 90E1 90E1 49BB C0E3 C0E3 C0E3 C0E3 4063 7C62 C0E3 96EA      E99BAA       E99BAA       F3B0B2A4
90E2     90E2 90E2 90E2 90E2 4AF3 C0E4 C0E4 C0E4 C0E4 4064 7C63 C0E4 7D76      E7B5B6       E7B5B6       F3B0B2A5
90E3     90E3 90E3 90E3 90E3 5094 C0E5 C0E5 C0E5 C0E5 4065 7C64 C0E5 820C      E8888C       E8888C       F3B0B2A6
90E4     90E4 90E4 90E4 90E4 53E8 C0E6 C0E6 C0E6 63B3 CDDC A2BC C0E6 8749      E89D89       E89D89       F3B0B2A7
90E5     90E5 90E5 90E5 90E5 48A2 C0E7 C0E7 C0E7 C0E7 4067 7C66 C0E7 4ED9      E4BB99       E4BB99       F3B0B2A8
90E6     90E6 90E6 90E6 90E6 47A2 C0E8 C0E8 C0E8 C0E8 4068 7C67 C0E8 5148      E58588       E58588       F3B0B2A9
90E7     90E7 90E7 90E7 90E7 454C C0E9 C0E9 C0E9 C0E9 4069 7C68 C0E9 5343      E58D83       E58D83       F3B0B2AA
90E8     90E8 90E8 90E8 90E8 49EE C0EA C0EA C0EA C0EA 406A 7C69 C0EA 5360      E58DA0       E58DA0       F3B0B2AB
90E9     90E9 90E9 90E9 90E9 498F C0EB C0EB C0EB C0EB 406B 7C70 C0EB 5BA3      E5AEA3       E5AEA3       F3B0B2AC
90EA     90EA 90EA 90EA 90EA 49FB C0EC C0EC C0EC C0EC 406C 7C71 C0EC 5C02      E5B082       E5B082       F3B0B2AD
90EB     90EB 90EB 90EB 90EB 53C0 C0ED C0ED C0ED C0ED 406D 7C72 C0ED 5C16      E5B096       E5B096       F3B0B2AE
90EC     90EC 90EC 90EC 90EC 4567 C0EE C0EE C0EE C0EE 406E 7C73 C0EE 5DDD      E5B79D       E5B79D       F3B0B2AF
90ED     90ED 90ED 90ED 90ED 4764 C0EF C0EF C0EF C0EF 406F 7C74 C0EF 6226      E688A6       E688A6       F3B0B2B0
90EE     90EE 90EE 90EE 90EE 4DCA C0F0 C0F0 C0F0 C0F0 4070 7C75 C0F0 6247      E68987       E68987       F3B0B2B1
90EF     90EF 90EF 90EF 90EF 5372 C0F1 C0F1 C0F1 C0F1 4071 7C76 C0F1 64B0      E692B0       E692B0       F3B0B2B2
90F0     90F0 90F0 90F0 90F0 50E3 C0F2 C0F2 C0F2 C0F2 4072 7C77 C0F2 6813      E6A093       E6A093       F3B0B2B3
90F1     90F1 90F1 90F1 90F1 55DA C0F3 C0F3 C0F3 C0F3 4073 7C78 C0F3 6834      E6A0B4       E6A0B4       F3B0B2B4
90F2     90F2 90F2 90F2 90F2 46EE C0F4 C0F4 C0F4 C0F4 4074 7C80 C0F4 6CC9      E6B389       E6B389       F3B0B2B5
90F3     90F3 90F3 90F3 90F3 468D C0F5 C0F5 C0F5 C0F5 4075 7C8B C0F5 6D45      E6B585       E6B585       F3B0B2B6
90F4     90F4 90F4 90F4 90F4 4A4E C0F6 C0F6 C0F6 C0F6 4076 7C9B C0F6 6D17      E6B497       E6B497       F3B0B2B7
90F5     90F5 90F5 90F5 90F5 489B C0F7 C0F7 C0F7 C0F7 4077 7C9C C0F7 67D3      E69F93       E69F93       F3B0B2B8
90F6     90F6 90F6 90F6 90F6 4DC0 C0F8 C0F8 C0F8 C0F8 4078 7CA0 C0F8 6F5C      E6BD9C       E6BD9C       F3B0B2B9
90F7     90F7 90F7 90F7 90F7 5490 C0F9 C0F9 C0F9 C0F9 4079 7CAB C0F9 714E      E7858E       E7858E       F3B0B2BA
90F8     90F8 90F8 90F8 90F8 5590 C0FA C0FA C0FA C0FA 407A 7CB0 C0FA 717D      E785BD       E785BD       F3B0B2BB
90F9     90F9 90F9 90F9 90F9 4D7B C0FB C0FB C0FB C0FB 407B 7CC0 C0FB 65CB      E6978B       E6978B       F3B0B2BC
90FA     90FA 90FA 90FA 90FA 4E91 C0FC C0FC C0FC C0FC 407C 7C6A C0FC 7A7F      E7A9BF       E7A9BF       F3B0B2BD
90FB     90FB 90FB 90FB 90FB 518D C0FD C0FD C0FD C0FD 407D 7CD0 C0FD 7BAD      E7AEAD       E7AEAD       F3B0B2BE
90FC     90FC 90FC 90FC 90FC 4687 C0FE C0FE C0FE C0FE 407E 7CA1 C0FE 7DDA      E7B79A       E7B79A       F3B0B2BF
9140     9140 9140 9140 9140 4868 C1A1 C1A1 C1A1 C1A1 4121 C14F C1A1 7E4A      E7B98A       E7B98A       F3B0B380
9141     9141 9141 9141 9141 5596 C1A2 C1A2 C1A2 C1A2 4122 C17F C1A2 7FA8      E7BEA8       E7BEA8       F3B0B381
9142     9142 9142 9142 9142 526F C1A3 C1A3 C1A3 C1A3 4123 C17B C1A3 817A      E885BA       E885BA       F3B0B382
9143     9143 9143 9143 9143 4FC1 C1A4 C1A4 C1A4 C1A4 4124 C1E0 C1A4 821B      E8889B       E8889B       F3B0B383
9144     9144 9144 9144 9144 467B C1A5 C1A5 C1A5 C1A5 4125 C16C C1A5 8239      E888B9       E888B9       F3B0B384
9145     9145 9145 9145 9145 4FBF C1A6 C1A6 C1A6 C1A6 4126 C150 C1A6 85A6      E896A6       E896A6       F3B0B385
9146     9146 9146 9146 9146 518E C1A7 C1A7 C1A7 C1A7 4127 C17D C1A7 8A6E      E8A9AE       E8A9AE       F3B0B386
9147     9147 9147 9147 9147 6478 ECCD C1A8 C1A8 ECCD 6C4D 71D4 C1A8 8CCE      E8B38E       E8B38E       F3B0B387
9148     9148 9148 9148 9148 50D1 C1A9 C1A9 C1A9 C1A9 4129 C15D C1A9 8DF5      E8B7B5       E8B7B5       F3B0B388
9149     9149 9149 9149 9149 485D C1AA C1AA C1AA C1AA 412A C15C C1AA 9078      E981B8       E981B8       F3B0B389
914A     914A 914A 914A 914A 535B C1AB C1AB C1AB C1AB 412B C14E C1AB 9077      E981B7       E981B7       F3B0B38A
914B     914B 914B 914B 914B 4D48 C1AC C1AC C1AC C1AC 412C C16B C1AC 92AD      E98AAD       E98AAD       F3B0B38B
914C     914C 914C 914C 914C 4CAD C1AD C1AD C1AD C1AD 412D C160 C1AD 9291      E98A91       E98A91       F3B0B38C
914D     914D 914D 914D 914D 519C C1AE C1AE C1AE C1AE 412E C14B C1AE 9583      E99683       E99683       F3B0B38D
914E     914E 914E 914E 914E 4CE9 C1AF C1AF C1AF C1AF 412F C161 C1AF 9BAE      E9AEAE       E9AEAE       F3B0B38E
914F     914F 914F 914F 914F 45AF C1B0 C1B0 C1B0 C1B0 4130 C1F0 C1B0 524D      E5898D       E5898D       F3B0B38F
9150     9150 9150 9150 9150 46B3 C1B1 C1B1 C1B1 C1B1 4131 C1F1 C1B1 5584      E59684       E59684       F3B0B390
9151     9151 9151 9151 9151 4D73 C1B2 C1B2 C1B2 C1B2 4132 C1F2 C1B2 6F38      E6BCB8       E6BCB8       F3B0B391
9152     9152 9152 9152 9152 47EF C1B3 C1B3 C1B3 C1B3 4133 C1F3 C1B3 7136      E784B6       E784B6       F3B0B392
9153     9153 9153 9153 9153 46B1 C1B4 C1B4 C1B4 C1B4 4134 C1F4 C1B4 5168      E585A8       E585A8       F3B0B393
9154     9154 9154 9154 9154 4F87 C1B5 C1B5 C1B5 C1B5 4135 C1F5 C1B5 7985      E7A685       E7A685       F3B0B394
9155     9155 9155 9155 9155 51D4 C1B6 C1B6 C1B6 C1B6 4136 C1F6 C1B6 7E55      E7B995       E7B995       F3B0B395
9156     9156 9156 9156 9156 5076 C1B7 C1B7 C1B7 C1B7 4137 C1F7 C1B7 81B3      E886B3       E886B3       F3B0B396
9157     9157 9157 9157 9157 5160 C1B8 C1B8 C1B8 C1B8 4138 C1F8 C1B8 7CCE      E7B38E       E7B38E       F3B0B397
9158     9158 9158 9158 9158 5062 C1B9 C1B9 C1B9 C1B9 4139 C1F9 C1B9 564C      E5998C       E5998C       F3B0B398
9159     9159 9159 9159 9159 4EC6 C1BA C1BA C1BA C1BA 413A C17A C1BA 5851      E5A191       E5A191       F3B0B399
915A     915A 915A 915A 915A 54B9 C1BB C1BB C1BB C1BB 413B C15E C1BB 5CA8      E5B2A8       E5B2A8       F3B0B39A
915B     915B 915B 915B 915B 4A60 C1BC C1BC C1BC C1BC 413C C14C C1BC 63AA      E68EAA       E68EAA       F3B0B39B
915C     915C 915C 915C 915C 5166 C1BD C1BD C1BD C1BD 413D C17E C1BD 66FE      E69BBE       E69BBE       F3B0B39C
915D     915D 915D 915D 915D 495F C1BE C1BE C1BE C1BE 413E C16E C1BE 66FD      E69BBD       E69BBD       F3B0B39D
915E     915E 915E 915E 915E 534C C1BF C1BF C1BF C1BF 413F C16F C1BF 695A      E6A59A       E6A59A       F3B0B39E
915F     915F 915F 915F 915F 4ECF C1C0 C1C0 C1C0 C1C0 4140 C17C C1C0 72D9      E78B99       E78B99       F3B0B39F
9160     9160 9160 9160 9160 55B0 C1C1 C1C1 C1C1 C1C1 4141 C1C1 C1C1 758F      E7968F       E7968F       F3B0B3A0
9161     9161 9161 9161 9161 5171 C1C2 C1C2 C1C2 C1C2 4142 C1C2 C1C2 758E      E7968E       E7968E       F3B0B3A1
9162     9162 9162 9162 9162 4C9C C1C3 C1C3 C1C3 C1C3 4143 C1C3 C1C3 790E      E7A48E       E7A48E       F3B0B3A2
9163     9163 9163 9163 9163 4CDC C1C4 C1C4 C1C4 C1C4 4144 C1C4 C1C4 7956      E7A596       E7A596       F3B0B3A3
9164     9164 9164 9164 9164 4D83 C1C5 C1C5 C1C5 C1C5 4145 C1C5 C1C5 79DF      E7A79F       E7A79F       F3B0B3A4
9165     9165 9165 9165 9165 4D59 C1C6 C1C6 C1C6 C1C6 4146 C1C6 C1C6 7C97      E7B297       E7B297       F3B0B3A5
9166     9166 9166 9166 9166 479F C1C7 C1C7 C1C7 C1C7 4147 C1C7 C1C7 7D20      E7B4A0       E7B4A0       F3B0B3A6
9167     9167 9167 9167 9167 46B9 C1C8 C1C8 C1C8 C1C8 4148 C1C8 C1C8 7D44      E7B584       E7B584       F3B0B3A7
9168     9168 9168 9168 9168 4CBF C1C9 C1C9 C1C9 C1C9 4149 C1C9 C1C9 8607      E89887       E89887       F3B0B3A8
9169     9169 9169 9169 9169 4CDF C1CA C1CA C1CA C1CA 414A C1D1 C1CA 8A34      E8A8B4       E8A8B4       F3B0B3A9
916A     916A 916A 916A 916A 4FEF C1CB C1CB C1CB C1CB 414B C1D2 C1CB 963B      E998BB       E998BB       F3B0B3AA
916B     916B 916B 916B 916B 544C C1CC C1CC C1CC C1CC 414C C1D3 C1CC 9061      E981A1       E981A1       F3B0B3AB
916C     916C 916C 916C 916C 50CB C1CD C1CD C1CD C1CD 414D C1D4 C1CD 9F20      E9BCA0       E9BCA0       F3B0B3AC
916D     916D 916D 916D 916D 4E84 C1CE C1CE C1CE C1CE 414E C1D5 C1CE 50E7      E583A7       E583A7       F3B0B3AD
916E     916E 916E 916E 916E 4BE1 C1CF C1CF C1CF C1CF 414F C1D6 C1CF 5275      E589B5       E589B5       F3B0B3AE
916F     916F 916F 916F 916F 4B71 C1D0 C1D0 C1D0 C1D0 4150 C1D7 C1D0 53CC      E58F8C       E58F8C       F3B0B3AF
9170     9170 9170 9170 9170 53F5 C1D1 C1D1 C1D1 C1D1 4151 C1D8 C1D1 53E2      E58FA2       E58FA2       F3B0B3B0
9171     9171 9171 9171 9171 45D6 C1D2 C1D2 C1D2 C1D2 4152 C1D9 C1D2 5009      E58089       E58089       F3B0B3B1
9172     9172 9172 9172 9172 5168 C1D3 C1D3 C1D3 C1D3 4153 C1E2 C1D3 55AA      E596AA       E596AA       F3B0B3B2
9173     9173 9173 9173 9173 4C9F C1D4 C1D4 C1D4 C1D4 4154 C1E3 C1D4 58EE      E5A3AE       E5A3AE       F3B0B3B3
9174     9174 9174 9174 9174 4C53 C1D5 C1D5 C1D5 C1D5 4155 C1E4 C1D5 594F      E5A58F       E5A58F       F3B0B3B4
9175     9175 9175 9175 9175 52DD C1D6 C1D6 C1D6 C1D6 4156 C1E5 C1D6 723D      E788BD       E788BD       F3B0B3B5
9176     9176 9176 9176 9176 524D C1D7 C1D7 C1D7 C1D7 4157 C1E6 C1D7 5B8B      E5AE8B       E5AE8B       F3B0B3B6
9177     9177 9177 9177 9177 4A8E C1D8 C1D8 C1D8 C1D8 4158 C1E7 C1D8 5C64      E5B1A4       E5B1A4       F3B0B3B7
9178     9178 9178 9178 9178 51B7 C1D9 C1D9 C1D9 C1D9 4159 C1E8 C1D9 531D      E58C9D       E58C9D       F3B0B3B8
9179     9179 9179 9179 9179 4B68 C1DA C1DA C1DA C1DA 415A C1E9 C1DA 60E3      E683A3       E683A3       F3B0B3B9
917A     917A 917A 917A 917A 48B3 C1DB C1DB C1DB C1DB 415B C14A C1DB 60F3      E683B3       E683B3       F3B0B3BA
917B     917B 917B 917B 917B 4DDD C1DC C1DC C1DC C1DC 415C C15B C1DC 635C      E68D9C       E68D9C       F3B0B3BB
917C     917C 917C 917C 917C 4DC3 C1DD C1DD C1DD C1DD 415D C15A C1DD 6383      E68E83       E68E83       F3B0B3BC
917D     917D 917D 917D 917D 5555 C1DE C1DE C1DE C1DE 415E C15F C1DE 633F      E68CBF       E68CBF       F3B0B3BD
917E     917E 917E 917E 917E 54A3 C1DF C1DF C1DF 50B9 D6D4 ACA9 C1DF 63BB      E68EBB       E68EBB       F3B0B3BE
9180     9180 9180 9180 9180 499B C1E0 C1E0 C1E0 C1E0 4160 C179 C1E0 64CD      E6938D       E6938D       F3B0B3BF
9181     9181 9181 9181 9181 46E9 C1E1 C1E1 C1E1 C1E1 4161 C157 C1E1 65E9      E697A9       E697A9       F3B0B480
9182     9182 9182 9182 9182 4ED2 C1E2 C1E2 C1E2 C1E2 4162 C159 C1E2 66F9      E69BB9       E69BB9       F3B0B481
9183     9183 9183 9183 9183 4C67 C1E3 C1E3 C1E3 C1E3 4163 C162 C1E3 5DE3      E5B7A3       E5B7A3       F3B0B482
9184     9184 9184 9184 9184 52DB C1E4 C1E4 C1E4 C1E4 4164 C163 C1E4 69CD      E6A78D       E6A78D       F3B0B483
9185     9185 9185 9185 9185 4DB7 C1E5 C1E5 C1E5 C1E5 4165 C164 C1E5 69FD      E6A7BD       E6A7BD       F3B0B484
9186     9186 9186 9186 9186 54A1 C1E6 C1E6 C1E6 C1E6 4166 C165 C1E6 6F15      E6BC95       E6BC95       F3B0B485
9187     9187 9187 9187 9187 4C66 C1E7 C1E7 C1E7 C1E7 4167 C166 C1E7 71E5      E787A5       E787A5       F3B0B486
9188     9188 9188 9188 9188 4896 C1E8 C1E8 C1E8 C1E8 4168 C167 C1E8 4E89      E4BA89       E4BA89       F3B0B487
9189     9189 9189 9189 9189 54A4 C1E9 C1E9 C1E9 5AF5 D6D5 ACAA C1E9 75E9      E797A9       E797A9       F3B0B488
918A     918A 918A 918A 918A 4678 C1EA C1EA C1EA C1EA 416A C169 C1EA 76F8      E79BB8       E79BB8       F3B0B489
918B     918B 918B 918B 918B 4BE9 C1EB C1EB C1EB C1EB 416B C170 C1EB 7A93      E7AA93       E7AA93       F3B0B48A
918C     918C 918C 918C 918C 518C C1EC C1EC C1EC C1EC 416C C171 C1EC 7CDF      E7B39F       E7B39F       F3B0B48B
918D     918D 918D 918D 918D 4793 C1ED C1ED C1ED C1ED 416D C172 C1ED 7DCF      E7B78F       E7B78F       F3B0B48C
918E     918E 918E 918E 918E 5070 C1EE C1EE C1EE C1EE 416E C173 C1EE 7D9C      E7B69C       E7B69C       F3B0B48D
918F     918F 918F 918F 918F 4B89 C1EF C1EF C1EF C1EF 416F C174 C1EF 8061      E881A1       E881A1       F3B0B48E
9190     9190 9190 9190 9190 4842 C1F0 C1F0 C1F0 C1F0 4170 C175 C1F0 8349      E88D89       E88D89       F3B0B48F
9191     9191 9191 9191 9191 4B62 C1F1 C1F1 C1F1 C1F1 4171 C176 C1F1 8358      E88D98       E88D98       F3B0B490
9192     9192 9192 9192 9192 4F79 C1F2 C1F2 C1F2 C1F2 4172 C177 C1F2 846C      E891AC       E891AC       F3B0B491
9193     9193 9193 9193 9193 50B9 C1F3 C1F3 C1F3 C1F3 4173 C178 C1F3 84BC      E892BC       E892BC       F3B0B492
9194     9194 9194 9194 9194 505F C1F4 C1F4 C1F4 C1F4 4174 C180 C1F4 85FB      E897BB       E897BB       F3B0B493
9195     9195 9195 9195 9195 4898 C1F5 C1F5 C1F5 C1F5 4175 C18B C1F5 88C5      E8A385       E8A385       F3B0B494
9196     9196 9196 9196 9196 4AAA C1F6 C1F6 C1F6 C1F6 4176 C19B C1F6 8D70      E8B5B0       E8B5B0       F3B0B495
9197     9197 9197 9197 9197 4744 C1F7 C1F7 C1F7 C1F7 4177 C19C C1F7 9001      E98081       E98081       F3B0B496
9198     9198 9198 9198 9198 5143 C1F8 C1F8 C1F8 C1F8 4178 C1A0 C1F8 906D      E981AD       E981AD       F3B0B497
9199     9199 9199 9199 9199 5197 C1F9 C1F9 C1F9 C1F9 4179 C1AB C1F9 9397      E98E97       E98E97       F3B0B498
919A     919A 919A 919A 919A 4DE8 C1FA C1FA C1FA C1FA 417A C1B0 C1FA 971C      E99C9C       E99C9C       F3B0B499
919B     919B 919B 919B 919B 4DD2 C1FB C1FB C1FB C1FB 417B C1C0 C1FB 9A12      E9A892       E9A892       F3B0B49A
919C     919C 919C 919C 919C 4BBA C1FC C1FC C1FC C1FC 417C C16A C1FC 50CF      E5838F       E5838F       F3B0B49B
919D     919D 919D 919D 919D 464D C1FD C1FD C1FD C1FD 417D C1D0 C1FD 5897      E5A297       E5A297       F3B0B49C
919E     919E 919E 919E 919E 4F4A C1FE C1FE C1FE C1FE 417E C1A1 C1FE 618E      E6868E       E6868E       F3B0B49D
919F     919F 919F 919F 919F 4DBD C2A1 C2A1 C2A1 C2A1 4221 C24F C2A1 81D3      E88793       E88793       F3B0B49E
91A0     91A0 91A0 91A0 91A0 465E C2A2 C2A2 C2A2 C2A2 4222 C27F C2A2 8535      E894B5       E894B5       F3B0B49F
91A1     91A1 91A1 91A1 91A1 4C7C C2A3 C2A3 C2A3 C2A3 4223 C27B C2A3 8D08      E8B488       E8B488       F3B0B4A0
91A2     91A2 91A2 91A2 91A2 45B8 C2A4 C2A4 C2A4 C2A4 4224 C2E0 C2A4 9020      E980A0       E980A0       F3B0B4A1
91A3     91A3 91A3 91A3 91A3 4A6A C2A5 C2A5 C2A5 C2A5 4225 C26C C2A5 4FC3      E4BF83       E4BF83       F3B0B4A2
91A4     91A4 91A4 91A4 91A4 4943 C2A6 C2A6 C2A6 C2A6 4226 C250 C2A6 5074      E581B4       E581B4       F3B0B4A3
91A5     91A5 91A5 91A5 91A5 474D C2A7 C2A7 C2A7 C2A7 4227 C27D C2A7 5247      E58987       E58987       F3B0B4A4
91A6     91A6 91A6 91A6 91A6 4B6D C2A8 C2A8 C2A8 C2A8 4228 C24D C2A8 5373      E58DB3       E58DB3       F3B0B4A5
91A7     91A7 91A7 91A7 91A7 4BB0 C2A9 C2A9 C2A9 C2A9 4229 C25D C2A9 606F      E681AF       E681AF       F3B0B4A6
91A8     91A8 91A8 91A8 91A8 53A0 C2AA C2AA C2AA C2AA 422A C25C C2AA 6349      E68D89       E68D89       F3B0B4A7
91A9     91A9 91A9 91A9 91A9 4B4B C2AB C2AB C2AB C2AB 422B C24E C2AB 675F      E69D9F       E69D9F       F3B0B4A8
91AA     91AA 91AA 91AA 91AA 4945 C2AC C2AC C2AC C2AC 422C C26B C2AC 6E2C      E6B8AC       E6B8AC       F3B0B4A9
91AB     91AB 91AB 91AB 91AB 46FA C2AD C2AD C2AD C2AD 422D C260 C2AD 8DB3      E8B6B3       E8B6B3       F3B0B4AA
91AC     91AC 91AC 91AC 91AC 48CD C2AE C2AE C2AE C2AE 422E C24B C2AE 901F      E9809F       E9809F       F3B0B4AB
91AD     91AD 91AD 91AD 91AD 4E97 C2AF C2AF C2AF C2AF 422F C261 C2AF 4FD7      E4BF97       E4BF97       F3B0B4AC
91AE     91AE 91AE 91AE 91AE 476D C2B0 C2B0 C2B0 C2B0 4230 C2F0 C2B0 5C5E      E5B19E       E5B19E       F3B0B4AD
91AF     91AF 91AF 91AF 91AF 4FFD C2B1 C2B1 C2B1 C2B1 4231 C2F1 C2B1 8CCA      E8B38A       E8B38A       F3B0B4AE
91B0     91B0 91B0 91B0 91B0 4ADC C2B2 C2B2 C2B2 C2B2 4232 C2F2 C2B2 65CF      E6978F       E6978F       F3B0B4AF
91B1     91B1 91B1 91B1 91B1 47CB C2B3 C2B3 C2B3 C2B3 4233 C2F3 C2B3 7D9A      E7B69A       E7B69A       F3B0B4B0
91B2     91B2 91B2 91B2 91B2 4AF2 C2B4 C2B4 C2B4 C2B4 4234 C2F4 C2B4 5352      E58D92       E58D92       F3B0B4B1
91B3     91B3 91B3 91B3 91B3 4A57 C2B5 C2B5 C2B5 C2B5 4235 C2F5 C2B5 8896      E8A296       E8A296       F3B0B4B2
91B4     91B4 91B4 91B4 91B4 4F67 C2B6 C2B6 C2B6 C2B6 4236 C2F6 C2B6 5176      E585B6       E585B6       F3B0B4B3
91B5     91B5 91B5 91B5 91B5 4EFD C2B7 C2B7 C2B7 C2B7 4237 C2F7 C2B7 63C3      E68F83       E68F83       F3B0B4B4
91B6     91B6 91B6 91B6 91B6 4A41 C2B8 C2B8 C2B8 C2B8 4238 C2F8 C2B8 5B58      E5AD98       E5AD98       F3B0B4B5
91B7     91B7 91B7 91B7 91B7 4C5E C2B9 C2B9 C2B9 C2B9 4239 C2F9 C2B9 5B6B      E5ADAB       E5ADAB       F3B0B4B6
91B8     91B8 91B8 91B8 91B8 4CEF C2BA C2BA C2BA C2BA 423A C27A C2BA 5C0A      E5B08A       E5B08A       F3B0B4B7
91B9     91B9 91B9 91B9 91B9 4BD8 C2BB C2BB C2BB C2BB 423B C25E C2BB 640D      E6908D       E6908D       F3B0B4B8
91BA     91BA 91BA 91BA 91BA 4556 C2BC C2BC C2BC C2BC 423C C24C C2BC 6751      E69D91       E69D91       F3B0B4B9
91BB     91BB 91BB 91BB 91BB 5455 C2BD C2BD C2BD C2BD 423D C27E C2BD 905C      E9819C       E9819C       F3B0B4BA
91BC     91BC 91BC 91BC 91BC 4755 C2BE C2BE C2BE C2BE 423E C26E C2BE 4ED6      E4BB96       E4BB96       F3B0B4BB
91BD     91BD 91BD 91BD 91BD 45FE C2BF C2BF C2BF C2BF 423F C26F C2BF 591A      E5A49A       E5A49A       F3B0B4BC
91BE     91BE 91BE 91BE 91BE 45AB C2C0 C2C0 C2C0 C2C0 4240 C27C C2C0 592A      E5A4AA       E5A4AA       F3B0B4BD
91BF     91BF 91BF 91BF 91BF 50D2 C2C1 C2C1 C2C1 C2C1 4241 C2C1 C2C1 6C70      E6B1B0       E6B1B0       F3B0B4BE
91C0     91C0 91C0 91C0 91C0 55E3 C2C2 C2C2 C2C2 C2C2 4242 C2C2 C2C2 8A51      E8A991       E8A991       F3B0B4BF
91C1     91C1 91C1 91C1 91C1 5478 C2C3 C2C3 C2C3 C2C3 4243 C2C3 C2C3 553E      E594BE       E594BE       F3B0B580
91C2     91C2 91C2 91C2 91C2 53B0 C2C4 C2C4 C2C4 C2C4 4244 C2C4 C2C4 5815      E5A095       E5A095       F3B0B581
91C3     91C3 91C3 91C3 91C3 4D9E C2C5 C2C5 C2C5 C2C5 4245 C2C5 C2C5 59A5      E5A6A5       E5A6A5       F3B0B582
91C4     91C4 91C4 91C4 91C4 54B8 C2C6 C2C6 C2C6 C2C6 4246 C2C6 C2C6 60F0      E683B0       E683B0       F3B0B583
91C5     91C5 91C5 91C5 91C5 47EA C2C7 C2C7 C2C7 C2C7 4247 C2C7 C2C7 6253      E68993       E68993       F3B0B584
91C6     91C6 91C6 91C6 91C6 55CD C2C8 C2C8 C2C8 C2C8 4248 C2C8 C2C8 67C1      E69F81       E69F81       F3B0B585
91C7     91C7 91C7 91C7 91C7 54DB C2C9 C2C9 C2C9 C2C9 4249 C2C9 C2C9 8235      E888B5       E888B5       F3B0B586
91C8     91C8 91C8 91C8 91C8 55C5 C2CA C2CA C2CA C2CA 424A C2D1 C2CA 6955      E6A595       E6A595       F3B0B587
91C9     91C9 91C9 91C9 91C9 4FE8 C2CB C2CB C2CB C2CB 424B C2D2 C2CB 9640      E99980       E99980       F3B0B588
91CA     91CA 91CA 91CA 91CA 4DE5 C2CC C2CC C2CC C2CC 424C C2D3 C2CC 99C4      E9A784       E9A784       F3B0B589
91CB     91CB 91CB 91CB 91CB 53DA C2CD C2CD C2CD 6DF8 D4D1 A9A6 C2CD 9A28      E9A8A8       E9A8A8       F3B0B58A
91CC     91CC 91CC 91CC 91CC 4696 C2CE C2CE C2CE C2CE 424E C2D5 C2CE 4F53      E4BD93       E4BD93       F3B0B58B
91CD     91CD 91CD 91CD 91CD 5371 C2CF C2CF C2CF C2CF 424F C2D6 C2CF 5806      E5A086       E5A086       F3B0B58C
91CE     91CE 91CE 91CE 91CE 4656 C2D0 C2D0 C2D0 C2D0 4250 C2D7 C2D0 5BFE      E5AFBE       E5AFBE       F3B0B58D
91CF     91CF 91CF 91CF 91CF 498E C2D1 C2D1 C2D1 C2D1 4251 C2D8 C2D1 8010      E88090       E88090       F3B0B58E
91D0     91D0 91D0 91D0 91D0 5262 C2D2 C2D2 C2D2 C2D2 4252 C2D9 C2D2 5CB1      E5B2B1       E5B2B1       F3B0B58F
91D1     91D1 91D1 91D1 91D1 4960 C2D3 C2D3 C2D3 C2D3 4253 C2E2 C2D3 5E2F      E5B8AF       E5B8AF       F3B0B590
91D2     91D2 91D2 91D2 91D2 48A3 C2D4 C2D4 C2D4 C2D4 4254 C2E3 C2D4 5F85      E5BE85       E5BE85       F3B0B591
91D3     91D3 91D3 91D3 91D3 5081 C2D5 C2D5 C2D5 C2D5 4255 C2E4 C2D5 6020      E680A0       E680A0       F3B0B592
91D4     91D4 91D4 91D4 91D4 48FD C2D6 C2D6 C2D6 C2D6 4256 C2E5 C2D6 614B      E6858B       E6858B       F3B0B593
91D5     91D5 91D5 91D5 91D5 5092 C2D7 C2D7 C2D7 C2D7 4257 C2E6 C2D7 6234      E688B4       E688B4       F3B0B594
91D6     91D6 91D6 91D6 91D6 4BC3 C2D8 C2D8 C2D8 C2D8 4258 C2E7 C2D8 66FF      E69BBF       E69BBF       F3B0B595
91D7     91D7 91D7 91D7 91D7 46F3 C2D9 C2D9 C2D9 C2D9 4259 C2E8 C2D9 6CF0      E6B3B0       E6B3B0       F3B0B596
91D8     91D8 91D8 91D8 91D8 4DA1 C2DA C2DA C2DA C2DA 425A C2E9 C2DA 6EDE      E6BB9E       E6BB9E       F3B0B597
91D9     91D9 91D9 91D9 91D9 5263 C2DB C2DB C2DB C2DB 425B C24A C2DB 80CE      E8838E       E8838E       F3B0B598
91DA     91DA 91DA 91DA 91DA 529A C2DC C2DC C2DC C2DC 425C C25B C2DC 817F      E885BF       E885BF       F3B0B599
91DB     91DB 91DB 91DB 91DB 52D7 C2DD C2DD C2DD C2DD 425D C25A C2DD 82D4      E88B94       E88B94       F3B0B59A
91DC     91DC 91DC 91DC 91DC 4A54 C2DE C2DE C2DE C2DE 425E C25F C2DE 888B      E8A28B       E8A28B       F3B0B59B
91DD     91DD 91DD 91DD 91DD 4AF4 C2DF C2DF C2DF C2DF 425F C26D C2DF 8CB8      E8B2B8       E8B2B8       F3B0B59C
91DE     91DE 91DE 91DE 91DE 4B49 C2E0 C2E0 C2E0 C2E0 4260 C279 C2E0 9000      E98080       E98080       F3B0B59D
91DF     91DF 91DF 91DF 91DF 4F77 C2E1 C2E1 C2E1 C2E1 4261 C257 C2E1 902E      E980AE       E980AE       F3B0B59E
91E0     91E0 91E0 91E0 91E0 4A93 C2E2 C2E2 C2E2 C2E2 4262 C259 C2E2 968A      E99A8A       E99A8A       F3B0B59F
91E1     91E1 91E1 91E1 91E1 51DD C2E3 C2E3 C2E3 C2E3 4263 C262 C2E3 9EDB      E9BB9B       E9BB9B       F3B0B5A0
91E2     91E2 91E2 91E2 91E2 50D8 C2E4 C2E4 C2E4 C2E4 4264 C263 C2E4 9BDB      E9AF9B       E9AF9B       F3B0B5A1
91E3     91E3 91E3 91E3 91E3 459A C2E5 C2E5 C2E5 C2E5 4265 C264 C2E5 4EE3      E4BBA3       E4BBA3       F3B0B5A2
91E4     91E4 91E4 91E4 91E4 4778 C2E6 C2E6 C2E6 C2E6 4266 C265 C2E6 53F0      E58FB0       E58FB0       F3B0B5A3
91E5     91E5 91E5 91E5 91E5 455B C2E7 C2E7 C2E7 C2E7 4267 C266 C2E7 5927      E5A4A7       E5A4A7       F3B0B5A4
91E6     91E6 91E6 91E6 91E6 467E C2E8 C2E8 C2E8 C2E8 4268 C267 C2E8 7B2C      E7ACAC       E7ACAC       F3B0B5A5
91E7     91E7 91E7 91E7 91E7 516F C2E9 C2E9 C2E9 C2E9 4269 C268 C2E9 918D      E9868D       E9868D       F3B0B5A6
91E8     91E8 91E8 91E8 91E8 47FE C2EA C2EA C2EA C2EA 426A C269 C2EA 984C      E9A18C       E9A18C       F3B0B5A7
91E9     91E9 91E9 91E9 91E9 4B7A C2EB C2EB C2EB C2EB 426B C270 C2EB 9DF9      E9B7B9       E9B7B9       F3B0B5A8
91EA     91EA 91EA 91EA 91EA 47A3 C2EC C2EC C2EC C2EC 426C C271 C2EC 6EDD      E6BB9D       E6BB9D       F3B0B5A9
91EB     91EB 91EB 91EB 91EB 52AE C2ED C2ED C2ED C2ED 426D C272 C2ED 7027      E780A7       E780A7       F3B0B5AA
91EC     91EC 91EC 91EC 91EC 4996 C2EE C2EE C2EE C2EE 426E C273 C2EE 5353      E58D93       E58D93       F3B0B5AB
91ED     91ED 91ED 91ED 91ED 5383 C2EF C2EF C2EF C2EF 426F C274 C2EF 5544      E59584       E59584       F3B0B5AC
91EE     91EE 91EE 91EE 91EE 486F C2F0 C2F0 C2F0 C2F0 4270 C275 C2F0 5B85      E5AE85       E5AE85       F3B0B5AD
91EF     91EF 91EF 91EF 91EF 55D5 C2F1 C2F1 C2F1 C2F1 4271 C276 C2F1 6258      E68998       E68998       F3B0B5AE
91F0     91F0 91F0 91F0 91F0 4E7E C2F2 C2F2 C2F2 C2F2 4272 C277 C2F2 629E      E68A9E       E68A9E       F3B0B5AF
91F1     91F1 91F1 91F1 91F1 4BDB C2F3 C2F3 C2F3 C2F3 4273 C278 C2F3 62D3      E68B93       E68B93       F3B0B5B0
91F2     91F2 91F2 91F2 91F2 4585 C2F4 C2F4 C2F4 C2F4 4274 C280 C2F4 6CA2      E6B2A2       E6B2A2       F3B0B5B1
91F3     91F3 91F3 91F3 91F3 4DD4 C2F5 C2F5 C2F5 C2F5 4275 C28B C2F5 6FEF      E6BFAF       E6BFAF       F3B0B5B2
91F4     91F4 91F4 91F4 91F4 4E5C C2F6 C2F6 C2F6 59A1 C4C8 9599 C2F6 7422      E790A2       E790A2       F3B0B5B3
91F5     91F5 91F5 91F5 91F5 4AB6 C2F7 C2F7 C2F7 C2F7 4277 C29C C2F7 8A17      E8A897       E8A897       F3B0B5B4
91F6     91F6 91F6 91F6 91F6 5388 C2F8 C2F8 C2F8 C2F8 4278 C2A0 C2F8 9438      E990B8       E990B8       F3B0B5B5
91F7     91F7 91F7 91F7 91F7 50BE C2F9 C2F9 C2F9 C2F9 4279 C2AB C2F9 6FC1      E6BF81       E6BF81       F3B0B5B6
91F8     91F8 91F8 91F8 91F8 4FE1 C2FA C2FA C2FA C2FA 427A C2B0 C2FA 8AFE      E8ABBE       E8ABBE       F3B0B5B7
91F9     91F9 91F9 91F9 91F9 4DF7 C2FB C2FB C2FB C2FB 427B C2C0 C2FB 8338      E88CB8       E88CB8       F3B0B5B8
91FA     91FA 91FA 91FA 91FA 558A C2FC C2FC C2FC C2FC 427C C26A C2FC 51E7      E587A7       E587A7       F3B0B5B9
91FB     91FB 91FB 91FB 91FB 524E C2FD C2FD C2FD C2FD 427D C2D0 C2FD 86F8      E89BB8       E89BB8       F3B0B5BA
91FC     91FC 91FC 91FC 91FC 4E41 C2FE C2FE C2FE C2FE 427E C2A1 C2FE 53EA      E58FAA       E58FAA       F3B0B5BB
9240     9240 9240 9240 9240 4F71 C3A1 C3A1 C3A1 C3A1 4321 C34F C3A1 53E9      E58FA9       E58FA9       F3B0B5BC
9241     9241 9241 9241 9241 4EFE C3A2 C3A2 C3A2 C3A2 4322 C37F C3A2 4F46      E4BD86       E4BD86       F3B0B5BD
9242     9242 9242 9242 9242 46A4 C3A3 C3A3 C3A3 C3A3 4323 C37B C3A3 9054      E98194       E98194       F3B0B5BE
9243     9243 9243 9243 9243 47DF C3A4 C3A4 C3A4 C3A4 4324 C3E0 C3A4 8FB0      E8BEB0       E8BEB0       F3B0B5BF
9244     9244 9244 9244 9244 4EBB C3A5 C3A5 C3A5 C3A5 4325 C36C C3A5 596A      E5A5AA       E5A5AA       F3B0B680
9245     9245 9245 9245 9245 4AB4 C3A6 C3A6 C3A6 C3A6 4326 C350 C3A6 8131      E884B1       E884B1       F3B0B681
9246     9246 9246 9246 9246 4E46 C3A7 C3A7 C3A7 C3A7 4327 C37D C3A7 5DFD      E5B7BD       E5B7BD       F3B0B682
9247     9247 9247 9247 9247 51C8 C3A8 C3A8 C3A8 C3A8 4328 C34D C3A8 7AEA      E7ABAA       E7ABAA       F3B0B683
9248     9248 9248 9248 9248 547E C3A9 C3A9 C3A9 C3A9 4329 C35D C3A9 8FBF      E8BEBF       E8BEBF       F3B0B684
9249     9249 9249 9249 9249 4BEF C3AA C3AA C3AA C3AA 432A C35C C3AA 68DA      E6A39A       E6A39A       F3B0B685
924A     924A 924A 924A 924A 4581 C3AB C3AB C3AB C3AB 432B C34E C3AB 8C37      E8B0B7       E8B0B7       F3B0B686
924B     924B 924B 924B 924B 52FE C3AC C3AC C3AC C3AC 432C C36B C3AC 72F8      E78BB8       E78BB8       F3B0B687
924C     924C 924C 924C 924C 5560 C3AD C3AD C3AD C3AD 432D C360 C3AD 9C48      E9B188       E9B188       F3B0B688
924D     924D 924D 924D 924D 4BBE C3AE C3AE C3AE C3AE 432E C34B C3AE 6A3D      E6A8BD       E6A8BD       F3B0B689
924E     924E 924E 924E 924E 4B83 C3AF C3AF C3AF C3AF 432F C361 C3AF 8AB0      E8AAB0       E8AAB0       F3B0B68A
924F     924F 924F 924F 924F 48AB C3B0 C3B0 C3B0 C3B0 4330 C3F0 C3B0 4E39      E4B8B9       E4B8B9       F3B0B68B
9250     9250 9250 9250 9250 48E3 C3B1 C3B1 C3B1 C3B1 4331 C3F1 C3B1 5358      E58D98       E58D98       F3B0B68C
9251     9251 9251 9251 9251 5063 C3B2 C3B2 C3B2 C3B2 4332 C3F2 C3B2 5606      E59886       E59886       F3B0B68D
9252     9252 9252 9252 9252 51C4 C3B3 C3B3 C3B3 C3B3 4333 C3F3 C3B3 5766      E59DA6       E59DA6       F3B0B68E
9253     9253 9253 9253 9253 4AA1 C3B4 C3B4 C3B4 C3B4 4334 C3F4 C3B4 62C5      E68B85       E68B85       F3B0B68F
9254     9254 9254 9254 9254 4B7F C3B5 C3B5 C3B5 C3B5 4335 C3F5 C3B5 63A2      E68EA2       E68EA2       F3B0B690
9255     9255 9255 9255 9255 4D9D C3B6 C3B6 C3B6 C3B6 4336 C3F6 C3B6 65E6      E697A6       E697A6       F3B0B691
9256     9256 9256 9256 9256 5487 C3B7 C3B7 C3B7 C3B7 4337 C3F7 C3B7 6B4E      E6AD8E       E6AD8E       F3B0B692
9257     9257 9257 9257 9257 4CA5 C3B8 C3B8 C3B8 C3B8 4338 C3F8 C3B8 6DE1      E6B7A1       E6B7A1       F3B0B693
9258     9258 9258 9258 9258 53F3 C3B9 C3B9 C3B9 C3B9 4339 C3F9 C3B9 6E5B      E6B99B       E6B99B       F3B0B694
9259     9259 9259 9259 9259 4861 C3BA C3BA C3BA C3BA 433A C37A C3BA 70AD      E782AD       E782AD       F3B0B695
925A     925A 925A 925A 925A 4A82 C3BB C3BB C3BB C3BB 433B C35E C3BB 77ED      E79FAD       E79FAD       F3B0B696
925B     925B 925B 925B 925B 48EF C3BC C3BC C3BC C3BC 433C C34C C3BC 7AEF      E7ABAF       E7ABAF       F3B0B697
925C     925C 925C 925C 925C 54CA C3BD C3BD C3BD 5EC2 D6DB ACBB C3BD 7BAA      E7AEAA       E7AEAA       F3B0B698
925D     925D 925D 925D 925D 54A6 C3BE C3BE C3BE C3BE 433E C36E C3BE 7DBB      E7B6BB       E7B6BB       F3B0B699
925E     925E 925E 925E 925E 55B8 C3BF C3BF C3BF C3BF 433F C36F C3BF 803D      E880BD       E880BD       F3B0B69A
925F     925F 925F 925F 925F 4E9B C3C0 C3C0 C3C0 C3C0 4340 C37C C3C0 80C6      E88386       E88386       F3B0B69B
9260     9260 9260 9260 9260 53D8 C3C1 C3C1 C3C1 C3C1 4341 C3C1 C3C1 86CB      E89B8B       E89B8B       F3B0B69C
9261     9261 9261 9261 9261 4FA2 C3C2 C3C2 C3C2 C3C2 4342 C3C2 C3C2 8A95      E8AA95       E8AA95       F3B0B69D
9262     9262 9262 9262 9262 48C7 C3C3 C3C3 C3C3 C3C3 4343 C3C3 C3C3 935B      E98D9B       E98D9B       F3B0B69E
9263     9263 9263 9263 9263 47F0 C3C4 C3C4 C3C4 C3C4 4344 C3C4 C3C4 56E3      E59BA3       E59BA3       F3B0B69F
9264     9264 9264 9264 9264 4E8E C3C5 C3C5 C3C5 C3C5 4345 C3C5 C3C5 58C7      E5A387       E5A387       F3B0B6A0
9265     9265 9265 9265 9265 4AD4 C3C6 C3C6 C3C6 C3C6 4346 C3C6 C3C6 5F3E      E5BCBE       E5BCBE       F3B0B6A1
9266     9266 9266 9266 9266 4989 C3C7 C3C7 C3C7 C3C7 4347 C3C7 C3C7 65AD      E696AD       E696AD       F3B0B6A2
9267     9267 9267 9267 9267 4D6C C3C8 C3C8 C3C8 C3C8 4348 C3C8 C3C8 6696      E69A96       E69A96       F3B0B6A3
9268     9268 9268 9268 9268 4FF5 C3C9 C3C9 C3C9 C3C9 4349 C3C9 C3C9 6A80      E6AA80       E6AA80       F3B0B6A4
9269     9269 9269 9269 9269 47ED C3CA C3CA C3CA C3CA 434A C3D1 C3CA 6BB5      E6AEB5       E6AEB5       F3B0B6A5
926A     926A 926A 926A 926A 457C C3CB C3CB C3CB C3CB 434B C3D2 C3CB 7537      E794B7       E794B7       F3B0B6A6
926B     926B 926B 926B 926B 4963 C3CC C3CC C3CC C3CC 434C C3D3 C3CC 8AC7      E8AB87       E8AB87       F3B0B6A7
926C     926C 926C 926C 926C 487D C3CD C3CD C3CD C3CD 434D C3D4 C3CD 5024      E580A4       E580A4       F3B0B6A8
926D     926D 926D 926D 926D 45B0 C3CE C3CE C3CE C3CE 434E C3D5 C3CE 77E5      E79FA5       E79FA5       F3B0B6A9
926E     926E 926E 926E 926E 45C2 C3CF C3CF C3CF C3CF 434F C3D6 C3CF 5730      E59CB0       E59CB0       F3B0B6AA
926F     926F 926F 926F 926F 5597 C3D0 C3D0 C3D0 C3D0 4350 C3D7 C3D0 5F1B      E5BC9B       E5BC9B       F3B0B6AB
9270     9270 9270 9270 9270 4ECE C3D1 C3D1 C3D1 C3D1 4351 C3D8 C3D1 6065      E681A5       E681A5       F3B0B6AC
9271     9271 9271 9271 9271 46A8 C3D2 C3D2 C3D2 C3D2 4352 C3D9 C3D2 667A      E699BA       E699BA       F3B0B6AD
9272     9272 9272 9272 9272 45E8 C3D3 C3D3 C3D3 C3D3 4353 C3E2 C3D3 6C60      E6B1A0       E6B1A0       F3B0B6AE
9273     9273 9273 9273 9273 5089 C3D4 C3D4 C3D4 C3D4 4354 C3E3 C3D4 75F4      E797B4       E797B4       F3B0B6AF
9274     9274 9274 9274 9274 4DCF C3D5 C3D5 C3D5 C3D5 4355 C3E4 C3D5 7A1A      E7A89A       E7A89A       F3B0B6B0
9275     9275 9275 9275 9275 468C C3D6 C3D6 C3D6 C3D6 4356 C3E5 C3D6 7F6E      E7BDAE       E7BDAE       F3B0B6B1
9276     9276 9276 9276 9276 4C6A C3D7 C3D7 C3D7 C3D7 4357 C3E6 C3D7 81F4      E887B4       E887B4       F3B0B6B2
9277     9277 9277 9277 9277 54DC C3D8 C3D8 C3D8 C3D8 4358 C3E7 C3D8 8718      E89C98       E89C98       F3B0B6B3
9278     9278 9278 9278 9278 4C6D C3D9 C3D9 C3D9 C3D9 4359 C3E8 C3D9 9045      E98185       E98185       F3B0B6B4
9279     9279 9279 9279 9279 52DF C3DA C3DA C3DA C3DA 435A C3E9 C3DA 99B3      E9A6B3       E9A6B3       F3B0B6B5
927A     927A 927A 927A 927A 49A8 C3DB C3DB C3DB C3DB 435B C34A C3DB 7BC9      E7AF89       E7AF89       F3B0B6B6
927B     927B 927B 927B 927B 4D72 C3DC C3DC C3DC C3DC 435C C35B C3DC 755C      E7959C       E7959C       F3B0B6B7
927C     927C 927C 927C 927C 45F5 C3DD C3DD C3DD C3DD 435D C35A C3DD 7AF9      E7ABB9       E7ABB9       F3B0B6B8
927D     927D 927D 927D 927D 4B45 C3DE C3DE C3DE C3DE 435E C35F C3DE 7B51      E7AD91       E7AD91       F3B0B6B9
927E     927E 927E 927E 927E 4C94 C3DF C3DF C3DF C3DF 435F C36D C3DF 84C4      E89384       E89384       F3B0B6BA
9280     9280 9280 9280 9280 50BB C3E0 C3E0 C3E0 C3E0 4360 C379 C3E0 9010      E98090       E98090       F3B0B6BB
9281     9281 9281 9281 9281 4D9F C3E1 C3E1 C3E1 C3E1 4361 C357 C3E1 79E9      E7A7A9       E7A7A9       F3B0B6BC
9282     9282 9282 9282 9282 4FB0 C3E2 C3E2 C3E2 C3E2 4362 C359 C3E2 7A92      E7AA92       E7AA92       F3B0B6BD
9283     9283 9283 9283 9283 4977 C3E3 C3E3 C3E3 C3E3 4363 C362 C3E3 8336      E88CB6       E88CB6       F3B0B6BE
9284     9284 9284 9284 9284 5567 C3E4 C3E4 C3E4 C3E4 4364 C363 C3E4 5AE1      E5ABA1       E5ABA1       F3B0B6BF
9285     9285 9285 9285 9285 47AA C3E5 C3E5 C3E5 C3E5 4365 C364 C3E5 7740      E79D80       E79D80       F3B0B780
9286     9286 9286 9286 9286 455C C3E6 C3E6 C3E6 C3E6 4366 C365 C3E6 4E2D      E4B8AD       E4B8AD       F3B0B781
9287     9287 9287 9287 9287 4869 C3E7 C3E7 C3E7 C3E7 4367 C366 C3E7 4EF2      E4BBB2       E4BBB2       F3B0B782
9288     9288 9288 9288 9288 4D64 C3E8 C3E8 C3E8 C3E8 4368 C367 C3E8 5B99      E5AE99       E5AE99       F3B0B783
9289     9289 9289 9289 9289 464F C3E9 C3E9 C3E9 C3E9 4369 C368 C3E9 5FE0      E5BFA0       E5BFA0       F3B0B784
928A     928A 928A 928A 928A 4D53 C3EA C3EA C3EA C3EA 436A C369 C3EA 62BD      E68ABD       E68ABD       F3B0B785
928B     928B 928B 928B 928B 4C98 C3EB C3EB C3EB C3EB 436B C370 C3EB 663C      E698BC       E698BC       F3B0B786
928C     928C 928C 928C 928C 4BEA C3EC C3EC C3EC C3EC 436C C371 C3EC 67F1      E69FB1       E69FB1       F3B0B787
928D     928D 928D 928D 928D 4944 C3ED C3ED C3ED C3ED 436D C372 C3ED 6CE8      E6B3A8       E6B3A8       F3B0B788
928E     928E 928E 928E 928E 4BCB C3EE C3EE C3EE C3EE 436E C373 C3EE 866B      E899AB       E899AB       F3B0B789
928F     928F 928F 928F 928F 52E9 C3EF C3EF C3EF C3EF 436F C374 C3EF 8877      E8A1B7       E8A1B7       F3B0B78A
9290     9290 9290 9290 9290 4F66 C3F0 C3F0 C3F0 C3F0 4370 C375 C3F0 8A3B      E8A8BB       E8A8BB       F3B0B78B
9291     9291 9291 9291 9291 5591 C3F1 C3F1 C3F1 C3F1 4371 C376 C3F1 914E      E9858E       E9858E       F3B0B78C
9292     9292 9292 9292 9292 48C1 C3F2 C3F2 C3F2 C3F2 4372 C377 C3F2 92F3      E98BB3       E98BB3       F3B0B78D
9293     9293 9293 9293 9293 4DF5 C3F3 C3F3 C3F3 C3F3 4373 C378 C3F3 99D0      E9A790       E9A790       F3B0B78E
9294     9294 9294 9294 9294 538B C3F4 C3F4 C3F4 C3F4 4374 C380 C3F4 6A17      E6A897       E6A897       F3B0B78F
9295     9295 9295 9295 9295 4FDE C3F5 C3F5 C3F5 C3F5 4375 C38B C3F5 7026      E780A6       E780A6       F3B0B790
9296     9296 9296 9296 9296 49C5 C3F6 C3F6 C3F6 C3F6 4376 C39B C3F6 732A      E78CAA       E78CAA       F3B0B791
9297     9297 9297 9297 9297 54B0 C3F7 C3F7 C3F7 C3F7 4377 C39C C3F7 82E7      E88BA7       E88BA7       F3B0B792
9298     9298 9298 9298 9298 4A59 C3F8 C3F8 C3F8 C3F8 4378 C3A0 C3F8 8457      E89197       E89197       F3B0B793
9299     9299 9299 9299 9299 4C62 C3F9 C3F9 C3F9 C3F9 4379 C3AB C3F9 8CAF      E8B2AF       E8B2AF       F3B0B794
929A     929A 929A 929A 929A 4BCE C3FA C3FA C3FA C3FA 437A C3B0 C3FA 4E01      E4B881       E4B881       F3B0B795
929B     929B 929B 929B 929B 4DF3 C3FB C3FB C3FB C3FB 437B C3C0 C3FB 5146      E58586       E58586       F3B0B796
929C     929C 929C 929C 929C 55F9 C3FC C3FC C3FC C3FC 437C C36A C3FC 51CB      E5878B       E5878B       F3B0B797
929D     929D 929D 929D 929D 5558 C3FD C3FD C3FD C3FD 437D C3D0 C3FD 558B      E5968B       E5968B       F3B0B798
929E     929E 929E 929E 929E 5556 C3FE C3FE C3FE C3FE 437E C3A1 C3FE 5BF5      E5AFB5       E5AFB5       F3B0B799
929F     929F 929F 929F 929F 51C7 C4A1 C4A1 C4A1 C4A1 4421 C44F C4A1 5E16      E5B896       E5B896       F3B0B79A
92A0     92A0 92A0 92A0 92A0 4DC5 C4A2 C4A2 C4A2 C4A2 4422 C47F C4A2 5E33      E5B8B3       E5B8B3       F3B0B79B
92A1     92A1 92A1 92A1 92A1 4B92 C4A3 C4A3 C4A3 C4A3 4423 C47B C4A3 5E81      E5BA81       E5BA81       F3B0B79C
92A2     92A2 92A2 92A2 92A2 51E5 C4A4 C4A4 C4A4 C4A4 4424 C4E0 C4A4 5F14      E5BC94       E5BC94       F3B0B79D
92A3     92A3 92A3 92A3 92A3 48BA C4A5 C4A5 C4A5 C4A5 4425 C46C C4A5 5F35      E5BCB5       E5BCB5       F3B0B79E
92A4     92A4 92A4 92A4 92A4 4E86 C4A6 C4A6 C4A6 C4A6 4426 C450 C4A6 5F6B      E5BDAB       E5BDAB       F3B0B79F
92A5     92A5 92A5 92A5 92A5 4C74 C4A7 C4A7 C4A7 C4A7 4427 C47D C4A7 5FB4      E5BEB4       E5BEB4       F3B0B7A0
92A6     92A6 92A6 92A6 92A6 5153 C4A8 C4A8 C4A8 C4A8 4428 C44D C4A8 61F2      E687B2       E687B2       F3B0B7A1
92A7     92A7 92A7 92A7 92A7 4FDB C4A9 C4A9 C4A9 C4A9 4429 C45D C4A9 6311      E68C91       E68C91       F3B0B7A2
92A8     92A8 92A8 92A8 92A8 4CC7 C4AA C4AA C4AA C4AA 442A C45C C4AA 66A2      E69AA2       E69AA2       F3B0B7A3
92A9     92A9 92A9 92A9 92A9 475C C4AB C4AB C4AB C4AB 442B C44E C4AB 671D      E69C9D       E69C9D       F3B0B7A4
92AA     92AA 92AA 92AA 92AA 4C93 C4AC C4AC C4AC C4AC 442C C46B C4AC 6F6E      E6BDAE       E6BDAE       F3B0B7A5
92AB     92AB 92AB 92AB 92AB 55DD C4AD C4AD C4AD C4AD 442D C460 C4AD 7252      E78992       E78992       F3B0B7A6
92AC     92AC 92AC 92AC 92AC 4555 C4AE C4AE C4AE C4AE 442E C44B C4AE 753A      E794BA       E794BA       F3B0B7A7
92AD     92AD 92AD 92AD 92AD 4E5F C4AF C4AF C4AF C4AF 442F C461 C4AF 773A      E79CBA       E79CBA       F3B0B7A8
92AE     92AE 92AE 92AE 92AE 4DA8 C4B0 C4B0 C4B0 C4B0 4430 C4F0 C4B0 8074      E881B4       E881B4       F3B0B7A9
92AF     92AF 92AF 92AF 92AF 55E9 C4B1 C4B1 C4B1 C4B1 4431 C4F1 C4B1 8139      E884B9       E884B9       F3B0B7AA
92B0     92B0 92B0 92B0 92B0 4DB6 C4B2 C4B2 C4B2 C4B2 4432 C4F2 C4B2 8178      E885B8       E885B8       F3B0B7AB
92B1     92B1 92B1 92B1 92B1 4E6E C4B3 C4B3 C4B3 C4B3 4433 C4F3 C4B3 8776      E89DB6       E89DB6       F3B0B7AC
92B2     92B2 92B2 92B2 92B2 4689 C4B4 C4B4 C4B4 C4B4 4434 C4F4 C4B4 8ABF      E8AABF       E8AABF       F3B0B7AD
92B3     92B3 92B3 92B3 92B3 5570 C4B5 C4B5 C4B5 C4B5 4435 C4F5 C4B5 8ADC      E8AB9C       E8AB9C       F3B0B7AE
92B4     92B4 92B4 92B4 92B4 4A78 C4B6 C4B6 C4B6 C4B6 4436 C4F6 C4B6 8D85      E8B685       E8B685       F3B0B7AF
92B5     92B5 92B5 92B5 92B5 50B4 C4B7 C4B7 C4B7 C4B7 4437 C4F7 C4B7 8DF3      E8B7B3       E8B7B3       F3B0B7B0
92B6     92B6 92B6 92B6 92B6 4E80 C4B8 C4B8 C4B8 C4B8 4438 C4F8 C4B8 929A      E98A9A       E98A9A       F3B0B7B1
92B7     92B7 92B7 92B7 92B7 4583 C4B9 C4B9 C4B9 C4B9 4439 C4F9 C4B9 9577      E995B7       E995B7       F3B0B7B2
92B8     92B8 92B8 92B8 92B8 4D5B C4BA C4BA C4BA C4BA 443A C47A C4BA 9802      E9A082       E9A082       F3B0B7B3
92B9     92B9 92B9 92B9 92B9 4779 C4BB C4BB C4BB C4BB 443B C45E C4BB 9CE5      E9B3A5       E9B3A5       F3B0B7B4
92BA     92BA 92BA 92BA 92BA 4EEF C4BC C4BC C4BC C4BC 443C C44C C4BC 52C5      E58B85       E58B85       F3B0B7B5
92BB     92BB 92BB 92BB 92BB 5546 C4BD C4BD C4BD C4BD 443D C47E C4BD 6357      E68D97       E68D97       F3B0B7B6
92BC     92BC 92BC 92BC 92BC 4674 C4BE C4BE C4BE C4BE 443E C46E C4BE 76F4      E79BB4       E79BB4       F3B0B7B7
92BD     92BD 92BD 92BD 92BD 55F3 C4BF C4BF C4BF C4BF 443F C46F C4BF 6715      E69C95       E69C95       F3B0B7B8
92BE     92BE 92BE 92BE 92BE 4C8F C4C0 C4C0 C4C0 C4C0 4440 C47C C4C0 6C88      E6B288       E6B288       F3B0B7B9
92BF     92BF 92BF 92BF 92BF 4D5D C4C1 C4C1 C4C1 C4C1 4441 C4C1 C4C1 73CD      E78F8D       E78F8D       F3B0B7BA
92C0     92C0 92C0 92C0 92C0 4B4D C4C2 C4C2 C4C2 C4C2 4442 C4C2 C4C2 8CC3      E8B383       E8B383       F3B0B7BB
92C1     92C1 92C1 92C1 92C1 4C89 C4C3 C4C3 C4C3 C4C3 4443 C4C3 C4C3 93AE      E98EAE       E98EAE       F3B0B7BC
92C2     92C2 92C2 92C2 92C2 4E61 C4C4 C4C4 C4C4 C4C4 4444 C4C4 C4C4 9673      E999B3       E999B3       F3B0B7BD
92C3     92C3 92C3 92C3 92C3 45DC C4C5 C4C5 C4C5 C4C5 4445 C4C5 C4C5 6D25      E6B4A5       E6B4A5       F3B0B7BE
92C4     92C4 92C4 92C4 92C4 5150 C4C6 C4C6 C4C6 C4C6 4446 C4C6 C4C6 589C      E5A29C       E5A29C       F3B0B7BF
92C5     92C5 92C5 92C5 92C5 4BA0 C4C7 C4C7 C4C7 C4C7 4447 C4C7 C4C7 690E      E6A48E       E6A48E       F3B0B880
92C6     92C6 92C6 92C6 92C6 4EFA C4C8 C4C8 C4C8 C4C8 4448 C4C8 C4C8 69CC      E6A78C       E6A78C       F3B0B881
92C7     92C7 92C7 92C7 92C7 49D1 C4C9 C4C9 C4C9 C4C9 4449 C4C9 C4C9 8FFD      E8BFBD       E8BFBD       F3B0B882
92C8     92C8 92C8 92C8 92C8 54FE C4CA C4CA C4CA C4CA 444A C4D1 C4CA 939A      E98E9A       E98E9A       F3B0B883
92C9     92C9 92C9 92C9 92C9 4C47 C4CB C4CB C4CB C4CB 444B C4D2 C4CB 75DB      E7979B       E7979B       F3B0B884
92CA     92CA 92CA 92CA 92CA 45E3 C4CC C4CC C4CC C4CC 444C C4D3 C4CC 901A      E9809A       E9809A       F3B0B885
#
# U+585A
# https://glyphwiki.org/wiki/u585a
#
92CB     92CB 92CB 92CB 92CB 4657 5CC4 C4CD C4CD 47C9 B6F5 86EB C4CD 585A      E5A19A       E5A19A       F3B0B886
92CC     92CC 92CC 92CC 92CC 52D1 C4CE C4CE C4CE C4CE 444E C4D5 C4CE 6802      E6A082       E6A082       F3B0B887
92CD     92CD 92CD 92CD 92CD 54CD C4CF C4CF C4CF 50BB D6E2 ACB4 C4CF 63B4      E68EB4       E68EB4       F3B0B888
92CE     92CE 92CE 92CE 92CE 4B5C C4D0 C4D0 C4D0 C4D0 4450 C4D7 C4D0 69FB      E6A7BB       E6A7BB       F3B0B889
92CF     92CF 92CF 92CF 92CF 4EDF C4D1 C4D1 C4D1 C4D1 4451 C4D8 C4D1 4F43      E4BD83       E4BD83       F3B0B88A
92D0     92D0 92D0 92D0 92D0 4E8A C4D2 C4D2 C4D2 C4D2 4452 C4D9 C4D2 6F2C      E6BCAC       E6BCAC       F3B0B88B
92D1     92D1 92D1 92D1 92D1 4FD8 C4D3 C4D3 C4D3 C4D3 4453 C4E2 C4D3 67D8      E69F98       E69F98       F3B0B88C
92D2     92D2 92D2 92D2 92D2 47F5 C4D4 C4D4 C4D4 C4D4 4454 C4E3 C4D4 8FBB      E8BEBB       E8BEBB       F3B0B88D
92D3     92D3 92D3 92D3 92D3 4FA1 C4D5 C4D5 C4D5 C4D5 4455 C4E4 C4D5 8526      E894A6       E894A6       F3B0B88E
92D4     92D4 92D4 92D4 92D4 4DC9 C4D6 C4D6 C4D6 C4D6 4456 C4E5 C4D6 7DB4      E7B6B4       E7B6B4       F3B0B88F
92D5     92D5 92D5 92D5 92D5 54AA C4D7 C4D7 C4D7 C4D7 4457 C4E6 C4D7 9354      E98D94       E98D94       F3B0B890
92D6     92D6 92D6 92D6 92D6 4CBA C4D8 C4D8 C4D8 C4D8 4458 C4E7 C4D8 693F      E6A4BF       E6A4BF       F3B0B891
92D7     92D7 92D7 92D7 92D7 5446 C4D9 C4D9 C4D9 C4D9 4459 C4E8 C4D9 6F70      E6BDB0       E6BDB0       F3B0B892
92D8     92D8 92D8 92D8 92D8 4949 C4DA C4DA C4DA C4DA 445A C4E9 C4DA 576A      E59DAA       E59DAA       F3B0B893
92D9     92D9 92D9 92D9 92D9 589B D4E4 C4DB C4DB D4E4 5464 E363 C4DB 58F7      E5A3B7       E5A3B7       F3B0B894
92DA     92DA 92DA 92DA 92DA 52EA C4DC C4DC C4DC C4DC 445C C45B C4DC 5B2C      E5ACAC       E5ACAC       F3B0B895
92DB     92DB 92DB 92DB 92DB 5592 C4DD C4DD C4DD C4DD 445D C45A C4DD 7D2C      E7B4AC       E7B4AC       F3B0B896
92DC     92DC 92DC 92DC 92DC 4C87 C4DE C4DE C4DE C4DE 445E C45F C4DE 722A      E788AA       E788AA       F3B0B897
92DD     92DD 92DD 92DD 92DD 53D3 C4DF C4DF C4DF C4DF 445F C46D C4DF 540A      E5908A       E5908A       F3B0B898
92DE     92DE 92DE 92DE 92DE 4CE3 C4E0 C4E0 C4E0 C4E0 4460 C479 C4E0 91E3      E987A3       E987A3       F3B0B899
92DF     92DF 92DF 92DF 92DF 477A C4E1 C4E1 C4E1 C4E1 4461 C457 C4E1 9DB4      E9B6B4       E9B6B4       F3B0B89A
92E0     92E0 92E0 92E0 92E0 4E52 C4E2 C4E2 C4E2 C4E2 4462 C459 C4E2 4EAD      E4BAAD       E4BAAD       F3B0B89B
92E1     92E1 92E1 92E1 92E1 48A4 C4E3 C4E3 C4E3 C4E3 4463 C462 C4E3 4F4E      E4BD8E       E4BD8E       F3B0B89C
92E2     92E2 92E2 92E2 92E2 4E70 C4E4 C4E4 C4E4 C4E4 4464 C463 C4E4 505C      E5819C       E5819C       F3B0B89D
92E3     92E3 92E3 92E3 92E3 4F95 C4E5 C4E5 C4E5 C4E5 4465 C464 C4E5 5075      E581B5       E581B5       F3B0B89E
92E4     92E4 92E4 92E4 92E4 5379 C4E6 C4E6 C4E6 C4E6 4466 C465 C4E6 5243      E58983       E58983       F3B0B89F
92E5     92E5 92E5 92E5 92E5 46B6 C4E7 C4E7 C4E7 C4E7 4467 C466 C4E7 8C9E      E8B29E       E8B29E       F3B0B8A0
92E6     92E6 92E6 92E6 92E6 4F56 C4E8 C4E8 C4E8 C4E8 4468 C467 C4E8 5448      E59188       E59188       F3B0B8A1
92E7     92E7 92E7 92E7 92E7 4B61 C4E9 C4E9 C4E9 C4E9 4469 C468 C4E9 5824      E5A0A4       E5A0A4       F3B0B8A2
92E8     92E8 92E8 92E8 92E8 45E7 C4EA C4EA C4EA C4EA 446A C469 C4EA 5B9A      E5AE9A       E5AE9A       F3B0B8A3
92E9     92E9 92E9 92E9 92E9 489C C4EB C4EB C4EB C4EB 446B C470 C4EB 5E1D      E5B89D       E5B89D       F3B0B8A4
92EA     92EA 92EA 92EA 92EA 4B64 C4EC C4EC C4EC C4EC 446C C471 C4EC 5E95      E5BA95       E5BA95       F3B0B8A5
92EB     92EB 92EB 92EB 92EB 48E0 C4ED C4ED C4ED C4ED 446D C472 C4ED 5EAD      E5BAAD       E5BAAD       F3B0B8A6
92EC     92EC 92EC 92EC 92EC 505B C4EE C4EE C4EE C4EE 446E C473 C4EE 5EF7      E5BBB7       E5BBB7       F3B0B8A7
92ED     92ED 92ED 92ED 92ED 4C9B C4EF C4EF C4EF C4EF 446F C474 C4EF 5F1F      E5BC9F       E5BC9F       F3B0B8A8
92EE     92EE 92EE 92EE 92EE 4DE3 C4F0 C4F0 C4F0 C4F0 4470 C475 C4F0 608C      E6828C       E6828C       F3B0B8A9
92EF     92EF 92EF 92EF 92EF 4B90 C4F1 C4F1 C4F1 C4F1 4471 C476 C4F1 62B5      E68AB5       E68AB5       F3B0B8AA
92F0     92F0 92F0 92F0 92F0 5387 C4F2 C4F2 C4F2 C4F2 4472 C477 C4F2 633A      E68CBA       E68CBA       F3B0B8AB
92F1     92F1 92F1 92F1 92F1 49B9 C4F3 C4F3 C4F3 C4F3 4473 C478 C4F3 63D0      E68F90       E68F90       F3B0B8AC
92F2     92F2 92F2 92F2 92F2 51E4 C4F4 C4F4 C4F4 C4F4 4474 C480 C4F4 68AF      E6A2AF       E6A2AF       F3B0B8AD
92F3     92F3 92F3 92F3 92F3 534D C4F5 C4F5 C4F5 C4F5 4475 C48B C4F5 6C40      E6B180       E6B180       F3B0B8AE
92F4     92F4 92F4 92F4 92F4 5195 C4F6 C4F6 C4F6 C4F6 4476 C49B C4F6 7887      E7A287       E7A287       F3B0B8AF
92F5     92F5 92F5 92F5 92F5 4BA7 C4F7 C4F7 C4F7 C4F7 4477 C49C C4F7 798E      E7A68E       E7A68E       F3B0B8B0
92F6     92F6 92F6 92F6 92F6 47F4 C4F8 C4F8 C4F8 C4F8 4478 C4A0 C4F8 7A0B      E7A88B       E7A88B       F3B0B8B1
92F7     92F7 92F7 92F7 92F7 4A80 C4F9 C4F9 C4F9 C4F9 4479 C4AB C4F9 7DE0      E7B7A0       E7B7A0       F3B0B8B2
92F8     92F8 92F8 92F8 92F8 4DFC C4FA C4FA C4FA C4FA 447A C4B0 C4FA 8247      E88987       E88987       F3B0B8B3
92F9     92F9 92F9 92F9 92F9 4FC4 C4FB C4FB C4FB C4FB 447B C4C0 C4FB 8A02      E8A882       E8A882       F3B0B8B4
92FA     92FA 92FA 92FA 92FA 5246 C4FC C4FC C4FC C4FC 447C C46A C4FC 8AE6      E8ABA6       E8ABA6       F3B0B8B5
92FB     92FB 92FB 92FB 92FB 55A7 C4FD C4FD C4FD C4FD 447D C4D0 C4FD 8E44      E8B984       E8B984       F3B0B8B6
92FC     92FC 92FC 92FC 92FC 515E C4FE C4FE C4FE C4FE 447E C4A1 C4FE 9013      E98093       E98093       F3B0B8B7
9340     9340 9340 9340 9340 4D81 C5A1 C5A1 C5A1 C5A1 4521 C54F C5A1 90B8      E982B8       E982B8       F3B0B8B8
9341     9341 9341 9341 9341 5154 C5A2 C5A2 C5A2 C5A2 4522 C57F C5A2 912D      E984AD       E984AD       F3B0B8B9
9342     9342 9342 9342 9342 4FF4 C5A3 C5A3 C5A3 C5A3 4523 C57B C5A3 91D8      E98798       E98798       F3B0B8BA
9343     9343 9343 9343 9343 51AA C5A4 C5A4 C5A4 C5A4 4524 C5E0 C5A4 9F0E      E9BC8E       E9BC8E       F3B0B8BB
9344     9344 9344 9344 9344 4E56 C5A5 C5A5 C5A5 C5A5 4525 C56C C5A5 6CE5      E6B3A5       E6B3A5       F3B0B8BC
9345     9345 9345 9345 9345 4DCB C5A6 C5A6 C5A6 C5A6 4526 C550 C5A6 6458      E69198       E69198       F3B0B8BD
9346     9346 9346 9346 9346 5543 C5A7 C5A7 C5A7 C5A7 4527 C57D C5A7 64E2      E693A2       E693A2       F3B0B8BE
9347     9347 9347 9347 9347 4BF3 C5A8 C5A8 C5A8 C5A8 4528 C54D C5A8 6575      E695B5       E695B5       F3B0B8BF
9348     9348 9348 9348 9348 4FB7 C5A9 C5A9 C5A9 C5A9 4529 C55D C5A9 6EF4      E6BBB4       E6BBB4       F3B0B980
9349     9349 9349 9349 9349 45EF C5AA C5AA C5AA C5AA 452A C55C C5AA 7684      E79A84       E79A84       F3B0B981
934A     934A 934A 934A 934A 4F41 C5AB C5AB C5AB C5AB 452B C54E C5AB 7B1B      E7AC9B       E7AC9B       F3B0B982
934B     934B 934B 934B 934B 49CC C5AC C5AC C5AC C5AC 452C C56B C5AC 9069      E981A9       E981A9       F3B0B983
934C     934C 934C 934C 934C 5041 C5AD C5AD C5AD C5AD 452D C560 C5AD 93D1      E98F91       E98F91       F3B0B984
934D     934D 934D 934D 934D 5497 C5AE C5AE C5AE C5AE 452E C54B C5AE 6EBA      E6BABA       E6BABA       F3B0B985
934E     934E 934E 934E 934E 4747 C5AF C5AF C5AF C5AF 452F C561 C5AF 54F2      E593B2       E593B2       F3B0B986
934F     934F 934F 934F 934F 48F8 C5B0 C5B0 C5B0 C5B0 4530 C5F0 C5B0 5FB9      E5BEB9       E5BEB9       F3B0B987
9350     9350 9350 9350 9350 4E7A C5B1 C5B1 C5B1 C5B1 4531 C5F1 C5B1 64A4      E692A4       E692A4       F3B0B988
9351     9351 9351 9351 9351 54EE C5B2 C5B2 C5B2 C5B2 4532 C5F2 C5B2 8F4D      E8BD8D       E8BD8D       F3B0B989
9352     9352 9352 9352 9352 53DF C5B3 C5B3 C5B3 C5B3 4533 C5F3 C5B3 8FED      E8BFAD       E8BFAD       F3B0B98A
9353     9353 9353 9353 9353 45D7 C5B4 C5B4 C5B4 C5B4 4534 C5F4 C5B4 9244      E98984       E98984       F3B0B98B
9354     9354 9354 9354 9354 47C9 C5B5 C5B5 C5B5 C5B5 4535 C5F5 C5B5 5178      E585B8       E585B8       F3B0B98C
9355     9355 9355 9355 9355 4F5E C5B6 C5B6 C5B6 47CA D6E6 ACB8 C5B6 586B      E5A1AB       E5A1AB       F3B0B98D
9356     9356 9356 9356 9356 468A C5B7 C5B7 C5B7 C5B7 4537 C5F7 C5B7 5929      E5A4A9       E5A4A9       F3B0B98E
9357     9357 9357 9357 9357 48DD C5B8 C5B8 C5B8 C5B8 4538 C5F8 C5B8 5C55      E5B195       E5B195       F3B0B98F
9358     9358 9358 9358 9358 4854 C5B9 C5B9 C5B9 C5B9 4539 C5F9 C5B9 5E97      E5BA97       E5BA97       F3B0B990
9359     9359 9359 9359 9359 4A91 C5BA C5BA C5BA C5BA 453A C57A C5BA 6DFB      E6B7BB       E6B7BB       F3B0B991
935A     935A 935A 935A 935A 55FA C5BB C5BB C5BB C5BB 453B C55E C5BB 7E8F      E7BA8F       E7BA8F       F3B0B992
935B     935B 935B 935B 935B 52A7 C5BC C5BC C5BC C5BC 453C C54C C5BC 751C      E7949C       E7949C       F3B0B993
935C     935C 935C 935C 935C 507B C5BD C5BD C5BD C5BD 453D C57E C5BD 8CBC      E8B2BC       E8B2BC       F3B0B994
935D     935D 935D 935D 935D 4752 C5BE C5BE C5BE C5BE 453E C56E C5BE 8EE2      E8BBA2       E8BBA2       F3B0B995
935E     935E 935E 935E 935E 55C1 C5BF C5BF C5BF 6CFC D4A3 A943 C5BF 985B      E9A19B       E9A19B       F3B0B996
935F     935F 935F 935F 935F 4788 C5C0 C5C0 C5C0 C5C0 4540 C57C C5C0 70B9      E782B9       E782B9       F3B0B997
9360     9360 9360 9360 9360 48D9 C5C1 C5C1 C5C1 C5C1 4541 C5C1 C5C1 4F1D      E4BC9D       E4BC9D       F3B0B998
9361     9361 9361 9361 9361 4849 C5C2 C5C2 C5C2 C5C2 4542 C5C2 C5C2 6BBF      E6AEBF       E6AEBF       F3B0B999
9362     9362 9362 9362 9362 5589 C5C3 C5C3 C5C3 C5C3 4543 C5C3 C5C3 6FB1      E6BEB1       E6BEB1       F3B0B99A
9363     9363 9363 9363 9363 4563 C5C4 C5C4 C5C4 C5C4 4544 C5C4 C5C4 7530      E794B0       E794B0       F3B0B99B
9364     9364 9364 9364 9364 4582 C5C5 C5C5 C5C5 C5C5 4545 C5C5 C5C5 96FB      E99BBB       E99BBB       F3B0B99C
9365     9365 9365 9365 9365 5050 C5C6 C5C6 C5C6 C5C6 4546 C5C6 C5C6 514E      E5858E       E5858E       F3B0B99D
9366     9366 9366 9366 9366 4EB2 C5C7 C5C7 C5C7 C5C7 4547 C5C7 C5C7 5410      E59090       E59090       F3B0B99E
9367     9367 9367 9367 9367 54F6 C5C8 C5C8 C5C8 C5C8 4548 C5C8 C5C8 5835      E5A0B5       E5A0B5       F3B0B99F
9368     9368 9368 9368 9368 486A C5C9 C5C9 C5C9 C5C9 4549 C5C9 C5C9 5857      E5A197       E5A197       F3B0B9A0
9369     9369 9369 9369 9369 537F C5CA C5CA C5CA C5CA 454A C5D1 C5CA 59AC      E5A6AC       E5A6AC       F3B0B9A1
936A     936A 936A 936A 936A 53FB C5CB C5CB C5CB C5CB 454B C5D2 C5CB 5C60      E5B1A0       E5B1A0       F3B0B9A2
936B     936B 936B 936B 936B 4C6C C5CC C5CC C5CC C5CC 454C C5D3 C5CC 5F92      E5BE92       E5BE92       F3B0B9A3
936C     936C 936C 936C 936C 4E83 C5CD C5CD C5CD C5CD 454D C5D4 C5CD 6597      E69697       E69697       F3B0B9A4
936D     936D 936D 936D 936D 525D C5CE C5CE C5CE C5CE 454E C5D5 C5CE 675C      E69D9C       E69D9C       F3B0B9A5
936E     936E 936E 936E 936E 45D2 C5CF C5CF C5CF C5CF 454F C5D6 C5CF 6E21      E6B8A1       E6B8A1       F3B0B9A6
936F     936F 936F 936F 936F 46E8 C5D0 C5D0 C5D0 C5D0 4550 C5D7 C5D0 767B      E799BB       E799BB       F3B0B9A7
9370     9370 9370 9370 9370 53D4 C5D1 C5D1 C5D1 C5D1 4551 C5D8 C5D1 83DF      E88F9F       E88F9F       F3B0B9A8
9371     9371 9371 9371 9371 52C1 C5D2 C5D2 C5D2 C5D2 4552 C5D9 C5D2 8CED      E8B3AD       E8B3AD       F3B0B9A9
9372     9372 9372 9372 9372 48F0 C5D3 C5D3 C5D3 C5D3 4553 C5E2 C5D3 9014      E98094       E98094       F3B0B9AA
9373     9373 9373 9373 9373 454F C5D4 C5D4 C5D4 C5D4 4554 C5E3 C5D4 90FD      E983BD       E983BD       F3B0B9AB
9374     9374 9374 9374 9374 55D3 C5D5 C5D5 C5D5 C5D5 4555 C5E4 C5D5 934D      E98D8D       E98D8D       F3B0B9AC
9375     9375 9375 9375 9375 4D45 C5D6 C5D6 C5D6 C5D6 4556 C5E5 C5D6 7825      E7A0A5       E7A0A5       F3B0B9AD
9376     9376 9376 9376 9376 50DF E2EA C5D7 C5D7 E2EA 626A 5969 C5D7 783A      E7A0BA       E7A0BA       F3B0B9AE
9377     9377 9377 9377 9377 4A64 C5D8 C5D8 C5D8 C5D8 4558 C5E7 C5D8 52AA      E58AAA       E58AAA       F3B0B9AF
9378     9378 9378 9378 9378 45DE C5D9 C5D9 C5D9 C5D9 4559 C5E8 C5D9 5EA6      E5BAA6       E5BAA6       F3B0B9B0
9379     9379 9379 9379 9379 4684 C5DA C5DA C5DA C5DA 455A C5E9 C5DA 571F      E59C9F       E59C9F       F3B0B9B1
937A     937A 937A 937A 937A 4D90 C5DB C5DB C5DB C5DB 455B C54A C5DB 5974      E5A5B4       E5A5B4       F3B0B9B2
937B     937B 937B 937B 937B 4CD6 C5DC C5DC C5DC C5DC 455C C55B C5DC 6012      E68092       E68092       F3B0B9B3
937C     937C 937C 937C 937C 4B6A C5DD C5DD C5DD C5DD 455D C55A C5DD 5012      E58092       E58092       F3B0B9B4
937D     937D 937D 937D 937D 494B C5DE C5DE C5DE C5DE 455E C55F C5DE 515A      E5859A       E5859A       F3B0B9B5
937E     937E 937E 937E 937E 4BDE C5DF C5DF C5DF C5DF 455F C56D C5DF 51AC      E586AC       E586AC       F3B0B9B6
9380     9380 9380 9380 9380 4BE6 C5E0 C5E0 C5E0 C5E0 4560 C579 C5E0 51CD      E5878D       E5878D       F3B0B9B7
9381     9381 9381 9381 9381 4C50 C5E1 C5E1 C5E1 C5E1 4561 C557 C5E1 5200      E58880       E58880       F3B0B9B8
9382     9382 9382 9382 9382 4BAB C5E2 C5E2 C5E2 C5E2 4562 C559 C5E2 5510      E59490       E59490       F3B0B9B9
9383     9383 9383 9383 9383 4CCB C5E3 C5E3 C5E3 C5E3 4563 C562 C5E3 5854      E5A194       E5A194       F3B0B9BA
9384     9384 9384 9384 9384 52F8 C5E4 C5E4 C5E4 C5E4 4564 C563 C5E4 5858      E5A198       E5A198       F3B0B9BB
9385     9385 9385 9385 9385 5583 C5E5 C5E5 C5E5 C5E5 4565 C564 C5E5 5957      E5A597       E5A597       F3B0B9BC
9386     9386 9386 9386 9386 515C C5E6 C5E6 C5E6 C5E6 4566 C565 C5E6 5B95      E5AE95       E5AE95       F3B0B9BD
9387     9387 9387 9387 9387 456F C5E7 C5E7 C5E7 C5E7 4567 C566 C5E7 5CF6      E5B3B6       E5B3B6       F3B0B9BE
9388     9388 9388 9388 9388 47D1 C5E8 C5E8 C5E8 C5E8 4568 C567 C5E8 5D8B      E5B68B       E5B68B       F3B0B9BF
9389     9389 9389 9389 9389 5145 C5E9 C5E9 C5E9 C5E9 4569 C568 C5E9 60BC      E682BC       E682BC       F3B0BA80
938A     938A 938A 938A 938A 4751 C5EA C5EA C5EA C5EA 456A C569 C5EA 6295      E68A95       E68A95       F3B0BA81
938B     938B 938B 938B 938B 528F C5EB C5EB C5EB C5EB 456B C570 C5EB 642D      E690AD       E690AD       F3B0BA82
938C     938C 938C 938C 938C 4557 C5EC C5EC C5EC C5EC 456C C571 C5EC 6771      E69DB1       E69DB1       F3B0BA83
938D     938D 938D 938D 938D 4ADA C5ED C5ED C5ED C5ED 456D C572 C5ED 6843      E6A183       E6A183       F3B0BA84
938E     938E 938E 938E 938E 5C49 DBED C5EE C5EE DBED 5B6D 4A72 C5EE 68BC      E6A2BC       E6A2BC       F3B0BA85
938F     938F 938F 938F 938F 4EA9 C5EF C5EF C5EF C5EF 456F C574 C5EF 68DF      E6A39F       E6A39F       F3B0BA86
9390     9390 9390 9390 9390 4D65 C5F0 C5F0 C5F0 C5F0 4570 C575 C5F0 76D7      E79B97       E79B97       F3B0BA87
9391     9391 9391 9391 9391 55E1 C5F1 C5F1 C5F1 C5F1 4571 C576 C5F1 6DD8      E6B798       E6B798       F3B0BA88
9392     9392 9392 9392 9392 48BD C5F2 C5F2 C5F2 C5F2 4572 C577 C5F2 6E6F      E6B9AF       E6B9AF       F3B0BA89
9393     9393 9393 9393 9393 5D69 DEB9 C5F3 C5F3 DEB9 5E39 5FF9 C5F3 6D9B      E6B69B       E6B69B       F3B0BA8A
9394     9394 9394 9394 9394 4B65 C5F4 C5F4 C5F4 C5F4 4574 C580 C5F4 706F      E781AF       E781AF       F3B0BA8B
9395     9395 9395 9395 9395 4EF8 C5F5 C5F5 C5F5 C5F5 4575 C58B C5F5 71C8      E78788       E78788       F3B0BA8C
9396     9396 9396 9396 9396 465F C5F6 C5F6 C5F6 C5F6 4576 C59B C5F6 5F53      E5BD93       E5BD93       F3B0BA8D
9397     9397 9397 9397 9397 5549 C5F7 C5F7 C5F7 C5F7 4577 C59C C5F7 75D8      E79798       E79798       F3B0BA8E
9398     9398 9398 9398 9398 5553 C5F8 C5F8 C5F8 5CEC C6FB 97FB C5F8 7977      E7A5B7       E7A5B7       F3B0BA8F
9399     9399 9399 9399 9399 46A7 C5F9 C5F9 C5F9 C5F9 4579 C5AB C5F9 7B49      E7AD89       E7AD89       F3B0BA90
939A     939A 939A 939A 939A 49D0 C5FA C5FA C5FA C5FA 457A C5B0 C5FA 7B54      E7AD94       E7AD94       F3B0BA91
939B     939B 939B 939B 939B 4B7C C5FB C5FB C5FB C5FB 457B C5C0 C5FB 7B52      E7AD92       E7AD92       F3B0BA92
939C     939C 939C 939C 939C 4884 C5FC C5FC C5FC C5FC 457C C56A C5FC 7CD6      E7B396       E7B396       F3B0BA93
939D     939D 939D 939D 939D 4881 C5FD C5FD C5FD C5FD 457D C5D0 C5FD 7D71      E7B5B1       E7B5B1       F3B0BA94
939E     939E 939E 939E 939E 4CDD C5FE C5FE C5FE C5FE 457E C5A1 C5FE 5230      E588B0       E588B0       F3B0BA95
939F     939F 939F 939F 939F 5071 C6A1 C6A1 C6A1 C6A1 4621 C64F C6A1 8463      E891A3       E891A3       F3B0BA96
93A0     93A0 93A0 93A0 93A0 54F0 C6A2 C6A2 C6A2 C6A2 4622 C67F C6A2 8569      E895A9       E895A9       F3B0BA97
93A1     93A1 93A1 93A1 93A1 4568 C6A3 C6A3 C6A3 C6A3 4623 C67B C6A3 85E4      E897A4       E897A4       F3B0BA98
93A2     93A2 93A2 93A2 93A2 49D2 C6A4 C6A4 C6A4 C6A4 4624 C6E0 C6A4 8A0E      E8A88E       E8A88E       F3B0BA99
93A3     93A3 93A3 93A3 93A3 4FE2 C6A5 C6A5 C6A5 C6A5 4625 C66C C6A5 8B04      E8AC84       E8AC84       F3B0BA9A
93A4     93A4 93A4 93A4 93A4 4A99 C6A6 C6A6 C6A6 C6A6 4626 C650 C6A6 8C46      E8B186       E8B186       F3B0BA9B
93A5     93A5 93A5 93A5 93A5 4D7C C6A7 C6A7 C6A7 C6A7 4627 C67D C6A7 8E0F      E8B88F       E8B88F       F3B0BA9C
93A6     93A6 93A6 93A6 93A6 4C72 C6A8 C6A8 C6A8 C6A8 4628 C64D C6A8 9003      E98083       E98083       F3B0BA9D
93A7     93A7 93A7 93A7 93A7 4BDA C6A9 C6A9 C6A9 C6A9 4629 C65D C6A9 900F      E9808F       E9808F       F3B0BA9E
93A8     93A8 93A8 93A8 93A8 52CF C6AA C6AA C6AA C6AA 462A C65C C6AA 9419      E99099       E99099       F3B0BA9F
93A9     93A9 93A9 93A9 93A9 4AB7 C6AB C6AB C6AB C6AB 462B C64E C6AB 9676      E999B6       E999B6       F3B0BAA0
93AA     93AA 93AA 93AA 93AA 47EB C6AC C6AC C6AC C6AC 462C C66B C6AC 982D      E9A0AD       E9A0AD       F3B0BAA1
93AB     93AB 93AB 93AB 93AB 4E5E C6AD C6AD C6AD C6AD 462D C660 C6AD 9A30      E9A8B0       E9A8B0       F3B0BAA2
93AC     93AC 93AC 93AC 93AC 4BCC C6AE C6AE C6AE C6AE 462E C64B C6AE 95D8      E99798       E99798       F3B0BAA3
93AD     93AD 93AD 93AD 93AD 48EC C6AF C6AF C6AF C6AF 462F C661 C6AF 50CD      E5838D       E5838D       F3B0BAA4
93AE     93AE 93AE 93AE 93AE 45E4 C6B0 C6B0 C6B0 C6B0 4630 C6F0 C6B0 52D5      E58B95       E58B95       F3B0BAA5
93AF     93AF 93AF 93AF 93AF 466D C6B1 C6B1 C6B1 C6B1 4631 C6F1 C6B1 540C      E5908C       E5908C       F3B0BAA6
93B0     93B0 93B0 93B0 93B0 497F C6B2 C6B2 C6B2 C6B2 4632 C6F2 C6B2 5802      E5A082       E5A082       F3B0BAA7
93B1     93B1 93B1 93B1 93B1 48AE C6B3 C6B3 C6B3 C6B3 4633 C6F3 C6B3 5C0E      E5B08E       E5B08E       F3B0BAA8
93B2     93B2 93B2 93B2 93B2 53C2 C6B4 C6B4 C6B4 C6B4 4634 C6F4 C6B4 61A7      E686A7       E686A7       F3B0BAA9
93B3     93B3 93B3 93B3 93B3 55DB C6B5 C6B5 C6B5 C6B5 4635 C6F5 C6B5 649E      E6929E       E6929E       F3B0BAAA
93B4     93B4 93B4 93B4 93B4 4D98 C6B6 C6B6 C6B6 C6B6 4636 C6F6 C6B6 6D1E      E6B49E       E6B49E       F3B0BAAB
93B5     93B5 93B5 93B5 93B5 4F9B C6B7 C6B7 C6B7 C6B7 4637 C6F7 C6B7 77B3      E79EB3       E79EB3       F3B0BAAC
93B6     93B6 93B6 93B6 93B6 4D50 C6B8 C6B8 C6B8 C6B8 4638 C6F8 C6B8 7AE5      E7ABA5       E7ABA5       F3B0BAAD
93B7     93B7 93B7 93B7 93B7 4EAB C6B9 C6B9 C6B9 C6B9 4639 C6F9 C6B9 80F4      E883B4       E883B4       F3B0BAAE
93B8     93B8 93B8 93B8 93B8 54F7 C6BA C6BA C6BA C6BA 463A C67A C6BA 8404      E89084       E89084       F3B0BAAF
93B9     93B9 93B9 93B9 93B9 4550 C6BB C6BB C6BB C6BB 463B C65E C6BB 9053      E98193       E98193       F3B0BAB0
93BA     93BA 93BA 93BA 93BA 48C0 C6BC C6BC C6BC C6BC 463C C64C C6BC 9285      E98A85       E98A85       F3B0BAB1
93BB     93BB 93BB 93BB 93BB 4F96 C6BD C6BD C6BD C6BD 463D C67E C6BD 5CE0      E5B3A0       E5B3A0       F3B0BAB2
93BC     93BC 93BC 93BC 93BC 51A1 C6BE C6BE C6BE C6BE 463E C66E C6BE 9D07      E9B487       E9B487       F3B0BAB3
93BD     93BD 93BD 93BD 93BD 50F1 C6BF C6BF C6BF C6BF 463F C66F C6BF 533F      E58CBF       E58CBF       F3B0BAB4
93BE     93BE 93BE 93BE 93BE 47E8 C6C0 C6C0 C6C0 C6C0 4640 C67C C6C0 5F97      E5BE97       E5BE97       F3B0BAB5
93BF     93BF 93BF 93BF 93BF 4654 C6C1 C6C1 C6C1 C6C1 4641 C6C1 C6C1 5FB3      E5BEB3       E5BEB3       F3B0BAB6
93C0     93C0 93C0 93C0 93C0 5550 C6C2 C6C2 C6C2 70D7 D6EC ACCE C6C2 6D9C      E6B69C       E6B69C       F3B0BAB7
93C1     93C1 93C1 93C1 93C1 46A2 C6C3 C6C3 C6C3 C6C3 4643 C6C3 C6C3 7279      E789B9       E789B9       F3B0BAB8
93C2     93C2 93C2 93C2 93C2 4A73 C6C4 C6C4 C6C4 C6C4 4644 C6C4 C6C4 7763      E79DA3       E79DA3       F3B0BAB9
93C3     93C3 93C3 93C3 93C3 53D0 C6C5 C6C5 C6C5 C6C5 4645 C6C5 C6C5 79BF      E7A6BF       E7A6BF       F3B0BABA
93C4     93C4 93C4 93C4 93C4 4AD8 C6C6 C6C6 C6C6 C6C6 4646 C6C6 C6C6 7BE4      E7AFA4       E7AFA4       F3B0BABB
93C5     93C5 93C5 93C5 93C5 4C97 C6C7 C6C7 C6C7 C6C7 4647 C6C7 C6C7 6BD2      E6AF92       E6AF92       F3B0BABC
93C6     93C6 93C6 93C6 93C6 4ADB C6C8 C6C8 C6C8 C6C8 4648 C6C8 C6C8 72EC      E78BAC       E78BAC       F3B0BABD
93C7     93C7 93C7 93C7 93C7 4A4F C6C9 C6C9 C6C9 C6C9 4649 C6C9 C6C9 8AAD      E8AAAD       E8AAAD       F3B0BABE
93C8     93C8 93C8 93C8 93C8 47D3 C6CA C6CA C6CA C6CA 464A C6D1 C6CA 6803      E6A083       E6A083       F3B0BABF
93C9     93C9 93C9 93C9 93C9 55D4 C6CB C6CB C6CB C6CB 464B C6D2 C6CB 6A61      E6A9A1       E6A9A1       F3B0BB80
93CA     93CA 93CA 93CA 93CA 50CF C6CC C6CC C6CC C6CC 464C C6D3 C6CC 51F8      E587B8       E587B8       F3B0BB81
93CB     93CB 93CB 93CB 93CB 4AFD C6CD C6CD C6CD C6CD 464D C6D4 C6CD 7A81      E7AA81       E7AA81       F3B0BB82
93CC     93CC 93CC 93CC 93CC 54A7 C6CE C6CE C6CE C6CE 464E C6D5 C6CE 6934      E6A4B4       E6A4B4       F3B0BB83
93CD     93CD 93CD 93CD 93CD 4DA7 C6CF C6CF C6CF C6CF 464F C6D6 C6CF 5C4A      E5B18A       E5B18A       F3B0BB84
93CE     93CE 93CE 93CE 93CE 54C4 C6D0 C6D0 C6D0 C6D0 4650 C6D7 C6D0 9CF6      E9B3B6       E9B3B6       F3B0BB85
93CF     93CF 93CF 93CF 93CF 4D5A C6D1 C6D1 C6D1 C6D1 4651 C6D8 C6D1 82EB      E88BAB       E88BAB       F3B0BB86
93D0     93D0 93D0 93D0 93D0 4A84 C6D2 C6D2 C6D2 C6D2 4652 C6D9 C6D2 5BC5      E5AF85       E5AF85       F3B0BB87
93D1     93D1 93D1 93D1 93D1 4F73 C6D3 C6D3 C6D3 C6D3 4653 C6E2 C6D3 9149      E98589       E98589       F3B0BB88
93D2     93D2 93D2 93D2 93D2 535C C6D4 C6D4 C6D4 C6D4 4654 C6E3 C6D4 701E      E7809E       E7809E       F3B0BB89
93D3     93D3 93D3 93D3 93D3 55F1 C6D5 C6D5 C6D5 C6D5 4655 C6E4 C6D5 5678      E599B8       E599B8       F3B0BB8A
93D4     93D4 93D4 93D4 93D4 4FDD C6D6 C6D6 C6D6 C6D6 4656 C6E5 C6D6 5C6F      E5B1AF       E5B1AF       F3B0BB8B
93D5     93D5 93D5 93D5 93D5 4E65 C6D7 C6D7 C6D7 C6D7 4657 C6E6 C6D7 60C7      E68387       E68387       F3B0BB8C
93D6     93D6 93D6 93D6 93D6 49F2 C6D8 C6D8 C6D8 C6D8 4658 C6E7 C6D8 6566      E695A6       E695A6       F3B0BB8D
93D7     93D7 93D7 93D7 93D7 5598 C6D9 C6D9 C6D9 C6D9 4659 C6E8 C6D9 6C8C      E6B28C       E6B28C       F3B0BB8E
93D8     93D8 93D8 93D8 93D8 4CCF C6DA C6DA C6DA C6DA 465A C6E9 C6DA 8C5A      E8B19A       E8B19A       F3B0BB8F
93D9     93D9 93D9 93D9 93D9 54F8 C6DB C6DB C6DB C6DB 465B C64A C6DB 9041      E98181       E98181       F3B0BB90
93DA     93DA 93DA 93DA 93DA 517A C6DC C6DC C6DC C6DC 465C C65B C6DC 9813      E9A093       E9A093       F3B0BB91
93DB     93DB 93DB 93DB 93DB 50B0 C6DD C6DD C6DD C6DD 465D C65A C6DD 5451      E59191       E59191       F3B0BB92
93DC     93DC 93DC 93DC 93DC 4C56 C6DE C6DE C6DE C6DE 465E C65F C6DE 66C7      E69B87       E69B87       F3B0BB93
93DD     93DD 93DD 93DD 93DD 4E63 C6DF C6DF C6DF C6DF 465F C66D C6DF 920D      E9888D       E9888D       F3B0BB94
93DE     93DE 93DE 93DE 93DE 45E6 C6E0 C6E0 C6E0 C6E0 4660 C679 C6E0 5948      E5A588       E5A588       F3B0BB95
93DF     93DF 93DF 93DF 93DF 4846 C6E1 C6E1 C6E1 C6E1 4661 C657 C6E1 90A3      E982A3       E982A3       F3B0BB96
93E0     93E0 93E0 93E0 93E0 458E C6E2 C6E2 C6E2 C6E2 4662 C659 C6E2 5185      E58685       E58685       F3B0BB97
93E1     93E1 93E1 93E1 93E1 55A9 C6E3 C6E3 C6E3 C6E3 4663 C662 C6E3 4E4D      E4B98D       E4B98D       F3B0BB98
93E2     93E2 93E2 93E2 93E2 549E C6E4 C6E4 C6E4 C6E4 4664 C663 C6E4 51EA      E587AA       E587AA       F3B0BB99
93E3     93E3 93E3 93E3 93E3 51AD C6E5 C6E5 C6E5 C6E5 4665 C664 C6E5 8599      E89699       E89699       F3B0BB9A
93E4     93E4 93E4 93E4 93E4 52F9 C6E6 C6E6 C6E6 C6E6 4666 C665 C6E6 8B0E      E8AC8E       E8AC8E       F3B0BB9B
93E5     93E5 93E5 93E5 93E5 4B96 C6E7 C6E7 C6E7 C6E7 4667 C666 C6E7 7058      E78198       E78198       F3B0BB9C
93E6     93E6 93E6 93E6 93E6 4FC0 C6E8 C6E8 C6E8 C6E8 4668 C667 C6E8 637A      E68DBA       E68DBA       F3B0BB9D
93E7     93E7 93E7 93E7 93E7 4B48 C6E9 C6E9 C6E9 C6E9 4669 C668 C6E9 934B      E98D8B       E98D8B       F3B0BB9E
93E8     93E8 93E8 93E8 93E8 4D93 C6EA C6EA C6EA C6EA 466A C669 C6EA 6962      E6A5A2       E6A5A2       F3B0BB9F
93E9     93E9 93E9 93E9 93E9 50A1 C6EB C6EB C6EB C6EB 466B C670 C6EB 99B4      E9A6B4       E9A6B4       F3B0BBA0
93EA     93EA 93EA 93EA 93EA 4859 C6EC C6EC C6EC C6EC 466C C671 C6EC 7E04      E7B884       E7B884       F3B0BBA1
93EB     93EB 93EB 93EB 93EB 50DD C6ED C6ED C6ED C6ED 466D C672 C6ED 7577      E795B7       E795B7       F3B0BBA2
93EC     93EC 93EC 93EC 93EC 4559 C6EE C6EE C6EE C6EE 466E C673 C6EE 5357      E58D97       E58D97       F3B0BBA3
93ED     93ED 93ED 93ED 93ED 4ADF C6EF C6EF C6EF C6EF 466F C674 C6EF 6960      E6A5A0       E6A5A0       F3B0BBA4
93EE     93EE 93EE 93EE 93EE 4CB1 C6F0 C6F0 C6F0 C6F0 4670 C675 C6F0 8EDF      E8BB9F       E8BB9F       F3B0BBA5
93EF     93EF 93EF 93EF 93EF 495A C6F1 C6F1 C6F1 C6F1 4671 C676 C6F1 96E3      E99BA3       E99BA3       F3B0BBA6
93F0     93F0 93F0 93F0 93F0 555E C6F2 C6F2 C6F2 C6F2 4672 C677 C6F2 6C5D      E6B19D       E6B19D       F3B0BBA7
93F1     93F1 93F1 93F1 93F1 4542 C6F3 C6F3 C6F3 C6F3 4673 C678 C6F3 4E8C      E4BA8C       E4BA8C       F3B0BBA8
93F2     93F2 93F2 93F2 93F2 4A86 C6F4 C6F4 C6F4 C6F4 4674 C680 C6F4 5C3C      E5B0BC       E5B0BC       F3B0BBA9
93F3     93F3 93F3 93F3 93F3 50E1 C6F5 C6F5 C6F5 C6F5 4675 C68B C6F5 5F10      E5BC90       E5BC90       F3B0BBAA
93F4     93F4 93F4 93F4 93F4 64F6 EDEE C6F6 C6F6 EDEE 6D6E 7273 C6F6 8FE9      E8BFA9       E8BFA9       F3B0BBAB
93F5     93F5 93F5 93F5 93F5 4ECB C6F7 C6F7 C6F7 C6F7 4677 C69C C6F7 5302      E58C82       E58C82       F3B0BBAC
93F6     93F6 93F6 93F6 93F6 539E C6F8 C6F8 C6F8 C6F8 4678 C6A0 C6F8 8CD1      E8B391       E8B391       F3B0BBAD
93F7     93F7 93F7 93F7 93F7 4A42 C6F9 C6F9 C6F9 C6F9 4679 C6AB C6F9 8089      E88289       E88289       F3B0BBAE
93F8     93F8 93F8 93F8 93F8 51B6 C6FA C6FA C6FA C6FA 467A C6B0 C6FA 8679      E899B9       E899B9       F3B0BBAF
93F9     93F9 93F9 93F9 93F9 51D9 C6FB C6FB C6FB C6FB 467B C6C0 C6FB 5EFF      E5BBBF       E5BBBF       F3B0BBB0
93FA     93FA 93FA 93FA 93FA 4562 C6FC C6FC C6FC C6FC 467C C66A C6FC 65E5      E697A5       E697A5       F3B0BBB1
93FB     93FB 93FB 93FB 93FB 4A98 C6FD C6FD C6FD C6FD 467D C6D0 C6FD 4E73      E4B9B3       E4B9B3       F3B0BBB2
93FC     93FC 93FC 93FC 93FC 4665 C6FE C6FE C6FE C6FE 467E C6A1 C6FE 5165      E585A5       E585A5       F3B0BBB3
9440     9440 9440 9440 9440 4BF8 C7A1 C7A1 C7A1 C7A1 4721 C74F C7A1 5982      E5A682       E5A682       F3B0BBB4
9441     9441 9441 9441 9441 4CC6 C7A2 C7A2 C7A2 C7A2 4722 C77F C7A2 5C3F      E5B0BF       E5B0BF       F3B0BBB5
9442     9442 9442 9442 9442 4FDF C7A3 C7A3 C7A3 C7A3 4723 C77B C7A3 97EE      E99FAE       E99FAE       F3B0BBB6
9443     9443 9443 9443 9443 4A55 C7A4 C7A4 C7A4 C7A4 4724 C7E0 C7A4 4EFB      E4BBBB       E4BBBB       F3B0BBB7
9444     9444 9444 9444 9444 5043 C7A5 C7A5 C7A5 C7A5 4725 C76C C7A5 598A      E5A68A       E5A68A       F3B0BBB8
9445     9445 9445 9445 9445 4BD7 C7A6 C7A6 C7A6 C7A6 4726 C750 C7A6 5FCD      E5BF8D       E5BF8D       F3B0BBB9
9446     9446 9446 9446 9446 4A7E C7A7 C7A7 C7A7 C7A7 4727 C77D C7A7 8A8D      E8AA8D       E8AA8D       F3B0BBBA
9447     9447 9447 9447 9447 507E C7A8 C7A8 C7A8 C7A8 4728 C74D C7A8 6FE1      E6BFA1       E6BFA1       F3B0BBBB
9448     9448 9448 9448 9448 50FA C7A9 C7A9 C7A9 C7A9 4729 C75D C7A9 79B0      E7A6B0       E7A6B0       F3B0BBBC
9449     9449 9449 9449 9449 4F89 C7AA C7AA C7AA C7AA 472A C75C C7AA 7962      E7A5A2       E7A5A2       F3B0BBBD
944A     944A 944A 944A 944A 4FA9 C7AB C7AB C7AB C7AB 472B C74E C7AB 5BE7      E5AFA7       E5AFA7       F3B0BBBE
944B     944B 944B 944B 944B 544E C7AC C7AC C7AC C7AC 472C C76B C7AC 8471      E891B1       E891B1       F3B0BBBF
944C     944C 944C 944C 944C 4FD0 C7AD C7AD C7AD C7AD 472D C760 C7AD 732B      E78CAB       E78CAB       F3B0BC80
944D     944D 944D 944D 944D 47D5 C7AE C7AE C7AE C7AE 472E C74B C7AE 71B1      E786B1       E786B1       F3B0BC81
944E     944E 944E 944E 944E 4560 C7AF C7AF C7AF C7AF 472F C761 C7AF 5E74      E5B9B4       E5B9B4       F3B0BC82
944F     944F 944F 944F 944F 4AAE C7B0 C7B0 C7B0 C7B0 4730 C7F0 C7B0 5FF5      E5BFB5       E5BFB5       F3B0BC83
9450     9450 9450 9450 9450 54C0 C7B1 C7B1 C7B1 C7B1 4731 C7F1 C7B1 637B      E68DBB       E68DBB       F3B0BC84
9451     9451 9451 9451 9451 5348 C7B2 C7B2 C7B2 C7B2 4732 C7F2 C7B2 649A      E6929A       E6929A       F3B0BC85
9452     9452 9452 9452 9452 49D5 C7B3 C7B3 C7B3 C7B3 4733 C7F3 C7B3 71C3      E78783       E78783       F3B0BC86
9453     9453 9453 9453 9453 4BF7 C7B4 C7B4 C7B4 C7B4 4734 C7F4 C7B4 7C98      E7B298       E7B298       F3B0BC87
9454     9454 9454 9454 9454 4B9F C7B5 C7B5 C7B5 C7B5 4735 C7F5 C7B5 4E43      E4B983       E4B983       F3B0BC88
9455     9455 9455 9455 9455 5564 C7B6 C7B6 C7B6 C7B6 4736 C7F6 C7B6 5EFC      E5BBBC       E5BBBC       F3B0BC89
9456     9456 9456 9456 9456 45D8 C7B7 C7B7 C7B7 C7B7 4737 C7F7 C7B7 4E4B      E4B98B       E4B98B       F3B0BC8A
9457     9457 9457 9457 9457 4DF4 C7B8 C7B8 C7B8 C7B8 4738 C7F8 C7B8 57DC      E59F9C       E59F9C       F3B0BC8B
9458     9458 9458 9458 9458 5190 C7B9 C7B9 C7B9 70BA D6F2 ACDE C7B9 56A2      E59AA2       E59AA2       F3B0BC8C
9459     9459 9459 9459 9459 4BCD C7BA C7BA C7BA C7BA 473A C77A C7BA 60A9      E682A9       E682A9       F3B0BC8D
945A     945A 945A 945A 945A 4ACE C7BB C7BB C7BB C7BB 473B C75E C7BB 6FC3      E6BF83       E6BF83       F3B0BC8E
945B     945B 945B 945B 945B 49E2 C7BC C7BC C7BC C7BC 473C C74C C7BC 7D0D      E7B48D       E7B48D       F3B0BC8F
945C     945C 945C 945C 945C 474F C7BD C7BD C7BD C7BD 473D C77E C7BD 80FD      E883BD       E883BD       F3B0BC90
945D     945D 945D 945D 945D 4CB3 C7BE C7BE C7BE C7BE 473E C76E C7BE 8133      E884B3       E884B3       F3B0BC91
945E     945E 945E 945E 945E 535F C7BF C7BF C7BF C7BF 473F C76F C7BF 81BF      E886BF       E886BF       F3B0BC92
945F     945F 945F 945F 945F 4845 C7C0 C7C0 C7C0 C7C0 4740 C77C C7C0 8FB2      E8BEB2       E8BEB2       F3B0BC93
9460     9460 9460 9460 9460 53EB C7C1 C7C1 C7C1 C7C1 4741 C7C1 C7C1 8997      E8A697       E8A697       F3B0BC94
9461     9461 9461 9461 9461 55A2 C7C2 C7C2 C7C2 C7C2 4742 C7C2 C7C2 86A4      E89AA4       E89AA4       F3B0BC95
9462     9462 9462 9462 9462 4E7D C7C3 C7C3 C7C3 C7C3 4743 C7C3 C7C3 5DF4      E5B7B4       E5B7B4       F3B0BC96
9463     9463 9463 9463 9463 4EE3 C7C4 C7C4 C7C4 C7C4 4744 C7C4 C7C4 628A      E68A8A       E68A8A       F3B0BC97
9464     9464 9464 9464 9464 4E62 C7C5 C7C5 C7C5 C7C5 4745 C7C5 C7C5 64AD      E692AD       E692AD       F3B0BC98
9465     9465 9465 9465 9465 4BFA C7C6 C7C6 C7C6 C7C6 4746 C7C6 C7C6 8987      E8A687       E8A687       F3B0BC99
9466     9466 9466 9466 9466 5280 C7C7 C7C7 C7C7 C7C7 4747 C7C7 C7C7 6777      E69DB7       E69DB7       F3B0BC9A
9467     9467 9467 9467 9467 46CA C7C8 C7C8 C7C8 C7C8 4748 C7C8 C7C8 6CE2      E6B3A2       E6B3A2       F3B0BC9B
9468     9468 9468 9468 9468 4A72 C7C9 C7C9 C7C9 C7C9 4749 C7C9 C7C9 6D3E      E6B4BE       E6B4BE       F3B0BC9C
9469     9469 9469 9469 9469 529F C7CA C7CA C7CA C7CA 474A C7D1 C7CA 7436      E790B6       E790B6       F3B0BC9D
946A     946A 946A 946A 946A 4ABE C7CB C7CB C7CB C7CB 474B C7D2 C7CB 7834      E7A0B4       E7A0B4       F3B0BC9E
946B     946B 946B 946B 946B 4ECC C7CC C7CC C7CC C7CC 474C C7D3 C7CC 5A46      E5A986       E5A986       F3B0BC9F
946C     946C 946C 946C 946C 5579 C7CD C7CD C7CD C7CD 474D C7D4 C7CD 7F75      E7BDB5       E7BDB5       F3B0BCA0
946D     946D 946D 946D 946D 5442 C7CE C7CE C7CE C7CE 474E C7D5 C7CE 82AD      E88AAD       E88AAD       F3B0BCA1
946E     946E 946E 946E 946E 4664 C7CF C7CF C7CF C7CF 474F C7D6 C7CF 99AC      E9A6AC       E9A6AC       F3B0BCA2
946F     946F 946F 946F 946F 4CE1 C7D0 C7D0 C7D0 C7D0 4750 C7D7 C7D0 4FF3      E4BFB3       E4BFB3       F3B0BCA3
9470     9470 9470 9470 9470 4BC7 C7D1 C7D1 C7D1 C7D1 4751 C7D8 C7D1 5EC3      E5BB83       E5BB83       F3B0BCA4
9471     9471 9471 9471 9471 4EE2 C7D2 C7D2 C7D2 C7D2 4752 C7D9 C7D2 62DD      E68B9D       E68B9D       F3B0BCA5
9472     9472 9472 9472 9472 4CFA C7D3 C7D3 C7D3 C7D3 4753 C7E2 C7D3 6392      E68E92       E68E92       F3B0BCA6
9473     9473 9473 9473 9473 4B8A C7D4 C7D4 C7D4 C7D4 4754 C7E3 C7D4 6557      E69597       E69597       F3B0BCA7
9474     9474 9474 9474 9474 4C63 C7D5 C7D5 C7D5 C7D5 4755 C7E4 C7D5 676F      E69DAF       E69DAF       F3B0BCA8
9475     9475 9475 9475 9475 53BA C7D6 C7D6 C7D6 C7D6 4756 C7E5 C7D6 76C3      E79B83       E79B83       F3B0BCA9
9476     9476 9476 9476 9476 556F C7D7 C7D7 C7D7 C7D7 4757 C7E6 C7D7 724C      E7898C       E7898C       F3B0BCAA
9477     9477 9477 9477 9477 4AB2 C7D8 C7D8 C7D8 C7D8 4758 C7E7 C7D8 80CC      E8838C       E8838C       F3B0BCAB
9478     9478 9478 9478 9478 504E C7D9 C7D9 C7D9 C7D9 4759 C7E8 C7D9 80BA      E882BA       E882BA       F3B0BCAC
9479     9479 9479 9479 9479 4EF4 C7DA C7DA C7DA C7DA 475A C7E9 C7DA 8F29      E8BCA9       E8BCA9       F3B0BCAD
947A     947A 947A 947A 947A 47F3 C7DB C7DB C7DB C7DB 475B C74A C7DB 914D      E9858D       E9858D       F3B0BCAE
947B     947B 947B 947B 947B 4AC4 C7DC C7DC C7DC C7DC 475C C75B C7DC 500D      E5808D       E5808D       F3B0BCAF
947C     947C 947C 947C 947C 4DAC C7DD C7DD C7DD C7DD 475D C75A C7DD 57F9      E59FB9       E59FB9       F3B0BCB0
947D     947D 947D 947D 947D 4D80 C7DE C7DE C7DE C7DE 475E C75F C7DE 5A92      E5AA92       E5AA92       F3B0BCB1
947E     947E 947E 947E 947E 4798 C7DF C7DF C7DF C7DF 475F C76D C7DF 6885      E6A285       E6A285       F3B0BCB2
9480     9480 9480 9480 9480 52B3 C7E0 C7E0 C7E0 C7E0 4760 C779 C7E0 6973      E6A5B3       E6A5B3       F3B0BCB3
9481     9481 9481 9481 9481 54EA C7E1 C7E1 C7E1 C7E1 4761 C757 C7E1 7164      E785A4       E785A4       F3B0BCB4
9482     9482 9482 9482 9482 54D6 C7E2 C7E2 C7E2 C7E2 4762 C759 C7E2 72FD      E78BBD       E78BBD       F3B0BCB5
9483     9483 9483 9483 9483 4974 C7E3 C7E3 C7E3 C7E3 4763 C762 C7E3 8CB7      E8B2B7       E8B2B7       F3B0BCB6
9484     9484 9484 9484 9484 4775 C7E4 C7E4 C7E4 C7E4 4764 C763 C7E4 58F2      E5A3B2       E5A3B2       F3B0BCB7
9485     9485 9485 9485 9485 4F4F C7E5 C7E5 C7E5 C7E5 4765 C764 C7E5 8CE0      E8B3A0       E8B3A0       F3B0BCB8
9486     9486 9486 9486 9486 55C3 C7E6 C7E6 C7E6 C7E6 4766 C765 C7E6 966A      E999AA       E999AA       F3B0BCB9
9487     9487 9487 9487 9487 4FCA C7E7 C7E7 C7E7 C7E7 4767 C766 C7E7 9019      E98099       E98099       F3B0BCBA
9488     9488 9488 9488 9488 634B EAA4 C7E8 C7E8 EAA4 6A24 69E0 C7E8 877F      E89DBF       E89DBF       F3B0BCBB
9489     9489 9489 9489 9489 508F C7E9 C7E9 C7E9 C7E9 4769 C768 C7E9 79E4      E7A7A4       E7A7A4       F3B0BCBC
948A     948A 948A 948A 948A 55C9 C7EA C7EA C7EA C7EA 476A C769 C7EA 77E7      E79FA7       E79FA7       F3B0BCBD
948B     948B 948B 948B 948B 494A C7EB C7EB C7EB C7EB 476B C770 C7EB 8429      E890A9       E890A9       F3B0BCBE
948C     948C 948C 948C 948C 49A6 C7EC C7EC C7EC C7EC 476C C771 C7EC 4F2F      E4BCAF       E4BCAF       F3B0BCBF
948D     948D 948D 948D 948D 5481 C7ED C7ED C7ED 44D0 B4BC 848D C7ED 5265      E589A5       E589A5       F3B0BD80
948E     948E 948E 948E 948E 45F6 C7EE C7EE C7EE C7EE 476E C773 C7EE 535A      E58D9A       E58D9A       F3B0BD81
948F     948F 948F 948F 948F 4FBA C7EF C7EF C7EF C7EF 476F C774 C7EF 62CD      E68B8D       E68B8D       F3B0BD82
9490     9490 9490 9490 9490 4980 C7F0 C7F0 C7F0 C7F0 4770 C775 C7F0 67CF      E69F8F       E69F8F       F3B0BD83
9491     9491 9491 9491 9491 4DB8 C7F1 C7F1 C7F1 C7F1 4771 C776 C7F1 6CCA      E6B38A       E6B38A       F3B0BD84
9492     9492 9492 9492 9492 4659 C7F2 C7F2 C7F2 C7F2 4772 C777 C7F2 767D      E799BD       E799BD       F3B0BD85
9493     9493 9493 9493 9493 4EE0 C7F3 C7F3 C7F3 C7F3 4773 C778 C7F3 7B94      E7AE94       E7AE94       F3B0BD86
9494     9494 9494 9494 9494 4C4B C7F4 C7F4 C7F4 C7F4 4774 C780 C7F4 7C95      E7B295       E7B295       F3B0BD87
9495     9495 9495 9495 9495 4B87 C7F5 C7F5 C7F5 C7F5 4775 C78B C7F5 8236      E888B6       E888B6       F3B0BD88
9496     9496 9496 9496 9496 4A92 C7F6 C7F6 C7F6 C7F6 4776 C79B C7F6 8584      E89684       E89684       F3B0BD89
9497     9497 9497 9497 9497 49EF C7F7 C7F7 C7F7 C7F7 4777 C79C C7F7 8FEB      E8BFAB       E8BFAB       F3B0BD8A
9498     9498 9498 9498 9498 52A6 C7F8 C7F8 C7F8 C7F8 4778 C7A0 C7F8 66DD      E69B9D       E69B9D       F3B0BD8B
9499     9499 9499 9499 9499 508E C7F9 C7F9 C7F9 C7F9 4779 C7AB C7F9 6F20      E6BCA0       E6BCA0       F3B0BD8C
949A     949A 949A 949A 949A 4AD9 C7FA C7FA C7FA C7FA 477A C7B0 C7FA 7206      E78886       E78886       F3B0BD8D
949B     949B 949B 949B 949B 52E6 C7FB C7FB C7FB C7FB 477B C7C0 C7FB 7E1B      E7B89B       E7B89B       F3B0BD8E
949C     949C 949C 949C 949C 54B3 C7FC C7FC C7FC C7FC 477C C76A C7FC 83AB      E88EAB       E88EAB       F3B0BD8F
949D     949D 949D 949D 949D 55E5 C7FD C7FD C7FD C7FD 477D C7D0 C7FD 99C1      E9A781       E9A781       F3B0BD90
949E     949E 949E 949E 949E 4AE0 C7FE C7FE C7FE C7FE 477E C7A1 C7FE 9EA6      E9BAA6       E9BAA6       F3B0BD91
949F     949F 949F 949F 949F 4BB6 C8A1 C8A1 C8A1 C8A1 4821 C84F C8A1 51FD      E587BD       E587BD       F3B0BD92
94A0     94A0 94A0 94A0 94A0 4AFB C8A2 C8A2 C8A2 C8A2 4822 C87F C8A2 7BB1      E7AEB1       E7AEB1       F3B0BD93
94A1     94A1 94A1 94A1 94A1 52BF C8A3 C8A3 C8A3 C8A3 4823 C87B C8A3 7872      E7A1B2       E7A1B2       F3B0BD94
94A2     94A2 94A2 94A2 94A2 505C C8A4 C8A4 C8A4 C8A4 4824 C8E0 C8A4 7BB8      E7AEB8       E7AEB8       F3B0BD95
94A3     94A3 94A3 94A3 94A3 4B41 C8A5 C8A5 C8A5 C8A5 4825 C86C C8A5 8087      E88287       E88287       F3B0BD96
94A4     94A4 94A4 94A4 94A4 4D9B C8A6 C8A6 C8A6 C8A6 4826 C850 C8A6 7B48      E7AD88       E7AD88       F3B0BD97
94A5     94A5 94A5 94A5 94A5 51C5 C8A7 C8A7 C8A7 C8A7 4827 C87D C8A7 6AE8      E6ABA8       E6ABA8       F3B0BD98
94A6     94A6 94A6 94A6 94A6 48DE C8A8 C8A8 C8A8 C8A8 4828 C84D C8A8 5E61      E5B9A1       E5B9A1       F3B0BD99
94A7     94A7 94A7 94A7 94A7 4E50 C8A9 C8A9 C8A9 C8A9 4829 C85D C8A9 808C      E8828C       E8828C       F3B0BD9A
94A8     94A8 94A8 94A8 94A8 478B C8AA C8AA C8AA C8AA 482A C85C C8AA 7551      E79591       E79591       F3B0BD9B
94A9     94A9 94A9 94A9 94A9 49FE C8AB C8AB C8AB C8AB 482B C84E C8AB 7560      E795A0       E795A0       F3B0BD9C
94AA     94AA 94AA 94AA 94AA 4548 C8AC C8AC C8AC C8AC 482C C86B C8AC 516B      E585AB       E585AB       F3B0BD9D
94AB     94AB 94AB 94AB 94AB 4EA8 C8AD C8AD C8AD C8AD 482D C860 C8AD 9262      E989A2       E989A2       F3B0BD9E
94AC     94AC 94AC 94AC 94AC 54FA C8AE C8AE C8AE 70DA C1ED 92CF C8AE 6E8C      E6BA8C       E6BA8C       F3B0BD9F
94AD     94AD 94AD 94AD 94AD 464B C8AF C8AF C8AF C8AF 482F C861 C8AF 767A      E799BA       E799BA       F3B0BDA0
94AE     94AE 94AE 94AE 94AE 51FA C8B0 C8B0 C8B0 69A2 D1AC A653 C8B0 9197      E98697       E98697       F3B0BDA1
94AF     94AF 94AF 94AF 94AF 4CF6 C8B1 C8B1 C8B1 C8B1 4831 C8F1 C8B1 9AEA      E9ABAA       E9ABAA       F3B0BDA2
94B0     94B0 94B0 94B0 94B0 50F6 C8B2 C8B2 C8B2 C8B2 4832 C8F2 C8B2 4F10      E4BC90       E4BC90       F3B0BDA3
94B1     94B1 94B1 94B1 94B1 4F82 C8B3 C8B3 C8B3 C8B3 4833 C8F3 C8B3 7F70      E7BDB0       E7BDB0       F3B0BDA4
94B2     94B2 94B2 94B2 94B2 4A85 C8B4 C8B4 C8B4 C8B4 4834 C8F4 C8B4 629C      E68A9C       E68A9C       F3B0BDA5
94B3     94B3 94B3 94B3 94B3 52F5 C8B5 C8B5 C8B5 C8B5 4835 C8F5 C8B5 7B4F      E7AD8F       E7AD8F       F3B0BDA6
94B4     94B4 94B4 94B4 94B4 5172 C8B6 C8B6 C8B6 C8B6 4836 C8F6 C8B6 95A5      E996A5       E996A5       F3B0BDA7
94B5     94B5 94B5 94B5 94B5 4D95 C8B7 C8B7 C8B7 C8B7 4837 C8F7 C8B7 9CE9      E9B3A9       E9B3A9       F3B0BDA8
94B6     94B6 94B6 94B6 94B6 5582 C8B8 C8B8 C8B8 C8B8 4838 C8F8 C8B8 567A      E599BA       E599BA       F3B0BDA9
94B7     94B7 94B7 94B7 94B7 4F6F C8B9 C8B9 C8B9 C8B9 4839 C8F9 C8B9 5859      E5A199       E5A199       F3B0BDAA
94B8     94B8 94B8 94B8 94B8 55C2 C8BA C8BA C8BA C8BA 483A C87A C8BA 86E4      E89BA4       E89BA4       F3B0BDAB
94B9     94B9 94B9 94B9 94B9 4ED5 C8BB C8BB C8BB C8BB 483B C85E C8BB 96BC      E99ABC       E99ABC       F3B0BDAC
94BA     94BA 94BA 94BA 94BA 49C9 C8BC C8BC C8BC C8BC 483C C84C C8BC 4F34      E4BCB4       E4BCB4       F3B0BDAD
94BB     94BB 94BB 94BB 94BB 495C C8BD C8BD C8BD C8BD 483D C87E C8BD 5224      E588A4       E588A4       F3B0BDAE
94BC     94BC 94BC 94BC 94BC 4790 C8BE C8BE C8BE C8BE 483E C86E C8BE 534A      E58D8A       E58D8A       F3B0BDAF
94BD     94BD 94BD 94BD 94BD 47F9 C8BF C8BF C8BF C8BF 483F C86F C8BF 53CD      E58F8D       E58F8D       F3B0BDB0
94BE     94BE 94BE 94BE 94BE 55B2 C8C0 C8C0 C8C0 C8C0 4840 C87C C8C0 53DB      E58F9B       E58F9B       F3B0BDB1
94BF     94BF 94BF 94BF 94BF 4F75 C8C1 C8C1 C8C1 C8C1 4841 C8C1 C8C1 5E06      E5B886       E5B886       F3B0BDB2
94C0     94C0 94C0 94C0 94C0 4DC8 C8C2 C8C2 C8C2 C8C2 4842 C8C2 C8C2 642C      E690AC       E690AC       F3B0BDB3
94C1     94C1 94C1 94C1 94C1 52E4 C8C3 C8C3 C8C3 C8C3 4843 C8C3 C8C3 6591      E69691       E69691       F3B0BDB4
94C2     94C2 94C2 94C2 94C2 46B4 C8C4 C8C4 C8C4 C8C4 4844 C8C4 C8C4 677F      E69DBF       E69DBF       F3B0BDB5
94C3     94C3 94C3 94C3 94C3 556E C8C5 C8C5 C8C5 C8C5 4845 C8C5 C8C5 6C3E      E6B0BE       E6B0BE       F3B0BDB6
94C4     94C4 94C4 94C4 94C4 505E C8C6 C8C6 C8C6 C8C6 4846 C8C6 C8C6 6C4E      E6B18E       E6B18E       F3B0BDB7
94C5     94C5 94C5 94C5 94C5 4ABD C8C7 C8C7 C8C7 C8C7 4847 C8C7 C8C7 7248      E78988       E78988       F3B0BDB8
94C6     94C6 94C6 94C6 94C6 4CA7 C8C8 C8C8 C8C8 C8C8 4848 C8C8 C8C8 72AF      E78AAF       E78AAF       F3B0BDB9
94C7     94C7 94C7 94C7 94C7 5066 C8C9 C8C9 C8C9 C8C9 4849 C8C9 C8C9 73ED      E78FAD       E78FAD       F3B0BDBA
94C8     94C8 94C8 94C8 94C8 4EDC C8CA C8CA C8CA C8CA 484A C8D1 C8CA 7554      E79594       E79594       F3B0BDBB
94C9     94C9 94C9 94C9 94C9 4759 C8CB C8CB C8CB C8CB 484B C8D2 C8CB 7E41      E7B981       E7B981       F3B0BDBC
94CA     94CA 94CA 94CA 94CA 4956 C8CC C8CC C8CC C8CC 484C C8D3 C8CC 822C      E888AC       E888AC       F3B0BDBD
94CB     94CB 94CB 94CB 94CB 4FF6 C8CD C8CD C8CD C8CD 484D C8D4 C8CD 85E9      E897A9       E897A9       F3B0BDBE
94CC     94CC 94CC 94CC 94CC 4A5D C8CE C8CE C8CE C8CE 484E C8D5 C8CE 8CA9      E8B2A9       E8B2A9       F3B0BDBF
94CD     94CD 94CD 94CD 94CD 49E8 C8CF C8CF C8CF C8CF 484F C8D6 C8CF 7BC4      E7AF84       E7AF84       F3B0BE80
94CE     94CE 94CE 94CE 94CE 5346 C8D0 C8D0 C8D0 C8D0 4850 C8D7 C8D0 91C6      E98786       E98786       F3B0BE81
94CF     94CF 94CF 94CF 94CF 54A0 C8D1 C8D1 C8D1 C8D1 4851 C8D8 C8D1 7169      E785A9       E785A9       F3B0BE82
94D0     94D0 94D0 94D0 94D0 5557 C8D2 C8D2 C8D2 C8D2 4852 C8D9 C8D2 9812      E9A092       E9A092       F3B0BE83
94D1     94D1 94D1 94D1 94D1 46B5 C8D3 C8D3 C8D3 C8D3 4853 C8E2 C8D3 98EF      E9A3AF       E9A3AF       F3B0BE84
94D2     94D2 94D2 94D2 94D2 52A8 C8D4 C8D4 C8D4 C8D4 4854 C8E3 C8D4 633D      E68CBD       E68CBD       F3B0BE85
94D3     94D3 94D3 94D3 94D3 4CB6 C8D5 C8D5 C8D5 C8D5 4855 C8E4 C8D5 6669      E699A9       E699A9       F3B0BE86
94D4     94D4 94D4 94D4 94D4 48B9 C8D6 C8D6 C8D6 C8D6 4856 C8E5 C8D6 756A      E795AA       E795AA       F3B0BE87
94D5     94D5 94D5 94D5 94D5 4890 C8D7 C8D7 C8D7 C8D7 4857 C8E6 C8D7 76E4      E79BA4       E79BA4       F3B0BE88
94D6     94D6 94D6 94D6 94D6 4B9C C8D8 C8D8 C8D8 C8D8 4858 C8E7 C8D8 78D0      E7A390       E7A390       F3B0BE89
94D7     94D7 94D7 94D7 94D7 52F7 C8D9 C8D9 C8D9 C8D9 4859 C8E8 C8D9 8543      E89583       E89583       F3B0BE8A
94D8     94D8 94D8 94D8 94D8 53A5 C8DA C8DA C8DA C8DA 485A C8E9 C8DA 86EE      E89BAE       E89BAE       F3B0BE8B
94D9     94D9 94D9 94D9 94D9 55ED C8DB C8DB C8DB C8DB 485B C84A C8DB 532A      E58CAA       E58CAA       F3B0BE8C
94DA     94DA 94DA 94DA 94DA 53AD C8DC C8DC C8DC C8DC 485C C85B C8DC 5351      E58D91       E58D91       F3B0BE8D
94DB     94DB 94DB 94DB 94DB 4C4F C8DD C8DD C8DD C8DD 485D C85A C8DD 5426      E590A6       E590A6       F3B0BE8E
94DC     94DC 94DC 94DC 94DC 51A2 C8DE C8DE C8DE C8DE 485E C85F C8DE 5983      E5A683       E5A683       F3B0BE8F
94DD     94DD 94DD 94DD 94DD 558D C8DF C8DF C8DF C8DF 485F C86D C8DF 5E87      E5BA87       E5BA87       F3B0BE90
94DE     94DE 94DE 94DE 94DE 4772 C8E0 C8E0 C8E0 C8E0 4860 C879 C8E0 5F7C      E5BDBC       E5BDBC       F3B0BE91
94DF     94DF 94DF 94DF 94DF 4C9D C8E1 C8E1 C8E1 C8E1 4861 C857 C8E1 60B2      E682B2       E682B2       F3B0BE92
94E0     94E0 94E0 94E0 94E0 50AD C8E2 C8E2 C8E2 C8E2 4862 C859 C8E2 6249      E68989       E68989       F3B0BE93
94E1     94E1 94E1 94E1 94E1 4C76 C8E3 C8E3 C8E3 C8E3 4863 C862 C8E3 6279      E689B9       E689B9       F3B0BE94
94E2     94E2 94E2 94E2 94E2 4FA6 C8E4 C8E4 C8E4 C8E4 4864 C863 C8E4 62AB      E68AAB       E68AAB       F3B0BE95
94E3     94E3 94E3 94E3 94E3 4B80 C8E5 C8E5 C8E5 C8E5 4865 C864 C8E5 6590      E69690       E69690       F3B0BE96
94E4     94E4 94E4 94E4 94E4 474E C8E6 C8E6 C8E6 C8E6 4866 C865 C8E6 6BD4      E6AF94       E6AF94       F3B0BE97
94E5     94E5 94E5 94E5 94E5 51FD C8E7 C8E7 C8E7 C8E7 4867 C866 C8E7 6CCC      E6B38C       E6B38C       F3B0BE98
94E6     94E6 94E6 94E6 94E6 4D97 C8E8 C8E8 C8E8 C8E8 4868 C867 C8E8 75B2      E796B2       E796B2       F3B0BE99
94E7     94E7 94E7 94E7 94E7 49DA C8E9 C8E9 C8E9 C8E9 4869 C868 C8E9 76AE      E79AAE       E79AAE       F3B0BE9A
94E8     94E8 94E8 94E8 94E8 517B C8EA C8EA C8EA C8EA 486A C869 C8EA 7891      E7A291       E7A291       F3B0BE9B
94E9     94E9 94E9 94E9 94E9 4C70 C8EB C8EB C8EB C8EB 486B C870 C8EB 79D8      E7A798       E7A798       F3B0BE9C
94EA     94EA 94EA 94EA 94EA 537D C8EC C8EC C8EC C8EC 486C C871 C8EC 7DCB      E7B78B       E7B78B       F3B0BE9D
94EB     94EB 94EB 94EB 94EB 55A5 C8ED C8ED C8ED C8ED 486D C872 C8ED 7F77      E7BDB7       E7BDB7       F3B0BE9E
94EC     94EC 94EC 94EC 94EC 497B C8EE C8EE C8EE C8EE 486E C873 C8EE 80A5      E882A5       E882A5       F3B0BE9F
94ED     94ED 94ED 94ED 94ED 4B8D C8EF C8EF C8EF C8EF 486F C874 C8EF 88AB      E8A2AB       E8A2AB       F3B0BEA0
94EE     94EE 94EE 94EE 94EE 55F2 C8F0 C8F0 C8F0 C8F0 4870 C875 C8F0 8AB9      E8AAB9       E8AAB9       F3B0BEA1
94EF     94EF 94EF 94EF 94EF 46D0 C8F1 C8F1 C8F1 C8F1 4871 C876 C8F1 8CBB      E8B2BB       E8B2BB       F3B0BEA2
94F0     94F0 94F0 94F0 94F0 4CC4 C8F2 C8F2 C8F2 C8F2 4872 C877 C8F2 907F      E981BF       E981BF       F3B0BEA3
94F1     94F1 94F1 94F1 94F1 48E1 C8F3 C8F3 C8F3 C8F3 4873 C878 C8F3 975E      E99D9E       E99D9E       F3B0BEA4
94F2     94F2 94F2 94F2 94F2 488A C8F4 C8F4 C8F4 C8F4 4874 C880 C8F4 98DB      E9A39B       E9A39B       F3B0BEA5
94F3     94F3 94F3 94F3 94F3 494F C8F5 C8F5 C8F5 C8F5 4875 C88B C8F5 6A0B      E6A88B       E6A88B       F3B0BEA6
94F4     94F4 94F4 94F4 94F4 51AB C8F6 C8F6 C8F6 C8F6 4876 C89B C8F6 7C38      E7B0B8       E7B0B8       F3B0BEA7
94F5     94F5 94F5 94F5 94F5 46AC C8F7 C8F7 C8F7 C8F7 4877 C89C C8F7 5099      E58299       E58299       F3B0BEA8
94F6     94F6 94F6 94F6 94F6 45C8 C8F8 C8F8 C8F8 C8F8 4878 C8A0 C8F8 5C3E      E5B0BE       E5B0BE       F3B0BEA9
94F7     94F7 94F7 94F7 94F7 4B5D C8F9 C8F9 C8F9 C8F9 4879 C8AB C8F9 5FAE      E5BEAE       E5BEAE       F3B0BEAA
94F8     94F8 94F8 94F8 94F8 525E C8FA C8FA C8FA C8FA 487A C8B0 C8FA 6787      E69E87       E69E87       F3B0BEAB
94F9     94F9 94F9 94F9 94F9 53D7 C8FB C8FB C8FB C8FB 487B C8C0 C8FB 6BD8      E6AF98       E6AF98       F3B0BEAC
94FA     94FA 94FA 94FA 94FA 52A3 C8FC C8FC C8FC C8FC 487C C86A C8FC 7435      E790B5       E790B5       F3B0BEAD
94FB     94FB 94FB 94FB 94FB 52BC C8FD C8FD C8FD C8FD 487D C8D0 C8FD 7709      E79C89       E79C89       F3B0BEAE
94FC     94FC 94FC 94FC 94FC 457D C8FE C8FE C8FE C8FE 487E C8A1 C8FE 7F8E      E7BE8E       E7BE8E       F3B0BEAF
9540     9540 9540 9540 9540 4CFD C9A1 C9A1 C9A1 C9A1 4921 C94F C9A1 9F3B      E9BCBB       E9BCBB       F3B0BEB0
9541     9541 9541 9541 9541 52D9 C9A2 C9A2 C9A2 C9A2 4922 C97F C9A2 67CA      E69F8A       E69F8A       F3B0BEB1
9542     9542 9542 9542 9542 5087 C9A3 C9A3 C9A3 C9A3 4923 C97B C9A3 7A17      E7A897       E7A897       F3B0BEB2
9543     9543 9543 9543 9543 4FD1 C9A4 C9A4 C9A4 C9A4 4924 C9E0 C9A4 5339      E58CB9       E58CB9       F3B0BEB3
9544     9544 9544 9544 9544 4EA0 C9A5 C9A5 C9A5 C9A5 4925 C96C C9A5 758B      E7968B       E7968B       F3B0BEB4
9545     9545 9545 9545 9545 509D C9A6 C9A6 C9A6 C9A6 4926 C950 C9A6 9AED      E9ABAD       E9ABAD       F3B0BEB5
9546     9546 9546 9546 9546 45E1 C9A7 C9A7 C9A7 C9A7 4927 C97D C9A7 5F66      E5BDA6       E5BDA6       F3B0BEB6
9547     9547 9547 9547 9547 5060 C9A8 C9A8 C9A8 C9A8 4928 C94D C9A8 819D      E8869D       E8869D       F3B0BEB7
9548     9548 9548 9548 9548 48CC C9A9 C9A9 C9A9 C9A9 4929 C95D C9A9 83F1      E88FB1       E88FB1       F3B0BEB8
9549     9549 9549 9549 9549 526A C9AA C9AA C9AA C9AA 492A C95C C9AA 8098      E88298       E88298       F3B0BEB9
954A     954A 954A 954A 954A 52E0 C9AB C9AB C9AB C9AB 492B C94E C9AB 5F3C      E5BCBC       E5BCBC       F3B0BEBA
954B     954B 954B 954B 954B 47CA C9AC C9AC C9AC C9AC 492C C96B C9AC 5FC5      E5BF85       E5BF85       F3B0BEBB
954C     954C 954C 954C 954C 559A C9AD C9AD C9AD C9AD 492D C960 C9AD 7562      E795A2       E795A2       F3B0BEBC
954D     954D 954D 954D 954D 4AC7 C9AE C9AE C9AE C9AE 492E C94B C9AE 7B46      E7AD86       E7AD86       F3B0BEBD
954E     954E 954E 954E 954E 55AC C9AF C9AF C9AF C9AF 492F C961 C9AF 903C      E980BC       E980BC       F3B0BEBE
954F     954F 954F 954F 954F 5046 DBD8 C9B0 C9B0 DBD8 5B58 4AE7 C9B0 6867      E6A1A7       E6A1A7       F3B0BEBF
9550     9550 9550 9550 9550 4AB3 C9B1 C9B1 C9B1 C9B1 4931 C9F1 C9B1 59EB      E5A7AB       E5A7AB       F3B0BF80
9551     9551 9551 9551 9551 4855 C9B2 C9B2 C9B2 C9B2 4932 C9F2 C9B2 5A9B      E5AA9B       E5AA9B       F3B0BF81
9552     9552 9552 9552 9552 4FB5 C9B3 C9B3 C9B3 C9B3 4933 C9F3 C9B3 7D10      E7B490       E7B490       F3B0BF82
9553     9553 9553 9553 9553 454B C9B4 C9B4 C9B4 C9B4 4934 C9F4 C9B4 767E      E799BE       E799BE       F3B0BF83
9554     9554 9554 9554 9554 55A1 C9B5 C9B5 C9B5 C9B5 4935 C9F5 C9B5 8B2C      E8ACAC       E8ACAC       F3B0BF84
9555     9555 9555 9555 9555 4DB1 C9B6 C9B6 C9B6 C9B6 4936 C9F6 C9B6 4FF5      E4BFB5       E4BFB5       F3B0BF85
9556     9556 9556 9556 9556 5159 C9B7 C9B7 C9B7 C9B7 4937 C9F7 C9B7 5F6A      E5BDAA       E5BDAA       F3B0BF86
9557     9557 9557 9557 9557 4966 C9B8 C9B8 C9B8 C9B8 4938 C9F8 C9B8 6A19      E6A899       E6A899       F3B0BF87
9558     9558 9558 9558 9558 4CA4 C9B9 C9B9 C9B9 C9B9 4939 C9F9 C9B9 6C37      E6B0B7       E6B0B7       F3B0BF88
9559     9559 9559 9559 9559 4FBC C9BA C9BA C9BA C9BA 493A C97A C9BA 6F02      E6BC82       E6BC82       F3B0BF89
955A     955A 955A 955A 955A 5452 C9BB C9BB C9BB C9BB 493B C95E C9BB 74E2      E793A2       E793A2       F3B0BF8A
955B     955B 955B 955B 955B 4DBF C9BC C9BC C9BC C9BC 493C C94C C9BC 7968      E7A5A8       E7A5A8       F3B0BF8B
955C     955C 955C 955C 955C 46C0 C9BD C9BD C9BD C9BD 493D C97E C9BD 8868      E8A1A8       E8A1A8       F3B0BF8C
955D     955D 955D 955D 955D 49FD C9BE C9BE C9BE C9BE 493E C96E C9BE 8A55      E8A995       E8A995       F3B0BF8D
955E     955E 955E 955E 955E 5593 C9BF C9BF C9BF C9BF 493F C96F C9BF 8C79      E8B1B9       E8B1B9       F3B0BF8E
955F     955F 955F 955F 955F 55AF C9C0 C9C0 C9C0 C9C0 4940 C97C C9C0 5EDF      E5BB9F       E5BB9F       F3B0BF8F
9560     9560 9560 9560 9560 4BCF C9C1 C9C1 C9C1 C9C1 4941 C9C1 C9C1 63CF      E68F8F       E68F8F       F3B0BF90
9561     9561 9561 9561 9561 49BF C9C2 C9C2 C9C2 C9C2 4942 C9C2 C9C2 75C5      E79785       E79785       F3B0BF91
9562     9562 9562 9562 9562 4DBA C9C3 C9C3 C9C3 C9C3 4943 C9C3 C9C3 79D2      E7A792       E7A792       F3B0BF92
9563     9563 9563 9563 9563 4AB8 C9C4 C9C4 C9C4 C9C4 4944 C9C4 C9C4 82D7      E88B97       E88B97       F3B0BF93
9564     9564 9564 9564 9564 53F0 C9C5 C9C5 C9C5 C9C5 4945 C9C5 C9C5 9328      E98CA8       E98CA8       F3B0BF94
9565     9565 9565 9565 9565 55BE C9C6 C9C6 C9C6 C9C6 4946 C9C6 C9C6 92F2      E98BB2       E98BB2       F3B0BF95
9566     9566 9566 9566 9566 5259 C9C7 C9C7 C9C7 C9C7 4947 C9C7 C9C7 849C      E8929C       E8929C       F3B0BF96
9567     9567 9567 9567 9567 4E66 C9C8 C9C8 C9C8 C9C8 4948 C9C8 C9C8 86ED      E89BAD       E89BAD       F3B0BF97
9568     9568 9568 9568 9568 546E C9C9 C9C9 C9C9 C9C9 4949 C9C9 C9C9 9C2D      E9B0AD       E9B0AD       F3B0BF98
9569     9569 9569 9569 9569 45D0 C9CA C9CA C9CA C9CA 494A C9D1 C9CA 54C1      E59381       E59381       F3B0BF99
956A     956A 956A 956A 956A 4E88 C9CB C9CB C9CB C9CB 494B C9D2 C9CB 5F6C      E5BDAC       E5BDAC       F3B0BF9A
956B     956B 956B 956B 956B 519D C9CC C9CC C9CC C9CC 494C C9D3 C9CC 658C      E6968C       E6968C       F3B0BF9B
956C     956C 956C 956C 956C 45EA C9CD C9CD C9CD C9CD 494D C9D4 C9CD 6D5C      E6B59C       E6B59C       F3B0BF9C
956D     956D 956D 956D 956D 55D1 C9CE C9CE C9CE C9CE 494E C9D5 C9CE 7015      E78095       E78095       F3B0BF9D
956E     956E 956E 956E 956E 4DA3 C9CF C9CF C9CF C9CF 494F C9D6 C9CF 8CA7      E8B2A7       E8B2A7       F3B0BF9E
956F     956F 956F 956F 956F 50D7 C9D0 C9D0 C9D0 C9D0 4950 C9D7 C9D0 8CD3      E8B393       E8B393       F3B0BF9F
9570     9570 9570 9570 9570 53AB C9D1 C9D1 C9D1 6CFB D3FB A8FB C9D1 983B      E9A0BB       E9A0BB       F3B0BFA0
9571     9571 9571 9571 9571 4647 C9D2 C9D2 C9D2 C9D2 4952 C9D9 C9D2 654F      E6958F       E6958F       F3B0BFA1
9572     9572 9572 9572 9572 4CD4 C9D3 C9D3 C9D3 C9D3 4953 C9E2 C9D3 74F6      E793B6       E793B6       F3B0BFA2
9573     9573 9573 9573 9573 46AF C9D4 C9D4 C9D4 C9D4 4954 C9E3 C9D4 4E0D      E4B88D       E4B88D       F3B0BFA3
9574     9574 9574 9574 9574 475D C9D5 C9D5 C9D5 C9D5 4955 C9E4 C9D5 4ED8      E4BB98       E4BB98       F3B0BFA4
9575     9575 9575 9575 9575 517E C9D6 C9D6 C9D6 C9D6 4956 C9E5 C9D6 57E0      E59FA0       E59FA0       F3B0BFA5
9576     9576 9576 9576 9576 4573 C9D7 C9D7 C9D7 C9D7 4957 C9E6 C9D7 592B      E5A4AB       E5A4AB       F3B0BFA6
9577     9577 9577 9577 9577 496F C9D8 C9D8 C9D8 C9D8 4958 C9E7 C9D8 5A66      E5A9A6       E5A9A6       F3B0BFA7
9578     9578 9578 9578 9578 45AC C9D9 C9D9 C9D9 C9D9 4959 C9E8 C9D9 5BCC      E5AF8C       E5AF8C       F3B0BFA8
9579     9579 9579 9579 9579 4878 C9DA C9DA C9DA C9DA 495A C9E9 C9DA 51A8      E586A8       E586A8       F3B0BFA9
957A     957A 957A 957A 957A 476B C9DB C9DB C9DB C9DB 495B C94A C9DB 5E03      E5B883       E5B883       F3B0BFAA
957B     957B 957B 957B 957B 4551 C9DC C9DC C9DC C9DC 495C C95B C9DC 5E9C      E5BA9C       E5BA9C       F3B0BFAB
957C     957C 957C 957C 957C 4EF0 C9DD C9DD C9DD C9DD 495D C95A C9DD 6016      E68096       E68096       F3B0BFAC
957D     957D 957D 957D 957D 4D7A C9DE C9DE C9DE C9DE 495E C95F C9DE 6276      E689B6       E689B6       F3B0BFAD
957E     957E 957E 957E 957E 4951 C9DF C9DF C9DF C9DF 495F C96D C9DF 6577      E695B7       E695B7       F3B0BFAE
9580     9580 9580 9580 9580 51C2 C9E0 C9E0 C9E0 C9E0 4960 C979 C9E0 65A7      E696A7       E696A7       F3B0BFAF
9581     9581 9581 9581 9581 49F8 C9E1 C9E1 C9E1 C9E1 4961 C957 C9E1 666E      E699AE       E699AE       F3B0BFB0
9582     9582 9582 9582 9582 4A8C C9E2 C9E2 C9E2 C9E2 4962 C959 C9E2 6D6E      E6B5AE       E6B5AE       F3B0BFB1
9583     9583 9583 9583 9583 45CC C9E3 C9E3 C9E3 C9E3 4963 C962 C9E3 7236      E788B6       E788B6       F3B0BFB2
9584     9584 9584 9584 9584 4FD9 C9E4 C9E4 C9E4 C9E4 4964 C963 C9E4 7B26      E7ACA6       E7ACA6       F3B0BFB3
9585     9585 9585 9585 9585 4D62 C9E5 C9E5 C9E5 C9E5 4965 C964 C9E5 8150      E88590       E88590       F3B0BFB4
9586     9586 9586 9586 9586 5191 C9E6 C9E6 C9E6 C9E6 4966 C965 C9E6 819A      E8869A       E8869A       F3B0BFB5
9587     9587 9587 9587 9587 4EBD C9E7 C9E7 C9E7 C9E7 4967 C966 C9E7 8299      E88A99       E88A99       F3B0BFB6
9588     9588 9588 9588 9588 4E4F C9E8 C9E8 C9E8 C9E8 4968 C967 C9E8 8B5C      E8AD9C       E8AD9C       F3B0BFB7
9589     9589 9589 9589 9589 49C8 C9E9 C9E9 C9E9 C9E9 4969 C968 C9E9 8CA0      E8B2A0       E8B2A0       F3B0BFB8
958A     958A 958A 958A 958A 4FA7 C9EA C9EA C9EA C9EA 496A C969 C9EA 8CE6      E8B3A6       E8B3A6       F3B0BFB9
958B     958B 958B 958B 958B 50EE C9EB C9EB C9EB C9EB 496B C970 C9EB 8D74      E8B5B4       E8B5B4       F3B0BFBA
958C     958C 958C 958C 958C 4786 C9EC C9EC C9EC C9EC 496C C971 C9EC 961C      E9989C       E9989C       F3B0BFBB
958D     958D 958D 958D 958D 4B8C C9ED C9ED C9ED C9ED 496D C972 C9ED 9644      E99984       E99984       F3B0BFBC
958E     958E 958E 958E 958E 53B9 C9EE C9EE C9EE C9EE 496E C973 C9EE 4FAE      E4BEAE       E4BEAE       F3B0BFBD
958F     958F 958F 958F 958F 51E1 C9EF C9EF C9EF C9EF 496F C974 C9EF 64AB      E692AB       E692AB       F3B0BFBE
9590     9590 9590 9590 9590 45B1 C9F0 C9F0 C9F0 C9F0 4970 C975 C9F0 6B66      E6ADA6       E6ADA6       F3B0BFBF
9591     9591 9591 9591 9591 4A63 C9F1 C9F1 C9F1 C9F1 4971 C976 C9F1 821E      E8889E       E8889E       F3B18080
9592     9592 9592 9592 9592 5547 C9F2 C9F2 C9F2 C9F2 4972 C977 C9F2 8461      E891A1       E891A1       F3B18081
9593     9593 9593 9593 9593 51D3 C9F3 C9F3 C9F3 C9F3 4973 C978 C9F3 856A      E895AA       E895AA       F3B18082
9594     9594 9594 9594 9594 4595 C9F4 C9F4 C9F4 C9F4 4974 C980 C9F4 90E8      E983A8       E983A8       F3B18083
9595     9595 9595 9595 9595 4D76 C9F5 C9F5 C9F5 C9F5 4975 C98B C9F5 5C01      E5B081       E5B081       F3B18084
9596     9596 9596 9596 9596 5295 C9F6 C9F6 C9F6 C9F6 4976 C99B C9F6 6953      E6A593       E6A593       F3B18085
9597     9597 9597 9597 9597 47E7 C9F7 C9F7 C9F7 C9F7 4977 C99C C9F7 98A8      E9A2A8       E9A2A8       F3B18086
9598     9598 9598 9598 9598 50E0 C9F8 C9F8 C9F8 C9F8 4978 C9A0 C9F8 847A      E891BA       E891BA       F3B18087
9599     9599 9599 9599 9599 5491 C9F9 C9F9 C9F9 C9F9 4979 C9AB C9F9 8557      E89597       E89597       F3B18088
959A     959A 959A 959A 959A 4A83 C9FA C9FA C9FA C9FA 497A C9B0 C9FA 4F0F      E4BC8F       E4BC8F       F3B18089
959B     959B 959B 959B 959B 4B4E C9FB C9FB C9FB C9FB 497B C9C0 C9FB 526F      E589AF       E589AF       F3B1808A
959C     959C 959C 959C 959C 4A70 C9FC C9FC C9FC C9FC 497C C96A C9FC 5FA9      E5BEA9       E5BEA9       F3B1808B
959D     959D 959D 959D 959D 4A5E C9FD C9FD C9FD C9FD 497D C9D0 C9FD 5E45      E5B985       E5B985       F3B1808C
959E     959E 959E 959E 959E 47D4 C9FE C9FE C9FE C9FE 497E C9A1 C9FE 670D      E69C8D       E69C8D       F3B1808D
959F     959F 959F 959F 959F 458B CAA1 CAA1 CAA1 CAA1 4A21 D14F CAA1 798F      E7A68F       E7A68F       F3B1808E
95A0     95A0 95A0 95A0 95A0 4CA0 CAA2 CAA2 CAA2 CAA2 4A22 D17F CAA2 8179      E885B9       E885B9       F3B1808F
95A1     95A1 95A1 95A1 95A1 4B51 CAA3 CAA3 CAA3 CAA3 4A23 D17B CAA3 8907      E8A487       E8A487       F3B18090
95A2     95A2 95A2 95A2 95A2 4DF2 CAA4 CAA4 CAA4 CAA4 4A24 D1E0 CAA4 8986      E8A686       E8A686       F3B18091
95A3     95A3 95A3 95A3 95A3 48EA CAA5 CAA5 CAA5 CAA5 4A25 D16C CAA5 6DF5      E6B7B5       E6B7B5       F3B18092
95A4     95A4 95A4 95A4 95A4 54AC CAA6 CAA6 CAA6 CAA6 4A26 D150 CAA6 5F17      E5BC97       E5BC97       F3B18093
95A5     95A5 95A5 95A5 95A5 4981 CAA7 CAA7 CAA7 CAA7 4A27 D17D CAA7 6255      E68995       E68995       F3B18094
95A6     95A6 95A6 95A6 95A6 4EAA CAA8 CAA8 CAA8 CAA8 4A28 D14D CAA8 6CB8      E6B2B8       E6B2B8       F3B18095
95A7     95A7 95A7 95A7 95A7 4BE5 CAA9 CAA9 CAA9 CAA9 4A29 D15D CAA9 4ECF      E4BB8F       E4BB8F       F3B18096
95A8     95A8 95A8 95A8 95A8 4652 CAAA CAAA CAAA CAAA 4A2A D15C CAAA 7269      E789A9       E789A9       F3B18097
95A9     95A9 95A9 95A9 95A9 528D CAAB CAAB CAAB CAAB 4A2B D14E CAAB 9B92      E9AE92       E9AE92       F3B18098
95AA     95AA 95AA 95AA 95AA 45B4 CAAC CAAC CAAC CAAC 4A2C D16B CAAC 5206      E58886       E58886       F3B18099
95AB     95AB 95AB 95AB 95AB 53E5 CAAD CAAD CAAD CAAD 4A2D D160 CAAD 543B      E590BB       E590BB       F3B1809A
95AC     95AC 95AC 95AC 95AC 4E6D CAAE CAAE CAAE CAAE 4A2E D14B CAAE 5674      E599B4       E599B4       F3B1809B
95AD     95AD 95AD 95AD 95AD 5289 CAAF CAAF CAAF CAAF 4A2F D161 CAAF 58B3      E5A2B3       E5A2B3       F3B1809C
95AE     95AE 95AE 95AE 95AE 4FFC CAB0 CAB0 CAB0 CAB0 4A30 D1F0 CAB0 61A4      E686A4       E686A4       F3B1809D
95AF     95AF 95AF 95AF 95AF 54B7 CAB1 CAB1 CAB1 CAB1 4A31 D1F1 CAB1 626E      E689AE       E689AE       F3B1809E
95B0     95B0 95B0 95B0 95B0 544F CAB2 CAB2 CAB2 CAB2 4A32 D1F2 CAB2 711A      E7849A       E7849A       F3B1809F
95B1     95B1 95B1 95B1 95B1 4EDD CAB3 CAB3 CAB3 CAB3 4A33 D1F3 CAB3 596E      E5A5AE       E5A5AE       F3B180A0
95B2     95B2 95B2 95B2 95B2 489E CAB4 CAB4 CAB4 CAB4 4A34 D1F4 CAB4 7C89      E7B289       E7B289       F3B180A1
95B3     95B3 95B3 95B3 95B3 5492 CAB5 CAB5 CAB5 CAB5 4A35 D1F5 CAB5 7CDE      E7B39E       E7B39E       F3B180A2
95B4     95B4 95B4 95B4 95B4 4FF3 CAB6 CAB6 CAB6 CAB6 4A36 D1F6 CAB6 7D1B      E7B49B       E7B49B       F3B180A3
95B5     95B5 95B5 95B5 95B5 4F90 CAB7 CAB7 CAB7 CAB7 4A37 D1F7 CAB7 96F0      E99BB0       E99BB0       F3B180A4
95B6     95B6 95B6 95B6 95B6 45CA CAB8 CAB8 CAB8 CAB8 4A38 D1F8 CAB8 6587      E69687       E69687       F3B180A5
95B7     95B7 95B7 95B7 95B7 48BB CAB9 CAB9 CAB9 CAB9 4A39 D1F9 CAB9 805E      E8819E       E8819E       F3B180A6
95B8     95B8 95B8 95B8 95B8 50C5 CABA CABA CABA CABA 4A3A D17A CABA 4E19      E4B899       E4B899       F3B180A7
95B9     95B9 95B9 95B9 95B9 4B94 CABB CABB CABB CABB 4A3B D15E CABB 4F75      E4BDB5       E4BDB5       F3B180A8
95BA     95BA 95BA 95BA 95BA 4649 CABC CABC CABC CABC 4A3C D14C CABC 5175      E585B5       E585B5       F3B180A9
95BB     95BB 95BB 95BB 95BB 539F CABD CABD CABD CABD 4A3D D17E CABD 5840      E5A180       E5A180       F3B180AA
95BC     95BC 95BC 95BC 95BC 4E7F CABE CABE CABE CABE 4A3E D16E CABE 5E63      E5B9A3       E5B9A3       F3B180AB
95BD     95BD 95BD 95BD 95BD 458D CABF CABF CABF CABF 4A3F D16F CABF 5E73      E5B9B3       E5B9B3       F3B180AC
95BE     95BE 95BE 95BE 95BE 50CD CAC0 CAC0 CAC0 CAC0 4A40 D17C CAC0 5F0A      E5BC8A       E5BC8A       F3B180AD
95BF     95BF 95BF 95BF 95BF 4A7F CAC1 CAC1 CAC1 CAC1 4A41 D1C1 CAC1 67C4      E69F84       E69F84       F3B180AE
95C0     95C0 95C0 95C0 95C0 487C CAC2 CAC2 CAC2 CAC2 4A42 D1C2 CAC2 4E26      E4B8A6       E4B8A6       F3B180AF
95C1     95C1 95C1 95C1 95C1 54D1 CAC3 CAC3 CAC3 CAC3 4A43 D1C3 CAC3 853D      E894BD       E894BD       F3B180B0
95C2     95C2 95C2 95C2 95C2 4B7D CAC4 CAC4 CAC4 CAC4 4A44 D1C4 CAC4 9589      E99689       E99689       F3B180B1
95C3     95C3 95C3 95C3 95C3 5157 CAC5 CAC5 CAC5 CAC5 4A45 D1C5 CAC5 965B      E9999B       E9999B       F3B180B2
95C4     95C4 95C4 95C4 95C4 466F CAC6 CAC6 CAC6 CAC6 4A46 D1C6 CAC6 7C73      E7B1B3       E7B1B3       F3B180B3
95C5     95C5 95C5 95C5 95C5 4CD9 CAC7 CAC7 CAC7 CAC7 4A47 D1C7 CAC7 9801      E9A081       E9A081       F3B180B4
95C6     95C6 95C6 95C6 95C6 5578 CAC8 CAC8 CAC8 CAC8 4A48 D1C8 CAC8 50FB      E583BB       E583BB       F3B180B5
95C7     95C7 95C7 95C7 95C7 4B95 CAC9 CAC9 CAC9 CAC9 4A49 D1C9 CAC9 58C1      E5A381       E5A381       F3B180B6
95C8     95C8 95C8 95C8 95C8 537E CACA CACA CACA CACA 4A4A D1D1 CACA 7656      E79996       E79996       F3B180B7
95C9     95C9 95C9 95C9 95C9 4DF8 CACB CACB CACB CACB 4A4B D1D2 CACB 78A7      E7A2A7       E7A2A7       F3B180B8
95CA     95CA 95CA 95CA 95CA 46DD CACC CACC CACC CACC 4A4C D1D3 CACC 5225      E588A5       E588A5       F3B180B9
95CB     95CB 95CB 95CB 95CB 5572 CACD CACD CACD CACD 4A4D D1D4 CACD 77A5      E79EA5       E79EA5       F3B180BA
95CC     95CC 95CC 95CC 95CC 556C CACE CACE CACE CACE 4A4E D1D5 CACE 8511      E89491       E89491       F3B180BB
95CD     95CD 95CD 95CD 95CD 559B CACF CACF CACF CACF 4A4F D1D6 CACF 7B86      E7AE86       E7AE86       F3B180BC
95CE     95CE 95CE 95CE 95CE 5049 CAD0 CAD0 CAD0 CAD0 4A50 D1D7 CAD0 504F      E5818F       E5818F       F3B180BD
95CF     95CF 95CF 95CF 95CF 476F CAD1 CAD1 CAD1 CAD1 4A51 D1D8 CAD1 5909      E5A489       E5A489       F3B180BE
95D0     95D0 95D0 95D0 95D0 4767 CAD2 CAD2 CAD2 CAD2 4A52 D1D9 CAD2 7247      E78987       E78987       F3B180BF
95D1     95D1 95D1 95D1 95D1 4FB6 CAD3 CAD3 CAD3 CAD3 4A53 D1E2 CAD3 7BC7      E7AF87       E7AF87       F3B18180
95D2     95D2 95D2 95D2 95D2 484E CAD4 CAD4 CAD4 CAD4 4A54 D1E3 CAD4 7DE8      E7B7A8       E7B7A8       F3B18181
95D3     95D3 95D3 95D3 95D3 45D5 CAD5 CAD5 CAD5 CAD5 4A55 D1E4 CAD5 8FBA      E8BEBA       E8BEBA       F3B18182
95D4     95D4 95D4 95D4 95D4 48D6 CAD6 CAD6 CAD6 CAD6 4A56 D1E5 CAD6 8FD4      E8BF94       E8BF94       F3B18183
95D5     95D5 95D5 95D5 95D5 505D CAD7 CAD7 CAD7 CAD7 4A57 D1E6 CAD7 904D      E9818D       E9818D       F3B18184
95D6     95D6 95D6 95D6 95D6 4B4C CAD8 CAD8 CAD8 CAD8 4A58 D1E7 CAD8 4FBF      E4BEBF       E4BEBF       F3B18185
95D7     95D7 95D7 95D7 95D7 48CE CAD9 CAD9 CAD9 CAD9 4A59 D1E8 CAD9 52C9      E58B89       E58B89       F3B18186
95D8     95D8 95D8 95D8 95D8 549F CADA CADA CADA CADA 4A5A D1E9 CADA 5A29      E5A8A9       E5A8A9       F3B18187
95D9     95D9 95D9 95D9 95D9 4B47 CADB CADB CADB CADB 4A5B D14A CADB 5F01      E5BC81       E5BC81       F3B18188
95DA     95DA 95DA 95DA 95DA 548B CADC CADC CADC CADC 4A5C D15B CADC 97AD      E99EAD       E99EAD       F3B18189
95DB     95DB 95DB 95DB 95DB 45AA CADD CADD CADD CADD 4A5D D15A CADD 4FDD      E4BF9D       E4BF9D       F3B1818A
95DC     95DC 95DC 95DC 95DC 4E5B CADE CADE CADE CADE 4A5E D15F CADE 8217      E88897       E88897       F3B1818B
95DD     95DD 95DD 95DD 95DD 5363 CADF CADF CADF CADF 4A5F D16D CADF 92EA      E98BAA       E98BAA       F3B1818C
95DE     95DE 95DE 95DE 95DE 53CF CAE0 CAE0 CAE0 CAE0 4A60 D179 CAE0 5703      E59C83       E59C83       F3B1818D
95DF     95DF 95DF 95DF 95DF 4C41 CAE1 CAE1 CAE1 CAE1 4A61 D157 CAE1 6355      E68D95       E68D95       F3B1818E
95E0     95E0 95E0 95E0 95E0 47D8 CAE2 CAE2 CAE2 CAE2 4A62 D159 CAE2 6B69      E6ADA9       E6ADA9       F3B1818F
95E1     95E1 95E1 95E1 95E1 4D58 CAE3 CAE3 CAE3 CAE3 4A63 D162 CAE3 752B      E794AB       E794AB       F3B18190
95E2     95E2 95E2 95E2 95E2 496A CAE4 CAE4 CAE4 CAE4 4A64 D163 CAE4 88DC      E8A39C       E8A39C       F3B18191
95E3     95E3 95E3 95E3 95E3 4A68 CAE5 CAE5 CAE5 CAE5 4A65 D164 CAE5 8F14      E8BC94       E8BC94       F3B18192
95E4     95E4 95E4 95E4 95E4 48D4 CAE6 CAE6 CAE6 CAE6 4A66 D165 CAE6 7A42      E7A982       E7A982       F3B18193
95E5     95E5 95E5 95E5 95E5 4AE1 CAE7 CAE7 CAE7 CAE7 4A67 D166 CAE7 52DF      E58B9F       E58B9F       F3B18194
95E6     95E6 95E6 95E6 95E6 4EBE CAE8 CAE8 CAE8 CAE8 4A68 D167 CAE8 5893      E5A293       E5A293       F3B18195
95E7     95E7 95E7 95E7 95E7 52E3 CAE9 CAE9 CAE9 CAE9 4A69 D168 CAE9 6155      E68595       E68595       F3B18196
95E8     95E8 95E8 95E8 95E8 5161 CAEA CAEA CAEA CAEA 4A6A D169 CAEA 620A      E6888A       E6888A       F3B18197
95E9     95E9 95E9 95E9 95E9 4AFE CAEB CAEB CAEB CAEB 4A6B D170 CAEB 66AE      E69AAE       E69AAE       F3B18198
95EA     95EA 95EA 95EA 95EA 4969 CAEC CAEC CAEC CAEC 4A6C D171 CAEC 6BCD      E6AF8D       E6AF8D       F3B18199
95EB     95EB 95EB 95EB 95EB 4F4C CAED CAED CAED CAED 4A6D D172 CAED 7C3F      E7B0BF       E7B0BF       F3B1819A
95EC     95EC 95EC 95EC 95EC 516A CAEE CAEE CAEE CAEE 4A6E D173 CAEE 83E9      E88FA9       E88FA9       F3B1819B
95ED     95ED 95ED 95ED 95ED 5569 CAEF CAEF CAEF CAEF 4A6F D174 CAEF 5023      E580A3       E580A3       F3B1819C
95EE     95EE 95EE 95EE 95EE 50F9 CAF0 CAF0 CAF0 CAF0 4A70 D175 CAF0 4FF8      E4BFB8       E4BFB8       F3B1819D
95EF     95EF 95EF 95EF 95EF 4AA8 CAF1 CAF1 CAF1 CAF1 4A71 D176 CAF1 5305      E58C85       E58C85       F3B1819E
95F0     95F0 95F0 95F0 95F0 5073 CAF2 CAF2 CAF2 CAF2 4A72 D177 CAF2 5446      E59186       E59186       F3B1819F
95F1     95F1 95F1 95F1 95F1 49B8 CAF3 CAF3 CAF3 CAF3 4A73 D178 CAF3 5831      E5A0B1       E5A0B1       F3B181A0
95F2     95F2 95F2 95F2 95F2 4DB2 CAF4 CAF4 CAF4 CAF4 4A74 D180 CAF4 5949      E5A589       E5A589       F3B181A1
95F3     95F3 95F3 95F3 95F3 494D CAF5 CAF5 CAF5 CAF5 4A75 D18B CAF5 5B9D      E5AE9D       E5AE9D       F3B181A2
95F4     95F4 95F4 95F4 95F4 48FE CAF6 CAF6 CAF6 CAF6 4A76 D19B CAF6 5CF0      E5B3B0       E5B3B0       F3B181A3
95F5     95F5 95F5 95F5 95F5 4AAC CAF7 CAF7 CAF7 CAF7 4A77 D19C CAF7 5CEF      E5B3AF       E5B3AF       F3B181A4
95F6     95F6 95F6 95F6 95F6 4F5F CAF8 CAF8 CAF8 CAF8 4A78 D1A0 CAF8 5D29      E5B4A9       E5B4A9       F3B181A5
95F7     95F7 95F7 95F7 95F7 554F CAF9 CAF9 CAF9 CAF9 4A79 D1AB CAF9 5E96      E5BA96       E5BA96       F3B181A6
95F8     95F8 95F8 95F8 95F8 4C86 CAFA CAFA CAFA CAFA 4A7A D1B0 CAFA 62B1      E68AB1       E68AB1       F3B181A7
95F9     95F9 95F9 95F9 95F9 5285 CAFB CAFB CAFB CAFB 4A7B D1C0 CAFB 6367      E68DA7       E68DA7       F3B181A8
95FA     95FA 95FA 95FA 95FA 48AF CAFC CAFC CAFC CAFC 4A7C D16A CAFC 653E      E694BE       E694BE       F3B181A9
95FB     95FB 95FB 95FB 95FB 45CD CAFD CAFD CAFD CAFD 4A7D D1D0 CAFD 65B9      E696B9       E696B9       F3B181AA
95FC     95FC 95FC 95FC 95FC 4BC4 CAFE CAFE CAFE CAFE 4A7E D1A1 CAFE 670B      E69C8B       E69C8B       F3B181AB
9640     9640 9640 9640 9640 464A CBA1 CBA1 CBA1 CBA1 4B21 D24F CBA1 6CD5      E6B395       E6B395       F3B181AC
9641     9641 9641 9641 9641 4E87 CBA2 CBA2 CBA2 CBA2 4B22 D27F CBA2 6CE1      E6B3A1       E6B3A1       F3B181AD
9642     9642 9642 9642 9642 5587 CBA3 CBA3 CBA3 CBA3 4B23 D27B CBA3 70F9      E783B9       E783B9       F3B181AE
9643     9643 9643 9643 9643 4CC2 CBA4 CBA4 CBA4 CBA4 4B24 D2E0 CBA4 7832      E7A0B2       E7A0B2       F3B181AF
9644     9644 9644 9644 9644 49E4 CBA5 CBA5 CBA5 CBA5 4B25 D26C CBA5 7E2B      E7B8AB       E7B8AB       F3B181B0
9645     9645 9645 9645 9645 50B6 CBA6 CBA6 CBA6 CBA6 4B26 D250 CBA6 80DE      E8839E       E8839E       F3B181B1
9646     9646 9646 9646 9646 4683 CBA7 CBA7 CBA7 CBA7 4B27 D27D CBA7 82B3      E88AB3       E88AB3       F3B181B2
9647     9647 9647 9647 9647 50DB CBA8 CBA8 CBA8 CBA8 4B28 D24D CBA8 840C      E8908C       E8908C       F3B181B3
9648     9648 9648 9648 9648 5088 CBA9 CBA9 CBA9 CBA9 4B29 D25D CBA9 84EC      E893AC       E893AC       F3B181B4
9649     9649 9649 9649 9649 4ECD CBAA CBAA CBAA CBAA 4B2A D25C CBAA 8702      E89C82       E89C82       F3B181B5
964A     964A 964A 964A 964A 51BA CBAB CBAB CBAB CBAB 4B2B D24E CBAB 8912      E8A492       E8A492       F3B181B6
964B     964B 964B 964B 964B 4A71 CBAC CBAC CBAC CBAC 4B2C D26B CBAC 8A2A      E8A8AA       E8A8AA       F3B181B7
964C     964C 964C 964C 964C 4648 CBAD CBAD CBAD CBAD 4B2D D260 CBAD 8C4A      E8B18A       E8B18A       F3B181B8
964D     964D 964D 964D 964D 46C5 CBAE CBAE CBAE CBAE 4B2E D24B CBAE 90A6      E982A6       E982A6       F3B181B9
964E     964E 964E 964E 964E 5449 CBAF CBAF CBAF CBAF 4B2F D261 CBAF 92D2      E98B92       E98B92       F3B181BA
964F     964F 964F 964F 964F 4D49 CBB0 CBB0 CBB0 CBB0 4B30 D2F0 CBB0 98FD      E9A3BD       E9A3BD       F3B181BB
9650     9650 9650 9650 9650 4EA6 CBB1 CBB1 CBB1 CBB1 4B31 D2F1 CBB1 9CF3      E9B3B3       E9B3B3       F3B181BC
9651     9651 9651 9651 9651 518A CBB2 CBB2 CBB2 CBB2 4B32 D2F2 CBB2 9D6C      E9B5AC       E9B5AC       F3B181BD
9652     9652 9652 9652 9652 4F6C CBB3 CBB3 CBB3 CBB3 4B33 D2F3 CBB3 4E4F      E4B98F       E4B98F       F3B181BE
9653     9653 9653 9653 9653 4CCE CBB4 CBB4 CBB4 CBB4 4B34 D2F4 CBB4 4EA1      E4BAA1       E4BAA1       F3B181BF
9654     9654 9654 9654 9654 4EE7 CBB5 CBB5 CBB5 CBB5 4B35 D2F5 CBB5 508D      E5828D       E5828D       F3B18280
9655     9655 9655 9655 9655 5199 CBB6 CBB6 CBB6 CBB6 4B36 D2F6 CBB6 5256      E58996       E58996       F3B18281
9656     9656 9656 9656 9656 4C55 CBB7 CBB7 CBB7 CBB7 4B37 D2F7 CBB7 574A      E59D8A       E59D8A       F3B18282
9657     9657 9657 9657 9657 50EC CBB8 CBB8 CBB8 CBB8 4B38 D2F8 CBB8 59A8      E5A6A8       E5A6A8       F3B18283
9658     9658 9658 9658 9658 4C85 CBB9 CBB9 CBB9 CBB9 4B39 D2F9 CBB9 5E3D      E5B8BD       E5B8BD       F3B18284
9659     9659 9659 9659 9659 4C45 CBBA CBBA CBBA CBBA 4B3A D27A CBBA 5FD8      E5BF98       E5BF98       F3B18285
965A     965A 965A 965A 965A 4EA3 CBBB CBBB CBBB CBBB 4B3B D25E CBBB 5FD9      E5BF99       E5BF99       F3B18286
965B     965B 965B 965B 965B 47DC CBBC CBBC CBBC CBBC 4B3C D24C CBBC 623F      E688BF       E688BF       F3B18287
965C     965C 965C 965C 965C 4CB2 CBBD CBBD CBBD CBBD 4B3D D27E CBBD 66B4      E69AB4       E69AB4       F3B18288
965D     965D 965D 965D 965D 47BB CBBE CBBE CBBE CBBE 4B3E D26E CBBE 671B      E69C9B       E69C9B       F3B18289
965E     965E 965E 965E 965E 504A CBBF CBBF CBBF CBBF 4B3F D26F CBBF 67D0      E69F90       E69F90       F3B1828A
965F     965F 965F 965F 965F 4B5A CBC0 CBC0 CBC0 CBC0 4B40 D27C CBC0 68D2      E6A392       E6A392       F3B1828B
9660     9660 9660 9660 9660 4F8F CBC1 CBC1 CBC1 CBC1 4B41 D2C1 CBC1 5192      E58692       E58692       F3B1828C
9661     9661 9661 9661 9661 4877 CBC2 CBC2 CBC2 CBC2 4B42 D2C2 CBC2 7D21      E7B4A1       E7B4A1       F3B1828D
9662     9662 9662 9662 9662 4CFB CBC3 CBC3 CBC3 CBC3 4B43 D2C3 CBC3 80AA      E882AA       E882AA       F3B1828E
9663     9663 9663 9663 9663 4EF6 CBC4 CBC4 CBC4 CBC4 4B44 D2C4 CBC4 81A8      E886A8       E886A8       F3B1828F
9664     9664 9664 9664 9664 4F4D CBC5 CBC5 CBC5 CBC5 4B45 D2C5 CBC5 8B00      E8AC80       E8AC80       F3B18290
9665     9665 9665 9665 9665 5064 CBC6 CBC6 CBC6 CBC6 4B46 D2C6 CBC6 8C8C      E8B28C       E8B28C       F3B18291
9666     9666 9666 9666 9666 4872 CBC7 CBC7 CBC7 CBC7 4B47 D2C7 CBC7 8CBF      E8B2BF       E8B2BF       F3B18292
9667     9667 9667 9667 9667 5183 CBC8 CBC8 CBC8 CBC8 4B48 D2C8 CBC8 927E      E989BE       E989BE       F3B18293
9668     9668 9668 9668 9668 47BC CBC9 CBC9 CBC9 CBC9 4B49 D2C9 CBC9 9632      E998B2       E998B2       F3B18294
9669     9669 9669 9669 9669 53E1 CBCA CBCA CBCA CBCA 4B4A D2D1 CBCA 5420      E590A0       E590A0       F3B18295
966A     966A 966A 966A 966A 4EB3 CBCB CBCB CBCB 6CFD D3F7 A8ED CBCB 982C      E9A0AC       E9A0AC       F3B18296
966B     966B 966B 966B 966B 455A CBCC CBCC CBCC CBCC 4B4C D2D3 CBCC 5317      E58C97       E58C97       F3B18297
966C     966C 966C 966C 966C 4ABB CBCD CBCD CBCD CBCD 4B4D D2D4 CBCD 50D5      E58395       E58395       F3B18298
966D     966D 966D 966D 966D 51C0 CBCE CBCE CBCE CBCE 4B4E D2D5 CBCE 535C      E58D9C       E58D9C       F3B18299
966E     966E 966E 966E 966E 4AFC CBCF CBCF CBCF CBCF 4B4F D2D6 CBCF 58A8      E5A2A8       E5A2A8       F3B1829A
966F     966F 966F 966F 966F 4EC7 CBD0 CBD0 CBD0 CBD0 4B50 D2D7 CBD0 64B2      E692B2       E692B2       F3B1829B
9670     9670 9670 9670 9670 4EA2 CBD1 CBD1 CBD1 CBD1 4B51 D2D8 CBD1 6734      E69CB4       E69CB4       F3B1829C
9671     9671 9671 9671 9671 487B CBD2 CBD2 CBD2 CBD2 4B52 D2D9 CBD2 7267      E789A7       E789A7       F3B1829D
9672     9672 9672 9672 9672 4A96 CBD3 CBD3 CBD3 CBD3 4B53 D2E2 CBD3 7766      E79DA6       E79DA6       F3B1829E
9673     9673 9673 9673 9673 52BE CBD4 CBD4 CBD4 CBD4 4B54 D2E3 CBD4 7A46      E7A986       E7A986       F3B1829F
9674     9674 9674 9674 9674 534A CBD5 CBD5 CBD5 CBD5 4B55 D2E4 CBD5 91E6      E987A6       E987A6       F3B182A0
9675     9675 9675 9675 9675 54B1 CBD6 CBD6 CBD6 CBD6 4B56 D2E5 CBD6 52C3      E58B83       E58B83       F3B182A1
9676     9676 9676 9676 9676 5075 CBD7 CBD7 CBD7 CBD7 4B57 D2E6 CBD7 6CA1      E6B2A1       E6B2A1       F3B182A2
9677     9677 9677 9677 9677 4F42 CBD8 CBD8 CBD8 CBD8 4B58 D2E7 CBD8 6B86      E6AE86       E6AE86       F3B182A3
9678     9678 9678 9678 9678 469D CBD9 CBD9 CBD9 CBD9 4B59 D2E8 CBD9 5800      E5A080       E5A080       F3B182A4
9679     9679 9679 9679 9679 496B CBDA CBDA CBDA CBDA 4B5A D2E9 CBDA 5E4C      E5B98C       E5B98C       F3B182A5
967A     967A 967A 967A 967A 5284 CBDB CBDB CBDB CBDB 4B5B D24A CBDB 5954      E5A594       E5A594       F3B182A6
967B     967B 967B 967B 967B 4566 CBDC CBDC CBDC CBDC 4B5C D25B CBDC 672C      E69CAC       E69CAC       F3B182A7
967C     967C 967C 967C 967C 51F6 CBDD CBDD CBDD CBDD 4B5D D25A CBDD 7FFB      E7BFBB       E7BFBB       F3B182A8
967D     967D 967D 967D 967D 4DE1 CBDE CBDE CBDE CBDE 4B5E D25F CBDE 51E1      E587A1       E587A1       F3B182A9
967E     967E 967E 967E 967E 5057 CBDF CBDF CBDF CBDF 4B5F D26D CBDF 76C6      E79B86       E79B86       F3B182AA
9680     9680 9680 9680 9680 48D1 CBE0 CBE0 CBE0 CBE0 4B60 D279 CBE0 6469      E691A9       E691A9       F3B182AB
9681     9681 9681 9681 9681 49B4 CBE1 CBE1 CBE1 CBE1 4B61 D257 CBE1 78E8      E7A3A8       E7A3A8       F3B182AC
9682     9682 9682 9682 9682 4DE2 CBE2 CBE2 CBE2 CBE2 4B62 D259 CBE2 9B54      E9AD94       E9AD94       F3B182AD
9683     9683 9683 9683 9683 48A7 CBE3 CBE3 CBE3 CBE3 4B63 D262 CBE3 9EBB      E9BABB       E9BABB       F3B182AE
9684     9684 9684 9684 9684 4EAC CBE4 CBE4 CBE4 CBE4 4B64 D263 CBE4 57CB      E59F8B       E59F8B       F3B182AF
9685     9685 9685 9685 9685 4BDC CBE5 CBE5 CBE5 CBE5 4B65 D264 CBE5 59B9      E5A6B9       E5A6B9       F3B182B0
9686     9686 9686 9686 9686 559D CBE6 CBE6 CBE6 CBE6 4B66 D265 CBE6 6627      E698A7       E698A7       F3B182B1
9687     9687 9687 9687 9687 49D9 CBE7 CBE7 CBE7 CBE7 4B67 D266 CBE7 679A      E69E9A       E69E9A       F3B182B2
9688     9688 9688 9688 9688 49F5 CBE8 CBE8 CBE8 CBE8 4B68 D267 CBE8 6BCE      E6AF8E       E6AF8E       F3B182B3
9689     9689 9689 9689 9689 55F0 CBE9 CBE9 CBE9 CBE9 4B69 D268 CBE9 54E9      E593A9       E593A9       F3B182B4
968A     968A 968A 968A 968A 6881 CBEA CBEA CBEA 54C4 BFCE 90A3 CBEA 69D9      E6A799       E6A799       F3B182B5
968B     968B 968B 968B 968B 4CCA CBEB CBEB CBEB CBEB 4B6B D270 CBEB 5E55      E5B995       E5B995       F3B182B6
968C     968C 968C 968C 968C 4D8A CBEC CBEC CBEC CBEC 4B6C D271 CBEC 819C      E8869C       E8869C       F3B182B7
968D     968D 968D 968D 968D 4F78 CBED CBED CBED CBED 4B6D D272 CBED 6795      E69E95       E69E95       F3B182B8
968E     968E 968E 968E 968E 55CE CBEE CBEE CBEE CBEE 4B6E D273 CBEE 9BAA      E9AEAA       E9AEAA       F3B182B9
968F     968F 968F 968F 968F 5054 CBEF CBEF CBEF CBEF 4B6F D274 CBEF 67FE      E69FBE       E69FBE       F3B182BA
9690     9690 9690 9690 9690 53C6 CBF0 CBF0 CBF0 CBF0 4B70 D275 CBF0 9C52      E9B192       E9B192       F3B182BB
9691     9691 9691 9691 9691 4D88 CBF1 CBF1 CBF1 CBF1 4B71 D276 CBF1 685D      E6A19D       E6A19D       F3B182BC
9692     9692 9692 9692 9692 4FFA CBF2 CBF2 CBF2 CBF2 4B72 D277 CBF2 4EA6      E4BAA6       E4BAA6       F3B182BD
9693     9693 9693 9693 9693 4B4A CBF3 CBF3 CBF3 CBF3 4B73 D278 CBF3 4FE3      E4BFA3       E4BFA3       F3B182BE
9694     9694 9694 9694 9694 48F5 CBF4 CBF4 CBF4 CBF4 4B74 D280 CBF4 53C8      E58F88       E58F88       F3B182BF
9695     9695 9695 9695 9695 51E0 CBF5 CBF5 CBF5 CBF5 4B75 D28B CBF5 62B9      E68AB9       E68AB9       F3B18380
9696     9696 9696 9696 9696 4746 CBF6 CBF6 CBF6 CBF6 4B76 D29B CBF6 672B      E69CAB       E69CAB       F3B18381
9697     9697 9697 9697 9697 53EC CBF7 CBF7 CBF7 CBF7 4B77 D29C CBF7 6CAB      E6B2AB       E6B2AB       F3B18382
9698     9698 9698 9698 9698 49CD CBF8 CBF8 CBF8 CBF8 4B78 D2A0 CBF8 8FC4      E8BF84       E8BF84       F3B18383
9699     9699 9699 9699 9699 5675 D0D6 CBF9 CBF9 D0D6 5056 D7E5 CBF9 4FAD      E4BEAD       E4BEAD       F3B18384
969A     969A 969A 969A 969A 4D9A CBFA CBFA CBFA CBFA 4B7A D2B0 CBFA 7E6D      E7B9AD       E7B9AD       F3B18385
969B     969B 969B 969B 969B 4EEA CBFB CBFB CBFB CBFB 4B7B D2C0 CBFB 9EBF      E9BABF       E9BABF       F3B18386
969C     969C 969C 969C 969C 454D CBFC CBFC CBFC CBFC 4B7C D26A CBFC 4E07      E4B887       E4B887       F3B18387
969D     969D 969D 969D 969D 4DDB CBFD CBFD CBFD CBFD 4B7D D2D0 CBFD 6162      E685A2       E685A2       F3B18388
969E     969E 969E 969E 969E 475B CBFE CBFE CBFE CBFE 4B7E D2A1 CBFE 6E80      E6BA80       E6BA80       F3B18389
969F     969F 969F 969F 969F 4F6E CCA1 CCA1 CCA1 CCA1 4C21 D34F CCA1 6F2B      E6BCAB       E6BCAB       F3B1838A
96A0     96A0 96A0 96A0 96A0 557B CCA2 CCA2 CCA2 CCA2 4C22 D37F CCA2 8513      E89493       E89493       F3B1838B
96A1     96A1 96A1 96A1 96A1 477F CCA3 CCA3 CCA3 CCA3 4C23 D37B CCA3 5473      E591B3       E591B3       F3B1838C
96A2     96A2 96A2 96A2 96A2 49C6 CCA4 CCA4 CCA4 CCA4 4C24 D3E0 CCA4 672A      E69CAA       E69CAA       F3B1838D
96A3     96A3 96A3 96A3 96A3 4DB4 CCA5 CCA5 CCA5 CCA5 4C25 D36C CCA5 9B45      E9AD85       E9AD85       F3B1838E
96A4     96A4 96A4 96A4 96A4 4AE6 CCA6 CCA6 CCA6 CCA6 4C26 D350 CCA6 5DF3      E5B7B3       E5B7B3       F3B1838F
96A5     96A5 96A5 96A5 96A5 4CCC CCA7 CCA7 CCA7 CCA7 4C27 D37D CCA7 7B95      E7AE95       E7AE95       F3B18390
96A6     96A6 96A6 96A6 96A6 50D5 CCA8 CCA8 CCA8 CCA8 4C28 D34D CCA8 5CAC      E5B2AC       E5B2AC       F3B18391
96A7     96A7 96A7 96A7 96A7 4A56 CCA9 CCA9 CCA9 CCA9 4C29 D35D CCA9 5BC6      E5AF86       E5AF86       F3B18392
96A8     96A8 96A8 96A8 96A8 53A1 CCAA CCAA CCAA CCAA 4C2A D35C CCAA 871C      E89C9C       E89C9C       F3B18393
96A9     96A9 96A9 96A9 96A9 4C88 CCAB CCAB CCAB CCAB 4C2B D34E CCAB 6E4A      E6B98A       E6B98A       F3B18394
96AA     96AA 96AA 96AA 96AA 4F9A CCAC CCAC CCAC CCAC 4C2C D36B CCAC 84D1      E89391       E89391       F3B18395
96AB     96AB 96AB 96AB 96AB 47FA CCAD CCAD CCAD CCAD 4C2D D360 CCAD 7A14      E7A894       E7A894       F3B18396
96AC     96AC 96AC 96AC 96AC 5079 CCAE CCAE CCAE CCAE 4C2E D34B CCAE 8108      E88488       E88488       F3B18397
96AD     96AD 96AD 96AD 96AD 4A66 CCAF CCAF CCAF CCAF 4C2F D361 CCAF 5999      E5A699       E5A699       F3B18398
96AE     96AE 96AE 96AE 96AE 53C9 CCB0 CCB0 CCB0 CCB0 4C30 D3F0 CCB0 7C8D      E7B28D       E7B28D       F3B18399
96AF     96AF 96AF 96AF 96AF 4756 CCB1 CCB1 CCB1 CCB1 4C31 D3F1 CCB1 6C11      E6B091       E6B091       F3B1839A
96B0     96B0 96B0 96B0 96B0 4E6A CCB2 CCB2 CCB2 CCB2 4C32 D3F2 CCB2 7720      E79CA0       E79CA0       F3B1839B
96B1     96B1 96B1 96B1 96B1 46ED CCB3 CCB3 CCB3 CCB3 4C33 D3F3 CCB3 52D9      E58B99       E58B99       F3B1839C
96B2     96B2 96B2 96B2 96B2 4B8B CCB4 CCB4 CCB4 CCB4 4C34 D3F4 CCB4 5922      E5A4A2       E5A4A2       F3B1839D
96B3     96B3 96B3 96B3 96B3 475F CCB5 CCB5 CCB5 CCB5 4C35 D3F5 CCB5 7121      E784A1       E784A1       F3B1839E
96B4     96B4 96B4 96B4 96B4 4AAF CCB6 CCB6 CCB6 CCB6 4C36 D3F6 CCB6 725F      E7899F       E7899F       F3B1839F
96B5     96B5 96B5 96B5 96B5 50ED CCB7 CCB7 CCB7 CCB7 4C37 D3F7 CCB7 77DB      E79F9B       E79F9B       F3B183A0
96B6     96B6 96B6 96B6 96B6 4E48 CCB8 CCB8 CCB8 CCB8 4C38 D3F8 CCB8 9727      E99CA7       E99CA7       F3B183A1
96B7     96B7 96B7 96B7 96B7 557E CCB9 CCB9 CCB9 CCB9 4C39 D3F9 CCB9 9D61      E9B5A1       E9B5A1       F3B183A2
96B8     96B8 96B8 96B8 96B8 4DFD CCBA CCBA CCBA CCBA 4C3A D37A CCBA 690B      E6A48B       E6A48B       F3B183A3
96B9     96B9 96B9 96B9 96B9 5454 CCBB CCBB CCBB CCBB 4C3B D35E CCBB 5A7F      E5A9BF       E5A9BF       F3B183A4
96BA     96BA 96BA 96BA 96BA 4B54 CCBC CCBC CCBC CCBC 4C3C D34C CCBC 5A18      E5A898       E5A898       F3B183A5
96BB     96BB 96BB 96BB 96BB 5467 CCBD CCBD CCBD CCBD 4C3D D37E CCBD 51A5      E586A5       E586A5       F3B183A6
96BC     96BC 96BC 96BC 96BC 45F3 CCBE CCBE CCBE CCBE 4C3E D36E CCBE 540D      E5908D       E5908D       F3B183A7
96BD     96BD 96BD 96BD 96BD 49B0 CCBF CCBF CCBF CCBF 4C3F D36F CCBF 547D      E591BD       E591BD       F3B183A8
96BE     96BE 96BE 96BE 96BE 45A6 CCC0 CCC0 CCC0 CCC0 4C40 D37C CCC0 660E      E6988E       E6988E       F3B183A9
96BF     96BF 96BF 96BF 96BF 4B84 CCC1 CCC1 CCC1 CCC1 4C41 D3C1 CCC1 76DF      E79B9F       E79B9F       F3B183AA
96C0     96C0 96C0 96C0 96C0 4E58 CCC2 CCC2 CCC2 CCC2 4C42 D3C2 CCC2 8FF7      E8BFB7       E8BFB7       F3B183AB
96C1     96C1 96C1 96C1 96C1 4DDC CCC3 CCC3 CCC3 CCC3 4C43 D3C3 CCC3 9298      E98A98       E98A98       F3B183AC
96C2     96C2 96C2 96C2 96C2 4B70 CCC4 CCC4 CCC4 CCC4 4C44 D3C4 CCC4 9CF4      E9B3B4       E9B3B4       F3B183AD
96C3     96C3 96C3 96C3 96C3 558C CCC5 CCC5 CCC5 CCC5 4C45 D3C5 CCC5 59EA      E5A7AA       E5A7AA       F3B183AE
96C4     96C4 96C4 96C4 96C4 55B7 CCC6 CCC6 CCC6 CCC6 4C46 D3C6 CCC6 725D      E7899D       E7899D       F3B183AF
96C5     96C5 96C5 96C5 96C5 4EE6 CCC7 CCC7 CCC7 CCC7 4C47 D3C7 CCC7 6EC5      E6BB85       E6BB85       F3B183B0
96C6     96C6 96C6 96C6 96C6 4B56 CCC8 CCC8 CCC8 CCC8 4C48 D3C8 CCC8 514D      E5858D       E5858D       F3B183B1
96C7     96C7 96C7 96C7 96C7 55CA CCC9 CCC9 CCC9 CCC9 4C49 D3C9 CCC9 68C9      E6A389       E6A389       F3B183B2
96C8     96C8 96C8 96C8 96C8 489D CCCA CCCA CCCA CCCA 4C4A D3D1 CCCA 7DBF      E7B6BF       E7B6BF       F3B183B3
96C9     96C9 96C9 96C9 96C9 556B CCCB CCCB CCCB CCCB 4C4B D3D2 CCCB 7DEC      E7B7AC       E7B7AC       F3B183B4
96CA     96CA 96CA 96CA 96CA 4691 CCCC CCCC CCCC CCCC 4C4C D3D3 CCCC 9762      E99DA2       E99DA2       F3B183B5
96CB     96CB 96CB 96CB 96CB 555F CCCD CCCD CCCD 6FC4 D7A7 AD47 CCCD 9EBA      E9BABA       E9BABA       F3B183B6
96CC     96CC 96CC 96CC 96CC 55F5 CCCE CCCE CCCE CCCE 4C4E D3D5 CCCE 6478      E691B8       E691B8       F3B183B7
96CD     96CD 96CD 96CD 96CD 4864 CCCF CCCF CCCF CCCF 4C4F D3D6 CCCF 6A21      E6A8A1       E6A8A1       F3B183B8
96CE     96CE 96CE 96CE 96CE 4646 CCD0 CCD0 CCD0 CCD0 4C50 D3D7 CCD0 8302      E88C82       E88C82       F3B183B9
96CF     96CF 96CF 96CF 96CF 554D CCD1 CCD1 CCD1 CCD1 4C51 D3D8 CCD1 5984      E5A684       E5A684       F3B183BA
96D0     96D0 96D0 96D0 96D0 4F83 CCD2 CCD2 CCD2 CCD2 4C52 D3D9 CCD2 5B5F      E5AD9F       E5AD9F       F3B183BB
96D1     96D1 96D1 96D1 96D1 46E2 CCD3 CCD3 CCD3 CCD3 4C53 D3E2 CCD3 6BDB      E6AF9B       E6AF9B       F3B183BC
96D2     96D2 96D2 96D2 96D2 4A51 CCD4 CCD4 CCD4 CCD4 4C54 D3E3 CCD4 731B      E78C9B       E78C9B       F3B183BD
96D3     96D3 96D3 96D3 96D3 5080 CCD5 CCD5 CCD5 CCD5 4C55 D3E4 CCD5 76F2      E79BB2       E79BB2       F3B183BE
96D4     96D4 96D4 96D4 96D4 4A53 CCD6 CCD6 CCD6 CCD6 4C56 D3E5 CCD6 7DB2      E7B6B2       E7B6B2       F3B183BF
96D5     96D5 96D5 96D5 96D5 4FEC CCD7 CCD7 CCD7 CCD7 4C57 D3E6 CCD7 8017      E88097       E88097       F3B18480
96D6     96D6 96D6 96D6 96D6 5095 CCD8 CCD8 CCD8 CCD8 4C58 D3E7 CCD8 8499      E89299       E89299       F3B18481
96D7     96D7 96D7 96D7 96D7 54BD CCD9 CCD9 CCD9 CCD9 4C59 D3E8 CCD9 5132      E584B2       E584B2       F3B18482
96D8     96D8 96D8 96D8 96D8 456C CCDA CCDA CCDA CCDA 4C5A D3E9 CCDA 6728      E69CA8       E69CA8       F3B18483
96D9     96D9 96D9 96D9 96D9 4DBC CCDB CCDB CCDB CCDB 4C5B D34A CCDB 9ED9      E9BB99       E9BB99       F3B18484
96DA     96DA 96DA 96DA 96DA 4675 CCDC CCDC CCDC CCDC 4C5C D35B CCDC 76EE      E79BAE       E79BAE       F3B18485
96DB     96DB 96DB 96DB 96DB 51A6 CCDD CCDD CCDD CCDD 4C5D D35A CCDD 6762      E69DA2       E69DA2       F3B18486
96DC     96DC 96DC 96DC 96DC 5297 CCDE CCDE CCDE CCDE 4C5E D35F CCDE 52FF      E58BBF       E58BBF       F3B18487
96DD     96DD 96DD 96DD 96DD 50BF CCDF CCDF CCDF CCDF 4C5F D36D CCDF 9905      E9A485       E9A485       F3B18488
96DE     96DE 96DE 96DE 96DE 53AE CCE0 CCE0 CCE0 CCE0 4C60 D379 CCE0 5C24      E5B0A4       E5B0A4       F3B18489
96DF     96DF 96DF 96DF 96DF 4C44 CCE1 CCE1 CCE1 CCE1 4C61 D357 CCE1 623B      E688BB       E688BB       F3B1848A
96E0     96E0 96E0 96E0 96E0 4FF0 CCE2 CCE2 CCE2 CCE2 4C62 D359 CCE2 7C7E      E7B1BE       E7B1BE       F3B1848B
96E1     96E1 96E1 96E1 96E1 4DEF CCE3 CCE3 CCE3 CCE3 4C63 D362 CCE3 8CB0      E8B2B0       E8B2B0       F3B1848C
96E2     96E2 96E2 96E2 96E2 46E4 CCE4 CCE4 CCE4 CCE4 4C64 D363 CCE4 554F      E5958F       E5958F       F3B1848D
96E3     96E3 96E3 96E3 96E3 5554 CCE5 CCE5 CCE5 CCE5 4C65 D364 CCE5 60B6      E682B6       E682B6       F3B1848E
96E4     96E4 96E4 96E4 96E4 4CF9 CCE6 CCE6 CCE6 CCE6 4C66 D365 CCE6 7D0B      E7B48B       E7B48B       F3B1848F
96E5     96E5 96E5 96E5 96E5 46AA CCE7 CCE7 CCE7 CCE7 4C67 D366 CCE7 9580      E99680       E99680       F3B18490
96E6     96E6 96E6 96E6 96E6 4F5B CCE8 CCE8 CCE8 CCE8 4C68 D367 CCE8 5301      E58C81       E58C81       F3B18491
96E7     96E7 96E7 96E7 96E7 4749 CCE9 CCE9 CCE9 CCE9 4C69 D368 CCE9 4E5F      E4B99F       E4B99F       F3B18492
96E8     96E8 96E8 96E8 96E8 5146 CCEA CCEA CCEA CCEA 4C6A D369 CCEA 51B6      E586B6       E586B6       F3B18493
96E9     96E9 96E9 96E9 96E9 48C8 CCEB CCEB CCEB CCEB 4C6B D370 CCEB 591C      E5A49C       E5A49C       F3B18494
96EA     96EA 96EA 96EA 96EA 5096 CCEC CCEC CCEC CCEC 4C6C D371 CCEC 723A      E788BA       E788BA       F3B18495
96EB     96EB 96EB 96EB 96EB 4E99 CCED CCED CCED CCED 4C6D D372 CCED 8036      E880B6       E880B6       F3B18496
96EC     96EC 96EC 96EC 96EC 4569 CCEE CCEE CCEE CCEE 4C6E D373 CCEE 91CE      E9878E       E9878E       F3B18497
96ED     96ED 96ED 96ED 96ED 47E6 CCEF CCEF CCEF CCEF 4C6F D374 CCEF 5F25      E5BCA5       E5BCA5       F3B18498
96EE     96EE 96EE 96EE 96EE 46C1 CCF0 CCF0 CCF0 CCF0 4C70 D375 CCF0 77E2      E79FA2       E79FA2       F3B18499
96EF     96EF 96EF 96EF 96EF 5352 CCF1 CCF1 CCF1 CCF1 4C71 D376 CCF1 5384      E58E84       E58E84       F3B1849A
96F0     96F0 96F0 96F0 96F0 495D CCF2 CCF2 CCF2 CCF2 4C72 D377 CCF2 5F79      E5BDB9       E5BDB9       F3B1849B
96F1     96F1 96F1 96F1 96F1 47BD CCF3 CCF3 CCF3 CCF3 4C73 D378 CCF3 7D04      E7B484       E7B484       F3B1849C
96F2     96F2 96F2 96F2 96F2 46E1 CCF4 CCF4 CCF4 CCF4 4C74 D380 CCF4 85AC      E896AC       E896AC       F3B1849D
96F3     96F3 96F3 96F3 96F3 4BAD CCF5 CCF5 CCF5 CCF5 4C75 D38B CCF5 8A33      E8A8B3       E8A8B3       F3B1849E
96F4     96F4 96F4 96F4 96F4 4CA1 CCF6 CCF6 CCF6 CCF6 4C76 D39B CCF6 8E8D      E8BA8D       E8BA8D       F3B1849F
96F5     96F5 96F5 96F5 96F5 48C3 CCF7 CCF7 CCF7 CCF7 4C77 D39C CCF7 9756      E99D96       E99D96       F3B184A0
96F6     96F6 96F6 96F6 96F6 46EA CCF8 CCF8 CCF8 CCF8 4C78 D3A0 CCF8 67F3      E69FB3       E69FB3       F3B184A1
96F7     96F7 96F7 96F7 96F7 62B3 E9AE CCF9 CCF9 E9AE 692E 684B CCF9 85AE      E896AE       E896AE       F3B184A2
96F8     96F8 96F8 96F8 96F8 5241 CCFA CCFA CCFA CCFA 4C7A D3B0 CCFA 9453      E99193       E99193       F3B184A3
96F9     96F9 96F9 96F9 96F9 5098 CCFB CCFB CCFB CCFB 4C7B D3C0 CCFB 6109      E68489       E68489       F3B184A4
96FA     96FA 96FA 96FA 96FA 53C8 CCFC CCFC CCFC CCFC 4C7C D36A CCFC 6108      E68488       E68488       F3B184A5
96FB     96FB 96FB 96FB 96FB 465C CCFD CCFD CCFD CCFD 4C7D D3D0 CCFD 6CB9      E6B2B9       E6B2B9       F3B184A6
96FC     96FC 96FC 96FC 96FC 53AA CCFE CCFE CCFE CCFE 4C7E D3A1 CCFE 7652      E79992       E79992       F3B184A7
9740     9740 9740 9740 9740 4E9C CDA1 CDA1 CDA1 CDA1 4D21 D44F CDA1 8AED      E8ABAD       E8ABAD       F3B184A8
9741     9741 9741 9741 9741 4693 CDA2 CDA2 CDA2 CDA2 4D22 D47F CDA2 8F38      E8BCB8       E8BCB8       F3B184A9
9742     9742 9742 9742 9742 4BE0 CDA3 CDA3 CDA3 CDA3 4D23 D47B CDA3 552F      E594AF       E594AF       F3B184AA
9743     9743 9743 9743 9743 4CE6 CDA4 CDA4 CDA4 CDA4 4D24 D4E0 CDA4 4F51      E4BD91       E4BD91       F3B184AB
9744     9744 9744 9744 9744 47DA CDA5 CDA5 CDA5 CDA5 4D25 D46C CDA5 512A      E584AA       E584AA       F3B184AC
9745     9745 9745 9745 9745 469C CDA6 CDA6 CDA6 CDA6 4D26 D450 CDA6 52C7      E58B87       E58B87       F3B184AD
9746     9746 9746 9746 9746 46D4 CDA7 CDA7 CDA7 CDA7 4D27 D47D CDA7 53CB      E58F8B       E58F8B       F3B184AE
9747     9747 9747 9747 9747 4F7A CDA8 CDA8 CDA8 CDA8 4D28 D44D CDA8 5BA5      E5AEA5       E5AEA5       F3B184AF
9748     9748 9748 9748 9748 525F CDA9 CDA9 CDA9 CDA9 4D29 D45D CDA9 5E7D      E5B9BD       E5B9BD       F3B184B0
9749     9749 9749 9749 9749 4DD8 CDAA CDAA CDAA CDAA 4D2A D45C CDAA 60A0      E682A0       E682A0       F3B184B1
974A     974A 974A 974A 974A 507C CDAB CDAB CDAB CDAB 4D2B D44E CDAB 6182      E68682       E68682       F3B184B2
974B     974B 974B 974B 974B 4CF2 CDAC CDAC CDAC CDAC 4D2C D46B CDAC 63D6      E68F96       E68F96       F3B184B3
974C     974C 974C 974C 974C 4685 CDAD CDAD CDAD CDAD 4D2D D460 CDAD 6709      E69C89       E69C89       F3B184B4
974D     974D 974D 974D 974D 4EE1 CDAE CDAE CDAE CDAE 4D2E D44B CDAE 67DA      E69F9A       E69F9A       F3B184B5
974E     974E 974E 974E 974E 4FBD CDAF CDAF CDAF CDAF 4D2F D461 CDAF 6E67      E6B9A7       E6B9A7       F3B184B6
974F     974F 974F 974F 974F 4F7D CDB0 CDB0 CDB0 CDB0 4D30 D4F0 CDB0 6D8C      E6B68C       E6B68C       F3B184B7
9750     9750 9750 9750 9750 504F CDB1 CDB1 CDB1 CDB1 4D31 D4F1 CDB1 7336      E78CB6       E78CB6       F3B184B8
9751     9751 9751 9751 9751 53DD CDB2 CDB2 CDB2 CDB2 4D32 D4F2 CDB2 7337      E78CB7       E78CB7       F3B184B9
9752     9752 9752 9752 9752 467C CDB3 CDB3 CDB3 CDB3 4D33 D4F3 CDB3 7531      E794B1       E794B1       F3B184BA
9753     9753 9753 9753 9753 488C CDB4 CDB4 CDB4 CDB4 4D34 D4F4 CDB4 7950      E7A590       E7A590       F3B184BB
9754     9754 9754 9754 9754 46F8 CDB5 CDB5 CDB5 CDB5 4D35 D4F5 CDB5 88D5      E8A395       E8A395       F3B184BC
9755     9755 9755 9755 9755 4ACB CDB6 CDB6 CDB6 CDB6 4D36 D4F6 CDB6 8A98      E8AA98       E8AA98       F3B184BD
9756     9756 9756 9756 9756 4A8B CDB7 CDB7 CDB7 CDB7 4D37 D4F7 CDB7 904A      E9818A       E9818A       F3B184BE
9757     9757 9757 9757 9757 4C92 CDB8 CDB8 CDB8 CDB8 4D38 D4F8 CDB8 9091      E98291       E98291       F3B184BF
9758     9758 9758 9758 9758 4CB7 CDB9 CDB9 CDB9 CDB9 4D39 D4F9 CDB9 90F5      E983B5       E983B5       F3B18580
9759     9759 9759 9759 9759 4570 CDBA CDBA CDBA CDBA 4D3A D47A CDBA 96C4      E99B84       E99B84       F3B18581
975A     975A 975A 975A 975A 48C2 CDBB CDBB CDBB CDBB 4D3B D45E CDBB 878D      E89E8D       E89E8D       F3B18582
975B     975B 975B 975B 975B 4AC3 CDBC CDBC CDBC CDBC 4D3C D44C CDBC 5915      E5A495       E5A495       F3B18583
975C     975C 975C 975C 975C 47C2 CDBD CDBD CDBD CDBD 4D3D D47E CDBD 4E88      E4BA88       E4BA88       F3B18584
975D     975D 975D 975D 975D 49C1 CDBE CDBE CDBE CDBE 4D3E D46E CDBE 4F59      E4BD99       E4BD99       F3B18585
975E     975E 975E 975E 975E 4789 CDBF CDBF CDBF CDBF 4D3F D46F CDBF 4E0E      E4B88E       E4B88E       F3B18586
975F     975F 975F 975F 975F 4D8C CDC0 CDC0 CDC0 CDC0 4D40 D47C CDC0 8A89      E8AA89       E8AA89       F3B18587
9760     9760 9760 9760 9760 506C CDC1 CDC1 CDC1 CDC1 4D41 D4C1 CDC1 8F3F      E8BCBF       E8BCBF       F3B18588
9761     9761 9761 9761 9761 4D4A CDC2 CDC2 CDC2 CDC2 4D42 D4C2 CDC2 9810      E9A090       E9A090       F3B18589
9762     9762 9762 9762 9762 54DA CDC3 CDC3 CDC3 CDC3 4D43 D4C3 CDC3 50AD      E582AD       E582AD       F3B1858A
9763     9763 9763 9763 9763 4DAF CDC4 CDC4 CDC4 CDC4 4D44 D4C4 CDC4 5E7C      E5B9BC       E5B9BC       F3B1858B
9764     9764 9764 9764 9764 53D1 CDC5 CDC5 CDC5 CDC5 4D45 D4C5 CDC5 5996      E5A696       E5A696       F3B1858C
9765     9765 9765 9765 9765 4889 CDC6 CDC6 CDC6 CDC6 4D46 D4C6 CDC6 5BB9      E5AEB9       E5AEB9       F3B1858D
9766     9766 9766 9766 9766 4C5D CDC7 CDC7 CDC7 CDC7 4D47 D4C7 CDC7 5EB8      E5BAB8       E5BAB8       F3B1858E
9767     9767 9767 9767 9767 4C46 CDC8 CDC8 CDC8 CDC8 4D48 D4C8 CDC8 63DA      E68F9A       E68F9A       F3B1858F
9768     9768 9768 9768 9768 4F7E CDC9 CDC9 CDC9 CDC9 4D49 D4C9 CDC9 63FA      E68FBA       E68FBA       F3B18590
9769     9769 9769 9769 9769 514E CDCA CDCA CDCA CDCA 4D4A D4D1 CDCA 64C1      E69381       E69381       F3B18591
976A     976A 976A 976A 976A 4BFE CDCB CDCB CDCB CDCB 4D4B D4D2 CDCB 66DC      E69B9C       E69B9C       F3B18592
976B     976B 976B 976B 976B 51D7 CDCC CDCC CDCC CDCC 4D4C D4D3 CDCC 694A      E6A58A       E6A58A       F3B18593
976C     976C 976C 976C 976C 4848 CDCD CDCD CDCD CDCD 4D4D D4D4 CDCD 69D8      E6A798       E6A798       F3B18594
976D     976D 976D 976D 976D 45DA CDCE CDCE CDCE CDCE 4D4E D4D5 CDCE 6D0B      E6B48B       E6B48B       F3B18595
976E     976E 976E 976E 976E 48BE CDCF CDCF CDCF CDCF 4D4F D4D6 CDCF 6EB6      E6BAB6       E6BAB6       F3B18596
976F     976F 976F 976F 976F 529D CDD0 CDD0 CDD0 CDD0 4D50 D4D7 CDD0 7194      E78694       E78694       F3B18597
9770     9770 9770 9770 9770 45B6 CDD1 CDD1 CDD1 CDD1 4D51 D4D8 CDD1 7528      E794A8       E794A8       F3B18598
9771     9771 9771 9771 9771 49A7 CDD2 CDD2 CDD2 CDD2 4D52 D4D9 CDD2 7AAF      E7AAAF       E7AAAF       F3B18599
9772     9772 9772 9772 9772 4AD1 CDD3 CDD3 CDD3 CDD3 4D53 D4E2 CDD3 7F8A      E7BE8A       E7BE8A       F3B1859A
9773     9773 9773 9773 9773 52A0 CDD4 CDD4 CDD4 CDD4 4D54 D4E3 CDD4 8000      E88080       E88080       F3B1859B
9774     9774 9774 9774 9774 45F9 CDD5 CDD5 CDD5 CDD5 4D55 D4E4 CDD5 8449      E89189       E89189       F3B1859C
9775     9775 9775 9775 9775 5083 CDD6 CDD6 CDD6 CDD6 4D56 D4E5 CDD6 84C9      E89389       E89389       F3B1859D
9776     9776 9776 9776 9776 4655 CDD7 CDD7 CDD7 CDD7 4D57 D4E6 CDD7 8981      E8A681       E8A681       F3B1859E
9777     9777 9777 9777 9777 4D7D CDD8 CDD8 CDD8 CDD8 4D58 D4E7 CDD8 8B21      E8ACA1       E8ACA1       F3B1859F
9778     9778 9778 9778 9778 4CA8 CDD9 CDD9 CDD9 CDD9 4D59 D4E8 CDD9 8E0A      E8B88A       E8B88A       F3B185A0
9779     9779 9779 9779 9779 6882 CDDA CDDA CDDA 70EB D0C4 A595 CDDA 9065      E981A5       E981A5       F3B185A1
977A     977A 977A 977A 977A 47BF CDDB CDDB CDDB CDDB 4D5B D44A CDDB 967D      E999BD       E999BD       F3B185A2
977B     977B 977B 977B 977B 4979 CDDC CDDC CDDC CDDC 4D5C D45B CDDC 990A      E9A48A       E9A48A       F3B185A3
977C     977C 977C 977C 977C 53F4 CDDD CDDD CDDD CDDD 4D5D D45A CDDD 617E      E685BE       E685BE       F3B185A4
977D     977D 977D 977D 977D 4D66 CDDE CDDE CDDE CDDE 4D5E D45F CDDE 6291      E68A91       E68A91       F3B185A5
977E     977E 977E 977E 977E 4BF6 CDDF CDDF CDDF CDDF 4D5F D46D CDDF 6B32      E6ACB2       E6ACB2       F3B185A6
9780     9780 9780 9780 9780 55D9 CDE0 CDE0 CDE0 CDE0 4D60 D479 CDE0 6C83      E6B283       E6B283       F3B185A7
9781     9781 9781 9781 9781 4B7E CDE1 CDE1 CDE1 CDE1 4D61 D457 CDE1 6D74      E6B5B4       E6B5B4       F3B185A8
9782     9782 9782 9782 9782 4D9C CDE2 CDE2 CDE2 CDE2 4D62 D459 CDE2 7FCC      E7BF8C       E7BF8C       F3B185A9
9783     9783 9783 9783 9783 4DB5 CDE3 CDE3 CDE3 CDE3 4D63 D462 CDE3 7FFC      E7BFBC       E7BFBC       F3B185AA
9784     9784 9784 9784 9784 4A88 CDE4 CDE4 CDE4 CDE4 4D64 D463 CDE4 6DC0      E6B780       E6B780       F3B185AB
9785     9785 9785 9785 9785 4DFA CDE5 CDE5 CDE5 CDE5 4D65 D464 CDE5 7F85      E7BE85       E7BE85       F3B185AC
9786     9786 9786 9786 9786 51D6 CDE6 CDE6 CDE6 CDE6 4D66 D465 CDE6 87BA      E89EBA       E89EBA       F3B185AD
9787     9787 9787 9787 9787 4F43 CDE7 CDE7 CDE7 CDE7 4D67 D466 CDE7 88F8      E8A3B8       E8A3B8       F3B185AE
9788     9788 9788 9788 9788 4676 CDE8 CDE8 CDE8 CDE8 4D68 D467 CDE8 6765      E69DA5       E69DA5       F3B185AF
9789     9789 9789 9789 9789 52C9 CDE9 CDE9 CDE9 63A2 CBE3 9EB5 CDE9 83B1      E88EB1       E88EB1       F3B185B0
978A     978A 978A 978A 978A 4A7A CDEA CDEA CDEA CDEA 4D6A D469 CDEA 983C      E9A0BC       E9A0BC       F3B185B1
978B     978B 978B 978B 978B 4BD6 CDEB CDEB CDEB CDEB 4D6B D470 CDEB 96F7      E99BB7       E99BB7       F3B185B2
978C     978C 978C 978C 978C 5450 CDEC CDEC CDEC CDEC 4D6C D471 CDEC 6D1B      E6B49B       E6B49B       F3B185B3
978D     978D 978D 978D 978D 4CE7 CDED CDED CDED CDED 4D6D D472 CDED 7D61      E7B5A1       E7B5A1       F3B185B4
978E     978E 978E 978E 978E 47F2 CDEE CDEE CDEE CDEE 4D6E D473 CDEE 843D      E890BD       E890BD       F3B185B5
978F     978F 978F 978F 978F 50BD CDEF CDEF CDEF CDEF 4D6F D474 CDEF 916A      E985AA       E985AA       F3B185B6
9790     9790 9790 9790 9790 4BBF CDF0 CDF0 CDF0 CDF0 4D70 D475 CDF0 4E71      E4B9B1       E4B9B1       F3B185B7
9791     9791 9791 9791 9791 4CF5 CDF1 CDF1 CDF1 CDF1 4D71 D476 CDF1 5375      E58DB5       E58DB5       F3B185B8
9792     9792 9792 9792 9792 498C CDF2 CDF2 CDF2 CDF2 4D72 D477 CDF2 5D50      E5B590       E5B590       F3B185B9
9793     9793 9793 9793 9793 4EDE CDF3 CDF3 CDF3 CDF3 4D73 D478 CDF3 6B04      E6AC84       E6AC84       F3B185BA
9794     9794 9794 9794 9794 55A3 CDF4 CDF4 CDF4 CDF4 4D74 D480 CDF4 6FEB      E6BFAB       E6BFAB       F3B185BB
9795     9795 9795 9795 9795 4EED CDF5 CDF5 CDF5 CDF5 4D75 D48B CDF5 85CD      E8978D       E8978D       F3B185BC
9796     9796 9796 9796 9796 4CA6 CDF6 CDF6 CDF6 CDF6 4D76 D49B CDF6 862D      E898AD       E898AD       F3B185BD
9797     9797 9797 9797 9797 4ED0 CDF7 CDF7 CDF7 CDF7 4D77 D49C CDF7 89A7      E8A6A7       E8A6A7       F3B185BE
9798     9798 9798 9798 9798 45BC CDF8 CDF8 CDF8 CDF8 4D78 D4A0 CDF8 5229      E588A9       E588A9       F3B185BF
9799     9799 9799 9799 9799 50C6 CDF9 CDF9 CDF9 CDF9 4D79 D4AB CDF9 540F      E5908F       E5908F       F3B18680
979A     979A 979A 979A 979A 4CA2 CDFA CDFA CDFA CDFA 4D7A D4B0 CDFA 5C65      E5B1A5       E5B1A5       F3B18681
979B     979B 979B 979B 979B 4E51 CDFB CDFB CDFB CDFB 4D7B D4C0 CDFB 674E      E69D8E       E69D8E       F3B18682
979C     979C 979C 979C 979C 4858 CDFC CDFC CDFC CDFC 4D7C D46A CDFC 68A8      E6A2A8       E6A2A8       F3B18683
979D     979D 979D 979D 979D 45EB CDFD CDFD CDFD CDFD 4D7D D4D0 CDFD 7406      E79086       E79086       F3B18684
979E     979E 979E 979E 979E 528C CDFE CDFE CDFE CDFE 4D7E D4A1 CDFE 7483      E79283       E79283       F3B18685
979F     979F 979F 979F 979F 5174 CEA1 CEA1 CEA1 CEA1 4E21 D54F CEA1 75E2      E797A2       E797A2       F3B18686
97A0     97A0 97A0 97A0 97A0 49A4 CEA2 CEA2 CEA2 CEA2 4E22 D57F CEA2 88CF      E8A38F       E8A38F       F3B18687
97A1     97A1 97A1 97A1 97A1 544D CEA3 CEA3 CEA3 CEA3 4E23 D57B CEA3 88E1      E8A3A1       E8A3A1       F3B18688
97A2     97A2 97A2 97A2 97A2 476C CEA4 CEA4 CEA4 CEA4 4E24 D5E0 CEA4 91CC      E9878C       E9878C       F3B18689
97A3     97A3 97A3 97A3 97A3 4997 CEA5 CEA5 CEA5 CEA5 4E25 D56C CEA5 96E2      E99BA2       E99BA2       F3B1868A
97A4     97A4 97A4 97A4 97A4 4961 CEA6 CEA6 CEA6 CEA6 4E26 D550 CEA6 9678      E999B8       E999B8       F3B1868B
97A5     97A5 97A5 97A5 97A5 4991 CEA7 CEA7 CEA7 CEA7 4E27 D57D CEA7 5F8B      E5BE8B       E5BE8B       F3B1868C
97A6     97A6 97A6 97A6 97A6 4783 CEA8 CEA8 CEA8 CEA8 4E28 D54D CEA8 7387      E78E87       E78E87       F3B1868D
97A7     97A7 97A7 97A7 97A7 45DD CEA9 CEA9 CEA9 CEA9 4E29 D55D CEA9 7ACB      E7AB8B       E7AB8B       F3B1868E
97A8     97A8 97A8 97A8 97A8 559F CEAA CEAA CEAA CEAA 4E2A D55C CEAA 844E      E8918E       E8918E       F3B1868F
97A9     97A9 97A9 97A9 97A9 55E6 CEAB CEAB CEAB CEAB 4E2B D54E CEAB 63A0      E68EA0       E68EA0       F3B18690
97AA     97AA 97AA 97AA 97AA 4CF0 CEAC CEAC CEAC CEAC 4E2C D56B CEAC 7565      E795A5       E795A5       F3B18691
97AB     97AB 97AB 97AB 97AB 5185 CEAD CEAD CEAD CEAD 4E2D D560 CEAD 5289      E58A89       E58A89       F3B18692
97AC     97AC 97AC 97AC 97AC 46FC CEAE CEAE CEAE CEAE 4E2E D54B CEAE 6D41      E6B581       E6B581       F3B18693
97AD     97AD 97AD 97AD 97AD 4DC4 CEAF CEAF CEAF CEAF 4E2F D561 CEAF 6E9C      E6BA9C       E6BA9C       F3B18694
97AE     97AE 97AE 97AE 97AE 50F7 CEB0 CEB0 CEB0 CEB0 4E30 D5F0 CEB0 7409      E79089       E79089       F3B18695
97AF     97AF 97AF 97AF 97AF 47AB CEB1 CEB1 CEB1 CEB1 4E31 D5F1 CEB1 7559      E79599       E79599       F3B18696
97B0     97B0 97B0 97B0 97B0 4984 CEB2 CEB2 CEB2 CEB2 4E32 D5F2 CEB2 786B      E7A1AB       E7A1AB       F3B18697
97B1     97B1 97B1 97B1 97B1 4CA9 CEB3 CEB3 CEB3 CEB3 4E33 D5F3 CEB3 7C92      E7B292       E7B292       F3B18698
97B2     97B2 97B2 97B2 97B2 465D CEB4 CEB4 CEB4 CEB4 4E34 D5F4 CEB4 9686      E99A86       E99A86       F3B18699
97B3     97B3 97B3 97B3 97B3 49E7 CEB5 CEB5 CEB5 CEB5 4E35 D5F5 CEB5 7ADC      E7AB9C       E7AB9C       F3B1869A
97B4     97B4 97B4 97B4 97B4 48CA CEB6 CEB6 CEB6 CEB6 4E36 D5F6 CEB6 9F8D      E9BE8D       E9BE8D       F3B1869B
97B5     97B5 97B5 97B5 97B5 557C CEB7 CEB7 CEB7 CEB7 4E37 D5F7 CEB7 4FB6      E4BEB6       E4BEB6       F3B1869C
97B6     97B6 97B6 97B6 97B6 4C49 CEB8 CEB8 CEB8 CEB8 4E38 D5F8 CEB8 616E      E685AE       E685AE       F3B1869D
97B7     97B7 97B7 97B7 97B7 4ACF CEB9 CEB9 CEB9 CEB9 4E39 D5F9 CEB9 65C5      E69785       E69785       F3B1869E
97B8     97B8 97B8 97B8 97B8 5250 CEBA CEBA CEBA CEBA 4E3A D57A CEBA 865C      E8999C       E8999C       F3B1869F
97B9     97B9 97B9 97B9 97B9 4BB1 CEBB CEBB CEBB CEBB 4E3B D55E CEBB 4E86      E4BA86       E4BA86       F3B186A0
97BA     97BA 97BA 97BA 97BA 499C CEBC CEBC CEBC CEBC 4E3C D54C CEBC 4EAE      E4BAAE       E4BAAE       F3B186A1
97BB     97BB 97BB 97BB 97BB 4D82 CEBD CEBD CEBD CEBD 4E3D D57E CEBD 50DA      E5839A       E5839A       F3B186A2
97BC     97BC 97BC 97BC 97BC 4883 CEBE CEBE CEBE CEBE 4E3E D56E CEBE 4E21      E4B8A1       E4B8A1       F3B186A3
97BD     97BD 97BD 97BD 97BD 5344 CEBF CEBF CEBF CEBF 4E3F D56F CEBF 51CC      E5878C       E5878C       F3B186A4
97BE     97BE 97BE 97BE 97BE 4C7A CEC0 CEC0 CEC0 CEC0 4E40 D57C CEC0 5BEE      E5AFAE       E5AFAE       F3B186A5
97BF     97BF 97BF 97BF 97BF 464E CEC1 CEC1 CEC1 CEC1 4E41 D5C1 CEC1 6599      E69699       E69699       F3B186A6
97C0     97C0 97C0 97C0 97C0 4E44 CEC2 CEC2 CEC2 CEC2 4E42 D5C2 CEC2 6881      E6A281       E6A281       F3B186A7
97C1     97C1 97C1 97C1 97C1 4E74 CEC3 CEC3 CEC3 CEC3 4E43 D5C3 CEC3 6DBC      E6B6BC       E6B6BC       F3B186A8
97C2     97C2 97C2 97C2 97C2 4E9D CEC4 CEC4 CEC4 CEC4 4E44 D5C4 CEC4 731F      E78C9F       E78C9F       F3B186A9
97C3     97C3 97C3 97C3 97C3 4BB7 CEC5 CEC5 CEC5 CEC5 4E45 D5C5 CEC5 7642      E79982       E79982       F3B186AA
97C4     97C4 97C4 97C4 97C4 527D CEC6 CEC6 CEC6 CEC6 4E46 D5C6 CEC6 77AD      E79EAD       E79EAD       F3B186AB
97C5     97C5 97C5 97C5 97C5 5296 CEC7 CEC7 CEC7 CEC7 4E47 D5C7 CEC7 7A1C      E7A89C       E7A89C       F3B186AC
97C6     97C6 97C6 97C6 97C6 4EC5 CEC8 CEC8 CEC8 CEC8 4E48 D5C8 CEC8 7CE7      E7B3A7       E7B3A7       F3B186AD
97C7     97C7 97C7 97C7 97C7 45C4 CEC9 CEC9 CEC9 CEC9 4E49 D5C9 CEC9 826F      E889AF       E889AF       F3B186AE
97C8     97C8 97C8 97C8 97C8 5181 CECA CECA CECA CECA 4E4A D5D1 CECA 8AD2      E8AB92       E8AB92       F3B186AF
97C9     97C9 97C9 97C9 97C9 526B CECB CECB CECB CECB 4E4B D5D2 CECB 907C      E981BC       E981BC       F3B186B0
97CA     97CA 97CA 97CA 97CA 47B5 CECC CECC CECC CECC 4E4C D5D3 CECC 91CF      E9878F       E9878F       F3B186B1
97CB     97CB 97CB 97CB 97CB 50C7 CECD CECD CECD CECD 4E4D D5D4 CECD 9675      E999B5       E999B5       F3B186B2
97CC     97CC 97CC 97CC 97CC 4ACD CECE CECE CECE CECE 4E4E D5D5 CECE 9818      E9A098       E9A098       F3B186B3
97CD     97CD 97CD 97CD 97CD 45F7 CECF CECF CECF CECF 4E4F D5D6 CECF 529B      E58A9B       E58A9B       F3B186B4
97CE     97CE 97CE 97CE 97CE 48A5 CED0 CED0 CED0 CED0 4E50 D5D7 CED0 7DD1      E7B791       E7B791       F3B186B5
97CF     97CF 97CF 97CF 97CF 4D51 CED1 CED1 CED1 CED1 4E51 D5D8 CED1 502B      E580AB       E580AB       F3B186B6
97D0     97D0 97D0 97D0 97D0 4F55 CED2 CED2 CED2 CED2 4E52 D5D9 CED2 5398      E58E98       E58E98       F3B186B7
97D1     97D1 97D1 97D1 97D1 45A2 CED3 CED3 CED3 CED3 4E53 D5E2 CED3 6797      E69E97       E69E97       F3B186B8
97D2     97D2 97D2 97D2 97D2 52C7 CED4 CED4 CED4 CED4 4E54 D5E3 CED4 6DCB      E6B78B       E6B78B       F3B186B9
97D3     97D3 97D3 97D3 97D3 4A79 CED5 CED5 CED5 CED5 4E55 D5E4 CED5 71D0      E78790       E78790       F3B186BA
97D4     97D4 97D4 97D4 97D4 5453 CED6 CED6 CED6 CED6 4E56 D5E5 CED6 7433      E790B3       E790B3       F3B186BB
97D5     97D5 97D5 97D5 97D5 4CD0 CED7 CED7 CED7 CED7 4E57 D5E6 CED7 81E8      E887A8       E887A8       F3B186BC
97D6     97D6 97D6 97D6 97D6 48B7 CED8 CED8 CED8 CED8 4E58 D5E7 CED8 8F2A      E8BCAA       E8BCAA       F3B186BD
97D7     97D7 97D7 97D7 97D7 4CE0 CED9 CED9 CED9 CED9 4E59 D5E8 CED9 96A3      E99AA3       E99AA3       F3B186BE
97D8     97D8 97D8 97D8 97D8 554C CEDA CEDA CEDA CEDA 4E5A D5E9 CEDA 9C57      E9B197       E9B197       F3B186BF
97D9     97D9 97D9 97D9 97D9 519B CEDB CEDB CEDB CEDB 4E5B D54A CEDB 9E9F      E9BA9F       E9BA9F       F3B18780
97DA     97DA 97DA 97DA 97DA 508C CEDC CEDC CEDC CEDC 4E5C D55B CEDC 7460      E791A0       E791A0       F3B18781
97DB     97DB 97DB 97DB 97DB 4BEB CEDD CEDD CEDD CEDD 4E5D D55A CEDD 5841      E5A181       E5A181       F3B18782
97DC     97DC 97DC 97DC 97DC 4DDA CEDE CEDE CEDE CEDE 4E5E D55F CEDE 6D99      E6B699       E6B699       F3B18783
97DD     97DD 97DD 97DD 97DD 5148 CEDF CEDF CEDF CEDF 4E5F D56D CEDF 7D2F      E7B4AF       E7B4AF       F3B18784
97DE     97DE 97DE 97DE 97DE 46BA CEE0 CEE0 CEE0 CEE0 4E60 D579 CEE0 985E      E9A19E       E9A19E       F3B18785
97DF     97DF 97DF 97DF 97DF 49C3 CEE1 CEE1 CEE1 CEE1 4E61 D557 CEE1 4EE4      E4BBA4       E4BBA4       F3B18786
97E0     97E0 97E0 97E0 97E0 544A CEE2 CEE2 CEE2 CEE2 4E62 D559 CEE2 4F36      E4BCB6       E4BCB6       F3B18787
97E1     97E1 97E1 97E1 97E1 4A46 CEE3 CEE3 CEE3 CEE3 4E63 D562 CEE3 4F8B      E4BE8B       E4BE8B       F3B18788
97E2     97E2 97E2 97E2 97E2 48FC CEE4 CEE4 CEE4 CEE4 4E64 D563 CEE4 51B7      E586B7       E586B7       F3B18789
97E3     97E3 97E3 97E3 97E3 4D92 CEE5 CEE5 CEE5 CEE5 4E65 D564 CEE5 52B1      E58AB1       E58AB1       F3B1878A
97E4     97E4 97E4 97E4 97E4 4CE8 CEE6 CEE6 CEE6 CEE6 4E66 D565 CEE6 5DBA      E5B6BA       E5B6BA       F3B1878B
97E5     97E5 97E5 97E5 97E5 509F CEE7 CEE7 CEE7 CEE7 4E67 D566 CEE7 601C      E6809C       E6809C       F3B1878C
97E6     97E6 97E6 97E6 97E6 4AF9 CEE8 CEE8 CEE8 CEE8 4E68 D567 CEE8 73B2      E78EB2       E78EB2       F3B1878D
97E7     97E7 97E7 97E7 97E7 4958 CEE9 CEE9 CEE9 CEE9 4E69 D568 CEE9 793C      E7A4BC       E7A4BC       F3B1878E
97E8     97E8 97E8 97E8 97E8 5559 CEEA CEEA CEEA CEEA 4E6A D569 CEEA 82D3      E88B93       E88B93       F3B1878F
97E9     97E9 97E9 97E9 97E9 45AE CEEB CEEB CEEB CEEB 4E6B D570 CEEB 9234      E988B4       E988B4       F3B18790
97EA     97EA 97EA 97EA 97EA 5545 CEEC CEEC CEEC CEEC 4E6C D571 CEEC 96B7      E99AB7       E99AB7       F3B18791
97EB     97EB 97EB 97EB 97EB 5069 CEED CEED CEED CEED 4E6D D572 CEED 96F6      E99BB6       E99BB6       F3B18792
97EC     97EC 97EC 97EC 97EC 4F4B CEEE CEEE CEEE CEEE 4E6E D573 CEEE 970A      E99C8A       E99C8A       F3B18793
97ED     97ED 97ED 97ED 97ED 4CF3 CEEF CEEF CEEF CEEF 4E6F D574 CEEF 9E97      E9BA97       E9BA97       F3B18794
97EE     97EE 97EE 97EE 97EE 4C80 CEF0 CEF0 CEF0 CEF0 4E70 D575 CEF0 9F62      E9BDA2       E9BDA2       F3B18795
97EF     97EF 97EF 97EF 97EF 50CC CEF1 CEF1 CEF1 CEF1 4E71 D576 CEF1 66A6      E69AA6       E69AA6       F3B18796
97F0     97F0 97F0 97F0 97F0 4955 CEF2 CEF2 CEF2 CEF2 4E72 D577 CEF2 6B74      E6ADB4       E6ADB4       F3B18797
97F1     97F1 97F1 97F1 97F1 4BB4 CEF3 CEF3 CEF3 CEF3 4E73 D578 CEF3 5217      E58897       E58897       F3B18798
97F2     97F2 97F2 97F2 97F2 4E68 CEF4 CEF4 CEF4 CEF4 4E74 D580 CEF4 52A3      E58AA3       E58AA3       F3B18799
97F3     97F3 97F3 97F3 97F3 4DD1 CEF5 CEF5 CEF5 CEF5 4E75 D58B CEF5 70C8      E78388       E78388       F3B1879A
97F4     97F4 97F4 97F4 97F4 4F60 CEF6 CEF6 CEF6 CEF6 4E76 D59B CEF6 88C2      E8A382       E8A382       F3B1879B
97F5     97F5 97F5 97F5 97F5 4EDB CEF7 CEF7 CEF7 CEF7 4E77 D59C CEF7 5EC9      E5BB89       E5BB89       F3B1879C
97F6     97F6 97F6 97F6 97F6 4BCA CEF8 CEF8 CEF8 CEF8 4E78 D5A0 CEF8 604B      E6818B       E6818B       F3B1879D
97F7     97F7 97F7 97F7 97F7 555B CEF9 CEF9 CEF9 CEF9 4E79 D5AB CEF9 6190      E68690       E68690       F3B1879E
97F8     97F8 97F8 97F8 97F8 54E0 CEFA CEFA CEFA CEFA 4E7A D5B0 CEFA 6F23      E6BCA3       E6BCA3       F3B1879F
97F9     97F9 97F9 97F9 97F9 4C6F CEFB CEFB CEFB CEFB 4E7B D5C0 CEFB 7149      E78589       E78589       F3B187A0
97FA     97FA 97FA 97FA 97FA 5270 CEFC CEFC CEFC CEFC 4E7C D56A CEFC 7C3E      E7B0BE       E7B0BE       F3B187A1
97FB     97FB 97FB 97FB 97FB 4860 CEFD CEFD CEFD CEFD 4E7D D5D0 CEFD 7DF4      E7B7B4       E7B7B4       F3B187A2
97FC     97FC 97FC 97FC 97FC 53CA CEFE CEFE CEFE CEFE 4E7E D5A1 CEFE 806F      E881AF       E881AF       F3B187A3
9840     9840 9840 9840 9840 4BF4 CFA1 CFA1 CFA1 CFA1 4F21 D64F CFA1 84EE      E893AE       E893AE       F3B187A4
9841     9841 9841 9841 9841 4754 CFA2 CFA2 CFA2 CFA2 4F22 D67F CFA2 9023      E980A3       E980A3       F3B187A5
9842     9842 9842 9842 9842 4FE6 CFA3 CFA3 CFA3 CFA3 4F23 D67B CFA3 932C      E98CAC       E98CAC       F3B187A6
9843     9843 9843 9843 9843 4B50 CFA4 CFA4 CFA4 CFA4 4F24 D6E0 CFA4 5442      E59182       E59182       F3B187A7
9844     9844 9844 9844 9844 51F4 CFA5 CFA5 CFA5 CFA5 4F25 D66C CFA5 9B6F      E9ADAF       E9ADAF       F3B187A8
9845     9845 9845 9845 9845 5586 CFA6 CFA6 CFA6 CFA6 4F26 D650 CFA6 6AD3      E6AB93       E6AB93       F3B187A9
9846     9846 9846 9846 9846 497D CFA7 CFA7 CFA7 CFA7 4F27 D67D CFA7 7089      E78289       E78289       F3B187AA
9847     9847 9847 9847 9847 55F8 CFA8 CFA8 CFA8 CFA8 4F28 D64D CFA8 8CC2      E8B382       E8B382       F3B187AB
9848     9848 9848 9848 9848 4784 CFA9 CFA9 CFA9 CFA9 4F29 D65D CFA9 8DEF      E8B7AF       E8B7AF       F3B187AC
9849     9849 9849 9849 9849 4BC6 CFAA CFAA CFAA CFAA 4F2A D65C CFAA 9732      E99CB2       E99CB2       F3B187AD
984A     984A 984A 984A 984A 48A9 CFAB CFAB CFAB CFAB 4F2B D64E CFAB 52B4      E58AB4       E58AB4       F3B187AE
984B     984B 984B 984B 984B 50F8 CFAC CFAC CFAC CFAC 4F2C D66B CFAC 5A41      E5A981       E5A981       F3B187AF
984C     984C 984C 984C 984C 4EA7 CFAD CFAD CFAD CFAD 4F2D D660 CFAD 5ECA      E5BB8A       E5BB8A       F3B187B0
984D     984D 984D 984D 984D 54BF CFAE CFAE CFAE CFAE 4F2E D64B CFAE 5F04      E5BC84       E5BC84       F3B187B1
984E     984E 984E 984E 984E 486E CFAF CFAF CFAF CFAF 4F2F D661 CFAF 6717      E69C97       E69C97       F3B187B2
984F     984F 984F 984F 984F 52BA CFB0 CFB0 CFB0 CFB0 4F30 D6F0 CFB0 697C      E6A5BC       E6A5BC       F3B187B3
9850     9850 9850 9850 9850 55C7 CFB1 CFB1 CFB1 CFB1 4F31 D6F1 CFB1 6994      E6A694       E6A694       F3B187B4
9851     9851 9851 9851 9851 4A9D CFB2 CFB2 CFB2 CFB2 4F32 D6F2 CFB2 6D6A      E6B5AA       E6B5AA       F3B187B5
9852     9852 9852 9852 9852 50C4 CFB3 CFB3 CFB3 CFB3 4F33 D6F3 CFB3 6F0F      E6BC8F       E6BC8F       F3B187B6
9853     9853 9853 9853 9853 55C4 CFB4 CFB4 CFB4 CFB4 4F34 D6F4 CFB4 7262      E789A2       E789A2       F3B187B7
9854     9854 9854 9854 9854 506F CFB5 CFB5 CFB5 CFB5 4F35 D6F5 CFB5 72FC      E78BBC       E78BBC       F3B187B8
9855     9855 9855 9855 9855 5444 E4C6 CFB6 CFB6 E4C6 6446 63C6 CFB6 7BED      E7AFAD       E7AFAD       F3B187B9
9856     9856 9856 9856 9856 4985 CFB7 CFB7 CFB7 CFB7 4F37 D6F7 CFB7 8001      E88081       E88081       F3B187BA
9857     9857 9857 9857 9857 53E4 CFB8 CFB8 CFB8 CFB8 4F38 D6F8 CFB8 807E      E881BE       E881BE       F3B187BB
9858     9858 9858 9858 9858 52A1 CFB9 CFB9 CFB9 63B5 D7B3 AD83 CFB9 874B      E89D8B       E89D8B       F3B187BC
9859     9859 9859 9859 9859 456E CFBA CFBA CFBA CFBA 4F3A D67A CFBA 90CE      E9838E       E9838E       F3B187BD
985A     985A 985A 985A 985A 4546 CFBB CFBB CFBB CFBB 4F3B D65E CFBB 516D      E585AD       E585AD       F3B187BE
985B     985B 985B 985B 985B 519E CFBC CFBC CFBC CFBC 4F3C D64C CFBC 9E93      E9BA93       E9BA93       F3B187BF
985C     985C 985C 985C 985C 50AE CFBD CFBD CFBD CFBD 4F3D D67E CFBD 7984      E7A684       E7A684       F3B18880
985D     985D 985D 985D 985D 557F CFBE CFBE CFBE CFBE 4F3E D66E CFBE 808B      E8828B       E8828B       F3B18881
985E     985E 985E 985E 985E 4886 CFBF CFBF CFBF CFBF 4F3F D66F CFBF 9332      E98CB2       E98CB2       F3B18882
985F     985F 985F 985F 985F 48F3 CFC0 CFC0 CFC0 CFC0 4F40 D67C CFC0 8AD6      E8AB96       E8AB96       F3B18883
9860     9860 9860 9860 9860 4FF9 CFC1 CFC1 CFC1 CFC1 4F41 D6C1 CFC1 502D      E580AD       E580AD       F3B18884
9861     9861 9861 9861 9861 457A CFC2 CFC2 CFC2 CFC2 4F42 D6C2 CFC2 548C      E5928C       E5928C       F3B18885
9862     9862 9862 9862 9862 4750 CFC3 CFC3 CFC3 CFC3 4F43 D6C3 CFC3 8A71      E8A9B1       E8A9B1       F3B18886
9863     9863 9863 9863 9863 558E CFC4 CFC4 CFC4 CFC4 4F44 D6C4 CFC4 6B6A      E6ADAA       E6ADAA       F3B18887
9864     9864 9864 9864 9864 5067 CFC5 CFC5 CFC5 CFC5 4F45 D6C5 CFC5 8CC4      E8B384       E8B384       F3B18888
9865     9865 9865 9865 9865 47E3 CFC6 CFC6 CFC6 CFC6 4F46 D6C6 CFC6 8107      E88487       E88487       F3B18889
9866     9866 9866 9866 9866 4E78 CFC7 CFC7 CFC7 CFC7 4F47 D6C7 CFC7 60D1      E68391       E68391       F3B1888A
9867     9867 9867 9867 9867 4FD2 CFC8 CFC8 CFC8 CFC8 4F48 D6C8 CFC8 67A0      E69EA0       E69EA0       F3B1888B
9868     9868 9868 9868 9868 4BF1 CFC9 CFC9 CFC9 CFC9 4F49 D6C9 CFC9 9DF2      E9B7B2       E9B7B2       F3B1888C
9869     9869 9869 9869 9869 50E8 CFCA CFCA CFCA CFCA 4F4A D6D1 CFCA 4E99      E4BA99       E4BA99       F3B1888D
986A     986A 986A 986A 986A 4BE4 CFCB CFCB CFCB CFCB 4F4B D6D2 CFCB 4E98      E4BA98       E4BA98       F3B1888E
986B     986B 986B 986B 986B 5293 CFCC CFCC CFCC CFCC 4F4C D6D3 CFCC 9C10      E9B090       E9B090       F3B1888F
986C     986C 986C 986C 986C 50D3 CFCD CFCD CFCD CFCD 4F4D D6D4 CFCD 8A6B      E8A9AB       E8A9AB       F3B18890
986D     986D 986D 986D 986D 4FBB CFCE CFCE CFCE CFCE 4F4E D6D5 CFCE 85C1      E89781       E89781       F3B18891
986E     986E 986E 986E 986E 50DE CFCF CFCF CFCF CFCF 4F4F D6D6 CFCF 8568      E895A8       E895A8       F3B18892
986F     986F 986F 986F 986F 5468 CFD0 CFD0 CFD0 CFD0 4F50 D6D7 CFD0 6900      E6A480       E6A480       F3B18893
9870     9870 9870 9870 9870 4D6B CFD1 CFD1 CFD1 CFD1 4F51 D6D8 CFD1 6E7E      E6B9BE       E6B9BE       F3B18894
9871     9871 9871 9871 9871 545B CFD2 CFD2 CFD2 CFD2 4F52 D6D9 CFD2 7897      E7A297       E7A297       F3B18895
9872     9872 9872 9872 9872 4CAF CFD3 CFD3 CFD3 CFD3 4F53 D6E2 CFD3 8155      E88595       E88595       F3B18896
989F     989F 989F 989F 989F 5641 D0A1 D0A1 D0A1 D0A1 5021 D74F D0A1 5F0C      E5BC8C       E5BC8C       F3B18982
98A0     98A0 98A0 98A0 98A0 5642 D0A2 D0A2 D0A2 D0A2 5022 D77F D0A2 4E10      E4B890       E4B890       F3B18983
98A1     98A1 98A1 98A1 98A1 5643 D0A3 D0A3 D0A3 D0A3 5023 D77B D0A3 4E15      E4B895       E4B895       F3B18984
98A2     98A2 98A2 98A2 98A2 5645 D0A4 D0A4 D0A4 D0A4 5024 D7E0 D0A4 4E2A      E4B8AA       E4B8AA       F3B18985
98A3     98A3 98A3 98A3 98A3 5646 D0A5 D0A5 D0A5 D0A5 5025 D76C D0A5 4E31      E4B8B1       E4B8B1       F3B18986
98A4     98A4 98A4 98A4 98A4 5647 D0A6 D0A6 D0A6 D0A6 5026 D750 D0A6 4E36      E4B8B6       E4B8B6       F3B18987
98A5     98A5 98A5 98A5 98A5 54C8 D0A7 D0A7 D0A7 D0A7 5027 D77D D0A7 4E3C      E4B8BC       E4B8BC       F3B18988
98A6     98A6 98A6 98A6 98A6 5648 D0A8 D0A8 D0A8 D0A8 5028 D74D D0A8 4E3F      E4B8BF       E4B8BF       F3B18989
98A7     98A7 98A7 98A7 98A7 5649 D0A9 D0A9 D0A9 D0A9 5029 D75D D0A9 4E42      E4B982       E4B982       F3B1898A
98A8     98A8 98A8 98A8 98A8 564A D0AA D0AA D0AA D0AA 502A D75C D0AA 4E56      E4B996       E4B996       F3B1898B
98A9     98A9 98A9 98A9 98A9 564B D0AB D0AB D0AB D0AB 502B D74E D0AB 4E58      E4B998       E4B998       F3B1898C
98AA     98AA 98AA 98AA 98AA 564D D0AC D0AC D0AC D0AC 502C D76B D0AC 4E82      E4BA82       E4BA82       F3B1898D
98AB     98AB 98AB 98AB 98AB 564E D0AD D0AD D0AD D0AD 502D D760 D0AD 4E85      E4BA85       E4BA85       F3B1898E
98AC     98AC 98AC 98AC 98AC 5355 D0AE D0AE D0AE D0AE 502E D74B D0AE 8C6B      E8B1AB       E8B1AB       F3B1898F
98AD     98AD 98AD 98AD 98AD 564F D0AF D0AF D0AF D0AF 502F D761 D0AF 4E8A      E4BA8A       E4BA8A       F3B18990
98AE     98AE 98AE 98AE 98AE 61DB D0B0 D0B0 D0B0 D0B0 5030 D7F0 D0B0 8212      E88892       E88892       F3B18991
98AF     98AF 98AF 98AF 98AF 5651 D0B1 D0B1 D0B1 D0B1 5031 D7F1 D0B1 5F0D      E5BC8D       E5BC8D       F3B18992
98B0     98B0 98B0 98B0 98B0 5650 D0B2 D0B2 D0B2 D0B2 5032 D7F2 D0B2 4E8E      E4BA8E       E4BA8E       F3B18993
98B1     98B1 98B1 98B1 98B1 5652 D0B3 D0B3 D0B3 D0B3 5033 D7F3 D0B3 4E9E      E4BA9E       E4BA9E       F3B18994
98B2     98B2 98B2 98B2 98B2 5167 D0B4 D0B4 D0B4 D0B4 5034 D7F4 D0B4 4E9F      E4BA9F       E4BA9F       F3B18995
98B3     98B3 98B3 98B3 98B3 5653 D0B5 D0B5 D0B5 D0B5 5035 D7F5 D0B5 4EA0      E4BAA0       E4BAA0       F3B18996
98B4     98B4 98B4 98B4 98B4 5654 D0B6 D0B6 D0B6 D0B6 5036 D7F6 D0B6 4EA2      E4BAA2       E4BAA2       F3B18997
98B5     98B5 98B5 98B5 98B5 5655 D0B7 D0B7 D0B7 D0B7 5037 D7F7 D0B7 4EB0      E4BAB0       E4BAB0       F3B18998
98B6     98B6 98B6 98B6 98B6 5656 D0B8 D0B8 D0B8 D0B8 5038 D7F8 D0B8 4EB3      E4BAB3       E4BAB3       F3B18999
98B7     98B7 98B7 98B7 98B7 5657 D0B9 D0B9 D0B9 D0B9 5039 D7F9 D0B9 4EB6      E4BAB6       E4BAB6       F3B1899A
98B8     98B8 98B8 98B8 98B8 5658 D0BA D0BA D0BA D0BA 503A D77A D0BA 4ECE      E4BB8E       E4BB8E       F3B1899B
98B9     98B9 98B9 98B9 98B9 5659 D0BB D0BB D0BB D0BB 503B D75E D0BB 4ECD      E4BB8D       E4BB8D       F3B1899C
98BA     98BA 98BA 98BA 98BA 565A D0BC D0BC D0BC D0BC 503C D74C D0BC 4EC4      E4BB84       E4BB84       F3B1899D
98BB     98BB 98BB 98BB 98BB 565B D0BD D0BD D0BD D0BD 503D D77E D0BD 4EC6      E4BB86       E4BB86       F3B1899E
98BC     98BC 98BC 98BC 98BC 565C D0BE D0BE D0BE D0BE 503E D76E D0BE 4EC2      E4BB82       E4BB82       F3B1899F
98BD     98BD 98BD 98BD 98BD 565E D0BF D0BF D0BF D0BF 503F D76F D0BF 4ED7      E4BB97       E4BB97       F3B189A0
98BE     98BE 98BE 98BE 98BE 565F D0C0 D0C0 D0C0 D0C0 5040 D77C D0C0 4EDE      E4BB9E       E4BB9E       F3B189A1
98BF     98BF 98BF 98BF 98BF 5660 D0C1 D0C1 D0C1 D0C1 5041 D7C1 D0C1 4EED      E4BBAD       E4BBAD       F3B189A2
98C0     98C0 98C0 98C0 98C0 5661 D0C2 D0C2 D0C2 D0C2 5042 D7C2 D0C2 4EDF      E4BB9F       E4BB9F       F3B189A3
98C1     98C1 98C1 98C1 98C1 5341 D0C3 D0C3 D0C3 D0C3 5043 D7C3 D0C3 4EF7      E4BBB7       E4BBB7       F3B189A4
98C2     98C2 98C2 98C2 98C2 5663 D0C4 D0C4 D0C4 D0C4 5044 D7C4 D0C4 4F09      E4BC89       E4BC89       F3B189A5
98C3     98C3 98C3 98C3 98C3 5667 D0C5 D0C5 D0C5 D0C5 5045 D7C5 D0C5 4F5A      E4BD9A       E4BD9A       F3B189A6
98C4     98C4 98C4 98C4 98C4 5668 D0C6 D0C6 D0C6 D0C6 5046 D7C6 D0C6 4F30      E4BCB0       E4BCB0       F3B189A7
98C5     98C5 98C5 98C5 98C5 566F D0C7 D0C7 D0C7 D0C7 5047 D7C7 D0C7 4F5B      E4BD9B       E4BD9B       F3B189A8
98C6     98C6 98C6 98C6 98C6 5669 D0C8 D0C8 D0C8 D0C8 5048 D7C8 D0C8 4F5D      E4BD9D       E4BD9D       F3B189A9
98C7     98C7 98C7 98C7 98C7 566B D0C9 D0C9 D0C9 D0C9 5049 D7C9 D0C9 4F57      E4BD97       E4BD97       F3B189AA
98C8     98C8 98C8 98C8 98C8 566C D0CA D0CA D0CA D0CA 504A D7D1 D0CA 4F47      E4BD87       E4BD87       F3B189AB
98C9     98C9 98C9 98C9 98C9 51E2 D0CB D0CB D0CB D0CB 504B D7D2 D0CB 4F76      E4BDB6       E4BDB6       F3B189AC
98CA     98CA 98CA 98CA 98CA 5672 D0CC D0CC D0CC D0CC 504C D7D3 D0CC 4F88      E4BE88       E4BE88       F3B189AD
98CB     98CB 98CB 98CB 98CB 5673 D0CD D0CD D0CD D0CD 504D D7D4 D0CD 4F8F      E4BE8F       E4BE8F       F3B189AE
98CC     98CC 98CC 98CC 98CC 5676 D0CE D0CE D0CE D0CE 504E D7D5 D0CE 4F98      E4BE98       E4BE98       F3B189AF
98CD     98CD 98CD 98CD 98CD 5677 D0CF D0CF D0CF D0CF 504F D7D6 D0CF 4F7B      E4BDBB       E4BDBB       F3B189B0
98CE     98CE 98CE 98CE 98CE 5679 D0D0 D0D0 D0D0 D0D0 5050 D7D7 D0D0 4F69      E4BDA9       E4BDA9       F3B189B1
98CF     98CF 98CF 98CF 98CF 567A D0D1 D0D1 D0D1 D0D1 5051 D7D8 D0D1 4F70      E4BDB0       E4BDB0       F3B189B2
98D0     98D0 98D0 98D0 98D0 4F8D D0D2 D0D2 D0D2 D0D2 5052 D7D9 D0D2 4F91      E4BE91       E4BE91       F3B189B3
98D1     98D1 98D1 98D1 98D1 567C D0D3 D0D3 D0D3 D0D3 5053 D7E2 D0D3 4F6F      E4BDAF       E4BDAF       F3B189B4
98D2     98D2 98D2 98D2 98D2 567D D0D4 D0D4 D0D4 D0D4 5054 D7E3 D0D4 4F86      E4BE86       E4BE86       F3B189B5
98D3     98D3 98D3 98D3 98D3 567E D0D5 D0D5 D0D5 D0D5 5055 D7E4 D0D5 4F96      E4BE96       E4BE96       F3B189B6
98D4     98D4 98D4 98D4 98D4 5290 CBF9 D0D6 D0D6 CBF9 4B79 D2AB D0D6 5118      E58498       E58498       F3B189B7
98D5     98D5 98D5 98D5 98D5 567F D0D7 D0D7 D0D7 D0D7 5057 D7E6 D0D7 4FD4      E4BF94       E4BF94       F3B189B8
98D6     98D6 98D6 98D6 98D6 53C4 D0D8 D0D8 D0D8 D0D8 5058 D7E7 D0D8 4FDF      E4BF9F       E4BF9F       F3B189B9
98D7     98D7 98D7 98D7 98D7 5680 D0D9 D0D9 D0D9 D0D9 5059 D7E8 D0D9 4FCE      E4BF8E       E4BF8E       F3B189BA
98D8     98D8 98D8 98D8 98D8 5681 D0DA D0DA D0DA D0DA 505A D7E9 D0DA 4FD8      E4BF98       E4BF98       F3B189BB
98D9     98D9 98D9 98D9 98D9 5682 D0DB D0DB D0DB D0DB 505B D74A D0DB 4FDB      E4BF9B       E4BF9B       F3B189BC
98DA     98DA 98DA 98DA 98DA 5683 D0DC D0DC D0DC D0DC 505C D75B D0DC 4FD1      E4BF91       E4BF91       F3B189BD
98DB     98DB 98DB 98DB 98DB 5684 D0DD D0DD D0DD D0DD 505D D75A D0DD 4FDA      E4BF9A       E4BF9A       F3B189BE
98DC     98DC 98DC 98DC 98DC 5685 D0DE D0DE D0DE D0DE 505E D75F D0DE 4FD0      E4BF90       E4BF90       F3B189BF
98DD     98DD 98DD 98DD 98DD 5687 D0DF D0DF D0DF D0DF 505F D76D D0DF 4FE4      E4BFA4       E4BFA4       F3B18A80
98DE     98DE 98DE 98DE 98DE 5688 D0E0 D0E0 D0E0 D0E0 5060 D779 D0E0 4FE5      E4BFA5       E4BFA5       F3B18A81
98DF     98DF 98DF 98DF 98DF 5689 D0E1 D0E1 D0E1 D0E1 5061 D757 D0E1 501A      E5809A       E5809A       F3B18A82
98E0     98E0 98E0 98E0 98E0 568B D0E2 D0E2 D0E2 D0E2 5062 D759 D0E2 5028      E580A8       E580A8       F3B18A83
98E1     98E1 98E1 98E1 98E1 568C D0E3 D0E3 D0E3 D0E3 5063 D762 D0E3 5014      E58094       E58094       F3B18A84
98E2     98E2 98E2 98E2 98E2 568D D0E4 D0E4 D0E4 D0E4 5064 D763 D0E4 502A      E580AA       E580AA       F3B18A85
98E3     98E3 98E3 98E3 98E3 568E D0E5 D0E5 D0E5 D0E5 5065 D764 D0E5 5025      E580A5       E580A5       F3B18A86
98E4     98E4 98E4 98E4 98E4 568F D0E6 D0E6 D0E6 D0E6 5066 D765 D0E6 5005      E58085       E58085       F3B18A87
98E5     98E5 98E5 98E5 98E5 5664 D0E7 D0E7 D0E7 D0E7 5067 D766 D0E7 4F1C      E4BC9C       E4BC9C       F3B18A88
98E6     98E6 98E6 98E6 98E6 51F8 D0E8 D0E8 D0E8 D0E8 5068 D767 D0E8 4FF6      E4BFB6       E4BFB6       F3B18A89
98E7     98E7 98E7 98E7 98E7 5690 D0E9 D0E9 D0E9 D0E9 5069 D768 D0E9 5021      E580A1       E580A1       F3B18A8A
98E8     98E8 98E8 98E8 98E8 5692 D0EA D0EA D0EA D0EA 506A D769 D0EA 5029      E580A9       E580A9       F3B18A8B
98E9     98E9 98E9 98E9 98E9 5693 D0EB D0EB D0EB D0EB 506B D770 D0EB 502C      E580AC       E580AC       F3B18A8C
98EA     98EA 98EA 98EA 98EA 5695 D0EC D0EC D0EC D0EC 506C D771 D0EC 4FFE      E4BFBE       E4BFBE       F3B18A8D
98EB     98EB 98EB 98EB 98EB 5696 D0ED D0ED D0ED D0ED 506D D772 D0ED 4FEF      E4BFAF       E4BFAF       F3B18A8E
98EC     98EC 98EC 98EC 98EC 5697 D0EE D0EE D0EE D0EE 506E D773 D0EE 5011      E58091       E58091       F3B18A8F
98ED     98ED 98ED 98ED 98ED 5699 D0EF D0EF D0EF D0EF 506F D774 D0EF 5006      E58086       E58086       F3B18A90
98EE     98EE 98EE 98EE 98EE 569A D0F0 D0F0 D0F0 D0F0 5070 D775 D0F0 5043      E58183       E58183       F3B18A91
98EF     98EF 98EF 98EF 98EF 569B D0F1 D0F1 D0F1 D0F1 5071 D776 D0F1 5047      E58187       E58187       F3B18A92
98F0     98F0 98F0 98F0 98F0 56AC D0F2 D0F2 D0F2 D0F2 5072 D777 D0F2 6703      E69C83       E69C83       F3B18A93
98F1     98F1 98F1 98F1 98F1 569C D0F3 D0F3 D0F3 D0F3 5073 D778 D0F3 5055      E58195       E58195       F3B18A94
98F2     98F2 98F2 98F2 98F2 569D D0F4 D0F4 D0F4 D0F4 5074 D780 D0F4 5050      E58190       E58190       F3B18A95
98F3     98F3 98F3 98F3 98F3 569E D0F5 D0F5 D0F5 D0F5 5075 D78B D0F5 5048      E58188       E58188       F3B18A96
98F4     98F4 98F4 98F4 98F4 569F D0F6 D0F6 D0F6 D0F6 5076 D79B D0F6 505A      E5819A       E5819A       F3B18A97
98F5     98F5 98F5 98F5 98F5 56A0 D0F7 D0F7 D0F7 D0F7 5077 D79C D0F7 5056      E58196       E58196       F3B18A98
98F6     98F6 98F6 98F6 98F6 56A5 D0F8 D0F8 D0F8 D0F8 5078 D7A0 D0F8 506C      E581AC       E581AC       F3B18A99
98F7     98F7 98F7 98F7 98F7 56A6 D0F9 D0F9 D0F9 D0F9 5079 D7AB D0F9 5078      E581B8       E581B8       F3B18A9A
98F8     98F8 98F8 98F8 98F8 56A7 D0FA D0FA D0FA D0FA 507A D7B0 D0FA 5080      E58280       E58280       F3B18A9B
98F9     98F9 98F9 98F9 98F9 56A9 D0FB D0FB D0FB D0FB 507B D7C0 D0FB 509A      E5829A       E5829A       F3B18A9C
98FA     98FA 98FA 98FA 98FA 56AA D0FC D0FC D0FC D0FC 507C D76A D0FC 5085      E58285       E58285       F3B18A9D
98FB     98FB 98FB 98FB 98FB 56AB D0FD D0FD D0FD D0FD 507D D7D0 D0FD 50B4      E582B4       E582B4       F3B18A9E
98FC     98FC 98FC 98FC 98FC 56AD D0FE D0FE D0FE D0FE 507E D7A1 D0FE 50B2      E582B2       E582B2       F3B18A9F
9940     9940 9940 9940 9940 56AE D1A1 D1A1 D1A1 D1A1 5121 D84F D1A1 50C9      E58389       E58389       F3B18AA0
9941     9941 9941 9941 9941 56AF D1A2 D1A2 D1A2 D1A2 5122 D87F D1A2 50CA      E5838A       E5838A       F3B18AA1
9942     9942 9942 9942 9942 56B0 D1A3 D1A3 D1A3 D1A3 5123 D87B D1A3 50B3      E582B3       E582B3       F3B18AA2
9943     9943 9943 9943 9943 56B1 D1A4 D1A4 D1A4 D1A4 5124 D8E0 D1A4 50C2      E58382       E58382       F3B18AA3
9944     9944 9944 9944 9944 52D4 D1A5 D1A5 D1A5 D1A5 5125 D86C D1A5 50D6      E58396       E58396       F3B18AA4
9945     9945 9945 9945 9945 56B3 D1A6 D1A6 D1A6 D1A6 5126 D850 D1A6 50DE      E5839E       E5839E       F3B18AA5
9946     9946 9946 9946 9946 56B4 D1A7 D1A7 D1A7 D1A7 5127 D87D D1A7 50E5      E583A5       E583A5       F3B18AA6
9947     9947 9947 9947 9947 56B6 D1A8 D1A8 D1A8 D1A8 5128 D84D D1A8 50ED      E583AD       E583AD       F3B18AA7
9948     9948 9948 9948 9948 56B7 D1A9 D1A9 D1A9 D1A9 5129 D85D D1A9 50E3      E583A3       E583A3       F3B18AA8
9949     9949 9949 9949 9949 56B8 D1AA D1AA D1AA D1AA 512A D85C D1AA 50EE      E583AE       E583AE       F3B18AA9
994A     994A 994A 994A 994A 56B9 D1AB D1AB D1AB D1AB 512B D84E D1AB 50F9      E583B9       E583B9       F3B18AAA
994B     994B 994B 994B 994B 56BA D1AC D1AC D1AC D1AC 512C D86B D1AC 50F5      E583B5       E583B5       F3B18AAB
994C     994C 994C 994C 994C 56BB D1AD D1AD D1AD D1AD 512D D860 D1AD 5109      E58489       E58489       F3B18AAC
994D     994D 994D 994D 994D 56BC D1AE D1AE D1AE D1AE 512E D84B D1AE 5101      E58481       E58481       F3B18AAD
994E     994E 994E 994E 994E 56BD D1AF D1AF D1AF D1AF 512F D861 D1AF 5102      E58482       E58482       F3B18AAE
994F     994F 994F 994F 994F 56C1 D1B0 D1B0 D1B0 D1B0 5130 D8F0 D1B0 5116      E58496       E58496       F3B18AAF
9950     9950 9950 9950 9950 56BF D1B1 D1B1 D1B1 D1B1 5131 D8F1 D1B1 5115      E58495       E58495       F3B18AB0
9951     9951 9951 9951 9951 56C0 D1B2 D1B2 D1B2 D1B2 5132 D8F2 D1B2 5114      E58494       E58494       F3B18AB1
9952     9952 9952 9952 9952 56BE D1B3 D1B3 D1B3 D1B3 5133 D8F3 D1B3 511A      E5849A       E5849A       F3B18AB2
9953     9953 9953 9953 9953 56C2 D1B4 D1B4 D1B4 D1B4 5134 D8F4 D1B4 5121      E584A1       E584A1       F3B18AB3
9954     9954 9954 9954 9954 56C3 D1B5 D1B5 D1B5 D1B5 5135 D8F5 D1B5 513A      E584BA       E584BA       F3B18AB4
9955     9955 9955 9955 9955 56C4 D1B6 D1B6 D1B6 D1B6 5136 D8F6 D1B6 5137      E584B7       E584B7       F3B18AB5
9956     9956 9956 9956 9956 56C5 D1B7 D1B7 D1B7 D1B7 5137 D8F7 D1B7 513C      E584BC       E584BC       F3B18AB6
9957     9957 9957 9957 9957 56C6 D1B8 D1B8 D1B8 D1B8 5138 D8F8 D1B8 513B      E584BB       E584BB       F3B18AB7
9958     9958 9958 9958 9958 56C7 D1B9 D1B9 D1B9 D1B9 5139 D8F9 D1B9 513F      E584BF       E584BF       F3B18AB8
9959     9959 9959 9959 9959 56C8 D1BA D1BA D1BA D1BA 513A D87A D1BA 5140      E58580       E58580       F3B18AB9
995A     995A 995A 995A 995A 56CB D1BB D1BB D1BB D1BB 513B D85E D1BB 5152      E58592       E58592       F3B18ABA
995B     995B 995B 995B 995B 56CA D1BC D1BC D1BC D1BC 513C D84C D1BC 514C      E5858C       E5858C       F3B18ABB
995C     995C 995C 995C 995C 56CC D1BD D1BD D1BD D1BD 513D D87E D1BD 5154      E58594       E58594       F3B18ABC
995D     995D 995D 995D 995D 56CD D1BE D1BE D1BE D1BE 513E D86E D1BE 5162      E585A2       E585A2       F3B18ABD
995E     995E 995E 995E 995E 605D D1BF D1BF D1BF D1BF 513F D86F D1BF 7AF8      E7ABB8       E7ABB8       F3B18ABE
995F     995F 995F 995F 995F 56CF D1C0 D1C0 D1C0 D1C0 5140 D87C D1C0 5169      E585A9       E585A9       F3B18ABF
9960     9960 9960 9960 9960 56D0 D1C1 D1C1 D1C1 D1C1 5141 D8C1 D1C1 516A      E585AA       E585AA       F3B18B80
9961     9961 9961 9961 9961 56D1 D1C2 D1C2 D1C2 D1C2 5142 D8C2 D1C2 516E      E585AE       E585AE       F3B18B81
9962     9962 9962 9962 9962 56D2 D1C3 D1C3 D1C3 D1C3 5143 D8C3 D1C3 5180      E58680       E58680       F3B18B82
9963     9963 9963 9963 9963 56D3 D1C4 D1C4 D1C4 D1C4 5144 D8C4 D1C4 5182      E58682       E58682       F3B18B83
9964     9964 9964 9964 9964 56D4 D1C5 D1C5 D1C5 D1C5 5145 D8C5 D1C5 56D8      E59B98       E59B98       F3B18B84
9965     9965 9965 9965 9965 56D5 D1C6 D1C6 D1C6 D1C6 5146 D8C6 D1C6 518C      E5868C       E5868C       F3B18B85
9966     9966 9966 9966 9966 56D6 D1C7 D1C7 D1C7 D1C7 5147 D8C7 D1C7 5189      E58689       E58689       F3B18B86
9967     9967 9967 9967 9967 56D7 D1C8 D1C8 D1C8 D1C8 5148 D8C8 D1C8 518F      E5868F       E5868F       F3B18B87
9968     9968 9968 9968 9968 56D8 D1C9 D1C9 D1C9 D1C9 5149 D8C9 D1C9 5191      E58691       E58691       F3B18B88
9969     9969 9969 9969 9969 56D9 D1CA D1CA D1CA D1CA 514A D8D1 D1CA 5193      E58693       E58693       F3B18B89
996A     996A 996A 996A 996A 56DA D1CB D1CB D1CB D1CB 514B D8D2 D1CB 5195      E58695       E58695       F3B18B8A
996B     996B 996B 996B 996B 56DB D1CC D1CC D1CC D1CC 514C D8D3 D1CC 5196      E58696       E58696       F3B18B8B
996C     996C 996C 996C 996C 56DD D1CD D1CD D1CD D1CD 514D D8D4 D1CD 51A4      E586A4       E586A4       F3B18B8C
996D     996D 996D 996D 996D 56DE D1CE D1CE D1CE D1CE 514E D8D5 D1CE 51A6      E586A6       E586A6       F3B18B8D
996E     996E 996E 996E 996E 56DF D1CF D1CF D1CF D1CF 514F D8D6 D1CF 51A2      E586A2       E586A2       F3B18B8E
996F     996F 996F 996F 996F 56E0 D1D0 D1D0 D1D0 D1D0 5150 D8D7 D1D0 51A9      E586A9       E586A9       F3B18B8F
9970     9970 9970 9970 9970 56E1 D1D1 D1D1 D1D1 D1D1 5151 D8D8 D1D1 51AA      E586AA       E586AA       F3B18B90
9971     9971 9971 9971 9971 56E2 D1D2 D1D2 D1D2 D1D2 5152 D8D9 D1D2 51AB      E586AB       E586AB       F3B18B91
9972     9972 9972 9972 9972 56E3 D1D3 D1D3 D1D3 D1D3 5153 D8E2 D1D3 51B3      E586B3       E586B3       F3B18B92
9973     9973 9973 9973 9973 56E4 D1D4 D1D4 D1D4 D1D4 5154 D8E3 D1D4 51B1      E586B1       E586B1       F3B18B93
9974     9974 9974 9974 9974 56E5 D1D5 D1D5 D1D5 D1D5 5155 D8E4 D1D5 51B2      E586B2       E586B2       F3B18B94
9975     9975 9975 9975 9975 56E6 D1D6 D1D6 D1D6 D1D6 5156 D8E5 D1D6 51B0      E586B0       E586B0       F3B18B95
9976     9976 9976 9976 9976 56E7 D1D7 D1D7 D1D7 D1D7 5157 D8E6 D1D7 51B5      E586B5       E586B5       F3B18B96
9977     9977 9977 9977 9977 56E9 D1D8 D1D8 D1D8 D1D8 5158 D8E7 D1D8 51BD      E586BD       E586BD       F3B18B97
9978     9978 9978 9978 9978 56EA D1D9 D1D9 D1D9 D1D9 5159 D8E8 D1D9 51C5      E58785       E58785       F3B18B98
9979     9979 9979 9979 9979 56EB D1DA D1DA D1DA D1DA 515A D8E9 D1DA 51C9      E58789       E58789       F3B18B99
997A     997A 997A 997A 997A 56EC D1DB D1DB D1DB D1DB 515B D84A D1DB 51DB      E5879B       E5879B       F3B18B9A
997B     997B 997B 997B 997B 56ED D1DC D1DC D1DC D1DC 515C D85B D1DC 51E0      E587A0       E587A0       F3B18B9B
997C     997C 997C 997C 997C 62D1 D1DD D1DD D1DD D1DD 515D D85A D1DD 8655      E89995       E89995       F3B18B9C
997D     997D 997D 997D 997D 56EE D1DE D1DE D1DE D1DE 515E D85F D1DE 51E9      E587A9       E587A9       F3B18B9D
997E     997E 997E 997E 997E 56F0 D1DF D1DF D1DF D1DF 515F D86D D1DF 51ED      E587AD       E587AD       F3B18B9E
9980     9980 9980 9980 9980 56F1 D1E0 D1E0 D1E0 D1E0 5160 D879 D1E0 51F0      E587B0       E587B0       F3B18B9F
9981     9981 9981 9981 9981 56F2 D1E1 D1E1 D1E1 D1E1 5161 D857 D1E1 51F5      E587B5       E587B5       F3B18BA0
9982     9982 9982 9982 9982 56F3 D1E2 D1E2 D1E2 D1E2 5162 D859 D1E2 51FE      E587BE       E587BE       F3B18BA1
9983     9983 9983 9983 9983 56F4 D1E3 D1E3 D1E3 D1E3 5163 D862 D1E3 5204      E58884       E58884       F3B18BA2
9984     9984 9984 9984 9984 56F5 D1E4 D1E4 D1E4 D1E4 5164 D863 D1E4 520B      E5888B       E5888B       F3B18BA3
9985     9985 9985 9985 9985 56F6 D1E5 D1E5 D1E5 D1E5 5165 D864 D1E5 5214      E58894       E58894       F3B18BA4
9986     9986 9986 9986 9986 54E3 D1E6 D1E6 D1E6 D1E6 5166 D865 D1E6 520E      E5888E       E5888E       F3B18BA5
9987     9987 9987 9987 9987 56F8 D1E7 D1E7 D1E7 D1E7 5167 D866 D1E7 5227      E588A7       E588A7       F3B18BA6
9988     9988 9988 9988 9988 56F9 D1E8 D1E8 D1E8 D1E8 5168 D867 D1E8 522A      E588AA       E588AA       F3B18BA7
9989     9989 9989 9989 9989 56FA D1E9 D1E9 D1E9 D1E9 5169 D868 D1E9 522E      E588AE       E588AE       F3B18BA8
998A     998A 998A 998A 998A 56FB D1EA D1EA D1EA D1EA 516A D869 D1EA 5233      E588B3       E588B3       F3B18BA9
998B     998B 998B 998B 998B 56FC D1EB D1EB D1EB D1EB 516B D870 D1EB 5239      E588B9       E588B9       F3B18BAA
998C     998C 998C 998C 998C 5741 D1EC D1EC D1EC D1EC 516C D871 D1EC 524F      E5898F       E5898F       F3B18BAB
998D     998D 998D 998D 998D 56FD D1ED D1ED D1ED D1ED 516D D872 D1ED 5244      E58984       E58984       F3B18BAC
998E     998E 998E 998E 998E 56FE D1EE D1EE D1EE D1EE 516E D873 D1EE 524B      E5898B       E5898B       F3B18BAD
998F     998F 998F 998F 998F 51AE D1EF D1EF D1EF D1EF 516F D874 D1EF 524C      E5898C       E5898C       F3B18BAE
9990     9990 9990 9990 9990 5742 D1F0 D1F0 D1F0 D1F0 5170 D875 D1F0 525E      E5899E       E5899E       F3B18BAF
9991     9991 9991 9991 9991 5743 D1F1 D1F1 D1F1 D1F1 5171 D876 D1F1 5254      E58994       E58994       F3B18BB0
9992     9992 9992 9992 9992 5745 D1F2 D1F2 D1F2 D1F2 5172 D877 D1F2 526A      E589AA       E589AA       F3B18BB1
9993     9993 9993 9993 9993 5747 D1F3 D1F3 D1F3 D1F3 5173 D878 D1F3 5274      E589B4       E589B4       F3B18BB2
9994     9994 9994 9994 9994 5748 D1F4 D1F4 D1F4 D1F4 5174 D880 D1F4 5269      E589A9       E589A9       F3B18BB3
9995     9995 9995 9995 9995 5746 D1F5 D1F5 D1F5 D1F5 5175 D88B D1F5 5273      E589B3       E589B3       F3B18BB4
9996     9996 9996 9996 9996 5749 D1F6 D1F6 D1F6 D1F6 5176 D89B D1F6 527F      E589BF       E589BF       F3B18BB5
9997     9997 9997 9997 9997 574A D1F7 D1F7 D1F7 D1F7 5177 D89C D1F7 527D      E589BD       E589BD       F3B18BB6
9998     9998 9998 9998 9998 574B D1F8 D1F8 D1F8 D1F8 5178 D8A0 D1F8 528D      E58A8D       E58A8D       F3B18BB7
9999     9999 9999 9999 9999 5580 D1F9 D1F9 D1F9 D1F9 5179 D8AB D1F9 5294      E58A94       E58A94       F3B18BB8
999A     999A 999A 999A 999A 574D D1FA D1FA D1FA D1FA 517A D8B0 D1FA 5292      E58A92       E58A92       F3B18BB9
999B     999B 999B 999B 999B 5744 D1FB D1FB D1FB D1FB 517B D8C0 D1FB 5271      E589B1       E589B1       F3B18BBA
999C     999C 999C 999C 999C 574C D1FC D1FC D1FC D1FC 517C D86A D1FC 5288      E58A88       E58A88       F3B18BBB
999D     999D 999D 999D 999D 574E D1FD D1FD D1FD D1FD 517D D8D0 D1FD 5291      E58A91       E58A91       F3B18BBC
999E     999E 999E 999E 999E 64EC D1FE D1FE D1FE D1FE 517E D8A1 D1FE 8FA8      E8BEA8       E8BEA8       F3B18BBD
999F     999F 999F 999F 999F 64ED D2A1 D2A1 D2A1 D2A1 5221 D94F D2A1 8FA7      E8BEA7       E8BEA7       F3B18BBE
99A0     99A0 99A0 99A0 99A0 5751 D2A2 D2A2 D2A2 D2A2 5222 D97F D2A2 52AC      E58AAC       E58AAC       F3B18BBF
99A1     99A1 99A1 99A1 99A1 5752 D2A3 D2A3 D2A3 D2A3 5223 D97B D2A3 52AD      E58AAD       E58AAD       F3B18C80
99A2     99A2 99A2 99A2 99A2 5753 D2A4 D2A4 D2A4 D2A4 5224 D9E0 D2A4 52BC      E58ABC       E58ABC       F3B18C81
99A3     99A3 99A3 99A3 99A3 5754 D2A5 D2A5 D2A5 D2A5 5225 D96C D2A5 52B5      E58AB5       E58AB5       F3B18C82
99A4     99A4 99A4 99A4 99A4 5755 D2A6 D2A6 D2A6 D2A6 5226 D950 D2A6 52C1      E58B81       E58B81       F3B18C83
99A5     99A5 99A5 99A5 99A5 5757 D2A7 D2A7 D2A7 D2A7 5227 D97D D2A7 52CD      E58B8D       E58B8D       F3B18C84
99A6     99A6 99A6 99A6 99A6 5392 D2A8 D2A8 D2A8 D2A8 5228 D94D D2A8 52D7      E58B97       E58B97       F3B18C85
99A7     99A7 99A7 99A7 99A7 5759 D2A9 D2A9 D2A9 D2A9 5229 D95D D2A9 52DE      E58B9E       E58B9E       F3B18C86
99A8     99A8 99A8 99A8 99A8 575A D2AA D2AA D2AA D2AA 522A D95C D2AA 52E3      E58BA3       E58BA3       F3B18C87
99A9     99A9 99A9 99A9 99A9 575B D2AB D2AB D2AB D2AB 522B D94E D2AB 52E6      E58BA6       E58BA6       F3B18C88
99AA     99AA 99AA 99AA 99AA 66DB D2AC D2AC D2AC D2AC 522C D96B D2AC 98ED      E9A3AD       E9A3AD       F3B18C89
99AB     99AB 99AB 99AB 99AB 575C D2AD D2AD D2AD D2AD 522D D960 D2AD 52E0      E58BA0       E58BA0       F3B18C8A
99AC     99AC 99AC 99AC 99AC 575D D2AE D2AE D2AE D2AE 522E D94B D2AE 52F3      E58BB3       E58BB3       F3B18C8B
99AD     99AD 99AD 99AD 99AD 575E D2AF D2AF D2AF D2AF 522F D961 D2AF 52F5      E58BB5       E58BB5       F3B18C8C
99AE     99AE 99AE 99AE 99AE 575F D2B0 D2B0 D2B0 D2B0 5230 D9F0 D2B0 52F8      E58BB8       E58BB8       F3B18C8D
99AF     99AF 99AF 99AF 99AF 5760 D2B1 D2B1 D2B1 D2B1 5231 D9F1 D2B1 52F9      E58BB9       E58BB9       F3B18C8E
99B0     99B0 99B0 99B0 99B0 5762 D2B2 D2B2 D2B2 D2B2 5232 D9F2 D2B2 5306      E58C86       E58C86       F3B18C8F
99B1     99B1 99B1 99B1 99B1 5764 D2B3 D2B3 D2B3 D2B3 5233 D9F3 D2B3 5308      E58C88       E58C88       F3B18C90
99B2     99B2 99B2 99B2 99B2 5765 D2B4 D2B4 D2B4 D2B4 5234 D9F4 D2B4 7538      E794B8       E794B8       F3B18C91
99B3     99B3 99B3 99B3 99B3 5766 D2B5 D2B5 D2B5 D2B5 5235 D9F5 D2B5 530D      E58C8D       E58C8D       F3B18C92
99B4     99B4 99B4 99B4 99B4 5767 D2B6 D2B6 D2B6 D2B6 5236 D9F6 D2B6 5310      E58C90       E58C90       F3B18C93
99B5     99B5 99B5 99B5 99B5 5768 D2B7 D2B7 D2B7 D2B7 5237 D9F7 D2B7 530F      E58C8F       E58C8F       F3B18C94
99B6     99B6 99B6 99B6 99B6 5769 D2B8 D2B8 D2B8 D2B8 5238 D9F8 D2B8 5315      E58C95       E58C95       F3B18C95
99B7     99B7 99B7 99B7 99B7 576A D2B9 D2B9 D2B9 D2B9 5239 D9F9 D2B9 531A      E58C9A       E58C9A       F3B18C96
99B8     99B8 99B8 99B8 99B8 576C D2BA D2BA D2BA D2BA 523A D97A D2BA 5323      E58CA3       E58CA3       F3B18C97
99B9     99B9 99B9 99B9 99B9 576D D2BB D2BB D2BB D2BB 523B D95E D2BB 532F      E58CAF       E58CAF       F3B18C98
99BA     99BA 99BA 99BA 99BA 576E D2BC D2BC D2BC D2BC 523C D94C D2BC 5331      E58CB1       E58CB1       F3B18C99
99BB     99BB 99BB 99BB 99BB 576F D2BD D2BD D2BD D2BD 523D D97E D2BD 5333      E58CB3       E58CB3       F3B18C9A
99BC     99BC 99BC 99BC 99BC 5770 D2BE D2BE D2BE D2BE 523E D96E D2BE 5338      E58CB8       E58CB8       F3B18C9B
99BD     99BD 99BD 99BD 99BD 5771 D2BF D2BF D2BF D2BF 523F D96F D2BF 5340      E58D80       E58D80       F3B18C9C
99BE     99BE 99BE 99BE 99BE 5773 D2C0 D2C0 D2C0 D2C0 5240 D97C D2C0 5346      E58D86       E58D86       F3B18C9D
99BF     99BF 99BF 99BF 99BF 5772 D2C1 D2C1 D2C1 D2C1 5241 D9C1 D2C1 5345      E58D85       E58D85       F3B18C9E
99C0     99C0 99C0 99C0 99C0 5775 D2C2 D2C2 D2C2 D2C2 5242 D9C2 D2C2 4E17      E4B897       E4B897       F3B18C9F
99C1     99C1 99C1 99C1 99C1 5774 D2C3 D2C3 D2C3 D2C3 5243 D9C3 D2C3 5349      E58D89       E58D89       F3B18CA0
99C2     99C2 99C2 99C2 99C2 5776 D2C4 D2C4 D2C4 D2C4 5244 D9C4 D2C4 534D      E58D8D       E58D8D       F3B18CA1
99C3     99C3 99C3 99C3 99C3 5777 D2C5 D2C5 D2C5 D2C5 5245 D9C5 D2C5 51D6      E58796       E58796       F3B18CA2
99C4     99C4 99C4 99C4 99C4 5779 D2C6 D2C6 D2C6 D2C6 5246 D9C6 D2C6 535E      E58D9E       E58D9E       F3B18CA3
99C5     99C5 99C5 99C5 99C5 577A D2C7 D2C7 D2C7 D2C7 5247 D9C7 D2C7 5369      E58DA9       E58DA9       F3B18CA4
99C6     99C6 99C6 99C6 99C6 577B D2C8 D2C8 D2C8 D2C8 5248 D9C8 D2C8 536E      E58DAE       E58DAE       F3B18CA5
99C7     99C7 99C7 99C7 99C7 52C8 D2C9 D2C9 D2C9 D2C9 5249 D9C9 D2C9 5918      E5A498       E5A498       F3B18CA6
99C8     99C8 99C8 99C8 99C8 577E D2CA D2CA D2CA D2CA 524A D9D1 D2CA 537B      E58DBB       E58DBB       F3B18CA7
99C9     99C9 99C9 99C9 99C9 577D D2CB D2CB D2CB D2CB 524B D9D2 D2CB 5377      E58DB7       E58DB7       F3B18CA8
99CA     99CA 99CA 99CA 99CA 577F D2CC D2CC D2CC D2CC 524C D9D3 D2CC 5382      E58E82       E58E82       F3B18CA9
99CB     99CB 99CB 99CB 99CB 5781 D2CD D2CD D2CD D2CD 524D D9D4 D2CD 5396      E58E96       E58E96       F3B18CAA
99CC     99CC 99CC 99CC 99CC 5782 D2CE D2CE D2CE D2CE 524E D9D5 D2CE 53A0      E58EA0       E58EA0       F3B18CAB
99CD     99CD 99CD 99CD 99CD 5783 D2CF D2CF D2CF D2CF 524F D9D6 D2CF 53A6      E58EA6       E58EA6       F3B18CAC
99CE     99CE 99CE 99CE 99CE 5784 D2D0 D2D0 D2D0 D2D0 5250 D9D7 D2D0 53A5      E58EA5       E58EA5       F3B18CAD
99CF     99CF 99CF 99CF 99CF 5785 D2D1 D2D1 D2D1 D2D1 5251 D9D8 D2D1 53AE      E58EAE       E58EAE       F3B18CAE
99D0     99D0 99D0 99D0 99D0 5786 D2D2 D2D2 D2D2 D2D2 5252 D9D9 D2D2 53B0      E58EB0       E58EB0       F3B18CAF
99D1     99D1 99D1 99D1 99D1 5788 D2D3 D2D3 D2D3 D2D3 5253 D9E2 D2D3 53B6      E58EB6       E58EB6       F3B18CB0
99D2     99D2 99D2 99D2 99D2 5789 D2D4 D2D4 D2D4 D2D4 5254 D9E3 D2D4 53C3      E58F83       E58F83       F3B18CB1
99D3     99D3 99D3 99D3 99D3 578A D2D5 D2D5 D2D5 D2D5 5255 D9E4 D2D5 7C12      E7B092       E7B092       F3B18CB2
99D4     99D4 99D4 99D4 99D4 6688 D2D6 D2D6 D2D6 D2D6 5256 D9E5 D2D6 96D9      E99B99       E99B99       F3B18CB3
99D5     99D5 99D5 99D5 99D5 578C D2D7 D2D7 D2D7 D2D7 5257 D9E6 D2D7 53DF      E58F9F       E58F9F       F3B18CB4
99D6     99D6 99D6 99D6 99D6 578D D2D8 D2D8 D2D8 D2D8 5258 D9E7 D2D8 66FC      E69BBC       E69BBC       F3B18CB5
99D7     99D7 99D7 99D7 99D7 578F D2D9 D2D9 D2D9 D2D9 5259 D9E8 D2D9 71EE      E787AE       E787AE       F3B18CB6
99D8     99D8 99D8 99D8 99D8 5790 D2DA D2DA D2DA D2DA 525A D9E9 D2DA 53EE      E58FAE       E58FAE       F3B18CB7
99D9     99D9 99D9 99D9 99D9 5791 D2DB D2DB D2DB D2DB 525B D94A D2DB 53E8      E58FA8       E58FA8       F3B18CB8
99DA     99DA 99DA 99DA 99DA 5792 D2DC D2DC D2DC D2DC 525C D95B D2DC 53ED      E58FAD       E58FAD       F3B18CB9
99DB     99DB 99DB 99DB 99DB 5793 D2DD D2DD D2DD D2DD 525D D95A D2DD 53FA      E58FBA       E58FBA       F3B18CBA
99DC     99DC 99DC 99DC 99DC 5794 D2DE D2DE D2DE D2DE 525E D95F D2DE 5401      E59081       E59081       F3B18CBB
99DD     99DD 99DD 99DD 99DD 5795 D2DF D2DF D2DF D2DF 525F D96D D2DF 543D      E590BD       E590BD       F3B18CBC
99DE     99DE 99DE 99DE 99DE 5796 D2E0 D2E0 D2E0 D2E0 5260 D979 D2E0 5440      E59180       E59180       F3B18CBD
99DF     99DF 99DF 99DF 99DF 5797 D2E1 D2E1 D2E1 D2E1 5261 D957 D2E1 542C      E590AC       E590AC       F3B18CBE
99E0     99E0 99E0 99E0 99E0 5798 D2E2 D2E2 D2E2 D2E2 5262 D959 D2E2 542D      E590AD       E590AD       F3B18CBF
99E1     99E1 99E1 99E1 99E1 5799 D2E3 D2E3 D2E3 D2E3 5263 D962 D2E3 543C      E590BC       E590BC       F3B18D80
99E2     99E2 99E2 99E2 99E2 579A D2E4 D2E4 D2E4 D2E4 5264 D963 D2E4 542E      E590AE       E590AE       F3B18D81
99E3     99E3 99E3 99E3 99E3 579B D2E5 D2E5 D2E5 D2E5 5265 D964 D2E5 5436      E590B6       E590B6       F3B18D82
99E4     99E4 99E4 99E4 99E4 579C D2E6 D2E6 D2E6 D2E6 5266 D965 D2E6 5429      E590A9       E590A9       F3B18D83
99E5     99E5 99E5 99E5 99E5 579D D2E7 D2E7 D2E7 D2E7 5267 D966 D2E7 541D      E5909D       E5909D       F3B18D84
99E6     99E6 99E6 99E6 99E6 579E D2E8 D2E8 D2E8 D2E8 5268 D967 D2E8 544E      E5918E       E5918E       F3B18D85
99E7     99E7 99E7 99E7 99E7 579F D2E9 D2E9 D2E9 D2E9 5269 D968 D2E9 548F      E5928F       E5928F       F3B18D86
99E8     99E8 99E8 99E8 99E8 57A0 D2EA D2EA D2EA D2EA 526A D969 D2EA 5475      E591B5       E591B5       F3B18D87
99E9     99E9 99E9 99E9 99E9 57A1 D2EB D2EB D2EB D2EB 526B D970 D2EB 548E      E5928E       E5928E       F3B18D88
99EA     99EA 99EA 99EA 99EA 57A2 D2EC D2EC D2EC D2EC 526C D971 D2EC 545F      E5919F       E5919F       F3B18D89
99EB     99EB 99EB 99EB 99EB 57A3 D2ED D2ED D2ED D2ED 526D D972 D2ED 5471      E591B1       E591B1       F3B18D8A
99EC     99EC 99EC 99EC 99EC 57A4 D2EE D2EE D2EE D2EE 526E D973 D2EE 5477      E591B7       E591B7       F3B18D8B
99ED     99ED 99ED 99ED 99ED 57A5 D2EF D2EF D2EF D2EF 526F D974 D2EF 5470      E591B0       E591B0       F3B18D8C
99EE     99EE 99EE 99EE 99EE 57A6 D2F0 D2F0 D2F0 D2F0 5270 D975 D2F0 5492      E59292       E59292       F3B18D8D
99EF     99EF 99EF 99EF 99EF 57A7 D2F1 D2F1 D2F1 D2F1 5271 D976 D2F1 547B      E591BB       E591BB       F3B18D8E
99F0     99F0 99F0 99F0 99F0 57A8 D2F2 D2F2 D2F2 D2F2 5272 D977 D2F2 5480      E59280       E59280       F3B18D8F
99F1     99F1 99F1 99F1 99F1 57AA D2F3 D2F3 D2F3 D2F3 5273 D978 D2F3 5476      E591B6       E591B6       F3B18D90
99F2     99F2 99F2 99F2 99F2 57AB D2F4 D2F4 D2F4 D2F4 5274 D980 D2F4 5484      E59284       E59284       F3B18D91
99F3     99F3 99F3 99F3 99F3 57AC D2F5 D2F5 D2F5 D2F5 5275 D98B D2F5 5490      E59290       E59290       F3B18D92
99F4     99F4 99F4 99F4 99F4 57AD D2F6 D2F6 D2F6 D2F6 5276 D99B D2F6 5486      E59286       E59286       F3B18D93
99F5     99F5 99F5 99F5 99F5 57AF D2F7 D2F7 D2F7 D2F7 5277 D99C D2F7 54C7      E59387       E59387       F3B18D94
99F6     99F6 99F6 99F6 99F6 57B2 D2F8 D2F8 D2F8 D2F8 5278 D9A0 D2F8 54A2      E592A2       E592A2       F3B18D95
99F7     99F7 99F7 99F7 99F7 57B3 D2F9 D2F9 D2F9 D2F9 5279 D9AB D2F9 54B8      E592B8       E592B8       F3B18D96
99F8     99F8 99F8 99F8 99F8 57B4 D2FA D2FA D2FA D2FA 527A D9B0 D2FA 54A5      E592A5       E592A5       F3B18D97
99F9     99F9 99F9 99F9 99F9 57B5 D2FB D2FB D2FB D2FB 527B D9C0 D2FB 54AC      E592AC       E592AC       F3B18D98
99FA     99FA 99FA 99FA 99FA 57B6 D2FC D2FC D2FC D2FC 527C D96A D2FC 54C4      E59384       E59384       F3B18D99
99FB     99FB 99FB 99FB 99FB 57B8 D2FD D2FD D2FD D2FD 527D D9D0 D2FD 54C8      E59388       E59388       F3B18D9A
99FC     99FC 99FC 99FC 99FC 57B9 D2FE D2FE D2FE D2FE 527E D9A1 D2FE 54A8      E592A8       E592A8       F3B18D9B
9A40     9A40 9A40 9A40 9A40 57BA D3A1 D3A1 D3A1 D3A1 5321 E24F D3A1 54AB      E592AB       E592AB       F3B18D9C
9A41     9A41 9A41 9A41 9A41 57BB D3A2 D3A2 D3A2 D3A2 5322 E27F D3A2 54C2      E59382       E59382       F3B18D9D
9A42     9A42 9A42 9A42 9A42 57BC D3A3 D3A3 D3A3 D3A3 5323 E27B D3A3 54A4      E592A4       E592A4       F3B18D9E
9A43     9A43 9A43 9A43 9A43 57BE D3A4 D3A4 D3A4 D3A4 5324 E2E0 D3A4 54BE      E592BE       E592BE       F3B18D9F
9A44     9A44 9A44 9A44 9A44 57B0 D3A5 D3A5 D3A5 D3A5 5325 E26C D3A5 54BC      E592BC       E592BC       F3B18DA0
9A45     9A45 9A45 9A45 9A45 57B7 D3A6 D3A6 D3A6 D3A6 5326 E250 D3A6 54D8      E59398       E59398       F3B18DA1
9A46     9A46 9A46 9A46 9A46 57BF D3A7 D3A7 D3A7 D3A7 5327 E27D D3A7 54E5      E593A5       E593A5       F3B18DA2
9A47     9A47 9A47 9A47 9A47 57C1 D3A8 D3A8 D3A8 D3A8 5328 E24D D3A8 54E6      E593A6       E593A6       F3B18DA3
9A48     9A48 9A48 9A48 9A48 57C2 D3A9 D3A9 D3A9 D3A9 5329 E25D D3A9 550F      E5948F       E5948F       F3B18DA4
9A49     9A49 9A49 9A49 9A49 57C3 D3AA D3AA D3AA D3AA 532A E25C D3AA 5514      E59494       E59494       F3B18DA5
9A4A     9A4A 9A4A 9A4A 9A4A 57C4 D3AB D3AB D3AB D3AB 532B E24E D3AB 54FD      E593BD       E593BD       F3B18DA6
9A4B     9A4B 9A4B 9A4B 9A4B 57C5 D3AC D3AC D3AC D3AC 532C E26B D3AC 54EE      E593AE       E593AE       F3B18DA7
9A4C     9A4C 9A4C 9A4C 9A4C 57C6 D3AD D3AD D3AD D3AD 532D E260 D3AD 54ED      E593AD       E593AD       F3B18DA8
9A4D     9A4D 9A4D 9A4D 9A4D 547C D3AE D3AE D3AE D3AE 532E E24B D3AE 54FA      E593BA       E593BA       F3B18DA9
9A4E     9A4E 9A4E 9A4E 9A4E 57C7 D3AF D3AF D3AF D3AF 532F E261 D3AF 54E2      E593A2       E593A2       F3B18DAA
9A4F     9A4F 9A4F 9A4F 9A4F 57C8 D3B0 D3B0 D3B0 D3B0 5330 E2F0 D3B0 5539      E594B9       E594B9       F3B18DAB
9A50     9A50 9A50 9A50 9A50 57C9 D3B1 D3B1 D3B1 D3B1 5331 E2F1 D3B1 5540      E59580       E59580       F3B18DAC
9A51     9A51 9A51 9A51 9A51 57CA D3B2 D3B2 D3B2 D3B2 5332 E2F2 D3B2 5563      E595A3       E595A3       F3B18DAD
9A52     9A52 9A52 9A52 9A52 57CB D3B3 D3B3 D3B3 D3B3 5333 E2F3 D3B3 554C      E5958C       E5958C       F3B18DAE
9A53     9A53 9A53 9A53 9A53 57CC D3B4 D3B4 D3B4 D3B4 5334 E2F4 D3B4 552E      E594AE       E594AE       F3B18DAF
9A54     9A54 9A54 9A54 9A54 57CD D3B5 D3B5 D3B5 D3B5 5335 E2F5 D3B5 555C      E5959C       E5959C       F3B18DB0
9A55     9A55 9A55 9A55 9A55 57CE D3B6 D3B6 D3B6 D3B6 5336 E2F6 D3B6 5545      E59585       E59585       F3B18DB1
9A56     9A56 9A56 9A56 9A56 57CF D3B7 D3B7 D3B7 D3B7 5337 E2F7 D3B7 5556      E59596       E59596       F3B18DB2
9A57     9A57 9A57 9A57 9A57 57D0 D3B8 D3B8 D3B8 D3B8 5338 E2F8 D3B8 5557      E59597       E59597       F3B18DB3
9A58     9A58 9A58 9A58 9A58 57D1 D3B9 D3B9 D3B9 D3B9 5339 E2F9 D3B9 5538      E594B8       E594B8       F3B18DB4
9A59     9A59 9A59 9A59 9A59 57D2 D3BA D3BA D3BA D3BA 533A E27A D3BA 5533      E594B3       E594B3       F3B18DB5
9A5A     9A5A 9A5A 9A5A 9A5A 57D3 D3BB D3BB D3BB D3BB 533B E25E D3BB 555D      E5959D       E5959D       F3B18DB6
9A5B     9A5B 9A5B 9A5B 9A5B 57D4 D3BC D3BC D3BC D3BC 533C E24C D3BC 5599      E59699       E59699       F3B18DB7
9A5C     9A5C 9A5C 9A5C 9A5C 57D5 D3BD D3BD D3BD D3BD 533D E27E D3BD 5580      E59680       E59680       F3B18DB8
9A5D     9A5D 9A5D 9A5D 9A5D 57B1 D3BE D3BE D3BE D3BE 533E E26E D3BE 54AF      E592AF       E592AF       F3B18DB9
9A5E     9A5E 9A5E 9A5E 9A5E 57D6 D3BF D3BF D3BF D3BF 533F E26F D3BF 558A      E5968A       E5968A       F3B18DBA
9A5F     9A5F 9A5F 9A5F 9A5F 57D7 D3C0 D3C0 D3C0 D3C0 5340 E27C D3C0 559F      E5969F       E5969F       F3B18DBB
9A60     9A60 9A60 9A60 9A60 57D8 D3C1 D3C1 D3C1 D3C1 5341 E2C1 D3C1 557B      E595BB       E595BB       F3B18DBC
9A61     9A61 9A61 9A61 9A61 57D9 D3C2 D3C2 D3C2 D3C2 5342 E2C2 D3C2 557E      E595BE       E595BE       F3B18DBD
9A62     9A62 9A62 9A62 9A62 57DA D3C3 D3C3 D3C3 D3C3 5343 E2C3 D3C3 5598      E59698       E59698       F3B18DBE
9A63     9A63 9A63 9A63 9A63 57DB D3C4 D3C4 D3C4 D3C4 5344 E2C4 D3C4 559E      E5969E       E5969E       F3B18DBF
9A64     9A64 9A64 9A64 9A64 57DC D3C5 D3C5 D3C5 D3C5 5345 E2C5 D3C5 55AE      E596AE       E596AE       F3B18E80
9A65     9A65 9A65 9A65 9A65 57DD D3C6 D3C6 D3C6 D3C6 5346 E2C6 D3C6 557C      E595BC       E595BC       F3B18E81
9A66     9A66 9A66 9A66 9A66 57DF D3C7 D3C7 D3C7 D3C7 5347 E2C7 D3C7 5583      E59683       E59683       F3B18E82
9A67     9A67 9A67 9A67 9A67 57E0 D3C8 D3C8 D3C8 D3C8 5348 E2C8 D3C8 55A9      E596A9       E596A9       F3B18E83
9A68     9A68 9A68 9A68 9A68 57E1 D3C9 D3C9 D3C9 D3C9 5349 E2C9 D3C9 5587      E59687       E59687       F3B18E84
9A69     9A69 9A69 9A69 9A69 57E2 D3CA D3CA D3CA D3CA 534A E2D1 D3CA 55A8      E596A8       E596A8       F3B18E85
9A6A     9A6A 9A6A 9A6A 9A6A 52A2 D3CB D3CB D3CB D3CB 534B E2D2 D3CB 55DA      E5979A       E5979A       F3B18E86
9A6B     9A6B 9A6B 9A6B 9A6B 57E3 D3CC D3CC D3CC D3CC 534C E2D3 D3CC 55C5      E59785       E59785       F3B18E87
9A6C     9A6C 9A6C 9A6C 9A6C 57E4 D3CD D3CD D3CD D3CD 534D E2D4 D3CD 55DF      E5979F       E5979F       F3B18E88
9A6D     9A6D 9A6D 9A6D 9A6D 57E5 D3CE D3CE D3CE D3CE 534E E2D5 D3CE 55C4      E59784       E59784       F3B18E89
9A6E     9A6E 9A6E 9A6E 9A6E 57E6 D3CF D3CF D3CF D3CF 534F E2D6 D3CF 55DC      E5979C       E5979C       F3B18E8A
9A6F     9A6F 9A6F 9A6F 9A6F 57E7 D3D0 D3D0 D3D0 D3D0 5350 E2D7 D3D0 55E4      E597A4       E597A4       F3B18E8B
9A70     9A70 9A70 9A70 9A70 57E8 D3D1 D3D1 D3D1 D3D1 5351 E2D8 D3D1 55D4      E59794       E59794       F3B18E8C
9A71     9A71 9A71 9A71 9A71 57EA D3D2 D3D2 D3D2 D3D2 5352 E2D9 D3D2 5614      E59894       E59894       F3B18E8D
9A72     9A72 9A72 9A72 9A72 57EB D3D3 D3D3 D3D3 D3D3 5353 E2E2 D3D3 55F7      E597B7       E597B7       F3B18E8E
9A73     9A73 9A73 9A73 9A73 57EC D3D4 D3D4 D3D4 D3D4 5354 E2E3 D3D4 5616      E59896       E59896       F3B18E8F
9A74     9A74 9A74 9A74 9A74 57ED D3D5 D3D5 D3D5 D3D5 5355 E2E4 D3D5 55FE      E597BE       E597BE       F3B18E90
9A75     9A75 9A75 9A75 9A75 57EE D3D6 D3D6 D3D6 D3D6 5356 E2E5 D3D6 55FD      E597BD       E597BD       F3B18E91
9A76     9A76 9A76 9A76 9A76 57EF D3D7 D3D7 D3D7 D3D7 5357 E2E6 D3D7 561B      E5989B       E5989B       F3B18E92
9A77     9A77 9A77 9A77 9A77 57E9 D3D8 D3D8 D3D8 D3D8 5358 E2E7 D3D8 55F9      E597B9       E597B9       F3B18E93
9A78     9A78 9A78 9A78 9A78 57F0 D3D9 D3D9 D3D9 D3D9 5359 E2E8 D3D9 564E      E5998E       E5998E       F3B18E94
9A79     9A79 9A79 9A79 9A79 57F1 D3DA D3DA D3DA D3DA 535A E2E9 D3DA 5650      E59990       E59990       F3B18E95
9A7A     9A7A 9A7A 9A7A 9A7A 57FB D3DB D3DB D3DB D3DB 535B E24A D3DB 71DF      E7879F       E7879F       F3B18E96
9A7B     9A7B 9A7B 9A7B 9A7B 547F D3DC D3DC D3DC D3DC 535C E25B D3DC 5634      E598B4       E598B4       F3B18E97
9A7C     9A7C 9A7C 9A7C 9A7C 57F2 D3DD D3DD D3DD D3DD 535D E25A D3DD 5636      E598B6       E598B6       F3B18E98
9A7D     9A7D 9A7D 9A7D 9A7D 57F3 D3DE D3DE D3DE D3DE 535E E25F D3DE 5632      E598B2       E598B2       F3B18E99
9A7E     9A7E 9A7E 9A7E 9A7E 57F4 D3DF D3DF D3DF D3DF 535F E26D D3DF 5638      E598B8       E598B8       F3B18E9A
9A80     9A80 9A80 9A80 9A80 57F5 D3E0 D3E0 D3E0 D3E0 5360 E279 D3E0 566B      E599AB       E599AB       F3B18E9B
9A81     9A81 9A81 9A81 9A81 57F6 D3E1 D3E1 D3E1 D3E1 5361 E257 D3E1 5664      E599A4       E599A4       F3B18E9C
9A82     9A82 9A82 9A82 9A82 57F8 D3E2 D3E2 D3E2 D3E2 5362 E259 D3E2 562F      E598AF       E598AF       F3B18E9D
9A83     9A83 9A83 9A83 9A83 57F9 D3E3 D3E3 D3E3 D3E3 5363 E262 D3E3 566C      E599AC       E599AC       F3B18E9E
9A84     9A84 9A84 9A84 9A84 57FA D3E4 D3E4 D3E4 D3E4 5364 E263 D3E4 566A      E599AA       E599AA       F3B18E9F
9A85     9A85 9A85 9A85 9A85 57F7 D3E5 D3E5 D3E5 D3E5 5365 E264 D3E5 5686      E59A86       E59A86       F3B18EA0
9A86     9A86 9A86 9A86 9A86 57FE D3E6 D3E6 D3E6 D3E6 5366 E265 D3E6 5680      E59A80       E59A80       F3B18EA1
9A87     9A87 9A87 9A87 9A87 5841 D3E7 D3E7 D3E7 D3E7 5367 E266 D3E7 568A      E59A8A       E59A8A       F3B18EA2
9A88     9A88 9A88 9A88 9A88 5842 D3E8 D3E8 D3E8 D3E8 5368 E267 D3E8 56A0      E59AA0       E59AA0       F3B18EA3
9A89     9A89 9A89 9A89 9A89 57FC D3E9 D3E9 D3E9 D3E9 5369 E268 D3E9 5694      E59A94       E59A94       F3B18EA4
9A8A     9A8A 9A8A 9A8A 9A8A 57FD D3EA D3EA D3EA D3EA 536A E269 D3EA 568F      E59A8F       E59A8F       F3B18EA5
9A8B     9A8B 9A8B 9A8B 9A8B 5843 D3EB D3EB D3EB D3EB 536B E270 D3EB 56A5      E59AA5       E59AA5       F3B18EA6
9A8C     9A8C 9A8C 9A8C 9A8C 5844 D3EC D3EC D3EC D3EC 536C E271 D3EC 56AE      E59AAE       E59AAE       F3B18EA7
9A8D     9A8D 9A8D 9A8D 9A8D 5845 D3ED D3ED D3ED D3ED 536D E272 D3ED 56B6      E59AB6       E59AB6       F3B18EA8
9A8E     9A8E 9A8E 9A8E 9A8E 5846 D3EE D3EE D3EE D3EE 536E E273 D3EE 56B4      E59AB4       E59AB4       F3B18EA9
9A8F     9A8F 9A8F 9A8F 9A8F 5848 D3EF D3EF D3EF D3EF 536F E274 D3EF 56C2      E59B82       E59B82       F3B18EAA
9A90     9A90 9A90 9A90 9A90 5849 D3F0 D3F0 D3F0 D3F0 5370 E275 D3F0 56BC      E59ABC       E59ABC       F3B18EAB
9A91     9A91 9A91 9A91 9A91 584A D3F1 D3F1 D3F1 D3F1 5371 E276 D3F1 56C1      E59B81       E59B81       F3B18EAC
9A92     9A92 9A92 9A92 9A92 584B D3F2 D3F2 D3F2 D3F2 5372 E277 D3F2 56C3      E59B83       E59B83       F3B18EAD
9A93     9A93 9A93 9A93 9A93 584C D3F3 D3F3 D3F3 D3F3 5373 E278 D3F3 56C0      E59B80       E59B80       F3B18EAE
9A94     9A94 9A94 9A94 9A94 5847 D3F4 D3F4 D3F4 D3F4 5374 E280 D3F4 56C8      E59B88       E59B88       F3B18EAF
9A95     9A95 9A95 9A95 9A95 584D D3F5 D3F5 D3F5 D3F5 5375 E28B D3F5 56CE      E59B8E       E59B8E       F3B18EB0
9A96     9A96 9A96 9A96 9A96 584F D3F6 D3F6 D3F6 D3F6 5376 E29B D3F6 56D1      E59B91       E59B91       F3B18EB1
9A97     9A97 9A97 9A97 9A97 584E D3F7 D3F7 D3F7 D3F7 5377 E29C D3F7 56D3      E59B93       E59B93       F3B18EB2
9A98     9A98 9A98 9A98 9A98 5850 D3F8 D3F8 D3F8 D3F8 5378 E2A0 D3F8 56D7      E59B97       E59B97       F3B18EB3
9A99     9A99 9A99 9A99 9A99 5851 D3F9 D3F9 D3F9 D3F9 5379 E2AB D3F9 56EE      E59BAE       E59BAE       F3B18EB4
9A9A     9A9A 9A9A 9A9A 9A9A 5852 D3FA D3FA D3FA D3FA 537A E2B0 D3FA 56F9      E59BB9       E59BB9       F3B18EB5
9A9B     9A9B 9A9B 9A9B 9A9B 5266 D3FB D3FB D3FB D3FB 537B E2C0 D3FB 5700      E59C80       E59C80       F3B18EB6
9A9C     9A9C 9A9C 9A9C 9A9C 5853 D3FC D3FC D3FC D3FC 537C E26A D3FC 56FF      E59BBF       E59BBF       F3B18EB7
9A9D     9A9D 9A9D 9A9D 9A9D 5854 D3FD D3FD D3FD D3FD 537D E2D0 D3FD 5704      E59C84       E59C84       F3B18EB8
9A9E     9A9E 9A9E 9A9E 9A9E 5855 D3FE D3FE D3FE D3FE 537E E2A1 D3FE 5709      E59C89       E59C89       F3B18EB9
9A9F     9A9F 9A9F 9A9F 9A9F 5856 D4A1 D4A1 D4A1 D4A1 5421 E34F D4A1 5708      E59C88       E59C88       F3B18EBA
9AA0     9AA0 9AA0 9AA0 9AA0 51BD D4A2 D4A2 D4A2 D4A2 5422 E37F D4A2 570B      E59C8B       E59C8B       F3B18EBB
9AA1     9AA1 9AA1 9AA1 9AA1 5857 D4A3 D4A3 D4A3 D4A3 5423 E37B D4A3 570D      E59C8D       E59C8D       F3B18EBC
9AA2     9AA2 9AA2 9AA2 9AA2 54E7 D4A4 D4A4 D4A4 D4A4 5424 E3E0 D4A4 5713      E59C93       E59C93       F3B18EBD
9AA3     9AA3 9AA3 9AA3 9AA3 5859 D4A5 D4A5 D4A5 D4A5 5425 E36C D4A5 5718      E59C98       E59C98       F3B18EBE
9AA4     9AA4 9AA4 9AA4 9AA4 585A D4A6 D4A6 D4A6 D4A6 5426 E350 D4A6 5716      E59C96       E59C96       F3B18EBF
9AA5     9AA5 9AA5 9AA5 9AA5 5858 D4A7 D4A7 D4A7 D4A7 5427 E37D D4A7 55C7      E59787       E59787       F3B18F80
9AA6     9AA6 9AA6 9AA6 9AA6 585B D4A8 D4A8 D4A8 D4A8 5428 E34D D4A8 571C      E59C9C       E59C9C       F3B18F81
9AA7     9AA7 9AA7 9AA7 9AA7 585C D4A9 D4A9 D4A9 D4A9 5429 E35D D4A9 5726      E59CA6       E59CA6       F3B18F82
9AA8     9AA8 9AA8 9AA8 9AA8 51D1 D4AA D4AA D4AA D4AA 542A E35C D4AA 5737      E59CB7       E59CB7       F3B18F83
9AA9     9AA9 9AA9 9AA9 9AA9 585D D4AB D4AB D4AB D4AB 542B E34E D4AB 5738      E59CB8       E59CB8       F3B18F84
9AAA     9AAA 9AAA 9AAA 9AAA 585E D4AC D4AC D4AC D4AC 542C E36B D4AC 574E      E59D8E       E59D8E       F3B18F85
9AAB     9AAB 9AAB 9AAB 9AAB 585F D4AD D4AD D4AD D4AD 542D E360 D4AD 573B      E59CBB       E59CBB       F3B18F86
9AAC     9AAC 9AAC 9AAC 9AAC 5861 D4AE D4AE D4AE D4AE 542E E34B D4AE 5740      E59D80       E59D80       F3B18F87
9AAD     9AAD 9AAD 9AAD 9AAD 5862 D4AF D4AF D4AF D4AF 542F E361 D4AF 574F      E59D8F       E59D8F       F3B18F88
9AAE     9AAE 9AAE 9AAE 9AAE 5398 D4B0 D4B0 D4B0 D4B0 5430 E3F0 D4B0 5769      E59DA9       E59DA9       F3B18F89
9AAF     9AAF 9AAF 9AAF 9AAF 5871 D4B1 D4B1 D4B1 D4B1 5431 E3F1 D4B1 57C0      E59F80       E59F80       F3B18F8A
9AB0     9AB0 9AB0 9AB0 9AB0 5864 D4B2 D4B2 D4B2 D4B2 5432 E3F2 D4B2 5788      E59E88       E59E88       F3B18F8B
9AB1     9AB1 9AB1 9AB1 9AB1 5865 D4B3 D4B3 D4B3 D4B3 5433 E3F3 D4B3 5761      E59DA1       E59DA1       F3B18F8C
9AB2     9AB2 9AB2 9AB2 9AB2 5866 D4B4 D4B4 D4B4 D4B4 5434 E3F4 D4B4 577F      E59DBF       E59DBF       F3B18F8D
9AB3     9AB3 9AB3 9AB3 9AB3 5867 D4B5 D4B5 D4B5 D4B5 5435 E3F5 D4B5 5789      E59E89       E59E89       F3B18F8E
9AB4     9AB4 9AB4 9AB4 9AB4 5868 D4B6 D4B6 D4B6 D4B6 5436 E3F6 D4B6 5793      E59E93       E59E93       F3B18F8F
9AB5     9AB5 9AB5 9AB5 9AB5 5869 D4B7 D4B7 D4B7 D4B7 5437 E3F7 D4B7 57A0      E59EA0       E59EA0       F3B18F90
9AB6     9AB6 9AB6 9AB6 9AB6 586B D4B8 D4B8 D4B8 D4B8 5438 E3F8 D4B8 57B3      E59EB3       E59EB3       F3B18F91
9AB7     9AB7 9AB7 9AB7 9AB7 586A D4B9 D4B9 D4B9 D4B9 5439 E3F9 D4B9 57A4      E59EA4       E59EA4       F3B18F92
9AB8     9AB8 9AB8 9AB8 9AB8 586D D4BA D4BA D4BA D4BA 543A E37A D4BA 57AA      E59EAA       E59EAA       F3B18F93
9AB9     9AB9 9AB9 9AB9 9AB9 53CD D4BB D4BB D4BB D4BB 543B E35E D4BB 57B0      E59EB0       E59EB0       F3B18F94
9ABA     9ABA 9ABA 9ABA 9ABA 586E D4BC D4BC D4BC D4BC 543C E34C D4BC 57C3      E59F83       E59F83       F3B18F95
9ABB     9ABB 9ABB 9ABB 9ABB 586F D4BD D4BD D4BD D4BD 543D E37E D4BD 57C6      E59F86       E59F86       F3B18F96
9ABC     9ABC 9ABC 9ABC 9ABC 5872 D4BE D4BE D4BE D4BE 543E E36E D4BE 57D4      E59F94       E59F94       F3B18F97
9ABD     9ABD 9ABD 9ABD 9ABD 5874 D4BF D4BF D4BF D4BF 543F E36F D4BF 57D2      E59F92       E59F92       F3B18F98
9ABE     9ABE 9ABE 9ABE 9ABE 5875 D4C0 D4C0 D4C0 D4C0 5440 E37C D4C0 57D3      E59F93       E59F93       F3B18F99
9ABF     9ABF 9ABF 9ABF 9ABF 5878 D4C1 D4C1 D4C1 D4C1 5441 E3C1 D4C1 580A      E5A08A       E5A08A       F3B18F9A
9AC0     9AC0 9AC0 9AC0 9AC0 5876 D4C2 D4C2 D4C2 D4C2 5442 E3C2 D4C2 57D6      E59F96       E59F96       F3B18F9B
9AC1     9AC1 9AC1 9AC1 9AC1 5879 D4C3 D4C3 D4C3 D4C3 5443 E3C3 D4C3 57E3      E59FA3       E59FA3       F3B18F9C
9AC2     9AC2 9AC2 9AC2 9AC2 587A D4C4 D4C4 D4C4 D4C4 5444 E3C4 D4C4 580B      E5A08B       E5A08B       F3B18F9D
9AC3     9AC3 9AC3 9AC3 9AC3 587B D4C5 D4C5 D4C5 D4C5 5445 E3C5 D4C5 5819      E5A099       E5A099       F3B18F9E
9AC4     9AC4 9AC4 9AC4 9AC4 53A7 D4C6 D4C6 D4C6 D4C6 5446 E3C6 D4C6 581D      E5A09D       E5A09D       F3B18F9F
9AC5     9AC5 9AC5 9AC5 9AC5 5886 D4C7 D4C7 D4C7 D4C7 5447 E3C7 D4C7 5872      E5A1B2       E5A1B2       F3B18FA0
9AC6     9AC6 9AC6 9AC6 9AC6 587C D4C8 D4C8 D4C8 D4C8 5448 E3C8 D4C8 5821      E5A0A1       E5A0A1       F3B18FA1
9AC7     9AC7 9AC7 9AC7 9AC7 587E D4C9 D4C9 D4C9 D4C9 5449 E3C9 D4C9 5862      E5A1A2       E5A1A2       F3B18FA2
9AC8     9AC8 9AC8 9AC8 9AC8 587D D4CA D4CA D4CA D4CA 544A E3D1 D4CA 584B      E5A18B       E5A18B       F3B18FA3
9AC9     9AC9 9AC9 9AC9 9AC9 5883 D4CB D4CB D4CB D4CB 544B E3D2 D4CB 5870      E5A1B0       E5A1B0       F3B18FA4
9ACA     9ACA 9ACA 9ACA 9ACA 587F D4CC D4CC D4CC D4CC 544C E3D3 D4CC 6BC0      E6AF80       E6AF80       F3B18FA5
9ACB     9ACB 9ACB 9ACB 9ACB 5881 D4CD D4CD D4CD D4CD 544D E3D4 D4CD 5852      E5A192       E5A192       F3B18FA6
9ACC     9ACC 9ACC 9ACC 9ACC 5880 D4CE D4CE D4CE D4CE 544E E3D5 D4CE 583D      E5A0BD       E5A0BD       F3B18FA7
9ACD     9ACD 9ACD 9ACD 9ACD 5884 D4CF D4CF D4CF D4CF 544F E3D6 D4CF 5879      E5A1B9       E5A1B9       F3B18FA8
9ACE     9ACE 9ACE 9ACE 9ACE 5885 D4D0 D4D0 D4D0 D4D0 5450 E3D7 D4D0 5885      E5A285       E5A285       F3B18FA9
9ACF     9ACF 9ACF 9ACF 9ACF 588D D4D1 D4D1 D4D1 D4D1 5451 E3D8 D4D1 58B9      E5A2B9       E5A2B9       F3B18FAA
9AD0     9AD0 9AD0 9AD0 9AD0 5887 D4D2 D4D2 D4D2 D4D2 5452 E3D9 D4D2 589F      E5A29F       E5A29F       F3B18FAB
9AD1     9AD1 9AD1 9AD1 9AD1 5888 D4D3 D4D3 D4D3 D4D3 5453 E3E2 D4D3 58AB      E5A2AB       E5A2AB       F3B18FAC
9AD2     9AD2 9AD2 9AD2 9AD2 588E D4D4 D4D4 D4D4 D4D4 5454 E3E3 D4D4 58BA      E5A2BA       E5A2BA       F3B18FAD
9AD3     9AD3 9AD3 9AD3 9AD3 5895 D4D5 D4D5 D4D5 D4D5 5455 E3E4 D4D5 58DE      E5A39E       E5A39E       F3B18FAE
9AD4     9AD4 9AD4 9AD4 9AD4 5552 D4D6 D4D6 D4D6 D4D6 5456 E3E5 D4D6 58BB      E5A2BB       E5A2BB       F3B18FAF
9AD5     9AD5 9AD5 9AD5 9AD5 5889 D4D7 D4D7 D4D7 D4D7 5457 E3E6 D4D7 58B8      E5A2B8       E5A2B8       F3B18FB0
9AD6     9AD6 9AD6 9AD6 9AD6 588B D4D8 D4D8 D4D8 D4D8 5458 E3E7 D4D8 58AE      E5A2AE       E5A2AE       F3B18FB1
9AD7     9AD7 9AD7 9AD7 9AD7 588F D4D9 D4D9 D4D9 D4D9 5459 E3E8 D4D9 58C5      E5A385       E5A385       F3B18FB2
9AD8     9AD8 9AD8 9AD8 9AD8 5890 D4DA D4DA D4DA D4DA 545A E3E9 D4DA 58D3      E5A393       E5A393       F3B18FB3
9AD9     9AD9 9AD9 9AD9 9AD9 5891 D4DB D4DB D4DB D4DB 545B E34A D4DB 58D1      E5A391       E5A391       F3B18FB4
9ADA     9ADA 9ADA 9ADA 9ADA 5892 D4DC D4DC D4DC D4DC 545C E35B D4DC 58D7      E5A397       E5A397       F3B18FB5
9ADB     9ADB 9ADB 9ADB 9ADB 5893 D4DD D4DD D4DD D4DD 545D E35A D4DD 58D9      E5A399       E5A399       F3B18FB6
9ADC     9ADC 9ADC 9ADC 9ADC 5894 D4DE D4DE D4DE D4DE 545E E35F D4DE 58D8      E5A398       E5A398       F3B18FB7
9ADD     9ADD 9ADD 9ADD 9ADD 5899 D4DF D4DF D4DF D4DF 545F E36D D4DF 58E5      E5A3A5       E5A3A5       F3B18FB8
9ADE     9ADE 9ADE 9ADE 9ADE 5896 D4E0 D4E0 D4E0 D4E0 5460 E379 D4E0 58DC      E5A39C       E5A39C       F3B18FB9
9ADF     9ADF 9ADF 9ADF 9ADF 5898 D4E1 D4E1 D4E1 D4E1 5461 E357 D4E1 58E4      E5A3A4       E5A3A4       F3B18FBA
9AE0     9AE0 9AE0 9AE0 9AE0 5897 D4E2 D4E2 D4E2 D4E2 5462 E359 D4E2 58DF      E5A39F       E5A39F       F3B18FBB
9AE1     9AE1 9AE1 9AE1 9AE1 589A D4E3 D4E3 D4E3 D4E3 5463 E362 D4E3 58EF      E5A3AF       E5A3AF       F3B18FBC
9AE2     9AE2 9AE2 9AE2 9AE2 5077 C4DB D4E4 D4E4 C4DB 445B C44A D4E4 58FA      E5A3BA       E5A3BA       F3B18FBD
9AE3     9AE3 9AE3 9AE3 9AE3 589C D4E5 D4E5 D4E5 D4E5 5465 E364 D4E5 58F9      E5A3B9       E5A3B9       F3B18FBE
9AE4     9AE4 9AE4 9AE4 9AE4 589D D4E6 D4E6 D4E6 D4E6 5466 E365 D4E6 58FB      E5A3BB       E5A3BB       F3B18FBF
9AE5     9AE5 9AE5 9AE5 9AE5 589E D4E7 D4E7 D4E7 D4E7 5467 E366 D4E7 58FC      E5A3BC       E5A3BC       F3B19080
9AE6     9AE6 9AE6 9AE6 9AE6 5275 D4E8 D4E8 D4E8 D4E8 5468 E367 D4E8 58FD      E5A3BD       E5A3BD       F3B19081
9AE7     9AE7 9AE7 9AE7 9AE7 589F D4E9 D4E9 D4E9 D4E9 5469 E368 D4E9 5902      E5A482       E5A482       F3B19082
9AE8     9AE8 9AE8 9AE8 9AE8 58A0 D4EA D4EA D4EA D4EA 546A E369 D4EA 590A      E5A48A       E5A48A       F3B19083
9AE9     9AE9 9AE9 9AE9 9AE9 58A2 D4EB D4EB D4EB D4EB 546B E370 D4EB 5910      E5A490       E5A490       F3B19084
9AEA     9AEA 9AEA 9AEA 9AEA 58A3 D4EC D4EC D4EC D4EC 546C E371 D4EC 591B      E5A49B       E5A49B       F3B19085
9AEB     9AEB 9AEB 9AEB 9AEB 58A4 D4ED D4ED D4ED D4ED 546D E372 D4ED 68A6      E6A2A6       E6A2A6       F3B19086
9AEC     9AEC 9AEC 9AEC 9AEC 58A5 D4EE D4EE D4EE D4EE 546E E373 D4EE 5925      E5A4A5       E5A4A5       F3B19087
9AED     9AED 9AED 9AED 9AED 58A6 D4EF D4EF D4EF D4EF 546F E374 D4EF 592C      E5A4AC       E5A4AC       F3B19088
9AEE     9AEE 9AEE 9AEE 9AEE 58A7 D4F0 D4F0 D4F0 D4F0 5470 E375 D4F0 592D      E5A4AD       E5A4AD       F3B19089
9AEF     9AEF 9AEF 9AEF 9AEF 58A8 D4F1 D4F1 D4F1 D4F1 5471 E376 D4F1 5932      E5A4B2       E5A4B2       F3B1908A
9AF0     9AF0 9AF0 9AF0 9AF0 58A9 D4F2 D4F2 D4F2 D4F2 5472 E377 D4F2 5938      E5A4B8       E5A4B8       F3B1908B
9AF1     9AF1 9AF1 9AF1 9AF1 58AA D4F3 D4F3 D4F3 D4F3 5473 E378 D4F3 593E      E5A4BE       E5A4BE       F3B1908C
9AF2     9AF2 9AF2 9AF2 9AF2 604E D4F4 D4F4 D4F4 D4F4 5474 E380 D4F4 7AD2      E7AB92       E7AB92       F3B1908D
9AF3     9AF3 9AF3 9AF3 9AF3 58AB D4F5 D4F5 D4F5 D4F5 5475 E38B D4F5 5955      E5A595       E5A595       F3B1908E
9AF4     9AF4 9AF4 9AF4 9AF4 58AC D4F6 D4F6 D4F6 D4F6 5476 E39B D4F6 5950      E5A590       E5A590       F3B1908F
9AF5     9AF5 9AF5 9AF5 9AF5 518F D4F7 D4F7 D4F7 D4F7 5477 E39C D4F7 594E      E5A58E       E5A58E       F3B19090
9AF6     9AF6 9AF6 9AF6 9AF6 58AE D4F8 D4F8 D4F8 D4F8 5478 E3A0 D4F8 595A      E5A59A       E5A59A       F3B19091
9AF7     9AF7 9AF7 9AF7 9AF7 58AF D4F9 D4F9 D4F9 D4F9 5479 E3AB D4F9 5958      E5A598       E5A598       F3B19092
9AF8     9AF8 9AF8 9AF8 9AF8 58B3 D4FA D4FA D4FA D4FA 547A E3B0 D4FA 5962      E5A5A2       E5A5A2       F3B19093
9AF9     9AF9 9AF9 9AF9 9AF9 58B4 D4FB D4FB D4FB D4FB 547B E3C0 D4FB 5960      E5A5A0       E5A5A0       F3B19094
9AFA     9AFA 9AFA 9AFA 9AFA 58B5 D4FC D4FC D4FC D4FC 547C E36A D4FC 5967      E5A5A7       E5A5A7       F3B19095
9AFB     9AFB 9AFB 9AFB 9AFB 58B6 D4FD D4FD D4FD D4FD 547D E3D0 D4FD 596C      E5A5AC       E5A5AC       F3B19096
9AFC     9AFC 9AFC 9AFC 9AFC 58B7 D4FE D4FE D4FE D4FE 547E E3A1 D4FE 5969      E5A5A9       E5A5A9       F3B19097
9B40     9B40 9B40 9B40 9B40 58B8 D5A1 D5A1 D5A1 D5A1 5521 E44F D5A1 5978      E5A5B8       E5A5B8       F3B19098
9B41     9B41 9B41 9B41 9B41 58B9 D5A2 D5A2 D5A2 D5A2 5522 E47F D5A2 5981      E5A681       E5A681       F3B19099
9B42     9B42 9B42 9B42 9B42 58BC D5A3 D5A3 D5A3 D5A3 5523 E47B D5A3 599D      E5A69D       E5A69D       F3B1909A
9B43     9B43 9B43 9B43 9B43 566D D5A4 D5A4 D5A4 D5A4 5524 E4E0 D5A4 4F5E      E4BD9E       E4BD9E       F3B1909B
9B44     9B44 9B44 9B44 9B44 5678 D5A5 D5A5 D5A5 D5A5 5525 E46C D5A5 4FAB      E4BEAB       E4BEAB       F3B1909C
9B45     9B45 9B45 9B45 9B45 58BD D5A6 D5A6 D5A6 D5A6 5526 E450 D5A6 59A3      E5A6A3       E5A6A3       F3B1909D
9B46     9B46 9B46 9B46 9B46 58BF D5A7 D5A7 D5A7 D5A7 5527 E47D D5A7 59B2      E5A6B2       E5A6B2       F3B1909E
9B47     9B47 9B47 9B47 9B47 58C1 D5A8 D5A8 D5A8 D5A8 5528 E44D D5A8 59C6      E5A786       E5A786       F3B1909F
9B48     9B48 9B48 9B48 9B48 58C2 D5A9 D5A9 D5A9 D5A9 5529 E45D D5A9 59E8      E5A7A8       E5A7A8       F3B190A0
9B49     9B49 9B49 9B49 9B49 5184 D5AA D5AA D5AA D5AA 552A E45C D5AA 59DC      E5A79C       E5A79C       F3B190A1
9B4A     9B4A 9B4A 9B4A 9B4A 58BA D5AB D5AB D5AB D5AB 552B E44E D5AB 598D      E5A68D       E5A68D       F3B190A2
9B4B     9B4B 9B4B 9B4B 9B4B 58C3 D5AC D5AC D5AC D5AC 552C E46B D5AC 59D9      E5A799       E5A799       F3B190A3
9B4C     9B4C 9B4C 9B4C 9B4C 58C4 D5AD D5AD D5AD D5AD 552D E460 D5AD 59DA      E5A79A       E5A79A       F3B190A4
9B4D     9B4D 9B4D 9B4D 9B4D 58C5 D5AE D5AE D5AE D5AE 552E E44B D5AE 5A25      E5A8A5       E5A8A5       F3B190A5
9B4E     9B4E 9B4E 9B4E 9B4E 58C6 D5AF D5AF D5AF D5AF 552F E461 D5AF 5A1F      E5A89F       E5A89F       F3B190A6
9B4F     9B4F 9B4F 9B4F 9B4F 58C7 D5B0 D5B0 D5B0 D5B0 5530 E4F0 D5B0 5A11      E5A891       E5A891       F3B190A7
9B50     9B50 9B50 9B50 9B50 58C8 D5B1 D5B1 D5B1 D5B1 5531 E4F1 D5B1 5A1C      E5A89C       E5A89C       F3B190A8
9B51     9B51 9B51 9B51 9B51 58CA D5B2 D5B2 D5B2 D5B2 5532 E4F2 D5B2 5A09      E5A889       E5A889       F3B190A9
9B52     9B52 9B52 9B52 9B52 58C9 D5B3 D5B3 D5B3 D5B3 5533 E4F3 D5B3 5A1A      E5A89A       E5A89A       F3B190AA
9B53     9B53 9B53 9B53 9B53 58CB D5B4 D5B4 D5B4 D5B4 5534 E4F4 D5B4 5A40      E5A980       E5A980       F3B190AB
9B54     9B54 9B54 9B54 9B54 58CC D5B5 D5B5 D5B5 D5B5 5535 E4F5 D5B5 5A6C      E5A9AC       E5A9AC       F3B190AC
9B55     9B55 9B55 9B55 9B55 58CD D5B6 D5B6 D5B6 D5B6 5536 E4F6 D5B6 5A49      E5A989       E5A989       F3B190AD
9B56     9B56 9B56 9B56 9B56 58CE D5B7 D5B7 D5B7 D5B7 5537 E4F7 D5B7 5A35      E5A8B5       E5A8B5       F3B190AE
9B57     9B57 9B57 9B57 9B57 58CF D5B8 D5B8 D5B8 D5B8 5538 E4F8 D5B8 5A36      E5A8B6       E5A8B6       F3B190AF
9B58     9B58 9B58 9B58 9B58 58D0 D5B9 D5B9 D5B9 D5B9 5539 E4F9 D5B9 5A62      E5A9A2       E5A9A2       F3B190B0
9B59     9B59 9B59 9B59 9B59 58D1 D5BA D5BA D5BA D5BA 553A E47A D5BA 5A6A      E5A9AA       E5A9AA       F3B190B1
9B5A     9B5A 9B5A 9B5A 9B5A 58D2 D5BB D5BB D5BB D5BB 553B E45E D5BB 5A9A      E5AA9A       E5AA9A       F3B190B2
9B5B     9B5B 9B5B 9B5B 9B5B 58D3 D5BC D5BC D5BC D5BC 553C E44C D5BC 5ABC      E5AABC       E5AABC       F3B190B3
9B5C     9B5C 9B5C 9B5C 9B5C 58D4 D5BD D5BD D5BD D5BD 553D E47E D5BD 5ABE      E5AABE       E5AABE       F3B190B4
9B5D     9B5D 9B5D 9B5D 9B5D 58D6 D5BE D5BE D5BE D5BE 553E E46E D5BE 5ACB      E5AB8B       E5AB8B       F3B190B5
9B5E     9B5E 9B5E 9B5E 9B5E 58D7 D5BF D5BF D5BF D5BF 553F E46F D5BF 5AC2      E5AB82       E5AB82       F3B190B6
9B5F     9B5F 9B5F 9B5F 9B5F 58D8 D5C0 D5C0 D5C0 D5C0 5540 E47C D5C0 5ABD      E5AABD       E5AABD       F3B190B7
9B60     9B60 9B60 9B60 9B60 58D9 D5C1 D5C1 D5C1 D5C1 5541 E4C1 D5C1 5AE3      E5ABA3       E5ABA3       F3B190B8
9B61     9B61 9B61 9B61 9B61 58DA D5C2 D5C2 D5C2 D5C2 5542 E4C2 D5C2 5AD7      E5AB97       E5AB97       F3B190B9
9B62     9B62 9B62 9B62 9B62 58DB D5C3 D5C3 D5C3 D5C3 5543 E4C3 D5C3 5AE6      E5ABA6       E5ABA6       F3B190BA
9B63     9B63 9B63 9B63 9B63 58DC D5C4 D5C4 D5C4 D5C4 5544 E4C4 D5C4 5AE9      E5ABA9       E5ABA9       F3B190BB
9B64     9B64 9B64 9B64 9B64 58DD D5C5 D5C5 D5C5 D5C5 5545 E4C5 D5C5 5AD6      E5AB96       E5AB96       F3B190BC
9B65     9B65 9B65 9B65 9B65 58DE D5C6 D5C6 D5C6 D5C6 5546 E4C6 D5C6 5AFA      E5ABBA       E5ABBA       F3B190BD
9B66     9B66 9B66 9B66 9B66 58DF D5C7 D5C7 D5C7 D5C7 5547 E4C7 D5C7 5AFB      E5ABBB       E5ABBB       F3B190BE
9B67     9B67 9B67 9B67 9B67 58E0 D5C8 D5C8 D5C8 D5C8 5548 E4C8 D5C8 5B0C      E5AC8C       E5AC8C       F3B190BF
9B68     9B68 9B68 9B68 9B68 58E1 D5C9 D5C9 D5C9 D5C9 5549 E4C9 D5C9 5B0B      E5AC8B       E5AC8B       F3B19180
9B69     9B69 9B69 9B69 9B69 58E2 D5CA D5CA D5CA D5CA 554A E4D1 D5CA 5B16      E5AC96       E5AC96       F3B19181
9B6A     9B6A 9B6A 9B6A 9B6A 58E3 D5CB D5CB D5CB D5CB 554B E4D2 D5CB 5B32      E5ACB2       E5ACB2       F3B19182
9B6B     9B6B 9B6B 9B6B 9B6B 58D5 D5CC D5CC D5CC D5CC 554C E4D3 D5CC 5AD0      E5AB90       E5AB90       F3B19183
9B6C     9B6C 9B6C 9B6C 9B6C 58E4 D5CD D5CD D5CD D5CD 554D E4D4 D5CD 5B2A      E5ACAA       E5ACAA       F3B19184
9B6D     9B6D 9B6D 9B6D 9B6D 58E5 D5CE D5CE D5CE D5CE 554E E4D5 D5CE 5B36      E5ACB6       E5ACB6       F3B19185
9B6E     9B6E 9B6E 9B6E 9B6E 58E6 D5CF D5CF D5CF D5CF 554F E4D6 D5CF 5B3E      E5ACBE       E5ACBE       F3B19186
9B6F     9B6F 9B6F 9B6F 9B6F 58E7 D5D0 D5D0 D5D0 D5D0 5550 E4D7 D5D0 5B43      E5AD83       E5AD83       F3B19187
9B70     9B70 9B70 9B70 9B70 58E8 D5D1 D5D1 D5D1 D5D1 5551 E4D8 D5D1 5B45      E5AD85       E5AD85       F3B19188
9B71     9B71 9B71 9B71 9B71 58E9 D5D2 D5D2 D5D2 D5D2 5552 E4D9 D5D2 5B40      E5AD80       E5AD80       F3B19189
9B72     9B72 9B72 9B72 9B72 58EA D5D3 D5D3 D5D3 D5D3 5553 E4E2 D5D3 5B51      E5AD91       E5AD91       F3B1918A
9B73     9B73 9B73 9B73 9B73 58EB D5D4 D5D4 D5D4 D5D4 5554 E4E3 D5D4 5B55      E5AD95       E5AD95       F3B1918B
9B74     9B74 9B74 9B74 9B74 5258 D5D5 D5D5 D5D5 D5D5 5555 E4E4 D5D5 5B5A      E5AD9A       E5AD9A       F3B1918C
9B75     9B75 9B75 9B75 9B75 58EE D5D6 D5D6 D5D6 D5D6 5556 E4E5 D5D6 5B5B      E5AD9B       E5AD9B       F3B1918D
9B76     9B76 9B76 9B76 9B76 58EF D5D7 D5D7 D5D7 D5D7 5557 E4E6 D5D7 5B65      E5ADA5       E5ADA5       F3B1918E
9B77     9B77 9B77 9B77 9B77 58F0 D5D8 D5D8 D5D8 D5D8 5558 E4E7 D5D8 5B69      E5ADA9       E5ADA9       F3B1918F
9B78     9B78 9B78 9B78 9B78 58F1 D5D9 D5D9 D5D9 D5D9 5559 E4E8 D5D9 5B70      E5ADB0       E5ADB0       F3B19190
9B79     9B79 9B79 9B79 9B79 58F2 D5DA D5DA D5DA D5DA 555A E4E9 D5DA 5B73      E5ADB3       E5ADB3       F3B19191
9B7A     9B7A 9B7A 9B7A 9B7A 58F3 D5DB D5DB D5DB D5DB 555B E44A D5DB 5B75      E5ADB5       E5ADB5       F3B19192
9B7B     9B7B 9B7B 9B7B 9B7B 58F4 D5DC D5DC D5DC D5DC 555C E45B D5DC 5B78      E5ADB8       E5ADB8       F3B19193
9B7C     9B7C 9B7C 9B7C 9B7C 58ED D5DD D5DD D5DD D5DD 555D E45A D5DD 6588      E69688       E69688       F3B19194
9B7D     9B7D 9B7D 9B7D 9B7D 58F5 D5DE D5DE D5DE D5DE 555E E45F D5DE 5B7A      E5ADBA       E5ADBA       F3B19195
9B7E     9B7E 9B7E 9B7E 9B7E 58F6 D5DF D5DF D5DF D5DF 555F E46D D5DF 5B80      E5AE80       E5AE80       F3B19196
9B80     9B80 9B80 9B80 9B80 58F7 D5E0 D5E0 D5E0 D5E0 5560 E479 D5E0 5B83      E5AE83       E5AE83       F3B19197
9B81     9B81 9B81 9B81 9B81 58F8 D5E1 D5E1 D5E1 D5E1 5561 E457 D5E1 5BA6      E5AEA6       E5AEA6       F3B19198
9B82     9B82 9B82 9B82 9B82 58F9 D5E2 D5E2 D5E2 D5E2 5562 E459 D5E2 5BB8      E5AEB8       E5AEB8       F3B19199
9B83     9B83 9B83 9B83 9B83 58FA D5E3 D5E3 D5E3 D5E3 5563 E462 D5E3 5BC3      E5AF83       E5AF83       F3B1919A
9B84     9B84 9B84 9B84 9B84 58FB D5E4 D5E4 D5E4 D5E4 5564 E463 D5E4 5BC7      E5AF87       E5AF87       F3B1919B
9B85     9B85 9B85 9B85 9B85 58FD D5E5 D5E5 D5E5 D5E5 5565 E464 D5E5 5BC9      E5AF89       E5AF89       F3B1919C
9B86     9B86 9B86 9B86 9B86 5244 D5E6 D5E6 D5E6 D5E6 5566 E465 D5E6 5BD4      E5AF94       E5AF94       F3B1919D
9B87     9B87 9B87 9B87 9B87 5941 D5E7 D5E7 D5E7 D5E7 5567 E466 D5E7 5BD0      E5AF90       E5AF90       F3B1919E
9B88     9B88 9B88 9B88 9B88 5945 D5E8 D5E8 D5E8 D5E8 5568 E467 D5E8 5BE4      E5AFA4       E5AFA4       F3B1919F
9B89     9B89 9B89 9B89 9B89 51FC D5E9 D5E9 D5E9 D5E9 5569 E468 D5E9 5BE6      E5AFA6       E5AFA6       F3B191A0
9B8A     9B8A 9B8A 9B8A 9B8A 5946 D5EA D5EA D5EA D5EA 556A E469 D5EA 5BE2      E5AFA2       E5AFA2       F3B191A1
9B8B     9B8B 9B8B 9B8B 9B8B 5943 D5EB D5EB D5EB D5EB 556B E470 D5EB 5BDE      E5AF9E       E5AF9E       F3B191A2
9B8C     9B8C 9B8C 9B8C 9B8C 5947 D5EC D5EC D5EC D5EC 556C E471 D5EC 5BE5      E5AFA5       E5AFA5       F3B191A3
9B8D     9B8D 9B8D 9B8D 9B8D 5948 D5ED D5ED D5ED D5ED 556D E472 D5ED 5BEB      E5AFAB       E5AFAB       F3B191A4
9B8E     9B8E 9B8E 9B8E 9B8E 5949 D5EE D5EE D5EE D5EE 556E E473 D5EE 5BF0      E5AFB0       E5AFB0       F3B191A5
9B8F     9B8F 9B8F 9B8F 9B8F 594B D5EF D5EF D5EF D5EF 556F E474 D5EF 5BF6      E5AFB6       E5AFB6       F3B191A6
9B90     9B90 9B90 9B90 9B90 594A D5F0 D5F0 D5F0 D5F0 5570 E475 D5F0 5BF3      E5AFB3       E5AFB3       F3B191A7
9B91     9B91 9B91 9B91 9B91 594C D5F1 D5F1 D5F1 D5F1 5571 E476 D5F1 5C05      E5B085       E5B085       F3B191A8
9B92     9B92 9B92 9B92 9B92 594D D5F2 D5F2 D5F2 D5F2 5572 E477 D5F2 5C07      E5B087       E5B087       F3B191A9
9B93     9B93 9B93 9B93 9B93 594E D5F3 D5F3 D5F3 D5F3 5573 E478 D5F3 5C08      E5B088       E5B088       F3B191AA
9B94     9B94 9B94 9B94 9B94 594F D5F4 D5F4 D5F4 D5F4 5574 E480 D5F4 5C0D      E5B08D       E5B08D       F3B191AB
9B95     9B95 9B95 9B95 9B95 5950 D5F5 D5F5 D5F5 D5F5 5575 E48B D5F5 5C13      E5B093       E5B093       F3B191AC
9B96     9B96 9B96 9B96 9B96 5952 D5F6 D5F6 D5F6 D5F6 5576 E49B D5F6 5C20      E5B0A0       E5B0A0       F3B191AD
9B97     9B97 9B97 9B97 9B97 5953 D5F7 D5F7 D5F7 D5F7 5577 E49C D5F7 5C22      E5B0A2       E5B0A2       F3B191AE
9B98     9B98 9B98 9B98 9B98 5954 D5F8 D5F8 D5F8 D5F8 5578 E4A0 D5F8 5C28      E5B0A8       E5B0A8       F3B191AF
9B99     9B99 9B99 9B99 9B99 5955 D5F9 D5F9 D5F9 D5F9 5579 E4AB D5F9 5C38      E5B0B8       E5B0B8       F3B191B0
9B9A     9B9A 9B9A 9B9A 9B9A 515D D5FA D5FA D5FA D5FA 557A E4B0 D5FA 5C39      E5B0B9       E5B0B9       F3B191B1
9B9B     9B9B 9B9B 9B9B 9B9B 5956 D5FB D5FB D5FB D5FB 557B E4C0 D5FB 5C41      E5B181       E5B181       F3B191B2
9B9C     9B9C 9B9C 9B9C 9B9C 5957 D5FC D5FC D5FC D5FC 557C E46A D5FC 5C46      E5B186       E5B186       F3B191B3
9B9D     9B9D 9B9D 9B9D 9B9D 5958 D5FD D5FD D5FD D5FD 557D E4D0 D5FD 5C4E      E5B18E       E5B18E       F3B191B4
9B9E     9B9E 9B9E 9B9E 9B9E 5959 D5FE D5FE D5FE D5FE 557E E4A1 D5FE 5C53      E5B193       E5B193       F3B191B5
9B9F     9B9F 9B9F 9B9F 9B9F 595A D6A1 D6A1 D6A1 D6A1 5621 E54F D6A1 5C50      E5B190       E5B190       F3B191B6
9BA0     9BA0 9BA0 9BA0 9BA0 5443 D6A2 D6A2 D6A2 4AE7 B8E7 88B9 D6A2 5C4F      E5B18F       E5B18F       F3B191B7
9BA1     9BA1 9BA1 9BA1 9BA1 595B D6A3 D6A3 D6A3 D6A3 5623 E57B D6A3 5B71      E5ADB1       E5ADB1       F3B191B8
9BA2     9BA2 9BA2 9BA2 9BA2 595C D6A4 D6A4 D6A4 D6A4 5624 E5E0 D6A4 5C6C      E5B1AC       E5B1AC       F3B191B9
9BA3     9BA3 9BA3 9BA3 9BA3 595D D6A5 D6A5 D6A5 D6A5 5625 E56C D6A5 5C6E      E5B1AE       E5B1AE       F3B191BA
9BA4     9BA4 9BA4 9BA4 9BA4 564C D6A6 D6A6 D6A6 D6A6 5626 E550 D6A6 4E62      E4B9A2       E4B9A2       F3B191BB
9BA5     9BA5 9BA5 9BA5 9BA5 595E D6A7 D6A7 D6A7 D6A7 5627 E57D D6A7 5C76      E5B1B6       E5B1B6       F3B191BC
9BA6     9BA6 9BA6 9BA6 9BA6 595F D6A8 D6A8 D6A8 D6A8 5628 E54D D6A8 5C79      E5B1B9       E5B1B9       F3B191BD
9BA7     9BA7 9BA7 9BA7 9BA7 5960 D6A9 D6A9 D6A9 D6A9 5629 E55D D6A9 5C8C      E5B28C       E5B28C       F3B191BE
9BA8     9BA8 9BA8 9BA8 9BA8 525A D6AA D6AA D6AA D6AA 562A E55C D6AA 5C91      E5B291       E5B291       F3B191BF
9BA9     9BA9 9BA9 9BA9 9BA9 5961 D6AB D6AB D6AB D6AB 562B E54E D6AB 5C94      E5B294       E5B294       F3B19280
9BAA     9BAA 9BAA 9BAA 9BAA 58BB D6AC D6AC D6AC D6AC 562C E56B D6AC 599B      E5A69B       E5A69B       F3B19281
9BAB     9BAB 9BAB 9BAB 9BAB 5963 D6AD D6AD D6AD D6AD 562D E560 D6AD 5CAB      E5B2AB       E5B2AB       F3B19282
9BAC     9BAC 9BAC 9BAC 9BAC 5964 D6AE D6AE D6AE D6AE 562E E54B D6AE 5CBB      E5B2BB       E5B2BB       F3B19283
9BAD     9BAD 9BAD 9BAD 9BAD 5965 D6AF D6AF D6AF D6AF 562F E561 D6AF 5CB6      E5B2B6       E5B2B6       F3B19284
9BAE     9BAE 9BAE 9BAE 9BAE 596A D6B0 D6B0 D6B0 D6B0 5630 E5F0 D6B0 5CBC      E5B2BC       E5B2BC       F3B19285
9BAF     9BAF 9BAF 9BAF 9BAF 5966 D6B1 D6B1 D6B1 D6B1 5631 E5F1 D6B1 5CB7      E5B2B7       E5B2B7       F3B19286
9BB0     9BB0 9BB0 9BB0 9BB0 5969 D6B2 D6B2 D6B2 D6B2 5632 E5F2 D6B2 5CC5      E5B385       E5B385       F3B19287
9BB1     9BB1 9BB1 9BB1 9BB1 5962 D6B3 D6B3 D6B3 D6B3 5633 E5F3 D6B3 5CBE      E5B2BE       E5B2BE       F3B19288
9BB2     9BB2 9BB2 9BB2 9BB2 596B D6B4 D6B4 D6B4 D6B4 5634 E5F4 D6B4 5CC7      E5B387       E5B387       F3B19289
9BB3     9BB3 9BB3 9BB3 9BB3 596C D6B5 D6B5 D6B5 D6B5 5635 E5F5 D6B5 5CD9      E5B399       E5B399       F3B1928A
9BB4     9BB4 9BB4 9BB4 9BB4 596D D6B6 D6B6 D6B6 D6B6 5636 E5F6 D6B6 5CE9      E5B3A9       E5B3A9       F3B1928B
9BB5     9BB5 9BB5 9BB5 9BB5 596E D6B7 D6B7 D6B7 D6B7 5637 E5F7 D6B7 5CFD      E5B3BD       E5B3BD       F3B1928C
9BB6     9BB6 9BB6 9BB6 9BB6 596F D6B8 D6B8 D6B8 D6B8 5638 E5F8 D6B8 5CFA      E5B3BA       E5B3BA       F3B1928D
9BB7     9BB7 9BB7 9BB7 9BB7 5971 D6B9 D6B9 D6B9 D6B9 5639 E5F9 D6B9 5CED      E5B3AD       E5B3AD       F3B1928E
9BB8     9BB8 9BB8 9BB8 9BB8 598C D6BA D6BA D6BA D6BA 563A E57A D6BA 5D8C      E5B68C       E5B68C       F3B1928F
9BB9     9BB9 9BB9 9BB9 9BB9 5972 D6BB D6BB D6BB D6BB 563B E55E D6BB 5CEA      E5B3AA       E5B3AA       F3B19290
9BBA     9BBA 9BBA 9BBA 9BBA 5973 D6BC D6BC D6BC D6BC 563C E54C D6BC 5D0B      E5B48B       E5B48B       F3B19291
9BBB     9BBB 9BBB 9BBB 9BBB 5974 D6BD D6BD D6BD D6BD 563D E57E D6BD 5D15      E5B495       E5B495       F3B19292
9BBC     9BBC 9BBC 9BBC 9BBC 54F1 D6BE D6BE D6BE D6BE 563E E56E D6BE 5D17      E5B497       E5B497       F3B19293
9BBD     9BBD 9BBD 9BBD 9BBD 5981 D6BF D6BF D6BF D6BF 563F E56F D6BF 5D5C      E5B59C       E5B59C       F3B19294
9BBE     9BBE 9BBE 9BBE 9BBE 5975 D6C0 D6C0 D6C0 D6C0 5640 E57C D6C0 5D1F      E5B49F       E5B49F       F3B19295
9BBF     9BBF 9BBF 9BBF 9BBF 5976 D6C1 D6C1 D6C1 D6C1 5641 E5C1 D6C1 5D1B      E5B49B       E5B49B       F3B19296
9BC0     9BC0 9BC0 9BC0 9BC0 5977 D6C2 D6C2 D6C2 D6C2 5642 E5C2 D6C2 5D11      E5B491       E5B491       F3B19297
9BC1     9BC1 9BC1 9BC1 9BC1 514D D6C3 D6C3 D6C3 D6C3 5643 E5C3 D6C3 5D14      E5B494       E5B494       F3B19298
9BC2     9BC2 9BC2 9BC2 9BC2 5979 D6C4 D6C4 D6C4 D6C4 5644 E5C4 D6C4 5D22      E5B4A2       E5B4A2       F3B19299
9BC3     9BC3 9BC3 9BC3 9BC3 597A D6C5 D6C5 D6C5 D6C5 5645 E5C5 D6C5 5D1A      E5B49A       E5B49A       F3B1929A
9BC4     9BC4 9BC4 9BC4 9BC4 597B D6C6 D6C6 D6C6 D6C6 5646 E5C6 D6C6 5D19      E5B499       E5B499       F3B1929B
9BC5     9BC5 9BC5 9BC5 9BC5 597C D6C7 D6C7 D6C7 D6C7 5647 E5C7 D6C7 5D18      E5B498       E5B498       F3B1929C
9BC6     9BC6 9BC6 9BC6 9BC6 597D D6C8 D6C8 D6C8 D6C8 5648 E5C8 D6C8 5D4C      E5B58C       E5B58C       F3B1929D
9BC7     9BC7 9BC7 9BC7 9BC7 597E D6C9 D6C9 D6C9 D6C9 5649 E5C9 D6C9 5D52      E5B592       E5B592       F3B1929E
9BC8     9BC8 9BC8 9BC8 9BC8 5982 D6CA D6CA D6CA D6CA 564A E5D1 D6CA 5D4E      E5B58E       E5B58E       F3B1929F
9BC9     9BC9 9BC9 9BC9 9BC9 5983 D6CB D6CB D6CB D6CB 564B E5D2 D6CB 5D4B      E5B58B       E5B58B       F3B192A0
9BCA     9BCA 9BCA 9BCA 9BCA 5985 D6CC D6CC D6CC D6CC 564C E5D3 D6CC 5D6C      E5B5AC       E5B5AC       F3B192A1
9BCB     9BCB 9BCB 9BCB 9BCB 5986 D6CD D6CD D6CD D6CD 564D E5D4 D6CD 5D73      E5B5B3       E5B5B3       F3B192A2
9BCC     9BCC 9BCC 9BCC 9BCC 5988 D6CE D6CE D6CE D6CE 564E E5D5 D6CE 5D76      E5B5B6       E5B5B6       F3B192A3
9BCD     9BCD 9BCD 9BCD 9BCD 5989 D6CF D6CF D6CF D6CF 564F E5D6 D6CF 5D87      E5B687       E5B687       F3B192A4
9BCE     9BCE 9BCE 9BCE 9BCE 598A D6D0 D6D0 D6D0 D6D0 5650 E5D7 D6D0 5D84      E5B684       E5B684       F3B192A5
9BCF     9BCF 9BCF 9BCF 9BCF 598B D6D1 D6D1 D6D1 D6D1 5651 E5D8 D6D1 5D82      E5B682       E5B682       F3B192A6
9BD0     9BD0 9BD0 9BD0 9BD0 598D D6D2 D6D2 D6D2 D6D2 5652 E5D9 D6D2 5DA2      E5B6A2       E5B6A2       F3B192A7
9BD1     9BD1 9BD1 9BD1 9BD1 598E D6D3 D6D3 D6D3 D6D3 5653 E5E2 D6D3 5D9D      E5B69D       E5B69D       F3B192A8
9BD2     9BD2 9BD2 9BD2 9BD2 5990 D6D4 D6D4 D6D4 D6D4 5654 E5E3 D6D4 5DAC      E5B6AC       E5B6AC       F3B192A9
9BD3     9BD3 9BD3 9BD3 9BD3 5991 D6D5 D6D5 D6D5 D6D5 5655 E5E4 D6D5 5DAE      E5B6AE       E5B6AE       F3B192AA
9BD4     9BD4 9BD4 9BD4 9BD4 4F84 D6D6 D6D6 D6D6 D6D6 5656 E5E5 D6D6 5DBD      E5B6BD       E5B6BD       F3B192AB
9BD5     9BD5 9BD5 9BD5 9BD5 598F D6D7 D6D7 D6D7 D6D7 5657 E5E6 D6D7 5D90      E5B690       E5B690       F3B192AC
9BD6     9BD6 9BD6 9BD6 9BD6 5992 D6D8 D6D8 D6D8 D6D8 5658 E5E7 D6D8 5DB7      E5B6B7       E5B6B7       F3B192AD
9BD7     9BD7 9BD7 9BD7 9BD7 5994 D6D9 D6D9 D6D9 D6D9 5659 E5E8 D6D9 5DBC      E5B6BC       E5B6BC       F3B192AE
9BD8     9BD8 9BD8 9BD8 9BD8 5996 D6DA D6DA D6DA D6DA 565A E5E9 D6DA 5DC9      E5B789       E5B789       F3B192AF
9BD9     9BD9 9BD9 9BD9 9BD9 5281 D6DB D6DB D6DB D6DB 565B E54A D6DB 5DCD      E5B78D       E5B78D       F3B192B0
9BDA     9BDA 9BDA 9BDA 9BDA 5998 D6DC D6DC D6DC D6DC 565C E55B D6DC 5DD3      E5B793       E5B793       F3B192B1
9BDB     9BDB 9BDB 9BDB 9BDB 5999 D6DD D6DD D6DD D6DD 565D E55A D6DD 5DD2      E5B792       E5B792       F3B192B2
9BDC     9BDC 9BDC 9BDC 9BDC 51DF D6DE D6DE D6DE D6DE 565E E55F D6DE 5DD6      E5B796       E5B796       F3B192B3
9BDD     9BDD 9BDD 9BDD 9BDD 599A D6DF D6DF D6DF D6DF 565F E56D D6DF 5DDB      E5B79B       E5B79B       F3B192B4
9BDE     9BDE 9BDE 9BDE 9BDE 599B D6E0 D6E0 D6E0 D6E0 5660 E579 D6E0 5DEB      E5B7AB       E5B7AB       F3B192B5
9BDF     9BDF 9BDF 9BDF 9BDF 4A4B D6E1 D6E1 D6E1 D6E1 5661 E557 D6E1 5DF2      E5B7B2       E5B7B2       F3B192B6
9BE0     9BE0 9BE0 9BE0 9BE0 599C D6E2 D6E2 D6E2 D6E2 5662 E559 D6E2 5DF5      E5B7B5       E5B7B5       F3B192B7
9BE1     9BE1 9BE1 9BE1 9BE1 599D D6E3 D6E3 D6E3 D6E3 5663 E562 D6E3 5E0B      E5B88B       E5B88B       F3B192B8
9BE2     9BE2 9BE2 9BE2 9BE2 599E D6E4 D6E4 D6E4 D6E4 5664 E563 D6E4 5E1A      E5B89A       E5B89A       F3B192B9
9BE3     9BE3 9BE3 9BE3 9BE3 599F D6E5 D6E5 D6E5 D6E5 5665 E564 D6E5 5E19      E5B899       E5B899       F3B192BA
9BE4     9BE4 9BE4 9BE4 9BE4 59A0 D6E6 D6E6 D6E6 D6E6 5666 E565 D6E6 5E11      E5B891       E5B891       F3B192BB
9BE5     9BE5 9BE5 9BE5 9BE5 59A1 D6E7 D6E7 D6E7 D6E7 5667 E566 D6E7 5E1B      E5B89B       E5B89B       F3B192BC
9BE6     9BE6 9BE6 9BE6 9BE6 59A2 D6E8 D6E8 D6E8 D6E8 5668 E567 D6E8 5E36      E5B8B6       E5B8B6       F3B192BD
9BE7     9BE7 9BE7 9BE7 9BE7 54BE D6E9 D6E9 D6E9 D6E9 5669 E568 D6E9 5E37      E5B8B7       E5B8B7       F3B192BE
9BE8     9BE8 9BE8 9BE8 9BE8 59A3 D6EA D6EA D6EA D6EA 566A E569 D6EA 5E44      E5B984       E5B984       F3B192BF
9BE9     9BE9 9BE9 9BE9 9BE9 59A4 D6EB D6EB D6EB D6EB 566B E570 D6EB 5E43      E5B983       E5B983       F3B19380
9BEA     9BEA 9BEA 9BEA 9BEA 59A5 D6EC D6EC D6EC D6EC 566C E571 D6EC 5E40      E5B980       E5B980       F3B19381
9BEB     9BEB 9BEB 9BEB 9BEB 59A7 D6ED D6ED D6ED D6ED 566D E572 D6ED 5E4E      E5B98E       E5B98E       F3B19382
9BEC     9BEC 9BEC 9BEC 9BEC 59A8 D6EE D6EE D6EE D6EE 566E E573 D6EE 5E57      E5B997       E5B997       F3B19383
9BED     9BED 9BED 9BED 9BED 59A9 D6EF D6EF D6EF D6EF 566F E574 D6EF 5E54      E5B994       E5B994       F3B19384
9BEE     9BEE 9BEE 9BEE 9BEE 5483 D6F0 D6F0 D6F0 D6F0 5670 E575 D6F0 5E5F      E5B99F       E5B99F       F3B19385
9BEF     9BEF 9BEF 9BEF 9BEF 59AA D6F1 D6F1 D6F1 D6F1 5671 E576 D6F1 5E62      E5B9A2       E5B9A2       F3B19386
9BF0     9BF0 9BF0 9BF0 9BF0 59AB D6F2 D6F2 D6F2 D6F2 5672 E577 D6F2 5E64      E5B9A4       E5B9A4       F3B19387
9BF1     9BF1 9BF1 9BF1 9BF1 59A6 D6F3 D6F3 D6F3 D6F3 5673 E578 D6F3 5E47      E5B987       E5B987       F3B19388
9BF2     9BF2 9BF2 9BF2 9BF2 59AC D6F4 D6F4 D6F4 D6F4 5674 E580 D6F4 5E75      E5B9B5       E5B9B5       F3B19389
9BF3     9BF3 9BF3 9BF3 9BF3 59AD D6F5 D6F5 D6F5 D6F5 5675 E58B D6F5 5E76      E5B9B6       E5B9B6       F3B1938A
9BF4     9BF4 9BF4 9BF4 9BF4 59AE D6F6 D6F6 D6F6 D6F6 5676 E59B D6F6 5E7A      E5B9BA       E5B9BA       F3B1938B
9BF5     9BF5 9BF5 9BF5 9BF5 6850 D6F7 D6F7 D6F7 D6F7 5677 E59C D6F7 9EBC      E9BABC       E9BABC       F3B1938C
9BF6     9BF6 9BF6 9BF6 9BF6 59AF D6F8 D6F8 D6F8 D6F8 5678 E5A0 D6F8 5E7F      E5B9BF       E5B9BF       F3B1938D
9BF7     9BF7 9BF7 9BF7 9BF7 59B0 D6F9 D6F9 D6F9 D6F9 5679 E5AB D6F9 5EA0      E5BAA0       E5BAA0       F3B1938E
9BF8     9BF8 9BF8 9BF8 9BF8 59B1 D6FA D6FA D6FA D6FA 567A E5B0 D6FA 5EC1      E5BB81       E5BB81       F3B1938F
9BF9     9BF9 9BF9 9BF9 9BF9 59B2 D6FB D6FB D6FB D6FB 567B E5C0 D6FB 5EC2      E5BB82       E5BB82       F3B19390
9BFA     9BFA 9BFA 9BFA 9BFA 59B3 D6FC D6FC D6FC D6FC 567C E56A D6FC 5EC8      E5BB88       E5BB88       F3B19391
9BFB     9BFB 9BFB 9BFB 9BFB 59B4 D6FD D6FD D6FD D6FD 567D E5D0 D6FD 5ED0      E5BB90       E5BB90       F3B19392
9BFC     9BFC 9BFC 9BFC 9BFC 59B5 D6FE D6FE D6FE D6FE 567E E5A1 D6FE 5ECF      E5BB8F       E5BB8F       F3B19393
9C40     9C40 9C40 9C40 9C40 536C D7A1 D7A1 D7A1 D7A1 5721 E64F D7A1 5ED6      E5BB96       E5BB96       F3B19394
9C41     9C41 9C41 9C41 9C41 50E9 D7A2 D7A2 D7A2 D7A2 5722 E67F D7A2 5EE3      E5BBA3       E5BBA3       F3B19395
9C42     9C42 9C42 9C42 9C42 59B6 D7A3 D7A3 D7A3 D7A3 5723 E67B D7A3 5EDD      E5BB9D       E5BB9D       F3B19396
9C43     9C43 9C43 9C43 9C43 59B7 D7A4 D7A4 D7A4 D7A4 5724 E6E0 D7A4 5EDA      E5BB9A       E5BB9A       F3B19397
9C44     9C44 9C44 9C44 9C44 59B8 D7A5 D7A5 D7A5 D7A5 5725 E66C D7A5 5EDB      E5BB9B       E5BB9B       F3B19398
9C45     9C45 9C45 9C45 9C45 59B9 D7A6 D7A6 D7A6 D7A6 5726 E650 D7A6 5EE2      E5BBA2       E5BBA2       F3B19399
9C46     9C46 9C46 9C46 9C46 59BA D7A7 D7A7 D7A7 D7A7 5727 E67D D7A7 5EE1      E5BBA1       E5BBA1       F3B1939A
9C47     9C47 9C47 9C47 9C47 59BB D7A8 D7A8 D7A8 D7A8 5728 E64D D7A8 5EE8      E5BBA8       E5BBA8       F3B1939B
9C48     9C48 9C48 9C48 9C48 59BC D7A9 D7A9 D7A9 D7A9 5729 E65D D7A9 5EE9      E5BBA9       E5BBA9       F3B1939C
9C49     9C49 9C49 9C49 9C49 59BD D7AA D7AA D7AA D7AA 572A E65C D7AA 5EEC      E5BBAC       E5BBAC       F3B1939D
9C4A     9C4A 9C4A 9C4A 9C4A 59BF D7AB D7AB D7AB D7AB 572B E64E D7AB 5EF1      E5BBB1       E5BBB1       F3B1939E
9C4B     9C4B 9C4B 9C4B 9C4B 59C0 D7AC D7AC D7AC D7AC 572C E66B D7AC 5EF3      E5BBB3       E5BBB3       F3B1939F
9C4C     9C4C 9C4C 9C4C 9C4C 59BE D7AD D7AD D7AD D7AD 572D E660 D7AD 5EF0      E5BBB0       E5BBB0       F3B193A0
9C4D     9C4D 9C4D 9C4D 9C4D 59C1 D7AE D7AE D7AE D7AE 572E E64B D7AE 5EF4      E5BBB4       E5BBB4       F3B193A1
9C4E     9C4E 9C4E 9C4E 9C4E 52D6 D7AF D7AF D7AF D7AF 572F E661 D7AF 5EF8      E5BBB8       E5BBB8       F3B193A2
9C4F     9C4F 9C4F 9C4F 9C4F 544B D7B0 D7B0 D7B0 D7B0 5730 E6F0 D7B0 5EFE      E5BBBE       E5BBBE       F3B193A3
9C50     9C50 9C50 9C50 9C50 59C2 D7B1 D7B1 D7B1 D7B1 5731 E6F1 D7B1 5F03      E5BC83       E5BC83       F3B193A4
9C51     9C51 9C51 9C51 9C51 59C3 D7B2 D7B2 D7B2 D7B2 5732 E6F2 D7B2 5F09      E5BC89       E5BC89       F3B193A5
9C52     9C52 9C52 9C52 9C52 59D6 D7B3 D7B3 D7B3 D7B3 5733 E6F3 D7B3 5F5D      E5BD9D       E5BD9D       F3B193A6
9C53     9C53 9C53 9C53 9C53 59D5 D7B4 D7B4 D7B4 D7B4 5734 E6F4 D7B4 5F5C      E5BD9C       E5BD9C       F3B193A7
9C54     9C54 9C54 9C54 9C54 59C4 D7B5 D7B5 D7B5 D7B5 5735 E6F5 D7B5 5F0B      E5BC8B       E5BC8B       F3B193A8
9C55     9C55 9C55 9C55 9C55 59C5 D7B6 D7B6 D7B6 D7B6 5736 E6F6 D7B6 5F11      E5BC91       E5BC91       F3B193A9
9C56     9C56 9C56 9C56 9C56 59C6 D7B7 D7B7 D7B7 D7B7 5737 E6F7 D7B7 5F16      E5BC96       E5BC96       F3B193AA
9C57     9C57 9C57 9C57 9C57 59C8 D7B8 D7B8 D7B8 D7B8 5738 E6F8 D7B8 5F29      E5BCA9       E5BCA9       F3B193AB
9C58     9C58 9C58 9C58 9C58 59C9 D7B9 D7B9 D7B9 D7B9 5739 E6F9 D7B9 5F2D      E5BCAD       E5BCAD       F3B193AC
9C59     9C59 9C59 9C59 9C59 59CC D7BA D7BA D7BA D7BA 573A E67A D7BA 5F38      E5BCB8       E5BCB8       F3B193AD
#
# One 9C5A meaning requires 9C5A9C5A
#
# 9C5A9C5A 9C5A 9C5A 59CD D7BB D7BB D7BB D7BB 573B E65E D7BB 5F41      E5BD81       F3B483BE
#
9C5B     9C5B 9C5B 9C5B 9C5B 59CE D7BC D7BC D7BC D7BC 573C E64C D7BC 5F48      E5BD88       E5BD88       F3B193AF
9C5C     9C5C 9C5C 9C5C 9C5C 59CF D7BD D7BD D7BD D7BD 573D E67E D7BD 5F4C      E5BD8C       E5BD8C       F3B193B0
9C5D     9C5D 9C5D 9C5D 9C5D 59D0 D7BE D7BE D7BE D7BE 573E E66E D7BE 5F4E      E5BD8E       E5BD8E       F3B193B1
9C5E     9C5E 9C5E 9C5E 9C5E 59CA D7BF D7BF D7BF D7BF 573F E66F D7BF 5F2F      E5BCAF       E5BCAF       F3B193B2
9C5F     9C5F 9C5F 9C5F 9C5F 59D1 D7C0 D7C0 D7C0 D7C0 5740 E67C D7C0 5F51      E5BD91       E5BD91       F3B193B3
9C60     9C60 9C60 9C60 9C60 59D2 D7C1 D7C1 D7C1 D7C1 5741 E6C1 D7C1 5F56      E5BD96       E5BD96       F3B193B4
9C61     9C61 9C61 9C61 9C61 59D3 D7C2 D7C2 D7C2 D7C2 5742 E6C2 D7C2 5F57      E5BD97       E5BD97       F3B193B5
9C62     9C62 9C62 9C62 9C62 59D4 D7C3 D7C3 D7C3 D7C3 5743 E6C3 D7C3 5F59      E5BD99       E5BD99       F3B193B6
9C63     9C63 9C63 9C63 9C63 59D7 D7C4 D7C4 D7C4 D7C4 5744 E6C4 D7C4 5F61      E5BDA1       E5BDA1       F3B193B7
9C64     9C64 9C64 9C64 9C64 529C D7C5 D7C5 D7C5 D7C5 5745 E6C5 D7C5 5F6D      E5BDAD       E5BDAD       F3B193B8
9C65     9C65 9C65 9C65 9C65 59D9 D7C6 D7C6 D7C6 D7C6 5746 E6C6 D7C6 5F73      E5BDB3       E5BDB3       F3B193B9
9C66     9C66 9C66 9C66 9C66 59DA D7C7 D7C7 D7C7 D7C7 5747 E6C7 D7C7 5F77      E5BDB7       E5BDB7       F3B193BA
9C67     9C67 9C67 9C67 9C67 59DB D7C8 D7C8 D7C8 D7C8 5748 E6C8 D7C8 5F83      E5BE83       E5BE83       F3B193BB
9C68     9C68 9C68 9C68 9C68 59DC D7C9 D7C9 D7C9 D7C9 5749 E6C9 D7C9 5F82      E5BE82       E5BE82       F3B193BC
9C69     9C69 9C69 9C69 9C69 59DD D7CA D7CA D7CA D7CA 574A E6D1 D7CA 5F7F      E5BDBF       E5BDBF       F3B193BD
9C6A     9C6A 9C6A 9C6A 9C6A 59DE D7CB D7CB D7CB D7CB 574B E6D2 D7CB 5F8A      E5BE8A       E5BE8A       F3B193BE
9C6B     9C6B 9C6B 9C6B 9C6B 59DF D7CC D7CC D7CC D7CC 574C E6D3 D7CC 5F88      E5BE88       E5BE88       F3B193BF
9C6C     9C6C 9C6C 9C6C 9C6C 59E1 D7CD D7CD D7CD D7CD 574D E6D4 D7CD 5F91      E5BE91       E5BE91       F3B19480
9C6D     9C6D 9C6D 9C6D 9C6D 59E0 D7CE D7CE D7CE D7CE 574E E6D5 D7CE 5F87      E5BE87       E5BE87       F3B19481
9C6E     9C6E 9C6E 9C6E 9C6E 59E3 D7CF D7CF D7CF D7CF 574F E6D6 D7CF 5F9E      E5BE9E       E5BE9E       F3B19482
9C6F     9C6F 9C6F 9C6F 9C6F 59E2 D7D0 D7D0 D7D0 D7D0 5750 E6D7 D7D0 5F99      E5BE99       E5BE99       F3B19483
9C70     9C70 9C70 9C70 9C70 59E4 D7D1 D7D1 D7D1 D7D1 5751 E6D8 D7D1 5F98      E5BE98       E5BE98       F3B19484
9C71     9C71 9C71 9C71 9C71 59E5 D7D2 D7D2 D7D2 D7D2 5752 E6D9 D7D2 5FA0      E5BEA0       E5BEA0       F3B19485
9C72     9C72 9C72 9C72 9C72 59E6 D7D3 D7D3 D7D3 D7D3 5753 E6E2 D7D3 5FA8      E5BEA8       E5BEA8       F3B19486
9C73     9C73 9C73 9C73 9C73 59E7 D7D4 D7D4 D7D4 D7D4 5754 E6E3 D7D4 5FAD      E5BEAD       E5BEAD       F3B19487
9C74     9C74 9C74 9C74 9C74 59E9 D7D5 D7D5 D7D5 D7D5 5755 E6E4 D7D5 5FBC      E5BEBC       E5BEBC       F3B19488
9C75     9C75 9C75 9C75 9C75 59EA D7D6 D7D6 D7D6 D7D6 5756 E6E5 D7D6 5FD6      E5BF96       E5BF96       F3B19489
9C76     9C76 9C76 9C76 9C76 59EB D7D7 D7D7 D7D7 D7D7 5757 E6E6 D7D7 5FFB      E5BFBB       E5BFBB       F3B1948A
9C77     9C77 9C77 9C77 9C77 59EC D7D8 D7D8 D7D8 D7D8 5758 E6E7 D7D8 5FE4      E5BFA4       E5BFA4       F3B1948B
9C78     9C78 9C78 9C78 9C78 59ED D7D9 D7D9 D7D9 D7D9 5759 E6E8 D7D9 5FF8      E5BFB8       E5BFB8       F3B1948C
9C79     9C79 9C79 9C79 9C79 59EE D7DA D7DA D7DA D7DA 575A E6E9 D7DA 5FF1      E5BFB1       E5BFB1       F3B1948D
9C7A     9C7A 9C7A 9C7A 9C7A 59F0 D7DB D7DB D7DB D7DB 575B E64A D7DB 5FDD      E5BF9D       E5BF9D       F3B1948E
9C7B     9C7B 9C7B 9C7B 9C7B 5276 D7DC D7DC D7DC D7DC 575C E65B D7DC 60B3      E682B3       E682B3       F3B1948F
9C7C     9C7C 9C7C 9C7C 9C7C 59F2 D7DD D7DD D7DD D7DD 575D E65A D7DD 5FFF      E5BFBF       E5BFBF       F3B19490
9C7D     9C7D 9C7D 9C7D 9C7D 59F3 D7DE D7DE D7DE D7DE 575E E65F D7DE 6021      E680A1       E680A1       F3B19491
9C7E     9C7E 9C7E 9C7E 9C7E 5A43 D7DF D7DF D7DF D7DF 575F E66D D7DF 6060      E681A0       E681A0       F3B19492
9C80     9C80 9C80 9C80 9C80 59F4 D7E0 D7E0 D7E0 D7E0 5760 E679 D7E0 6019      E68099       E68099       F3B19493
9C81     9C81 9C81 9C81 9C81 59F5 D7E1 D7E1 D7E1 D7E1 5761 E657 D7E1 6010      E68090       E68090       F3B19494
9C82     9C82 9C82 9C82 9C82 59F6 D7E2 D7E2 D7E2 D7E2 5762 E659 D7E2 6029      E680A9       E680A9       F3B19495
9C83     9C83 9C83 9C83 9C83 59F7 D7E3 D7E3 D7E3 D7E3 5763 E662 D7E3 600E      E6808E       E6808E       F3B19496
9C84     9C84 9C84 9C84 9C84 59F8 D7E4 D7E4 D7E4 D7E4 5764 E663 D7E4 6031      E680B1       E680B1       F3B19497
9C85     9C85 9C85 9C85 9C85 59F9 D7E5 D7E5 D7E5 D7E5 5765 E664 D7E5 601B      E6809B       E6809B       F3B19498
9C86     9C86 9C86 9C86 9C86 59FA D7E6 D7E6 D7E6 D7E6 5766 E665 D7E6 6015      E68095       E68095       F3B19499
9C87     9C87 9C87 9C87 9C87 59FB D7E7 D7E7 D7E7 D7E7 5767 E666 D7E7 602B      E680AB       E680AB       F3B1949A
9C88     9C88 9C88 9C88 9C88 59FC D7E8 D7E8 D7E8 D7E8 5768 E667 D7E8 6026      E680A6       E680A6       F3B1949B
9C89     9C89 9C89 9C89 9C89 59FD D7E9 D7E9 D7E9 D7E9 5769 E668 D7E9 600F      E6808F       E6808F       F3B1949C
9C8A     9C8A 9C8A 9C8A 9C8A 59FE D7EA D7EA D7EA D7EA 576A E669 D7EA 603A      E680BA       E680BA       F3B1949D
9C8B     9C8B 9C8B 9C8B 9C8B 5A41 D7EB D7EB D7EB D7EB 576B E670 D7EB 605A      E6819A       E6819A       F3B1949E
9C8C     9C8C 9C8C 9C8C 9C8C 5A42 D7EC D7EC D7EC D7EC 576C E671 D7EC 6041      E68181       E68181       F3B1949F
9C8D     9C8D 9C8D 9C8D 9C8D 5A45 D7ED D7ED D7ED D7ED 576D E672 D7ED 606A      E681AA       E681AA       F3B194A0
9C8E     9C8E 9C8E 9C8E 9C8E 5A46 D7EE D7EE D7EE D7EE 576E E673 D7EE 6077      E681B7       E681B7       F3B194A1
9C8F     9C8F 9C8F 9C8F 9C8F 5A47 D7EF D7EF D7EF D7EF 576F E674 D7EF 605F      E6819F       E6819F       F3B194A2
9C90     9C90 9C90 9C90 9C90 5A48 D7F0 D7F0 D7F0 D7F0 5770 E675 D7F0 604A      E6818A       E6818A       F3B194A3
9C91     9C91 9C91 9C91 9C91 5A49 D7F1 D7F1 D7F1 D7F1 5771 E676 D7F1 6046      E68186       E68186       F3B194A4
9C92     9C92 9C92 9C92 9C92 5A4A D7F2 D7F2 D7F2 D7F2 5772 E677 D7F2 604D      E6818D       E6818D       F3B194A5
9C93     9C93 9C93 9C93 9C93 5A4B D7F3 D7F3 D7F3 D7F3 5773 E678 D7F3 6063      E681A3       E681A3       F3B194A6
9C94     9C94 9C94 9C94 9C94 5A4C D7F4 D7F4 D7F4 D7F4 5774 E680 D7F4 6043      E68183       E68183       F3B194A7
9C95     9C95 9C95 9C95 9C95 5A4D D7F5 D7F5 D7F5 D7F5 5775 E68B D7F5 6064      E681A4       E681A4       F3B194A8
9C96     9C96 9C96 9C96 9C96 5256 D7F6 D7F6 D7F6 D7F6 5776 E69B D7F6 6042      E68182       E68182       F3B194A9
9C97     9C97 9C97 9C97 9C97 5A4E D7F7 D7F7 D7F7 D7F7 5777 E69C D7F7 606C      E681AC       E681AC       F3B194AA
9C98     9C98 9C98 9C98 9C98 5A4F D7F8 D7F8 D7F8 D7F8 5778 E6A0 D7F8 606B      E681AB       E681AB       F3B194AB
9C99     9C99 9C99 9C99 9C99 5A50 D7F9 D7F9 D7F9 D7F9 5779 E6AB D7F9 6059      E68199       E68199       F3B194AC
9C9A     9C9A 9C9A 9C9A 9C9A 5A52 D7FA D7FA D7FA D7FA 577A E6B0 D7FA 6081      E68281       E68281       F3B194AD
9C9B     9C9B 9C9B 9C9B 9C9B 5544 D7FB D7FB D7FB D7FB 577B E6C0 D7FB 608D      E6828D       E6828D       F3B194AE
9C9C     9C9C 9C9C 9C9C 9C9C 554A D7FC D7FC D7FC D7FC 577C E66A D7FC 60E7      E683A7       E683A7       F3B194AF
9C9D     9C9D 9C9D 9C9D 9C9D 5A53 D7FD D7FD D7FD D7FD 577D E6D0 D7FD 6083      E68283       E68283       F3B194B0
9C9E     9C9E 9C9E 9C9E 9C9E 5A54 D7FE D7FE D7FE D7FE 577E E6A1 D7FE 609A      E6829A       E6829A       F3B194B1
9C9F     9C9F 9C9F 9C9F 9C9F 5A55 D8A1 D8A1 D8A1 D8A1 5821 E74F D8A1 6084      E68284       E68284       F3B194B2
9CA0     9CA0 9CA0 9CA0 9CA0 5A56 D8A2 D8A2 D8A2 D8A2 5822 E77F D8A2 609B      E6829B       E6829B       F3B194B3
9CA1     9CA1 9CA1 9CA1 9CA1 5A58 D8A3 D8A3 D8A3 D8A3 5823 E77B D8A3 6096      E68296       E68296       F3B194B4
9CA2     9CA2 9CA2 9CA2 9CA2 5A59 D8A4 D8A4 D8A4 D8A4 5824 E7E0 D8A4 6097      E68297       E68297       F3B194B5
9CA3     9CA3 9CA3 9CA3 9CA3 5A5A D8A5 D8A5 D8A5 D8A5 5825 E76C D8A5 6092      E68292       E68292       F3B194B6
9CA4     9CA4 9CA4 9CA4 9CA4 5A5B D8A6 D8A6 D8A6 D8A6 5826 E750 D8A6 60A7      E682A7       E682A7       F3B194B7
9CA5     9CA5 9CA5 9CA5 9CA5 5A5C D8A7 D8A7 D8A7 D8A7 5827 E77D D8A7 608B      E6828B       E6828B       F3B194B8
9CA6     9CA6 9CA6 9CA6 9CA6 5A5D D8A8 D8A8 D8A8 D8A8 5828 E74D D8A8 60E1      E683A1       E683A1       F3B194B9
9CA7     9CA7 9CA7 9CA7 9CA7 5A5E D8A9 D8A9 D8A9 D8A9 5829 E75D D8A9 60B8      E682B8       E682B8       F3B194BA
9CA8     9CA8 9CA8 9CA8 9CA8 5A60 D8AA D8AA D8AA D8AA 582A E75C D8AA 60E0      E683A0       E683A0       F3B194BB
9CA9     9CA9 9CA9 9CA9 9CA9 5A61 D8AB D8AB D8AB D8AB 582B E74E D8AB 60D3      E68393       E68393       F3B194BC
9CAA     9CAA 9CAA 9CAA 9CAA 53A3 D8AC D8AC D8AC D8AC 582C E76B D8AC 60B4      E682B4       E682B4       F3B194BD
9CAB     9CAB 9CAB 9CAB 9CAB 59EF D8AD D8AD D8AD D8AD 582D E760 D8AD 5FF0      E5BFB0       E5BFB0       F3B194BE
9CAC     9CAC 9CAC 9CAC 9CAC 5A62 D8AE D8AE D8AE D8AE 582E E74B D8AE 60BD      E682BD       E682BD       F3B194BF
9CAD     9CAD 9CAD 9CAD 9CAD 5A63 D8AF D8AF D8AF D8AF 582F E761 D8AF 60C6      E68386       E68386       F3B19580
9CAE     9CAE 9CAE 9CAE 9CAE 5A64 D8B0 D8B0 D8B0 D8B0 5830 E7F0 D8B0 60B5      E682B5       E682B5       F3B19581
9CAF     9CAF 9CAF 9CAF 9CAF 5A66 D8B1 D8B1 D8B1 D8B1 5831 E7F1 D8B1 60D8      E68398       E68398       F3B19582
9CB0     9CB0 9CB0 9CB0 9CB0 5A78 D8B2 D8B2 D8B2 D8B2 5832 E7F2 D8B2 614D      E6858D       E6858D       F3B19583
9CB1     9CB1 9CB1 9CB1 9CB1 5A69 D8B3 D8B3 D8B3 D8B3 5833 E7F3 D8B3 6115      E68495       E68495       F3B19584
9CB2     9CB2 9CB2 9CB2 9CB2 5A6A D8B4 D8B4 D8B4 D8B4 5834 E7F4 D8B4 6106      E68486       E68486       F3B19585
9CB3     9CB3 9CB3 9CB3 9CB3 5A6B D8B5 D8B5 D8B5 D8B5 5835 E7F5 D8B5 60F6      E683B6       E683B6       F3B19586
9CB4     9CB4 9CB4 9CB4 9CB4 5A6C D8B6 D8B6 D8B6 D8B6 5836 E7F6 D8B6 60F7      E683B7       E683B7       F3B19587
9CB5     9CB5 9CB5 9CB5 9CB5 5A6D D8B7 D8B7 D8B7 D8B7 5837 E7F7 D8B7 6100      E68480       E68480       F3B19588
9CB6     9CB6 9CB6 9CB6 9CB6 5A6E D8B8 D8B8 D8B8 D8B8 5838 E7F8 D8B8 60F4      E683B4       E683B4       F3B19589
9CB7     9CB7 9CB7 9CB7 9CB7 5A6F D8B9 D8B9 D8B9 D8B9 5839 E7F9 D8B9 60FA      E683BA       E683BA       F3B1958A
9CB8     9CB8 9CB8 9CB8 9CB8 5A70 D8BA D8BA D8BA D8BA 583A E77A D8BA 6103      E68483       E68483       F3B1958B
9CB9     9CB9 9CB9 9CB9 9CB9 5A71 D8BB D8BB D8BB D8BB 583B E75E D8BB 6121      E684A1       E684A1       F3B1958C
9CBA     9CBA 9CBA 9CBA 9CBA 5A72 D8BC D8BC D8BC D8BC 583C E74C D8BC 60FB      E683BB       E683BB       F3B1958D
9CBB     9CBB 9CBB 9CBB 9CBB 5A73 D8BD D8BD D8BD D8BD 583D E77E D8BD 60F1      E683B1       E683B1       F3B1958E
9CBC     9CBC 9CBC 9CBC 9CBC 5A74 D8BE D8BE D8BE D8BE 583E E76E D8BE 610D      E6848D       E6848D       F3B1958F
9CBD     9CBD 9CBD 9CBD 9CBD 5A75 D8BF D8BF D8BF D8BF 583F E76F D8BF 610E      E6848E       E6848E       F3B19590
9CBE     9CBE 9CBE 9CBE 9CBE 5A77 D8C0 D8C0 D8C0 D8C0 5840 E77C D8C0 6147      E68587       E68587       F3B19591
9CBF     9CBF 9CBF 9CBF 9CBF 5A7C D8C1 D8C1 D8C1 D8C1 5841 E7C1 D8C1 613E      E684BE       E684BE       F3B19592
9CC0     9CC0 9CC0 9CC0 9CC0 5A7A D8C2 D8C2 D8C2 D8C2 5842 E7C2 D8C2 6128      E684A8       E684A8       F3B19593
9CC1     9CC1 9CC1 9CC1 9CC1 5A7B D8C3 D8C3 D8C3 D8C3 5843 E7C3 D8C3 6127      E684A7       E684A7       F3B19594
9CC2     9CC2 9CC2 9CC2 9CC2 5A7D D8C4 D8C4 D8C4 D8C4 5844 E7C4 D8C4 614A      E6858A       E6858A       F3B19595
9CC3     9CC3 9CC3 9CC3 9CC3 51E3 D8C5 D8C5 D8C5 D8C5 5845 E7C5 D8C5 613F      E684BF       E684BF       F3B19596
9CC4     9CC4 9CC4 9CC4 9CC4 5A7F D8C6 D8C6 D8C6 D8C6 5846 E7C6 D8C6 613C      E684BC       E684BC       F3B19597
9CC5     9CC5 9CC5 9CC5 9CC5 5A80 D8C7 D8C7 D8C7 D8C7 5847 E7C7 D8C7 612C      E684AC       E684AC       F3B19598
9CC6     9CC6 9CC6 9CC6 9CC6 5A81 D8C8 D8C8 D8C8 D8C8 5848 E7C8 D8C8 6134      E684B4       E684B4       F3B19599
9CC7     9CC7 9CC7 9CC7 9CC7 5A84 D8C9 D8C9 D8C9 D8C9 5849 E7C9 D8C9 613D      E684BD       E684BD       F3B1959A
9CC8     9CC8 9CC8 9CC8 9CC8 5A85 D8CA D8CA D8CA D8CA 584A E7D1 D8CA 6142      E68582       E68582       F3B1959B
9CC9     9CC9 9CC9 9CC9 9CC9 5A86 D8CB D8CB D8CB D8CB 584B E7D2 D8CB 6144      E68584       E68584       F3B1959C
9CCA     9CCA 9CCA 9CCA 9CCA 5A87 D8CC D8CC D8CC D8CC 584C E7D3 D8CC 6173      E685B3       E685B3       F3B1959D
9CCB     9CCB 9CCB 9CCB 9CCB 5A89 D8CD D8CD D8CD D8CD 584D E7D4 D8CD 6177      E685B7       E685B7       F3B1959E
9CCC     9CCC 9CCC 9CCC 9CCC 5A8A D8CE D8CE D8CE D8CE 584E E7D5 D8CE 6158      E68598       E68598       F3B1959F
9CCD     9CCD 9CCD 9CCD 9CCD 5A8B D8CF D8CF D8CF D8CF 584F E7D6 D8CF 6159      E68599       E68599       F3B195A0
9CCE     9CCE 9CCE 9CCE 9CCE 5A8C D8D0 D8D0 D8D0 D8D0 5850 E7D7 D8D0 615A      E6859A       E6859A       F3B195A1
9CCF     9CCF 9CCF 9CCF 9CCF 5A8D D8D1 D8D1 D8D1 D8D1 5851 E7D8 D8D1 616B      E685AB       E685AB       F3B195A2
9CD0     9CD0 9CD0 9CD0 9CD0 5A8E D8D2 D8D2 D8D2 D8D2 5852 E7D9 D8D2 6174      E685B4       E685B4       F3B195A3
9CD1     9CD1 9CD1 9CD1 9CD1 5A8F D8D3 D8D3 D8D3 D8D3 5853 E7E2 D8D3 616F      E685AF       E685AF       F3B195A4
9CD2     9CD2 9CD2 9CD2 9CD2 5A82 D8D4 D8D4 D8D4 D8D4 5854 E7E3 D8D4 6165      E685A5       E685A5       F3B195A5
9CD3     9CD3 9CD3 9CD3 9CD3 5A90 D8D5 D8D5 D8D5 D8D5 5855 E7E4 D8D5 6171      E685B1       E685B1       F3B195A6
9CD4     9CD4 9CD4 9CD4 9CD4 5A91 D8D6 D8D6 D8D6 D8D6 5856 E7E5 D8D6 615F      E6859F       E6859F       F3B195A7
9CD5     9CD5 9CD5 9CD5 9CD5 5A83 D8D7 D8D7 D8D7 D8D7 5857 E7E6 D8D7 615D      E6859D       E6859D       F3B195A8
9CD6     9CD6 9CD6 9CD6 9CD6 5A92 D8D8 D8D8 D8D8 D8D8 5858 E7E7 D8D8 6153      E68593       E68593       F3B195A9
9CD7     9CD7 9CD7 9CD7 9CD7 5A93 D8D9 D8D9 D8D9 D8D9 5859 E7E8 D8D9 6175      E685B5       E685B5       F3B195AA
9CD8     9CD8 9CD8 9CD8 9CD8 5A95 D8DA D8DA D8DA D8DA 585A E7E9 D8DA 6199      E68699       E68699       F3B195AB
9CD9     9CD9 9CD9 9CD9 9CD9 5A96 D8DB D8DB D8DB D8DB 585B E74A D8DB 6196      E68696       E68696       F3B195AC
9CDA     9CDA 9CDA 9CDA 9CDA 5A88 D8DC D8DC D8DC D8DC 585C E75B D8DC 6187      E68687       E68687       F3B195AD
9CDB     9CDB 9CDB 9CDB 9CDB 5A97 D8DD D8DD D8DD D8DD 585D E75A D8DD 61AC      E686AC       E686AC       F3B195AE
9CDC     9CDC 9CDC 9CDC 9CDC 5A98 D8DE D8DE D8DE D8DE 585E E75F D8DE 6194      E68694       E68694       F3B195AF
9CDD     9CDD 9CDD 9CDD 9CDD 55CF D8DF D8DF D8DF D8DF 585F E76D D8DF 619A      E6869A       E6869A       F3B195B0
9CDE     9CDE 9CDE 9CDE 9CDE 5A99 D8E0 D8E0 D8E0 D8E0 5860 E779 D8E0 618A      E6868A       E6868A       F3B195B1
9CDF     9CDF 9CDF 9CDF 9CDF 5A9A D8E1 D8E1 D8E1 D8E1 5861 E757 D8E1 6191      E68691       E68691       F3B195B2
9CE0     9CE0 9CE0 9CE0 9CE0 5A9B D8E2 D8E2 D8E2 D8E2 5862 E759 D8E2 61AB      E686AB       E686AB       F3B195B3
9CE1     9CE1 9CE1 9CE1 9CE1 5A9C D8E3 D8E3 D8E3 D8E3 5863 E762 D8E3 61AE      E686AE       E686AE       F3B195B4
9CE2     9CE2 9CE2 9CE2 9CE2 5A9D D8E4 D8E4 D8E4 D8E4 5864 E763 D8E4 61CC      E6878C       E6878C       F3B195B5
9CE3     9CE3 9CE3 9CE3 9CE3 5A9E D8E5 D8E5 D8E5 D8E5 5865 E764 D8E5 61CA      E6878A       E6878A       F3B195B6
9CE4     9CE4 9CE4 9CE4 9CE4 5A9F D8E6 D8E6 D8E6 D8E6 5866 E765 D8E6 61C9      E68789       E68789       F3B195B7
9CE5     9CE5 9CE5 9CE5 9CE5 5AAA D8E7 D8E7 D8E7 D8E7 5867 E766 D8E7 61F7      E687B7       E687B7       F3B195B8
9CE6     9CE6 9CE6 9CE6 9CE6 5AA0 D8E8 D8E8 D8E8 D8E8 5868 E767 D8E8 61C8      E68788       E68788       F3B195B9
9CE7     9CE7 9CE7 9CE7 9CE7 5AA1 D8E9 D8E9 D8E9 D8E9 5869 E768 D8E9 61C3      E68783       E68783       F3B195BA
9CE8     9CE8 9CE8 9CE8 9CE8 5AA2 D8EA D8EA D8EA D8EA 586A E769 D8EA 61C6      E68786       E68786       F3B195BB
9CE9     9CE9 9CE9 9CE9 9CE9 5AA3 D8EB D8EB D8EB D8EB 586B E770 D8EB 61BA      E686BA       E686BA       F3B195BC
9CEA     9CEA 9CEA 9CEA 9CEA 5AA4 D8EC D8EC D8EC D8EC 586C E771 D8EC 61CB      E6878B       E6878B       F3B195BD
9CEB     9CEB 9CEB 9CEB 9CEB 5AA5 D8ED D8ED D8ED D8ED 586D E772 D8ED 7F79      E7BDB9       E7BDB9       F3B195BE
9CEC     9CEC 9CEC 9CEC 9CEC 5AA6 D8EE D8EE D8EE D8EE 586E E773 D8EE 61CD      E6878D       E6878D       F3B195BF
9CED     9CED 9CED 9CED 9CED 5AA7 D8EF D8EF D8EF D8EF 586F E774 D8EF 61E6      E687A6       E687A6       F3B19680
9CEE     9CEE 9CEE 9CEE 9CEE 5AA8 D8F0 D8F0 D8F0 D8F0 5870 E775 D8F0 61E3      E687A3       E687A3       F3B19681
9CEF     9CEF 9CEF 9CEF 9CEF 5AAB D8F1 D8F1 D8F1 D8F1 5871 E776 D8F1 61F6      E687B6       E687B6       F3B19682
9CF0     9CF0 9CF0 9CF0 9CF0 5AAD D8F2 D8F2 D8F2 D8F2 5872 E777 D8F2 61FA      E687BA       E687BA       F3B19683
9CF1     9CF1 9CF1 9CF1 9CF1 5AA9 D8F3 D8F3 D8F3 D8F3 5873 E778 D8F3 61F4      E687B4       E687B4       F3B19684
9CF2     9CF2 9CF2 9CF2 9CF2 5AAE D8F4 D8F4 D8F4 D8F4 5874 E780 D8F4 61FF      E687BF       E687BF       F3B19685
9CF3     9CF3 9CF3 9CF3 9CF3 5AAC D8F5 D8F5 D8F5 D8F5 5875 E78B D8F5 61FD      E687BD       E687BD       F3B19686
9CF4     9CF4 9CF4 9CF4 9CF4 5AAF D8F6 D8F6 D8F6 D8F6 5876 E79B D8F6 61FC      E687BC       E687BC       F3B19687
9CF5     9CF5 9CF5 9CF5 9CF5 5AB0 D8F7 D8F7 D8F7 D8F7 5877 E79C D8F7 61FE      E687BE       E687BE       F3B19688
9CF6     9CF6 9CF6 9CF6 9CF6 5AB1 D8F8 D8F8 D8F8 D8F8 5878 E7A0 D8F8 6200      E68880       E68880       F3B19689
9CF7     9CF7 9CF7 9CF7 9CF7 5AB2 D8F9 D8F9 D8F9 D8F9 5879 E7AB D8F9 6208      E68888       E68888       F3B1968A
9CF8     9CF8 9CF8 9CF8 9CF8 5AB3 D8FA D8FA D8FA D8FA 587A E7B0 D8FA 6209      E68889       E68889       F3B1968B
9CF9     9CF9 9CF9 9CF9 9CF9 5AB4 D8FB D8FB D8FB D8FB 587B E7C0 D8FB 620D      E6888D       E6888D       F3B1968C
9CFA     9CFA 9CFA 9CFA 9CFA 5460 D8FC D8FC D8FC D8FC 587C E76A D8FC 620C      E6888C       E6888C       F3B1968D
9CFB     9CFB 9CFB 9CFB 9CFB 5AB6 D8FD D8FD D8FD D8FD 587D E7D0 D8FD 6214      E68894       E68894       F3B1968E
9CFC     9CFC 9CFC 9CFC 9CFC 5AB7 D8FE D8FE D8FE D8FE 587E E7A1 D8FE 621B      E6889B       E6889B       F3B1968F
9D40     9D40 9D40 9D40 9D40 5AB8 D9A1 D9A1 D9A1 D9A1 5921 E84F D9A1 621E      E6889E       E6889E       F3B19690
9D41     9D41 9D41 9D41 9D41 5AB9 D9A2 D9A2 D9A2 D9A2 5922 E87F D9A2 6221      E688A1       E688A1       F3B19691
9D42     9D42 9D42 9D42 9D42 5ABA D9A3 D9A3 D9A3 D9A3 5923 E87B D9A3 622A      E688AA       E688AA       F3B19692
9D43     9D43 9D43 9D43 9D43 5ABB D9A4 D9A4 D9A4 D9A4 5924 E8E0 D9A4 622E      E688AE       E688AE       F3B19693
9D44     9D44 9D44 9D44 9D44 5ABC D9A5 D9A5 D9A5 D9A5 5925 E86C D9A5 6230      E688B0       E688B0       F3B19694
9D45     9D45 9D45 9D45 9D45 5ABD D9A6 D9A6 D9A6 D9A6 5926 E850 D9A6 6232      E688B2       E688B2       F3B19695
9D46     9D46 9D46 9D46 9D46 5ABE D9A7 D9A7 D9A7 D9A7 5927 E87D D9A7 6233      E688B3       E688B3       F3B19696
9D47     9D47 9D47 9D47 9D47 5ABF D9A8 D9A8 D9A8 D9A8 5928 E84D D9A8 6241      E68981       E68981       F3B19697
9D48     9D48 9D48 9D48 9D48 5AC0 D9A9 D9A9 D9A9 D9A9 5929 E85D D9A9 624E      E6898E       E6898E       F3B19698
9D49     9D49 9D49 9D49 9D49 5AC1 D9AA D9AA D9AA D9AA 592A E85C D9AA 625E      E6899E       E6899E       F3B19699
9D4A     9D4A 9D4A 9D4A 9D4A 5AC2 D9AB D9AB D9AB D9AB 592B E84E D9AB 6263      E689A3       E689A3       F3B1969A
9D4B     9D4B 9D4B 9D4B 9D4B 5AC3 D9AC D9AC D9AC D9AC 592C E86B D9AC 625B      E6899B       E6899B       F3B1969B
9D4C     9D4C 9D4C 9D4C 9D4C 5AC4 D9AD D9AD D9AD D9AD 592D E860 D9AD 6260      E689A0       E689A0       F3B1969C
9D4D     9D4D 9D4D 9D4D 9D4D 5AC5 D9AE D9AE D9AE D9AE 592E E84B D9AE 6268      E689A8       E689A8       F3B1969D
9D4E     9D4E 9D4E 9D4E 9D4E 5AC6 D9AF D9AF D9AF D9AF 592F E861 D9AF 627C      E689BC       E689BC       F3B1969E
9D4F     9D4F 9D4F 9D4F 9D4F 5AC7 D9B0 D9B0 D9B0 D9B0 5930 E8F0 D9B0 6282      E68A82       E68A82       F3B1969F
9D50     9D50 9D50 9D50 9D50 5AC8 D9B1 D9B1 D9B1 D9B1 5931 E8F1 D9B1 6289      E68A89       E68A89       F3B196A0
9D51     9D51 9D51 9D51 9D51 5ACA D9B2 D9B2 D9B2 D9B2 5932 E8F2 D9B2 627E      E689BE       E689BE       F3B196A1
9D52     9D52 9D52 9D52 9D52 5AC9 D9B3 D9B3 D9B3 D9B3 5933 E8F3 D9B3 6292      E68A92       E68A92       F3B196A2
9D53     9D53 9D53 9D53 9D53 5ACB D9B4 D9B4 D9B4 D9B4 5934 E8F4 D9B4 6293      E68A93       E68A93       F3B196A3
9D54     9D54 9D54 9D54 9D54 5ACC D9B5 D9B5 D9B5 D9B5 5935 E8F5 D9B5 6296      E68A96       E68A96       F3B196A4
9D55     9D55 9D55 9D55 9D55 5AD7 D9B6 D9B6 D9B6 D9B6 5936 E8F6 D9B6 62D4      E68B94       E68B94       F3B196A5
9D56     9D56 9D56 9D56 9D56 5ACD D9B7 D9B7 D9B7 D9B7 5937 E8F7 D9B7 6283      E68A83       E68A83       F3B196A6
9D57     9D57 9D57 9D57 9D57 5ACE D9B8 D9B8 D9B8 D9B8 5938 E8F8 D9B8 6294      E68A94       E68A94       F3B196A7
9D58     9D58 9D58 9D58 9D58 5ACF D9B9 D9B9 D9B9 D9B9 5939 E8F9 D9B9 62D7      E68B97       E68B97       F3B196A8
9D59     9D59 9D59 9D59 9D59 5AD0 D9BA D9BA D9BA D9BA 593A E87A D9BA 62D1      E68B91       E68B91       F3B196A9
9D5A     9D5A 9D5A 9D5A 9D5A 5AD1 D9BB D9BB D9BB D9BB 593B E85E D9BB 62BB      E68ABB       E68ABB       F3B196AA
9D5B     9D5B 9D5B 9D5B 9D5B 5AD2 D9BC D9BC D9BC D9BC 593C E84C D9BC 62CF      E68B8F       E68B8F       F3B196AB
9D5C     9D5C 9D5C 9D5C 9D5C 5AE6 D9BD D9BD D9BD D9BD 593D E87E D9BD 62FF      E68BBF       E68BBF       F3B196AC
9D5D     9D5D 9D5D 9D5D 9D5D 5AD4 D9BE D9BE D9BE D9BE 593E E86E D9BE 62C6      E68B86       E68B86       F3B196AD
9D5E     9D5E 9D5E 9D5E 9D5E 5B60 D9BF D9BF D9BF D9BF 593F E86F D9BF 64D4      E69394       E69394       F3B196AE
9D5F     9D5F 9D5F 9D5F 9D5F 5AD5 D9C0 D9C0 D9C0 D9C0 5940 E87C D9C0 62C8      E68B88       E68B88       F3B196AF
9D60     9D60 9D60 9D60 9D60 5AD6 D9C1 D9C1 D9C1 D9C1 5941 E8C1 D9C1 62DC      E68B9C       E68B9C       F3B196B0
9D61     9D61 9D61 9D61 9D61 5345 D9C2 D9C2 D9C2 D9C2 5942 E8C2 D9C2 62CC      E68B8C       E68B8C       F3B196B1
9D62     9D62 9D62 9D62 9D62 5AD8 D9C3 D9C3 D9C3 D9C3 5943 E8C3 D9C3 62CA      E68B8A       E68B8A       F3B196B2
9D63     9D63 9D63 9D63 9D63 5AD9 D9C4 D9C4 D9C4 D9C4 5944 E8C4 D9C4 62C2      E68B82       E68B82       F3B196B3
9D64     9D64 9D64 9D64 9D64 5ADB D9C5 D9C5 D9C5 D9C5 5945 E8C5 D9C5 62C7      E68B87       E68B87       F3B196B4
9D65     9D65 9D65 9D65 9D65 5ADC D9C6 D9C6 D9C6 D9C6 5946 E8C6 D9C6 629B      E68A9B       E68A9B       F3B196B5
9D66     9D66 9D66 9D66 9D66 5ADD D9C7 D9C7 D9C7 D9C7 5947 E8C7 D9C7 62C9      E68B89       E68B89       F3B196B6
9D67     9D67 9D67 9D67 9D67 5ADE D9C8 D9C8 D9C8 D9C8 5948 E8C8 D9C8 630C      E68C8C       E68C8C       F3B196B7
9D68     9D68 9D68 9D68 9D68 5ADF D9C9 D9C9 D9C9 D9C9 5949 E8C9 D9C9 62EE      E68BAE       E68BAE       F3B196B8
9D69     9D69 9D69 9D69 9D69 5AE0 D9CA D9CA D9CA D9CA 594A E8D1 D9CA 62F1      E68BB1       E68BB1       F3B196B9
9D6A     9D6A 9D6A 9D6A 9D6A 5AE1 D9CB D9CB D9CB D9CB 594B E8D2 D9CB 6327      E68CA7       E68CA7       F3B196BA
9D6B     9D6B 9D6B 9D6B 9D6B 5AE2 D9CC D9CC D9CC D9CC 594C E8D3 D9CC 6302      E68C82       E68C82       F3B196BB
9D6C     9D6C 9D6C 9D6C 9D6C 5AE3 D9CD D9CD D9CD D9CD 594D E8D4 D9CD 6308      E68C88       E68C88       F3B196BC
9D6D     9D6D 9D6D 9D6D 9D6D 5AE4 D9CE D9CE D9CE D9CE 594E E8D5 D9CE 62EF      E68BAF       E68BAF       F3B196BD
9D6E     9D6E 9D6E 9D6E 9D6E 5AE5 D9CF D9CF D9CF D9CF 594F E8D6 D9CF 62F5      E68BB5       E68BB5       F3B196BE
9D6F     9D6F 9D6F 9D6F 9D6F 5AE7 D9D0 D9D0 D9D0 D9D0 5950 E8D7 D9D0 6350      E68D90       E68D90       F3B196BF
9D70     9D70 9D70 9D70 9D70 5AE9 D9D1 D9D1 D9D1 D9D1 5951 E8D8 D9D1 633E      E68CBE       E68CBE       F3B19780
9D71     9D71 9D71 9D71 9D71 5AE8 D9D2 D9D2 D9D2 D9D2 5952 E8D9 D9D2 634D      E68D8D       E68D8D       F3B19781
9D72     9D72 9D72 9D72 9D72 5B48 D9D3 D9D3 D9D3 D9D3 5953 E8E2 D9D3 641C      E6909C       E6909C       F3B19782
9D73     9D73 9D73 9D73 9D73 5AEA D9D4 D9D4 D9D4 D9D4 5954 E8E3 D9D4 634F      E68D8F       E68D8F       F3B19783
9D74     9D74 9D74 9D74 9D74 5AEB D9D5 D9D5 D9D5 D9D5 5955 E8E4 D9D5 6396      E68E96       E68E96       F3B19784
9D75     9D75 9D75 9D75 9D75 5AEC D9D6 D9D6 D9D6 D9D6 5956 E8E5 D9D6 638E      E68E8E       E68E8E       F3B19785
9D76     9D76 9D76 9D76 9D76 5AED D9D7 D9D7 D9D7 D9D7 5957 E8E6 D9D7 6380      E68E80       E68E80       F3B19786
9D77     9D77 9D77 9D77 9D77 5AEE D9D8 D9D8 D9D8 D9D8 5958 E8E7 D9D8 63AB      E68EAB       E68EAB       F3B19787
9D78     9D78 9D78 9D78 9D78 5AEF D9D9 D9D9 D9D9 D9D9 5959 E8E8 D9D9 6376      E68DB6       E68DB6       F3B19788
9D79     9D79 9D79 9D79 9D79 5AF0 D9DA D9DA D9DA D9DA 595A E8E9 D9DA 63A3      E68EA3       E68EA3       F3B19789
9D7A     9D7A 9D7A 9D7A 9D7A 5AF1 D9DB D9DB D9DB D9DB 595B E84A D9DB 638F      E68E8F       E68E8F       F3B1978A
9D7B     9D7B 9D7B 9D7B 9D7B 5AF2 D9DC D9DC D9DC D9DC 595C E85B D9DC 6389      E68E89       E68E89       F3B1978B
9D7C     9D7C 9D7C 9D7C 9D7C 5AF3 D9DD D9DD D9DD D9DD 595D E85A D9DD 639F      E68E9F       E68E9F       F3B1978C
9D7D     9D7D 9D7D 9D7D 9D7D 5AF6 D9DE D9DE D9DE D9DE 595E E85F D9DE 63B5      E68EB5       E68EB5       F3B1978D
9D7E     9D7E 9D7E 9D7E 9D7E 5AF4 D9DF D9DF D9DF D9DF 595F E86D D9DF 636B      E68DAB       E68DAB       F3B1978E
9D80     9D80 9D80 9D80 9D80 5AF5 D9E0 D9E0 D9E0 D9E0 5960 E879 D9E0 6369      E68DA9       E68DA9       F3B1978F
9D81     9D81 9D81 9D81 9D81 5AF7 D9E1 D9E1 D9E1 D9E1 5961 E857 D9E1 63BE      E68EBE       E68EBE       F3B19790
9D82     9D82 9D82 9D82 9D82 5AF8 D9E2 D9E2 D9E2 D9E2 5962 E859 D9E2 63E9      E68FA9       E68FA9       F3B19791
9D83     9D83 9D83 9D83 9D83 5AF9 D9E3 D9E3 D9E3 D9E3 5963 E862 D9E3 63C0      E68F80       E68F80       F3B19792
9D84     9D84 9D84 9D84 9D84 5AFA D9E4 D9E4 D9E4 D9E4 5964 E863 D9E4 63C6      E68F86       E68F86       F3B19793
9D85     9D85 9D85 9D85 9D85 5AFC D9E5 D9E5 D9E5 D9E5 5965 E864 D9E5 63E3      E68FA3       E68FA3       F3B19794
9D86     9D86 9D86 9D86 9D86 5AFD D9E6 D9E6 D9E6 D9E6 5966 E865 D9E6 63C9      E68F89       E68F89       F3B19795
9D87     9D87 9D87 9D87 9D87 5AFE D9E7 D9E7 D9E7 D9E7 5967 E866 D9E7 63D2      E68F92       E68F92       F3B19796
9D88     9D88 9D88 9D88 9D88 5B41 D9E8 D9E8 D9E8 D9E8 5968 E867 D9E8 63F6      E68FB6       E68FB6       F3B19797
9D89     9D89 9D89 9D89 9D89 5B42 D9E9 D9E9 D9E9 D9E9 5969 E868 D9E9 63C4      E68F84       E68F84       F3B19798
9D8A     9D8A 9D8A 9D8A 9D8A 5B4C D9EA D9EA D9EA D9EA 596A E869 D9EA 6416      E69096       E69096       F3B19799
9D8B     9D8B 9D8B 9D8B 9D8B 5B43 D9EB D9EB D9EB D9EB 596B E870 D9EB 6434      E690B4       E690B4       F3B1979A
9D8C     9D8C 9D8C 9D8C 9D8C 5B44 D9EC D9EC D9EC D9EC 596C E871 D9EC 6406      E69086       E69086       F3B1979B
9D8D     9D8D 9D8D 9D8D 9D8D 5B45 D9ED D9ED D9ED D9ED 596D E872 D9ED 6413      E69093       E69093       F3B1979C
9D8E     9D8E 9D8E 9D8E 9D8E 5B46 D9EE D9EE D9EE D9EE 596E E873 D9EE 6426      E690A6       E690A6       F3B1979D
9D8F     9D8F 9D8F 9D8F 9D8F 5B47 D9EF D9EF D9EF D9EF 596F E874 D9EF 6436      E690B6       E690B6       F3B1979E
9D90     9D90 9D90 9D90 9D90 5B71 D9F0 D9F0 D9F0 D9F0 5970 E875 D9F0 651D      E6949D       E6949D       F3B1979F
9D91     9D91 9D91 9D91 9D91 5B49 D9F1 D9F1 D9F1 D9F1 5971 E876 D9F1 6417      E69097       E69097       F3B197A0
9D92     9D92 9D92 9D92 9D92 5B4A D9F2 D9F2 D9F2 D9F2 5972 E877 D9F2 6428      E690A8       E690A8       F3B197A1
9D93     9D93 9D93 9D93 9D93 5B4B D9F3 D9F3 D9F3 D9F3 5973 E878 D9F3 640F      E6908F       E6908F       F3B197A2
9D94     9D94 9D94 9D94 9D94 5B4E D9F4 D9F4 D9F4 D9F4 5974 E880 D9F4 6467      E691A7       E691A7       F3B197A3
9D95     9D95 9D95 9D95 9D95 5B4F D9F5 D9F5 D9F5 D9F5 5975 E88B D9F5 646F      E691AF       E691AF       F3B197A4
9D96     9D96 9D96 9D96 9D96 5B51 D9F6 D9F6 D9F6 D9F6 5976 E89B D9F6 6476      E691B6       E691B6       F3B197A5
9D97     9D97 9D97 9D97 9D97 5B4D D9F7 D9F7 D9F7 D9F7 5977 E89C D9F7 644E      E6918E       E6918E       F3B197A6
9D98     9D98 9D98 9D98 9D98 527F B3C9 D9F8 D9F8 B3C9 3349 F3C9 D9F8 652A      E694AA       E694AA       F3B197A7
9D99     9D99 9D99 9D99 9D99 5B55 D9F9 D9F9 D9F9 D9F9 5979 E8AB D9F9 6495      E69295       E69295       F3B197A8
9D9A     9D9A 9D9A 9D9A 9D9A 5B57 D9FA D9FA D9FA D9FA 597A E8B0 D9FA 6493      E69293       E69293       F3B197A9
9D9B     9D9B 9D9B 9D9B 9D9B 5B58 D9FB D9FB D9FB D9FB 597B E8C0 D9FB 64A5      E692A5       E692A5       F3B197AA
9D9C     9D9C 9D9C 9D9C 9D9C 5B59 D9FC D9FC D9FC D9FC 597C E86A D9FC 64A9      E692A9       E692A9       F3B197AB
9D9D     9D9D 9D9D 9D9D 9D9D 5B5A D9FD D9FD D9FD D9FD 597D E8D0 D9FD 6488      E69288       E69288       F3B197AC
9D9E     9D9E 9D9E 9D9E 9D9E 5B5B D9FE D9FE D9FE D9FE 597E E8A1 D9FE 64BC      E692BC       E692BC       F3B197AD
9D9F     9D9F 9D9F 9D9F 9D9F 5B5C DAA1 DAA1 DAA1 DAA1 5A21 E94F DAA1 64DA      E6939A       E6939A       F3B197AE
9DA0     9DA0 9DA0 9DA0 9DA0 5B5D DAA2 DAA2 DAA2 DAA2 5A22 E97F DAA2 64D2      E69392       E69392       F3B197AF
9DA1     9DA1 9DA1 9DA1 9DA1 5B5E DAA3 DAA3 DAA3 DAA3 5A23 E97B DAA3 64C5      E69385       E69385       F3B197B0
9DA2     9DA2 9DA2 9DA2 9DA2 5B5F DAA4 DAA4 DAA4 DAA4 5A24 E9E0 DAA4 64C7      E69387       E69387       F3B197B1
9DA3     9DA3 9DA3 9DA3 9DA3 5B56 DAA5 DAA5 DAA5 DAA5 5A25 E96C DAA5 64BB      E692BB       E692BB       F3B197B2
9DA4     9DA4 9DA4 9DA4 9DA4 5B61 DAA6 DAA6 DAA6 DAA6 5A26 E950 DAA6 64D8      E69398       E69398       F3B197B3
9DA5     9DA5 9DA5 9DA5 9DA5 5B62 DAA7 DAA7 DAA7 DAA7 5A27 E97D DAA7 64C2      E69382       E69382       F3B197B4
9DA6     9DA6 9DA6 9DA6 9DA6 5B63 DAA8 DAA8 DAA8 DAA8 5A28 E94D DAA8 64F1      E693B1       E693B1       F3B197B5
9DA7     9DA7 9DA7 9DA7 9DA7 5B64 DAA9 DAA9 DAA9 DAA9 5A29 E95D DAA9 64E7      E693A7       E693A7       F3B197B6
9DA8     9DA8 9DA8 9DA8 9DA8 5778 DAAA DAAA DAAA DAAA 5A2A E95C DAAA 8209      E88889       E88889       F3B197B7
9DA9     9DA9 9DA9 9DA9 9DA9 5B65 DAAB DAAB DAAB DAAB 5A2B E94E DAAB 64E0      E693A0       E693A0       F3B197B8
9DAA     9DAA 9DAA 9DAA 9DAA 5B66 DAAC DAAC DAAC DAAC 5A2C E96B DAAC 64E1      E693A1       E693A1       F3B197B9
9DAB     9DAB 9DAB 9DAB 9DAB 5AD3 DAAD DAAD DAAD DAAD 5A2D E960 DAAD 62AC      E68AAC       E68AAC       F3B197BA
9DAC     9DAC 9DAC 9DAC 9DAC 5B67 DAAE DAAE DAAE DAAE 5A2E E94B DAAE 64E3      E693A3       E693A3       F3B197BB
9DAD     9DAD 9DAD 9DAD 9DAD 5B68 DAAF DAAF DAAF DAAF 5A2F E961 DAAF 64EF      E693AF       E693AF       F3B197BC
9DAE     9DAE 9DAE 9DAE 9DAE 5B76 DAB0 DAB0 DAB0 DAB0 5A30 E9F0 DAB0 652C      E694AC       E694AC       F3B197BD
9DAF     9DAF 9DAF 9DAF 9DAF 5B6A DAB1 DAB1 DAB1 DAB1 5A31 E9F1 DAB1 64F6      E693B6       E693B6       F3B197BE
9DB0     9DB0 9DB0 9DB0 9DB0 5B69 DAB2 DAB2 DAB2 DAB2 5A32 E9F2 DAB2 64F4      E693B4       E693B4       F3B197BF
9DB1     9DB1 9DB1 9DB1 9DB1 5B6B DAB3 DAB3 DAB3 DAB3 5A33 E9F3 DAB3 64F2      E693B2       E693B2       F3B19880
9DB2     9DB2 9DB2 9DB2 9DB2 5B6C DAB4 DAB4 DAB4 DAB4 5A34 E9F4 DAB4 64FA      E693BA       E693BA       F3B19881
9DB3     9DB3 9DB3 9DB3 9DB3 5B6D DAB5 DAB5 DAB5 DAB5 5A35 E9F5 DAB5 6500      E69480       E69480       F3B19882
9DB4     9DB4 9DB4 9DB4 9DB4 5B6E DAB6 DAB6 DAB6 DAB6 5A36 E9F6 DAB6 64FD      E693BD       E693BD       F3B19883
9DB5     9DB5 9DB5 9DB5 9DB5 5B6F DAB7 DAB7 DAB7 DAB7 5A37 E9F7 DAB7 6518      E69498       E69498       F3B19884
9DB6     9DB6 9DB6 9DB6 9DB6 5B70 DAB8 DAB8 DAB8 DAB8 5A38 E9F8 DAB8 651C      E6949C       E6949C       F3B19885
9DB7     9DB7 9DB7 9DB7 9DB7 5B72 DAB9 DAB9 DAB9 50B1 BCE9 8DCB DAB9 6505      E69485       E69485       F3B19886
9DB8     9DB8 9DB8 9DB8 9DB8 5B73 DABA DABA DABA DABA 5A3A E97A DABA 6524      E694A4       E694A4       F3B19887
9DB9     9DB9 9DB9 9DB9 9DB9 5B74 DABB DABB DABB DABB 5A3B E95E DABB 6523      E694A3       E694A3       F3B19888
9DBA     9DBA 9DBA 9DBA 9DBA 5B75 DABC DABC DABC DABC 5A3C E94C DABC 652B      E694AB       E694AB       F3B19889
9DBB     9DBB 9DBB 9DBB 9DBB 5B77 DABD DABD DABD DABD 5A3D E97E DABD 6534      E694B4       E694B4       F3B1988A
9DBC     9DBC 9DBC 9DBC 9DBC 5B78 DABE DABE DABE DABE 5A3E E96E DABE 6535      E694B5       E694B5       F3B1988B
9DBD     9DBD 9DBD 9DBD 9DBD 5B79 DABF DABF DABF DABF 5A3F E96F DABF 6537      E694B7       E694B7       F3B1988C
9DBE     9DBE 9DBE 9DBE 9DBE 5B7A DAC0 DAC0 DAC0 DAC0 5A40 E97C DAC0 6536      E694B6       E694B6       F3B1988D
9DBF     9DBF 9DBF 9DBF 9DBF 5B7B DAC1 DAC1 DAC1 DAC1 5A41 E9C1 DAC1 6538      E694B8       E694B8       F3B1988E
9DC0     9DC0 9DC0 9DC0 9DC0 5B7C DAC2 DAC2 DAC2 DAC2 5A42 E9C2 DAC2 754B      E7958B       E7958B       F3B1988F
9DC1     9DC1 9DC1 9DC1 9DC1 5B7D DAC3 DAC3 DAC3 DAC3 5A43 E9C3 DAC3 6548      E69588       E69588       F3B19890
9DC2     9DC2 9DC2 9DC2 9DC2 5B7F DAC4 DAC4 DAC4 DAC4 5A44 E9C4 DAC4 6556      E69596       E69596       F3B19891
9DC3     9DC3 9DC3 9DC3 9DC3 5B82 DAC5 DAC5 DAC5 DAC5 5A45 E9C5 DAC5 6555      E69595       E69595       F3B19892
9DC4     9DC4 9DC4 9DC4 9DC4 5B80 DAC6 DAC6 DAC6 DAC6 5A46 E9C6 DAC6 654D      E6958D       E6958D       F3B19893
9DC5     9DC5 9DC5 9DC5 9DC5 5B81 DAC7 DAC7 DAC7 DAC7 5A47 E9C7 DAC7 6558      E69598       E69598       F3B19894
9DC6     9DC6 9DC6 9DC6 9DC6 51B1 DAC8 DAC8 DAC8 DAC8 5A48 E9C8 DAC8 655E      E6959E       E6959E       F3B19895
9DC7     9DC7 9DC7 9DC7 9DC7 5B83 DAC9 DAC9 DAC9 DAC9 5A49 E9C9 DAC9 655D      E6959D       E6959D       F3B19896
9DC8     9DC8 9DC8 9DC8 9DC8 5B84 DACA DACA DACA DACA 5A4A E9D1 DACA 6572      E695B2       E695B2       F3B19897
9DC9     9DC9 9DC9 9DC9 9DC9 5B85 DACB DACB DACB DACB 5A4B E9D2 DACB 6578      E695B8       E695B8       F3B19898
9DCA     9DCA 9DCA 9DCA 9DCA 5B86 DACC DACC DACC DACC 5A4C E9D3 DACC 6582      E69682       E69682       F3B19899
9DCB     9DCB 9DCB 9DCB 9DCB 5B87 DACD DACD DACD DACD 5A4D E9D4 DACD 6583      E69683       E69683       F3B1989A
9DCC     9DCC 9DCC 9DCC 9DCC 5B88 DACE DACE DACE DACE 5A4E E9D5 DACE 8B8A      E8AE8A       E8AE8A       F3B1989B
9DCD     9DCD 9DCD 9DCD 9DCD 5B89 DACF DACF DACF DACF 5A4F E9D6 DACF 659B      E6969B       E6969B       F3B1989C
9DCE     9DCE 9DCE 9DCE 9DCE 5B8A DAD0 DAD0 DAD0 DAD0 5A50 E9D7 DAD0 659F      E6969F       E6969F       F3B1989D
9DCF     9DCF 9DCF 9DCF 9DCF 5B8B DAD1 DAD1 DAD1 DAD1 5A51 E9D8 DAD1 65AB      E696AB       E696AB       F3B1989E
9DD0     9DD0 9DD0 9DD0 9DD0 5B8C DAD2 DAD2 DAD2 DAD2 5A52 E9D9 DAD2 65B7      E696B7       E696B7       F3B1989F
9DD1     9DD1 9DD1 9DD1 9DD1 5B8D DAD3 DAD3 DAD3 DAD3 5A53 E9E2 DAD3 65C3      E69783       E69783       F3B198A0
9DD2     9DD2 9DD2 9DD2 9DD2 5B8E DAD4 DAD4 DAD4 DAD4 5A54 E9E3 DAD4 65C6      E69786       E69786       F3B198A1
9DD3     9DD3 9DD3 9DD3 9DD3 5B8F DAD5 DAD5 DAD5 DAD5 5A55 E9E4 DAD5 65C1      E69781       E69781       F3B198A2
9DD4     9DD4 9DD4 9DD4 9DD4 5B90 DAD6 DAD6 DAD6 DAD6 5A56 E9E5 DAD6 65C4      E69784       E69784       F3B198A3
9DD5     9DD5 9DD5 9DD5 9DD5 5B91 DAD7 DAD7 DAD7 DAD7 5A57 E9E6 DAD7 65CC      E6978C       E6978C       F3B198A4
9DD6     9DD6 9DD6 9DD6 9DD6 5B92 DAD8 DAD8 DAD8 DAD8 5A58 E9E7 DAD8 65D2      E69792       E69792       F3B198A5
9DD7     9DD7 9DD7 9DD7 9DD7 5165 DAD9 DAD9 DAD9 DAD9 5A59 E9E8 DAD9 65DB      E6979B       E6979B       F3B198A6
9DD8     9DD8 9DD8 9DD8 9DD8 5B93 DADA DADA DADA DADA 5A5A E9E9 DADA 65D9      E69799       E69799       F3B198A7
9DD9     9DD9 9DD9 9DD9 9DD9 5B95 DADB DADB DADB DADB 5A5B E94A DADB 65E0      E697A0       E697A0       F3B198A8
9DDA     9DDA 9DDA 9DDA 9DDA 5B94 DADC DADC DADC DADC 5A5C E95B DADC 65E1      E697A1       E697A1       F3B198A9
9DDB     9DDB 9DDB 9DDB 9DDB 5B96 DADD DADD DADD DADD 5A5D E95A DADD 65F1      E697B1       E697B1       F3B198AA
9DDC     9DDC 9DDC 9DDC 9DDC 5B9A DADE DADE DADE DADE 5A5E E95F DADE 6772      E69DB2       E69DB2       F3B198AB
9DDD     9DDD 9DDD 9DDD 9DDD 528E DADF DADF DADF DADF 5A5F E96D DADF 660A      E6988A       E6988A       F3B198AC
9DDE     9DDE 9DDE 9DDE 9DDE 5B9B DAE0 DAE0 DAE0 DAE0 5A60 E979 DAE0 6603      E69883       E69883       F3B198AD
9DDF     9DDF 9DDF 9DDF 9DDF 5B9C DAE1 DAE1 DAE1 DAE1 5A61 E957 DAE1 65FB      E697BB       E697BB       F3B198AE
9DE0     9DE0 9DE0 9DE0 9DE0 5BE0 DAE2 DAE2 DAE2 DAE2 5A62 E959 DAE2 6773      E69DB3       E69DB3       F3B198AF
9DE1     9DE1 9DE1 9DE1 9DE1 5B9F DAE3 DAE3 DAE3 DAE3 5A63 E962 DAE3 6635      E698B5       E698B5       F3B198B0
9DE2     9DE2 9DE2 9DE2 9DE2 5193 DAE4 DAE4 DAE4 DAE4 5A64 E963 DAE4 6636      E698B6       E698B6       F3B198B1
9DE3     9DE3 9DE3 9DE3 9DE3 5BA2 DAE5 DAE5 DAE5 DAE5 5A65 E964 DAE5 6634      E698B4       E698B4       F3B198B2
9DE4     9DE4 9DE4 9DE4 9DE4 5BA3 DAE6 DAE6 DAE6 DAE6 5A66 E965 DAE6 661C      E6989C       E6989C       F3B198B3
9DE5     9DE5 9DE5 9DE5 9DE5 51EA DAE7 DAE7 DAE7 DAE7 5A67 E966 DAE7 664F      E6998F       E6998F       F3B198B4
9DE6     9DE6 9DE6 9DE6 9DE6 5BA5 DAE8 DAE8 DAE8 DAE8 5A68 E967 DAE8 6644      E69984       E69984       F3B198B5
9DE7     9DE7 9DE7 9DE7 9DE7 5BA6 DAE9 DAE9 DAE9 DAE9 5A69 E968 DAE9 6649      E69989       E69989       F3B198B6
9DE8     9DE8 9DE8 9DE8 9DE8 5251 DAEA DAEA DAEA DAEA 5A6A E969 DAEA 6641      E69981       E69981       F3B198B7
9DE9     9DE9 9DE9 9DE9 9DE9 5BA9 DAEB DAEB DAEB DAEB 5A6B E970 DAEB 665E      E6999E       E6999E       F3B198B8
9DEA     9DEA 9DEA 9DEA 9DEA 5BAD DAEC DAEC DAEC DAEC 5A6C E971 DAEC 665D      E6999D       E6999D       F3B198B9
9DEB     9DEB 9DEB 9DEB 9DEB 5BAA DAED DAED DAED DAED 5A6D E972 DAED 6664      E699A4       E699A4       F3B198BA
9DEC     9DEC 9DEC 9DEC 9DEC 50A0 DAEE DAEE DAEE DAEE 5A6E E973 DAEE 6667      E699A7       E699A7       F3B198BB
9DED     9DED 9DED 9DED 9DED 51B2 DAEF DAEF DAEF DAEF 5A6F E974 DAEF 6668      E699A8       E699A8       F3B198BC
9DEE     9DEE 9DEE 9DEE 9DEE 4FCE DAF0 DAF0 DAF0 DAF0 5A70 E975 DAF0 665F      E6999F       E6999F       F3B198BD
9DEF     9DEF 9DEF 9DEF 9DEF 5BAC DAF1 DAF1 DAF1 DAF1 5A71 E976 DAF1 6662      E699A2       E699A2       F3B198BE
9DF0     9DF0 9DF0 9DF0 9DF0 5BB0 DAF2 DAF2 DAF2 DAF2 5A72 E977 DAF2 6670      E699B0       E699B0       F3B198BF
9DF1     9DF1 9DF1 9DF1 9DF1 5BB1 DAF3 DAF3 DAF3 DAF3 5A73 E978 DAF3 6683      E69A83       E69A83       F3B19980
9DF2     9DF2 9DF2 9DF2 9DF2 5BB2 DAF4 DAF4 DAF4 DAF4 5A74 E980 DAF4 6688      E69A88       E69A88       F3B19981
9DF3     9DF3 9DF3 9DF3 9DF3 4FAB DAF5 DAF5 DAF5 DAF5 5A75 E98B DAF5 668E      E69A8E       E69A8E       F3B19982
9DF4     9DF4 9DF4 9DF4 9DF4 4ED1 DAF6 DAF6 DAF6 DAF6 5A76 E99B DAF6 6689      E69A89       E69A89       F3B19983
9DF5     9DF5 9DF5 9DF5 9DF5 5BB3 DAF7 DAF7 DAF7 DAF7 5A77 E99C DAF7 6684      E69A84       E69A84       F3B19984
9DF6     9DF6 9DF6 9DF6 9DF6 5BB5 DAF8 DAF8 DAF8 DAF8 5A78 E9A0 DAF8 6698      E69A98       E69A98       F3B19985
9DF7     9DF7 9DF7 9DF7 9DF7 5BB7 DAF9 DAF9 DAF9 DAF9 5A79 E9AB DAF9 669D      E69A9D       E69A9D       F3B19986
9DF8     9DF8 9DF8 9DF8 9DF8 5BBA DAFA DAFA DAFA DAFA 5A7A E9B0 DAFA 66C1      E69B81       E69B81       F3B19987
9DF9     9DF9 9DF9 9DF9 9DF9 5243 DAFB DAFB DAFB DAFB 5A7B E9C0 DAFB 66B9      E69AB9       E69AB9       F3B19988
9DFA     9DFA 9DFA 9DFA 9DFA 5BBC DAFC DAFC DAFC DAFC 5A7C E96A DAFC 66C9      E69B89       E69B89       F3B19989
9DFB     9DFB 9DFB 9DFB 9DFB 5BBD DAFD DAFD DAFD DAFD 5A7D E9D0 DAFD 66BE      E69ABE       E69ABE       F3B1998A
9DFC     9DFC 9DFC 9DFC 9DFC 5BBE DAFE DAFE DAFE DAFE 5A7E E9A1 DAFE 66BC      E69ABC       E69ABC       F3B1998B
9E40     9E40 9E40 9E40 9E40 5BB9 DBA1 DBA1 DBA1 DBA1 5B21 4A4F DBA1 66C4      E69B84       E69B84       F3B1998C
9E41     9E41 9E41 9E41 9E41 5BBF DBA2 DBA2 DBA2 DBA2 5B22 4A7F DBA2 66B8      E69AB8       E69AB8       F3B1998D
9E42     9E42 9E42 9E42 9E42 5BC0 DBA3 DBA3 DBA3 DBA3 5B23 4A7B DBA3 66D6      E69B96       E69B96       F3B1998E
9E43     9E43 9E43 9E43 9E43 5BC1 DBA4 DBA4 DBA4 DBA4 5B24 4AE0 DBA4 66DA      E69B9A       E69B9A       F3B1998F
9E44     9E44 9E44 9E44 9E44 51CC DBA5 DBA5 DBA5 DBA5 5B25 4A6C DBA5 66E0      E69BA0       E69BA0       F3B19990
9E45     9E45 9E45 9E45 9E45 5B9E DBA6 DBA6 DBA6 DBA6 5B26 4A50 DBA6 663F      E698BF       E698BF       F3B19991
9E46     9E46 9E46 9E46 9E46 5BC2 DBA7 DBA7 DBA7 DBA7 5B27 4A7D DBA7 66E6      E69BA6       E69BA6       F3B19992
9E47     9E47 9E47 9E47 9E47 5BC3 DBA8 DBA8 DBA8 DBA8 5B28 4A4D DBA8 66E9      E69BA9       E69BA9       F3B19993
9E48     9E48 9E48 9E48 9E48 5BC4 DBA9 DBA9 DBA9 DBA9 5B29 4A5D DBA9 66F0      E69BB0       E69BB0       F3B19994
9E49     9E49 9E49 9E49 9E49 5BC5 DBAA DBAA DBAA DBAA 5B2A 4A5C DBAA 66F5      E69BB5       E69BB5       F3B19995
9E4A     9E4A 9E4A 9E4A 9E4A 5BC6 DBAB DBAB DBAB DBAB 5B2B 4A4E DBAB 66F7      E69BB7       E69BB7       F3B19996
9E4B     9E4B 9E4B 9E4B 9E4B 54FD DBAC DBAC DBAC DBAC 5B2C 4A6B DBAC 670F      E69C8F       E69C8F       F3B19997
9E4C     9E4C 9E4C 9E4C 9E4C 5BCA DBAD DBAD DBAD DBAD 5B2D 4A60 DBAD 6716      E69C96       E69C96       F3B19998
9E4D     9E4D 9E4D 9E4D 9E4D 5BCB DBAE DBAE DBAE DBAE 5B2E 4A4B DBAE 671E      E69C9E       E69C9E       F3B19999
9E4E     9E4E 9E4E 9E4E 9E4E 5BCD DBAF DBAF DBAF DBAF 5B2F 4A61 DBAF 6726      E69CA6       E69CA6       F3B1999A
9E4F     9E4F 9E4F 9E4F 9E4F 5BCE DBB0 DBB0 DBB0 DBB0 5B30 4AF0 DBB0 6727      E69CA7       E69CA7       F3B1999B
9E50     9E50 9E50 9E50 9E50 5BCF DBB1 DBB1 DBB1 DBB1 5B31 4AF1 DBB1 9738      E99CB8       E99CB8       F3B1999C
9E51     9E51 9E51 9E51 9E51 5BD0 DBB2 DBB2 DBB2 DBB2 5B32 4AF2 DBB2 672E      E69CAE       E69CAE       F3B1999D
9E52     9E52 9E52 9E52 9E52 5BD1 DBB3 DBB3 DBB3 DBB3 5B33 4AF3 DBB3 673F      E69CBF       E69CBF       F3B1999E
9E53     9E53 9E53 9E53 9E53 5BD2 DBB4 DBB4 DBB4 DBB4 5B34 4AF4 DBB4 6736      E69CB6       E69CB6       F3B1999F
9E54     9E54 9E54 9E54 9E54 534F DBB5 DBB5 DBB5 DBB5 5B35 4AF5 DBB5 6741      E69D81       E69D81       F3B199A0
9E55     9E55 9E55 9E55 9E55 5BD4 DBB6 DBB6 DBB6 DBB6 5B36 4AF6 DBB6 6738      E69CB8       E69CB8       F3B199A1
9E56     9E56 9E56 9E56 9E56 5BD3 DBB7 DBB7 DBB7 DBB7 5B37 4AF7 DBB7 6737      E69CB7       E69CB7       F3B199A2
9E57     9E57 9E57 9E57 9E57 5BD5 DBB8 DBB8 DBB8 DBB8 5B38 4AF8 DBB8 6746      E69D86       E69D86       F3B199A3
9E58     9E58 9E58 9E58 9E58 5BD6 DBB9 DBB9 DBB9 DBB9 5B39 4AF9 DBB9 675E      E69D9E       E69D9E       F3B199A4
9E59     9E59 9E59 9E59 9E59 5480 DBBA DBBA DBBA DBBA 5B3A 4A7A DBBA 6760      E69DA0       E69DA0       F3B199A5
9E5A     9E5A 9E5A 9E5A 9E5A 5BD7 DBBB DBBB DBBB DBBB 5B3B 4A5E DBBB 6759      E69D99       E69D99       F3B199A6
9E5B     9E5B 9E5B 9E5B 9E5B 5291 DBBC DBBC DBBC DBBC 5B3C 4A4C DBBC 6763      E69DA3       E69DA3       F3B199A7
9E5C     9E5C 9E5C 9E5C 9E5C 5BD9 DBBD DBBD DBBD DBBD 5B3D 4A7E DBBD 6764      E69DA4       E69DA4       F3B199A8
9E5D     9E5D 9E5D 9E5D 9E5D 5BDA DBBE DBBE DBBE DBBE 5B3E 4A6E DBBE 6789      E69E89       E69E89       F3B199A9
9E5E     9E5E 9E5E 9E5E 9E5E 5BDC DBBF DBBF DBBF DBBF 5B3F 4A6F DBBF 6770      E69DB0       E69DB0       F3B199AA
9E5F     9E5F 9E5F 9E5F 9E5F 5BDD DBC0 DBC0 DBC0 DBC0 5B40 4A7C DBC0 67A9      E69EA9       E69EA9       F3B199AB
9E60     9E60 9E60 9E60 9E60 54DF DBC1 DBC1 DBC1 DBC1 5B41 4AC1 DBC1 677C      E69DBC       E69DBC       F3B199AC
9E61     9E61 9E61 9E61 9E61 5BDE DBC2 DBC2 DBC2 DBC2 5B42 4AC2 DBC2 676A      E69DAA       E69DAA       F3B199AD
9E62     9E62 9E62 9E62 9E62 54F2 DBC3 DBC3 DBC3 DBC3 5B43 4AC3 DBC3 678C      E69E8C       E69E8C       F3B199AE
9E63     9E63 9E63 9E63 9E63 5BDF DBC4 DBC4 DBC4 DBC4 5B44 4AC4 DBC4 678B      E69E8B       E69E8B       F3B199AF
9E64     9E64 9E64 9E64 9E64 5BE1 DBC5 DBC5 DBC5 DBC5 5B45 4AC5 DBC5 67A6      E69EA6       E69EA6       F3B199B0
9E65     9E65 9E65 9E65 9E65 5BE2 DBC6 DBC6 DBC6 DBC6 5B46 4AC6 DBC6 67A1      E69EA1       E69EA1       F3B199B1
9E66     9E66 9E66 9E66 9E66 5BDB DBC7 DBC7 DBC7 DBC7 5B47 4AC7 DBC7 6785      E69E85       E69E85       F3B199B2
9E67     9E67 9E67 9E67 9E67 5BE4 DBC8 DBC8 DBC8 DBC8 5B48 4AC8 DBC8 67B7      E69EB7       E69EB7       F3B199B3
9E68     9E68 9E68 9E68 9E68 5BE5 DBC9 DBC9 DBC9 DBC9 5B49 4AC9 DBC9 67EF      E69FAF       E69FAF       F3B199B4
9E69     9E69 9E69 9E69 9E69 5BE6 DBCA DBCA DBCA DBCA 5B4A 4AD1 DBCA 67B4      E69EB4       E69EB4       F3B199B5
9E6A     9E6A 9E6A 9E6A 9E6A 5BE7 DBCB DBCB DBCB DBCB 5B4B 4AD2 DBCB 67EC      E69FAC       E69FAC       F3B199B6
9E6B     9E6B 9E6B 9E6B 9E6B 54C9 DBCC DBCC DBCC DBCC 5B4C 4AD3 DBCC 67B3      E69EB3       E69EB3       F3B199B7
9E6C     9E6C 9E6C 9E6C 9E6C 5BE8 DBCD DBCD DBCD DBCD 5B4D 4AD4 DBCD 67E9      E69FA9       E69FA9       F3B199B8
9E6D     9E6D 9E6D 9E6D 9E6D 5BE9 DBCE DBCE DBCE DBCE 5B4E 4AD5 DBCE 67B8      E69EB8       E69EB8       F3B199B9
9E6E     9E6E 9E6E 9E6E 9E6E 5BEB DBCF DBCF DBCF DBCF 5B4F 4AD6 DBCF 67E4      E69FA4       E69FA4       F3B199BA
9E6F     9E6F 9E6F 9E6F 9E6F 54F3 DBD0 DBD0 DBD0 DBD0 5B50 4AD7 DBD0 67DE      E69F9E       E69F9E       F3B199BB
9E70     9E70 9E70 9E70 9E70 5BED DBD1 DBD1 DBD1 DBD1 5B51 4AD8 DBD1 67DD      E69F9D       E69F9D       F3B199BC
9E71     9E71 9E71 9E71 9E71 5BEE DBD2 DBD2 DBD2 DBD2 5B52 4AD9 DBD2 67E2      E69FA2       E69FA2       F3B199BD
9E72     9E72 9E72 9E72 9E72 5BEF DBD3 DBD3 DBD3 DBD3 5B53 4AE2 DBD3 67EE      E69FAE       E69FAE       F3B199BE
9E73     9E73 9E73 9E73 9E73 5BF2 DBD4 DBD4 DBD4 DBD4 5B54 4AE3 DBD4 67B9      E69EB9       E69EB9       F3B199BF
9E74     9E74 9E74 9E74 9E74 5BF1 DBD5 DBD5 DBD5 DBD5 5B55 4AE4 DBD5 67CE      E69F8E       E69F8E       F3B19A80
9E75     9E75 9E75 9E75 9E75 5BF4 DBD6 DBD6 DBD6 DBD6 5B56 4AE5 DBD6 67C6      E69F86       E69F86       F3B19A81
9E76     9E76 9E76 9E76 9E76 5BEA DBD7 DBD7 DBD7 DBD7 5B57 4AE6 DBD7 67E7      E69FA7       E69FA7       F3B19A82
9E77     9E77 9E77 9E77 9E77 4D55 C9B0 DBD8 DBD8 C9B0 4930 C9F0 DBD8 6A9C      E6AA9C       E6AA9C       F3B19A83
9E78     9E78 9E78 9E78 9E78 5BF5 DBD9 DBD9 DBD9 DBD9 5B59 4AE8 DBD9 681E      E6A09E       E6A09E       F3B19A84
9E79     9E79 9E79 9E79 9E79 5BF6 DBDA DBDA DBDA DBDA 5B5A 4AE9 DBDA 6846      E6A186       E6A186       F3B19A85
9E7A     9E7A 9E7A 9E7A 9E7A 52DE DBDB DBDB DBDB DBDB 5B5B 4A4A DBDB 6829      E6A0A9       E6A0A9       F3B19A86
9E7B     9E7B 9E7B 9E7B 9E7B 5BF8 DBDC DBDC DBDC DBDC 5B5C 4A5B DBDC 6840      E6A180       E6A180       F3B19A87
9E7C     9E7C 9E7C 9E7C 9E7C 5BF7 DBDD DBDD DBDD DBDD 5B5D 4A5A DBDD 684D      E6A18D       E6A18D       F3B19A88
9E7D     9E7D 9E7D 9E7D 9E7D 5BFA DBDE DBDE DBDE DBDE 5B5E 4A5F DBDE 6832      E6A0B2       E6A0B2       F3B19A89
9E7E     9E7E 9E7E 9E7E 9E7E 5BFB DBDF DBDF DBDF DBDF 5B5F 4A6D DBDF 684E      E6A18E       E6A18E       F3B19A8A
9E80     9E80 9E80 9E80 9E80 5361 DBE0 DBE0 DBE0 DBE0 5B60 4A79 DBE0 68B3      E6A2B3       E6A2B3       F3B19A8B
9E81     9E81 9E81 9E81 9E81 52E2 DBE1 DBE1 DBE1 DBE1 5B61 4A57 DBE1 682B      E6A0AB       E6A0AB       F3B19A8C
9E82     9E82 9E82 9E82 9E82 5BFD DBE2 DBE2 DBE2 DBE2 5B62 4A59 DBE2 6859      E6A199       E6A199       F3B19A8D
9E83     9E83 9E83 9E83 9E83 5BFC DBE3 DBE3 DBE3 DBE3 5B63 4A62 DBE3 6863      E6A1A3       E6A1A3       F3B19A8E
9E84     9E84 9E84 9E84 9E84 5C41 DBE4 DBE4 DBE4 DBE4 5B64 4A63 DBE4 6877      E6A1B7       E6A1B7       F3B19A8F
9E85     9E85 9E85 9E85 9E85 5C42 DBE5 DBE5 DBE5 DBE5 5B65 4A64 DBE5 687F      E6A1BF       E6A1BF       F3B19A90
9E86     9E86 9E86 9E86 9E86 5C43 DBE6 DBE6 DBE6 DBE6 5B66 4A65 DBE6 689F      E6A29F       E6A29F       F3B19A91
9E87     9E87 9E87 9E87 9E87 5C45 DBE7 DBE7 DBE7 DBE7 5B67 4A66 DBE7 688F      E6A28F       E6A28F       F3B19A92
9E88     9E88 9E88 9E88 9E88 5C46 DBE8 DBE8 DBE8 DBE8 5B68 4A67 DBE8 68AD      E6A2AD       E6A2AD       F3B19A93
9E89     9E89 9E89 9E89 9E89 5C47 DBE9 DBE9 DBE9 DBE9 5B69 4A68 DBE9 6894      E6A294       E6A294       F3B19A94
9E8A     9E8A 9E8A 9E8A 9E8A 4E92 DBEA DBEA DBEA DBEA 5B6A 4A69 DBEA 689D      E6A29D       E6A29D       F3B19A95
9E8B     9E8B 9E8B 9E8B 9E8B 53BC DBEB DBEB DBEB DBEB 5B6B 4A70 DBEB 689B      E6A29B       E6A29B       F3B19A96
9E8C     9E8C 9E8C 9E8C 9E8C 5C48 DBEC DBEC DBEC DBEC 5B6C 4A71 DBEC 6883      E6A283       E6A283       F3B19A97
9E8D     9E8D 9E8D 9E8D 9E8D 545D C5EE DBED DBED C5EE 456E C573 DBED 6AAE      E6AAAE       E6AAAE       F3B19A98
9E8E     9E8E 9E8E 9E8E 9E8E 5C4A DBEE DBEE DBEE DBEE 5B6E 4A73 DBEE 68B9      E6A2B9       E6A2B9       F3B19A99
9E8F     9E8F 9E8F 9E8F 9E8F 5C4B DBEF DBEF DBEF DBEF 5B6F 4A74 DBEF 6874      E6A1B4       E6A1B4       F3B19A9A
9E90     9E90 9E90 9E90 9E90 5C4C DBF0 DBF0 DBF0 DBF0 5B70 4A75 DBF0 68B5      E6A2B5       E6A2B5       F3B19A9B
9E91     9E91 9E91 9E91 9E91 52C6 DBF1 DBF1 DBF1 DBF1 5B71 4A76 DBF1 68A0      E6A2A0       E6A2A0       F3B19A9C
9E92     9E92 9E92 9E92 9E92 5C4D DBF2 DBF2 DBF2 DBF2 5B72 4A77 DBF2 68BA      E6A2BA       E6A2BA       F3B19A9D
9E93     9E93 9E93 9E93 9E93 5C4E DBF3 DBF3 DBF3 DBF3 5B73 4A78 DBF3 690F      E6A48F       E6A48F       F3B19A9E
9E94     9E94 9E94 9E94 9E94 5BFE DBF4 DBF4 DBF4 DBF4 5B74 4A80 DBF4 688D      E6A28D       E6A28D       F3B19A9F
9E95     9E95 9E95 9E95 9E95 5C44 DBF5 DBF5 DBF5 DBF5 5B75 4A8B DBF5 687E      E6A1BE       E6A1BE       F3B19AA0
9E96     9E96 9E96 9E96 9E96 5C4F DBF6 DBF6 DBF6 DBF6 5B76 4A9B DBF6 6901      E6A481       E6A481       F3B19AA1
9E97     9E97 9E97 9E97 9E97 5C50 DBF7 DBF7 DBF7 DBF7 5B77 4A9C DBF7 68CA      E6A38A       E6A38A       F3B19AA2
9E98     9E98 9E98 9E98 9E98 5C51 DBF8 DBF8 DBF8 DBF8 5B78 4AA0 DBF8 6908      E6A488       E6A488       F3B19AA3
9E99     9E99 9E99 9E99 9E99 5C52 DBF9 DBF9 DBF9 DBF9 5B79 4AAB DBF9 68D8      E6A398       E6A398       F3B19AA4
9E9A     9E9A 9E9A 9E9A 9E9A 5C61 DBFA DBFA DBFA DBFA 5B7A 4AB0 DBFA 6922      E6A4A2       E6A4A2       F3B19AA5
9E9B     9E9B 9E9B 9E9B 9E9B 5C53 DBFB DBFB DBFB DBFB 5B7B 4AC0 DBFB 6926      E6A4A6       E6A4A6       F3B19AA6
9E9C     9E9C 9E9C 9E9C 9E9C 5C54 DBFC DBFC DBFC DBFC 5B7C 4A6A DBFC 68E1      E6A3A1       E6A3A1       F3B19AA7
9E9D     9E9D 9E9D 9E9D 9E9D 5C55 DBFD DBFD DBFD DBFD 5B7D 4AD0 DBFD 690C      E6A48C       E6A48C       F3B19AA8
9E9E     9E9E 9E9E 9E9E 9E9E 5C56 DBFE DBFE DBFE DBFE 5B7E 4AA1 DBFE 68CD      E6A38D       E6A38D       F3B19AA9
9E9F     9E9F 9E9F 9E9F 9E9F 5C57 DCA1 DCA1 DCA1 DCA1 5C21 5B4F DCA1 68D4      E6A394       E6A394       F3B19AAA
9EA0     9EA0 9EA0 9EA0 9EA0 5C58 DCA2 DCA2 DCA2 DCA2 5C22 5B7F DCA2 68E7      E6A3A7       E6A3A7       F3B19AAB
9EA1     9EA1 9EA1 9EA1 9EA1 5C59 DCA3 DCA3 DCA3 DCA3 5C23 5B7B DCA3 68D5      E6A395       E6A395       F3B19AAC
9EA2     9EA2 9EA2 9EA2 9EA2 5C69 DCA4 DCA4 DCA4 DCA4 5C24 5BE0 DCA4 6936      E6A4B6       E6A4B6       F3B19AAD
9EA3     9EA3 9EA3 9EA3 9EA3 5C5A DCA5 DCA5 DCA5 DCA5 5C25 5B6C DCA5 6912      E6A492       E6A492       F3B19AAE
9EA4     9EA4 9EA4 9EA4 9EA4 5C5C DCA6 DCA6 DCA6 DCA6 5C26 5B50 DCA6 6904      E6A484       E6A484       F3B19AAF
9EA5     9EA5 9EA5 9EA5 9EA5 54C2 DCA7 DCA7 DCA7 DCA7 5C27 5B7D DCA7 68D7      E6A397       E6A397       F3B19AB0
9EA6     9EA6 9EA6 9EA6 9EA6 5C5D DCA8 DCA8 DCA8 DCA8 5C28 5B4D DCA8 68E3      E6A3A3       E6A3A3       F3B19AB1
9EA7     9EA7 9EA7 9EA7 9EA7 53FA DCA9 DCA9 DCA9 DCA9 5C29 5B5D DCA9 6925      E6A4A5       E6A4A5       F3B19AB2
9EA8     9EA8 9EA8 9EA8 9EA8 5494 DCAA DCAA DCAA DCAA 5C2A 5B5C DCAA 68F9      E6A3B9       E6A3B9       F3B19AB3
9EA9     9EA9 9EA9 9EA9 9EA9 5C5E DCAB DCAB DCAB DCAB 5C2B 5B4E DCAB 68E0      E6A3A0       E6A3A0       F3B19AB4
9EAA     9EAA 9EAA 9EAA 9EAA 5C5B DCAC DCAC DCAC DCAC 5C2C 5B6B DCAC 68EF      E6A3AF       E6A3AF       F3B19AB5
9EAB     9EAB 9EAB 9EAB 9EAB 5C65 DCAD DCAD DCAD DCAD 5C2D 5B60 DCAD 6928      E6A4A8       E6A4A8       F3B19AB6
9EAC     9EAC 9EAC 9EAC 9EAC 5C62 DCAE DCAE DCAE DCAE 5C2E 5B4B DCAE 692A      E6A4AA       E6A4AA       F3B19AB7
9EAD     9EAD 9EAD 9EAD 9EAD 5370 DCAF DCAF DCAF DCAF 5C2F 5B61 DCAF 691A      E6A49A       E6A49A       F3B19AB8
9EAE     9EAE 9EAE 9EAE 9EAE 5C64 DCB0 DCB0 DCB0 DCB0 5C30 5BF0 DCB0 6923      E6A4A3       E6A4A3       F3B19AB9
9EAF     9EAF 9EAF 9EAF 9EAF 5C63 DCB1 DCB1 DCB1 DCB1 5C31 5BF1 DCB1 6921      E6A4A1       E6A4A1       F3B19ABA
9EB0     9EB0 9EB0 9EB0 9EB0 5C60 DCB2 DCB2 DCB2 DCB2 5C32 5BF2 DCB2 68C6      E6A386       E6A386       F3B19ABB
9EB1     9EB1 9EB1 9EB1 9EB1 5C67 DCB3 DCB3 DCB3 DCB3 5C33 5BF3 DCB3 6979      E6A5B9       E6A5B9       F3B19ABC
9EB2     9EB2 9EB2 9EB2 9EB2 5C68 DCB4 DCB4 DCB4 DCB4 5C34 5BF4 DCB4 6977      E6A5B7       E6A5B7       F3B19ABD
9EB3     9EB3 9EB3 9EB3 9EB3 54E2 DCB5 DCB5 DCB5 DCB5 5C35 5BF5 DCB5 695C      E6A59C       E6A59C       F3B19ABE
9EB4     9EB4 9EB4 9EB4 9EB4 5C6A DCB6 DCB6 DCB6 DCB6 5C36 5BF6 DCB6 6978      E6A5B8       E6A5B8       F3B19ABF
9EB5     9EB5 9EB5 9EB5 9EB5 54A2 DCB7 DCB7 DCB7 DCB7 5C37 5BF7 DCB7 696B      E6A5AB       E6A5AB       F3B19B80
9EB6     9EB6 9EB6 9EB6 9EB6 5C6B DCB8 DCB8 DCB8 DCB8 5C38 5BF8 DCB8 6954      E6A594       E6A594       F3B19B81
9EB7     9EB7 9EB7 9EB7 9EB7 5C74 DCB9 DCB9 DCB9 DCB9 5C39 5BF9 DCB9 697E      E6A5BE       E6A5BE       F3B19B82
9EB8     9EB8 9EB8 9EB8 9EB8 53EF DCBA DCBA DCBA DCBA 5C3A 5B7A DCBA 696E      E6A5AE       E6A5AE       F3B19B83
9EB9     9EB9 9EB9 9EB9 9EB9 52ED DCBB DCBB DCBB DCBB 5C3B 5B5E DCBB 6939      E6A4B9       E6A4B9       F3B19B84
9EBA     9EBA 9EBA 9EBA 9EBA 5C6D DCBC DCBC DCBC DCBC 5C3C 5B4C DCBC 6974      E6A5B4       E6A5B4       F3B19B85
9EBB     9EBB 9EBB 9EBB 9EBB 5C6F DCBD DCBD DCBD DCBD 5C3D 5B7E DCBD 693D      E6A4BD       E6A4BD       F3B19B86
9EBC     9EBC 9EBC 9EBC 9EBC 5C70 DCBE DCBE DCBE DCBE 5C3E 5B6E DCBE 6959      E6A599       E6A599       F3B19B87
9EBD     9EBD 9EBD 9EBD 9EBD 5C71 DCBF DCBF DCBF DCBF 5C3F 5B6F DCBF 6930      E6A4B0       E6A4B0       F3B19B88
9EBE     9EBE 9EBE 9EBE 9EBE 527C DCC0 DCC0 DCC0 DCC0 5C40 5B7C DCC0 6961      E6A5A1       E6A5A1       F3B19B89
9EBF     9EBF 9EBF 9EBF 9EBF 5C72 DCC1 DCC1 DCC1 DCC1 5C41 5BC1 DCC1 695E      E6A59E       E6A59E       F3B19B8A
9EC0     9EC0 9EC0 9EC0 9EC0 5C73 DCC2 DCC2 DCC2 DCC2 5C42 5BC2 DCC2 695D      E6A59D       E6A59D       F3B19B8B
9EC1     9EC1 9EC1 9EC1 9EC1 5C75 DCC3 DCC3 DCC3 DCC3 5C43 5BC3 DCC3 6981      E6A681       E6A681       F3B19B8C
9EC2     9EC2 9EC2 9EC2 9EC2 5C6C DCC4 DCC4 DCC4 DCC4 5C44 5BC4 DCC4 696A      E6A5AA       E6A5AA       F3B19B8D
9EC3     9EC3 9EC3 9EC3 9EC3 5C76 DCC5 DCC5 DCC5 DCC5 5C45 5BC5 DCC5 69B2      E6A6B2       E6A6B2       F3B19B8E
9EC4     9EC4 9EC4 9EC4 9EC4 514A DCC6 DCC6 DCC6 DCC6 5C46 5BC6 DCC6 69AE      E6A6AE       E6A6AE       F3B19B8F
9EC5     9EC5 9EC5 9EC5 9EC5 548C DCC7 DCC7 DCC7 DCC7 5C47 5BC7 DCC7 69D0      E6A790       E6A790       F3B19B90
9EC6     9EC6 9EC6 9EC6 9EC6 5C77 DCC8 DCC8 DCC8 DCC8 5C48 5BC8 DCC8 69BF      E6A6BF       E6A6BF       F3B19B91
9EC7     9EC7 9EC7 9EC7 9EC7 5C7A DCC9 DCC9 DCC9 DCC9 5C49 5BC9 DCC9 69C1      E6A781       E6A781       F3B19B92
9EC8     9EC8 9EC8 9EC8 9EC8 5C7B DCCA DCCA DCCA DCCA 5C4A 5BD1 DCCA 69D3      E6A793       E6A793       F3B19B93
9EC9     9EC9 9EC9 9EC9 9EC9 5C7C DCCB DCCB DCCB DCCB 5C4B 5BD2 DCCB 69BE      E6A6BE       E6A6BE       F3B19B94
9ECA     9ECA 9ECA 9ECA 9ECA 5C7D DCCC DCCC DCCC DCCC 5C4C 5BD3 DCCC 69CE      E6A78E       E6A78E       F3B19B95
9ECB     9ECB 9ECB 9ECB 9ECB 5C7E DCCD DCCD DCCD DCCD 5C4D 5BD4 DCCD 5BE8      E5AFA8       E5AFA8       F3B19B96
9ECC     9ECC 9ECC 9ECC 9ECC 5C7F DCCE DCCE DCCE DCCE 5C4E 5BD5 DCCE 69CA      E6A78A       E6A78A       F3B19B97
9ECD     9ECD 9ECD 9ECD 9ECD 5C81 DCCF DCCF DCCF DCCF 5C4F 5BD6 DCCF 69DD      E6A79D       E6A79D       F3B19B98
9ECE     9ECE 9ECE 9ECE 9ECE 5C82 DCD0 DCD0 DCD0 DCD0 5C50 5BD7 DCD0 69BB      E6A6BB       E6A6BB       F3B19B99
9ECF     9ECF 9ECF 9ECF 9ECF 5C83 DCD1 DCD1 DCD1 DCD1 5C51 5BD8 DCD1 69C3      E6A783       E6A783       F3B19B9A
9ED0     9ED0 9ED0 9ED0 9ED0 538D DCD2 DCD2 DCD2 DCD2 5C52 5BD9 DCD2 69A7      E6A6A7       E6A6A7       F3B19B9B
9ED1     9ED1 9ED1 9ED1 9ED1 5C88 DCD3 DCD3 DCD3 DCD3 5C53 5BE2 DCD3 6A2E      E6A8AE       E6A8AE       F3B19B9C
9ED2     9ED2 9ED2 9ED2 9ED2 5476 DCD4 DCD4 DCD4 DCD4 5C54 5BE3 DCD4 6991      E6A691       E6A691       F3B19B9D
9ED3     9ED3 9ED3 9ED3 9ED3 5C84 DCD5 DCD5 DCD5 DCD5 5C55 5BE4 DCD5 69A0      E6A6A0       E6A6A0       F3B19B9E
9ED4     9ED4 9ED4 9ED4 9ED4 5C85 DCD6 DCD6 DCD6 DCD6 5C56 5BE5 DCD6 699C      E6A69C       E6A69C       F3B19B9F
9ED5     9ED5 9ED5 9ED5 9ED5 5C86 DCD7 DCD7 DCD7 DCD7 5C57 5BE6 DCD7 6995      E6A695       E6A695       F3B19BA0
9ED6     9ED6 9ED6 9ED6 9ED6 53B2 DCD8 DCD8 DCD8 DCD8 5C58 5BE7 DCD8 69B4      E6A6B4       E6A6B4       F3B19BA1
9ED7     9ED7 9ED7 9ED7 9ED7 5C87 DCD9 DCD9 DCD9 DCD9 5C59 5BE8 DCD9 69DE      E6A79E       E6A79E       F3B19BA2
9ED8     9ED8 9ED8 9ED8 9ED8 5C89 DCDA DCDA DCDA DCDA 5C5A 5BE9 DCDA 69E8      E6A7A8       E6A7A8       F3B19BA3
9ED9     9ED9 9ED9 9ED9 9ED9 5C8A DCDB DCDB DCDB DCDB 5C5B 5B4A DCDB 6A02      E6A882       E6A882       F3B19BA4
9EDA     9EDA 9EDA 9EDA 9EDA 5C8B DCDC DCDC DCDC DCDC 5C5C 5B5B DCDC 6A1B      E6A89B       E6A89B       F3B19BA5
9EDB     9EDB 9EDB 9EDB 9EDB 5C8C DCDD DCDD DCDD DCDD 5C5D 5B5A DCDD 69FF      E6A7BF       E6A7BF       F3B19BA6
9EDC     9EDC 9EDC 9EDC 9EDC 5CC1 DCDE DCDE DCDE DCDE 5C5E 5B5F DCDE 6B0A      E6AC8A       E6AC8A       F3B19BA7
9EDD     9EDD 9EDD 9EDD 9EDD 5C8D DCDF DCDF DCDF DCDF 5C5F 5B6D DCDF 69F9      E6A7B9       E6A7B9       F3B19BA8
9EDE     9EDE 9EDE 9EDE 9EDE 5C8E DCE0 DCE0 DCE0 DCE0 5C60 5B79 DCE0 69F2      E6A7B2       E6A7B2       F3B19BA9
9EDF     9EDF 9EDF 9EDF 9EDF 5C8F DCE1 DCE1 DCE1 DCE1 5C61 5B57 DCE1 69E7      E6A7A7       E6A7A7       F3B19BAA
9EE0     9EE0 9EE0 9EE0 9EE0 53BB DCE2 DCE2 DCE2 DCE2 5C62 5B59 DCE2 6A05      E6A885       E6A885       F3B19BAB
9EE1     9EE1 9EE1 9EE1 9EE1 5C80 DCE3 DCE3 DCE3 DCE3 5C63 5B62 DCE3 69B1      E6A6B1       E6A6B1       F3B19BAC
9EE2     9EE2 9EE2 9EE2 9EE2 5C91 DCE4 DCE4 DCE4 DCE4 5C64 5B63 DCE4 6A1E      E6A89E       E6A89E       F3B19BAD
9EE3     9EE3 9EE3 9EE3 9EE3 5C92 DCE5 DCE5 DCE5 DCE5 5C65 5B64 DCE5 69ED      E6A7AD       E6A7AD       F3B19BAE
9EE4     9EE4 9EE4 9EE4 9EE4 5C93 DCE6 DCE6 DCE6 DCE6 5C66 5B65 DCE6 6A14      E6A894       E6A894       F3B19BAF
9EE5     9EE5 9EE5 9EE5 9EE5 5C94 DCE7 DCE7 DCE7 DCE7 5C67 5B66 DCE7 69EB      E6A7AB       E6A7AB       F3B19BB0
9EE6     9EE6 9EE6 9EE6 9EE6 5C95 DCE8 DCE8 DCE8 DCE8 5C68 5B67 DCE8 6A0A      E6A88A       E6A88A       F3B19BB1
9EE7     9EE7 9EE7 9EE7 9EE7 5C97 DCE9 DCE9 DCE9 DCE9 5C69 5B68 DCE9 6A12      E6A892       E6A892       F3B19BB2
9EE8     9EE8 9EE8 9EE8 9EE8 5CB7 DCEA DCEA DCEA DCEA 5C6A 5B69 DCEA 6AC1      E6AB81       E6AB81       F3B19BB3
9EE9     9EE9 9EE9 9EE9 9EE9 5C98 DCEB DCEB DCEB DCEB 5C6B 5B70 DCEB 6A23      E6A8A3       E6A8A3       F3B19BB4
9EEA     9EEA 9EEA 9EEA 9EEA 5C99 DCEC DCEC DCEC DCEC 5C6C 5B71 DCEC 6A13      E6A893       E6A893       F3B19BB5
9EEB     9EEB 9EEB 9EEB 9EEB 5C9C DCED DCED DCED DCED 5C6D 5B72 DCED 6A44      E6A984       E6A984       F3B19BB6
9EEC     9EEC 9EEC 9EEC 9EEC 5C9D DCEE DCEE DCEE DCEE 5C6E 5B73 DCEE 6A0C      E6A88C       E6A88C       F3B19BB7
9EED     9EED 9EED 9EED 9EED 5CA8 DCEF DCEF DCEF DCEF 5C6F 5B74 DCEF 6A72      E6A9B2       E6A9B2       F3B19BB8
9EEE     9EEE 9EEE 9EEE 9EEE 5C9F DCF0 DCF0 DCF0 DCF0 5C70 5B75 DCF0 6A36      E6A8B6       E6A8B6       F3B19BB9
9EEF     9EEF 9EEF 9EEF 9EEF 5CA0 DCF1 DCF1 DCF1 DCF1 5C71 5B76 DCF1 6A78      E6A9B8       E6A9B8       F3B19BBA
9EF0     9EF0 9EF0 9EF0 9EF0 5CA1 DCF2 DCF2 DCF2 DCF2 5C72 5B77 DCF2 6A47      E6A987       E6A987       F3B19BBB
9EF1     9EF1 9EF1 9EF1 9EF1 5CA2 DCF3 DCF3 DCF3 DCF3 5C73 5B78 DCF3 6A62      E6A9A2       E6A9A2       F3B19BBC
9EF2     9EF2 9EF2 9EF2 9EF2 5CA3 DCF4 DCF4 DCF4 DCF4 5C74 5B80 DCF4 6A59      E6A999       E6A999       F3B19BBD
9EF3     9EF3 9EF3 9EF3 9EF3 5CA4 DCF5 DCF5 DCF5 DCF5 5C75 5B8B DCF5 6A66      E6A9A6       E6A9A6       F3B19BBE
9EF4     9EF4 9EF4 9EF4 9EF4 5CA5 DCF6 DCF6 DCF6 DCF6 5C76 5B9B DCF6 6A48      E6A988       E6A988       F3B19BBF
9EF5     9EF5 9EF5 9EF5 9EF5 5CA7 DCF7 DCF7 DCF7 DCF7 5C77 5B9C DCF7 6A38      E6A8B8       E6A8B8       F3B19C80
9EF6     9EF6 9EF6 9EF6 9EF6 5C96 DCF8 DCF8 DCF8 DCF8 5C78 5BA0 DCF8 6A22      E6A8A2       E6A8A2       F3B19C81
9EF7     9EF7 9EF7 9EF7 9EF7 5CAA DCF9 DCF9 DCF9 DCF9 5C79 5BAB DCF9 6A90      E6AA90       E6AA90       F3B19C82
9EF8     9EF8 9EF8 9EF8 9EF8 5CAB DCFA DCFA DCFA DCFA 5C7A 5BB0 DCFA 6A8D      E6AA8D       E6AA8D       F3B19C83
9EF9     9EF9 9EF9 9EF9 9EF9 5C9E DCFB DCFB DCFB DCFB 5C7B 5BC0 DCFB 6AA0      E6AAA0       E6AAA0       F3B19C84
9EFA     9EFA 9EFA 9EFA 9EFA 5CAC DCFC DCFC DCFC DCFC 5C7C 5B6A DCFC 6A84      E6AA84       E6AA84       F3B19C85
9EFB     9EFB 9EFB 9EFB 9EFB 5CAD DCFD DCFD DCFD DCFD 5C7D 5BD0 DCFD 6AA2      E6AAA2       E6AAA2       F3B19C86
9EFC     9EFC 9EFC 9EFC 9EFC 5CAE DCFE DCFE DCFE DCFE 5C7E 5BA1 DCFE 6AA3      E6AAA3       E6AAA3       F3B19C87
9F40     9F40 9F40 9F40 9F40 5CB0 DDA1 DDA1 DDA1 DDA1 5D21 5A4F DDA1 6A97      E6AA97       E6AA97       F3B19C88
9F41     9F41 9F41 9F41 9F41 5CBE DDA2 DDA2 DDA2 DDA2 5D22 5A7F DDA2 8617      E89897       E89897       F3B19C89
9F42     9F42 9F42 9F42 9F42 5CB3 DDA3 DDA3 DDA3 DDA3 5D23 5A7B DDA3 6ABB      E6AABB       E6AABB       F3B19C8A
9F43     9F43 9F43 9F43 9F43 528B DDA4 DDA4 DDA4 DDA4 5D24 5AE0 DDA4 6AC3      E6AB83       E6AB83       F3B19C8B
9F44     9F44 9F44 9F44 9F44 5CB4 DDA5 DDA5 DDA5 DDA5 5D25 5A6C DDA5 6AC2      E6AB82       E6AB82       F3B19C8C
9F45     9F45 9F45 9F45 9F45 5CB5 DDA6 DDA6 DDA6 DDA6 5D26 5A50 DDA6 6AB8      E6AAB8       E6AAB8       F3B19C8D
9F46     9F46 9F46 9F46 9F46 5CB6 DDA7 DDA7 DDA7 DDA7 5D27 5A7D DDA7 6AB3      E6AAB3       E6AAB3       F3B19C8E
9F47     9F47 9F47 9F47 9F47 5CB1 DDA8 DDA8 DDA8 DDA8 5D28 5A4D DDA8 6AAC      E6AAAC       E6AAAC       F3B19C8F
9F48     9F48 9F48 9F48 9F48 5CB8 DDA9 DDA9 DDA9 DDA9 5D29 5A5D DDA9 6ADE      E6AB9E       E6AB9E       F3B19C90
9F49     9F49 9F49 9F49 9F49 5CBA DDAA DDAA DDAA DDAA 5D2A 5A5C DDAA 6AD1      E6AB91       E6AB91       F3B19C91
9F4A     9F4A 9F4A 9F4A 9F4A 5362 DDAB DDAB DDAB DDAB 5D2B 5A4E DDAB 6ADF      E6AB9F       E6AB9F       F3B19C92
9F4B     9F4B 9F4B 9F4B 9F4B 5CB2 DDAC DDAC DDAC DDAC 5D2C 5A6B DDAC 6AAA      E6AAAA       E6AAAA       F3B19C93
9F4C     9F4C 9F4C 9F4C 9F4C 5CBB DDAD DDAD DDAD DDAD 5D2D 5A60 DDAD 6ADA      E6AB9A       E6AB9A       F3B19C94
9F4D     9F4D 9F4D 9F4D 9F4D 5CBF DDAE DDAE DDAE DDAE 5D2E 5A4B DDAE 6AEA      E6ABAA       E6ABAA       F3B19C95
9F4E     9F4E 9F4E 9F4E 9F4E 52EE DDAF DDAF DDAF DDAF 5D2F 5A61 DDAF 6AFB      E6ABBB       E6ABBB       F3B19C96
9F4F     9F4F 9F4F 9F4F 9F4F 5CC0 DDB0 DDB0 DDB0 DDB0 5D30 5AF0 DDB0 6B05      E6AC85       E6AC85       F3B19C97
9F50     9F50 9F50 9F50 9F50 5CBD DDB1 DDB1 DDB1 DDB1 5D31 5AF1 DDB1 8616      E89896       E89896       F3B19C98
9F51     9F51 9F51 9F51 9F51 5CC2 DDB2 DDB2 DDB2 DDB2 5D32 5AF2 DDB2 6AFA      E6ABBA       E6ABBA       F3B19C99
9F52     9F52 9F52 9F52 9F52 5CC3 DDB3 DDB3 DDB3 DDB3 5D33 5AF3 DDB3 6B12      E6AC92       E6AC92       F3B19C9A
9F53     9F53 9F53 9F53 9F53 5CC4 DDB4 DDB4 DDB4 DDB4 5D34 5AF4 DDB4 6B16      E6AC96       E6AC96       F3B19C9B
9F54     9F54 9F54 9F54 9F54 6786 DDB5 DDB5 DDB5 DDB5 5D35 5AF5 DDB5 9B31      E9ACB1       E9ACB1       F3B19C9C
9F55     9F55 9F55 9F55 9F55 5CC5 DDB6 DDB6 DDB6 DDB6 5D36 5AF6 DDB6 6B1F      E6AC9F       E6AC9F       F3B19C9D
9F56     9F56 9F56 9F56 9F56 5CC6 DDB7 DDB7 DDB7 DDB7 5D37 5AF7 DDB7 6B38      E6ACB8       E6ACB8       F3B19C9E
9F57     9F57 9F57 9F57 9F57 5CC7 DDB8 DDB8 DDB8 DDB8 5D38 5AF8 DDB8 6B37      E6ACB7       E6ACB7       F3B19C9F
9F58     9F58 9F58 9F58 9F58 5CC9 DDB9 DDB9 DDB9 DDB9 5D39 5AF9 DDB9 76DC      E79B9C       E79B9C       F3B19CA0
9F59     9F59 9F59 9F59 9F59 5CC8 DDBA DDBA DDBA DDBA 5D3A 5A7A DDBA 6B39      E6ACB9       E6ACB9       F3B19CA1
9F5A     9F5A 9F5A 9F5A 9F5A 5CCA DDBB DDBB DDBB DDBB 5D3B 5A5E DDBB 98EE      E9A3AE       E9A3AE       F3B19CA2
9F5B     9F5B 9F5B 9F5B 9F5B 5CCB DDBC DDBC DDBC DDBC 5D3C 5A4C DDBC 6B47      E6AD87       E6AD87       F3B19CA3
9F5C     9F5C 9F5C 9F5C 9F5C 5CCC DDBD DDBD DDBD DDBD 5D3D 5A7E DDBD 6B43      E6AD83       E6AD83       F3B19CA4
9F5D     9F5D 9F5D 9F5D 9F5D 5CCD DDBE DDBE DDBE DDBE 5D3E 5A6E DDBE 6B49      E6AD89       E6AD89       F3B19CA5
9F5E     9F5E 9F5E 9F5E 9F5E 5CCE DDBF DDBF DDBF DDBF 5D3F 5A6F DDBF 6B50      E6AD90       E6AD90       F3B19CA6
9F5F     9F5F 9F5F 9F5F 9F5F 5CCF DDC0 DDC0 DDC0 DDC0 5D40 5A7C DDC0 6B59      E6AD99       E6AD99       F3B19CA7
9F60     9F60 9F60 9F60 9F60 5CD0 DDC1 DDC1 DDC1 DDC1 5D41 5AC1 DDC1 6B54      E6AD94       E6AD94       F3B19CA8
9F61     9F61 9F61 9F61 9F61 5CD1 DDC2 DDC2 DDC2 DDC2 5D42 5AC2 DDC2 6B5B      E6AD9B       E6AD9B       F3B19CA9
9F62     9F62 9F62 9F62 9F62 5CD2 DDC3 DDC3 DDC3 DDC3 5D43 5AC3 DDC3 6B5F      E6AD9F       E6AD9F       F3B19CAA
9F63     9F63 9F63 9F63 9F63 5CD3 DDC4 DDC4 DDC4 DDC4 5D44 5AC4 DDC4 6B61      E6ADA1       E6ADA1       F3B19CAB
9F64     9F64 9F64 9F64 9F64 5CD4 DDC5 DDC5 DDC5 DDC5 5D45 5AC5 DDC5 6B78      E6ADB8       E6ADB8       F3B19CAC
9F65     9F65 9F65 9F65 9F65 5CD5 DDC6 DDC6 DDC6 DDC6 5D46 5AC6 DDC6 6B79      E6ADB9       E6ADB9       F3B19CAD
9F66     9F66 9F66 9F66 9F66 5CD6 DDC7 DDC7 DDC7 DDC7 5D47 5AC7 DDC7 6B7F      E6ADBF       E6ADBF       F3B19CAE
9F67     9F67 9F67 9F67 9F67 5CD7 DDC8 DDC8 DDC8 DDC8 5D48 5AC8 DDC8 6B80      E6AE80       E6AE80       F3B19CAF
9F68     9F68 9F68 9F68 9F68 5CD8 DDC9 DDC9 DDC9 DDC9 5D49 5AC9 DDC9 6B84      E6AE84       E6AE84       F3B19CB0
9F69     9F69 9F69 9F69 9F69 5CD9 DDCA DDCA DDCA DDCA 5D4A 5AD1 DDCA 6B83      E6AE83       E6AE83       F3B19CB1
9F6A     9F6A 9F6A 9F6A 9F6A 5CDA DDCB DDCB DDCB DDCB 5D4B 5AD2 DDCB 6B8D      E6AE8D       E6AE8D       F3B19CB2
9F6B     9F6B 9F6B 9F6B 9F6B 5CDB DDCC DDCC DDCC DDCC 5D4C 5AD3 DDCC 6B98      E6AE98       E6AE98       F3B19CB3
9F6C     9F6C 9F6C 9F6C 9F6C 5CDC DDCD DDCD DDCD DDCD 5D4D 5AD4 DDCD 6B95      E6AE95       E6AE95       F3B19CB4
9F6D     9F6D 9F6D 9F6D 9F6D 5CDD DDCE DDCE DDCE DDCE 5D4E 5AD5 DDCE 6B9E      E6AE9E       E6AE9E       F3B19CB5
9F6E     9F6E 9F6E 9F6E 9F6E 5CDE DDCF DDCF DDCF DDCF 5D4F 5AD6 DDCF 6BA4      E6AEA4       E6AEA4       F3B19CB6
9F6F     9F6F 9F6F 9F6F 9F6F 5CDF DDD0 DDD0 DDD0 DDD0 5D50 5AD7 DDD0 6BAA      E6AEAA       E6AEAA       F3B19CB7
9F70     9F70 9F70 9F70 9F70 5CE0 DDD1 DDD1 DDD1 DDD1 5D51 5AD8 DDD1 6BAB      E6AEAB       E6AEAB       F3B19CB8
9F71     9F71 9F71 9F71 9F71 5CE1 DDD2 DDD2 DDD2 DDD2 5D52 5AD9 DDD2 6BAF      E6AEAF       E6AEAF       F3B19CB9
9F72     9F72 9F72 9F72 9F72 5CE3 DDD3 DDD3 DDD3 DDD3 5D53 5AE2 DDD3 6BB2      E6AEB2       E6AEB2       F3B19CBA
9F73     9F73 9F73 9F73 9F73 5CE2 DDD4 DDD4 DDD4 DDD4 5D54 5AE3 DDD4 6BB1      E6AEB1       E6AEB1       F3B19CBB
9F74     9F74 9F74 9F74 9F74 5CE4 DDD5 DDD5 DDD5 DDD5 5D55 5AE4 DDD5 6BB3      E6AEB3       E6AEB3       F3B19CBC
9F75     9F75 9F75 9F75 9F75 5CE5 DDD6 DDD6 DDD6 DDD6 5D56 5AE5 DDD6 6BB7      E6AEB7       E6AEB7       F3B19CBD
9F76     9F76 9F76 9F76 9F76 5CE6 DDD7 DDD7 DDD7 DDD7 5D57 5AE6 DDD7 6BBC      E6AEBC       E6AEBC       F3B19CBE
9F77     9F77 9F77 9F77 9F77 5CE7 DDD8 DDD8 DDD8 DDD8 5D58 5AE7 DDD8 6BC6      E6AF86       E6AF86       F3B19CBF
9F78     9F78 9F78 9F78 9F78 5CE8 DDD9 DDD9 DDD9 DDD9 5D59 5AE8 DDD9 6BCB      E6AF8B       E6AF8B       F3B19D80
9F79     9F79 9F79 9F79 9F79 5CE9 DDDA DDDA DDDA DDDA 5D5A 5AE9 DDDA 6BD3      E6AF93       E6AF93       F3B19D81
9F7A     9F7A 9F7A 9F7A 9F7A 5CEB DDDB DDDB DDDB DDDB 5D5B 5A4A DDDB 6BDF      E6AF9F       E6AF9F       F3B19D82
9F7B     9F7B 9F7B 9F7B 9F7B 5CEC DDDC DDDC DDDC DDDC 5D5C 5A5B DDDC 6BEC      E6AFAC       E6AFAC       F3B19D83
9F7C     9F7C 9F7C 9F7C 9F7C 5CED DDDD DDDD DDDD DDDD 5D5D 5A5A DDDD 6BEB      E6AFAB       E6AFAB       F3B19D84
9F7D     9F7D 9F7D 9F7D 9F7D 5CEE DDDE DDDE DDDE DDDE 5D5E 5A5F DDDE 6BF3      E6AFB3       E6AFB3       F3B19D85
9F7E     9F7E 9F7E 9F7E 9F7E 5CEF DDDF DDDF DDDF DDDF 5D5F 5A6D DDDF 6BEF      E6AFAF       E6AFAF       F3B19D86
9F80     9F80 9F80 9F80 9F80 6851 DDE0 DDE0 DDE0 DDE0 5D60 5A79 DDE0 9EBE      E9BABE       E9BABE       F3B19D87
9F81     9F81 9F81 9F81 9F81 5CF0 DDE1 DDE1 DDE1 DDE1 5D61 5A57 DDE1 6C08      E6B088       E6B088       F3B19D88
9F82     9F82 9F82 9F82 9F82 5CF1 DDE2 DDE2 DDE2 DDE2 5D62 5A59 DDE2 6C13      E6B093       E6B093       F3B19D89
9F83     9F83 9F83 9F83 9F83 5CF2 DDE3 DDE3 DDE3 DDE3 5D63 5A62 DDE3 6C14      E6B094       E6B094       F3B19D8A
9F84     9F84 9F84 9F84 9F84 5CF3 DDE4 DDE4 DDE4 DDE4 5D64 5A63 DDE4 6C1B      E6B09B       E6B09B       F3B19D8B
9F85     9F85 9F85 9F85 9F85 5CF4 DDE5 DDE5 DDE5 DDE5 5D65 5A64 DDE5 6C24      E6B0A4       E6B0A4       F3B19D8C
9F86     9F86 9F86 9F86 9F86 5CF5 DDE6 DDE6 DDE6 DDE6 5D66 5A65 DDE6 6C23      E6B0A3       E6B0A3       F3B19D8D
9F87     9F87 9F87 9F87 9F87 5CF7 DDE7 DDE7 DDE7 DDE7 5D67 5A66 DDE7 6C5E      E6B19E       E6B19E       F3B19D8E
9F88     9F88 9F88 9F88 9F88 5CF8 DDE8 DDE8 DDE8 DDE8 5D68 5A67 DDE8 6C55      E6B195       E6B195       F3B19D8F
9F89     9F89 9F89 9F89 9F89 5CFA DDE9 DDE9 DDE9 DDE9 5D69 5A68 DDE9 6C62      E6B1A2       E6B1A2       F3B19D90
9F8A     9F8A 9F8A 9F8A 9F8A 52B4 DDEA DDEA DDEA DDEA 5D6A 5A69 DDEA 6C6A      E6B1AA       E6B1AA       F3B19D91
9F8B     9F8B 9F8B 9F8B 9F8B 5CFB DDEB DDEB DDEB DDEB 5D6B 5A70 DDEB 6C82      E6B282       E6B282       F3B19D92
9F8C     9F8C 9F8C 9F8C 9F8C 5CFC DDEC DDEC DDEC DDEC 5D6C 5A71 DDEC 6C8D      E6B28D       E6B28D       F3B19D93
9F8D     9F8D 9F8D 9F8D 9F8D 5D41 DDED DDED DDED DDED 5D6D 5A72 DDED 6C9A      E6B29A       E6B29A       F3B19D94
9F8E     9F8E 9F8E 9F8E 9F8E 5D42 DDEE DDEE DDEE DDEE 5D6E 5A73 DDEE 6C81      E6B281       E6B281       F3B19D95
9F8F     9F8F 9F8F 9F8F 9F8F 5D43 DDEF DDEF DDEF DDEF 5D6F 5A74 DDEF 6C9B      E6B29B       E6B29B       F3B19D96
9F90     9F90 9F90 9F90 9F90 5D44 DDF0 DDF0 DDF0 DDF0 5D70 5A75 DDF0 6C7E      E6B1BE       E6B1BE       F3B19D97
9F91     9F91 9F91 9F91 9F91 5D45 DDF1 DDF1 DDF1 DDF1 5D71 5A76 DDF1 6C68      E6B1A8       E6B1A8       F3B19D98
9F92     9F92 9F92 9F92 9F92 5D46 DDF2 DDF2 DDF2 DDF2 5D72 5A77 DDF2 6C73      E6B1B3       E6B1B3       F3B19D99
9F93     9F93 9F93 9F93 9F93 5D47 DDF3 DDF3 DDF3 DDF3 5D73 5A78 DDF3 6C92      E6B292       E6B292       F3B19D9A
9F94     9F94 9F94 9F94 9F94 5D48 DDF4 DDF4 DDF4 DDF4 5D74 5A80 DDF4 6C90      E6B290       E6B290       F3B19D9B
9F95     9F95 9F95 9F95 9F95 5D49 DDF5 DDF5 DDF5 DDF5 5D75 5A8B DDF5 6CC4      E6B384       E6B384       F3B19D9C
9F96     9F96 9F96 9F96 9F96 5D4A DDF6 DDF6 DDF6 DDF6 5D76 5A9B DDF6 6CF1      E6B3B1       E6B3B1       F3B19D9D
9F97     9F97 9F97 9F97 9F97 54D7 DDF7 DDF7 DDF7 DDF7 5D77 5A9C DDF7 6CD3      E6B393       E6B393       F3B19D9E
9F98     9F98 9F98 9F98 9F98 5D4B DDF8 DDF8 DDF8 DDF8 5D78 5AA0 DDF8 6CBD      E6B2BD       E6B2BD       F3B19D9F
9F99     9F99 9F99 9F99 9F99 55C6 DDF9 DDF9 DDF9 DDF9 5D79 5AAB DDF9 6CD7      E6B397       E6B397       F3B19DA0
9F9A     9F9A 9F9A 9F9A 9F9A 5D4C DDFA DDFA DDFA DDFA 5D7A 5AB0 DDFA 6CC5      E6B385       E6B385       F3B19DA1
9F9B     9F9B 9F9B 9F9B 9F9B 5D4F DDFB DDFB DDFB DDFB 5D7B 5AC0 DDFB 6CDD      E6B39D       E6B39D       F3B19DA2
9F9C     9F9C 9F9C 9F9C 9F9C 5D4D DDFC DDFC DDFC DDFC 5D7C 5A6A DDFC 6CAE      E6B2AE       E6B2AE       F3B19DA3
9F9D     9F9D 9F9D 9F9D 9F9D 5D50 DDFD DDFD DDFD DDFD 5D7D 5AD0 DDFD 6CB1      E6B2B1       E6B2B1       F3B19DA4
9F9E     9F9E 9F9E 9F9E 9F9E 5D51 DDFE DDFE DDFE DDFE 5D7E 5AA1 DDFE 6CBE      E6B2BE       E6B2BE       F3B19DA5
9F9F     9F9F 9F9F 9F9F 9F9F 5D52 DEA1 DEA1 DEA1 DEA1 5E21 5F4F DEA1 6CBA      E6B2BA       E6B2BA       F3B19DA6
9FA0     9FA0 9FA0 9FA0 9FA0 5D53 DEA2 DEA2 DEA2 DEA2 5E22 5F7F DEA2 6CDB      E6B39B       E6B39B       F3B19DA7
9FA1     9FA1 9FA1 9FA1 9FA1 5D54 DEA3 DEA3 DEA3 DEA3 5E23 5F7B DEA3 6CEF      E6B3AF       E6B3AF       F3B19DA8
9FA2     9FA2 9FA2 9FA2 9FA2 5D55 DEA4 DEA4 DEA4 DEA4 5E24 5FE0 DEA4 6CD9      E6B399       E6B399       F3B19DA9
9FA3     9FA3 9FA3 9FA3 9FA3 5D56 DEA5 DEA5 DEA5 DEA5 5E25 5F6C DEA5 6CEA      E6B3AA       E6B3AA       F3B19DAA
9FA4     9FA4 9FA4 9FA4 9FA4 5D57 DEA6 DEA6 DEA6 DEA6 5E26 5F50 DEA6 6D1F      E6B49F       E6B49F       F3B19DAB
9FA5     9FA5 9FA5 9FA5 9FA5 6370 DEA7 DEA7 DEA7 DEA7 5E27 5F7D DEA7 884D      E8A18D       E8A18D       F3B19DAC
9FA6     9FA6 9FA6 9FA6 9FA6 5D59 DEA8 DEA8 DEA8 DEA8 5E28 5F4D DEA8 6D36      E6B4B6       E6B4B6       F3B19DAD
9FA7     9FA7 9FA7 9FA7 9FA7 5D5A DEA9 DEA9 DEA9 DEA9 5E29 5F5D DEA9 6D2B      E6B4AB       E6B4AB       F3B19DAE
9FA8     9FA8 9FA8 9FA8 9FA8 5D5B DEAA DEAA DEAA DEAA 5E2A 5F5C DEAA 6D3D      E6B4BD       E6B4BD       F3B19DAF
9FA9     9FA9 9FA9 9FA9 9FA9 50BC DEAB DEAB DEAB DEAB 5E2B 5F4E DEAB 6D38      E6B4B8       E6B4B8       F3B19DB0
9FAA     9FAA 9FAA 9FAA 9FAA 52B6 DEAC DEAC DEAC DEAC 5E2C 5F6B DEAC 6D19      E6B499       E6B499       F3B19DB1
9FAB     9FAB 9FAB 9FAB 9FAB 52AC DEAD DEAD DEAD DEAD 5E2D 5F60 DEAD 6D35      E6B4B5       E6B4B5       F3B19DB2
9FAC     9FAC 9FAC 9FAC 9FAC 5D5C DEAE DEAE DEAE DEAE 5E2E 5F4B DEAE 6D33      E6B4B3       E6B4B3       F3B19DB3
9FAD     9FAD 9FAD 9FAD 9FAD 5D5D DEAF DEAF DEAF DEAF 5E2F 5F61 DEAF 6D12      E6B492       E6B492       F3B19DB4
9FAE     9FAE 9FAE 9FAE 9FAE 5D5E DEB0 DEB0 DEB0 DEB0 5E30 5FF0 DEB0 6D0C      E6B48C       E6B48C       F3B19DB5
9FAF     9FAF 9FAF 9FAF 9FAF 5D5F DEB1 DEB1 DEB1 DEB1 5E31 5FF1 DEB1 6D63      E6B5A3       E6B5A3       F3B19DB6
9FB0     9FB0 9FB0 9FB0 9FB0 5D61 DEB2 DEB2 DEB2 DEB2 5E32 5FF2 DEB2 6D93      E6B693       E6B693       F3B19DB7
9FB1     9FB1 9FB1 9FB1 9FB1 5D63 DEB3 DEB3 DEB3 DEB3 5E33 5FF3 DEB3 6D64      E6B5A4       E6B5A4       F3B19DB8
9FB2     9FB2 9FB2 9FB2 9FB2 5D64 DEB4 DEB4 DEB4 DEB4 5E34 5FF4 DEB4 6D5A      E6B59A       E6B59A       F3B19DB9
9FB3     9FB3 9FB3 9FB3 9FB3 5D65 DEB5 DEB5 DEB5 DEB5 5E35 5FF5 DEB5 6D79      E6B5B9       E6B5B9       F3B19DBA
9FB4     9FB4 9FB4 9FB4 9FB4 5D66 DEB6 DEB6 DEB6 DEB6 5E36 5FF6 DEB6 6D59      E6B599       E6B599       F3B19DBB
9FB5     9FB5 9FB5 9FB5 9FB5 5D67 DEB7 DEB7 DEB7 DEB7 5E37 5FF7 DEB7 6D8E      E6B68E       E6B68E       F3B19DBC
9FB6     9FB6 9FB6 9FB6 9FB6 5D68 DEB8 DEB8 DEB8 DEB8 5E38 5FF8 DEB8 6D95      E6B695       E6B695       F3B19DBD
9FB7     9FB7 9FB7 9FB7 9FB7 51D2 C5F3 DEB9 DEB9 C5F3 4573 C578 DEB9 6FE4      E6BFA4       E6BFA4       F3B19DBE
9FB8     9FB8 9FB8 9FB8 9FB8 5D6A DEBA DEBA DEBA DEBA 5E3A 5F7A DEBA 6D85      E6B685       E6B685       F3B19DBF
9FB9     9FB9 9FB9 9FB9 9FB9 5D6C DEBB DEBB DEBB DEBB 5E3B 5F5E DEBB 6DF9      E6B7B9       E6B7B9       F3B19E80
9FBA     9FBA 9FBA 9FBA 9FBA 5164 DEBC DEBC DEBC DEBC 5E3C 5F4C DEBC 6E15      E6B895       E6B895       F3B19E81
9FBB     9FBB 9FBB 9FBB 9FBB 5D6D DEBD DEBD DEBD DEBD 5E3D 5F7E DEBD 6E0A      E6B88A       E6B88A       F3B19E82
9FBC     9FBC 9FBC 9FBC 9FBC 5D6F DEBE DEBE DEBE DEBE 5E3E 5F6E DEBE 6DB5      E6B6B5       E6B6B5       F3B19E83
9FBD     9FBD 9FBD 9FBD 9FBD 5D71 DEBF DEBF DEBF DEBF 5E3F 5F6F DEBF 6DC7      E6B787       E6B787       F3B19E84
9FBE     9FBE 9FBE 9FBE 9FBE 5D70 DEC0 DEC0 DEC0 DEC0 5E40 5F7C DEC0 6DE6      E6B7A6       E6B7A6       F3B19E85
9FBF     9FBF 9FBF 9FBF 9FBF 5D73 DEC1 DEC1 DEC1 DEC1 5E41 5FC1 DEC1 6DB8      E6B6B8       E6B6B8       F3B19E86
9FC0     9FC0 9FC0 9FC0 9FC0 5D75 DEC2 DEC2 DEC2 DEC2 5E42 5FC2 DEC2 6DC6      E6B786       E6B786       F3B19E87
9FC1     9FC1 9FC1 9FC1 9FC1 5D76 DEC3 DEC3 DEC3 DEC3 5E43 5FC3 DEC3 6DEC      E6B7AC       E6B7AC       F3B19E88
9FC2     9FC2 9FC2 9FC2 9FC2 5D77 DEC4 DEC4 DEC4 DEC4 5E44 5FC4 DEC4 6DDE      E6B79E       E6B79E       F3B19E89
9FC3     9FC3 9FC3 9FC3 9FC3 5D78 DEC5 DEC5 DEC5 DEC5 5E45 5FC5 DEC5 6DCC      E6B78C       E6B78C       F3B19E8A
9FC4     9FC4 9FC4 9FC4 9FC4 5D79 DEC6 DEC6 DEC6 DEC6 5E46 5FC6 DEC6 6DE8      E6B7A8       E6B7A8       F3B19E8B
9FC5     9FC5 9FC5 9FC5 9FC5 5D7B DEC7 DEC7 DEC7 DEC7 5E47 5FC7 DEC7 6DD2      E6B792       E6B792       F3B19E8C
9FC6     9FC6 9FC6 9FC6 9FC6 5D7C DEC8 DEC8 DEC8 DEC8 5E48 5FC8 DEC8 6DC5      E6B785       E6B785       F3B19E8D
9FC7     9FC7 9FC7 9FC7 9FC7 5D7D DEC9 DEC9 DEC9 DEC9 5E49 5FC9 DEC9 6DFA      E6B7BA       E6B7BA       F3B19E8E
9FC8     9FC8 9FC8 9FC8 9FC8 5D7E DECA DECA DECA DECA 5E4A 5FD1 DECA 6DD9      E6B799       E6B799       F3B19E8F
9FC9     9FC9 9FC9 9FC9 9FC9 5D81 DECB DECB DECB DECB 5E4B 5FD2 DECB 6DE4      E6B7A4       E6B7A4       F3B19E90
9FCA     9FCA 9FCA 9FCA 9FCA 5D82 DECC DECC DECC DECC 5E4C 5FD3 DECC 6DD5      E6B795       E6B795       F3B19E91
9FCB     9FCB 9FCB 9FCB 9FCB 5D83 DECD DECD DECD DECD 5E4D 5FD4 DECD 6DEA      E6B7AA       E6B7AA       F3B19E92
9FCC     9FCC 9FCC 9FCC 9FCC 5D84 DECE DECE DECE DECE 5E4E 5FD5 DECE 6DEE      E6B7AE       E6B7AE       F3B19E93
9FCD     9FCD 9FCD 9FCD 9FCD 5D85 DECF DECF DECF DECF 5E4F 5FD6 DECF 6E2D      E6B8AD       E6B8AD       F3B19E94
9FCE     9FCE 9FCE 9FCE 9FCE 5D86 DED0 DED0 DED0 DED0 5E50 5FD7 DED0 6E6E      E6B9AE       E6B9AE       F3B19E95
9FCF     9FCF 9FCF 9FCF 9FCF 5D6E DED1 DED1 DED1 DED1 5E51 5FD8 DED1 6E2E      E6B8AE       E6B8AE       F3B19E96
9FD0     9FD0 9FD0 9FD0 9FD0 5D87 DED2 DED2 DED2 DED2 5E52 5FD9 DED2 6E19      E6B899       E6B899       F3B19E97
9FD1     9FD1 9FD1 9FD1 9FD1 5D88 DED3 DED3 DED3 DED3 5E53 5FE2 DED3 6E72      E6B9B2       E6B9B2       F3B19E98
9FD2     9FD2 9FD2 9FD2 9FD2 5D89 DED4 DED4 DED4 DED4 5E54 5FE3 DED4 6E5F      E6B99F       E6B99F       F3B19E99
9FD3     9FD3 9FD3 9FD3 9FD3 5D8B DED5 DED5 DED5 DED5 5E55 5FE4 DED5 6E3E      E6B8BE       E6B8BE       F3B19E9A
9FD4     9FD4 9FD4 9FD4 9FD4 5D8C DED6 DED6 DED6 DED6 5E56 5FE5 DED6 6E23      E6B8A3       E6B8A3       F3B19E9B
9FD5     9FD5 9FD5 9FD5 9FD5 5D8D DED7 DED7 DED7 DED7 5E57 5FE6 DED7 6E6B      E6B9AB       E6B9AB       F3B19E9C
9FD6     9FD6 9FD6 9FD6 9FD6 5D8F DED8 DED8 DED8 DED8 5E58 5FE7 DED8 6E2B      E6B8AB       E6B8AB       F3B19E9D
9FD7     9FD7 9FD7 9FD7 9FD7 5D90 DED9 DED9 DED9 DED9 5E59 5FE8 DED9 6E76      E6B9B6       E6B9B6       F3B19E9E
9FD8     9FD8 9FD8 9FD8 9FD8 5D91 DEDA DEDA DEDA DEDA 5E5A 5FE9 DEDA 6E4D      E6B98D       E6B98D       F3B19E9F
9FD9     9FD9 9FD9 9FD9 9FD9 5D92 DEDB DEDB DEDB DEDB 5E5B 5F4A DEDB 6E1F      E6B89F       E6B89F       F3B19EA0
9FDA     9FDA 9FDA 9FDA 9FDA 5D94 DEDC DEDC DEDC DEDC 5E5C 5F5B DEDC 6E43      E6B983       E6B983       F3B19EA1
9FDB     9FDB 9FDB 9FDB 9FDB 5D96 DEDD DEDD DEDD DEDD 5E5D 5F5A DEDD 6E3A      E6B8BA       E6B8BA       F3B19EA2
9FDC     9FDC 9FDC 9FDC 9FDC 5D97 DEDE DEDE DEDE DEDE 5E5E 5F5F DEDE 6E4E      E6B98E       E6B98E       F3B19EA3
9FDD     9FDD 9FDD 9FDD 9FDD 5D98 DEDF DEDF DEDF DEDF 5E5F 5F6D DEDF 6E24      E6B8A4       E6B8A4       F3B19EA4
9FDE     9FDE 9FDE 9FDE 9FDE 5DB4 DEE0 DEE0 DEE0 DEE0 5E60 5F79 DEE0 6EFF      E6BBBF       E6BBBF       F3B19EA5
9FDF     9FDF 9FDF 9FDF 9FDF 5D99 DEE1 DEE1 DEE1 DEE1 5E61 5F57 DEE1 6E1D      E6B89D       E6B89D       F3B19EA6
9FE0     9FE0 9FE0 9FE0 9FE0 5D9A DEE2 DEE2 DEE2 DEE2 5E62 5F59 DEE2 6E38      E6B8B8       E6B8B8       F3B19EA7
9FE1     9FE1 9FE1 9FE1 9FE1 5D9B DEE3 DEE3 DEE3 DEE3 5E63 5F62 DEE3 6E82      E6BA82       E6BA82       F3B19EA8
9FE2     9FE2 9FE2 9FE2 9FE2 5D9C DEE4 DEE4 DEE4 DEE4 5E64 5F63 DEE4 6EAA      E6BAAA       E6BAAA       F3B19EA9
9FE3     9FE3 9FE3 9FE3 9FE3 5D9D DEE5 DEE5 DEE5 DEE5 5E65 5F64 DEE5 6E98      E6BA98       E6BA98       F3B19EAA
9FE4     9FE4 9FE4 9FE4 9FE4 52A9 DEE6 DEE6 DEE6 DEE6 5E66 5F65 DEE6 6EC9      E6BB89       E6BB89       F3B19EAB
9FE5     9FE5 9FE5 9FE5 9FE5 5D9E DEE7 DEE7 DEE7 DEE7 5E67 5F66 DEE7 6EB7      E6BAB7       E6BAB7       F3B19EAC
9FE6     9FE6 9FE6 9FE6 9FE6 524A DEE8 DEE8 DEE8 DEE8 5E68 5F67 DEE8 6ED3      E6BB93       E6BB93       F3B19EAD
9FE7     9FE7 9FE7 9FE7 9FE7 5D9F DEE9 DEE9 DEE9 DEE9 5E69 5F68 DEE9 6EBD      E6BABD       E6BABD       F3B19EAE
9FE8     9FE8 9FE8 9FE8 9FE8 5DA0 DEEA DEEA DEEA DEEA 5E6A 5F69 DEEA 6EAF      E6BAAF       E6BAAF       F3B19EAF
9FE9     9FE9 9FE9 9FE9 9FE9 5DA1 DEEB DEEB DEEB DEEB 5E6B 5F70 DEEB 6EC4      E6BB84       E6BB84       F3B19EB0
9FEA     9FEA 9FEA 9FEA 9FEA 5DA2 DEEC DEEC DEEC DEEC 5E6C 5F71 DEEC 6EB2      E6BAB2       E6BAB2       F3B19EB1
9FEB     9FEB 9FEB 9FEB 9FEB 5DA3 DEED DEED DEED DEED 5E6D 5F72 DEED 6ED4      E6BB94       E6BB94       F3B19EB2
9FEC     9FEC 9FEC 9FEC 9FEC 5DA4 DEEE DEEE DEEE DEEE 5E6E 5F73 DEEE 6ED5      E6BB95       E6BB95       F3B19EB3
9FED     9FED 9FED 9FED 9FED 5DA5 DEEF DEEF DEEF DEEF 5E6F 5F74 DEEF 6E8F      E6BA8F       E6BA8F       F3B19EB4
9FEE     9FEE 9FEE 9FEE 9FEE 5376 DEF0 DEF0 DEF0 DEF0 5E70 5F75 DEF0 6EA5      E6BAA5       E6BAA5       F3B19EB5
9FEF     9FEF 9FEF 9FEF 9FEF 5DA7 DEF1 DEF1 DEF1 DEF1 5E71 5F76 DEF1 6EC2      E6BB82       E6BB82       F3B19EB6
9FF0     9FF0 9FF0 9FF0 9FF0 5DA8 DEF2 DEF2 DEF2 DEF2 5E72 5F77 DEF2 6E9F      E6BA9F       E6BA9F       F3B19EB7
9FF1     9FF1 9FF1 9FF1 9FF1 5DA9 DEF3 DEF3 DEF3 DEF3 5E73 5F78 DEF3 6F41      E6BD81       E6BD81       F3B19EB8
9FF2     9FF2 9FF2 9FF2 9FF2 5575 DEF4 DEF4 DEF4 DEF4 5E74 5F80 DEF4 6F11      E6BC91       E6BC91       F3B19EB9
9FF3     9FF3 9FF3 9FF3 9FF3 53EA B4C3 DEF5 DEF5 B4C3 3443 F4C3 DEF5 704C      E7818C       E7818C       F3B19EBA
9FF4     9FF4 9FF4 9FF4 9FF4 5DAB DEF6 DEF6 DEF6 DEF6 5E76 5F9B DEF6 6EEC      E6BBAC       E6BBAC       F3B19EBB
9FF5     9FF5 9FF5 9FF5 9FF5 5DAC DEF7 DEF7 DEF7 DEF7 5E77 5F9C DEF7 6EF8      E6BBB8       E6BBB8       F3B19EBC
9FF6     9FF6 9FF6 9FF6 9FF6 5DAD DEF8 DEF8 DEF8 DEF8 5E78 5FA0 DEF8 6EFE      E6BBBE       E6BBBE       F3B19EBD
9FF7     9FF7 9FF7 9FF7 9FF7 5DAE DEF9 DEF9 DEF9 DEF9 5E79 5FAB DEF9 6F3F      E6BCBF       E6BCBF       F3B19EBE
9FF8     9FF8 9FF8 9FF8 9FF8 5DAF DEFA DEFA DEFA DEFA 5E7A 5FB0 DEFA 6EF2      E6BBB2       E6BBB2       F3B19EBF
9FF9     9FF9 9FF9 9FF9 9FF9 5DB0 DEFB DEFB DEFB DEFB 5E7B 5FC0 DEFB 6F31      E6BCB1       E6BCB1       F3B19F80
9FFA     9FFA 9FFA 9FFA 9FFA 5DB1 DEFC DEFC DEFC DEFC 5E7C 5F6A DEFC 6EEF      E6BBAF       E6BBAF       F3B19F81
9FFB     9FFB 9FFB 9FFB 9FFB 5DB2 DEFD DEFD DEFD DEFD 5E7D 5FD0 DEFD 6F32      E6BCB2       E6BCB2       F3B19F82
9FFC     9FFC 9FFC 9FFC 9FFC 5DB3 DEFE DEFE DEFE DEFE 5E7E 5FA1 DEFE 6ECC      E6BB8C       E6BB8C       F3B19F83
E040     E040 E040 E040 E040 5DB5 DFA1 DFA1 DFA1 DFA1 5F21 6D4F DFA1 6F3E      E6BCBE       E6BCBE       F3B19F84
E041     E041 E041 E041 E041 5DB6 DFA2 DFA2 DFA2 DFA2 5F22 6D7F DFA2 6F13      E6BC93       E6BC93       F3B19F85
E042     E042 E042 E042 E042 5DB7 DFA3 DFA3 DFA3 DFA3 5F23 6D7B DFA3 6EF7      E6BBB7       E6BBB7       F3B19F86
E043     E043 E043 E043 E043 5DB8 DFA4 DFA4 DFA4 DFA4 5F24 6DE0 DFA4 6F86      E6BE86       E6BE86       F3B19F87
E044     E044 E044 E044 E044 5DB9 DFA5 DFA5 DFA5 DFA5 5F25 6D6C DFA5 6F7A      E6BDBA       E6BDBA       F3B19F88
E045     E045 E045 E045 E045 5DBA DFA6 DFA6 DFA6 DFA6 5F26 6D50 DFA6 6F78      E6BDB8       E6BDB8       F3B19F89
E046     E046 E046 E046 E046 55A0 DFA7 DFA7 DFA7 DFA7 5F27 6D7D DFA7 6F81      E6BE81       E6BE81       F3B19F8A
E047     E047 E047 E047 E047 5DBB DFA8 DFA8 DFA8 DFA8 5F28 6D4D DFA8 6F80      E6BE80       E6BE80       F3B19F8B
E048     E048 E048 E048 E048 5DBC DFA9 DFA9 DFA9 DFA9 5F29 6D5D DFA9 6F6F      E6BDAF       E6BDAF       F3B19F8C
E049     E049 E049 E049 E049 5DBD DFAA DFAA DFAA DFAA 5F2A 6D5C DFAA 6F5B      E6BD9B       E6BD9B       F3B19F8D
E04A     E04A E04A E04A E04A 5DE2 DFAB DFAB DFAB DFAB 5F2B 6D4E DFAB 6FF3      E6BFB3       E6BFB3       F3B19F8E
E04B     E04B E04B E04B E04B 5DBE DFAC DFAC DFAC DFAC 5F2C 6D6B DFAC 6F6D      E6BDAD       E6BDAD       F3B19F8F
E04C     E04C E04C E04C E04C 5DC0 DFAD DFAD DFAD DFAD 5F2D 6D60 DFAD 6F82      E6BE82       E6BE82       F3B19F90
E04D     E04D E04D E04D E04D 5DC2 DFAE DFAE DFAE DFAE 5F2E 6D4B DFAE 6F7C      E6BDBC       E6BDBC       F3B19F91
E04E     E04E E04E E04E E04E 5DC3 DFAF DFAF DFAF DFAF 5F2F 6D61 DFAF 6F58      E6BD98       E6BD98       F3B19F92
E04F     E04F E04F E04F E04F 5DC5 DFB0 DFB0 DFB0 DFB0 5F30 6DF0 DFB0 6F8E      E6BE8E       E6BE8E       F3B19F93
E050     E050 E050 E050 E050 5DC6 DFB1 DFB1 DFB1 DFB1 5F31 6DF1 DFB1 6F91      E6BE91       E6BE91       F3B19F94
E051     E051 E051 E051 E051 5DCF DFB2 DFB2 DFB2 DFB2 5F32 6DF2 DFB2 6FC2      E6BF82       E6BF82       F3B19F95
E052     E052 E052 E052 E052 5DC7 DFB3 DFB3 DFB3 DFB3 5F33 6DF3 DFB3 6F66      E6BDA6       E6BDA6       F3B19F96
E053     E053 E053 E053 E053 5DC8 DFB4 DFB4 DFB4 DFB4 5F34 6DF4 DFB4 6FB3      E6BEB3       E6BEB3       F3B19F97
E054     E054 E054 E054 E054 5DC9 DFB5 DFB5 DFB5 DFB5 5F35 6DF5 DFB5 6FA3      E6BEA3       E6BEA3       F3B19F98
E055     E055 E055 E055 E055 5DCB DFB6 DFB6 DFB6 DFB6 5F36 6DF6 DFB6 6FA1      E6BEA1       E6BEA1       F3B19F99
E056     E056 E056 E056 E056 4E4B DFB7 DFB7 DFB7 DFB7 5F37 6DF7 DFB7 6FA4      E6BEA4       E6BEA4       F3B19F9A
E057     E057 E057 E057 E057 5DCC DFB8 DFB8 DFB8 DFB8 5F38 6DF8 DFB8 6FB9      E6BEB9       E6BEB9       F3B19F9B
E058     E058 E058 E058 E058 5DC4 DFB9 DFB9 DFB9 DFB9 5F39 6DF9 DFB9 6FC6      E6BF86       E6BF86       F3B19F9C
E059     E059 E059 E059 E059 5DCE DFBA DFBA DFBA DFBA 5F3A 6D7A DFBA 6FAA      E6BEAA       E6BEAA       F3B19F9D
E05A     E05A E05A E05A E05A 5DD0 DFBB DFBB DFBB DFBB 5F3B 6D5E DFBB 6FDF      E6BF9F       E6BF9F       F3B19F9E
E05B     E05B E05B E05B E05B 5DD1 DFBC DFBC DFBC DFBC 5F3C 6D4C DFBC 6FD5      E6BF95       E6BF95       F3B19F9F
E05C     E05C E05C E05C E05C 5DD2 DFBD DFBD DFBD DFBD 5F3D 6D7E DFBD 6FEC      E6BFAC       E6BFAC       F3B19FA0
E05D     E05D E05D E05D E05D 5DD4 DFBE DFBE DFBE DFBE 5F3E 6D6E DFBE 6FD4      E6BF94       E6BF94       F3B19FA1
E05E     E05E E05E E05E E05E 5DD3 DFBF DFBF DFBF DFBF 5F3F 6D6F DFBF 6FD8      E6BF98       E6BF98       F3B19FA2
E05F     E05F E05F E05F E05F 5055 DFC0 DFC0 DFC0 DFC0 5F40 6D7C DFC0 6FF1      E6BFB1       E6BFB1       F3B19FA3
E060     E060 E060 E060 E060 5DD6 DFC1 DFC1 DFC1 DFC1 5F41 6DC1 DFC1 6FEE      E6BFAE       E6BFAE       F3B19FA4
E061     E061 E061 E061 E061 5DCD DFC2 DFC2 DFC2 DFC2 5F42 6DC2 DFC2 6FDB      E6BF9B       E6BF9B       F3B19FA5
E062     E062 E062 E062 E062 5DD9 DFC3 DFC3 DFC3 DFC3 5F43 6DC3 DFC3 7009      E78089       E78089       F3B19FA6
E063     E063 E063 E063 E063 5DDA DFC4 DFC4 DFC4 DFC4 5F44 6DC4 DFC4 700B      E7808B       E7808B       F3B19FA7
E064     E064 E064 E064 E064 5DDB DFC5 DFC5 DFC5 DFC5 5F45 6DC5 DFC5 6FFA      E6BFBA       E6BFBA       F3B19FA8
E065     E065 E065 E065 E065 5DDC DFC6 DFC6 DFC6 DFC6 5F46 6DC6 DFC6 7011      E78091       E78091       F3B19FA9
E066     E066 E066 E066 E066 5DDD DFC7 DFC7 DFC7 DFC7 5F47 6DC7 DFC7 7001      E78081       E78081       F3B19FAA
E067     E067 E067 E067 E067 5DDE DFC8 DFC8 DFC8 DFC8 5F48 6DC8 DFC8 700F      E7808F       E7808F       F3B19FAB
E068     E068 E068 E068 E068 5187 DFC9 DFC9 DFC9 DFC9 5F49 6DC9 DFC9 6FFE      E6BFBE       E6BFBE       F3B19FAC
E069     E069 E069 E069 E069 5DDF DFCA DFCA DFCA DFCA 5F4A 6DD1 DFCA 701B      E7809B       E7809B       F3B19FAD
E06A     E06A E06A E06A E06A 5DE0 DFCB DFCB DFCB DFCB 5F4B 6DD2 DFCB 701A      E7809A       E7809A       F3B19FAE
E06B     E06B E06B E06B E06B 5DBF DFCC DFCC DFCC DFCC 5F4C 6DD3 DFCC 6F74      E6BDB4       E6BDB4       F3B19FAF
E06C     E06C E06C E06C E06C 52B0 DFCD DFCD DFCD DFCD 5F4D 6DD4 DFCD 701D      E7809D       E7809D       F3B19FB0
E06D     E06D E06D E06D E06D 5DE4 DFCE DFCE DFCE DFCE 5F4E 6DD5 DFCE 7018      E78098       E78098       F3B19FB1
E06E     E06E E06E E06E E06E 5DE1 DFCF DFCF DFCF DFCF 5F4F 6DD6 DFCF 701F      E7809F       E7809F       F3B19FB2
E06F     E06F E06F E06F E06F 5DE5 DFD0 DFD0 DFD0 DFD0 5F50 6DD7 DFD0 7030      E780B0       E780B0       F3B19FB3
E070     E070 E070 E070 E070 5DE6 DFD1 DFD1 DFD1 DFD1 5F51 6DD8 DFD1 703E      E780BE       E780BE       F3B19FB4
E071     E071 E071 E071 E071 5DE7 DFD2 DFD2 DFD2 DFD2 5F52 6DD9 DFD2 7032      E780B2       E780B2       F3B19FB5
E072     E072 E072 E072 E072 5DE8 DFD3 DFD3 DFD3 DFD3 5F53 6DE2 DFD3 7051      E78191       E78191       F3B19FB6
E073     E073 E073 E073 E073 5DE9 DFD4 DFD4 DFD4 DFD4 5F54 6DE3 DFD4 7063      E781A3       E781A3       F3B19FB7
E074     E074 E074 E074 E074 5DEB DFD5 DFD5 DFD5 DFD5 5F55 6DE4 DFD5 7099      E78299       E78299       F3B19FB8
E075     E075 E075 E075 E075 4FEE DFD6 DFD6 DFD6 DFD6 5F56 6DE5 DFD6 7092      E78292       E78292       F3B19FB9
E076     E076 E076 E076 E076 5DEC DFD7 DFD7 DFD7 DFD7 5F57 6DE6 DFD7 70AF      E782AF       E782AF       F3B19FBA
E077     E077 E077 E077 E077 5DF7 DFD8 DFD8 DFD8 DFD8 5F58 6DE7 DFD8 70F1      E783B1       E783B1       F3B19FBB
E078     E078 E078 E078 E078 5DEE DFD9 DFD9 DFD9 DFD9 5F59 6DE8 DFD9 70AC      E782AC       E782AC       F3B19FBC
E079     E079 E079 E079 E079 5DEF DFDA DFDA DFDA DFDA 5F5A 6DE9 DFDA 70B8      E782B8       E782B8       F3B19FBD
E07A     E07A E07A E07A E07A 52CD DFDB DFDB DFDB DFDB 5F5B 6D4A DFDB 70B3      E782B3       E782B3       F3B19FBE
E07B     E07B E07B E07B E07B 5DF0 DFDC DFDC DFDC DFDC 5F5C 6D5B DFDC 70AE      E782AE       E782AE       F3B19FBF
E07C     E07C E07C E07C E07C 5DF1 DFDD DFDD DFDD DFDD 5F5D 6D5A DFDD 70DF      E7839F       E7839F       F3B1A080
E07D     E07D E07D E07D E07D 5DF2 DFDE DFDE DFDE DFDE 5F5E 6D5F DFDE 70CB      E7838B       E7838B       F3B1A081
E07E     E07E E07E E07E E07E 538C DFDF DFDF DFDF DFDF 5F5F 6D6D DFDF 70DD      E7839D       E7839D       F3B1A082
E080     E080 E080 E080 E080 5DF3 DFE0 DFE0 DFE0 DFE0 5F60 6D79 DFE0 70D9      E78399       E78399       F3B1A083
E081     E081 E081 E081 E081 5DF4 DFE1 DFE1 DFE1 DFE1 5F61 6D57 DFE1 7109      E78489       E78489       F3B1A084
E082     E082 E082 E082 E082 5DF8 DFE2 DFE2 DFE2 DFE2 5F62 6D59 DFE2 70FD      E783BD       E783BD       F3B1A085
E083     E083 E083 E083 E083 5DF9 DFE3 DFE3 DFE3 DFE3 5F63 6D62 DFE3 711C      E7849C       E7849C       F3B1A086
E084     E084 E084 E084 E084 5DFA DFE4 DFE4 DFE4 DFE4 5F64 6D63 DFE4 7119      E78499       E78499       F3B1A087
E085     E085 E085 E085 E085 5357 DFE5 DFE5 DFE5 DFE5 5F65 6D64 DFE5 7165      E785A5       E785A5       F3B1A088
E086     E086 E086 E086 E086 6885 DFE6 DFE6 DFE6 DFE6 5F66 6D65 DFE6 7155      E78595       E78595       F3B1A089
E087     E087 E087 E087 E087 5E45 DFE7 DFE7 DFE7 DFE7 5F67 6D66 DFE7 7188      E78688       E78688       F3B1A08A
E088     E088 E088 E088 E088 5DFE DFE8 DFE8 DFE8 DFE8 5F68 6D67 DFE8 7166      E785A6       E785A6       F3B1A08B
E089     E089 E089 E089 E089 5E41 DFE9 DFE9 DFE9 DFE9 5F69 6D68 DFE9 7162      E785A2       E785A2       F3B1A08C
E08A     E08A E08A E08A E08A 5E42 DFEA DFEA DFEA DFEA 5F6A 6D69 DFEA 714C      E7858C       E7858C       F3B1A08D
E08B     E08B E08B E08B E08B 5E43 DFEB DFEB DFEB DFEB 5F6B 6D70 DFEB 7156      E78596       E78596       F3B1A08E
E08C     E08C E08C E08C E08C 5E44 DFEC DFEC DFEC DFEC 5F6C 6D71 DFEC 716C      E785AC       E785AC       F3B1A08F
E08D     E08D E08D E08D E08D 5E46 DFED DFED DFED DFED 5F6D 6D72 DFED 718F      E7868F       E7868F       F3B1A090
E08E     E08E E08E E08E E08E 5E57 DFEE DFEE DFEE DFEE 5F6E 6D73 DFEE 71FB      E787BB       E787BB       F3B1A091
E08F     E08F E08F E08F E08F 5E47 DFEF DFEF DFEF DFEF 5F6F 6D74 DFEF 7184      E78684       E78684       F3B1A092
E090     E090 E090 E090 E090 5E48 DFF0 DFF0 DFF0 DFF0 5F70 6D75 DFF0 7195      E78695       E78695       F3B1A093
E091     E091 E091 E091 E091 54BB DFF1 DFF1 DFF1 DFF1 5F71 6D76 DFF1 71A8      E786A8       E786A8       F3B1A094
E092     E092 E092 E092 E092 5E4A DFF2 DFF2 DFF2 DFF2 5F72 6D77 DFF2 71AC      E786AC       E786AC       F3B1A095
E093     E093 E093 E093 E093 5E52 DFF3 DFF3 DFF3 DFF3 5F73 6D78 DFF3 71D7      E78797       E78797       F3B1A096
E094     E094 E094 E094 E094 5E4C DFF4 DFF4 DFF4 DFF4 5F74 6D80 DFF4 71B9      E786B9       E786B9       F3B1A097
E095     E095 E095 E095 E095 5E4D DFF5 DFF5 DFF5 DFF5 5F75 6D8B DFF5 71BE      E786BE       E786BE       F3B1A098
E096     E096 E096 E096 E096 5E4E DFF6 DFF6 DFF6 DFF6 5F76 6D9B DFF6 71D2      E78792       E78792       F3B1A099
E097     E097 E097 E097 E097 5E50 DFF7 DFF7 DFF7 DFF7 5F77 6D9C DFF7 71C9      E78789       E78789       F3B1A09A
E098     E098 E098 E098 E098 5E51 DFF8 DFF8 DFF8 DFF8 5F78 6DA0 DFF8 71D4      E78794       E78794       F3B1A09B
E099     E099 E099 E099 E099 5E53 DFF9 DFF9 DFF9 DFF9 5F79 6DAB DFF9 71CE      E7878E       E7878E       F3B1A09C
E09A     E09A E09A E09A E09A 5E55 DFFA DFFA DFFA DFFA 5F7A 6DB0 DFFA 71E0      E787A0       E787A0       F3B1A09D
E09B     E09B E09B E09B E09B 5E56 DFFB DFFB DFFB DFFB 5F7B 6DC0 DFFB 71EC      E787AC       E787AC       F3B1A09E
E09C     E09C E09C E09C E09C 5E4F DFFC DFFC DFFC DFFC 5F7C 6D6A DFFC 71E7      E787A7       E787A7       F3B1A09F
E09D     E09D E09D E09D E09D 5E54 DFFD DFFD DFFD DFFD 5F7D 6DD0 DFFD 71F5      E787B5       E787B5       F3B1A0A0
E09E     E09E E09E E09E E09E 5E58 DFFE DFFE DFFE DFFE 5F7E 6DA1 DFFE 71FC      E787BC       E787BC       F3B1A0A1
E09F     E09F E09F E09F E09F 5E59 E0A1 E0A1 E0A1 E0A1 6021 794F E0A1 71F9      E787B9       E787B9       F3B1A0A2
E0A0     E0A0 E0A0 E0A0 E0A0 5E5B E0A2 E0A2 E0A2 E0A2 6022 797F E0A2 71FF      E787BF       E787BF       F3B1A0A3
E0A1     E0A1 E0A1 E0A1 E0A1 5E5C E0A3 E0A3 E0A3 E0A3 6023 797B E0A3 720D      E7888D       E7888D       F3B1A0A4
E0A2     E0A2 E0A2 E0A2 E0A2 5E5D E0A4 E0A4 E0A4 E0A4 6024 79E0 E0A4 7210      E78890       E78890       F3B1A0A5
E0A3     E0A3 E0A3 E0A3 E0A3 537C E0A5 E0A5 E0A5 E0A5 6025 796C E0A5 721B      E7889B       E7889B       F3B1A0A6
E0A4     E0A4 E0A4 E0A4 E0A4 5E5E E0A6 E0A6 E0A6 E0A6 6026 7950 E0A6 7228      E788A8       E788A8       F3B1A0A7
E0A5     E0A5 E0A5 E0A5 E0A5 5E5F E0A7 E0A7 E0A7 E0A7 6027 797D E0A7 722D      E788AD       E788AD       F3B1A0A8
E0A6     E0A6 E0A6 E0A6 E0A6 5E60 E0A8 E0A8 E0A8 E0A8 6028 794D E0A8 722C      E788AC       E788AC       F3B1A0A9
E0A7     E0A7 E0A7 E0A7 E0A7 5E61 E0A9 E0A9 E0A9 E0A9 6029 795D E0A9 7230      E788B0       E788B0       F3B1A0AA
E0A8     E0A8 E0A8 E0A8 E0A8 5E62 E0AA E0AA E0AA E0AA 602A 795C E0AA 7232      E788B2       E788B2       F3B1A0AB
E0A9     E0A9 E0A9 E0A9 E0A9 5E63 E0AB E0AB E0AB E0AB 602B 794E E0AB 723B      E788BB       E788BB       F3B1A0AC
E0AA     E0AA E0AA E0AA E0AA 5E64 E0AC E0AC E0AC E0AC 602C 796B E0AC 723C      E788BC       E788BC       F3B1A0AD
E0AB     E0AB E0AB E0AB E0AB 5E65 E0AD E0AD E0AD E0AD 602D 7960 E0AD 723F      E788BF       E788BF       F3B1A0AE
E0AC     E0AC E0AC E0AC E0AC 5E66 E0AE E0AE E0AE E0AE 602E 794B E0AE 7240      E78980       E78980       F3B1A0AF
E0AD     E0AD E0AD E0AD E0AD 5E67 E0AF E0AF E0AF E0AF 602F 7961 E0AF 7246      E78986       E78986       F3B1A0B0
E0AE     E0AE E0AE E0AE E0AE 5E68 E0B0 E0B0 E0B0 E0B0 6030 79F0 E0B0 724B      E7898B       E7898B       F3B1A0B1
E0AF     E0AF E0AF E0AF E0AF 5E69 E0B1 E0B1 E0B1 E0B1 6031 79F1 E0B1 7258      E78998       E78998       F3B1A0B2
E0B0     E0B0 E0B0 E0B0 E0B0 5E6A E0B2 E0B2 E0B2 E0B2 6032 79F2 E0B2 7274      E789B4       E789B4       F3B1A0B3
E0B1     E0B1 E0B1 E0B1 E0B1 5E6B E0B3 E0B3 E0B3 E0B3 6033 79F3 E0B3 727E      E789BE       E789BE       F3B1A0B4
E0B2     E0B2 E0B2 E0B2 E0B2 5E6E E0B4 E0B4 E0B4 E0B4 6034 79F4 E0B4 7282      E78A82       E78A82       F3B1A0B5
E0B3     E0B3 E0B3 E0B3 E0B3 5E6C E0B5 E0B5 E0B5 E0B5 6035 79F5 E0B5 7281      E78A81       E78A81       F3B1A0B6
E0B4     E0B4 E0B4 E0B4 E0B4 5E6D E0B6 E0B6 E0B6 E0B6 6036 79F6 E0B6 7287      E78A87       E78A87       F3B1A0B7
E0B5     E0B5 E0B5 E0B5 E0B5 5E6F E0B7 E0B7 E0B7 E0B7 6037 79F7 E0B7 7292      E78A92       E78A92       F3B1A0B8
E0B6     E0B6 E0B6 E0B6 E0B6 5E70 E0B8 E0B8 E0B8 E0B8 6038 79F8 E0B8 7296      E78A96       E78A96       F3B1A0B9
E0B7     E0B7 E0B7 E0B7 E0B7 5E71 E0B9 E0B9 E0B9 E0B9 6039 79F9 E0B9 72A2      E78AA2       E78AA2       F3B1A0BA
E0B8     E0B8 E0B8 E0B8 E0B8 5E72 E0BA E0BA E0BA E0BA 603A 797A E0BA 72A7      E78AA7       E78AA7       F3B1A0BB
E0B9     E0B9 E0B9 E0B9 E0B9 5E79 E0BB E0BB E0BB E0BB 603B 795E E0BB 72B9      E78AB9       E78AB9       F3B1A0BC
E0BA     E0BA E0BA E0BA E0BA 5E74 E0BC E0BC E0BC E0BC 603C 794C E0BC 72B2      E78AB2       E78AB2       F3B1A0BD
E0BB     E0BB E0BB E0BB E0BB 5E76 E0BD E0BD E0BD E0BD 603D 797E E0BD 72C3      E78B83       E78B83       F3B1A0BE
E0BC     E0BC E0BC E0BC E0BC 5E77 E0BE E0BE E0BE E0BE 603E 796E E0BE 72C6      E78B86       E78B86       F3B1A0BF
E0BD     E0BD E0BD E0BD E0BD 5E78 E0BF E0BF E0BF E0BF 603F 796F E0BF 72C4      E78B84       E78B84       F3B1A180
E0BE     E0BE E0BE E0BE E0BE 5E7A E0C0 E0C0 E0C0 E0C0 6040 797C E0C0 72CE      E78B8E       E78B8E       F3B1A181
E0BF     E0BF E0BF E0BF E0BF 5E7B E0C1 E0C1 E0C1 E0C1 6041 79C1 E0C1 72D2      E78B92       E78B92       F3B1A182
E0C0     E0C0 E0C0 E0C0 E0C0 5E7C E0C2 E0C2 E0C2 E0C2 6042 79C2 E0C2 72E2      E78BA2       E78BA2       F3B1A183
E0C1     E0C1 E0C1 E0C1 E0C1 5E7D E0C3 E0C3 E0C3 E0C3 6043 79C3 E0C3 72E0      E78BA0       E78BA0       F3B1A184
E0C2     E0C2 E0C2 E0C2 E0C2 5E7E E0C4 E0C4 E0C4 E0C4 6044 79C4 E0C4 72E1      E78BA1       E78BA1       F3B1A185
E0C3     E0C3 E0C3 E0C3 E0C3 5E7F E0C5 E0C5 E0C5 E0C5 6045 79C5 E0C5 72F9      E78BB9       E78BB9       F3B1A186
E0C4     E0C4 E0C4 E0C4 E0C4 5E80 E0C6 E0C6 E0C6 E0C6 6046 79C6 E0C6 72F7      E78BB7       E78BB7       F3B1A187
E0C5     E0C5 E0C5 E0C5 E0C5 56A1 E0C7 E0C7 E0C7 E0C7 6047 79C7 E0C7 500F      E5808F       E5808F       F3B1A188
E0C6     E0C6 E0C6 E0C6 E0C6 5E81 E0C8 E0C8 E0C8 E0C8 6048 79C8 E0C8 7317      E78C97       E78C97       F3B1A189
E0C7     E0C7 E0C7 E0C7 E0C7 5E82 E0C9 E0C9 E0C9 E0C9 6049 79C9 E0C9 730A      E78C8A       E78C8A       F3B1A18A
E0C8     E0C8 E0C8 E0C8 E0C8 5E83 E0CA E0CA E0CA E0CA 604A 79D1 E0CA 731C      E78C9C       E78C9C       F3B1A18B
E0C9     E0C9 E0C9 E0C9 E0C9 5E84 E0CB E0CB E0CB E0CB 604B 79D2 E0CB 7316      E78C96       E78C96       F3B1A18C
E0CA     E0CA E0CA E0CA E0CA 5E85 E0CC E0CC E0CC E0CC 604C 79D3 E0CC 731D      E78C9D       E78C9D       F3B1A18D
E0CB     E0CB E0CB E0CB E0CB 5E87 E0CD E0CD E0CD E0CD 604D 79D4 E0CD 7334      E78CB4       E78CB4       F3B1A18E
E0CC     E0CC E0CC E0CC E0CC 5E89 E0CE E0CE E0CE E0CE 604E 79D5 E0CE 732F      E78CAF       E78CAF       F3B1A18F
E0CD     E0CD E0CD E0CD E0CD 5E88 E0CF E0CF E0CF E0CF 604F 79D6 E0CF 7329      E78CA9       E78CA9       F3B1A190
E0CE     E0CE E0CE E0CE E0CE 5E8B E0D0 E0D0 E0D0 E0D0 6050 79D7 E0D0 7325      E78CA5       E78CA5       F3B1A191
E0CF     E0CF E0CF E0CF E0CF 5E8C E0D1 E0D1 E0D1 E0D1 6051 79D8 E0D1 733E      E78CBE       E78CBE       F3B1A192
E0D0     E0D0 E0D0 E0D0 E0D0 5E8E E0D2 E0D2 E0D2 E0D2 6052 79D9 E0D2 734E      E78D8E       E78D8E       F3B1A193
E0D1     E0D1 E0D1 E0D1 E0D1 5E8D E0D3 E0D3 E0D3 E0D3 6053 79E2 E0D3 734F      E78D8F       E78D8F       F3B1A194
E0D2     E0D2 E0D2 E0D2 E0D2 5E90 E0D4 E0D4 E0D4 E0D4 6054 79E3 E0D4 9ED8      E9BB98       E9BB98       F3B1A195
E0D3     E0D3 E0D3 E0D3 E0D3 5E8F E0D5 E0D5 E0D5 E0D5 6055 79E4 E0D5 7357      E78D97       E78D97       F3B1A196
E0D4     E0D4 E0D4 E0D4 E0D4 5E91 E0D6 E0D6 E0D6 E0D6 6056 79E5 E0D6 736A      E78DAA       E78DAA       F3B1A197
E0D5     E0D5 E0D5 E0D5 E0D5 5E92 E0D7 E0D7 E0D7 E0D7 6057 79E6 E0D7 7368      E78DA8       E78DA8       F3B1A198
E0D6     E0D6 E0D6 E0D6 E0D6 5E93 E0D8 E0D8 E0D8 E0D8 6058 79E7 E0D8 7370      E78DB0       E78DB0       F3B1A199
E0D7     E0D7 E0D7 E0D7 E0D7 5E95 E0D9 E0D9 E0D9 E0D9 6059 79E8 E0D9 7378      E78DB8       E78DB8       F3B1A19A
E0D8     E0D8 E0D8 E0D8 E0D8 5E96 E0DA E0DA E0DA E0DA 605A 79E9 E0DA 7375      E78DB5       E78DB5       F3B1A19B
E0D9     E0D9 E0D9 E0D9 E0D9 5E97 E0DB E0DB E0DB E0DB 605B 794A E0DB 737B      E78DBB       E78DBB       F3B1A19C
E0DA     E0DA E0DA E0DA E0DA 51CB E0DC E0DC E0DC E0DC 605C 795B E0DC 737A      E78DBA       E78DBA       F3B1A19D
E0DB     E0DB E0DB E0DB E0DB 5E98 E0DD E0DD E0DD E0DD 605D 795A E0DD 73C8      E78F88       E78F88       F3B1A19E
E0DC     E0DC E0DC E0DC E0DC 5E9A E0DE E0DE E0DE E0DE 605E 795F E0DE 73B3      E78EB3       E78EB3       F3B1A19F
E0DD     E0DD E0DD E0DD E0DD 5E9B E0DF E0DF E0DF E0DF 605F 796D E0DF 73CE      E78F8E       E78F8E       F3B1A1A0
E0DE     E0DE E0DE E0DE E0DE 5E9C E0E0 E0E0 E0E0 E0E0 6060 7979 E0E0 73BB      E78EBB       E78EBB       F3B1A1A1
E0DF     E0DF E0DF E0DF E0DF 5E9D E0E1 E0E1 E0E1 E0E1 6061 7957 E0E1 73C0      E78F80       E78F80       F3B1A1A2
E0E0     E0E0 E0E0 E0E0 E0E0 5EA0 E0E2 E0E2 E0E2 E0E2 6062 7959 E0E2 73E5      E78FA5       E78FA5       F3B1A1A3
E0E1     E0E1 E0E1 E0E1 E0E1 5EA3 E0E3 E0E3 E0E3 E0E3 6063 7962 E0E3 73EE      E78FAE       E78FAE       F3B1A1A4
E0E2     E0E2 E0E2 E0E2 E0E2 5EA5 E0E4 E0E4 E0E4 E0E4 6064 7963 E0E4 73DE      E78F9E       E78F9E       F3B1A1A5
E0E3     E0E3 E0E3 E0E3 E0E3 5EC2 E0E5 E0E5 E0E5 E0E5 6065 7964 E0E5 74A2      E792A2       E792A2       F3B1A1A6
E0E4     E0E4 E0E4 E0E4 E0E4 5EA9 E0E6 E0E6 E0E6 E0E6 6066 7965 E0E6 7405      E79085       E79085       F3B1A1A7
E0E5     E0E5 E0E5 E0E5 E0E5 5EBE E0E7 E0E7 E0E7 E0E7 6067 7966 E0E7 746F      E791AF       E791AF       F3B1A1A8
E0E6     E0E6 E0E6 E0E6 E0E6 5EAC E0E8 E0E8 E0E8 E0E8 6068 7967 E0E8 7425      E790A5       E790A5       F3B1A1A9
E0E7     E0E7 E0E7 E0E7 E0E7 5EA6 E0E9 E0E9 E0E9 E0E9 6069 7968 E0E9 73F8      E78FB8       E78FB8       F3B1A1AA
E0E8     E0E8 E0E8 E0E8 E0E8 5EAF E0EA E0EA E0EA E0EA 606A 7969 E0EA 7432      E790B2       E790B2       F3B1A1AB
E0E9     E0E9 E0E9 E0E9 E0E9 5EB0 E0EB E0EB E0EB E0EB 606B 7970 E0EB 743A      E790BA       E790BA       F3B1A1AC
E0EA     E0EA E0EA E0EA E0EA 5EB1 E0EC E0EC E0EC E0EC 606C 7971 E0EC 7455      E79195       E79195       F3B1A1AD
E0EB     E0EB E0EB E0EB E0EB 5EB2 E0ED E0ED E0ED E0ED 606D 7972 E0ED 743F      E790BF       E790BF       F3B1A1AE
E0EC     E0EC E0EC E0EC E0EC 5EB3 E0EE E0EE E0EE E0EE 606E 7973 E0EE 745F      E7919F       E7919F       F3B1A1AF
E0ED     E0ED E0ED E0ED E0ED 5EB4 E0EF E0EF E0EF E0EF 606F 7974 E0EF 7459      E79199       E79199       F3B1A1B0
E0EE     E0EE E0EE E0EE E0EE 5EB5 E0F0 E0F0 E0F0 E0F0 6070 7975 E0F0 7441      E79181       E79181       F3B1A1B1
E0EF     E0EF E0EF E0EF E0EF 5EB6 E0F1 E0F1 E0F1 E0F1 6071 7976 E0F1 745C      E7919C       E7919C       F3B1A1B2
E0F0     E0F0 E0F0 E0F0 E0F0 5EB7 E0F2 E0F2 E0F2 E0F2 6072 7977 E0F2 7469      E791A9       E791A9       F3B1A1B3
E0F1     E0F1 E0F1 E0F1 E0F1 5EB8 E0F3 E0F3 E0F3 E0F3 6073 7978 E0F3 7470      E791B0       E791B0       F3B1A1B4
E0F2     E0F2 E0F2 E0F2 E0F2 5EB9 E0F4 E0F4 E0F4 E0F4 6074 7980 E0F4 7463      E791A3       E791A3       F3B1A1B5
E0F3     E0F3 E0F3 E0F3 E0F3 5EBA E0F5 E0F5 E0F5 E0F5 6075 798B E0F5 746A      E791AA       E791AA       F3B1A1B6
E0F4     E0F4 E0F4 E0F4 E0F4 6883 E0F6 E0F6 E0F6 59A2 C4E8 95CA E0F6 7476      E791B6       E791B6       F3B1A1B7
E0F5     E0F5 E0F5 E0F5 E0F5 5EBF E0F7 E0F7 E0F7 E0F7 6077 799C E0F7 747E      E791BE       E791BE       F3B1A1B8
E0F6     E0F6 E0F6 E0F6 E0F6 504D E0F8 E0F8 E0F8 E0F8 6078 79A0 E0F8 748B      E7928B       E7928B       F3B1A1B9
E0F7     E0F7 E0F7 E0F7 E0F7 5EC1 E0F9 E0F9 E0F9 E0F9 6079 79AB E0F9 749E      E7929E       E7929E       F3B1A1BA
E0F8     E0F8 E0F8 E0F8 E0F8 5EC3 E0FA E0FA E0FA E0FA 607A 79B0 E0FA 74A7      E792A7       E792A7       F3B1A1BB
E0F9     E0F9 E0F9 E0F9 E0F9 5EC4 E0FB E0FB E0FB E0FB 607B 79C0 E0FB 74CA      E7938A       E7938A       F3B1A1BC
E0FA     E0FA E0FA E0FA E0FA 5EC5 E0FC E0FC E0FC E0FC 607C 796A E0FC 74CF      E7938F       E7938F       F3B1A1BD
E0FB     E0FB E0FB E0FB E0FB 5EC6 E0FD E0FD E0FD E0FD 607D 79D0 E0FD 74D4      E79394       E79394       F3B1A1BE
E0FC     E0FC E0FC E0FC E0FC 5EA4 E0FE E0FE E0FE E0FE 607E 79A1 E0FE 73F1      E78FB1       E78FB1       F3B1A1BF
E140     E140 E140 E140 E140 5EC7 E1A1 E1A1 E1A1 E1A1 6121 574F E1A1 74E0      E793A0       E793A0       F3B1A280
E141     E141 E141 E141 E141 5EC8 E1A2 E1A2 E1A2 E1A2 6122 577F E1A2 74E3      E793A3       E793A3       F3B1A281
E142     E142 E142 E142 E142 5EC9 E1A3 E1A3 E1A3 E1A3 6123 577B E1A3 74E7      E793A7       E793A7       F3B1A282
E143     E143 E143 E143 E143 5ECA E1A4 E1A4 E1A4 E1A4 6124 57E0 E1A4 74E9      E793A9       E793A9       F3B1A283
E144     E144 E144 E144 E144 5ECB E1A5 E1A5 E1A5 E1A5 6125 576C E1A5 74EE      E793AE       E793AE       F3B1A284
E145     E145 E145 E145 E145 5ECD E1A6 E1A6 E1A6 E1A6 6126 5750 E1A6 74F2      E793B2       E793B2       F3B1A285
E146     E146 E146 E146 E146 5ECC E1A7 E1A7 E1A7 E1A7 6127 577D E1A7 74F0      E793B0       E793B0       F3B1A286
E147     E147 E147 E147 E147 5ECE E1A8 E1A8 E1A8 E1A8 6128 574D E1A8 74F1      E793B1       E793B1       F3B1A287
E148     E148 E148 E148 E148 5ED0 E1A9 E1A9 E1A9 E1A9 6129 575D E1A9 74F8      E793B8       E793B8       F3B1A288
E149     E149 E149 E149 E149 5ECF E1AA E1AA E1AA E1AA 612A 575C E1AA 74F7      E793B7       E793B7       F3B1A289
E14A     E14A E14A E14A E14A 5ED2 E1AB E1AB E1AB E1AB 612B 574E E1AB 7504      E79484       E79484       F3B1A28A
E14B     E14B E14B E14B E14B 5ED3 E1AC E1AC E1AC E1AC 612C 576B E1AC 7503      E79483       E79483       F3B1A28B
E14C     E14C E14C E14C E14C 5ED4 E1AD E1AD E1AD E1AD 612D 5760 E1AD 7505      E79485       E79485       F3B1A28C
E14D     E14D E14D E14D E14D 5ED6 E1AE E1AE E1AE E1AE 612E 574B E1AE 750C      E7948C       E7948C       F3B1A28D
E14E     E14E E14E E14E E14E 5ED7 E1AF E1AF E1AF E1AF 612F 5761 E1AF 750E      E7948E       E7948E       F3B1A28E
E14F     E14F E14F E14F E14F 5ED5 E1B0 E1B0 E1B0 E1B0 6130 57F0 E1B0 750D      E7948D       E7948D       F3B1A28F
E150     E150 E150 E150 E150 53E6 E1B1 E1B1 E1B1 E1B1 6131 57F1 E1B1 7515      E79495       E79495       F3B1A290
E151     E151 E151 E151 E151 5ED8 E1B2 E1B2 E1B2 E1B2 6132 57F2 E1B2 7513      E79493       E79493       F3B1A291
E152     E152 E152 E152 E152 5ED9 E1B3 E1B3 E1B3 E1B3 6133 57F3 E1B3 751E      E7949E       E7949E       F3B1A292
E153     E153 E153 E153 E153 5EDA E1B4 E1B4 E1B4 E1B4 6134 57F4 E1B4 7526      E794A6       E794A6       F3B1A293
E154     E154 E154 E154 E154 5EDB E1B5 E1B5 E1B5 E1B5 6135 57F5 E1B5 752C      E794AC       E794AC       F3B1A294
E155     E155 E155 E155 E155 5EDC E1B6 E1B6 E1B6 E1B6 6136 57F6 E1B6 753C      E794BC       E794BC       F3B1A295
E156     E156 E156 E156 E156 5EDD E1B7 E1B7 E1B7 E1B7 6137 57F7 E1B7 7544      E79584       E79584       F3B1A296
E157     E157 E157 E157 E157 5EDE E1B8 E1B8 E1B8 E1B8 6138 57F8 E1B8 754D      E7958D       E7958D       F3B1A297
E158     E158 E158 E158 E158 5EDF E1B9 E1B9 E1B9 E1B9 6139 57F9 E1B9 754A      E7958A       E7958A       F3B1A298
E159     E159 E159 E159 E159 5EE0 E1BA E1BA E1BA E1BA 613A 577A E1BA 7549      E79589       E79589       F3B1A299
E15A     E15A E15A E15A E15A 5EE2 E1BB E1BB E1BB E1BB 613B 575E E1BB 755B      E7959B       E7959B       F3B1A29A
E15B     E15B E15B E15B E15B 5EE1 E1BC E1BC E1BC E1BC 613C 574C E1BC 7546      E79586       E79586       F3B1A29B
E15C     E15C E15C E15C E15C 5EE3 E1BD E1BD E1BD E1BD 613D 577E E1BD 755A      E7959A       E7959A       F3B1A29C
E15D     E15D E15D E15D E15D 52FD E1BE E1BE E1BE E1BE 613E 576E E1BE 7569      E795A9       E795A9       F3B1A29D
E15E     E15E E15E E15E E15E 5EE4 E1BF E1BF E1BF E1BF 613F 576F E1BF 7564      E795A4       E795A4       F3B1A29E
E15F     E15F E15F E15F E15F 5EE5 E1C0 E1C0 E1C0 E1C0 6140 577C E1C0 7567      E795A7       E795A7       F3B1A29F
E160     E160 E160 E160 E160 5EE6 E1C1 E1C1 E1C1 E1C1 6141 57C1 E1C1 756B      E795AB       E795AB       F3B1A2A0
E161     E161 E161 E161 E161 5EE9 E1C2 E1C2 E1C2 E1C2 6142 57C2 E1C2 756D      E795AD       E795AD       F3B1A2A1
E162     E162 E162 E162 E162 5EEA E1C3 E1C3 E1C3 E1C3 6143 57C3 E1C3 7578      E795B8       E795B8       F3B1A2A2
E163     E163 E163 E163 E163 5EEB E1C4 E1C4 E1C4 E1C4 6144 57C4 E1C4 7576      E795B6       E795B6       F3B1A2A3
E164     E164 E164 E164 E164 5EED E1C5 E1C5 E1C5 E1C5 6145 57C5 E1C5 7586      E79686       E79686       F3B1A2A4
E165     E165 E165 E165 E165 5EEE E1C6 E1C6 E1C6 E1C6 6146 57C6 E1C6 7587      E79687       E79687       F3B1A2A5
E166     E166 E166 E166 E166 5EE8 E1C7 E1C7 E1C7 E1C7 6147 57C7 E1C7 7574      E795B4       E795B4       F3B1A2A6
E167     E167 E167 E167 E167 5EEF E1C8 E1C8 E1C8 E1C8 6148 57C8 E1C8 758A      E7968A       E7968A       F3B1A2A7
E168     E168 E168 E168 E168 5EF0 E1C9 E1C9 E1C9 E1C9 6149 57C9 E1C9 7589      E79689       E79689       F3B1A2A8
E169     E169 E169 E169 E169 5EEC E1CA E1CA E1CA E1CA 614A 57D1 E1CA 7582      E79682       E79682       F3B1A2A9
E16A     E16A E16A E16A E16A 5EF1 E1CB E1CB E1CB E1CB 614B 57D2 E1CB 7594      E79694       E79694       F3B1A2AA
E16B     E16B E16B E16B E16B 5EF2 E1CC E1CC E1CC E1CC 614C 57D3 E1CC 759A      E7969A       E7969A       F3B1A2AB
E16C     E16C E16C E16C E16C 5EF3 E1CD E1CD E1CD E1CD 614D 57D4 E1CD 759D      E7969D       E7969D       F3B1A2AC
E16D     E16D E16D E16D E16D 5EF4 E1CE E1CE E1CE E1CE 614E 57D5 E1CE 75A5      E796A5       E796A5       F3B1A2AD
E16E     E16E E16E E16E E16E 5EF5 E1CF E1CF E1CF E1CF 614F 57D6 E1CF 75A3      E796A3       E796A3       F3B1A2AE
E16F     E16F E16F E16F E16F 5EF6 E1D0 E1D0 E1D0 E1D0 6150 57D7 E1D0 75C2      E79782       E79782       F3B1A2AF
E170     E170 E170 E170 E170 5EF7 E1D1 E1D1 E1D1 E1D1 6151 57D8 E1D1 75B3      E796B3       E796B3       F3B1A2B0
E171     E171 E171 E171 E171 5EF8 E1D2 E1D2 E1D2 E1D2 6152 57D9 E1D2 75C3      E79783       E79783       F3B1A2B1
E172     E172 E172 E172 E172 5EF9 E1D3 E1D3 E1D3 E1D3 6153 57E2 E1D3 75B5      E796B5       E796B5       F3B1A2B2
E173     E173 E173 E173 E173 5EFA E1D4 E1D4 E1D4 E1D4 6154 57E3 E1D4 75BD      E796BD       E796BD       F3B1A2B3
E174     E174 E174 E174 E174 5EFB E1D5 E1D5 E1D5 E1D5 6155 57E4 E1D5 75B8      E796B8       E796B8       F3B1A2B4
E175     E175 E175 E175 E175 5EFC E1D6 E1D6 E1D6 E1D6 6156 57E5 E1D6 75BC      E796BC       E796BC       F3B1A2B5
E176     E176 E176 E176 E176 5EFD E1D7 E1D7 E1D7 E1D7 6157 57E6 E1D7 75B1      E796B1       E796B1       F3B1A2B6
E177     E177 E177 E177 E177 5EFE E1D8 E1D8 E1D8 E1D8 6158 57E7 E1D8 75CD      E7978D       E7978D       F3B1A2B7
E178     E178 E178 E178 E178 5F41 E1D9 E1D9 E1D9 E1D9 6159 57E8 E1D9 75CA      E7978A       E7978A       F3B1A2B8
E179     E179 E179 E179 E179 5F42 E1DA E1DA E1DA E1DA 615A 57E9 E1DA 75D2      E79792       E79792       F3B1A2B9
E17A     E17A E17A E17A E17A 5F43 E1DB E1DB E1DB E1DB 615B 574A E1DB 75D9      E79799       E79799       F3B1A2BA
E17B     E17B E17B E17B E17B 5F44 E1DC E1DC E1DC E1DC 615C 575B E1DC 75E3      E797A3       E797A3       F3B1A2BB
E17C     E17C E17C E17C E17C 5F45 E1DD E1DD E1DD E1DD 615D 575A E1DD 75DE      E7979E       E7979E       F3B1A2BC
E17D     E17D E17D E17D E17D 5F46 E1DE E1DE E1DE E1DE 615E 575F E1DE 75FE      E797BE       E797BE       F3B1A2BD
E17E     E17E E17E E17E E17E 5F47 E1DF E1DF E1DF E1DF 615F 576D E1DF 75FF      E797BF       E797BF       F3B1A2BE
E180     E180 E180 E180 E180 5F48 E1E0 E1E0 E1E0 E1E0 6160 5779 E1E0 75FC      E797BC       E797BC       F3B1A2BF
E181     E181 E181 E181 E181 5F49 E1E1 E1E1 E1E1 E1E1 6161 5757 E1E1 7601      E79881       E79881       F3B1A380
E182     E182 E182 E182 E182 5F4A E1E2 E1E2 E1E2 E1E2 6162 5759 E1E2 75F0      E797B0       E797B0       F3B1A381
E183     E183 E183 E183 E183 5F4B E1E3 E1E3 E1E3 E1E3 6163 5762 E1E3 75FA      E797BA       E797BA       F3B1A382
E184     E184 E184 E184 E184 5F4C E1E4 E1E4 E1E4 E1E4 6164 5763 E1E4 75F2      E797B2       E797B2       F3B1A383
E185     E185 E185 E185 E185 5F4D E1E5 E1E5 E1E5 E1E5 6165 5764 E1E5 75F3      E797B3       E797B3       F3B1A384
E186     E186 E186 E186 E186 5F4E E1E6 E1E6 E1E6 E1E6 6166 5765 E1E6 760B      E7988B       E7988B       F3B1A385
E187     E187 E187 E187 E187 524F E1E7 E1E7 E1E7 E1E7 6167 5766 E1E7 760D      E7988D       E7988D       F3B1A386
E188     E188 E188 E188 E188 5F4F E1E8 E1E8 E1E8 E1E8 6168 5767 E1E8 7609      E79889       E79889       F3B1A387
E189     E189 E189 E189 E189 5F50 E1E9 E1E9 E1E9 E1E9 6169 5768 E1E9 761F      E7989F       E7989F       F3B1A388
E18A     E18A E18A E18A E18A 5F51 E1EA E1EA E1EA E1EA 616A 5769 E1EA 7627      E798A7       E798A7       F3B1A389
E18B     E18B E18B E18B E18B 5F52 E1EB E1EB E1EB E1EB 616B 5770 E1EB 7620      E798A0       E798A0       F3B1A38A
E18C     E18C E18C E18C E18C 5F53 E1EC E1EC E1EC E1EC 616C 5771 E1EC 7621      E798A1       E798A1       F3B1A38B
E18D     E18D E18D E18D E18D 5F54 E1ED E1ED E1ED E1ED 616D 5772 E1ED 7622      E798A2       E798A2       F3B1A38C
E18E     E18E E18E E18E E18E 5F55 E1EE E1EE E1EE E1EE 616E 5773 E1EE 7624      E798A4       E798A4       F3B1A38D
E18F     E18F E18F E18F E18F 5F56 E1EF E1EF E1EF E1EF 616F 5774 E1EF 7634      E798B4       E798B4       F3B1A38E
E190     E190 E190 E190 E190 5F57 E1F0 E1F0 E1F0 E1F0 6170 5775 E1F0 7630      E798B0       E798B0       F3B1A38F
E191     E191 E191 E191 E191 5F58 E1F1 E1F1 E1F1 E1F1 6171 5776 E1F1 763B      E798BB       E798BB       F3B1A390
E192     E192 E192 E192 E192 5F59 E1F2 E1F2 E1F2 E1F2 6172 5777 E1F2 7647      E79987       E79987       F3B1A391
E193     E193 E193 E193 E193 5F5A E1F3 E1F3 E1F3 E1F3 6173 5778 E1F3 7648      E79988       E79988       F3B1A392
E194     E194 E194 E194 E194 5F5C E1F4 E1F4 E1F4 E1F4 6174 5780 E1F4 7646      E79986       E79986       F3B1A393
E195     E195 E195 E195 E195 5F5D E1F5 E1F5 E1F5 E1F5 6175 578B E1F5 765C      E7999C       E7999C       F3B1A394
E196     E196 E196 E196 E196 5F5B E1F6 E1F6 E1F6 E1F6 6176 579B E1F6 7658      E79998       E79998       F3B1A395
E197     E197 E197 E197 E197 5F5E E1F7 E1F7 E1F7 E1F7 6177 579C E1F7 7661      E799A1       E799A1       F3B1A396
E198     E198 E198 E198 E198 5F5F E1F8 E1F8 E1F8 E1F8 6178 57A0 E1F8 7662      E799A2       E799A2       F3B1A397
E199     E199 E199 E199 E199 5F60 E1F9 E1F9 E1F9 E1F9 6179 57AB E1F9 7668      E799A8       E799A8       F3B1A398
E19A     E19A E19A E19A E19A 5F61 E1FA E1FA E1FA E1FA 617A 57B0 E1FA 7669      E799A9       E799A9       F3B1A399
E19B     E19B E19B E19B E19B 5F63 E1FB E1FB E1FB E1FB 617B 57C0 E1FB 766A      E799AA       E799AA       F3B1A39A
E19C     E19C E19C E19C E19C 5F62 E1FC E1FC E1FC E1FC 617C 576A E1FC 7667      E799A7       E799A7       F3B1A39B
E19D     E19D E19D E19D E19D 5F64 E1FD E1FD E1FD E1FD 617D 57D0 E1FD 766C      E799AC       E799AC       F3B1A39C
E19E     E19E E19E E19E E19E 5F65 E1FE E1FE E1FE E1FE 617E 57A1 E1FE 7670      E799B0       E799B0       F3B1A39D
E19F     E19F E19F E19F E19F 5F66 E2A1 E2A1 E2A1 E2A1 6221 594F E2A1 7672      E799B2       E799B2       F3B1A39E
E1A0     E1A0 E1A0 E1A0 E1A0 5F67 E2A2 E2A2 E2A2 E2A2 6222 597F E2A2 7676      E799B6       E799B6       F3B1A39F
E1A1     E1A1 E1A1 E1A1 E1A1 539A E2A3 E2A3 E2A3 E2A3 6223 597B E2A3 7678      E799B8       E799B8       F3B1A3A0
E1A2     E1A2 E1A2 E1A2 E1A2 5F68 E2A4 E2A4 E2A4 E2A4 6224 59E0 E2A4 767C      E799BC       E799BC       F3B1A3A1
E1A3     E1A3 E1A3 E1A3 E1A3 5F6A E2A5 E2A5 E2A5 E2A5 6225 596C E2A5 7680      E79A80       E79A80       F3B1A3A2
E1A4     E1A4 E1A4 E1A4 E1A4 5F6B E2A6 E2A6 E2A6 E2A6 6226 5950 E2A6 7683      E79A83       E79A83       F3B1A3A3
E1A5     E1A5 E1A5 E1A5 E1A5 5F6C E2A7 E2A7 E2A7 E2A7 6227 597D E2A7 7688      E79A88       E79A88       F3B1A3A4
E1A6     E1A6 E1A6 E1A6 E1A6 5F6D E2A8 E2A8 E2A8 E2A8 6228 594D E2A8 768B      E79A8B       E79A8B       F3B1A3A5
E1A7     E1A7 E1A7 E1A7 E1A7 5299 E2A9 E2A9 E2A9 E2A9 6229 595D E2A9 768E      E79A8E       E79A8E       F3B1A3A6
E1A8     E1A8 E1A8 E1A8 E1A8 5396 E2AA E2AA E2AA E2AA 622A 595C E2AA 7696      E79A96       E79A96       F3B1A3A7
E1A9     E1A9 E1A9 E1A9 E1A9 4E81 E2AB E2AB E2AB E2AB 622B 594E E2AB 7693      E79A93       E79A93       F3B1A3A8
E1AA     E1AA E1AA E1AA E1AA 5F6E E2AC E2AC E2AC E2AC 622C 596B E2AC 7699      E79A99       E79A99       F3B1A3A9
E1AB     E1AB E1AB E1AB E1AB 5F6F E2AD E2AD E2AD E2AD 622D 5960 E2AD 769A      E79A9A       E79A9A       F3B1A3AA
E1AC     E1AC E1AC E1AC E1AC 5F74 E2AE E2AE E2AE E2AE 622E 594B E2AE 76B0      E79AB0       E79AB0       F3B1A3AB
E1AD     E1AD E1AD E1AD E1AD 5F75 E2AF E2AF E2AF E2AF 622F 5961 E2AF 76B4      E79AB4       E79AB4       F3B1A3AC
E1AE     E1AE E1AE E1AE E1AE 5F76 E2B0 E2B0 E2B0 E2B0 6230 59F0 E2B0 76B8      E79AB8       E79AB8       F3B1A3AD
E1AF     E1AF E1AF E1AF E1AF 5F77 E2B1 E2B1 E2B1 E2B1 6231 59F1 E2B1 76B9      E79AB9       E79AB9       F3B1A3AE
E1B0     E1B0 E1B0 E1B0 E1B0 5F78 E2B2 E2B2 E2B2 E2B2 6232 59F2 E2B2 76BA      E79ABA       E79ABA       F3B1A3AF
E1B1     E1B1 E1B1 E1B1 E1B1 5F79 E2B3 E2B3 E2B3 E2B3 6233 59F3 E2B3 76C2      E79B82       E79B82       F3B1A3B0
E1B2     E1B2 E1B2 E1B2 E1B2 5F7B E2B4 E2B4 E2B4 E2B4 6234 59F4 E2B4 76CD      E79B8D       E79B8D       F3B1A3B1
E1B3     E1B3 E1B3 E1B3 E1B3 5F7C E2B5 E2B5 E2B5 E2B5 6235 59F5 E2B5 76D6      E79B96       E79B96       F3B1A3B2
E1B4     E1B4 E1B4 E1B4 E1B4 5F7D E2B6 E2B6 E2B6 E2B6 6236 59F6 E2B6 76D2      E79B92       E79B92       F3B1A3B3
E1B5     E1B5 E1B5 E1B5 E1B5 5F7E E2B7 E2B7 E2B7 E2B7 6237 59F7 E2B7 76DE      E79B9E       E79B9E       F3B1A3B4
E1B6     E1B6 E1B6 E1B6 E1B6 5F7F E2B8 E2B8 E2B8 E2B8 6238 59F8 E2B8 76E1      E79BA1       E79BA1       F3B1A3B5
E1B7     E1B7 E1B7 E1B7 E1B7 5F80 E2B9 E2B9 E2B9 E2B9 6239 59F9 E2B9 76E5      E79BA5       E79BA5       F3B1A3B6
E1B8     E1B8 E1B8 E1B8 E1B8 53F7 E2BA E2BA E2BA E2BA 623A 597A E2BA 76E7      E79BA7       E79BA7       F3B1A3B7
E1B9     E1B9 E1B9 E1B9 E1B9 5F81 E2BB E2BB E2BB E2BB 623B 595E E2BB 76EA      E79BAA       E79BAA       F3B1A3B8
E1BA     E1BA E1BA E1BA E1BA 5F82 E2BC E2BC E2BC E2BC 623C 594C E2BC 862F      E898AF       E898AF       F3B1A3B9
E1BB     E1BB E1BB E1BB E1BB 5F83 E2BD E2BD E2BD E2BD 623D 597E E2BD 76FB      E79BBB       E79BBB       F3B1A3BA
E1BC     E1BC E1BC E1BC E1BC 5F84 E2BE E2BE E2BE E2BE 623E 596E E2BE 7708      E79C88       E79C88       F3B1A3BB
E1BD     E1BD E1BD E1BD E1BD 5F85 E2BF E2BF E2BF E2BF 623F 596F E2BF 7707      E79C87       E79C87       F3B1A3BC
E1BE     E1BE E1BE E1BE E1BE 5F86 E2C0 E2C0 E2C0 E2C0 6240 597C E2C0 7704      E79C84       E79C84       F3B1A3BD
E1BF     E1BF E1BF E1BF E1BF 5F88 E2C1 E2C1 E2C1 E2C1 6241 59C1 E2C1 7729      E79CA9       E79CA9       F3B1A3BE
E1C0     E1C0 E1C0 E1C0 E1C0 5F87 E2C2 E2C2 E2C2 E2C2 6242 59C2 E2C2 7724      E79CA4       E79CA4       F3B1A3BF
E1C1     E1C1 E1C1 E1C1 E1C1 51CA E2C3 E2C3 E2C3 E2C3 6243 59C3 E2C3 771E      E79C9E       E79C9E       F3B1A480
E1C2     E1C2 E1C2 E1C2 E1C2 5F89 E2C4 E2C4 E2C4 E2C4 6244 59C4 E2C4 7725      E79CA5       E79CA5       F3B1A481
E1C3     E1C3 E1C3 E1C3 E1C3 5F8A E2C5 E2C5 E2C5 E2C5 6245 59C5 E2C5 7726      E79CA6       E79CA6       F3B1A482
E1C4     E1C4 E1C4 E1C4 E1C4 5F8B E2C6 E2C6 E2C6 E2C6 6246 59C6 E2C6 771B      E79C9B       E79C9B       F3B1A483
E1C5     E1C5 E1C5 E1C5 E1C5 5F8C E2C7 E2C7 E2C7 E2C7 6247 59C7 E2C7 7737      E79CB7       E79CB7       F3B1A484
E1C6     E1C6 E1C6 E1C6 E1C6 5F8D E2C8 E2C8 E2C8 E2C8 6248 59C8 E2C8 7738      E79CB8       E79CB8       F3B1A485
E1C7     E1C7 E1C7 E1C7 E1C7 5F8F E2C9 E2C9 E2C9 E2C9 6249 59C9 E2C9 7747      E79D87       E79D87       F3B1A486
E1C8     E1C8 E1C8 E1C8 E1C8 5F90 E2CA E2CA E2CA E2CA 624A 59D1 E2CA 775A      E79D9A       E79D9A       F3B1A487
E1C9     E1C9 E1C9 E1C9 E1C9 5F91 E2CB E2CB E2CB E2CB 624B 59D2 E2CB 7768      E79DA8       E79DA8       F3B1A488
E1CA     E1CA E1CA E1CA E1CA 5F92 E2CC E2CC E2CC E2CC 624C 59D3 E2CC 776B      E79DAB       E79DAB       F3B1A489
E1CB     E1CB E1CB E1CB E1CB 5F93 E2CD E2CD E2CD E2CD 624D 59D4 E2CD 775B      E79D9B       E79D9B       F3B1A48A
E1CC     E1CC E1CC E1CC E1CC 5F94 E2CE E2CE E2CE E2CE 624E 59D5 E2CE 7765      E79DA5       E79DA5       F3B1A48B
E1CD     E1CD E1CD E1CD E1CD 5F95 E2CF E2CF E2CF E2CF 624F 59D6 E2CF 777F      E79DBF       E79DBF       F3B1A48C
E1CE     E1CE E1CE E1CE E1CE 5F96 E2D0 E2D0 E2D0 E2D0 6250 59D7 E2D0 777E      E79DBE       E79DBE       F3B1A48D
E1CF     E1CF E1CF E1CF E1CF 5F97 E2D1 E2D1 E2D1 E2D1 6251 59D8 E2D1 7779      E79DB9       E79DB9       F3B1A48E
E1D0     E1D0 E1D0 E1D0 E1D0 5F98 E2D2 E2D2 E2D2 E2D2 6252 59D9 E2D2 778E      E79E8E       E79E8E       F3B1A48F
E1D1     E1D1 E1D1 E1D1 E1D1 5F99 E2D3 E2D3 E2D3 E2D3 6253 59E2 E2D3 778B      E79E8B       E79E8B       F3B1A490
E1D2     E1D2 E1D2 E1D2 E1D2 5F9A E2D4 E2D4 E2D4 E2D4 6254 59E3 E2D4 7791      E79E91       E79E91       F3B1A491
E1D3     E1D3 E1D3 E1D3 E1D3 5F9B E2D5 E2D5 E2D5 E2D5 6255 59E4 E2D5 77A0      E79EA0       E79EA0       F3B1A492
E1D4     E1D4 E1D4 E1D4 E1D4 5F9C E2D6 E2D6 E2D6 E2D6 6256 59E5 E2D6 779E      E79E9E       E79E9E       F3B1A493
E1D5     E1D5 E1D5 E1D5 E1D5 5F9D E2D7 E2D7 E2D7 E2D7 6257 59E6 E2D7 77B0      E79EB0       E79EB0       F3B1A494
E1D6     E1D6 E1D6 E1D6 E1D6 5F9E E2D8 E2D8 E2D8 E2D8 6258 59E7 E2D8 77B6      E79EB6       E79EB6       F3B1A495
E1D7     E1D7 E1D7 E1D7 E1D7 5F9F E2D9 E2D9 E2D9 E2D9 6259 59E8 E2D9 77B9      E79EB9       E79EB9       F3B1A496
E1D8     E1D8 E1D8 E1D8 E1D8 5FA0 E2DA E2DA E2DA E2DA 625A 59E9 E2DA 77BF      E79EBF       E79EBF       F3B1A497
E1D9     E1D9 E1D9 E1D9 E1D9 5FA1 E2DB E2DB E2DB E2DB 625B 594A E2DB 77BC      E79EBC       E79EBC       F3B1A498
E1DA     E1DA E1DA E1DA E1DA 5FA2 E2DC E2DC E2DC E2DC 625C 595B E2DC 77BD      E79EBD       E79EBD       F3B1A499
E1DB     E1DB E1DB E1DB E1DB 5FA3 E2DD E2DD E2DD E2DD 625D 595A E2DD 77BB      E79EBB       E79EBB       F3B1A49A
E1DC     E1DC E1DC E1DC E1DC 5FA4 E2DE E2DE E2DE E2DE 625E 595F E2DE 77C7      E79F87       E79F87       F3B1A49B
E1DD     E1DD E1DD E1DD E1DD 5FA5 E2DF E2DF E2DF E2DF 625F 596D E2DF 77CD      E79F8D       E79F8D       F3B1A49C
E1DE     E1DE E1DE E1DE E1DE 5350 E2E0 E2E0 E2E0 E2E0 6260 5979 E2E0 77D7      E79F97       E79F97       F3B1A49D
E1DF     E1DF E1DF E1DF E1DF 5FA6 E2E1 E2E1 E2E1 E2E1 6261 5957 E2E1 77DA      E79F9A       E79F9A       F3B1A49E
E1E0     E1E0 E1E0 E1E0 E1E0 5FA7 E2E2 E2E2 E2E2 E2E2 6262 5959 E2E2 77DC      E79F9C       E79F9C       F3B1A49F
E1E1     E1E1 E1E1 E1E1 E1E1 5FA8 E2E3 E2E3 E2E3 E2E3 6263 5962 E2E3 77E3      E79FA3       E79FA3       F3B1A4A0
E1E2     E1E2 E1E2 E1E2 E1E2 5FA9 E2E4 E2E4 E2E4 E2E4 6264 5963 E2E4 77EE      E79FAE       E79FAE       F3B1A4A1
E1E3     E1E3 E1E3 E1E3 E1E3 5FAB E2E5 E2E5 E2E5 E2E5 6265 5964 E2E5 77FC      E79FBC       E79FBC       F3B1A4A2
E1E4     E1E4 E1E4 E1E4 E1E4 5FAC E2E6 E2E6 E2E6 E2E6 6266 5965 E2E6 780C      E7A08C       E7A08C       F3B1A4A3
E1E5     E1E5 E1E5 E1E5 E1E5 5FAD E2E7 E2E7 E2E7 E2E7 6267 5966 E2E7 7812      E7A092       E7A092       F3B1A4A4
E1E6     E1E6 E1E6 E1E6 E1E6 51B9 B9DC E2E8 E2E8 B9DC 395C F95B E2E8 7926      E7A4A6       E7A4A6       F3B1A4A5
E1E7     E1E7 E1E7 E1E7 E1E7 5FB0 E2E9 E2E9 E2E9 E2E9 6269 5968 E2E9 7820      E7A0A0       E7A0A0       F3B1A4A6
E1E8     E1E8 E1E8 E1E8 E1E8 514C C5D7 E2EA E2EA C5D7 4557 C5E6 E2EA 792A      E7A4AA       E7A4AA       F3B1A4A7
E1E9     E1E9 E1E9 E1E9 E1E9 5FB1 E2EB E2EB E2EB E2EB 626B 5970 E2EB 7845      E7A185       E7A185       F3B1A4A8
E1EA     E1EA E1EA E1EA E1EA 5FB5 E2EC E2EC E2EC E2EC 626C 5971 E2EC 788E      E7A28E       E7A28E       F3B1A4A9
E1EB     E1EB E1EB E1EB E1EB 5FB4 E2ED E2ED E2ED E2ED 626D 5972 E2ED 7874      E7A1B4       E7A1B4       F3B1A4AA
E1EC     E1EC E1EC E1EC E1EC 5FB7 E2EE E2EE E2EE E2EE 626E 5973 E2EE 7886      E7A286       E7A286       F3B1A4AB
E1ED     E1ED E1ED E1ED E1ED 5FB9 E2EF E2EF E2EF E2EF 626F 5974 E2EF 787C      E7A1BC       E7A1BC       F3B1A4AC
E1EE     E1EE E1EE E1EE E1EE 5FB8 E2F0 E2F0 E2F0 E2F0 6270 5975 E2F0 789A      E7A29A       E7A29A       F3B1A4AD
E1EF     E1EF E1EF E1EF E1EF 5FBA E2F1 E2F1 E2F1 E2F1 6271 5976 E2F1 788C      E7A28C       E7A28C       F3B1A4AE
E1F0     E1F0 E1F0 E1F0 E1F0 5FBB E2F2 E2F2 E2F2 E2F2 6272 5977 E2F2 78A3      E7A2A3       E7A2A3       F3B1A4AF
E1F1     E1F1 E1F1 E1F1 E1F1 5FBC E2F3 E2F3 E2F3 E2F3 6273 5978 E2F3 78B5      E7A2B5       E7A2B5       F3B1A4B0
E1F2     E1F2 E1F2 E1F2 E1F2 5FBD E2F4 E2F4 E2F4 E2F4 6274 5980 E2F4 78AA      E7A2AA       E7A2AA       F3B1A4B1
E1F3     E1F3 E1F3 E1F3 E1F3 5FBE E2F5 E2F5 E2F5 E2F5 6275 598B E2F5 78AF      E7A2AF       E7A2AF       F3B1A4B2
E1F4     E1F4 E1F4 E1F4 E1F4 5FBF E2F6 E2F6 E2F6 E2F6 6276 599B E2F6 78D1      E7A391       E7A391       F3B1A4B3
E1F5     E1F5 E1F5 E1F5 E1F5 5FC0 E2F7 E2F7 E2F7 E2F7 6277 599C E2F7 78C6      E7A386       E7A386       F3B1A4B4
E1F6     E1F6 E1F6 E1F6 E1F6 5FC1 E2F8 E2F8 E2F8 E2F8 6278 59A0 E2F8 78CB      E7A38B       E7A38B       F3B1A4B5
E1F7     E1F7 E1F7 E1F7 E1F7 5FC2 E2F9 E2F9 E2F9 E2F9 6279 59AB E2F9 78D4      E7A394       E7A394       F3B1A4B6
E1F8     E1F8 E1F8 E1F8 E1F8 5FC3 E2FA E2FA E2FA E2FA 627A 59B0 E2FA 78BE      E7A2BE       E7A2BE       F3B1A4B7
E1F9     E1F9 E1F9 E1F9 E1F9 5FC4 E2FB E2FB E2FB E2FB 627B 59C0 E2FB 78BC      E7A2BC       E7A2BC       F3B1A4B8
E1FA     E1FA E1FA E1FA E1FA 5FC5 E2FC E2FC E2FC E2FC 627C 596A E2FC 78C5      E7A385       E7A385       F3B1A4B9
E1FB     E1FB E1FB E1FB E1FB 5FC6 E2FD E2FD E2FD E2FD 627D 59D0 E2FD 78CA      E7A38A       E7A38A       F3B1A4BA
E1FC     E1FC E1FC E1FC E1FC 5FC7 E2FE E2FE E2FE E2FE 627E 59A1 E2FE 78EC      E7A3AC       E7A3AC       F3B1A4BB
E240     E240 E240 E240 E240 5FC8 E3A1 E3A1 E3A1 E3A1 6321 624F E3A1 78E7      E7A3A7       E7A3A7       F3B1A4BC
E241     E241 E241 E241 E241 5FC9 E3A2 E3A2 E3A2 E3A2 6322 627F E3A2 78DA      E7A39A       E7A39A       F3B1A4BD
E242     E242 E242 E242 E242 5FCA E3A3 E3A3 E3A3 E3A3 6323 627B E3A3 78FD      E7A3BD       E7A3BD       F3B1A4BE
E243     E243 E243 E243 E243 5FCB E3A4 E3A4 E3A4 E3A4 6324 62E0 E3A4 78F4      E7A3B4       E7A3B4       F3B1A4BF
E244     E244 E244 E244 E244 5FCC E3A5 E3A5 E3A5 E3A5 6325 626C E3A5 7907      E7A487       E7A487       F3B1A580
E245     E245 E245 E245 E245 4DF0 E3A6 E3A6 E3A6 E3A6 6326 6250 E3A6 7912      E7A492       E7A492       F3B1A581
E246     E246 E246 E246 E246 5FCD E3A7 E3A7 E3A7 E3A7 6327 627D E3A7 7911      E7A491       E7A491       F3B1A582
E247     E247 E247 E247 E247 5FCE E3A8 E3A8 E3A8 E3A8 6328 624D E3A8 7919      E7A499       E7A499       F3B1A583
E248     E248 E248 E248 E248 5FCF E3A9 E3A9 E3A9 E3A9 6329 625D E3A9 792C      E7A4AC       E7A4AC       F3B1A584
E249     E249 E249 E249 E249 5FD0 E3AA E3AA E3AA E3AA 632A 625C E3AA 792B      E7A4AB       E7A4AB       F3B1A585
E24A     E24A E24A E24A E24A 5FD3 E3AB E3AB E3AB E3AB 632B 624E E3AB 7940      E7A580       E7A580       F3B1A586
E24B     E24B E24B E24B E24B 5FD4 E3AC E3AC E3AC E3AC 632C 626B E3AC 7960      E7A5A0       E7A5A0       F3B1A587
E24C     E24C E24C E24C E24C 5365 E3AD E3AD E3AD E3AD 632D 6260 E3AD 7957      E7A597       E7A597       F3B1A588
E24D     E24D E24D E24D E24D 5FD6 E3AE E3AE E3AE E3AE 632E 624B E3AE 795F      E7A59F       E7A59F       F3B1A589
E24E     E24E E24E E24E E24E 5FD7 E3AF E3AF E3AF E3AF 632F 6261 E3AF 795A      E7A59A       E7A59A       F3B1A58A
E24F     E24F E24F E24F E24F 5FD8 E3B0 E3B0 E3B0 E3B0 6330 62F0 E3B0 7955      E7A595       E7A595       F3B1A58B
E250     E250 E250 E250 E250 555C E3B1 E3B1 E3B1 E3B1 6331 62F1 E3B1 7953      E7A593       E7A593       F3B1A58C
E251     E251 E251 E251 E251 52D8 E3B2 E3B2 E3B2 E3B2 6332 62F2 E3B2 797A      E7A5BA       E7A5BA       F3B1A58D
E252     E252 E252 E252 E252 5FDA E3B3 E3B3 E3B3 E3B3 6333 62F3 E3B3 797F      E7A5BF       E7A5BF       F3B1A58E
E253     E253 E253 E253 E253 5FDB E3B4 E3B4 E3B4 E3B4 6334 62F4 E3B4 798A      E7A68A       E7A68A       F3B1A58F
E254     E254 E254 E254 E254 5FDE E3B5 E3B5 E3B5 E3B5 6335 62F5 E3B5 799D      E7A69D       E7A69D       F3B1A590
E255     E255 E255 E255 E255 4FD7 E3B6 E3B6 E3B6 E3B6 6336 62F6 E3B6 79A7      E7A6A7       E7A6A7       F3B1A591
E256     E256 E256 E256 E256 51F5 E3B7 E3B7 E3B7 E3B7 6337 62F7 E3B7 9F4B      E9BD8B       E9BD8B       F3B1A592
E257     E257 E257 E257 E257 5FE0 E3B8 E3B8 E3B8 E3B8 6338 62F8 E3B8 79AA      E7A6AA       E7A6AA       F3B1A593
E258     E258 E258 E258 E258 54F5 E3B9 E3B9 E3B9 E3B9 6339 62F9 E3B9 79AE      E7A6AE       E7A6AE       F3B1A594
E259     E259 E259 E259 E259 5FE1 E3BA E3BA E3BA E3BA 633A 627A E3BA 79B3      E7A6B3       E7A6B3       F3B1A595
E25A     E25A E25A E25A E25A 536A E3BB E3BB E3BB E3BB 633B 625E E3BB 79B9      E7A6B9       E7A6B9       F3B1A596
E25B     E25B E25B E25B E25B 5FE2 E3BC E3BC E3BC E3BC 633C 624C E3BC 79BA      E7A6BA       E7A6BA       F3B1A597
E25C     E25C E25C E25C E25C 5FE3 E3BD E3BD E3BD E3BD 633D 627E E3BD 79C9      E7A789       E7A789       F3B1A598
E25D     E25D E25D E25D E25D 5FE4 E3BE E3BE E3BE E3BE 633E 626E E3BE 79D5      E7A795       E7A795       F3B1A599
E25E     E25E E25E E25E E25E 5FE5 E3BF E3BF E3BF E3BF 633F 626F E3BF 79E7      E7A7A7       E7A7A7       F3B1A59A
E25F     E25F E25F E25F E25F 5FE6 E3C0 E3C0 E3C0 E3C0 6340 627C E3C0 79EC      E7A7AC       E7A7AC       F3B1A59B
E260     E260 E260 E260 E260 54B5 E3C1 E3C1 E3C1 E3C1 6341 62C1 E3C1 79E1      E7A7A1       E7A7A1       F3B1A59C
E261     E261 E261 E261 E261 5FE7 E3C2 E3C2 E3C2 E3C2 6342 62C2 E3C2 79E3      E7A7A3       E7A7A3       F3B1A59D
E262     E262 E262 E262 E262 5FE8 E3C3 E3C3 E3C3 E3C3 6343 62C3 E3C3 7A08      E7A888       E7A888       F3B1A59E
E263     E263 E263 E263 E263 5FE9 E3C4 E3C4 E3C4 E3C4 6344 62C4 E3C4 7A0D      E7A88D       E7A88D       F3B1A59F
E264     E264 E264 E264 E264 5FEA E3C5 E3C5 E3C5 E3C5 6345 62C5 E3C5 7A18      E7A898       E7A898       F3B1A5A0
E265     E265 E265 E265 E265 5FEB E3C6 E3C6 E3C6 E3C6 6346 62C6 E3C6 7A19      E7A899       E7A899       F3B1A5A1
E266     E266 E266 E266 E266 5366 E3C7 E3C7 E3C7 E3C7 6347 62C7 E3C7 7A20      E7A8A0       E7A8A0       F3B1A5A2
E267     E267 E267 E267 E267 5FEC E3C8 E3C8 E3C8 E3C8 6348 62C8 E3C8 7A1F      E7A89F       E7A89F       F3B1A5A3
E268     E268 E268 E268 E268 50E7 E3C9 E3C9 E3C9 E3C9 6349 62C9 E3C9 7980      E7A680       E7A680       F3B1A5A4
E269     E269 E269 E269 E269 5FED E3CA E3CA E3CA E3CA 634A 62D1 E3CA 7A31      E7A8B1       E7A8B1       F3B1A5A5
E26A     E26A E26A E26A E26A 5FF0 E3CB E3CB E3CB E3CB 634B 62D2 E3CB 7A3B      E7A8BB       E7A8BB       F3B1A5A6
E26B     E26B E26B E26B E26B 5FEE E3CC E3CC E3CC E3CC 634C 62D3 E3CC 7A3E      E7A8BE       E7A8BE       F3B1A5A7
E26C     E26C E26C E26C E26C 5FEF E3CD E3CD E3CD E3CD 634D 62D4 E3CD 7A37      E7A8B7       E7A8B7       F3B1A5A8
E26D     E26D E26D E26D E26D 5FF1 E3CE E3CE E3CE E3CE 634E 62D5 E3CE 7A43      E7A983       E7A983       F3B1A5A9
E26E     E26E E26E E26E E26E 5FF2 E3CF E3CF E3CF E3CF 634F 62D6 E3CF 7A57      E7A997       E7A997       F3B1A5AA
E26F     E26F E26F E26F E26F 5FF3 E3D0 E3D0 E3D0 E3D0 6350 62D7 E3D0 7A49      E7A989       E7A989       F3B1A5AB
E270     E270 E270 E270 E270 5FF5 E3D1 E3D1 E3D1 E3D1 6351 62D8 E3D1 7A61      E7A9A1       E7A9A1       F3B1A5AC
E271     E271 E271 E271 E271 5FF4 E3D2 E3D2 E3D2 E3D2 6352 62D9 E3D2 7A62      E7A9A2       E7A9A2       F3B1A5AD
E272     E272 E272 E272 E272 5FF6 E3D3 E3D3 E3D3 E3D3 6353 62E2 E3D3 7A69      E7A9A9       E7A9A9       F3B1A5AE
E273     E273 E273 E273 E273 5FF7 E3D4 E3D4 E3D4 E3D4 6354 62E3 E3D4 9F9D      E9BE9D       E9BE9D       F3B1A5AF
E274     E274 E274 E274 E274 5FF8 E3D5 E3D5 E3D5 E3D5 6355 62E4 E3D5 7A70      E7A9B0       E7A9B0       F3B1A5B0
E275     E275 E275 E275 E275 5FF9 E3D6 E3D6 E3D6 E3D6 6356 62E5 E3D6 7A79      E7A9B9       E7A9B9       F3B1A5B1
E276     E276 E276 E276 E276 5FFA E3D7 E3D7 E3D7 E3D7 6357 62E6 E3D7 7A7D      E7A9BD       E7A9BD       F3B1A5B2
E277     E277 E277 E277 E277 5FFB E3D8 E3D8 E3D8 E3D8 6358 62E7 E3D8 7A88      E7AA88       E7AA88       F3B1A5B3
E278     E278 E278 E278 E278 6041 E3D9 E3D9 E3D9 E3D9 6359 62E8 E3D9 7A97      E7AA97       E7AA97       F3B1A5B4
E279     E279 E279 E279 E279 5FFC E3DA E3DA E3DA E3DA 635A 62E9 E3DA 7A95      E7AA95       E7AA95       F3B1A5B5
E27A     E27A E27A E27A E27A 5FFD E3DB E3DB E3DB E3DB 635B 624A E3DB 7A98      E7AA98       E7AA98       F3B1A5B6
E27B     E27B E27B E27B E27B 5FFE E3DC E3DC E3DC E3DC 635C 625B E3DC 7A96      E7AA96       E7AA96       F3B1A5B7
E27C     E27C E27C E27C E27C 6042 E3DD E3DD E3DD E3DD 635D 625A E3DD 7AA9      E7AAA9       E7AAA9       F3B1A5B8
E27D     E27D E27D E27D E27D 52F0 B3F6 E3DE E3DE B3F6 3376 F39B E3DE 7AC8      E7AB88       E7AB88       F3B1A5B9
E27E     E27E E27E E27E E27E 6043 E3DF E3DF E3DF E3DF 635F 626D E3DF 7AB0      E7AAB0       E7AAB0       F3B1A5BA
E280     E280 E280 E280 E280 6044 E3E0 E3E0 E3E0 E3E0 6360 6279 E3E0 7AB6      E7AAB6       E7AAB6       F3B1A5BB
E281     E281 E281 E281 E281 6048 E3E1 E3E1 E3E1 E3E1 6361 6257 E3E1 7AC5      E7AB85       E7AB85       F3B1A5BC
E282     E282 E282 E282 E282 6049 E3E2 E3E2 E3E2 E3E2 6362 6259 E3E2 7AC4      E7AB84       E7AB84       F3B1A5BD
E283     E283 E283 E283 E283 6047 E3E3 E3E3 E3E3 E3E3 6363 6262 E3E3 7ABF      E7AABF       E7AABF       F3B1A5BE
E284     E284 E284 E284 E284 6045 E3E4 E3E4 E3E4 E3E4 6364 6263 E3E4 9083      E98283       E98283       F3B1A5BF
E285     E285 E285 E285 E285 604A E3E5 E3E5 E3E5 E3E5 6365 6264 E3E5 7AC7      E7AB87       E7AB87       F3B1A680
E286     E286 E286 E286 E286 604B E3E6 E3E6 E3E6 E3E6 6366 6265 E3E6 7ACA      E7AB8A       E7AB8A       F3B1A681
E287     E287 E287 E287 E287 604C E3E7 E3E7 E3E7 E3E7 6367 6266 E3E7 7ACD      E7AB8D       E7AB8D       F3B1A682
E288     E288 E288 E288 E288 604D E3E8 E3E8 E3E8 E3E8 6368 6267 E3E8 7ACF      E7AB8F       E7AB8F       F3B1A683
E289     E289 E289 E289 E289 6050 E3E9 E3E9 E3E9 E3E9 6369 6268 E3E9 7AD5      E7AB95       E7AB95       F3B1A684
E28A     E28A E28A E28A E28A 6051 E3EA E3EA E3EA E3EA 636A 6269 E3EA 7AD3      E7AB93       E7AB93       F3B1A685
E28B     E28B E28B E28B E28B 6052 E3EB E3EB E3EB E3EB 636B 6270 E3EB 7AD9      E7AB99       E7AB99       F3B1A686
E28C     E28C E28C E28C E28C 6053 E3EC E3EC E3EC E3EC 636C 6271 E3EC 7ADA      E7AB9A       E7AB9A       F3B1A687
E28D     E28D E28D E28D E28D 6054 E3ED E3ED E3ED E3ED 636D 6272 E3ED 7ADD      E7AB9D       E7AB9D       F3B1A688
E28E     E28E E28E E28E E28E 6055 E3EE E3EE E3EE E3EE 636E 6273 E3EE 7AE1      E7ABA1       E7ABA1       F3B1A689
E28F     E28F E28F E28F E28F 6056 E3EF E3EF E3EF E3EF 636F 6274 E3EF 7AE2      E7ABA2       E7ABA2       F3B1A68A
E290     E290 E290 E290 E290 6057 E3F0 E3F0 E3F0 E3F0 6370 6275 E3F0 7AE6      E7ABA6       E7ABA6       F3B1A68B
E291     E291 E291 E291 E291 605B E3F1 E3F1 E3F1 E3F1 6371 6276 E3F1 7AED      E7ABAD       E7ABAD       F3B1A68C
E292     E292 E292 E292 E292 605C E3F2 E3F2 E3F2 E3F2 6372 6277 E3F2 7AF0      E7ABB0       E7ABB0       F3B1A68D
E293     E293 E293 E293 E293 605E E3F3 E3F3 E3F3 E3F3 6373 6278 E3F3 7B02      E7AC82       E7AC82       F3B1A68E
E294     E294 E294 E294 E294 605F E3F4 E3F4 E3F4 E3F4 6374 6280 E3F4 7B0F      E7AC8F       E7AC8F       F3B1A68F
E295     E295 E295 E295 E295 6061 E3F5 E3F5 E3F5 E3F5 6375 628B E3F5 7B0A      E7AC8A       E7AC8A       F3B1A690
E296     E296 E296 E296 E296 6062 E3F6 E3F6 E3F6 E3F6 6376 629B E3F6 7B06      E7AC86       E7AC86       F3B1A691
E297     E297 E297 E297 E297 6063 E3F7 E3F7 E3F7 E3F7 6377 629C E3F7 7B33      E7ACB3       E7ACB3       F3B1A692
E298     E298 E298 E298 E298 53E7 E3F8 E3F8 E3F8 E3F8 6378 62A0 E3F8 7B18      E7AC98       E7AC98       F3B1A693
E299     E299 E299 E299 E299 6065 E3F9 E3F9 E3F9 E3F9 6379 62AB E3F9 7B19      E7AC99       E7AC99       F3B1A694
E29A     E29A E29A E29A E29A 6066 E3FA E3FA E3FA E3FA 637A 62B0 E3FA 7B1E      E7AC9E       E7AC9E       F3B1A695
E29B     E29B E29B E29B E29B 6067 E3FB E3FB E3FB E3FB 637B 62C0 E3FB 7B35      E7ACB5       E7ACB5       F3B1A696
E29C     E29C E29C E29C E29C 6068 E3FC E3FC E3FC E3FC 637C 626A E3FC 7B28      E7ACA8       E7ACA8       F3B1A697
E29D     E29D E29D E29D E29D 6064 E3FD E3FD E3FD E3FD 637D 62D0 E3FD 7B36      E7ACB6       E7ACB6       F3B1A698
E29E     E29E E29E E29E E29E 6069 E3FE E3FE E3FE E3FE 637E 62A1 E3FE 7B50      E7AD90       E7AD90       F3B1A699
E29F     E29F E29F E29F E29F 606F E4A1 E4A1 E4A1 E4A1 6421 634F E4A1 7B7A      E7ADBA       E7ADBA       F3B1A69A
E2A0     E2A0 E2A0 E2A0 E2A0 54D5 E4A2 E4A2 E4A2 E4A2 6422 637F E4A2 7B04      E7AC84       E7AC84       F3B1A69B
E2A1     E2A1 E2A1 E2A1 E2A1 606A E4A3 E4A3 E4A3 E4A3 6423 637B E4A3 7B4D      E7AD8D       E7AD8D       F3B1A69C
E2A2     E2A2 E2A2 E2A2 E2A2 6060 E4A4 E4A4 E4A4 E4A4 6424 63E0 E4A4 7B0B      E7AC8B       E7AC8B       F3B1A69D
E2A3     E2A3 E2A3 E2A3 E2A3 606B E4A5 E4A5 E4A5 E4A5 6425 636C E4A5 7B4C      E7AD8C       E7AD8C       F3B1A69E
E2A4     E2A4 E2A4 E2A4 E2A4 606C E4A6 E4A6 E4A6 E4A6 6426 6350 E4A6 7B45      E7AD85       E7AD85       F3B1A69F
E2A5     E2A5 E2A5 E2A5 E2A5 606E E4A7 E4A7 E4A7 E4A7 6427 637D E4A7 7B75      E7ADB5       E7ADB5       F3B1A6A0
E2A6     E2A6 E2A6 E2A6 E2A6 5484 E4A8 E4A8 E4A8 E4A8 6428 634D E4A8 7B65      E7ADA5       E7ADA5       F3B1A6A1
E2A7     E2A7 E2A7 E2A7 E2A7 6070 E4A9 E4A9 E4A9 E4A9 6429 635D E4A9 7B74      E7ADB4       E7ADB4       F3B1A6A2
E2A8     E2A8 E2A8 E2A8 E2A8 5048 E4AA E4AA E4AA E4AA 642A 635C E4AA 7B67      E7ADA7       E7ADA7       F3B1A6A3
E2A9     E2A9 E2A9 E2A9 E2A9 6071 E4AB E4AB E4AB E4AB 642B 634E E4AB 7B70      E7ADB0       E7ADB0       F3B1A6A4
E2AA     E2AA E2AA E2AA E2AA 6072 E4AC E4AC E4AC E4AC 642C 636B E4AC 7B71      E7ADB1       E7ADB1       F3B1A6A5
E2AB     E2AB E2AB E2AB E2AB 534E E4AD E4AD E4AD E4AD 642D 6360 E4AD 7B6C      E7ADAC       E7ADAC       F3B1A6A6
E2AC     E2AC E2AC E2AC E2AC 6073 E4AE E4AE E4AE E4AE 642E 634B E4AE 7B6E      E7ADAE       E7ADAE       F3B1A6A7
E2AD     E2AD E2AD E2AD E2AD 6074 E4AF E4AF E4AF E4AF 642F 6361 E4AF 7B9D      E7AE9D       E7AE9D       F3B1A6A8
E2AE     E2AE E2AE E2AE E2AE 6075 E4B0 E4B0 E4B0 E4B0 6430 63F0 E4B0 7B98      E7AE98       E7AE98       F3B1A6A9
E2AF     E2AF E2AF E2AF E2AF 6076 E4B1 E4B1 E4B1 E4B1 6431 63F1 E4B1 7B9F      E7AE9F       E7AE9F       F3B1A6AA
E2B0     E2B0 E2B0 E2B0 E2B0 6077 E4B2 E4B2 E4B2 E4B2 6432 63F2 E4B2 7B8D      E7AE8D       E7AE8D       F3B1A6AB
E2B1     E2B1 E2B1 E2B1 E2B1 6078 E4B3 E4B3 E4B3 E4B3 6433 63F3 E4B3 7B9C      E7AE9C       E7AE9C       F3B1A6AC
E2B2     E2B2 E2B2 E2B2 E2B2 6079 E4B4 E4B4 E4B4 E4B4 6434 63F4 E4B4 7B9A      E7AE9A       E7AE9A       F3B1A6AD
E2B3     E2B3 E2B3 E2B3 E2B3 55A4 E4B5 E4B5 E4B5 E4B5 6435 63F5 E4B5 7B8B      E7AE8B       E7AE8B       F3B1A6AE
E2B4     E2B4 E2B4 E2B4 E2B4 607A E4B6 E4B6 E4B6 E4B6 6436 63F6 E4B6 7B92      E7AE92       E7AE92       F3B1A6AF
E2B5     E2B5 E2B5 E2B5 E2B5 607B E4B7 E4B7 E4B7 E4B7 6437 63F7 E4B7 7B8F      E7AE8F       E7AE8F       F3B1A6B0
E2B6     E2B6 E2B6 E2B6 E2B6 606D E4B8 E4B8 E4B8 E4B8 6438 63F8 E4B8 7B5D      E7AD9D       E7AD9D       F3B1A6B1
E2B7     E2B7 E2B7 E2B7 E2B7 607C E4B9 E4B9 E4B9 E4B9 6439 63F9 E4B9 7B99      E7AE99       E7AE99       F3B1A6B2
E2B8     E2B8 E2B8 E2B8 E2B8 607E E4BA E4BA E4BA E4BA 643A 637A E4BA 7BCB      E7AF8B       E7AF8B       F3B1A6B3
E2B9     E2B9 E2B9 E2B9 E2B9 51E8 E4BB E4BB E4BB E4BB 643B 635E E4BB 7BC1      E7AF81       E7AF81       F3B1A6B4
E2BA     E2BA E2BA E2BA E2BA 607F E4BC E4BC E4BC E4BC 643C 634C E4BC 7BCC      E7AF8C       E7AF8C       F3B1A6B5
E2BB     E2BB E2BB E2BB E2BB 607D E4BD E4BD E4BD E4BD 643D 637E E4BD 7BCF      E7AF8F       E7AF8F       F3B1A6B6
E2BC     E2BC E2BC E2BC E2BC 6080 E4BE E4BE E4BE E4BE 643E 636E E4BE 7BB4      E7AEB4       E7AEB4       F3B1A6B7
E2BD     E2BD E2BD E2BD E2BD 6081 E4BF E4BF E4BF E4BF 643F 636F E4BF 7BC6      E7AF86       E7AF86       F3B1A6B8
E2BE     E2BE E2BE E2BE E2BE 6083 E4C0 E4C0 E4C0 E4C0 6440 637C E4C0 7BDD      E7AF9D       E7AF9D       F3B1A6B9
E2BF     E2BF E2BF E2BF E2BF 6084 E4C1 E4C1 E4C1 E4C1 6441 63C1 E4C1 7BE9      E7AFA9       E7AFA9       F3B1A6BA
E2C0     E2C0 E2C0 E2C0 E2C0 4F53 E4C2 E4C2 E4C2 E4C2 6442 63C2 E4C2 7C11      E7B091       E7B091       F3B1A6BB
E2C1     E2C1 E2C1 E2C1 E2C1 6088 E4C3 E4C3 E4C3 E4C3 6443 63C3 E4C3 7C14      E7B094       E7B094       F3B1A6BC
E2C2     E2C2 E2C2 E2C2 E2C2 6085 E4C4 E4C4 E4C4 E4C4 6444 63C4 E4C4 7BE6      E7AFA6       E7AFA6       F3B1A6BD
E2C3     E2C3 E2C3 E2C3 E2C3 6087 E4C5 E4C5 E4C5 E4C5 6445 63C5 E4C5 7BE5      E7AFA5       E7AFA5       F3B1A6BE
E2C4     E2C4 E2C4 E2C4 E2C4 4CEE CFB6 E4C6 E4C6 CFB6 4F36 D6F6 E4C6 7C60      E7B1A0       E7B1A0       F3B1A6BF
E2C5     E2C5 E2C5 E2C5 E2C5 6089 E4C7 E4C7 E4C7 E4C7 6447 63C7 E4C7 7C00      E7B080       E7B080       F3B1A780
E2C6     E2C6 E2C6 E2C6 E2C6 608B E4C8 E4C8 E4C8 E4C8 6448 63C8 E4C8 7C07      E7B087       E7B087       F3B1A781
E2C7     E2C7 E2C7 E2C7 E2C7 608A E4C9 E4C9 E4C9 E4C9 6449 63C9 E4C9 7C13      E7B093       E7B093       F3B1A782
E2C8     E2C8 E2C8 E2C8 E2C8 608C E4CA E4CA E4CA E4CA 644A 63D1 E4CA 7BF3      E7AFB3       E7AFB3       F3B1A783
E2C9     E2C9 E2C9 E2C9 E2C9 6086 E4CB E4CB E4CB E4CB 644B 63D2 E4CB 7BF7      E7AFB7       E7AFB7       F3B1A784
E2CA     E2CA E2CA E2CA E2CA 517C E4CC E4CC E4CC E4CC 644C 63D3 E4CC 7C17      E7B097       E7B097       F3B1A785
E2CB     E2CB E2CB E2CB E2CB 608D E4CD E4CD E4CD E4CD 644D 63D4 E4CD 7C0D      E7B08D       E7B08D       F3B1A786
E2CC     E2CC E2CC E2CC E2CC 608E E4CE E4CE E4CE E4CE 644E 63D5 E4CE 7BF6      E7AFB6       E7AFB6       F3B1A787
E2CD     E2CD E2CD E2CD E2CD 608F E4CF E4CF E4CF E4CF 644F 63D6 E4CF 7C23      E7B0A3       E7B0A3       F3B1A788
E2CE     E2CE E2CE E2CE E2CE 6090 E4D0 E4D0 E4D0 E4D0 6450 63D7 E4D0 7C27      E7B0A7       E7B0A7       F3B1A789
E2CF     E2CF E2CF E2CF E2CF 6091 E4D1 E4D1 E4D1 E4D1 6451 63D8 E4D1 7C2A      E7B0AA       E7B0AA       F3B1A78A
E2D0     E2D0 E2D0 E2D0 E2D0 6092 E4D2 E4D2 E4D2 E4D2 6452 63D9 E4D2 7C1F      E7B09F       E7B09F       F3B1A78B
E2D1     E2D1 E2D1 E2D1 E2D1 6093 E4D3 E4D3 E4D3 E4D3 6453 63E2 E4D3 7C37      E7B0B7       E7B0B7       F3B1A78C
E2D2     E2D2 E2D2 E2D2 E2D2 6094 E4D4 E4D4 E4D4 E4D4 6454 63E3 E4D4 7C2B      E7B0AB       E7B0AB       F3B1A78D
E2D3     E2D3 E2D3 E2D3 E2D3 6095 E4D5 E4D5 E4D5 E4D5 6455 63E4 E4D5 7C3D      E7B0BD       E7B0BD       F3B1A78E
E2D4     E2D4 E2D4 E2D4 E2D4 6097 E4D6 E4D6 E4D6 E4D6 6456 63E5 E4D6 7C4C      E7B18C       E7B18C       F3B1A78F
E2D5     E2D5 E2D5 E2D5 E2D5 6098 E4D7 E4D7 E4D7 E4D7 6457 63E6 E4D7 7C43      E7B183       E7B183       F3B1A790
E2D6     E2D6 E2D6 E2D6 E2D6 4F99 E4D8 E4D8 E4D8 E4D8 6458 63E7 E4D8 7C54      E7B194       E7B194       F3B1A791
E2D7     E2D7 E2D7 E2D7 E2D7 51F2 E4D9 E4D9 E4D9 E4D9 6459 63E8 E4D9 7C4F      E7B18F       E7B18F       F3B1A792
E2D8     E2D8 E2D8 E2D8 E2D8 6096 E4DA E4DA E4DA E4DA 645A 63E9 E4DA 7C40      E7B180       E7B180       F3B1A793
E2D9     E2D9 E2D9 E2D9 E2D9 609A E4DB E4DB E4DB E4DB 645B 634A E4DB 7C50      E7B190       E7B190       F3B1A794
E2DA     E2DA E2DA E2DA E2DA 609B E4DC E4DC E4DC E4DC 645C 635B E4DC 7C58      E7B198       E7B198       F3B1A795
E2DB     E2DB E2DB E2DB E2DB 609C E4DD E4DD E4DD E4DD 645D 635A E4DD 7C5F      E7B19F       E7B19F       F3B1A796
E2DC     E2DC E2DC E2DC E2DC 52AA E4DE E4DE E4DE E4DE 645E 635F E4DE 7C64      E7B1A4       E7B1A4       F3B1A797
E2DD     E2DD E2DD E2DD E2DD 6099 E4DF E4DF E4DF E4DF 645F 636D E4DF 7C56      E7B196       E7B196       F3B1A798
E2DE     E2DE E2DE E2DE E2DE 609D E4E0 E4E0 E4E0 E4E0 6460 6379 E4E0 7C65      E7B1A5       E7B1A5       F3B1A799
E2DF     E2DF E2DF E2DF E2DF 609E E4E1 E4E1 E4E1 E4E1 6461 6357 E4E1 7C6C      E7B1AC       E7B1AC       F3B1A79A
E2E0     E2E0 E2E0 E2E0 E2E0 609F E4E2 E4E2 E4E2 E4E2 6462 6359 E4E2 7C75      E7B1B5       E7B1B5       F3B1A79B
E2E1     E2E1 E2E1 E2E1 E2E1 60A0 E4E3 E4E3 E4E3 E4E3 6463 6362 E4E3 7C83      E7B283       E7B283       F3B1A79C
E2E2     E2E2 E2E2 E2E2 E2E2 60A1 E4E4 E4E4 E4E4 E4E4 6464 6363 E4E4 7C90      E7B290       E7B290       F3B1A79D
E2E3     E2E3 E2E3 E2E3 E2E3 60A2 E4E5 E4E5 E4E5 E4E5 6465 6364 E4E5 7CA4      E7B2A4       E7B2A4       F3B1A79E
E2E4     E2E4 E2E4 E2E4 E2E4 60A6 E4E6 E4E6 E4E6 E4E6 6466 6365 E4E6 7CAD      E7B2AD       E7B2AD       F3B1A79F
E2E5     E2E5 E2E5 E2E5 E2E5 60A3 E4E7 E4E7 E4E7 E4E7 6467 6366 E4E7 7CA2      E7B2A2       E7B2A2       F3B1A7A0
E2E6     E2E6 E2E6 E2E6 E2E6 60A4 E4E8 E4E8 E4E8 E4E8 6468 6367 E4E8 7CAB      E7B2AB       E7B2AB       F3B1A7A1
E2E7     E2E7 E2E7 E2E7 E2E7 60A5 E4E9 E4E9 E4E9 E4E9 6469 6368 E4E9 7CA1      E7B2A1       E7B2A1       F3B1A7A2
E2E8     E2E8 E2E8 E2E8 E2E8 60A7 E4EA E4EA E4EA E4EA 646A 6369 E4EA 7CA8      E7B2A8       E7B2A8       F3B1A7A3
E2E9     E2E9 E2E9 E2E9 E2E9 60A8 E4EB E4EB E4EB E4EB 646B 6370 E4EB 7CB3      E7B2B3       E7B2B3       F3B1A7A4
E2EA     E2EA E2EA E2EA E2EA 60A9 E4EC E4EC E4EC E4EC 646C 6371 E4EC 7CB2      E7B2B2       E7B2B2       F3B1A7A5
E2EB     E2EB E2EB E2EB E2EB 60AA E4ED E4ED E4ED E4ED 646D 6372 E4ED 7CB1      E7B2B1       E7B2B1       F3B1A7A6
E2EC     E2EC E2EC E2EC E2EC 60AB E4EE E4EE E4EE E4EE 646E 6373 E4EE 7CAE      E7B2AE       E7B2AE       F3B1A7A7
E2ED     E2ED E2ED E2ED E2ED 60AC E4EF E4EF E4EF E4EF 646F 6374 E4EF 7CB9      E7B2B9       E7B2B9       F3B1A7A8
E2EE     E2EE E2EE E2EE E2EE 60AE E4F0 E4F0 E4F0 E4F0 6470 6375 E4F0 7CBD      E7B2BD       E7B2BD       F3B1A7A9
E2EF     E2EF E2EF E2EF E2EF 51BC E4F1 E4F1 E4F1 E4F1 6471 6376 E4F1 7CC0      E7B380       E7B380       F3B1A7AA
E2F0     E2F0 E2F0 E2F0 E2F0 60AF E4F2 E4F2 E4F2 E4F2 6472 6377 E4F2 7CC5      E7B385       E7B385       F3B1A7AB
E2F1     E2F1 E2F1 E2F1 E2F1 60B0 E4F3 E4F3 E4F3 E4F3 6473 6378 E4F3 7CC2      E7B382       E7B382       F3B1A7AC
E2F2     E2F2 E2F2 E2F2 E2F2 60B3 E4F4 E4F4 E4F4 E4F4 6474 6380 E4F4 7CD8      E7B398       E7B398       F3B1A7AD
E2F3     E2F3 E2F3 E2F3 E2F3 60B1 E4F5 E4F5 E4F5 E4F5 6475 638B E4F5 7CD2      E7B392       E7B392       F3B1A7AE
E2F4     E2F4 E2F4 E2F4 E2F4 60B4 E4F6 E4F6 E4F6 E4F6 6476 639B E4F6 7CDC      E7B39C       E7B39C       F3B1A7AF
E2F5     E2F5 E2F5 E2F5 E2F5 60B2 E4F7 E4F7 E4F7 E4F7 6477 639C E4F7 7CE2      E7B3A2       E7B3A2       F3B1A7B0
E2F6     E2F6 E2F6 E2F6 E2F6 6788 E4F8 E4F8 E4F8 E4F8 6478 63A0 E4F8 9B3B      E9ACBB       E9ACBB       F3B1A7B1
E2F7     E2F7 E2F7 E2F7 E2F7 60B5 E4F9 E4F9 E4F9 E4F9 6479 63AB E4F9 7CEF      E7B3AF       E7B3AF       F3B1A7B2
E2F8     E2F8 E2F8 E2F8 E2F8 60B6 E4FA E4FA E4FA E4FA 647A 63B0 E4FA 7CF2      E7B3B2       E7B3B2       F3B1A7B3
E2F9     E2F9 E2F9 E2F9 E2F9 60B7 E4FB E4FB E4FB E4FB 647B 63C0 E4FB 7CF4      E7B3B4       E7B3B4       F3B1A7B4
E2FA     E2FA E2FA E2FA E2FA 60B8 E4FC E4FC E4FC E4FC 647C 636A E4FC 7CF6      E7B3B6       E7B3B6       F3B1A7B5
E2FB     E2FB E2FB E2FB E2FB 52C2 E4FD E4FD E4FD E4FD 647D 63D0 E4FD 7CFA      E7B3BA       E7B3BA       F3B1A7B6
E2FC     E2FC E2FC E2FC E2FC 60B9 E4FE E4FE E4FE E4FE 647E 63A1 E4FE 7D06      E7B486       E7B486       F3B1A7B7
E340     E340 E340 E340 E340 60BA E5A1 E5A1 E5A1 E5A1 6521 644F E5A1 7D02      E7B482       E7B482       F3B1A7B8
E341     E341 E341 E341 E341 60BB E5A2 E5A2 E5A2 E5A2 6522 647F E5A2 7D1C      E7B49C       E7B49C       F3B1A7B9
E342     E342 E342 E342 E342 60BC E5A3 E5A3 E5A3 E5A3 6523 647B E5A3 7D15      E7B495       E7B495       F3B1A7BA
E343     E343 E343 E343 E343 60BD E5A4 E5A4 E5A4 E5A4 6524 64E0 E5A4 7D0A      E7B48A       E7B48A       F3B1A7BB
E344     E344 E344 E344 E344 60BE E5A5 E5A5 E5A5 E5A5 6525 646C E5A5 7D45      E7B585       E7B585       F3B1A7BC
E345     E345 E345 E345 E345 60BF E5A6 E5A6 E5A6 E5A6 6526 6450 E5A6 7D4B      E7B58B       E7B58B       F3B1A7BD
E346     E346 E346 E346 E346 60C0 E5A7 E5A7 E5A7 E5A7 6527 647D E5A7 7D2E      E7B4AE       E7B4AE       F3B1A7BE
E347     E347 E347 E347 E347 60C1 E5A8 E5A8 E5A8 E5A8 6528 644D E5A8 7D32      E7B4B2       E7B4B2       F3B1A7BF
E348     E348 E348 E348 E348 60C2 E5A9 E5A9 E5A9 E5A9 6529 645D E5A9 7D3F      E7B4BF       E7B4BF       F3B1A880
E349     E349 E349 E349 E349 60C3 E5AA E5AA E5AA E5AA 652A 645C E5AA 7D35      E7B4B5       E7B4B5       F3B1A881
E34A     E34A E34A E34A E34A 60C5 E5AB E5AB E5AB E5AB 652B 644E E5AB 7D46      E7B586       E7B586       F3B1A882
E34B     E34B E34B E34B E34B 60C7 E5AC E5AC E5AC E5AC 652C 646B E5AC 7D73      E7B5B3       E7B5B3       F3B1A883
E34C     E34C E34C E34C E34C 60C8 E5AD E5AD E5AD E5AD 652D 6460 E5AD 7D56      E7B596       E7B596       F3B1A884
E34D     E34D E34D E34D E34D 60C9 E5AE E5AE E5AE E5AE 652E 644B E5AE 7D4E      E7B58E       E7B58E       F3B1A885
E34E     E34E E34E E34E E34E 5268 E5AF E5AF E5AF E5AF 652F 6461 E5AF 7D72      E7B5B2       E7B5B2       F3B1A886
E34F     E34F E34F E34F E34F 60CA E5B0 E5B0 E5B0 E5B0 6530 64F0 E5B0 7D68      E7B5A8       E7B5A8       F3B1A887
E350     E350 E350 E350 E350 60CB E5B1 E5B1 E5B1 E5B1 6531 64F1 E5B1 7D6E      E7B5AE       E7B5AE       F3B1A888
E351     E351 E351 E351 E351 60CC E5B2 E5B2 E5B2 E5B2 6532 64F2 E5B2 7D4F      E7B58F       E7B58F       F3B1A889
E352     E352 E352 E352 E352 60CD E5B3 E5B3 E5B3 E5B3 6533 64F3 E5B3 7D63      E7B5A3       E7B5A3       F3B1A88A
E353     E353 E353 E353 E353 60CE E5B4 E5B4 E5B4 E5B4 6534 64F4 E5B4 7D93      E7B693       E7B693       F3B1A88B
E354     E354 E354 E354 E354 60CF E5B5 E5B5 E5B5 E5B5 6535 64F5 E5B5 7D89      E7B689       E7B689       F3B1A88C
E355     E355 E355 E355 E355 60D0 E5B6 E5B6 E5B6 E5B6 6536 64F6 E5B6 7D5B      E7B59B       E7B59B       F3B1A88D
E356     E356 E356 E356 E356 5374 E5B7 E5B7 E5B7 E5B7 6537 64F7 E5B7 7D8F      E7B68F       E7B68F       F3B1A88E
E357     E357 E357 E357 E357 54ED E5B8 E5B8 E5B8 E5B8 6538 64F8 E5B8 7D7D      E7B5BD       E7B5BD       F3B1A88F
E358     E358 E358 E358 E358 54EB E5B9 E5B9 E5B9 E5B9 6539 64F9 E5B9 7D9B      E7B69B       E7B69B       F3B1A890
E359     E359 E359 E359 E359 54D3 E5BA E5BA E5BA E5BA 653A 647A E5BA 7DBA      E7B6BA       E7B6BA       F3B1A891
E35A     E35A E35A E35A E35A 60D1 E5BB E5BB E5BB E5BB 653B 645E E5BB 7DAE      E7B6AE       E7B6AE       F3B1A892
E35B     E35B E35B E35B E35B 60D2 E5BC E5BC E5BC E5BC 653C 644C E5BC 7DA3      E7B6A3       E7B6A3       F3B1A893
E35C     E35C E35C E35C E35C 60D3 E5BD E5BD E5BD E5BD 653D 647E E5BD 7DB5      E7B6B5       E7B6B5       F3B1A894
E35D     E35D E35D E35D E35D 60D5 E5BE E5BE E5BE E5BE 653E 646E E5BE 7DC7      E7B787       E7B787       F3B1A895
E35E     E35E E35E E35E E35E 60D6 E5BF E5BF E5BF E5BF 653F 646F E5BF 7DBD      E7B6BD       E7B6BD       F3B1A896
E35F     E35F E35F E35F E35F 60D7 E5C0 E5C0 E5C0 E5C0 6540 647C E5C0 7DAB      E7B6AB       E7B6AB       F3B1A897
E360     E360 E360 E360 E360 60F3 E5C1 E5C1 E5C1 E5C1 6541 64C1 E5C1 7E3D      E7B8BD       E7B8BD       F3B1A898
E361     E361 E361 E361 E361 60D8 E5C2 E5C2 E5C2 E5C2 6542 64C2 E5C2 7DA2      E7B6A2       E7B6A2       F3B1A899
E362     E362 E362 E362 E362 60D9 E5C3 E5C3 E5C3 E5C3 6543 64C3 E5C3 7DAF      E7B6AF       E7B6AF       F3B1A89A
E363     E363 E363 E363 E363 60E8 E5C4 E5C4 E5C4 E5C4 6544 64C4 E5C4 7DDC      E7B79C       E7B79C       F3B1A89B
E364     E364 E364 E364 E364 60DB E5C5 E5C5 E5C5 E5C5 6545 64C5 E5C5 7DB8      E7B6B8       E7B6B8       F3B1A89C
E365     E365 E365 E365 E365 60DC E5C6 E5C6 E5C6 E5C6 6546 64C6 E5C6 7D9F      E7B69F       E7B69F       F3B1A89D
E366     E366 E366 E366 E366 60DD E5C7 E5C7 E5C7 E5C7 6547 64C7 E5C7 7DB0      E7B6B0       E7B6B0       F3B1A89E
E367     E367 E367 E367 E367 60DF E5C8 E5C8 E5C8 E5C8 6548 64C8 E5C8 7DD8      E7B798       E7B798       F3B1A89F
E368     E368 E368 E368 E368 60E0 E5C9 E5C9 E5C9 E5C9 6549 64C9 E5C9 7DDD      E7B79D       E7B79D       F3B1A8A0
E369     E369 E369 E369 E369 60E2 E5CA E5CA E5CA E5CA 654A 64D1 E5CA 7DE4      E7B7A4       E7B7A4       F3B1A8A1
E36A     E36A E36A E36A E36A 60E3 E5CB E5CB E5CB E5CB 654B 64D2 E5CB 7DDE      E7B79E       E7B79E       F3B1A8A2
E36B     E36B E36B E36B E36B 60E4 E5CC E5CC E5CC E5CC 654C 64D3 E5CC 7DFB      E7B7BB       E7B7BB       F3B1A8A3
E36C     E36C E36C E36C E36C 60E6 E5CD E5CD E5CD E5CD 654D 64D4 E5CD 7DF2      E7B7B2       E7B7B2       F3B1A8A4
E36D     E36D E36D E36D E36D 60E7 E5CE E5CE E5CE E5CE 654E 64D5 E5CE 7DE1      E7B7A1       E7B7A1       F3B1A8A5
E36E     E36E E36E E36E E36E 60E9 E5CF E5CF E5CF E5CF 654F 64D6 E5CF 7E05      E7B885       E7B885       F3B1A8A6
E36F     E36F E36F E36F E36F 60EA E5D0 E5D0 E5D0 E5D0 6550 64D7 E5D0 7E0A      E7B88A       E7B88A       F3B1A8A7
E370     E370 E370 E370 E370 55A8 E5D1 E5D1 E5D1 E5D1 6551 64D8 E5D1 7E23      E7B8A3       E7B8A3       F3B1A8A8
E371     E371 E371 E371 E371 60EB E5D2 E5D2 E5D2 E5D2 6552 64D9 E5D2 7E21      E7B8A1       E7B8A1       F3B1A8A9
E372     E372 E372 E372 E372 60EC E5D3 E5D3 E5D3 E5D3 6553 64E2 E5D3 7E12      E7B892       E7B892       F3B1A8AA
E373     E373 E373 E373 E373 60F2 E5D4 E5D4 E5D4 E5D4 6554 64E3 E5D4 7E31      E7B8B1       E7B8B1       F3B1A8AB
E374     E374 E374 E374 E374 60ED E5D5 E5D5 E5D5 E5D5 6555 64E4 E5D5 7E1F      E7B89F       E7B89F       F3B1A8AC
E375     E375 E375 E375 E375 60EE E5D6 E5D6 E5D6 E5D6 6556 64E5 E5D6 7E09      E7B889       E7B889       F3B1A8AD
E376     E376 E376 E376 E376 60E5 E5D7 E5D7 E5D7 E5D7 6557 64E6 E5D7 7E0B      E7B88B       E7B88B       F3B1A8AE
E377     E377 E377 E377 E377 5BCC E5D8 E5D8 E5D8 E5D8 6558 64E7 E5D8 7E22      E7B8A2       E7B8A2       F3B1A8AF
E378     E378 E378 E378 E378 60F0 E5D9 E5D9 E5D9 E5D9 6559 64E8 E5D9 7E46      E7B986       E7B986       F3B1A8B0
E379     E379 E379 E379 E379 60F1 E5DA E5DA E5DA E5DA 655A 64E9 E5DA 7E66      E7B9A6       E7B9A6       F3B1A8B1
E37A     E37A E37A E37A E37A 60F5 E5DB E5DB E5DB E5DB 655B 644A E5DB 7E3B      E7B8BB       E7B8BB       F3B1A8B2
E37B     E37B E37B E37B E37B 60F4 E5DC E5DC E5DC E5DC 655C 645B E5DC 7E35      E7B8B5       E7B8B5       F3B1A8B3
E37C     E37C E37C E37C E37C 60F6 E5DD E5DD E5DD E5DD 655D 645A E5DD 7E39      E7B8B9       E7B8B9       F3B1A8B4
E37D     E37D E37D E37D E37D 60F7 E5DE E5DE E5DE E5DE 655E 645F E5DE 7E43      E7B983       E7B983       F3B1A8B5
E37E     E37E E37E E37E E37E 60F8 E5DF E5DF E5DF E5DF 655F 646D E5DF 7E37      E7B8B7       E7B8B7       F3B1A8B6
E380     E380 E380 E380 E380 60F9 E5E0 E5E0 E5E0 E5E0 6560 6479 E5E0 7E32      E7B8B2       E7B8B2       F3B1A8B7
E381     E381 E381 E381 E381 60EF E5E1 E5E1 E5E1 E5E1 6561 6457 E5E1 7E3A      E7B8BA       E7B8BA       F3B1A8B8
E382     E382 E382 E382 E382 6142 E5E2 E5E2 E5E2 E5E2 6562 6459 E5E2 7E67      E7B9A7       E7B9A7       F3B1A8B9
E383     E383 E383 E383 E383 60FA E5E3 E5E3 E5E3 E5E3 6563 6462 E5E3 7E5D      E7B99D       E7B99D       F3B1A8BA
E384     E384 E384 E384 E384 60FB E5E4 E5E4 E5E4 E5E4 6564 6463 E5E4 7E56      E7B996       E7B996       F3B1A8BB
E385     E385 E385 E385 E385 60FC E5E5 E5E5 E5E5 E5E5 6565 6464 E5E5 7E5E      E7B99E       E7B99E       F3B1A8BC
E386     E386 E386 E386 E386 60FE E5E6 E5E6 E5E6 E5E6 6566 6465 E5E6 7E59      E7B999       E7B999       F3B1A8BD
E387     E387 E387 E387 E387 6141 E5E7 E5E7 E5E7 E5E7 6567 6466 E5E7 7E5A      E7B99A       E7B99A       F3B1A8BE
E388     E388 E388 E388 E388 6143 E5E8 E5E8 E5E8 E5E8 6568 6467 E5E8 7E79      E7B9B9       E7B9B9       F3B1A8BF
E389     E389 E389 E389 E389 6144 E5E9 E5E9 E5E9 E5E9 6569 6468 E5E9 7E6A      E7B9AA       E7B9AA       F3B1A980
E38A     E38A E38A E38A E38A 6145 E5EA E5EA E5EA E5EA 656A 6469 E5EA 7E69      E7B9A9       E7B9A9       F3B1A981
E38B     E38B E38B E38B E38B 6146 E5EB E5EB E5EB E5EB 656B 6470 E5EB 7E7C      E7B9BC       E7B9BC       F3B1A982
E38C     E38C E38C E38C E38C 6147 E5EC E5EC E5EC E5EC 656C 6471 E5EC 7E7B      E7B9BB       E7B9BB       F3B1A983
E38D     E38D E38D E38D E38D 614B E5ED E5ED E5ED E5ED 656D 6472 E5ED 7E83      E7BA83       E7BA83       F3B1A984
E38E     E38E E38E E38E E38E 60DE E5EE E5EE E5EE E5EE 656E 6473 E5EE 7DD5      E7B795       E7B795       F3B1A985
E38F     E38F E38F E38F E38F 6148 E5EF E5EF E5EF E5EF 656F 6474 E5EF 7E7D      E7B9BD       E7B9BD       F3B1A986
E390     E390 E390 E390 E390 6149 E5F0 E5F0 E5F0 E5F0 6570 6475 E5F0 8FAE      E8BEAE       E8BEAE       F3B1A987
E391     E391 E391 E391 E391 614A E5F1 E5F1 E5F1 E5F1 6571 6476 E5F1 7E7F      E7B9BF       E7B9BF       F3B1A988
E392     E392 E392 E392 E392 5278 E5F2 E5F2 E5F2 E5F2 6572 6477 E5F2 7E88      E7BA88       E7BA88       F3B1A989
E393     E393 E393 E393 E393 614C E5F3 E5F3 E5F3 E5F3 6573 6478 E5F3 7E89      E7BA89       E7BA89       F3B1A98A
E394     E394 E394 E394 E394 614E E5F4 E5F4 E5F4 E5F4 6574 6480 E5F4 7E8C      E7BA8C       E7BA8C       F3B1A98B
E395     E395 E395 E395 E395 614F E5F5 E5F5 E5F5 E5F5 6575 648B E5F5 7E92      E7BA92       E7BA92       F3B1A98C
E396     E396 E396 E396 E396 5273 E5F6 E5F6 E5F6 E5F6 6576 649B E5F6 7E90      E7BA90       E7BA90       F3B1A98D
E397     E397 E397 E397 E397 6150 E5F7 E5F7 E5F7 E5F7 6577 649C E5F7 7E93      E7BA93       E7BA93       F3B1A98E
E398     E398 E398 E398 E398 6151 E5F8 E5F8 E5F8 E5F8 6578 64A0 E5F8 7E94      E7BA94       E7BA94       F3B1A98F
E399     E399 E399 E399 E399 6152 E5F9 E5F9 E5F9 E5F9 6579 64AB E5F9 7E96      E7BA96       E7BA96       F3B1A990
E39A     E39A E39A E39A E39A 614D E5FA E5FA E5FA E5FA 657A 64B0 E5FA 7E8E      E7BA8E       E7BA8E       F3B1A991
E39B     E39B E39B E39B E39B 6153 E5FB E5FB E5FB E5FB 657B 64C0 E5FB 7E9B      E7BA9B       E7BA9B       F3B1A992
E39C     E39C E39C E39C E39C 539C E5FC E5FC E5FC E5FC 657C 646A E5FC 7E9C      E7BA9C       E7BA9C       F3B1A993
E39D     E39D E39D E39D E39D 6154 E5FD E5FD E5FD E5FD 657D 64D0 E5FD 7F38      E7BCB8       E7BCB8       F3B1A994
E39E     E39E E39E E39E E39E 6155 E5FE E5FE E5FE E5FE 657E 64A1 E5FE 7F3A      E7BCBA       E7BCBA       F3B1A995
E39F     E39F E39F E39F E39F 6156 E6A1 E6A1 E6A1 E6A1 6621 654F E6A1 7F45      E7BD85       E7BD85       F3B1A996
E3A0     E3A0 E3A0 E3A0 E3A0 6158 E6A2 E6A2 E6A2 E6A2 6622 657F E6A2 7F4C      E7BD8C       E7BD8C       F3B1A997
E3A1     E3A1 E3A1 E3A1 E3A1 54CB E6A3 E6A3 E6A3 E6A3 6623 657B E6A3 7F4D      E7BD8D       E7BD8D       F3B1A998
E3A2     E3A2 E3A2 E3A2 E3A2 6159 E6A4 E6A4 E6A4 E6A4 6624 65E0 E6A4 7F4E      E7BD8E       E7BD8E       F3B1A999
E3A3     E3A3 E3A3 E3A3 E3A3 516E E6A5 E6A5 E6A5 E6A5 6625 656C E6A5 7F50      E7BD90       E7BD90       F3B1A99A
E3A4     E3A4 E3A4 E3A4 E3A4 615A E6A6 E6A6 E6A6 E6A6 6626 6550 E6A6 7F51      E7BD91       E7BD91       F3B1A99B
E3A5     E3A5 E3A5 E3A5 E3A5 615B E6A7 E6A7 E6A7 E6A7 6627 657D E6A7 7F55      E7BD95       E7BD95       F3B1A99C
E3A6     E3A6 E3A6 E3A6 E3A6 615C E6A8 E6A8 E6A8 E6A8 6628 654D E6A8 7F54      E7BD94       E7BD94       F3B1A99D
E3A7     E3A7 E3A7 E3A7 E3A7 615D E6A9 E6A9 E6A9 E6A9 6629 655D E6A9 7F58      E7BD98       E7BD98       F3B1A99E
E3A8     E3A8 E3A8 E3A8 E3A8 615E E6AA E6AA E6AA E6AA 662A 655C E6AA 7F5F      E7BD9F       E7BD9F       F3B1A99F
E3A9     E3A9 E3A9 E3A9 E3A9 615F E6AB E6AB E6AB E6AB 662B 654E E6AB 7F60      E7BDA0       E7BDA0       F3B1A9A0
E3AA     E3AA E3AA E3AA E3AA 6160 E6AC E6AC E6AC E6AC 662C 656B E6AC 7F68      E7BDA8       E7BDA8       F3B1A9A1
E3AB     E3AB E3AB E3AB E3AB 6162 E6AD E6AD E6AD E6AD 662D 6560 E6AD 7F69      E7BDA9       E7BDA9       F3B1A9A2
E3AC     E3AC E3AC E3AC E3AC 6161 E6AE E6AE E6AE E6AE 662E 654B E6AE 7F67      E7BDA7       E7BDA7       F3B1A9A3
E3AD     E3AD E3AD E3AD E3AD 6163 E6AF E6AF E6AF E6AF 662F 6561 E6AF 7F78      E7BDB8       E7BDB8       F3B1A9A4
E3AE     E3AE E3AE E3AE E3AE 6164 E6B0 E6B0 E6B0 E6B0 6630 65F0 E6B0 7F82      E7BE82       E7BE82       F3B1A9A5
E3AF     E3AF E3AF E3AF E3AF 6165 E6B1 E6B1 E6B1 E6B1 6631 65F1 E6B1 7F86      E7BE86       E7BE86       F3B1A9A6
E3B0     E3B0 E3B0 E3B0 E3B0 6166 E6B2 E6B2 E6B2 E6B2 6632 65F2 E6B2 7F83      E7BE83       E7BE83       F3B1A9A7
E3B1     E3B1 E3B1 E3B1 E3B1 6168 E6B3 E6B3 E6B3 E6B3 6633 65F3 E6B3 7F88      E7BE88       E7BE88       F3B1A9A8
E3B2     E3B2 E3B2 E3B2 E3B2 6167 E6B4 E6B4 E6B4 E6B4 6634 65F4 E6B4 7F87      E7BE87       E7BE87       F3B1A9A9
E3B3     E3B3 E3B3 E3B3 E3B3 6169 E6B5 E6B5 E6B5 E6B5 6635 65F5 E6B5 7F8C      E7BE8C       E7BE8C       F3B1A9AA
E3B4     E3B4 E3B4 E3B4 E3B4 616A E6B6 E6B6 E6B6 E6B6 6636 65F6 E6B6 7F94      E7BE94       E7BE94       F3B1A9AB
E3B5     E3B5 E3B5 E3B5 E3B5 616B E6B7 E6B7 E6B7 E6B7 6637 65F7 E6B7 7F9E      E7BE9E       E7BE9E       F3B1A9AC
E3B6     E3B6 E3B6 E3B6 E3B6 616C E6B8 E6B8 E6B8 E6B8 6638 65F8 E6B8 7F9D      E7BE9D       E7BE9D       F3B1A9AD
E3B7     E3B7 E3B7 E3B7 E3B7 616D E6B9 E6B9 E6B9 E6B9 6639 65F9 E6B9 7F9A      E7BE9A       E7BE9A       F3B1A9AE
E3B8     E3B8 E3B8 E3B8 E3B8 616F E6BA E6BA E6BA E6BA 663A 657A E6BA 7FA3      E7BEA3       E7BEA3       F3B1A9AF
E3B9     E3B9 E3B9 E3B9 E3B9 6170 E6BB E6BB E6BB E6BB 663B 655E E6BB 7FAF      E7BEAF       E7BEAF       F3B1A9B0
E3BA     E3BA E3BA E3BA E3BA 6172 E6BC E6BC E6BC E6BC 663C 654C E6BC 7FB2      E7BEB2       E7BEB2       F3B1A9B1
E3BB     E3BB E3BB E3BB E3BB 6173 E6BD E6BD E6BD E6BD 663D 657E E6BD 7FB9      E7BEB9       E7BEB9       F3B1A9B2
E3BC     E3BC E3BC E3BC E3BC 6171 E6BE E6BE E6BE E6BE 663E 656E E6BE 7FAE      E7BEAE       E7BEAE       F3B1A9B3
E3BD     E3BD E3BD E3BD E3BD 6174 E6BF E6BF E6BF E6BF 663F 656F E6BF 7FB6      E7BEB6       E7BEB6       F3B1A9B4
E3BE     E3BE E3BE E3BE E3BE 6175 E6C0 E6C0 E6C0 E6C0 6640 657C E6C0 7FB8      E7BEB8       E7BEB8       F3B1A9B5
E3BF     E3BF E3BF E3BF E3BF 6176 E6C1 E6C1 E6C1 E6C1 6641 65C1 E6C1 8B71      E8ADB1       E8ADB1       F3B1A9B6
E3C0     E3C0 E3C0 E3C0 E3C0 6178 E6C2 E6C2 E6C2 E6C2 6642 65C2 E6C2 7FC5      E7BF85       E7BF85       F3B1A9B7
E3C1     E3C1 E3C1 E3C1 E3C1 6179 E6C3 E6C3 E6C3 E6C3 6643 65C3 E6C3 7FC6      E7BF86       E7BF86       F3B1A9B8
E3C2     E3C2 E3C2 E3C2 E3C2 617A E6C4 E6C4 E6C4 E6C4 6644 65C4 E6C4 7FCA      E7BF8A       E7BF8A       F3B1A9B9
E3C3     E3C3 E3C3 E3C3 E3C3 617B E6C5 E6C5 E6C5 E6C5 6645 65C5 E6C5 7FD5      E7BF95       E7BF95       F3B1A9BA
E3C4     E3C4 E3C4 E3C4 E3C4 54F9 E6C6 E6C6 E6C6 E6C6 6646 65C6 E6C6 7FD4      E7BF94       E7BF94       F3B1A9BB
E3C5     E3C5 E3C5 E3C5 E3C5 617C E6C7 E6C7 E6C7 E6C7 6647 65C7 E6C7 7FE1      E7BFA1       E7BFA1       F3B1A9BC
E3C6     E3C6 E3C6 E3C6 E3C6 617D E6C8 E6C8 E6C8 E6C8 6648 65C8 E6C8 7FE6      E7BFA6       E7BFA6       F3B1A9BD
E3C7     E3C7 E3C7 E3C7 E3C7 617E E6C9 E6C9 E6C9 E6C9 6649 65C9 E6C9 7FE9      E7BFA9       E7BFA9       F3B1A9BE
E3C8     E3C8 E3C8 E3C8 E3C8 617F E6CA E6CA E6CA E6CA 664A 65D1 E6CA 7FF3      E7BFB3       E7BFB3       F3B1A9BF
E3C9     E3C9 E3C9 E3C9 E3C9 6180 E6CB E6CB E6CB E6CB 664B 65D2 E6CB 7FF9      E7BFB9       E7BFB9       F3B1AA80
E3CA     E3CA E3CA E3CA E3CA 66DA E6CC E6CC E6CC E6CC 664C 65D3 E6CC 98DC      E9A39C       E9A39C       F3B1AA81
E3CB     E3CB E3CB E3CB E3CB 53DC E6CD E6CD E6CD E6CD 664D 65D4 E6CD 8006      E88086       E88086       F3B1AA82
E3CC     E3CC E3CC E3CC E3CC 6181 E6CE E6CE E6CE E6CE 664E 65D5 E6CE 8004      E88084       E88084       F3B1AA83
E3CD     E3CD E3CD E3CD E3CD 6182 E6CF E6CF E6CF E6CF 664F 65D6 E6CF 800B      E8808B       E8808B       F3B1AA84
E3CE     E3CE E3CE E3CE E3CE 6183 E6D0 E6D0 E6D0 E6D0 6650 65D7 E6D0 8012      E88092       E88092       F3B1AA85
E3CF     E3CF E3CF E3CF E3CF 54E4 E6D1 E6D1 E6D1 E6D1 6651 65D8 E6D1 8018      E88098       E88098       F3B1AA86
E3D0     E3D0 E3D0 E3D0 E3D0 6184 E6D2 E6D2 E6D2 E6D2 6652 65D9 E6D2 8019      E88099       E88099       F3B1AA87
E3D1     E3D1 E3D1 E3D1 E3D1 6185 E6D3 E6D3 E6D3 E6D3 6653 65E2 E6D3 801C      E8809C       E8809C       F3B1AA88
E3D2     E3D2 E3D2 E3D2 E3D2 6186 E6D4 E6D4 E6D4 E6D4 6654 65E3 E6D4 8021      E880A1       E880A1       F3B1AA89
E3D3     E3D3 E3D3 E3D3 E3D3 6187 E6D5 E6D5 E6D5 E6D5 6655 65E4 E6D5 8028      E880A8       E880A8       F3B1AA8A
E3D4     E3D4 E3D4 E3D4 E3D4 6188 E6D6 E6D6 E6D6 E6D6 6656 65E5 E6D6 803F      E880BF       E880BF       F3B1AA8B
E3D5     E3D5 E3D5 E3D5 E3D5 6189 E6D7 E6D7 E6D7 E6D7 6657 65E6 E6D7 803B      E880BB       E880BB       F3B1AA8C
E3D6     E3D6 E3D6 E3D6 E3D6 618A E6D8 E6D8 E6D8 E6D8 6658 65E7 E6D8 804A      E8818A       E8818A       F3B1AA8D
E3D7     E3D7 E3D7 E3D7 E3D7 618B E6D9 E6D9 E6D9 E6D9 6659 65E8 E6D9 8046      E88186       E88186       F3B1AA8E
E3D8     E3D8 E3D8 E3D8 E3D8 618C E6DA E6DA E6DA E6DA 665A 65E9 E6DA 8052      E88192       E88192       F3B1AA8F
E3D9     E3D9 E3D9 E3D9 E3D9 618D E6DB E6DB E6DB E6DB 665B 654A E6DB 8058      E88198       E88198       F3B1AA90
E3DA     E3DA E3DA E3DA E3DA 5479 E6DC E6DC E6DC E6DC 665C 655B E6DC 805A      E8819A       E8819A       F3B1AA91
E3DB     E3DB E3DB E3DB E3DB 618E E6DD E6DD E6DD E6DD 665D 655A E6DD 805F      E8819F       E8819F       F3B1AA92
E3DC     E3DC E3DC E3DC E3DC 618F E6DE E6DE E6DE E6DE 665E 655F E6DE 8062      E881A2       E881A2       F3B1AA93
E3DD     E3DD E3DD E3DD E3DD 6190 E6DF E6DF E6DF E6DF 665F 656D E6DF 8068      E881A8       E881A8       F3B1AA94
E3DE     E3DE E3DE E3DE E3DE 6191 E6E0 E6E0 E6E0 E6E0 6660 6579 E6E0 8073      E881B3       E881B3       F3B1AA95
E3DF     E3DF E3DF E3DF E3DF 6192 E6E1 E6E1 E6E1 E6E1 6661 6557 E6E1 8072      E881B2       E881B2       F3B1AA96
E3E0     E3E0 E3E0 E3E0 E3E0 6193 E6E2 E6E2 E6E2 E6E2 6662 6559 E6E2 8070      E881B0       E881B0       F3B1AA97
E3E1     E3E1 E3E1 E3E1 E3E1 6194 E6E3 E6E3 E6E3 E6E3 6663 6562 E6E3 8076      E881B6       E881B6       F3B1AA98
E3E2     E3E2 E3E2 E3E2 E3E2 6195 E6E4 E6E4 E6E4 E6E4 6664 6563 E6E4 8079      E881B9       E881B9       F3B1AA99
E3E3     E3E3 E3E3 E3E3 E3E3 6196 E6E5 E6E5 E6E5 E6E5 6665 6564 E6E5 807D      E881BD       E881BD       F3B1AA9A
E3E4     E3E4 E3E4 E3E4 E3E4 6197 E6E6 E6E6 E6E6 E6E6 6666 6565 E6E6 807F      E881BF       E881BF       F3B1AA9B
E3E5     E3E5 E3E5 E3E5 E3E5 6198 E6E7 E6E7 E6E7 E6E7 6667 6566 E6E7 8084      E88284       E88284       F3B1AA9C
E3E6     E3E6 E3E6 E3E6 E3E6 53B6 E6E8 E6E8 E6E8 E6E8 6668 6567 E6E8 8086      E88286       E88286       F3B1AA9D
E3E7     E3E7 E3E7 E3E7 E3E7 6199 E6E9 E6E9 E6E9 E6E9 6669 6568 E6E9 8085      E88285       E88285       F3B1AA9E
E3E8     E3E8 E3E8 E3E8 E3E8 5292 E6EA E6EA E6EA E6EA 666A 6569 E6EA 809B      E8829B       E8829B       F3B1AA9F
E3E9     E3E9 E3E9 E3E9 E3E9 619A E6EB E6EB E6EB E6EB 666B 6570 E6EB 8093      E88293       E88293       F3B1AAA0
E3EA     E3EA E3EA E3EA E3EA 619B E6EC E6EC E6EC E6EC 666C 6571 E6EC 809A      E8829A       E8829A       F3B1AAA1
E3EB     E3EB E3EB E3EB E3EB 619C E6ED E6ED E6ED E6ED 666D 6572 E6ED 80AD      E882AD       E882AD       F3B1AAA2
E3EC     E3EC E3EC E3EC E3EC 619D E6EE E6EE E6EE E6EE 666E 6573 E6EE 5190      E58690       E58690       F3B1AAA3
E3ED     E3ED E3ED E3ED E3ED 619E E6EF E6EF E6EF E6EF 666F 6574 E6EF 80AC      E882AC       E882AC       F3B1AAA4
E3EE     E3EE E3EE E3EE E3EE 619F E6F0 E6F0 E6F0 E6F0 6670 6575 E6F0 80DB      E8839B       E8839B       F3B1AAA5
E3EF     E3EF E3EF E3EF E3EF 61A0 E6F1 E6F1 E6F1 E6F1 6671 6576 E6F1 80E5      E883A5       E883A5       F3B1AAA6
E3F0     E3F0 E3F0 E3F0 E3F0 61A1 E6F2 E6F2 E6F2 E6F2 6672 6577 E6F2 80D9      E88399       E88399       F3B1AAA7
E3F1     E3F1 E3F1 E3F1 E3F1 61A2 E6F3 E6F3 E6F3 E6F3 6673 6578 E6F3 80DD      E8839D       E8839D       F3B1AAA8
E3F2     E3F2 E3F2 E3F2 E3F2 61A3 E6F4 E6F4 E6F4 E6F4 6674 6580 E6F4 80C4      E88384       E88384       F3B1AAA9
E3F3     E3F3 E3F3 E3F3 E3F3 61A4 E6F5 E6F5 E6F5 E6F5 6675 658B E6F5 80DA      E8839A       E8839A       F3B1AAAA
E3F4     E3F4 E3F4 E3F4 E3F4 5288 E6F6 E6F6 E6F6 E6F6 6676 659B E6F6 80D6      E88396       E88396       F3B1AAAB
E3F5     E3F5 E3F5 E3F5 E3F5 61A5 E6F7 E6F7 E6F7 E6F7 6677 659C E6F7 8109      E88489       E88489       F3B1AAAC
E3F6     E3F6 E3F6 E3F6 E3F6 61A6 E6F8 E6F8 E6F8 E6F8 6678 65A0 E6F8 80EF      E883AF       E883AF       F3B1AAAD
E3F7     E3F7 E3F7 E3F7 E3F7 61A7 E6F9 E6F9 E6F9 E6F9 6679 65AB E6F9 80F1      E883B1       E883B1       F3B1AAAE
E3F8     E3F8 E3F8 E3F8 E3F8 61A8 E6FA E6FA E6FA E6FA 667A 65B0 E6FA 811B      E8849B       E8849B       F3B1AAAF
E3F9     E3F9 E3F9 E3F9 E3F9 4C96 E6FB E6FB E6FB E6FB 667B 65C0 E6FB 8129      E884A9       E884A9       F3B1AAB0
E3FA     E3FA E3FA E3FA E3FA 61A9 E6FC E6FC E6FC E6FC 667C 656A E6FC 8123      E884A3       E884A3       F3B1AAB1
E3FB     E3FB E3FB E3FB E3FB 61AA E6FD E6FD E6FD E6FD 667D 65D0 E6FD 812F      E884AF       E884AF       F3B1AAB2
E3FC     E3FC E3FC E3FC E3FC 61AB E6FE E6FE E6FE E6FE 667E 65A1 E6FE 814B      E8858B       E8858B       F3B1AAB3
E440     E440 E440 E440 E440 6673 E7A1 E7A1 E7A1 E7A1 6721 664F E7A1 968B      E99A8B       E99A8B       F3B1AAB4
E441     E441 E441 E441 E441 61AC E7A2 E7A2 E7A2 E7A2 6722 667F E7A2 8146      E88586       E88586       F3B1AAB5
E442     E442 E442 E442 E442 61AD E7A3 E7A3 E7A3 E7A3 6723 667B E7A3 813E      E884BE       E884BE       F3B1AAB6
E443     E443 E443 E443 E443 61AE E7A4 E7A4 E7A4 E7A4 6724 66E0 E7A4 8153      E88593       E88593       F3B1AAB7
E444     E444 E444 E444 E444 61AF E7A5 E7A5 E7A5 E7A5 6725 666C E7A5 8151      E88591       E88591       F3B1AAB8
E445     E445 E445 E445 E445 61B0 E7A6 E7A6 E7A6 E7A6 6726 6650 E7A6 80FC      E883BC       E883BC       F3B1AAB9
E446     E446 E446 E446 E446 61B1 E7A7 E7A7 E7A7 E7A7 6727 667D E7A7 8171      E885B1       E885B1       F3B1AABA
E447     E447 E447 E447 E447 61B2 E7A8 E7A8 E7A8 E7A8 6728 664D E7A8 816E      E885AE       E885AE       F3B1AABB
E448     E448 E448 E448 E448 61B3 E7A9 E7A9 E7A9 E7A9 6729 665D E7A9 8165      E885A5       E885A5       F3B1AABC
E449     E449 E449 E449 E449 61B5 E7AA E7AA E7AA E7AA 672A 665C E7AA 8166      E885A6       E885A6       F3B1AABD
E44A     E44A E44A E44A E44A 61B6 E7AB E7AB E7AB E7AB 672B 664E E7AB 8174      E885B4       E885B4       F3B1AABE
E44B     E44B E44B E44B E44B 61B7 E7AC E7AC E7AC E7AC 672C 666B E7AC 8183      E88683       E88683       F3B1AABF
E44C     E44C E44C E44C E44C 61B8 E7AD E7AD E7AD E7AD 672D 6660 E7AD 8188      E88688       E88688       F3B1AB80
E44D     E44D E44D E44D E44D 61B9 E7AE E7AE E7AE E7AE 672E 664B E7AE 818A      E8868A       E8868A       F3B1AB81
E44E     E44E E44E E44E E44E 61BA E7AF E7AF E7AF E7AF 672F 6661 E7AF 8180      E88680       E88680       F3B1AB82
E44F     E44F E44F E44F E44F 61BB E7B0 E7B0 E7B0 E7B0 6730 66F0 E7B0 8182      E88682       E88682       F3B1AB83
E450     E450 E450 E450 E450 61BC E7B1 E7B1 E7B1 E7B1 6731 66F1 E7B1 81A0      E886A0       E886A0       F3B1AB84
E451     E451 E451 E451 E451 61BD E7B2 E7B2 E7B2 E7B2 6732 66F2 E7B2 8195      E88695       E88695       F3B1AB85
E452     E452 E452 E452 E452 61C1 E7B3 E7B3 E7B3 E7B3 6733 66F3 E7B3 81A4      E886A4       E886A4       F3B1AB86
E453     E453 E453 E453 E453 61BE E7B4 E7B4 E7B4 E7B4 6734 66F4 E7B4 81A3      E886A3       E886A3       F3B1AB87
E454     E454 E454 E454 E454 61B4 E7B5 E7B5 E7B5 E7B5 6735 66F5 E7B5 815F      E8859F       E8859F       F3B1AB88
E455     E455 E455 E455 E455 61BF E7B6 E7B6 E7B6 E7B6 6736 66F6 E7B6 8193      E88693       E88693       F3B1AB89
E456     E456 E456 E456 E456 61C2 E7B7 E7B7 E7B7 E7B7 6737 66F7 E7B7 81A9      E886A9       E886A9       F3B1AB8A
E457     E457 E457 E457 E457 61C4 E7B8 E7B8 E7B8 E7B8 6738 66F8 E7B8 81B0      E886B0       E886B0       F3B1AB8B
E458     E458 E458 E458 E458 61C0 E7B9 E7B9 E7B9 E7B9 6739 66F9 E7B9 81B5      E886B5       E886B5       F3B1AB8C
E459     E459 E459 E459 E459 61C6 E7BA E7BA E7BA E7BA 673A 667A E7BA 81BE      E886BE       E886BE       F3B1AB8D
E45A     E45A E45A E45A E45A 61C3 E7BB E7BB E7BB E7BB 673B 665E E7BB 81B8      E886B8       E886B8       F3B1AB8E
E45B     E45B E45B E45B E45B 61C7 E7BC E7BC E7BC E7BC 673C 664C E7BC 81BD      E886BD       E886BD       F3B1AB8F
E45C     E45C E45C E45C E45C 61C8 E7BD E7BD E7BD E7BD 673D 667E E7BD 81C0      E88780       E88780       F3B1AB90
E45D     E45D E45D E45D E45D 61C9 E7BE E7BE E7BE E7BE 673E 666E E7BE 81C2      E88782       E88782       F3B1AB91
E45E     E45E E45E E45E E45E 61CA E7BF E7BF E7BF E7BF 673F 666F E7BF 81BA      E886BA       E886BA       F3B1AB92
E45F     E45F E45F E45F E45F 61CB E7C0 E7C0 E7C0 E7C0 6740 667C E7C0 81C9      E88789       E88789       F3B1AB93
E460     E460 E460 E460 E460 61CC E7C1 E7C1 E7C1 E7C1 6741 66C1 E7C1 81CD      E8878D       E8878D       F3B1AB94
E461     E461 E461 E461 E461 61CD E7C2 E7C2 E7C2 E7C2 6742 66C2 E7C2 81D1      E88791       E88791       F3B1AB95
E462     E462 E462 E462 E462 61CF E7C3 E7C3 E7C3 E7C3 6743 66C3 E7C3 81D9      E88799       E88799       F3B1AB96
E463     E463 E463 E463 E463 61CE E7C4 E7C4 E7C4 E7C4 6744 66C4 E7C4 81D8      E88798       E88798       F3B1AB97
E464     E464 E464 E464 E464 61C5 E7C5 E7C5 E7C5 E7C5 6745 66C5 E7C5 81C8      E88788       E88788       F3B1AB98
E465     E465 E465 E465 E465 61D0 E7C6 E7C6 E7C6 E7C6 6746 66C6 E7C6 81DA      E8879A       E8879A       F3B1AB99
E466     E466 E466 E466 E466 61D1 E7C7 E7C7 E7C7 E7C7 6747 66C7 E7C7 81DF      E8879F       E8879F       F3B1AB9A
E467     E467 E467 E467 E467 61D2 E7C8 E7C8 E7C8 E7C8 6748 66C8 E7C8 81E0      E887A0       E887A0       F3B1AB9B
E468     E468 E468 E468 E468 5173 E7C9 E7C9 E7C9 E7C9 6749 66C9 E7C9 81E7      E887A7       E887A7       F3B1AB9C
E469     E469 E469 E469 E469 61D3 E7CA E7CA E7CA E7CA 674A 66D1 E7CA 81FA      E887BA       E887BA       F3B1AB9D
E46A     E46A E46A E46A E46A 61D4 E7CB E7CB E7CB E7CB 674B 66D2 E7CB 81FB      E887BB       E887BB       F3B1AB9E
E46B     E46B E46B E46B E46B 61D5 E7CC E7CC E7CC E7CC 674C 66D3 E7CC 81FE      E887BE       E887BE       F3B1AB9F
E46C     E46C E46C E46C E46C 61D6 E7CD E7CD E7CD E7CD 674D 66D4 E7CD 8201      E88881       E88881       F3B1ABA0
E46D     E46D E46D E46D E46D 61D7 E7CE E7CE E7CE E7CE 674E 66D5 E7CE 8202      E88882       E88882       F3B1ABA1
E46E     E46E E46E E46E E46E 61D8 E7CF E7CF E7CF E7CF 674F 66D6 E7CF 8205      E88885       E88885       F3B1ABA2
E46F     E46F E46F E46F E46F 5358 E7D0 E7D0 E7D0 E7D0 6750 66D7 E7D0 8207      E88887       E88887       F3B1ABA3
E470     E470 E470 E470 E470 62BA E7D1 E7D1 E7D1 E7D1 6751 66D8 E7D1 820A      E8888A       E8888A       F3B1ABA4
E471     E471 E471 E471 E471 61D9 E7D2 E7D2 E7D2 E7D2 6752 66D9 E7D2 820D      E8888D       E8888D       F3B1ABA5
E472     E472 E472 E472 E472 61DA E7D3 E7D3 E7D3 E7D3 6753 66E2 E7D3 8210      E88890       E88890       F3B1ABA6
E473     E473 E473 E473 E473 61DC E7D4 E7D4 E7D4 E7D4 6754 66E3 E7D4 8216      E88896       E88896       F3B1ABA7
E474     E474 E474 E474 E474 61DD E7D5 E7D5 E7D5 E7D5 6755 66E4 E7D5 8229      E888A9       E888A9       F3B1ABA8
E475     E475 E475 E475 E475 61DE E7D6 E7D6 E7D6 E7D6 6756 66E5 E7D6 822B      E888AB       E888AB       F3B1ABA9
E476     E476 E476 E476 E476 61E0 E7D7 E7D7 E7D7 E7D7 6757 66E6 E7D7 8238      E888B8       E888B8       F3B1ABAA
E477     E477 E477 E477 E477 61E1 E7D8 E7D8 E7D8 E7D8 6758 66E7 E7D8 8233      E888B3       E888B3       F3B1ABAB
E478     E478 E478 E478 E478 61E2 E7D9 E7D9 E7D9 E7D9 6759 66E8 E7D9 8240      E88980       E88980       F3B1ABAC
E479     E479 E479 E479 E479 61E3 E7DA E7DA E7DA E7DA 675A 66E9 E7DA 8259      E88999       E88999       F3B1ABAD
E47A     E47A E47A E47A E47A 54AE E7DB E7DB E7DB E7DB 675B 664A E7DB 8258      E88998       E88998       F3B1ABAE
E47B     E47B E47B E47B E47B 61E5 E7DC E7DC E7DC E7DC 675C 665B E7DC 825D      E8899D       E8899D       F3B1ABAF
E47C     E47C E47C E47C E47C 61E4 E7DD E7DD E7DD E7DD 675D 665A E7DD 825A      E8899A       E8899A       F3B1ABB0
E47D     E47D E47D E47D E47D 61E6 E7DE E7DE E7DE E7DE 675E 665F E7DE 825F      E8899F       E8899F       F3B1ABB1
E47E     E47E E47E E47E E47E 61E7 E7DF E7DF E7DF E7DF 675F 666D E7DF 8264      E889A4       E889A4       F3B1ABB2
E480     E480 E480 E480 E480 61E8 E7E0 E7E0 E7E0 E7E0 6760 6679 E7E0 8262      E889A2       E889A2       F3B1ABB3
E481     E481 E481 E481 E481 61E9 E7E1 E7E1 E7E1 E7E1 6761 6657 E7E1 8268      E889A8       E889A8       F3B1ABB4
E482     E482 E482 E482 E482 61EA E7E2 E7E2 E7E2 E7E2 6762 6659 E7E2 826A      E889AA       E889AA       F3B1ABB5
E483     E483 E483 E483 E483 61EB E7E3 E7E3 E7E3 E7E3 6763 6662 E7E3 826B      E889AB       E889AB       F3B1ABB6
E484     E484 E484 E484 E484 61DF E7E4 E7E4 E7E4 E7E4 6764 6663 E7E4 822E      E888AE       E888AE       F3B1ABB7
E485     E485 E485 E485 E485 61EC E7E5 E7E5 E7E5 E7E5 6765 6664 E7E5 8271      E889B1       E889B1       F3B1ABB8
E486     E486 E486 E486 E486 61ED E7E6 E7E6 E7E6 E7E6 6766 6665 E7E6 8277      E889B7       E889B7       F3B1ABB9
E487     E487 E487 E487 E487 53C5 E7E7 E7E7 E7E7 E7E7 6767 6666 E7E7 8278      E889B8       E889B8       F3B1ABBA
E488     E488 E488 E488 E488 61EE E7E8 E7E8 E7E8 E7E8 6768 6667 E7E8 827E      E889BE       E889BE       F3B1ABBB
E489     E489 E489 E489 E489 61EF E7E9 E7E9 E7E9 E7E9 6769 6668 E7E9 828D      E88A8D       E88A8D       F3B1ABBC
E48A     E48A E48A E48A E48A 61F0 E7EA E7EA E7EA E7EA 676A 6669 E7EA 8292      E88A92       E88A92       F3B1ABBD
E48B     E48B E48B E48B E48B 61F1 E7EB E7EB E7EB E7EB 676B 6670 E7EB 82AB      E88AAB       E88AAB       F3B1ABBE
E48C     E48C E48C E48C E48C 61F2 E7EC E7EC E7EC E7EC 676C 6671 E7EC 829F      E88A9F       E88A9F       F3B1ABBF
E48D     E48D E48D E48D E48D 61F3 E7ED E7ED E7ED E7ED 676D 6672 E7ED 82BB      E88ABB       E88ABB       F3B1AC80
E48E     E48E E48E E48E E48E 61F4 E7EE E7EE E7EE E7EE 676E 6673 E7EE 82AC      E88AAC       E88AAC       F3B1AC81
E48F     E48F E48F E48F E48F 61F5 E7EF E7EF E7EF E7EF 676F 6674 E7EF 82E1      E88BA1       E88BA1       F3B1AC82
E490     E490 E490 E490 E490 61F6 E7F0 E7F0 E7F0 E7F0 6770 6675 E7F0 82E3      E88BA3       E88BA3       F3B1AC83
E491     E491 E491 E491 E491 61F7 E7F1 E7F1 E7F1 E7F1 6771 6676 E7F1 82DF      E88B9F       E88B9F       F3B1AC84
E492     E492 E492 E492 E492 61F9 E7F2 E7F2 E7F2 E7F2 6772 6677 E7F2 82D2      E88B92       E88B92       F3B1AC85
E493     E493 E493 E493 E493 61FA E7F3 E7F3 E7F3 E7F3 6773 6678 E7F3 82F4      E88BB4       E88BB4       F3B1AC86
E494     E494 E494 E494 E494 61FB E7F4 E7F4 E7F4 E7F4 6774 6680 E7F4 82F3      E88BB3       E88BB3       F3B1AC87
E495     E495 E495 E495 E495 6244 E7F5 E7F5 E7F5 E7F5 6775 668B E7F5 82FA      E88BBA       E88BBA       F3B1AC88
E496     E496 E496 E496 E496 6260 E7F6 E7F6 E7F6 E7F6 6776 669B E7F6 8393      E88E93       E88E93       F3B1AC89
E497     E497 E497 E497 E497 61FC E7F7 E7F7 E7F7 E7F7 6777 669C E7F7 8303      E88C83       E88C83       F3B1AC8A
E498     E498 E498 E498 E498 61FD E7F8 E7F8 E7F8 E7F8 6778 66A0 E7F8 82FB      E88BBB       E88BBB       F3B1AC8B
E499     E499 E499 E499 E499 61FE E7F9 E7F9 E7F9 E7F9 6779 66AB E7F9 82F9      E88BB9       E88BB9       F3B1AC8C
E49A     E49A E49A E49A E49A 6241 E7FA E7FA E7FA E7FA 677A 66B0 E7FA 82DE      E88B9E       E88B9E       F3B1AC8D
E49B     E49B E49B E49B E49B 6242 E7FB E7FB E7FB E7FB 677B 66C0 E7FB 8306      E88C86       E88C86       F3B1AC8E
E49C     E49C E49C E49C E49C 6243 E7FC E7FC E7FC E7FC 677C 666A E7FC 82DC      E88B9C       E88B9C       F3B1AC8F
E49D     E49D E49D E49D E49D 6245 E7FD E7FD E7FD E7FD 677D 66D0 E7FD 8309      E88C89       E88C89       F3B1AC90
E49E     E49E E49E E49E E49E 6246 E7FE E7FE E7FE E7FE 677E 66A1 E7FE 82D9      E88B99       E88B99       F3B1AC91
E49F     E49F E49F E49F E49F 6247 E8A1 E8A1 E8A1 E8A1 6821 674F E8A1 8335      E88CB5       E88CB5       F3B1AC92
E4A0     E4A0 E4A0 E4A0 E4A0 6249 E8A2 E8A2 E8A2 E8A2 6822 677F E8A2 8334      E88CB4       E88CB4       F3B1AC93
E4A1     E4A1 E4A1 E4A1 E4A1 624A E8A3 E8A3 E8A3 E8A3 6823 677B E8A3 8316      E88C96       E88C96       F3B1AC94
E4A2     E4A2 E4A2 E4A2 E4A2 547B E8A4 E8A4 E8A4 E8A4 6824 67E0 E8A4 8332      E88CB2       E88CB2       F3B1AC95
E4A3     E4A3 E4A3 E4A3 E4A3 624B E8A5 E8A5 E8A5 E8A5 6825 676C E8A5 8331      E88CB1       E88CB1       F3B1AC96
E4A4     E4A4 E4A4 E4A4 E4A4 624C E8A6 E8A6 E8A6 E8A6 6826 6750 E8A6 8340      E88D80       E88D80       F3B1AC97
E4A5     E4A5 E4A5 E4A5 E4A5 624D E8A7 E8A7 E8A7 E8A7 6827 677D E8A7 8339      E88CB9       E88CB9       F3B1AC98
E4A6     E4A6 E4A6 E4A6 E4A6 624E E8A8 E8A8 E8A8 E8A8 6828 674D E8A8 8350      E88D90       E88D90       F3B1AC99
E4A7     E4A7 E4A7 E4A7 E4A7 624F E8A9 E8A9 E8A9 E8A9 6829 675D E8A9 8345      E88D85       E88D85       F3B1AC9A
E4A8     E4A8 E4A8 E4A8 E4A8 6250 E8AA E8AA E8AA E8AA 682A 675C E8AA 832F      E88CAF       E88CAF       F3B1AC9B
E4A9     E4A9 E4A9 E4A9 E4A9 6251 E8AB E8AB E8AB E8AB 682B 674E E8AB 832B      E88CAB       E88CAB       F3B1AC9C
E4AA     E4AA E4AA E4AA E4AA 53F6 E8AC E8AC E8AC E8AC 682C 676B E8AC 8317      E88C97       E88C97       F3B1AC9D
E4AB     E4AB E4AB E4AB E4AB 6252 E8AD E8AD E8AD E8AD 682D 6760 E8AD 8318      E88C98       E88C98       F3B1AC9E
E4AC     E4AC E4AC E4AC E4AC 6262 E8AE E8AE E8AE E8AE 682E 674B E8AE 8385      E88E85       E88E85       F3B1AC9F
E4AD     E4AD E4AD E4AD E4AD 6253 E8AF E8AF E8AF E8AF 682F 6761 E8AF 839A      E88E9A       E88E9A       F3B1ACA0
E4AE     E4AE E4AE E4AE E4AE 6254 E8B0 E8B0 E8B0 E8B0 6830 67F0 E8B0 83AA      E88EAA       E88EAA       F3B1ACA1
E4AF     E4AF E4AF E4AF E4AF 6255 E8B1 E8B1 E8B1 E8B1 6831 67F1 E8B1 839F      E88E9F       E88E9F       F3B1ACA2
E4B0     E4B0 E4B0 E4B0 E4B0 6256 E8B2 E8B2 E8B2 E8B2 6832 67F2 E8B2 83A2      E88EA2       E88EA2       F3B1ACA3
E4B1     E4B1 E4B1 E4B1 E4B1 6257 E8B3 E8B3 E8B3 E8B3 6833 67F3 E8B3 8396      E88E96       E88E96       F3B1ACA4
E4B2     E4B2 E4B2 E4B2 E4B2 6258 E8B4 E8B4 E8B4 E8B4 6834 67F4 E8B4 8323      E88CA3       E88CA3       F3B1ACA5
E4B3     E4B3 E4B3 E4B3 E4B3 6259 E8B5 E8B5 E8B5 E8B5 6835 67F5 E8B5 838E      E88E8E       E88E8E       F3B1ACA6
E4B4     E4B4 E4B4 E4B4 E4B4 5486 E8B6 E8B6 E8B6 E8B6 6836 67F6 E8B6 8387      E88E87       E88E87       F3B1ACA7
E4B5     E4B5 E4B5 E4B5 E4B5 625C E8B7 E8B7 E8B7 E8B7 6837 67F7 E8B7 838A      E88E8A       E88E8A       F3B1ACA8
E4B6     E4B6 E4B6 E4B6 E4B6 625D E8B8 E8B8 E8B8 E8B8 6838 67F8 E8B8 837C      E88DBC       E88DBC       F3B1ACA9
E4B7     E4B7 E4B7 E4B7 E4B7 625E E8B9 E8B9 E8B9 E8B9 6839 67F9 E8B9 83B5      E88EB5       E88EB5       F3B1ACAA
E4B8     E4B8 E4B8 E4B8 E4B8 625F E8BA E8BA E8BA E8BA 683A 677A E8BA 8373      E88DB3       E88DB3       F3B1ACAB
E4B9     E4B9 E4B9 E4B9 E4B9 625A E8BB E8BB E8BB E8BB 683B 675E E8BB 8375      E88DB5       E88DB5       F3B1ACAC
E4BA     E4BA E4BA E4BA E4BA 6261 E8BC E8BC E8BC E8BC 683C 674C E8BC 83A0      E88EA0       E88EA0       F3B1ACAD
E4BB     E4BB E4BB E4BB E4BB 6263 E8BD E8BD E8BD E8BD 683D 677E E8BD 8389      E88E89       E88E89       F3B1ACAE
E4BC     E4BC E4BC E4BC E4BC 6264 E8BE E8BE E8BE E8BE 683E 676E E8BE 83A8      E88EA8       E88EA8       F3B1ACAF
E4BD     E4BD E4BD E4BD E4BD 6265 E8BF E8BF E8BF E8BF 683F 676F E8BF 83F4      E88FB4       E88FB4       F3B1ACB0
E4BE     E4BE E4BE E4BE E4BE 6266 E8C0 E8C0 E8C0 E8C0 6840 677C E8C0 8413      E89093       E89093       F3B1ACB1
E4BF     E4BF E4BF E4BF E4BF 54E9 E8C1 E8C1 E8C1 E8C1 6841 67C1 E8C1 83EB      E88FAB       E88FAB       F3B1ACB2
E4C0     E4C0 E4C0 E4C0 E4C0 6268 E8C2 E8C2 E8C2 E8C2 6842 67C2 E8C2 83CE      E88F8E       E88F8E       F3B1ACB3
E4C1     E4C1 E4C1 E4C1 E4C1 626A E8C3 E8C3 E8C3 E8C3 6843 67C3 E8C3 83FD      E88FBD       E88FBD       F3B1ACB4
E4C2     E4C2 E4C2 E4C2 E4C2 626B E8C4 E8C4 E8C4 E8C4 6844 67C4 E8C4 8403      E89083       E89083       F3B1ACB5
E4C3     E4C3 E4C3 E4C3 E4C3 626C E8C5 E8C5 E8C5 E8C5 6845 67C5 E8C5 83D8      E88F98       E88F98       F3B1ACB6
E4C4     E4C4 E4C4 E4C4 E4C4 626D E8C6 E8C6 E8C6 E8C6 6846 67C6 E8C6 840B      E8908B       E8908B       F3B1ACB7
E4C5     E4C5 E4C5 E4C5 E4C5 626E E8C7 E8C7 E8C7 E8C7 6847 67C7 E8C7 83C1      E88F81       E88F81       F3B1ACB8
E4C6     E4C6 E4C6 E4C6 E4C6 6269 E8C8 E8C8 E8C8 E8C8 6848 67C8 E8C8 83F7      E88FB7       E88FB7       F3B1ACB9
E4C7     E4C7 E4C7 E4C7 E4C7 626F E8C9 E8C9 E8C9 E8C9 6849 67C9 E8C9 8407      E89087       E89087       F3B1ACBA
E4C8     E4C8 E4C8 E4C8 E4C8 6270 E8CA E8CA E8CA E8CA 684A 67D1 E8CA 83E0      E88FA0       E88FA0       F3B1ACBB
E4C9     E4C9 E4C9 E4C9 E4C9 6271 E8CB E8CB E8CB E8CB 684B 67D2 E8CB 83F2      E88FB2       E88FB2       F3B1ACBC
E4CA     E4CA E4CA E4CA E4CA 6272 E8CC E8CC E8CC E8CC 684C 67D3 E8CC 840D      E8908D       E8908D       F3B1ACBD
E4CB     E4CB E4CB E4CB E4CB 54D8 E8CD E8CD E8CD E8CD 684D 67D4 E8CD 8422      E890A2       E890A2       F3B1ACBE
E4CC     E4CC E4CC E4CC E4CC 6273 E8CE E8CE E8CE E8CE 684E 67D5 E8CE 8420      E890A0       E890A0       F3B1ACBF
E4CD     E4CD E4CD E4CD E4CD 6275 E8CF E8CF E8CF E8CF 684F 67D6 E8CF 83BD      E88EBD       E88EBD       F3B1AD80
E4CE     E4CE E4CE E4CE E4CE 6284 E8D0 E8D0 E8D0 E8D0 6850 67D7 E8D0 8438      E890B8       E890B8       F3B1AD81
E4CF     E4CF E4CF E4CF E4CF 629D E8D1 E8D1 E8D1 E8D1 6851 67D8 E8D1 8506      E89486       E89486       F3B1AD82
E4D0     E4D0 E4D0 E4D0 E4D0 6276 E8D2 E8D2 E8D2 E8D2 6852 67D9 E8D2 83FB      E88FBB       E88FBB       F3B1AD83
E4D1     E4D1 E4D1 E4D1 E4D1 516C E8D3 E8D3 E8D3 E8D3 6853 67E2 E8D3 846D      E891AD       E891AD       F3B1AD84
E4D2     E4D2 E4D2 E4D2 E4D2 6277 E8D4 E8D4 E8D4 E8D4 6854 67E3 E8D4 842A      E890AA       E890AA       F3B1AD85
E4D3     E4D3 E4D3 E4D3 E4D3 6279 E8D5 E8D5 E8D5 E8D5 6855 67E4 E8D5 843C      E890BC       E890BC       F3B1AD86
E4D4     E4D4 E4D4 E4D4 E4D4 629F E8D6 E8D6 E8D6 E8D6 6856 67E5 E8D6 855A      E8959A       E8959A       F3B1AD87
E4D5     E4D5 E4D5 E4D5 E4D5 627A E8D7 E8D7 E8D7 E8D7 6857 67E6 E8D7 8484      E89284       E89284       F3B1AD88
E4D6     E4D6 E4D6 E4D6 E4D6 627B E8D8 E8D8 E8D8 E8D8 6858 67E7 E8D8 8477      E891B7       E891B7       F3B1AD89
E4D7     E4D7 E4D7 E4D7 E4D7 627C E8D9 E8D9 E8D9 E8D9 6859 67E8 E8D9 846B      E891AB       E891AB       F3B1AD8A
E4D8     E4D8 E4D8 E4D8 E4D8 6290 E8DA E8DA E8DA E8DA 685A 67E9 E8DA 84AD      E892AD       E892AD       F3B1AD8B
E4D9     E4D9 E4D9 E4D9 E4D9 627F E8DB E8DB E8DB E8DB 685B 674A E8DB 846E      E891AE       E891AE       F3B1AD8C
E4DA     E4DA E4DA E4DA E4DA 6280 E8DC E8DC E8DC E8DC 685C 675B E8DC 8482      E89282       E89282       F3B1AD8D
E4DB     E4DB E4DB E4DB E4DB 6281 E8DD E8DD E8DD E8DD 685D 675A E8DD 8469      E891A9       E891A9       F3B1AD8E
E4DC     E4DC E4DC E4DC E4DC 6282 E8DE E8DE E8DE E8DE 685E 675F E8DE 8446      E89186       E89186       F3B1AD8F
E4DD     E4DD E4DD E4DD E4DD 4B75 E8DF E8DF E8DF E8DF 685F 676D E8DF 842C      E890AC       E890AC       F3B1AD90
E4DE     E4DE E4DE E4DE E4DE 6283 E8E0 E8E0 E8E0 E8E0 6860 6779 E8E0 846F      E891AF       E891AF       F3B1AD91
E4DF     E4DF E4DF E4DF E4DF 627D E8E1 E8E1 E8E1 E8E1 6861 6757 E8E1 8479      E891B9       E891B9       F3B1AD92
E4E0     E4E0 E4E0 E4E0 E4E0 6285 E8E2 E8E2 E8E2 E8E2 6862 6759 E8E2 8435      E890B5       E890B5       F3B1AD93
E4E1     E4E1 E4E1 E4E1 E4E1 6286 E8E3 E8E3 E8E3 E8E3 6863 6762 E8E3 84CA      E8938A       E8938A       F3B1AD94
E4E2     E4E2 E4E2 E4E2 E4E2 6278 E8E4 E8E4 E8E4 E8E4 6864 6763 E8E4 8462      E891A2       E891A2       F3B1AD95
E4E3     E4E3 E4E3 E4E3 E4E3 6287 E8E5 E8E5 E8E5 E8E5 6865 6764 E8E5 84B9      E892B9       E892B9       F3B1AD96
E4E4     E4E4 E4E4 E4E4 E4E4 6288 E8E6 E8E6 E8E6 E8E6 6866 6765 E8E6 84BF      E892BF       E892BF       F3B1AD97
E4E5     E4E5 E4E5 E4E5 E4E5 6289 E8E7 E8E7 E8E7 E8E7 6867 6766 E8E7 849F      E8929F       E8929F       F3B1AD98
E4E6     E4E6 E4E6 E4E6 E4E6 6294 E8E8 E8E8 E8E8 E8E8 6868 6767 E8E8 84D9      E89399       E89399       F3B1AD99
E4E7     E4E7 E4E7 E4E7 E4E7 628B E8E9 E8E9 E8E9 E8E9 6869 6768 E8E9 84CD      E8938D       E8938D       F3B1AD9A
E4E8     E4E8 E4E8 E4E8 E4E8 628C E8EA E8EA E8EA E8EA 686A 6769 E8EA 84BB      E892BB       E892BB       F3B1AD9B
E4E9     E4E9 E4E9 E4E9 E4E9 628D E8EB E8EB E8EB E8EB 686B 6770 E8EB 84DA      E8939A       E8939A       F3B1AD9C
E4EA     E4EA E4EA E4EA E4EA 628E E8EC E8EC E8EC E8EC 686C 6771 E8EC 84D0      E89390       E89390       F3B1AD9D
E4EB     E4EB E4EB E4EB E4EB 628F E8ED E8ED E8ED E8ED 686D 6772 E8ED 84C1      E89381       E89381       F3B1AD9E
E4EC     E4EC E4EC E4EC E4EC 6291 E8EE E8EE E8EE E8EE 686E 6773 E8EE 84C6      E89386       E89386       F3B1AD9F
E4ED     E4ED E4ED E4ED E4ED 6292 E8EF E8EF E8EF E8EF 686F 6774 E8EF 84D6      E89396       E89396       F3B1ADA0
E4EE     E4EE E4EE E4EE E4EE 6293 E8F0 E8F0 E8F0 E8F0 6870 6775 E8F0 84A1      E892A1       E892A1       F3B1ADA1
E4EF     E4EF E4EF E4EF E4EF 54A8 E8F1 E8F1 E8F1 E8F1 6871 6776 E8F1 8521      E894A1       E894A1       F3B1ADA2
E4F0     E4F0 E4F0 E4F0 E4F0 6295 E8F2 E8F2 E8F2 E8F2 6872 6777 E8F2 84FF      E893BF       E893BF       F3B1ADA3
E4F1     E4F1 E4F1 E4F1 E4F1 6296 E8F3 E8F3 E8F3 E8F3 6873 6778 E8F3 84F4      E893B4       E893B4       F3B1ADA4
E4F2     E4F2 E4F2 E4F2 E4F2 6297 E8F4 E8F4 E8F4 E8F4 6874 6780 E8F4 8517      E89497       E89497       F3B1ADA5
E4F3     E4F3 E4F3 E4F3 E4F3 6298 E8F5 E8F5 E8F5 E8F5 6875 678B E8F5 8518      E89498       E89498       F3B1ADA6
E4F4     E4F4 E4F4 E4F4 E4F4 6299 E8F6 E8F6 E8F6 E8F6 6876 679B E8F6 852C      E894AC       E894AC       F3B1ADA7
E4F5     E4F5 E4F5 E4F5 E4F5 629A E8F7 E8F7 E8F7 E8F7 6877 679C E8F7 851F      E8949F       E8949F       F3B1ADA8
E4F6     E4F6 E4F6 E4F6 E4F6 629B E8F8 E8F8 E8F8 E8F8 6878 67A0 E8F8 8515      E89495       E89495       F3B1ADA9
E4F7     E4F7 E4F7 E4F7 E4F7 629C E8F9 E8F9 E8F9 E8F9 6879 67AB E8F9 8514      E89494       E89494       F3B1ADAA
E4F8     E4F8 E4F8 E4F8 E4F8 518B E8FA E8FA E8FA E8FA 687A 67B0 E8FA 84FC      E893BC       E893BC       F3B1ADAB
E4F9     E4F9 E4F9 E4F9 E4F9 62A0 E8FB E8FB E8FB E8FB 687B 67C0 E8FB 8540      E89580       E89580       F3B1ADAC
E4FA     E4FA E4FA E4FA E4FA 62A2 E8FC E8FC E8FC E8FC 687C 676A E8FC 8563      E895A3       E895A3       F3B1ADAD
E4FB     E4FB E4FB E4FB E4FB 62A3 E8FD E8FD E8FD E8FD 687D 67D0 E8FD 8558      E89598       E89598       F3B1ADAE
E4FC     E4FC E4FC E4FC E4FC 62A4 E8FE E8FE E8FE E8FE 687E 67A1 E8FE 8548      E89588       E89588       F3B1ADAF
E540     E540 E540 E540 E540 62A5 E9A1 E9A1 E9A1 E9A1 6921 684F E9A1 8541      E89581       E89581       F3B1ADB0
E541     E541 E541 E541 E541 5496 BCC9 E9A2 E9A2 BCC9 3C49 4CC9 E9A2 8602      E89882       E89882       F3B1ADB1
E542     E542 E542 E542 E542 62A7 E9A3 E9A3 E9A3 E9A3 6923 687B E9A3 854B      E8958B       E8958B       F3B1ADB2
E543     E543 E543 E543 E543 62A9 E9A4 E9A4 E9A4 E9A4 6924 68E0 E9A4 8555      E89595       E89595       F3B1ADB3
E544     E544 E544 E544 E544 62AA E9A5 E9A5 E9A5 E9A5 6925 686C E9A5 8580      E89680       E89680       F3B1ADB4
E545     E545 E545 E545 E545 62AB E9A6 E9A6 E9A6 E9A6 6926 6850 E9A6 85A4      E896A4       E896A4       F3B1ADB5
E546     E546 E546 E546 E546 62AC E9A7 E9A7 E9A7 E9A7 6927 687D E9A7 8588      E89688       E89688       F3B1ADB6
E547     E547 E547 E547 E547 62AD E9A8 E9A8 E9A8 E9A8 6928 684D E9A8 8591      E89691       E89691       F3B1ADB7
E548     E548 E548 E548 E548 62AE E9A9 E9A9 E9A9 E9A9 6929 685D E9A9 858A      E8968A       E8968A       F3B1ADB8
E549     E549 E549 E549 E549 62AF E9AA E9AA E9AA E9AA 692A 685C E9AA 85A8      E896A8       E896A8       F3B1ADB9
E54A     E54A E54A E54A E54A 62B0 E9AB E9AB E9AB E9AB 692B 684E E9AB 856D      E895AD       E895AD       F3B1ADBA
E54B     E54B E54B E54B E54B 62B1 E9AC E9AC E9AC E9AC 692C 686B E9AC 8594      E89694       E89694       F3B1ADBB
E54C     E54C E54C E54C E54C 62B2 E9AD E9AD E9AD E9AD 692D 6860 E9AD 859B      E8969B       E8969B       F3B1ADBC
E54D     E54D E54D E54D E54D 4C95 CCF9 E9AE E9AE CCF9 4C79 D3AB E9AE 85EA      E897AA       E897AA       F3B1ADBD
E54E     E54E E54E E54E E54E 62B4 E9AF E9AF E9AF E9AF 692F 6861 E9AF 8587      E89687       E89687       F3B1ADBE
E54F     E54F E54F E54F E54F 62B5 E9B0 E9B0 E9B0 E9B0 6930 68F0 E9B0 859C      E8969C       E8969C       F3B1ADBF
E550     E550 E550 E550 E550 62B6 E9B1 E9B1 E9B1 E9B1 6931 68F1 E9B1 8577      E895B7       E895B7       F3B1AE80
E551     E551 E551 E551 E551 62B7 E9B2 E9B2 E9B2 E9B2 6932 68F2 E9B2 857E      E895BE       E895BE       F3B1AE81
E552     E552 E552 E552 E552 62B8 E9B3 E9B3 E9B3 E9B3 6933 68F3 E9B3 8590      E89690       E89690       F3B1AE82
E553     E553 E553 E553 E553 62BC E9B4 E9B4 E9B4 E9B4 6934 68F4 E9B4 85C9      E89789       E89789       F3B1AE83
E554     E554 E554 E554 E554 62BD E9B5 E9B5 E9B5 E9B5 6935 68F5 E9B5 85BA      E896BA       E896BA       F3B1AE84
E555     E555 E555 E555 E555 62BE E9B6 E9B6 E9B6 E9B6 6936 68F6 E9B6 85CF      E8978F       E8978F       F3B1AE85
E556     E556 E556 E556 E556 62BF E9B7 E9B7 E9B7 E9B7 6937 68F7 E9B7 85B9      E896B9       E896B9       F3B1AE86
E557     E557 E557 E557 E557 62C0 E9B8 E9B8 E9B8 E9B8 6938 68F8 E9B8 85D0      E89790       E89790       F3B1AE87
E558     E558 E558 E558 E558 62C1 E9B9 E9B9 E9B9 E9B9 6939 68F9 E9B9 85D5      E89795       E89795       F3B1AE88
E559     E559 E559 E559 E559 62C2 E9BA E9BA E9BA E9BA 693A 687A E9BA 85DD      E8979D       E8979D       F3B1AE89
E55A     E55A E55A E55A E55A 62C3 E9BB E9BB E9BB E9BB 693B 685E E9BB 85E5      E897A5       E897A5       F3B1AE8A
E55B     E55B E55B E55B E55B 62C4 E9BC E9BC E9BC E9BC 693C 684C E9BC 85DC      E8979C       E8979C       F3B1AE8B
E55C     E55C E55C E55C E55C 62C5 E9BD E9BD E9BD E9BD 693D 687E E9BD 85F9      E897B9       E897B9       F3B1AE8C
E55D     E55D E55D E55D E55D 62C6 E9BE E9BE E9BE E9BE 693E 686E E9BE 860A      E8988A       E8988A       F3B1AE8D
E55E     E55E E55E E55E E55E 62C7 E9BF E9BF E9BF E9BF 693F 686F E9BF 8613      E89893       E89893       F3B1AE8E
E55F     E55F E55F E55F E55F 62C8 E9C0 E9C0 E9C0 E9C0 6940 687C E9C0 860B      E8988B       E8988B       F3B1AE8F
E560     E560 E560 E560 E560 62C9 E9C1 E9C1 E9C1 E9C1 6941 68C1 E9C1 85FE      E897BE       E897BE       F3B1AE90
E561     E561 E561 E561 E561 53ED E9C2 E9C2 E9C2 E9C2 6942 68C2 E9C2 85FA      E897BA       E897BA       F3B1AE91
E562     E562 E562 E562 E562 4EDA E9C3 E9C3 E9C3 E9C3 6943 68C3 E9C3 8606      E89886       E89886       F3B1AE92
E563     E563 E563 E563 E563 62CA E9C4 E9C4 E9C4 E9C4 6944 68C4 E9C4 8622      E898A2       E898A2       F3B1AE93
E564     E564 E564 E564 E564 62CB E9C5 E9C5 E9C5 E9C5 6945 68C5 E9C5 861A      E8989A       E8989A       F3B1AE94
E565     E565 E565 E565 E565 62CC E9C6 E9C6 E9C6 E9C6 6946 68C6 E9C6 8630      E898B0       E898B0       F3B1AE95
E566     E566 E566 E566 E566 62CD E9C7 E9C7 E9C7 E9C7 6947 68C7 E9C7 863F      E898BF       E898BF       F3B1AE96
E567     E567 E567 E567 E567 62CF E9C8 E9C8 E9C8 E9C8 6948 68C8 E9C8 864D      E8998D       E8998D       F3B1AE97
E568     E568 E568 E568 E568 62D0 E9C9 E9C9 E9C9 E9C9 6949 68C9 E9C9 4E55      E4B995       E4B995       F3B1AE98
E569     E569 E569 E569 E569 52FB E9CA E9CA E9CA E9CA 694A 68D1 E9CA 8654      E89994       E89994       F3B1AE99
E56A     E56A E56A E56A E56A 62D2 E9CB E9CB E9CB E9CB 694B 68D2 E9CB 865F      E8999F       E8999F       F3B1AE9A
E56B     E56B E56B E56B E56B 62D3 E9CC E9CC E9CC E9CC 694C 68D3 E9CC 8667      E899A7       E899A7       F3B1AE9B
E56C     E56C E56C E56C E56C 62D4 E9CD E9CD E9CD E9CD 694D 68D4 E9CD 8671      E899B1       E899B1       F3B1AE9C
E56D     E56D E56D E56D E56D 62D5 E9CE E9CE E9CE E9CE 694E 68D5 E9CE 8693      E89A93       E89A93       F3B1AE9D
E56E     E56E E56E E56E E56E 62D6 E9CF E9CF E9CF E9CF 694F 68D6 E9CF 86A3      E89AA3       E89AA3       F3B1AE9E
E56F     E56F E56F E56F E56F 62D7 E9D0 E9D0 E9D0 E9D0 6950 68D7 E9D0 86A9      E89AA9       E89AA9       F3B1AE9F
E570     E570 E570 E570 E570 62D9 E9D1 E9D1 E9D1 E9D1 6951 68D8 E9D1 86AA      E89AAA       E89AAA       F3B1AEA0
E571     E571 E571 E571 E571 62D8 E9D2 E9D2 E9D2 E9D2 6952 68D9 E9D2 868B      E89A8B       E89A8B       F3B1AEA1
E572     E572 E572 E572 E572 62DA E9D3 E9D3 E9D3 E9D3 6953 68E2 E9D3 868C      E89A8C       E89A8C       F3B1AEA2
E573     E573 E573 E573 E573 62DB E9D4 E9D4 E9D4 E9D4 6954 68E3 E9D4 86B6      E89AB6       E89AB6       F3B1AEA3
E574     E574 E574 E574 E574 62DC E9D5 E9D5 E9D5 E9D5 6955 68E4 E9D5 86AF      E89AAF       E89AAF       F3B1AEA4
E575     E575 E575 E575 E575 62DD E9D6 E9D6 E9D6 E9D6 6956 68E5 E9D6 86C4      E89B84       E89B84       F3B1AEA5
E576     E576 E576 E576 E576 62DE E9D7 E9D7 E9D7 E9D7 6957 68E6 E9D7 86C6      E89B86       E89B86       F3B1AEA6
E577     E577 E577 E577 E577 62DF E9D8 E9D8 E9D8 E9D8 6958 68E7 E9D8 86B0      E89AB0       E89AB0       F3B1AEA7
E578     E578 E578 E578 E578 62E0 E9D9 E9D9 E9D9 E9D9 6959 68E8 E9D9 86C9      E89B89       E89B89       F3B1AEA8
E579     E579 E579 E579 E579 5351 B3C2 E9DA E9DA B3C2 3342 F3C2 E9DA 8823      E8A0A3       E8A0A3       F3B1AEA9
E57A     E57A E57A E57A E57A 62E3 E9DB E9DB E9DB E9DB 695B 684A E9DB 86AB      E89AAB       E89AAB       F3B1AEAA
E57B     E57B E57B E57B E57B 62E4 E9DC E9DC E9DC E9DC 695C 685B E9DC 86D4      E89B94       E89B94       F3B1AEAB
E57C     E57C E57C E57C E57C 62E5 E9DD E9DD E9DD E9DD 695D 685A E9DD 86DE      E89B9E       E89B9E       F3B1AEAC
E57D     E57D E57D E57D E57D 62E6 E9DE E9DE E9DE E9DE 695E 685F E9DE 86E9      E89BA9       E89BA9       F3B1AEAD
E57E     E57E E57E E57E E57E 62E7 E9DF E9DF E9DF E9DF 695F 686D E9DF 86EC      E89BAC       E89BAC       F3B1AEAE
E580     E580 E580 E580 E580 62E8 E9E0 E9E0 E9E0 E9E0 6960 6879 E9E0 86DF      E89B9F       E89B9F       F3B1AEAF
E581     E581 E581 E581 E581 62E9 E9E1 E9E1 E9E1 E9E1 6961 6857 E9E1 86DB      E89B9B       E89B9B       F3B1AEB0
E582     E582 E582 E582 E582 4F74 E9E2 E9E2 E9E2 E9E2 6962 6859 E9E2 86EF      E89BAF       E89BAF       F3B1AEB1
E583     E583 E583 E583 E583 62EA E9E3 E9E3 E9E3 E9E3 6963 6862 E9E3 8712      E89C92       E89C92       F3B1AEB2
E584     E584 E584 E584 E584 62EB E9E4 E9E4 E9E4 E9E4 6964 6863 E9E4 8706      E89C86       E89C86       F3B1AEB3
E585     E585 E585 E585 E585 62EC E9E5 E9E5 E9E5 E9E5 6965 6864 E9E5 8708      E89C88       E89C88       F3B1AEB4
E586     E586 E586 E586 E586 62ED E9E6 E9E6 E9E6 E9E6 6966 6865 E9E6 8700      E89C80       E89C80       F3B1AEB5
E587     E587 E587 E587 E587 62EE E9E7 E9E7 E9E7 E9E7 6967 6866 E9E7 8703      E89C83       E89C83       F3B1AEB6
E588     E588 E588 E588 E588 62EF E9E8 E9E8 E9E8 E9E8 6968 6867 E9E8 86FB      E89BBB       E89BBB       F3B1AEB7
E589     E589 E589 E589 E589 62F0 E9E9 E9E9 E9E9 E9E9 6969 6868 E9E9 8711      E89C91       E89C91       F3B1AEB8
E58A     E58A E58A E58A E58A 62F1 E9EA E9EA E9EA E9EA 696A 6869 E9EA 8709      E89C89       E89C89       F3B1AEB9
E58B     E58B E58B E58B E58B 62F2 E9EB E9EB E9EB E9EB 696B 6870 E9EB 870D      E89C8D       E89C8D       F3B1AEBA
E58C     E58C E58C E58C E58C 62F3 E9EC E9EC E9EC E9EC 696C 6871 E9EC 86F9      E89BB9       E89BB9       F3B1AEBB
E58D     E58D E58D E58D E58D 62F4 E9ED E9ED E9ED E9ED 696D 6872 E9ED 870A      E89C8A       E89C8A       F3B1AEBC
E58E     E58E E58E E58E E58E 62F5 E9EE E9EE E9EE E9EE 696E 6873 E9EE 8734      E89CB4       E89CB4       F3B1AEBD
E58F     E58F E58F E58F E58F 62F6 E9EF E9EF E9EF E9EF 696F 6874 E9EF 873F      E89CBF       E89CBF       F3B1AEBE
E590     E590 E590 E590 E590 526D E9F0 E9F0 E9F0 E9F0 6970 6875 E9F0 8737      E89CB7       E89CB7       F3B1AEBF
E591     E591 E591 E591 E591 62F7 E9F1 E9F1 E9F1 E9F1 6971 6876 E9F1 873B      E89CBB       E89CBB       F3B1AF80
E592     E592 E592 E592 E592 62F8 E9F2 E9F2 E9F2 E9F2 6972 6877 E9F2 8725      E89CA5       E89CA5       F3B1AF81
E593     E593 E593 E593 E593 62F9 E9F3 E9F3 E9F3 E9F3 6973 6878 E9F3 8729      E89CA9       E89CA9       F3B1AF82
E594     E594 E594 E594 E594 62FA E9F4 E9F4 E9F4 E9F4 6974 6880 E9F4 871A      E89C9A       E89C9A       F3B1AF83
E595     E595 E595 E595 E595 6346 E9F5 E9F5 E9F5 E9F5 6975 688B E9F5 8760      E89DA0       E89DA0       F3B1AF84
E596     E596 E596 E596 E596 62FB E9F6 E9F6 E9F6 E9F6 6976 689B E9F6 875F      E89D9F       E89D9F       F3B1AF85
E597     E597 E597 E597 E597 62FC E9F7 E9F7 E9F7 E9F7 6977 689C E9F7 8778      E89DB8       E89DB8       F3B1AF86
E598     E598 E598 E598 E598 62FD E9F8 E9F8 E9F8 E9F8 6978 68A0 E9F8 874C      E89D8C       E89D8C       F3B1AF87
E599     E599 E599 E599 E599 62FE E9F9 E9F9 E9F9 E9F9 6979 68AB E9F9 874E      E89D8E       E89D8E       F3B1AF88
E59A     E59A E59A E59A E59A 6341 E9FA E9FA E9FA E9FA 697A 68B0 E9FA 8774      E89DB4       E89DB4       F3B1AF89
E59B     E59B E59B E59B E59B 6342 E9FB E9FB E9FB E9FB 697B 68C0 E9FB 8757      E89D97       E89D97       F3B1AF8A
E59C     E59C E59C E59C E59C 6343 E9FC E9FC E9FC E9FC 697C 686A E9FC 8768      E89DA8       E89DA8       F3B1AF8B
E59D     E59D E59D E59D E59D 6347 E9FD E9FD E9FD E9FD 697D 68D0 E9FD 876E      E89DAE       E89DAE       F3B1AF8C
E59E     E59E E59E E59E E59E 6348 E9FE E9FE E9FE E9FE 697E 68A1 E9FE 8759      E89D99       E89D99       F3B1AF8D
E59F     E59F E59F E59F E59F 6349 EAA1 EAA1 EAA1 EAA1 6A21 694F EAA1 8753      E89D93       E89D93       F3B1AF8E
E5A0     E5A0 E5A0 E5A0 E5A0 634A EAA2 EAA2 EAA2 EAA2 6A22 697F EAA2 8763      E89DA3       E89DA3       F3B1AF8F
E5A1     E5A1 E5A1 E5A1 E5A1 6345 EAA3 EAA3 EAA3 EAA3 6A23 697B EAA3 876A      E89DAA       E89DAA       F3B1AF90
E5A2     E5A2 E5A2 E5A2 E5A2 55BD C7E8 EAA4 EAA4 C7E8 4768 C767 EAA4 8805      E8A085       E8A085       F3B1AF91
E5A3     E5A3 E5A3 E5A3 E5A3 634C EAA5 EAA5 EAA5 EAA5 6A25 696C EAA5 87A2      E89EA2       E89EA2       F3B1AF92
E5A4     E5A4 E5A4 E5A4 E5A4 634E EAA6 EAA6 EAA6 EAA6 6A26 6950 EAA6 879F      E89E9F       E89E9F       F3B1AF93
E5A5     E5A5 E5A5 E5A5 E5A5 6344 EAA7 EAA7 EAA7 EAA7 6A27 697D EAA7 8782      E89E82       E89E82       F3B1AF94
E5A6     E5A6 E5A6 E5A6 E5A6 634F EAA8 EAA8 EAA8 EAA8 6A28 694D EAA8 87AF      E89EAF       E89EAF       F3B1AF95
E5A7     E5A7 E5A7 E5A7 E5A7 6350 EAA9 EAA9 EAA9 EAA9 6A29 695D EAA9 87CB      E89F8B       E89F8B       F3B1AF96
E5A8     E5A8 E5A8 E5A8 E5A8 6351 EAAA EAAA EAAA EAAA 6A2A 695C EAAA 87BD      E89EBD       E89EBD       F3B1AF97
E5A9     E5A9 E5A9 E5A9 E5A9 6352 EAAB EAAB EAAB EAAB 6A2B 694E EAAB 87C0      E89F80       E89F80       F3B1AF98
E5AA     E5AA E5AA E5AA E5AA 6353 EAAC EAAC EAAC EAAC 6A2C 696B EAAC 87D0      E89F90       E89F90       F3B1AF99
E5AB     E5AB E5AB E5AB E5AB 6354 EAAD EAAD EAAD EAAD 6A2D 6960 EAAD 96D6      E99B96       E99B96       F3B1AF9A
E5AC     E5AC E5AC E5AC E5AC 6355 EAAE EAAE EAAE EAAE 6A2E 694B EAAE 87AB      E89EAB       E89EAB       F3B1AF9B
E5AD     E5AD E5AD E5AD E5AD 6356 EAAF EAAF EAAF EAAF 6A2F 6961 EAAF 87C4      E89F84       E89F84       F3B1AF9C
E5AE     E5AE E5AE E5AE E5AE 6357 EAB0 EAB0 EAB0 EAB0 6A30 69F0 EAB0 87B3      E89EB3       E89EB3       F3B1AF9D
E5AF     E5AF E5AF E5AF E5AF 54F4 EAB1 EAB1 EAB1 EAB1 6A31 69F1 EAB1 87C7      E89F87       E89F87       F3B1AF9E
E5B0     E5B0 E5B0 E5B0 E5B0 634D EAB2 EAB2 EAB2 EAB2 6A32 69F2 EAB2 87C6      E89F86       E89F86       F3B1AF9F
E5B1     E5B1 E5B1 E5B1 E5B1 6359 EAB3 EAB3 EAB3 EAB3 6A33 69F3 EAB3 87BB      E89EBB       E89EBB       F3B1AFA0
E5B2     E5B2 E5B2 E5B2 E5B2 635A EAB4 EAB4 EAB4 EAB4 6A34 69F4 EAB4 87EF      E89FAF       E89FAF       F3B1AFA1
E5B3     E5B3 E5B3 E5B3 E5B3 635B EAB5 EAB5 EAB5 EAB5 6A35 69F5 EAB5 87F2      E89FB2       E89FB2       F3B1AFA2
E5B4     E5B4 E5B4 E5B4 E5B4 635C EAB6 EAB6 EAB6 EAB6 6A36 69F6 EAB6 87E0      E89FA0       E89FA0       F3B1AFA3
E5B5     E5B5 E5B5 E5B5 E5B5 635F EAB7 EAB7 EAB7 EAB7 6A37 69F7 EAB7 880F      E8A08F       E8A08F       F3B1AFA4
E5B6     E5B6 E5B6 E5B6 E5B6 6361 EAB8 EAB8 EAB8 EAB8 6A38 69F8 EAB8 880D      E8A08D       E8A08D       F3B1AFA5
E5B7     E5B7 E5B7 E5B7 E5B7 6362 EAB9 EAB9 EAB9 EAB9 6A39 69F9 EAB9 87FE      E89FBE       E89FBE       F3B1AFA6
E5B8     E5B8 E5B8 E5B8 E5B8 6363 EABA EABA EABA EABA 6A3A 697A EABA 87F6      E89FB6       E89FB6       F3B1AFA7
E5B9     E5B9 E5B9 E5B9 E5B9 6364 EABB EABB EABB EABB 6A3B 695E EABB 87F7      E89FB7       E89FB7       F3B1AFA8
E5BA     E5BA E5BA E5BA E5BA 635D EABC EABC EABC EABC 6A3C 694C EABC 880E      E8A08E       E8A08E       F3B1AFA9
E5BB     E5BB E5BB E5BB E5BB 6358 EABD EABD EABD EABD 6A3D 697E EABD 87D2      E89F92       E89F92       F3B1AFAA
E5BC     E5BC E5BC E5BC E5BC 6365 EABE EABE EABE EABE 6A3E 696E EABE 8811      E8A091       E8A091       F3B1AFAB
E5BD     E5BD E5BD E5BD E5BD 6360 EABF EABF EABF EABF 6A3F 696F EABF 8816      E8A096       E8A096       F3B1AFAC
E5BE     E5BE E5BE E5BE E5BE 6366 EAC0 EAC0 EAC0 EAC0 6A40 697C EAC0 8815      E8A095       E8A095       F3B1AFAD
E5BF     E5BF E5BF E5BF E5BF 6367 EAC1 EAC1 EAC1 EAC1 6A41 69C1 EAC1 8822      E8A0A2       E8A0A2       F3B1AFAE
E5C0     E5C0 E5C0 E5C0 E5C0 6368 EAC2 EAC2 EAC2 EAC2 6A42 69C2 EAC2 8821      E8A0A1       E8A0A1       F3B1AFAF
E5C1     E5C1 E5C1 E5C1 E5C1 636A EAC3 EAC3 EAC3 EAC3 6A43 69C3 EAC3 8831      E8A0B1       E8A0B1       F3B1AFB0
E5C2     E5C2 E5C2 E5C2 E5C2 636B EAC4 EAC4 EAC4 EAC4 6A44 69C4 EAC4 8836      E8A0B6       E8A0B6       F3B1AFB1
E5C3     E5C3 E5C3 E5C3 E5C3 636C EAC5 EAC5 EAC5 EAC5 6A45 69C5 EAC5 8839      E8A0B9       E8A0B9       F3B1AFB2
E5C4     E5C4 E5C4 E5C4 E5C4 6369 EAC6 EAC6 EAC6 EAC6 6A46 69C6 EAC6 8827      E8A0A7       E8A0A7       F3B1AFB3
E5C5     E5C5 E5C5 E5C5 E5C5 636D EAC7 EAC7 EAC7 EAC7 6A47 69C7 EAC7 883B      E8A0BB       E8A0BB       F3B1AFB4
E5C6     E5C6 E5C6 E5C6 E5C6 636F EAC8 EAC8 EAC8 EAC8 6A48 69C8 EAC8 8844      E8A184       E8A184       F3B1AFB5
E5C7     E5C7 E5C7 E5C7 E5C7 636E EAC9 EAC9 EAC9 EAC9 6A49 69C9 EAC9 8842      E8A182       E8A182       F3B1AFB6
E5C8     E5C8 E5C8 E5C8 E5C8 6371 EACA EACA EACA EACA 6A4A 69D1 EACA 8852      E8A192       E8A192       F3B1AFB7
E5C9     E5C9 E5C9 E5C9 E5C9 6372 EACB EACB EACB EACB 6A4B 69D2 EACB 8859      E8A199       E8A199       F3B1AFB8
E5CA     E5CA E5CA E5CA E5CA 6373 EACC EACC EACC EACC 6A4C 69D3 EACC 885E      E8A19E       E8A19E       F3B1AFB9
E5CB     E5CB E5CB E5CB E5CB 6374 EACD EACD EACD EACD 6A4D 69D4 EACD 8862      E8A1A2       E8A1A2       F3B1AFBA
E5CC     E5CC E5CC E5CC E5CC 6375 EACE EACE EACE EACE 6A4E 69D5 EACE 886B      E8A1AB       E8A1AB       F3B1AFBB
E5CD     E5CD E5CD E5CD E5CD 6376 EACF EACF EACF EACF 6A4F 69D6 EACF 8881      E8A281       E8A281       F3B1AFBC
E5CE     E5CE E5CE E5CE E5CE 6377 EAD0 EAD0 EAD0 EAD0 6A50 69D7 EAD0 887E      E8A1BE       E8A1BE       F3B1AFBD
E5CF     E5CF E5CF E5CF E5CF 637C EAD1 EAD1 EAD1 EAD1 6A51 69D8 EAD1 889E      E8A29E       E8A29E       F3B1AFBE
E5D0     E5D0 E5D0 E5D0 E5D0 6378 EAD2 EAD2 EAD2 EAD2 6A52 69D9 EAD2 8875      E8A1B5       E8A1B5       F3B1AFBF
E5D1     E5D1 E5D1 E5D1 E5D1 6379 EAD3 EAD3 EAD3 EAD3 6A53 69E2 EAD3 887D      E8A1BD       E8A1BD       F3B1B080
E5D2     E5D2 E5D2 E5D2 E5D2 6385 EAD4 EAD4 EAD4 EAD4 6A54 69E3 EAD4 88B5      E8A2B5       E8A2B5       F3B1B081
E5D3     E5D3 E5D3 E5D3 E5D3 637A EAD5 EAD5 EAD5 EAD5 6A55 69E4 EAD5 8872      E8A1B2       E8A1B2       F3B1B082
E5D4     E5D4 E5D4 E5D4 E5D4 637B EAD6 EAD6 EAD6 EAD6 6A56 69E5 EAD6 8882      E8A282       E8A282       F3B1B083
E5D5     E5D5 E5D5 E5D5 E5D5 637D EAD7 EAD7 EAD7 EAD7 6A57 69E6 EAD7 8897      E8A297       E8A297       F3B1B084
E5D6     E5D6 E5D6 E5D6 E5D6 637E EAD8 EAD8 EAD8 EAD8 6A58 69E7 EAD8 8892      E8A292       E8A292       F3B1B085
E5D7     E5D7 E5D7 E5D7 E5D7 637F EAD9 EAD9 EAD9 EAD9 6A59 69E8 EAD9 88AE      E8A2AE       E8A2AE       F3B1B086
E5D8     E5D8 E5D8 E5D8 E5D8 6380 EADA EADA EADA EADA 6A5A 69E9 EADA 8899      E8A299       E8A299       F3B1B087
E5D9     E5D9 E5D9 E5D9 E5D9 6381 EADB EADB EADB EADB 6A5B 694A EADB 88A2      E8A2A2       E8A2A2       F3B1B088
E5DA     E5DA E5DA E5DA E5DA 6382 EADC EADC EADC EADC 6A5C 695B EADC 888D      E8A28D       E8A28D       F3B1B089
E5DB     E5DB E5DB E5DB E5DB 6383 EADD EADD EADD EADD 6A5D 695A EADD 88A4      E8A2A4       E8A2A4       F3B1B08A
E5DC     E5DC E5DC E5DC E5DC 54C5 EADE EADE EADE EADE 6A5E 695F EADE 88B0      E8A2B0       E8A2B0       F3B1B08B
E5DD     E5DD E5DD E5DD E5DD 6384 EADF EADF EADF EADF 6A5F 696D EADF 88BF      E8A2BF       E8A2BF       F3B1B08C
E5DE     E5DE E5DE E5DE E5DE 6386 EAE0 EAE0 EAE0 EAE0 6A60 6979 EAE0 88B1      E8A2B1       E8A2B1       F3B1B08D
E5DF     E5DF E5DF E5DF E5DF 6387 EAE1 EAE1 EAE1 EAE1 6A61 6957 EAE1 88C3      E8A383       E8A383       F3B1B08E
E5E0     E5E0 E5E0 E5E0 E5E0 6388 EAE2 EAE2 EAE2 EAE2 6A62 6959 EAE2 88C4      E8A384       E8A384       F3B1B08F
E5E1     E5E1 E5E1 E5E1 E5E1 6389 EAE3 EAE3 EAE3 EAE3 6A63 6962 EAE3 88D4      E8A394       E8A394       F3B1B090
E5E2     E5E2 E5E2 E5E2 E5E2 638A EAE4 EAE4 EAE4 EAE4 6A64 6963 EAE4 88D8      E8A398       E8A398       F3B1B091
E5E3     E5E3 E5E3 E5E3 E5E3 638B EAE5 EAE5 EAE5 EAE5 6A65 6964 EAE5 88D9      E8A399       E8A399       F3B1B092
E5E4     E5E4 E5E4 E5E4 E5E4 638C EAE6 EAE6 EAE6 EAE6 6A66 6965 EAE6 88DD      E8A39D       E8A39D       F3B1B093
E5E5     E5E5 E5E5 E5E5 E5E5 638D EAE7 EAE7 EAE7 EAE7 6A67 6966 EAE7 88F9      E8A3B9       E8A3B9       F3B1B094
E5E6     E5E6 E5E6 E5E6 E5E6 638E EAE8 EAE8 EAE8 EAE8 6A68 6967 EAE8 8902      E8A482       E8A482       F3B1B095
E5E7     E5E7 E5E7 E5E7 E5E7 638F EAE9 EAE9 EAE9 EAE9 6A69 6968 EAE9 88FC      E8A3BC       E8A3BC       F3B1B096
E5E8     E5E8 E5E8 E5E8 E5E8 5349 EAEA EAEA EAEA EAEA 6A6A 6969 EAEA 88F4      E8A3B4       E8A3B4       F3B1B097
E5E9     E5E9 E5E9 E5E9 E5E9 6391 EAEB EAEB EAEB EAEB 6A6B 6970 EAEB 88E8      E8A3A8       E8A3A8       F3B1B098
E5EA     E5EA E5EA E5EA E5EA 6392 EAEC EAEC EAEC EAEC 6A6C 6971 EAEC 88F2      E8A3B2       E8A3B2       F3B1B099
E5EB     E5EB E5EB E5EB E5EB 6393 EAED EAED EAED EAED 6A6D 6972 EAED 8904      E8A484       E8A484       F3B1B09A
E5EC     E5EC E5EC E5EC E5EC 6394 EAEE EAEE EAEE EAEE 6A6E 6973 EAEE 890C      E8A48C       E8A48C       F3B1B09B
E5ED     E5ED E5ED E5ED E5ED 6397 EAEF EAEF EAEF EAEF 6A6F 6974 EAEF 890A      E8A48A       E8A48A       F3B1B09C
E5EE     E5EE E5EE E5EE E5EE 6398 EAF0 EAF0 EAF0 EAF0 6A70 6975 EAF0 8913      E8A493       E8A493       F3B1B09D
E5EF     E5EF E5EF E5EF E5EF 639F EAF1 EAF1 EAF1 EAF1 6A71 6976 EAF1 8943      E8A583       E8A583       F3B1B09E
E5F0     E5F0 E5F0 E5F0 E5F0 6399 EAF2 EAF2 EAF2 EAF2 6A72 6977 EAF2 891E      E8A49E       E8A49E       F3B1B09F
E5F1     E5F1 E5F1 E5F1 E5F1 639A EAF3 EAF3 EAF3 EAF3 6A73 6978 EAF3 8925      E8A4A5       E8A4A5       F3B1B0A0
E5F2     E5F2 E5F2 E5F2 E5F2 6395 EAF4 EAF4 EAF4 EAF4 6A74 6980 EAF4 892A      E8A4AA       E8A4AA       F3B1B0A1
E5F3     E5F3 E5F3 E5F3 E5F3 639B EAF5 EAF5 EAF5 EAF5 6A75 698B EAF5 892B      E8A4AB       E8A4AB       F3B1B0A2
E5F4     E5F4 E5F4 E5F4 E5F4 639C EAF6 EAF6 EAF6 EAF6 6A76 699B EAF6 8941      E8A581       E8A581       F3B1B0A3
E5F5     E5F5 E5F5 E5F5 E5F5 506B EAF7 EAF7 EAF7 EAF7 6A77 699C EAF7 8944      E8A584       E8A584       F3B1B0A4
E5F6     E5F6 E5F6 E5F6 E5F6 639D EAF8 EAF8 EAF8 EAF8 6A78 69A0 EAF8 893B      E8A4BB       E8A4BB       F3B1B0A5
E5F7     E5F7 E5F7 E5F7 E5F7 639E EAF9 EAF9 EAF9 EAF9 6A79 69AB EAF9 8936      E8A4B6       E8A4B6       F3B1B0A6
E5F8     E5F8 E5F8 E5F8 E5F8 63A0 EAFA EAFA EAFA EAFA 6A7A 69B0 EAFA 8938      E8A4B8       E8A4B8       F3B1B0A7
E5F9     E5F9 E5F9 E5F9 E5F9 63A2 EAFB EAFB EAFB EAFB 6A7B 69C0 EAFB 894C      E8A58C       E8A58C       F3B1B0A8
E5FA     E5FA E5FA E5FA E5FA 6396 EAFC EAFC EAFC EAFC 6A7C 696A EAFC 891D      E8A49D       E8A49D       F3B1B0A9
E5FB     E5FB E5FB E5FB E5FB 63A3 EAFD EAFD EAFD EAFD 6A7D 69D0 EAFD 8960      E8A5A0       E8A5A0       F3B1B0AA
E5FC     E5FC E5FC E5FC E5FC 63A4 EAFE EAFE EAFE EAFE 6A7E 69A1 EAFE 895E      E8A59E       E8A59E       F3B1B0AB
E640     E640 E640 E640 E640 63A5 EBA1 EBA1 EBA1 EBA1 6B21 704F EBA1 8966      E8A5A6       E8A5A6       F3B1B0AC
E641     E641 E641 E641 E641 63A7 EBA2 EBA2 EBA2 EBA2 6B22 707F EBA2 8964      E8A5A4       E8A5A4       F3B1B0AD
E642     E642 E642 E642 E642 63A8 EBA3 EBA3 EBA3 EBA3 6B23 707B EBA3 896D      E8A5AD       E8A5AD       F3B1B0AE
E643     E643 E643 E643 E643 63A6 EBA4 EBA4 EBA4 EBA4 6B24 70E0 EBA4 896A      E8A5AA       E8A5AA       F3B1B0AF
E644     E644 E644 E644 E644 63A9 EBA5 EBA5 EBA5 EBA5 6B25 706C EBA5 896F      E8A5AF       E8A5AF       F3B1B0B0
E645     E645 E645 E645 E645 63AA EBA6 EBA6 EBA6 EBA6 6B26 7050 EBA6 8974      E8A5B4       E8A5B4       F3B1B0B1
E646     E646 E646 E646 E646 63AB EBA7 EBA7 EBA7 EBA7 6B27 707D EBA7 8977      E8A5B7       E8A5B7       F3B1B0B2
E647     E647 E647 E647 E647 63AC EBA8 EBA8 EBA8 EBA8 6B28 704D EBA8 897E      E8A5BE       E8A5BE       F3B1B0B3
E648     E648 E648 E648 E648 63AD EBA9 EBA9 EBA9 EBA9 6B29 705D EBA9 8983      E8A683       E8A683       F3B1B0B4
E649     E649 E649 E649 E649 63AE EBAA EBAA EBAA EBAA 6B2A 705C EBAA 8988      E8A688       E8A688       F3B1B0B5
E64A     E64A E64A E64A E64A 63AF EBAB EBAB EBAB EBAB 6B2B 704E EBAB 898A      E8A68A       E8A68A       F3B1B0B6
E64B     E64B E64B E64B E64B 63B0 EBAC EBAC EBAC EBAC 6B2C 706B EBAC 8993      E8A693       E8A693       F3B1B0B7
E64C     E64C E64C E64C E64C 63B1 EBAD EBAD EBAD EBAD 6B2D 7060 EBAD 8998      E8A698       E8A698       F3B1B0B8
E64D     E64D E64D E64D E64D 63B2 EBAE EBAE EBAE EBAE 6B2E 704B EBAE 89A1      E8A6A1       E8A6A1       F3B1B0B9
E64E     E64E E64E E64E E64E 63B3 EBAF EBAF EBAF EBAF 6B2F 7061 EBAF 89A9      E8A6A9       E8A6A9       F3B1B0BA
E64F     E64F E64F E64F E64F 63B4 EBB0 EBB0 EBB0 EBB0 6B30 70F0 EBB0 89A6      E8A6A6       E8A6A6       F3B1B0BB
E650     E650 E650 E650 E650 63B5 EBB1 EBB1 EBB1 EBB1 6B31 70F1 EBB1 89AC      E8A6AC       E8A6AC       F3B1B0BC
E651     E651 E651 E651 E651 63B6 EBB2 EBB2 EBB2 EBB2 6B32 70F2 EBB2 89AF      E8A6AF       E8A6AF       F3B1B0BD
E652     E652 E652 E652 E652 63B7 EBB3 EBB3 EBB3 EBB3 6B33 70F3 EBB3 89B2      E8A6B2       E8A6B2       F3B1B0BE
E653     E653 E653 E653 E653 63B8 EBB4 EBB4 EBB4 EBB4 6B34 70F4 EBB4 89BA      E8A6BA       E8A6BA       F3B1B0BF
E654     E654 E654 E654 E654 63BA EBB5 EBB5 EBB5 EBB5 6B35 70F5 EBB5 89BD      E8A6BD       E8A6BD       F3B1B180
E655     E655 E655 E655 E655 63B9 EBB6 EBB6 EBB6 EBB6 6B36 70F6 EBB6 89BF      E8A6BF       E8A6BF       F3B1B181
E656     E656 E656 E656 E656 63BB EBB7 EBB7 EBB7 EBB7 6B37 70F7 EBB7 89C0      E8A780       E8A780       F3B1B182
E657     E657 E657 E657 E657 63BC EBB8 EBB8 EBB8 EBB8 6B38 70F8 EBB8 89DA      E8A79A       E8A79A       F3B1B183
E658     E658 E658 E658 E658 5360 EBB9 EBB9 EBB9 EBB9 6B39 70F9 EBB9 89DC      E8A79C       E8A79C       F3B1B184
E659     E659 E659 E659 E659 63BD EBBA EBBA EBBA EBBA 6B3A 707A EBBA 89DD      E8A79D       E8A79D       F3B1B185
E65A     E65A E65A E65A E65A 63BE EBBB EBBB EBBB EBBB 6B3B 705E EBBB 89E7      E8A7A7       E8A7A7       F3B1B186
E65B     E65B E65B E65B E65B 63BF EBBC EBBC EBBC EBBC 6B3C 704C EBBC 89F4      E8A7B4       E8A7B4       F3B1B187
E65C     E65C E65C E65C E65C 63C0 EBBD EBBD EBBD EBBD 6B3D 707E EBBD 89F8      E8A7B8       E8A7B8       F3B1B188
E65D     E65D E65D E65D E65D 63C1 EBBE EBBE EBBE EBBE 6B3E 706E EBBE 8A03      E8A883       E8A883       F3B1B189
E65E     E65E E65E E65E E65E 63C2 EBBF EBBF EBBF EBBF 6B3F 706F EBBF 8A16      E8A896       E8A896       F3B1B18A
E65F     E65F E65F E65F E65F 63C3 EBC0 EBC0 EBC0 EBC0 6B40 707C EBC0 8A10      E8A890       E8A890       F3B1B18B
E660     E660 E660 E660 E660 63C4 EBC1 EBC1 EBC1 EBC1 6B41 70C1 EBC1 8A0C      E8A88C       E8A88C       F3B1B18C
E661     E661 E661 E661 E661 63C6 EBC2 EBC2 EBC2 EBC2 6B42 70C2 EBC2 8A1B      E8A89B       E8A89B       F3B1B18D
E662     E662 E662 E662 E662 63C7 EBC3 EBC3 EBC3 EBC3 6B43 70C3 EBC3 8A1D      E8A89D       E8A89D       F3B1B18E
E663     E663 E663 E663 E663 63C8 EBC4 EBC4 EBC4 EBC4 6B44 70C4 EBC4 8A25      E8A8A5       E8A8A5       F3B1B18F
E664     E664 E664 E664 E664 63C9 EBC5 EBC5 EBC5 EBC5 6B45 70C5 EBC5 8A36      E8A8B6       E8A8B6       F3B1B190
E665     E665 E665 E665 E665 63CA EBC6 EBC6 EBC6 EBC6 6B46 70C6 EBC6 8A41      E8A981       E8A981       F3B1B191
E666     E666 E666 E666 E666 63CC EBC7 EBC7 EBC7 EBC7 6B47 70C7 EBC7 8A5B      E8A99B       E8A99B       F3B1B192
E667     E667 E667 E667 E667 63CD EBC8 EBC8 EBC8 EBC8 6B48 70C8 EBC8 8A52      E8A992       E8A992       F3B1B193
E668     E668 E668 E668 E668 63CE EBC9 EBC9 EBC9 EBC9 6B49 70C9 EBC9 8A46      E8A986       E8A986       F3B1B194
E669     E669 E669 E669 E669 63CF EBCA EBCA EBCA EBCA 6B4A 70D1 EBCA 8A48      E8A988       E8A988       F3B1B195
E66A     E66A E66A E66A E66A 63D0 EBCB EBCB EBCB EBCB 6B4B 70D2 EBCB 8A7C      E8A9BC       E8A9BC       F3B1B196
E66B     E66B E66B E66B E66B 63D1 EBCC EBCC EBCC EBCC 6B4C 70D3 EBCC 8A6D      E8A9AD       E8A9AD       F3B1B197
E66C     E66C E66C E66C E66C 63D2 EBCD EBCD EBCD EBCD 6B4D 70D4 EBCD 8A6C      E8A9AC       E8A9AC       F3B1B198
E66D     E66D E66D E66D E66D 63D3 EBCE EBCE EBCE EBCE 6B4E 70D5 EBCE 8A62      E8A9A2       E8A9A2       F3B1B199
E66E     E66E E66E E66E E66E 63D5 EBCF EBCF EBCF EBCF 6B4F 70D6 EBCF 8A85      E8AA85       E8AA85       F3B1B19A
E66F     E66F E66F E66F E66F 63D6 EBD0 EBD0 EBD0 EBD0 6B50 70D7 EBD0 8A82      E8AA82       E8AA82       F3B1B19B
E670     E670 E670 E670 E670 63D7 EBD1 EBD1 EBD1 EBD1 6B51 70D8 EBD1 8A84      E8AA84       E8AA84       F3B1B19C
E671     E671 E671 E671 E671 63D8 EBD2 EBD2 EBD2 EBD2 6B52 70D9 EBD2 8AA8      E8AAA8       E8AAA8       F3B1B19D
E672     E672 E672 E672 E672 63D9 EBD3 EBD3 EBD3 EBD3 6B53 70E2 EBD3 8AA1      E8AAA1       E8AAA1       F3B1B19E
E673     E673 E673 E673 E673 63DA EBD4 EBD4 EBD4 EBD4 6B54 70E3 EBD4 8A91      E8AA91       E8AA91       F3B1B19F
E674     E674 E674 E674 E674 63DB EBD5 EBD5 EBD5 EBD5 6B55 70E4 EBD5 8AA5      E8AAA5       E8AAA5       F3B1B1A0
E675     E675 E675 E675 E675 63DC EBD6 EBD6 EBD6 EBD6 6B56 70E5 EBD6 8AA6      E8AAA6       E8AAA6       F3B1B1A1
E676     E676 E676 E676 E676 63DD EBD7 EBD7 EBD7 EBD7 6B57 70E6 EBD7 8A9A      E8AA9A       E8AA9A       F3B1B1A2
E677     E677 E677 E677 E677 63DE EBD8 EBD8 EBD8 EBD8 6B58 70E7 EBD8 8AA3      E8AAA3       E8AAA3       F3B1B1A3
E678     E678 E678 E678 E678 50B2 EBD9 EBD9 EBD9 EBD9 6B59 70E8 EBD9 8AC4      E8AB84       E8AB84       F3B1B1A4
E679     E679 E679 E679 E679 63E2 EBDA EBDA EBDA EBDA 6B5A 70E9 EBDA 8ACD      E8AB8D       E8AB8D       F3B1B1A5
E67A     E67A E67A E67A E67A 63E3 EBDB EBDB EBDB EBDB 6B5B 704A EBDB 8AC2      E8AB82       E8AB82       F3B1B1A6
E67B     E67B E67B E67B E67B 63E4 EBDC EBDC EBDC EBDC 6B5C 705B EBDC 8ADA      E8AB9A       E8AB9A       F3B1B1A7
E67C     E67C E67C E67C E67C 4E96 B4D2 EBDD EBDD B4D2 3452 F4D9 EBDD 8AEB      E8ABAB       E8ABAB       F3B1B1A8
E67D     E67D E67D E67D E67D 63E5 EBDE EBDE EBDE EBDE 6B5E 705F EBDE 8AF3      E8ABB3       E8ABB3       F3B1B1A9
E67E     E67E E67E E67E E67E 63E6 EBDF EBDF EBDF EBDF 6B5F 706D EBDF 8AE7      E8ABA7       E8ABA7       F3B1B1AA
E680     E680 E680 E680 E680 63E7 EBE0 EBE0 EBE0 EBE0 6B60 7079 EBE0 8AE4      E8ABA4       E8ABA4       F3B1B1AB
E681     E681 E681 E681 E681 63E8 EBE1 EBE1 EBE1 EBE1 6B61 7057 EBE1 8AF1      E8ABB1       E8ABB1       F3B1B1AC
E682     E682 E682 E682 E682 63E9 EBE2 EBE2 EBE2 EBE2 6B62 7059 EBE2 8B14      E8AC94       E8AC94       F3B1B1AD
E683     E683 E683 E683 E683 63EA EBE3 EBE3 EBE3 EBE3 6B63 7062 EBE3 8AE0      E8ABA0       E8ABA0       F3B1B1AE
E684     E684 E684 E684 E684 63EB EBE4 EBE4 EBE4 EBE4 6B64 7063 EBE4 8AE2      E8ABA2       E8ABA2       F3B1B1AF
E685     E685 E685 E685 E685 63F0 EBE5 EBE5 EBE5 EBE5 6B65 7064 EBE5 8AF7      E8ABB7       E8ABB7       F3B1B1B0
E686     E686 E686 E686 E686 63F1 EBE6 EBE6 EBE6 EBE6 6B66 7065 EBE6 8ADE      E8AB9E       E8AB9E       F3B1B1B1
E687     E687 E687 E687 E687 63F2 EBE7 EBE7 EBE7 EBE7 6B67 7066 EBE7 8ADB      E8AB9B       E8AB9B       F3B1B1B2
E688     E688 E688 E688 E688 63F3 EBE8 EBE8 EBE8 EBE8 6B68 7067 EBE8 8B0C      E8AC8C       E8AC8C       F3B1B1B3
E689     E689 E689 E689 E689 63F4 EBE9 EBE9 EBE9 EBE9 6B69 7068 EBE9 8B07      E8AC87       E8AC87       F3B1B1B4
E68A     E68A E68A E68A E68A 63F5 EBEA EBEA EBEA EBEA 6B6A 7069 EBEA 8B1A      E8AC9A       E8AC9A       F3B1B1B5
E68B     E68B E68B E68B E68B 63EC EBEB EBEB EBEB EBEB 6B6B 7070 EBEB 8AE1      E8ABA1       E8ABA1       F3B1B1B6
E68C     E68C E68C E68C E68C 63F6 EBEC EBEC EBEC EBEC 6B6C 7071 EBEC 8B16      E8AC96       E8AC96       F3B1B1B7
E68D     E68D E68D E68D E68D 63F7 EBED EBED EBED EBED 6B6D 7072 EBED 8B10      E8AC90       E8AC90       F3B1B1B8
E68E     E68E E68E E68E E68E 63F8 EBEE EBEE EBEE EBEE 6B6E 7073 EBEE 8B17      E8AC97       E8AC97       F3B1B1B9
E68F     E68F E68F E68F E68F 63F9 EBEF EBEF EBEF EBEF 6B6F 7074 EBEF 8B20      E8ACA0       E8ACA0       F3B1B1BA
E690     E690 E690 E690 E690 63FA EBF0 EBF0 EBF0 EBF0 6B70 7075 EBF0 8B33      E8ACB3       E8ACB3       F3B1B1BB
E691     E691 E691 E691 E691 63FC EBF1 EBF1 EBF1 EBF1 6B71 7076 EBF1 97AB      E99EAB       E99EAB       F3B1B1BC
E692     E692 E692 E692 E692 63FD EBF2 EBF2 EBF2 EBF2 6B72 7077 EBF2 8B26      E8ACA6       E8ACA6       F3B1B1BD
E693     E693 E693 E693 E693 63FE EBF3 EBF3 EBF3 EBF3 6B73 7078 EBF3 8B2B      E8ACAB       E8ACAB       F3B1B1BE
E694     E694 E694 E694 E694 6441 EBF4 EBF4 EBF4 EBF4 6B74 7080 EBF4 8B3E      E8ACBE       E8ACBE       F3B1B1BF
E695     E695 E695 E695 E695 5381 EBF5 EBF5 EBF5 EBF5 6B75 708B EBF5 8B28      E8ACA8       E8ACA8       F3B1B280
E696     E696 E696 E696 E696 63FB EBF6 EBF6 EBF6 EBF6 6B76 709B EBF6 8B41      E8AD81       E8AD81       F3B1B281
E697     E697 E697 E697 E697 6442 EBF7 EBF7 EBF7 EBF7 6B77 709C EBF7 8B4C      E8AD8C       E8AD8C       F3B1B282
E698     E698 E698 E698 E698 6443 EBF8 EBF8 EBF8 EBF8 6B78 70A0 EBF8 8B4F      E8AD8F       E8AD8F       F3B1B283
E699     E699 E699 E699 E699 6444 EBF9 EBF9 EBF9 EBF9 6B79 70AB EBF9 8B4E      E8AD8E       E8AD8E       F3B1B284
E69A     E69A E69A E69A E69A 6446 EBFA EBFA EBFA EBFA 6B7A 70B0 EBFA 8B49      E8AD89       E8AD89       F3B1B285
E69B     E69B E69B E69B E69B 6447 EBFB EBFB EBFB EBFB 6B7B 70C0 EBFB 8B56      E8AD96       E8AD96       F3B1B286
E69C     E69C E69C E69C E69C 6448 EBFC EBFC EBFC EBFC 6B7C 706A EBFC 8B5B      E8AD9B       E8AD9B       F3B1B287
E69D     E69D E69D E69D E69D 6449 EBFD EBFD EBFD EBFD 6B7D 70D0 EBFD 8B5A      E8AD9A       E8AD9A       F3B1B288
E69E     E69E E69E E69E E69E 644B EBFE EBFE EBFE EBFE 6B7E 70A1 EBFE 8B6B      E8ADAB       E8ADAB       F3B1B289
E69F     E69F E69F E69F E69F 644C ECA1 ECA1 ECA1 ECA1 6C21 714F ECA1 8B5F      E8AD9F       E8AD9F       F3B1B28A
E6A0     E6A0 E6A0 E6A0 E6A0 644D ECA2 ECA2 ECA2 ECA2 6C22 717F ECA2 8B6C      E8ADAC       E8ADAC       F3B1B28B
E6A1     E6A1 E6A1 E6A1 E6A1 644E ECA3 ECA3 ECA3 ECA3 6C23 717B ECA3 8B6F      E8ADAF       E8ADAF       F3B1B28C
E6A2     E6A2 E6A2 E6A2 E6A2 644A ECA4 ECA4 ECA4 ECA4 6C24 71E0 ECA4 8B74      E8ADB4       E8ADB4       F3B1B28D
E6A3     E6A3 E6A3 E6A3 E6A3 644F ECA5 ECA5 ECA5 ECA5 6C25 716C ECA5 8B7D      E8ADBD       E8ADBD       F3B1B28E
E6A4     E6A4 E6A4 E6A4 E6A4 6451 ECA6 ECA6 ECA6 ECA6 6C26 7150 ECA6 8B80      E8AE80       E8AE80       F3B1B28F
E6A5     E6A5 E6A5 E6A5 E6A5 6452 ECA7 ECA7 ECA7 ECA7 6C27 717D ECA7 8B8C      E8AE8C       E8AE8C       F3B1B290
E6A6     E6A6 E6A6 E6A6 E6A6 6453 ECA8 ECA8 ECA8 ECA8 6C28 714D ECA8 8B8E      E8AE8E       E8AE8E       F3B1B291
E6A7     E6A7 E6A7 E6A7 E6A7 6455 ECA9 ECA9 ECA9 ECA9 6C29 715D ECA9 8B92      E8AE92       E8AE92       F3B1B292
E6A8     E6A8 E6A8 E6A8 E6A8 6456 ECAA ECAA ECAA ECAA 6C2A 715C ECAA 8B93      E8AE93       E8AE93       F3B1B293
E6A9     E6A9 E6A9 E6A9 E6A9 6457 ECAB ECAB ECAB ECAB 6C2B 714E ECAB 8B96      E8AE96       E8AE96       F3B1B294
E6AA     E6AA E6AA E6AA E6AA 6454 ECAC ECAC ECAC ECAC 6C2C 716B ECAC 8B99      E8AE99       E8AE99       F3B1B295
E6AB     E6AB E6AB E6AB E6AB 6458 ECAD ECAD ECAD ECAD 6C2D 7160 ECAD 8B9A      E8AE9A       E8AE9A       F3B1B296
E6AC     E6AC E6AC E6AC E6AC 6459 ECAE ECAE ECAE ECAE 6C2E 714B ECAE 8C3A      E8B0BA       E8B0BA       F3B1B297
E6AD     E6AD E6AD E6AD E6AD 645A ECAF ECAF ECAF ECAF 6C2F 7161 ECAF 8C41      E8B181       E8B181       F3B1B298
E6AE     E6AE E6AE E6AE E6AE 645B ECB0 ECB0 ECB0 ECB0 6C30 71F0 ECB0 8C3F      E8B0BF       E8B0BF       F3B1B299
E6AF     E6AF E6AF E6AF E6AF 645C ECB1 ECB1 ECB1 ECB1 6C31 71F1 ECB1 8C48      E8B188       E8B188       F3B1B29A
E6B0     E6B0 E6B0 E6B0 E6B0 645D ECB2 ECB2 ECB2 ECB2 6C32 71F2 ECB2 8C4C      E8B18C       E8B18C       F3B1B29B
E6B1     E6B1 E6B1 E6B1 E6B1 645E ECB3 ECB3 ECB3 ECB3 6C33 71F3 ECB3 8C4E      E8B18E       E8B18E       F3B1B29C
E6B2     E6B2 E6B2 E6B2 E6B2 645F ECB4 ECB4 ECB4 ECB4 6C34 71F4 ECB4 8C50      E8B190       E8B190       F3B1B29D
E6B3     E6B3 E6B3 E6B3 E6B3 6460 ECB5 ECB5 ECB5 ECB5 6C35 71F5 ECB5 8C55      E8B195       E8B195       F3B1B29E
E6B4     E6B4 E6B4 E6B4 E6B4 6461 ECB6 ECB6 ECB6 ECB6 6C36 71F6 ECB6 8C62      E8B1A2       E8B1A2       F3B1B29F
E6B5     E6B5 E6B5 E6B5 E6B5 6462 ECB7 ECB7 ECB7 ECB7 6C37 71F7 ECB7 8C6C      E8B1AC       E8B1AC       F3B1B2A0
E6B6     E6B6 E6B6 E6B6 E6B6 6463 ECB8 ECB8 ECB8 ECB8 6C38 71F8 ECB8 8C78      E8B1B8       E8B1B8       F3B1B2A1
E6B7     E6B7 E6B7 E6B7 E6B7 6464 ECB9 ECB9 ECB9 ECB9 6C39 71F9 ECB9 8C7A      E8B1BA       E8B1BA       F3B1B2A2
E6B8     E6B8 E6B8 E6B8 E6B8 6466 ECBA ECBA ECBA ECBA 6C3A 717A ECBA 8C82      E8B282       E8B282       F3B1B2A3
E6B9     E6B9 E6B9 E6B9 E6B9 6467 ECBB ECBB ECBB ECBB 6C3B 715E ECBB 8C89      E8B289       E8B289       F3B1B2A4
E6BA     E6BA E6BA E6BA E6BA 6468 ECBC ECBC ECBC ECBC 6C3C 714C ECBC 8C85      E8B285       E8B285       F3B1B2A5
E6BB     E6BB E6BB E6BB E6BB 6469 ECBD ECBD ECBD ECBD 6C3D 717E ECBD 8C8A      E8B28A       E8B28A       F3B1B2A6
E6BC     E6BC E6BC E6BC E6BC 646A ECBE ECBE ECBE ECBE 6C3E 716E ECBE 8C8D      E8B28D       E8B28D       F3B1B2A7
E6BD     E6BD E6BD E6BD E6BD 646B ECBF ECBF ECBF ECBF 6C3F 716F ECBF 8C8E      E8B28E       E8B28E       F3B1B2A8
E6BE     E6BE E6BE E6BE E6BE 646D ECC0 ECC0 ECC0 ECC0 6C40 717C ECC0 8C94      E8B294       E8B294       F3B1B2A9
E6BF     E6BF E6BF E6BF E6BF 6465 ECC1 ECC1 ECC1 ECC1 6C41 71C1 ECC1 8C7C      E8B1BC       E8B1BC       F3B1B2AA
E6C0     E6C0 E6C0 E6C0 E6C0 646C ECC2 ECC2 ECC2 ECC2 6C42 71C2 ECC2 8C98      E8B298       E8B298       F3B1B2AB
E6C1     E6C1 E6C1 E6C1 E6C1 646E ECC3 ECC3 ECC3 ECC3 6C43 71C3 ECC3 621D      E6889D       E6889D       F3B1B2AC
E6C2     E6C2 E6C2 E6C2 E6C2 646F ECC4 ECC4 ECC4 ECC4 6C44 71C4 ECC4 8CAD      E8B2AD       E8B2AD       F3B1B2AD
E6C3     E6C3 E6C3 E6C3 E6C3 6470 ECC5 ECC5 ECC5 ECC5 6C45 71C5 ECC5 8CAA      E8B2AA       E8B2AA       F3B1B2AE
E6C4     E6C4 E6C4 E6C4 E6C4 6472 ECC6 ECC6 ECC6 ECC6 6C46 71C6 ECC6 8CBD      E8B2BD       E8B2BD       F3B1B2AF
E6C5     E6C5 E6C5 E6C5 E6C5 6473 ECC7 ECC7 ECC7 ECC7 6C47 71C7 ECC7 8CB2      E8B2B2       E8B2B2       F3B1B2B0
E6C6     E6C6 E6C6 E6C6 E6C6 6474 ECC8 ECC8 ECC8 ECC8 6C48 71C8 ECC8 8CB3      E8B2B3       E8B2B3       F3B1B2B1
E6C7     E6C7 E6C7 E6C7 E6C7 6471 ECC9 ECC9 ECC9 ECC9 6C49 71C9 ECC9 8CAE      E8B2AE       E8B2AE       F3B1B2B2
E6C8     E6C8 E6C8 E6C8 E6C8 6476 ECCA ECCA ECCA ECCA 6C4A 71D1 ECCA 8CB6      E8B2B6       E8B2B6       F3B1B2B3
E6C9     E6C9 E6C9 E6C9 E6C9 6477 ECCB ECCB ECCB ECCB 6C4B 71D2 ECCB 8CC8      E8B388       E8B388       F3B1B2B4
E6CA     E6CA E6CA E6CA E6CA 6475 ECCC ECCC ECCC ECCC 6C4C 71D3 ECCC 8CC1      E8B381       E8B381       F3B1B2B5
E6CB     E6CB E6CB E6CB E6CB 5566 C1A8 ECCD ECCD C1A8 4128 C14D ECCD 8CE4      E8B3A4       E8B3A4       F3B1B2B6
E6CC     E6CC E6CC E6CC E6CC 647A ECCE ECCE ECCE ECCE 6C4E 71D5 ECCE 8CE3      E8B3A3       E8B3A3       F3B1B2B7
E6CD     E6CD E6CD E6CD E6CD 647B ECCF ECCF ECCF ECCF 6C4F 71D6 ECCF 8CDA      E8B39A       E8B39A       F3B1B2B8
E6CE     E6CE E6CE E6CE E6CE 647E ECD0 ECD0 ECD0 ECD0 6C50 71D7 ECD0 8CFD      E8B3BD       E8B3BD       F3B1B2B9
E6CF     E6CF E6CF E6CF E6CF 647F ECD1 ECD1 ECD1 ECD1 6C51 71D8 ECD1 8CFA      E8B3BA       E8B3BA       F3B1B2BA
E6D0     E6D0 E6D0 E6D0 E6D0 6480 ECD2 ECD2 ECD2 ECD2 6C52 71D9 ECD2 8CFB      E8B3BB       E8B3BB       F3B1B2BB
E6D1     E6D1 E6D1 E6D1 E6D1 535A ECD3 ECD3 ECD3 ECD3 6C53 71E2 ECD3 8D04      E8B484       E8B484       F3B1B2BC
E6D2     E6D2 E6D2 E6D2 E6D2 5574 ECD4 ECD4 ECD4 ECD4 6C54 71E3 ECD4 8D05      E8B485       E8B485       F3B1B2BD
E6D3     E6D3 E6D3 E6D3 E6D3 6482 ECD5 ECD5 ECD5 ECD5 6C55 71E4 ECD5 8D0A      E8B48A       E8B48A       F3B1B2BE
E6D4     E6D4 E6D4 E6D4 E6D4 6481 ECD6 ECD6 ECD6 ECD6 6C56 71E5 ECD6 8D07      E8B487       E8B487       F3B1B2BF
E6D5     E6D5 E6D5 E6D5 E6D5 6483 ECD7 ECD7 ECD7 ECD7 6C57 71E6 ECD7 8D0F      E8B48F       E8B48F       F3B1B380
E6D6     E6D6 E6D6 E6D6 E6D6 6484 ECD8 ECD8 ECD8 ECD8 6C58 71E7 ECD8 8D0D      E8B48D       E8B48D       F3B1B381
E6D7     E6D7 E6D7 E6D7 E6D7 6486 ECD9 ECD9 ECD9 ECD9 6C59 71E8 ECD9 8D10      E8B490       E8B490       F3B1B382
E6D8     E6D8 E6D8 E6D8 E6D8 686E ECDA ECDA ECDA ECDA 6C5A 71E9 ECDA 9F4E      E9BD8E       E9BD8E       F3B1B383
E6D9     E6D9 E6D9 E6D9 E6D9 6487 ECDB ECDB ECDB ECDB 6C5B 714A ECDB 8D13      E8B493       E8B493       F3B1B384
E6DA     E6DA E6DA E6DA E6DA 6479 ECDC ECDC ECDC ECDC 6C5C 715B ECDC 8CCD      E8B38D       E8B38D       F3B1B385
E6DB     E6DB E6DB E6DB E6DB 6488 ECDD ECDD ECDD ECDD 6C5D 715A ECDD 8D14      E8B494       E8B494       F3B1B386
E6DC     E6DC E6DC E6DC E6DC 6489 ECDE ECDE ECDE ECDE 6C5E 715F ECDE 8D16      E8B496       E8B496       F3B1B387
E6DD     E6DD E6DD E6DD E6DD 648A ECDF ECDF ECDF ECDF 6C5F 716D ECDF 8D67      E8B5A7       E8B5A7       F3B1B388
E6DE     E6DE E6DE E6DE E6DE 648B ECE0 ECE0 ECE0 ECE0 6C60 7179 ECE0 8D6D      E8B5AD       E8B5AD       F3B1B389
E6DF     E6DF E6DF E6DF E6DF 648C ECE1 ECE1 ECE1 ECE1 6C61 7157 ECE1 8D71      E8B5B1       E8B5B1       F3B1B38A
E6E0     E6E0 E6E0 E6E0 E6E0 51C9 ECE2 ECE2 ECE2 ECE2 6C62 7159 ECE2 8D73      E8B5B3       E8B5B3       F3B1B38B
E6E1     E6E1 E6E1 E6E1 E6E1 648F ECE3 ECE3 ECE3 ECE3 6C63 7162 ECE3 8D81      E8B681       E8B681       F3B1B38C
E6E2     E6E2 E6E2 E6E2 E6E2 51DE ECE4 ECE4 ECE4 ECE4 6C64 7163 ECE4 8D99      E8B699       E8B699       F3B1B38D
E6E3     E6E3 E6E3 E6E3 E6E3 6490 ECE5 ECE5 ECE5 ECE5 6C65 7164 ECE5 8DC2      E8B782       E8B782       F3B1B38E
E6E4     E6E4 E6E4 E6E4 E6E4 6491 ECE6 ECE6 ECE6 ECE6 6C66 7165 ECE6 8DBE      E8B6BE       E8B6BE       F3B1B38F
E6E5     E6E5 E6E5 E6E5 E6E5 6492 ECE7 ECE7 ECE7 ECE7 6C67 7166 ECE7 8DBA      E8B6BA       E8B6BA       F3B1B390
E6E6     E6E6 E6E6 E6E6 E6E6 6493 ECE8 ECE8 ECE8 ECE8 6C68 7167 ECE8 8DCF      E8B78F       E8B78F       F3B1B391
E6E7     E6E7 E6E7 E6E7 E6E7 6494 ECE9 ECE9 ECE9 ECE9 6C69 7168 ECE9 8DDA      E8B79A       E8B79A       F3B1B392
E6E8     E6E8 E6E8 E6E8 E6E8 6495 ECEA ECEA ECEA ECEA 6C6A 7169 ECEA 8DD6      E8B796       E8B796       F3B1B393
E6E9     E6E9 E6E9 E6E9 E6E9 6496 ECEB ECEB ECEB ECEB 6C6B 7170 ECEB 8DCC      E8B78C       E8B78C       F3B1B394
E6EA     E6EA E6EA E6EA E6EA 6497 ECEC ECEC ECEC ECEC 6C6C 7171 ECEC 8DDB      E8B79B       E8B79B       F3B1B395
E6EB     E6EB E6EB E6EB E6EB 6498 ECED ECED ECED ECED 6C6D 7172 ECED 8DCB      E8B78B       E8B78B       F3B1B396
E6EC     E6EC E6EC E6EC E6EC 6499 ECEE ECEE ECEE ECEE 6C6E 7173 ECEE 8DEA      E8B7AA       E8B7AA       F3B1B397
E6ED     E6ED E6ED E6ED E6ED 649A ECEF ECEF ECEF ECEF 6C6F 7174 ECEF 8DEB      E8B7AB       E8B7AB       F3B1B398
E6EE     E6EE E6EE E6EE E6EE 649B ECF0 ECF0 ECF0 ECF0 6C70 7175 ECF0 8DDF      E8B79F       E8B79F       F3B1B399
E6EF     E6EF E6EF E6EF E6EF 649C ECF1 ECF1 ECF1 ECF1 6C71 7176 ECF1 8DE3      E8B7A3       E8B7A3       F3B1B39A
E6F0     E6F0 E6F0 E6F0 E6F0 649D ECF2 ECF2 ECF2 ECF2 6C72 7177 ECF2 8DFC      E8B7BC       E8B7BC       F3B1B39B
E6F1     E6F1 E6F1 E6F1 E6F1 649E ECF3 ECF3 ECF3 ECF3 6C73 7178 ECF3 8E08      E8B888       E8B888       F3B1B39C
E6F2     E6F2 E6F2 E6F2 E6F2 64A0 ECF4 ECF4 ECF4 ECF4 6C74 7180 ECF4 8E09      E8B889       E8B889       F3B1B39D
E6F3     E6F3 E6F3 E6F3 E6F3 649F ECF5 ECF5 ECF5 ECF5 6C75 718B ECF5 8DFF      E8B7BF       E8B7BF       F3B1B39E
E6F4     E6F4 E6F4 E6F4 E6F4 64A1 ECF6 ECF6 ECF6 ECF6 6C76 719B ECF6 8E1D      E8B89D       E8B89D       F3B1B39F
E6F5     E6F5 E6F5 E6F5 E6F5 64A2 ECF7 ECF7 ECF7 ECF7 6C77 719C ECF7 8E1E      E8B89E       E8B89E       F3B1B3A0
E6F6     E6F6 E6F6 E6F6 E6F6 64A3 ECF8 ECF8 ECF8 ECF8 6C78 71A0 ECF8 8E10      E8B890       E8B890       F3B1B3A1
E6F7     E6F7 E6F7 E6F7 E6F7 64A4 ECF9 ECF9 ECF9 ECF9 6C79 71AB ECF9 8E1F      E8B89F       E8B89F       F3B1B3A2
E6F8     E6F8 E6F8 E6F8 E6F8 64A5 ECFA ECFA ECFA ECFA 6C7A 71B0 ECFA 8E42      E8B982       E8B982       F3B1B3A3
E6F9     E6F9 E6F9 E6F9 E6F9 64A6 ECFB ECFB ECFB ECFB 6C7B 71C0 ECFB 8E35      E8B8B5       E8B8B5       F3B1B3A4
E6FA     E6FA E6FA E6FA E6FA 64A7 ECFC ECFC ECFC ECFC 6C7C 716A ECFC 8E30      E8B8B0       E8B8B0       F3B1B3A5
E6FB     E6FB E6FB E6FB E6FB 64A8 ECFD ECFD ECFD ECFD 6C7D 71D0 ECFD 8E34      E8B8B4       E8B8B4       F3B1B3A6
E6FC     E6FC E6FC E6FC E6FC 64A9 ECFE ECFE ECFE ECFE 6C7E 71A1 ECFE 8E4A      E8B98A       E8B98A       F3B1B3A7
E740     E740 E740 E740 E740 64AA EDA1 EDA1 EDA1 EDA1 6D21 724F EDA1 8E47      E8B987       E8B987       F3B1B3A8
E741     E741 E741 E741 E741 64AB EDA2 EDA2 EDA2 EDA2 6D22 727F EDA2 8E49      E8B989       E8B989       F3B1B3A9
E742     E742 E742 E742 E742 64AC EDA3 EDA3 EDA3 EDA3 6D23 727B EDA3 8E4C      E8B98C       E8B98C       F3B1B3AA
E743     E743 E743 E743 E743 64AD EDA4 EDA4 EDA4 EDA4 6D24 72E0 EDA4 8E50      E8B990       E8B990       F3B1B3AB
E744     E744 E744 E744 E744 64AE EDA5 EDA5 EDA5 EDA5 6D25 726C EDA5 8E48      E8B988       E8B988       F3B1B3AC
E745     E745 E745 E745 E745 64AF EDA6 EDA6 EDA6 EDA6 6D26 7250 EDA6 8E59      E8B999       E8B999       F3B1B3AD
E746     E746 E746 E746 E746 64B0 EDA7 EDA7 EDA7 EDA7 6D27 727D EDA7 8E64      E8B9A4       E8B9A4       F3B1B3AE
E747     E747 E747 E747 E747 64B1 EDA8 EDA8 EDA8 EDA8 6D28 724D EDA8 8E60      E8B9A0       E8B9A0       F3B1B3AF
E748     E748 E748 E748 E748 50FC EDA9 EDA9 EDA9 EDA9 6D29 725D EDA9 8E2A      E8B8AA       E8B8AA       F3B1B3B0
E749     E749 E749 E749 E749 64B3 EDAA EDAA EDAA EDAA 6D2A 725C EDAA 8E63      E8B9A3       E8B9A3       F3B1B3B1
E74A     E74A E74A E74A E74A 64B2 EDAB EDAB EDAB EDAB 6D2B 724E EDAB 8E55      E8B995       E8B995       F3B1B3B2
E74B     E74B E74B E74B E74B 64B4 EDAC EDAC EDAC EDAC 6D2C 726B EDAC 8E76      E8B9B6       E8B9B6       F3B1B3B3
E74C     E74C E74C E74C E74C 64B5 EDAD EDAD EDAD EDAD 6D2D 7260 EDAD 8E72      E8B9B2       E8B9B2       F3B1B3B4
E74D     E74D E74D E74D E74D 64B7 EDAE EDAE EDAE EDAE 6D2E 724B EDAE 8E7C      E8B9BC       E8B9BC       F3B1B3B5
E74E     E74E E74E E74E E74E 64B8 EDAF EDAF EDAF EDAF 6D2F 7261 EDAF 8E81      E8BA81       E8BA81       F3B1B3B6
E74F     E74F E74F E74F E74F 64B6 EDB0 EDB0 EDB0 EDB0 6D30 72F0 EDB0 8E87      E8BA87       E8BA87       F3B1B3B7
E750     E750 E750 E750 E750 64B9 EDB1 EDB1 EDB1 EDB1 6D31 72F1 EDB1 8E85      E8BA85       E8BA85       F3B1B3B8
E751     E751 E751 E751 E751 64BA EDB2 EDB2 EDB2 EDB2 6D32 72F2 EDB2 8E84      E8BA84       E8BA84       F3B1B3B9
E752     E752 E752 E752 E752 64BB EDB3 EDB3 EDB3 EDB3 6D33 72F3 EDB3 8E8B      E8BA8B       E8BA8B       F3B1B3BA
E753     E753 E753 E753 E753 64BC EDB4 EDB4 EDB4 EDB4 6D34 72F4 EDB4 8E8A      E8BA8A       E8BA8A       F3B1B3BB
E754     E754 E754 E754 E754 64BD EDB5 EDB5 EDB5 EDB5 6D35 72F5 EDB5 8E93      E8BA93       E8BA93       F3B1B3BC
E755     E755 E755 E755 E755 64BE EDB6 EDB6 EDB6 EDB6 6D36 72F6 EDB6 8E91      E8BA91       E8BA91       F3B1B3BD
E756     E756 E756 E756 E756 64BF EDB7 EDB7 EDB7 EDB7 6D37 72F7 EDB7 8E94      E8BA94       E8BA94       F3B1B3BE
E757     E757 E757 E757 E757 64C0 EDB8 EDB8 EDB8 EDB8 6D38 72F8 EDB8 8E99      E8BA99       E8BA99       F3B1B3BF
E758     E758 E758 E758 E758 64C2 EDB9 EDB9 EDB9 EDB9 6D39 72F9 EDB9 8EAA      E8BAAA       E8BAAA       F3B1B480
E759     E759 E759 E759 E759 64C1 EDBA EDBA EDBA EDBA 6D3A 727A EDBA 8EA1      E8BAA1       E8BAA1       F3B1B481
E75A     E75A E75A E75A E75A 5044 EDBB EDBB EDBB EDBB 6D3B 725E EDBB 8EAC      E8BAAC       E8BAAC       F3B1B482
E75B     E75B E75B E75B E75B 537A EDBC EDBC EDBC EDBC 6D3C 724C EDBC 8EB0      E8BAB0       E8BAB0       F3B1B483
E75C     E75C E75C E75C E75C 64C5 EDBD EDBD EDBD EDBD 6D3D 727E EDBD 8EC6      E8BB86       E8BB86       F3B1B484
E75D     E75D E75D E75D E75D 64C3 EDBE EDBE EDBE EDBE 6D3E 726E EDBE 8EB1      E8BAB1       E8BAB1       F3B1B485
E75E     E75E E75E E75E E75E 64C4 EDBF EDBF EDBF EDBF 6D3F 726F EDBF 8EBE      E8BABE       E8BABE       F3B1B486
E75F     E75F E75F E75F E75F 64C6 EDC0 EDC0 EDC0 EDC0 6D40 727C EDC0 8EC5      E8BB85       E8BB85       F3B1B487
E760     E760 E760 E760 E760 64C7 EDC1 EDC1 EDC1 EDC1 6D41 72C1 EDC1 8EC8      E8BB88       E8BB88       F3B1B488
E761     E761 E761 E761 E761 64C8 EDC2 EDC2 EDC2 EDC2 6D42 72C2 EDC2 8ECB      E8BB8B       E8BB8B       F3B1B489
E762     E762 E762 E762 E762 64CA EDC3 EDC3 EDC3 EDC3 6D43 72C3 EDC3 8EDB      E8BB9B       E8BB9B       F3B1B48A
E763     E763 E763 E763 E763 64CB EDC4 EDC4 EDC4 EDC4 6D44 72C4 EDC4 8EE3      E8BBA3       E8BBA3       F3B1B48B
E764     E764 E764 E764 E764 64CC EDC5 EDC5 EDC5 EDC5 6D45 72C5 EDC5 8EFC      E8BBBC       E8BBBC       F3B1B48C
E765     E765 E765 E765 E765 64CD EDC6 EDC6 EDC6 EDC6 6D46 72C6 EDC6 8EFB      E8BBBB       E8BBBB       F3B1B48D
E766     E766 E766 E766 E766 64CE EDC7 EDC7 EDC7 EDC7 6D47 72C7 EDC7 8EEB      E8BBAB       E8BBAB       F3B1B48E
E767     E767 E767 E767 E767 64CF EDC8 EDC8 EDC8 EDC8 6D48 72C8 EDC8 8EFE      E8BBBE       E8BBBE       F3B1B48F
E768     E768 E768 E768 E768 64D0 EDC9 EDC9 EDC9 EDC9 6D49 72C9 EDC9 8F0A      E8BC8A       E8BC8A       F3B1B490
E769     E769 E769 E769 E769 64D2 EDCA EDCA EDCA EDCA 6D4A 72D1 EDCA 8F05      E8BC85       E8BC85       F3B1B491
E76A     E76A E76A E76A E76A 64D3 EDCB EDCB EDCB EDCB 6D4B 72D2 EDCB 8F15      E8BC95       E8BC95       F3B1B492
E76B     E76B E76B E76B E76B 64D4 EDCC EDCC EDCC EDCC 6D4C 72D3 EDCC 8F12      E8BC92       E8BC92       F3B1B493
E76C     E76C E76C E76C E76C 64D7 EDCD EDCD EDCD EDCD 6D4D 72D4 EDCD 8F19      E8BC99       E8BC99       F3B1B494
E76D     E76D E76D E76D E76D 64D5 EDCE EDCE EDCE EDCE 6D4E 72D5 EDCE 8F13      E8BC93       E8BC93       F3B1B495
E76E     E76E E76E E76E E76E 64D6 EDCF EDCF EDCF EDCF 6D4F 72D6 EDCF 8F1C      E8BC9C       E8BC9C       F3B1B496
E76F     E76F E76F E76F E76F 64D8 EDD0 EDD0 EDD0 EDD0 6D50 72D7 EDD0 8F1F      E8BC9F       E8BC9F       F3B1B497
E770     E770 E770 E770 E770 515B EDD1 EDD1 EDD1 EDD1 6D51 72D8 EDD1 8F1B      E8BC9B       E8BC9B       F3B1B498
E771     E771 E771 E771 E771 64D1 EDD2 EDD2 EDD2 EDD2 6D52 72D9 EDD2 8F0C      E8BC8C       E8BC8C       F3B1B499
E772     E772 E772 E772 E772 64D9 EDD3 EDD3 EDD3 EDD3 6D53 72E2 EDD3 8F26      E8BCA6       E8BCA6       F3B1B49A
E773     E773 E773 E773 E773 64DA EDD4 EDD4 EDD4 EDD4 6D54 72E3 EDD4 8F33      E8BCB3       E8BCB3       F3B1B49B
E774     E774 E774 E774 E774 64DB EDD5 EDD5 EDD5 EDD5 6D55 72E4 EDD5 8F3B      E8BCBB       E8BCBB       F3B1B49C
E775     E775 E775 E775 E775 64DC EDD6 EDD6 EDD6 EDD6 6D56 72E5 EDD6 8F39      E8BCB9       E8BCB9       F3B1B49D
E776     E776 E776 E776 E776 64DD EDD7 EDD7 EDD7 EDD7 6D57 72E6 EDD7 8F45      E8BD85       E8BD85       F3B1B49E
E777     E777 E777 E777 E777 64DE EDD8 EDD8 EDD8 EDD8 6D58 72E7 EDD8 8F42      E8BD82       E8BD82       F3B1B49F
E778     E778 E778 E778 E778 64DF EDD9 EDD9 EDD9 EDD9 6D59 72E8 EDD9 8F3E      E8BCBE       E8BCBE       F3B1B4A0
E779     E779 E779 E779 E779 64E2 EDDA EDDA EDDA EDDA 6D5A 72E9 EDDA 8F4C      E8BD8C       E8BD8C       F3B1B4A1
E77A     E77A E77A E77A E77A 64E0 EDDB EDDB EDDB EDDB 6D5B 724A EDDB 8F49      E8BD89       E8BD89       F3B1B4A2
E77B     E77B E77B E77B E77B 64E1 EDDC EDDC EDDC EDDC 6D5C 725B EDDC 8F46      E8BD86       E8BD86       F3B1B4A3
E77C     E77C E77C E77C E77C 64E3 EDDD EDDD EDDD EDDD 6D5D 725A EDDD 8F4E      E8BD8E       E8BD8E       F3B1B4A4
E77D     E77D E77D E77D E77D 64E4 EDDE EDDE EDDE EDDE 6D5E 725F EDDE 8F57      E8BD97       E8BD97       F3B1B4A5
E77E     E77E E77E E77E E77E 64E5 EDDF EDDF EDDF EDDF 6D5F 726D EDDF 8F5C      E8BD9C       E8BD9C       F3B1B4A6
E780     E780 E780 E780 E780 64E6 EDE0 EDE0 EDE0 EDE0 6D60 7279 EDE0 8F62      E8BDA2       E8BDA2       F3B1B4A7
E781     E781 E781 E781 E781 64E7 EDE1 EDE1 EDE1 EDE1 6D61 7257 EDE1 8F63      E8BDA3       E8BDA3       F3B1B4A8
E782     E782 E782 E782 E782 64E8 EDE2 EDE2 EDE2 EDE2 6D62 7259 EDE2 8F64      E8BDA4       E8BDA4       F3B1B4A9
E783     E783 E783 E783 E783 64E9 EDE3 EDE3 EDE3 EDE3 6D63 7262 EDE3 8F9C      E8BE9C       E8BE9C       F3B1B4AA
E784     E784 E784 E784 E784 64EA EDE4 EDE4 EDE4 EDE4 6D64 7263 EDE4 8F9F      E8BE9F       E8BE9F       F3B1B4AB
E785     E785 E785 E785 E785 64EB EDE5 EDE5 EDE5 EDE5 6D65 7264 EDE5 8FA3      E8BEA3       E8BEA3       F3B1B4AC
E786     E786 E786 E786 E786 64EE EDE6 EDE6 EDE6 EDE6 6D66 7265 EDE6 8FAD      E8BEAD       E8BEAD       F3B1B4AD
E787     E787 E787 E787 E787 64EF EDE7 EDE7 EDE7 EDE7 6D67 7266 EDE7 8FAF      E8BEAF       E8BEAF       F3B1B4AE
E788     E788 E788 E788 E788 64F0 EDE8 EDE8 EDE8 EDE8 6D68 7267 EDE8 8FB7      E8BEB7       E8BEB7       F3B1B4AF
E789     E789 E789 E789 E789 64F2 EDE9 EDE9 EDE9 EDE9 6D69 7268 EDE9 8FDA      E8BF9A       E8BF9A       F3B1B4B0
E78A     E78A E78A E78A E78A 64F3 EDEA EDEA EDEA EDEA 6D6A 7269 EDEA 8FE5      E8BFA5       E8BFA5       F3B1B4B1
E78B     E78B E78B E78B E78B 64F4 EDEB EDEB EDEB EDEB 6D6B 7270 EDEB 8FE2      E8BFA2       E8BFA2       F3B1B4B2
E78C     E78C E78C E78C E78C 4E9E EDEC EDEC EDEC EDEC 6D6C 7271 EDEC 8FEA      E8BFAA       E8BFAA       F3B1B4B3
E78D     E78D E78D E78D E78D 64F5 EDED EDED EDED EDED 6D6D 7272 EDED 8FEF      E8BFAF       E8BFAF       F3B1B4B4
E78E     E78E E78E E78E E78E 5186 C6F6 EDEE EDEE C6F6 4676 C69B EDEE 9087      E98287       E98287       F3B1B4B5
E78F     E78F E78F E78F E78F 64F7 EDEF EDEF EDEF EDEF 6D6F 7274 EDEF 8FF4      E8BFB4       E8BFB4       F3B1B4B6
E790     E790 E790 E790 E790 64F8 EDF0 EDF0 EDF0 EDF0 6D70 7275 EDF0 9005      E98085       E98085       F3B1B4B7
E791     E791 E791 E791 E791 64F9 EDF1 EDF1 EDF1 EDF1 6D71 7276 EDF1 8FF9      E8BFB9       E8BFB9       F3B1B4B8
E792     E792 E792 E792 E792 54A9 EDF2 EDF2 EDF2 EDF2 6D72 7277 EDF2 8FFA      E8BFBA       E8BFBA       F3B1B4B9
E793     E793 E793 E793 E793 64FB EDF3 EDF3 EDF3 EDF3 6D73 7278 EDF3 9011      E98091       E98091       F3B1B4BA
E794     E794 E794 E794 E794 64FC EDF4 EDF4 EDF4 EDF4 6D74 7280 EDF4 9015      E98095       E98095       F3B1B4BB
E795     E795 E795 E795 E795 64FE EDF5 EDF5 EDF5 EDF5 6D75 728B EDF5 9021      E980A1       E980A1       F3B1B4BC
E796     E796 E796 E796 E796 6541 EDF6 EDF6 EDF6 EDF6 6D76 729B EDF6 900D      E9808D       E9808D       F3B1B4BD
E797     E797 E797 E797 E797 6542 EDF7 EDF7 EDF7 EDF7 6D77 729C EDF7 901E      E9809E       E9809E       F3B1B4BE
E798     E798 E798 E798 E798 6543 EDF8 EDF8 EDF8 EDF8 6D78 72A0 EDF8 9016      E98096       E98096       F3B1B4BF
E799     E799 E799 E799 E799 6544 EDF9 EDF9 EDF9 EDF9 6D79 72AB EDF9 900B      E9808B       E9808B       F3B1B580
E79A     E79A E79A E79A E79A 6545 EDFA EDFA EDFA EDFA 6D7A 72B0 EDFA 9027      E980A7       E980A7       F3B1B581
E79B     E79B E79B E79B E79B 6546 EDFB EDFB EDFB EDFB 6D7B 72C0 EDFB 9036      E980B6       E980B6       F3B1B582
E79C     E79C E79C E79C E79C 5485 EDFC EDFC EDFC EDFC 6D7C 726A EDFC 9035      E980B5       E980B5       F3B1B583
E79D     E79D E79D E79D E79D 6547 EDFD EDFD EDFD EDFD 6D7D 72D0 EDFD 9039      E980B9       E980B9       F3B1B584
E79E     E79E E79E E79E E79E 64FA EDFE EDFE EDFE EDFE 6D7E 72A1 EDFE 8FF8      E8BFB8       E8BFB8       F3B1B585
E79F     E79F E79F E79F E79F 6548 EEA1 EEA1 EEA1 EEA1 6E21 734F EEA1 904F      E9818F       E9818F       F3B1B586
E7A0     E7A0 E7A0 E7A0 E7A0 654A EEA2 EEA2 EEA2 EEA2 6E22 737F EEA2 9050      E98190       E98190       F3B1B587
E7A1     E7A1 E7A1 E7A1 E7A1 654B EEA3 EEA3 EEA3 EEA3 6E23 737B EEA3 9051      E98191       E98191       F3B1B588
E7A2     E7A2 E7A2 E7A2 E7A2 654C EEA4 EEA4 EEA4 EEA4 6E24 73E0 EEA4 9052      E98192       E98192       F3B1B589
E7A3     E7A3 E7A3 E7A3 E7A3 64FD EEA5 EEA5 EEA5 EEA5 6E25 736C EEA5 900E      E9808E       E9808E       F3B1B58A
E7A4     E7A4 E7A4 E7A4 E7A4 654D EEA6 EEA6 EEA6 EEA6 6E26 7350 EEA6 9049      E98189       E98189       F3B1B58B
E7A5     E7A5 E7A5 E7A5 E7A5 654E EEA7 EEA7 EEA7 EEA7 6E27 737D EEA7 903E      E980BE       E980BE       F3B1B58C
E7A6     E7A6 E7A6 E7A6 E7A6 654F EEA8 EEA8 EEA8 EEA8 6E28 734D EEA8 9056      E98196       E98196       F3B1B58D
E7A7     E7A7 E7A7 E7A7 E7A7 6550 EEA9 EEA9 EEA9 EEA9 6E29 735D EEA9 9058      E98198       E98198       F3B1B58E
E7A8     E7A8 E7A8 E7A8 E7A8 6551 EEAA EEAA EEAA EEAA 6E2A 735C EEAA 905E      E9819E       E9819E       F3B1B58F
E7A9     E7A9 E7A9 E7A9 E7A9 6552 EEAB EEAB EEAB EEAB 6E2B 734E EEAB 9068      E981A8       E981A8       F3B1B590
E7AA     E7AA E7AA E7AA E7AA 6554 EEAC EEAC EEAC EEAC 6E2C 736B EEAC 906F      E981AF       E981AF       F3B1B591
E7AB     E7AB E7AB E7AB E7AB 6555 EEAD EEAD EEAD EEAD 6E2D 7360 EEAD 9076      E981B6       E981B6       F3B1B592
E7AC     E7AC E7AC E7AC E7AC 6556 EEAE EEAE EEAE EEAE 6E2E 734B EEAE 96A8      E99AA8       E99AA8       F3B1B593
E7AD     E7AD E7AD E7AD E7AD 6557 EEAF EEAF EEAF EEAF 6E2F 7361 EEAF 9072      E981B2       E981B2       F3B1B594
E7AE     E7AE E7AE E7AE E7AE 6558 EEB0 EEB0 EEB0 EEB0 6E30 73F0 EEB0 9082      E98282       E98282       F3B1B595
E7AF     E7AF E7AF E7AF E7AF 6559 EEB1 EEB1 EEB1 EEB1 6E31 73F1 EEB1 907D      E981BD       E981BD       F3B1B596
E7B0     E7B0 E7B0 E7B0 E7B0 537B EEB2 EEB2 EEB2 EEB2 6E32 73F2 EEB2 9081      E98281       E98281       F3B1B597
E7B1     E7B1 E7B1 E7B1 E7B1 655B EEB3 EEB3 EEB3 EEB3 6E33 73F3 EEB3 9080      E98280       E98280       F3B1B598
E7B2     E7B2 E7B2 E7B2 E7B2 50EA EEB4 EEB4 EEB4 EEB4 6E34 73F4 EEB4 908A      E9828A       E9828A       F3B1B599
E7B3     E7B3 E7B3 E7B3 E7B3 655A EEB5 EEB5 EEB5 EEB5 6E35 73F5 EEB5 9089      E98289       E98289       F3B1B59A
E7B4     E7B4 E7B4 E7B4 E7B4 655C EEB6 EEB6 EEB6 EEB6 6E36 73F6 EEB6 908F      E9828F       E9828F       F3B1B59B
E7B5     E7B5 E7B5 E7B5 E7B5 51A8 EEB7 EEB7 EEB7 EEB7 6E37 73F7 EEB7 90A8      E982A8       E982A8       F3B1B59C
E7B6     E7B6 E7B6 E7B6 E7B6 655E EEB8 EEB8 EEB8 EEB8 6E38 73F8 EEB8 90AF      E982AF       E982AF       F3B1B59D
E7B7     E7B7 E7B7 E7B7 E7B7 655F EEB9 EEB9 EEB9 EEB9 6E39 73F9 EEB9 90B1      E982B1       E982B1       F3B1B59E
E7B8     E7B8 E7B8 E7B8 E7B8 6560 EEBA EEBA EEBA EEBA 6E3A 737A EEBA 90B5      E982B5       E982B5       F3B1B59F
E7B9     E7B9 E7B9 E7B9 E7B9 6561 EEBB EEBB EEBB EEBB 6E3B 735E EEBB 90E2      E983A2       E983A2       F3B1B5A0
E7BA     E7BA E7BA E7BA E7BA 6562 EEBC EEBC EEBC EEBC 6E3C 734C EEBC 90E4      E983A4       E983A4       F3B1B5A1
E7BB     E7BB E7BB E7BB E7BB 655D EEBD EEBD EEBD EEBD 6E3D 737E EEBD 6248      E68988       E68988       F3B1B5A2
E7BC     E7BC E7BC E7BC E7BC 6563 EEBE EEBE EEBE EEBE 6E3E 736E EEBE 90DB      E9839B       E9839B       F3B1B5A3
E7BD     E7BD E7BD E7BD E7BD 6565 EEBF EEBF EEBF EEBF 6E3F 736F EEBF 9102      E98482       E98482       F3B1B5A4
E7BE     E7BE E7BE E7BE E7BE 6568 EEC0 EEC0 EEC0 EEC0 6E40 737C EEC0 9112      E98492       E98492       F3B1B5A5
E7BF     E7BF E7BF E7BF E7BF 6569 EEC1 EEC1 EEC1 EEC1 6E41 73C1 EEC1 9119      E98499       E98499       F3B1B5A6
E7C0     E7C0 E7C0 E7C0 E7C0 656A EEC2 EEC2 EEC2 EEC2 6E42 73C2 EEC2 9132      E984B2       E984B2       F3B1B5A7
E7C1     E7C1 E7C1 E7C1 E7C1 656C EEC3 EEC3 EEC3 EEC3 6E43 73C3 EEC3 9130      E984B0       E984B0       F3B1B5A8
E7C2     E7C2 E7C2 E7C2 E7C2 656D EEC4 EEC4 EEC4 EEC4 6E44 73C4 EEC4 914A      E9858A       E9858A       F3B1B5A9
E7C3     E7C3 E7C3 E7C3 E7C3 656E EEC5 EEC5 EEC5 EEC5 6E45 73C5 EEC5 9156      E98596       E98596       F3B1B5AA
E7C4     E7C4 E7C4 E7C4 E7C4 656F EEC6 EEC6 EEC6 EEC6 6E46 73C6 EEC6 9158      E98598       E98598       F3B1B5AB
E7C5     E7C5 E7C5 E7C5 E7C5 6570 EEC7 EEC7 EEC7 EEC7 6E47 73C7 EEC7 9163      E985A3       E985A3       F3B1B5AC
E7C6     E7C6 E7C6 E7C6 E7C6 6571 EEC8 EEC8 EEC8 EEC8 6E48 73C8 EEC8 9165      E985A5       E985A5       F3B1B5AD
E7C7     E7C7 E7C7 E7C7 E7C7 6572 EEC9 EEC9 EEC9 EEC9 6E49 73C9 EEC9 9169      E985A9       E985A9       F3B1B5AE
E7C8     E7C8 E7C8 E7C8 E7C8 6573 EECA EECA EECA EECA 6E4A 73D1 EECA 9173      E985B3       E985B3       F3B1B5AF
E7C9     E7C9 E7C9 E7C9 E7C9 6574 EECB EECB EECB EECB 6E4B 73D2 EECB 9172      E985B2       E985B2       F3B1B5B0
E7CA     E7CA E7CA E7CA E7CA 6575 EECC EECC EECC EECC 6E4C 73D3 EECC 918B      E9868B       E9868B       F3B1B5B1
E7CB     E7CB E7CB E7CB E7CB 6576 EECD EECD EECD EECD 6E4D 73D4 EECD 9189      E98689       E98689       F3B1B5B2
E7CC     E7CC E7CC E7CC E7CC 6577 EECE EECE EECE EECE 6E4E 73D5 EECE 9182      E98682       E98682       F3B1B5B3
E7CD     E7CD E7CD E7CD E7CD 6578 EECF EECF EECF EECF 6E4F 73D6 EECF 91A2      E986A2       E986A2       F3B1B5B4
E7CE     E7CE E7CE E7CE E7CE 6579 EED0 EED0 EED0 EED0 6E50 73D7 EED0 91AB      E986AB       E986AB       F3B1B5B5
E7CF     E7CF E7CF E7CF E7CF 657A EED1 EED1 EED1 EED1 6E51 73D8 EED1 91AF      E986AF       E986AF       F3B1B5B6
E7D0     E7D0 E7D0 E7D0 E7D0 657B EED2 EED2 EED2 EED2 6E52 73D9 EED2 91AA      E986AA       E986AA       F3B1B5B7
E7D1     E7D1 E7D1 E7D1 E7D1 657C EED3 EED3 EED3 EED3 6E53 73E2 EED3 91B5      E986B5       E986B5       F3B1B5B8
E7D2     E7D2 E7D2 E7D2 E7D2 657D EED4 EED4 EED4 EED4 6E54 73E3 EED4 91B4      E986B4       E986B4       F3B1B5B9
E7D3     E7D3 E7D3 E7D3 E7D3 657E EED5 EED5 EED5 EED5 6E55 73E4 EED5 91BA      E986BA       E986BA       F3B1B5BA
E7D4     E7D4 E7D4 E7D4 E7D4 657F EED6 EED6 EED6 EED6 6E56 73E5 EED6 91C0      E98780       E98780       F3B1B5BB
E7D5     E7D5 E7D5 E7D5 E7D5 6580 EED7 EED7 EED7 EED7 6E57 73E6 EED7 91C1      E98781       E98781       F3B1B5BC
E7D6     E7D6 E7D6 E7D6 E7D6 5252 EED8 EED8 EED8 EED8 6E58 73E7 EED8 91C9      E98789       E98789       F3B1B5BD
E7D7     E7D7 E7D7 E7D7 E7D7 6581 EED9 EED9 EED9 EED9 6E59 73E8 EED9 91CB      E9878B       E9878B       F3B1B5BE
E7D8     E7D8 E7D8 E7D8 E7D8 6582 EEDA EEDA EEDA EEDA 6E5A 73E9 EEDA 91D0      E98790       E98790       F3B1B5BF
E7D9     E7D9 E7D9 E7D9 E7D9 6587 EEDB EEDB EEDB EEDB 6E5B 734A EEDB 91D6      E98796       E98796       F3B1B680
E7DA     E7DA E7DA E7DA E7DA 6588 EEDC EEDC EEDC EEDC 6E5C 735B EEDC 91DF      E9879F       E9879F       F3B1B681
E7DB     E7DB E7DB E7DB E7DB 6589 EEDD EEDD EEDD EEDD 6E5D 735A EEDD 91E1      E987A1       E987A1       F3B1B682
E7DC     E7DC E7DC E7DC E7DC 6584 EEDE EEDE EEDE EEDE 6E5E 735F EEDE 91DB      E9879B       E9879B       F3B1B683
E7DD     E7DD E7DD E7DD E7DD 51D0 EEDF EEDF EEDF EEDF 6E5F 736D EEDF 91FC      E987BC       E987BC       F3B1B684
E7DE     E7DE E7DE E7DE E7DE 658B EEE0 EEE0 EEE0 EEE0 6E60 7379 EEE0 91F5      E987B5       E987B5       F3B1B685
E7DF     E7DF E7DF E7DF E7DF 658E EEE1 EEE1 EEE1 EEE1 6E61 7357 EEE1 91F6      E987B6       E987B6       F3B1B686
E7E0     E7E0 E7E0 E7E0 E7E0 6591 EEE2 EEE2 EEE2 EEE2 6E62 7359 EEE2 921E      E9889E       E9889E       F3B1B687
E7E1     E7E1 E7E1 E7E1 E7E1 6592 EEE3 EEE3 EEE3 EEE3 6E63 7362 EEE3 91FF      E987BF       E987BF       F3B1B688
E7E2     E7E2 E7E2 E7E2 E7E2 6594 EEE4 EEE4 EEE4 EEE4 6E64 7363 EEE4 9214      E98894       E98894       F3B1B689
E7E3     E7E3 E7E3 E7E3 E7E3 6596 EEE5 EEE5 EEE5 EEE5 6E65 7364 EEE5 922C      E988AC       E988AC       F3B1B68A
E7E4     E7E4 E7E4 E7E4 E7E4 6597 EEE6 EEE6 EEE6 EEE6 6E66 7365 EEE6 9215      E98895       E98895       F3B1B68B
E7E5     E7E5 E7E5 E7E5 E7E5 5269 EEE7 EEE7 EEE7 EEE7 6E67 7366 EEE7 9211      E98891       E98891       F3B1B68C
E7E6     E7E6 E7E6 E7E6 E7E6 5393 EEE8 EEE8 EEE8 EEE8 6E68 7367 EEE8 925E      E9899E       E9899E       F3B1B68D
E7E7     E7E7 E7E7 E7E7 E7E7 6599 EEE9 EEE9 EEE9 EEE9 6E69 7368 EEE9 9257      E98997       E98997       F3B1B68E
E7E8     E7E8 E7E8 E7E8 E7E8 659A EEEA EEEA EEEA EEEA 6E6A 7369 EEEA 9245      E98985       E98985       F3B1B68F
E7E9     E7E9 E7E9 E7E9 E7E9 659C EEEB EEEB EEEB EEEB 6E6B 7370 EEEB 9249      E98989       E98989       F3B1B690
E7EA     E7EA E7EA E7EA E7EA 659D EEEC EEEC EEEC EEEC 6E6C 7371 EEEC 9264      E989A4       E989A4       F3B1B691
E7EB     E7EB E7EB E7EB E7EB 65A0 EEED EEED EEED EEED 6E6D 7372 EEED 9248      E98988       E98988       F3B1B692
E7EC     E7EC E7EC E7EC E7EC 65B1 EEEE EEEE EEEE EEEE 6E6E 7373 EEEE 9295      E98A95       E98A95       F3B1B693
E7ED     E7ED E7ED E7ED E7ED 65A4 EEEF EEEF EEEF EEEF 6E6F 7374 EEEF 923F      E988BF       E988BF       F3B1B694
E7EE     E7EE E7EE E7EE E7EE 65A7 EEF0 EEF0 EEF0 EEF0 6E70 7375 EEF0 924B      E9898B       E9898B       F3B1B695
E7EF     E7EF E7EF E7EF E7EF 65A2 EEF1 EEF1 EEF1 EEF1 6E71 7376 EEF1 9250      E98990       E98990       F3B1B696
E7F0     E7F0 E7F0 E7F0 E7F0 65AA EEF2 EEF2 EEF2 EEF2 6E72 7377 EEF2 929C      E98A9C       E98A9C       F3B1B697
E7F1     E7F1 E7F1 E7F1 E7F1 65AE EEF3 EEF3 EEF3 EEF3 6E73 7378 EEF3 9296      E98A96       E98A96       F3B1B698
E7F2     E7F2 E7F2 E7F2 E7F2 65AF EEF4 EEF4 EEF4 EEF4 6E74 7380 EEF4 9293      E98A93       E98A93       F3B1B699
E7F3     E7F3 E7F3 E7F3 E7F3 65B0 EEF5 EEF5 EEF5 EEF5 6E75 738B EEF5 929B      E98A9B       E98A9B       F3B1B69A
E7F4     E7F4 E7F4 E7F4 E7F4 65A9 EEF6 EEF6 EEF6 EEF6 6E76 739B EEF6 925A      E9899A       E9899A       F3B1B69B
E7F5     E7F5 E7F5 E7F5 E7F5 65B3 EEF7 EEF7 EEF7 EEF7 6E77 739C EEF7 92CF      E98B8F       E98B8F       F3B1B69C
E7F6     E7F6 E7F6 E7F6 E7F6 65BA EEF8 EEF8 EEF8 EEF8 6E78 73A0 EEF8 92B9      E98AB9       E98AB9       F3B1B69D
E7F7     E7F7 E7F7 E7F7 E7F7 65BB EEF9 EEF9 EEF9 EEF9 6E79 73AB EEF9 92B7      E98AB7       E98AB7       F3B1B69E
E7F8     E7F8 E7F8 E7F8 E7F8 65B2 EEFA EEFA EEFA EEFA 6E7A 73B0 EEFA 92E9      E98BA9       E98BA9       F3B1B69F
E7F9     E7F9 E7F9 E7F9 E7F9 65C0 EEFB EEFB EEFB EEFB 6E7B 73C0 EEFB 930F      E98C8F       E98C8F       F3B1B6A0
E7FA     E7FA E7FA E7FA E7FA 65C2 EEFC EEFC EEFC EEFC 6E7C 736A EEFC 92FA      E98BBA       E98BBA       F3B1B6A1
E7FB     E7FB E7FB E7FB E7FB 65C4 EEFD EEFD EEFD EEFD 6E7D 73D0 EEFD 9344      E98D84       E98D84       F3B1B6A2
E7FC     E7FC E7FC E7FC E7FC 5283 EEFE EEFE EEFE EEFE 6E7E 73A1 EEFE 932E      E98CAE       E98CAE       F3B1B6A3
E840     E840 E840 E840 E840 65C7 EFA1 EFA1 EFA1 EFA1 6F21 744F EFA1 9319      E98C99       E98C99       F3B1B6A4
E841     E841 E841 E841 E841 65CA EFA2 EFA2 EFA2 EFA2 6F22 747F EFA2 9322      E98CA2       E98CA2       F3B1B6A5
E842     E842 E842 E842 E842 65CB EFA3 EFA3 EFA3 EFA3 6F23 747B EFA3 931A      E98C9A       E98C9A       F3B1B6A6
E843     E843 E843 E843 E843 65CD EFA4 EFA4 EFA4 EFA4 6F24 74E0 EFA4 9323      E98CA3       E98CA3       F3B1B6A7
E844     E844 E844 E844 E844 65BE EFA5 EFA5 EFA5 EFA5 6F25 746C EFA5 933A      E98CBA       E98CBA       F3B1B6A8
E845     E845 E845 E845 E845 65BF EFA6 EFA6 EFA6 EFA6 6F26 7450 EFA6 9335      E98CB5       E98CB5       F3B1B6A9
E846     E846 E846 E846 E846 65CF EFA7 EFA7 EFA7 EFA7 6F27 747D EFA7 933B      E98CBB       E98CBB       F3B1B6AA
E847     E847 E847 E847 E847 52AD EFA8 EFA8 EFA8 EFA8 6F28 744D EFA8 935C      E98D9C       E98D9C       F3B1B6AB
E848     E848 E848 E848 E848 65D1 EFA9 EFA9 EFA9 EFA9 6F29 745D EFA9 9360      E98DA0       E98DA0       F3B1B6AC
E849     E849 E849 E849 E849 65D2 EFAA EFAA EFAA EFAA 6F2A 745C EFAA 937C      E98DBC       E98DBC       F3B1B6AD
E84A     E84A E84A E84A E84A 65D3 EFAB EFAB EFAB EFAB 6F2B 744E EFAB 936E      E98DAE       E98DAE       F3B1B6AE
E84B     E84B E84B E84B E84B 65D4 EFAC EFAC EFAC EFAC 6F2C 746B EFAC 9356      E98D96       E98D96       F3B1B6AF
E84C     E84C E84C E84C E84C 65D7 EFAD EFAD EFAD EFAD 6F2D 7460 EFAD 93B0      E98EB0       E98EB0       F3B1B6B0
E84D     E84D E84D E84D E84D 536E EFAE EFAE EFAE EFAE 6F2E 744B EFAE 93AC      E98EAC       E98EAC       F3B1B6B1
E84E     E84E E84E E84E E84E 65D9 EFAF EFAF EFAF EFAF 6F2F 7461 EFAF 93AD      E98EAD       E98EAD       F3B1B6B2
E84F     E84F E84F E84F E84F 65DA EFB0 EFB0 EFB0 EFB0 6F30 74F0 EFB0 9394      E98E94       E98E94       F3B1B6B3
E850     E850 E850 E850 E850 65D6 EFB1 EFB1 EFB1 EFB1 6F31 74F1 EFB1 93B9      E98EB9       E98EB9       F3B1B6B4
E851     E851 E851 E851 E851 65DC EFB2 EFB2 EFB2 EFB2 6F32 74F2 EFB2 93D6      E98F96       E98F96       F3B1B6B5
E852     E852 E852 E852 E852 65DE EFB3 EFB3 EFB3 EFB3 6F33 74F3 EFB3 93D7      E98F97       E98F97       F3B1B6B6
E853     E853 E853 E853 E853 65DF EFB4 EFB4 EFB4 EFB4 6F34 74F4 EFB4 93E8      E98FA8       E98FA8       F3B1B6B7
E854     E854 E854 E854 E854 65E0 EFB5 EFB5 EFB5 EFB5 6F35 74F5 EFB5 93E5      E98FA5       E98FA5       F3B1B6B8
E855     E855 E855 E855 E855 65E1 EFB6 EFB6 EFB6 EFB6 6F36 74F6 EFB6 93D8      E98F98       E98F98       F3B1B6B9
E856     E856 E856 E856 E856 65E2 EFB7 EFB7 EFB7 EFB7 6F37 74F7 EFB7 93C3      E98F83       E98F83       F3B1B6BA
E857     E857 E857 E857 E857 65E3 EFB8 EFB8 EFB8 EFB8 6F38 74F8 EFB8 93DD      E98F9D       E98F9D       F3B1B6BB
E858     E858 E858 E858 E858 65E5 EFB9 EFB9 EFB9 EFB9 6F39 74F9 EFB9 93D0      E98F90       E98F90       F3B1B6BC
E859     E859 E859 E859 E859 65DB EFBA EFBA EFBA EFBA 6F3A 747A EFBA 93C8      E98F88       E98F88       F3B1B6BD
E85A     E85A E85A E85A E85A 65E6 EFBB EFBB EFBB EFBB 6F3B 745E EFBB 93E4      E98FA4       E98FA4       F3B1B6BE
E85B     E85B E85B E85B E85B 65E7 EFBC EFBC EFBC EFBC 6F3C 744C EFBC 941A      E9909A       E9909A       F3B1B6BF
E85C     E85C E85C E85C E85C 65E9 EFBD EFBD EFBD EFBD 6F3D 747E EFBD 9414      E99094       E99094       F3B1B780
E85D     E85D E85D E85D E85D 65EA EFBE EFBE EFBE EFBE 6F3E 746E EFBE 9413      E99093       E99093       F3B1B781
E85E     E85E E85E E85E E85E 65EC EFBF EFBF EFBF EFBF 6F3F 746F EFBF 9403      E99083       E99083       F3B1B782
E85F     E85F E85F E85F E85F 65ED EFC0 EFC0 EFC0 EFC0 6F40 747C EFC0 9407      E99087       E99087       F3B1B783
E860     E860 E860 E860 E860 51CD EFC1 EFC1 EFC1 EFC1 6F41 74C1 EFC1 9410      E99090       E99090       F3B1B784
E861     E861 E861 E861 E861 65EE EFC2 EFC2 EFC2 EFC2 6F42 74C2 EFC2 9436      E990B6       E990B6       F3B1B785
E862     E862 E862 E862 E862 65EF EFC3 EFC3 EFC3 EFC3 6F43 74C3 EFC3 942B      E990AB       E990AB       F3B1B786
E863     E863 E863 E863 E863 5156 EFC4 EFC4 EFC4 EFC4 6F44 74C4 EFC4 9435      E990B5       E990B5       F3B1B787
E864     E864 E864 E864 E864 65EB EFC5 EFC5 EFC5 EFC5 6F45 74C5 EFC5 9421      E990A1       E990A1       F3B1B788
E865     E865 E865 E865 E865 65F1 EFC6 EFC6 EFC6 EFC6 6F46 74C6 EFC6 943A      E990BA       E990BA       F3B1B789
E866     E866 E866 E866 E866 65F2 EFC7 EFC7 EFC7 EFC7 6F47 74C7 EFC7 9441      E99181       E99181       F3B1B78A
E867     E867 E867 E867 E867 65F3 EFC8 EFC8 EFC8 EFC8 6F48 74C8 EFC8 9452      E99192       E99192       F3B1B78B
E868     E868 E868 E868 E868 65F5 EFC9 EFC9 EFC9 EFC9 6F49 74C9 EFC9 9444      E99184       E99184       F3B1B78C
E869     E869 E869 E869 E869 65F7 EFCA EFCA EFCA EFCA 6F4A 74D1 EFCA 945B      E9919B       E9919B       F3B1B78D
E86A     E86A E86A E86A E86A 65F9 EFCB EFCB EFCB EFCB 6F4B 74D2 EFCB 9460      E991A0       E991A0       F3B1B78E
E86B     E86B E86B E86B E86B 65FA EFCC EFCC EFCC EFCC 6F4C 74D3 EFCC 9462      E991A2       E991A2       F3B1B78F
E86C     E86C E86C E86C E86C 65FB EFCD EFCD EFCD EFCD 6F4D 74D4 EFCD 945E      E9919E       E9919E       F3B1B790
E86D     E86D E86D E86D E86D 65FC EFCE EFCE EFCE EFCE 6F4E 74D5 EFCE 946A      E991AA       E991AA       F3B1B791
E86E     E86E E86E E86E E86E 6598 EFCF EFCF EFCF EFCF 6F4F 74D6 EFCF 9229      E988A9       E988A9       F3B1B792
E86F     E86F E86F E86F E86F 65FE EFD0 EFD0 EFD0 EFD0 6F50 74D7 EFD0 9470      E991B0       E991B0       F3B1B793
E870     E870 E870 E870 E870 65FD EFD1 EFD1 EFD1 EFD1 6F51 74D8 EFD1 9475      E991B5       E991B5       F3B1B794
E871     E871 E871 E871 E871 6641 EFD2 EFD2 EFD2 EFD2 6F52 74D9 EFD2 9477      E991B7       E991B7       F3B1B795
E872     E872 E872 E872 E872 6643 EFD3 EFD3 EFD3 EFD3 6F53 74E2 EFD3 947D      E991BD       E991BD       F3B1B796
E873     E873 E873 E873 E873 65F8 EFD4 EFD4 EFD4 EFD4 6F54 74E3 EFD4 945A      E9919A       E9919A       F3B1B797
E874     E874 E874 E874 E874 6644 EFD5 EFD5 EFD5 EFD5 6F55 74E4 EFD5 947C      E991BC       E991BC       F3B1B798
E875     E875 E875 E875 E875 6645 EFD6 EFD6 EFD6 EFD6 6F56 74E5 EFD6 947E      E991BE       E991BE       F3B1B799
E876     E876 E876 E876 E876 6646 EFD7 EFD7 EFD7 EFD7 6F57 74E6 EFD7 9481      E99281       E99281       F3B1B79A
E877     E877 E877 E877 E877 6642 EFD8 EFD8 EFD8 EFD8 6F58 74E7 EFD8 947F      E991BF       E991BF       F3B1B79B
E878     E878 E878 E878 E878 6647 EFD9 EFD9 EFD9 EFD9 6F59 74E8 EFD9 9582      E99682       E99682       F3B1B79C
E879     E879 E879 E879 E879 6648 EFDA EFDA EFDA EFDA 6F5A 74E9 EFDA 9587      E99687       E99687       F3B1B79D
E87A     E87A E87A E87A E87A 6649 EFDB EFDB EFDB EFDB 6F5B 744A EFDB 958A      E9968A       E9968A       F3B1B79E
E87B     E87B E87B E87B E87B 664B EFDC EFDC EFDC EFDC 6F5C 745B EFDC 9594      E99694       E99694       F3B1B79F
E87C     E87C E87C E87C E87C 664C EFDD EFDD EFDD EFDD 6F5D 745A EFDD 9596      E99696       E99696       F3B1B7A0
E87D     E87D E87D E87D E87D 664D EFDE EFDE EFDE EFDE 6F5E 745F EFDE 9598      E99698       E99698       F3B1B7A1
E87E     E87E E87E E87E E87E 664E EFDF EFDF EFDF EFDF 6F5F 746D EFDF 9599      E99699       E99699       F3B1B7A2
E880     E880 E880 E880 E880 664F EFE0 EFE0 EFE0 EFE0 6F60 7479 EFE0 95A0      E996A0       E996A0       F3B1B7A3
E881     E881 E881 E881 E881 6650 EFE1 EFE1 EFE1 EFE1 6F61 7457 EFE1 95A8      E996A8       E996A8       F3B1B7A4
E882     E882 E882 E882 E882 6651 EFE2 EFE2 EFE2 EFE2 6F62 7459 EFE2 95A7      E996A7       E996A7       F3B1B7A5
E883     E883 E883 E883 E883 6652 EFE3 EFE3 EFE3 EFE3 6F63 7462 EFE3 95AD      E996AD       E996AD       F3B1B7A6
E884     E884 E884 E884 E884 6653 EFE4 EFE4 EFE4 EFE4 6F64 7463 EFE4 95BC      E996BC       E996BC       F3B1B7A7
E885     E885 E885 E885 E885 6654 EFE5 EFE5 EFE5 EFE5 6F65 7464 EFE5 95BB      E996BB       E996BB       F3B1B7A8
E886     E886 E886 E886 E886 6655 EFE6 EFE6 EFE6 EFE6 6F66 7465 EFE6 95B9      E996B9       E996B9       F3B1B7A9
E887     E887 E887 E887 E887 6656 EFE7 EFE7 EFE7 EFE7 6F67 7466 EFE7 95BE      E996BE       E996BE       F3B1B7AA
E888     E888 E888 E888 E888 6657 EFE8 EFE8 EFE8 EFE8 6F68 7467 EFE8 95CA      E9978A       E9978A       F3B1B7AB
E889     E889 E889 E889 E889 6658 EFE9 EFE9 EFE9 EFE9 6F69 7468 EFE9 6FF6      E6BFB6       E6BFB6       F3B1B7AC
E88A     E88A E88A E88A E88A 6659 EFEA EFEA EFEA EFEA 6F6A 7469 EFEA 95C3      E99783       E99783       F3B1B7AD
E88B     E88B E88B E88B E88B 665A EFEB EFEB EFEB EFEB 6F6B 7470 EFEB 95CD      E9978D       E9978D       F3B1B7AE
E88C     E88C E88C E88C E88C 665B EFEC EFEC EFEC EFEC 6F6C 7471 EFEC 95CC      E9978C       E9978C       F3B1B7AF
E88D     E88D E88D E88D E88D 665C EFED EFED EFED EFED 6F6D 7472 EFED 95D5      E99795       E99795       F3B1B7B0
E88E     E88E E88E E88E E88E 665D EFEE EFEE EFEE EFEE 6F6E 7473 EFEE 95D4      E99794       E99794       F3B1B7B1
E88F     E88F E88F E88F E88F 665E EFEF EFEF EFEF EFEF 6F6F 7474 EFEF 95D6      E99796       E99796       F3B1B7B2
E890     E890 E890 E890 E890 665F EFF0 EFF0 EFF0 EFF0 6F70 7475 EFF0 95DC      E9979C       E9979C       F3B1B7B3
E891     E891 E891 E891 E891 6660 EFF1 EFF1 EFF1 EFF1 6F71 7476 EFF1 95E1      E997A1       E997A1       F3B1B7B4
E892     E892 E892 E892 E892 6661 EFF2 EFF2 EFF2 EFF2 6F72 7477 EFF2 95E5      E997A5       E997A5       F3B1B7B5
E893     E893 E893 E893 E893 6662 EFF3 EFF3 EFF3 EFF3 6F73 7478 EFF3 95E2      E997A2       E997A2       F3B1B7B6
E894     E894 E894 E894 E894 6663 EFF4 EFF4 EFF4 EFF4 6F74 7480 EFF4 9621      E998A1       E998A1       F3B1B7B7
E895     E895 E895 E895 E895 6664 EFF5 EFF5 EFF5 EFF5 6F75 748B EFF5 9628      E998A8       E998A8       F3B1B7B8
E896     E896 E896 E896 E896 6665 EFF6 EFF6 EFF6 EFF6 6F76 749B EFF6 962E      E998AE       E998AE       F3B1B7B9
E897     E897 E897 E897 E897 6666 EFF7 EFF7 EFF7 EFF7 6F77 749C EFF7 962F      E998AF       E998AF       F3B1B7BA
E898     E898 E898 E898 E898 6667 EFF8 EFF8 EFF8 EFF8 6F78 74A0 EFF8 9642      E99982       E99982       F3B1B7BB
E899     E899 E899 E899 E899 6669 EFF9 EFF9 EFF9 EFF9 6F79 74AB EFF9 964C      E9998C       E9998C       F3B1B7BC
E89A     E89A E89A E89A E89A 6668 EFFA EFFA EFFA EFFA 6F7A 74B0 EFFA 964F      E9998F       E9998F       F3B1B7BD
E89B     E89B E89B E89B E89B 666A EFFB EFFB EFFB EFFB 6F7B 74C0 EFFB 964B      E9998B       E9998B       F3B1B7BE
E89C     E89C E89C E89C E89C 666F EFFC EFFC EFFC EFFC 6F7C 746A EFFC 9677      E999B7       E999B7       F3B1B7BF
E89D     E89D E89D E89D E89D 666B EFFD EFFD EFFD EFFD 6F7D 74D0 EFFD 965C      E9999C       E9999C       F3B1B880
E89E     E89E E89E E89E E89E 5272 EFFE EFFE EFFE EFFE 6F7E 74A1 EFFE 965E      E9999E       E9999E       F3B1B881
E89F     E89F E89F E89F E89F 666C F0A1 F0A1 F0A1 F0A1 7021 754F F0A1 965D      E9999D       E9999D       F3B1B882
E8A0     E8A0 E8A0 E8A0 E8A0 666D F0A2 F0A2 F0A2 F0A2 7022 757F F0A2 965F      E9999F       E9999F       F3B1B883
E8A1     E8A1 E8A1 E8A1 E8A1 666E F0A3 F0A3 F0A3 F0A3 7023 757B F0A3 9666      E999A6       E999A6       F3B1B884
E8A2     E8A2 E8A2 E8A2 E8A2 6670 F0A4 F0A4 F0A4 F0A4 7024 75E0 F0A4 9672      E999B2       E999B2       F3B1B885
E8A3     E8A3 E8A3 E8A3 E8A3 6671 F0A5 F0A5 F0A5 F0A5 7025 756C F0A5 966C      E999AC       E999AC       F3B1B886
E8A4     E8A4 E8A4 E8A4 E8A4 6672 F0A6 F0A6 F0A6 F0A6 7026 7550 F0A6 968D      E99A8D       E99A8D       F3B1B887
E8A5     E8A5 E8A5 E8A5 E8A5 6675 F0A7 F0A7 F0A7 F0A7 7027 757D F0A7 9698      E99A98       E99A98       F3B1B888
E8A6     E8A6 E8A6 E8A6 E8A6 6676 F0A8 F0A8 F0A8 F0A8 7028 754D F0A8 9695      E99A95       E99A95       F3B1B889
E8A7     E8A7 E8A7 E8A7 E8A7 6677 F0A9 F0A9 F0A9 F0A9 7029 755D F0A9 9697      E99A97       E99A97       F3B1B88A
E8A8     E8A8 E8A8 E8A8 E8A8 667B F0AA F0AA F0AA F0AA 702A 755C F0AA 96AA      E99AAA       E99AAA       F3B1B88B
E8A9     E8A9 E8A9 E8A9 E8A9 667A F0AB F0AB F0AB F0AB 702B 754E F0AB 96A7      E99AA7       E99AA7       F3B1B88C
E8AA     E8AA E8AA E8AA E8AA 667C F0AC F0AC F0AC F0AC 702C 756B F0AC 96B1      E99AB1       E99AB1       F3B1B88D
E8AB     E8AB E8AB E8AB E8AB 667D F0AD F0AD F0AD F0AD 702D 7560 F0AD 96B2      E99AB2       E99AB2       F3B1B88E
E8AC     E8AC E8AC E8AC E8AC 667E F0AE F0AE F0AE F0AE 702E 754B F0AE 96B0      E99AB0       E99AB0       F3B1B88F
E8AD     E8AD E8AD E8AD E8AD 6680 F0AF F0AF F0AF F0AF 702F 7561 F0AF 96B4      E99AB4       E99AB4       F3B1B890
E8AE     E8AE E8AE E8AE E8AE 6681 F0B0 F0B0 F0B0 F0B0 7030 75F0 F0B0 96B6      E99AB6       E99AB6       F3B1B891
E8AF     E8AF E8AF E8AF E8AF 6682 F0B1 F0B1 F0B1 F0B1 7031 75F1 F0B1 96B8      E99AB8       E99AB8       F3B1B892
E8B0     E8B0 E8B0 E8B0 E8B0 6683 F0B2 F0B2 F0B2 F0B2 7032 75F2 F0B2 96B9      E99AB9       E99AB9       F3B1B893
E8B1     E8B1 E8B1 E8B1 E8B1 6684 F0B3 F0B3 F0B3 F0B3 7033 75F3 F0B3 96CE      E99B8E       E99B8E       F3B1B894
E8B2     E8B2 E8B2 E8B2 E8B2 6685 F0B4 F0B4 F0B4 F0B4 7034 75F4 F0B4 96CB      E99B8B       E99B8B       F3B1B895
E8B3     E8B3 E8B3 E8B3 E8B3 5477 F0B5 F0B5 F0B5 F0B5 7035 75F5 F0B5 96C9      E99B89       E99B89       F3B1B896
E8B4     E8B4 E8B4 E8B4 E8B4 525B F0B6 F0B6 F0B6 F0B6 7036 75F6 F0B6 96CD      E99B8D       E99B8D       F3B1B897
E8B5     E8B5 E8B5 E8B5 E8B5 63A1 F0B7 F0B7 F0B7 F0B7 7037 75F7 F0B7 894D      E8A58D       E8A58D       F3B1B898
E8B6     E8B6 E8B6 E8B6 E8B6 6687 F0B8 F0B8 F0B8 F0B8 7038 75F8 F0B8 96DC      E99B9C       E99B9C       F3B1B899
E8B7     E8B7 E8B7 E8B7 E8B7 668E F0B9 F0B9 F0B9 F0B9 7039 75F9 F0B9 970D      E99C8D       E99C8D       F3B1B89A
E8B8     E8B8 E8B8 E8B8 E8B8 6686 F0BA F0BA F0BA F0BA 703A 757A F0BA 96D5      E99B95       E99B95       F3B1B89B
E8B9     E8B9 E8B9 E8B9 E8B9 6689 F0BB F0BB F0BB F0BB 703B 755E F0BB 96F9      E99BB9       E99BB9       F3B1B89C
E8BA     E8BA E8BA E8BA E8BA 668A F0BC F0BC F0BC F0BC 703C 754C F0BC 9704      E99C84       E99C84       F3B1B89D
E8BB     E8BB E8BB E8BB E8BB 668B F0BD F0BD F0BD F0BD 703D 757E F0BD 9706      E99C86       E99C86       F3B1B89E
E8BC     E8BC E8BC E8BC E8BC 668C F0BE F0BE F0BE F0BE 703E 756E F0BE 9708      E99C88       E99C88       F3B1B89F
E8BD     E8BD E8BD E8BD E8BD 668F F0BF F0BF F0BF F0BF 703F 756F F0BF 9713      E99C93       E99C93       F3B1B8A0
E8BE     E8BE E8BE E8BE E8BE 6690 F0C0 F0C0 F0C0 F0C0 7040 757C F0C0 970E      E99C8E       E99C8E       F3B1B8A1
E8BF     E8BF E8BF E8BF E8BF 6691 F0C1 F0C1 F0C1 F0C1 7041 75C1 F0C1 9711      E99C91       E99C91       F3B1B8A2
E8C0     E8C0 E8C0 E8C0 E8C0 6692 F0C2 F0C2 F0C2 F0C2 7042 75C2 F0C2 970F      E99C8F       E99C8F       F3B1B8A3
E8C1     E8C1 E8C1 E8C1 E8C1 6693 F0C3 F0C3 F0C3 F0C3 7043 75C3 F0C3 9716      E99C96       E99C96       F3B1B8A4
E8C2     E8C2 E8C2 E8C2 E8C2 668D F0C4 F0C4 F0C4 F0C4 7044 75C4 F0C4 9719      E99C99       E99C99       F3B1B8A5
E8C3     E8C3 E8C3 E8C3 E8C3 6694 F0C5 F0C5 F0C5 F0C5 7045 75C5 F0C5 9724      E99CA4       E99CA4       F3B1B8A6
E8C4     E8C4 E8C4 E8C4 E8C4 6695 F0C6 F0C6 F0C6 F0C6 7046 75C6 F0C6 972A      E99CAA       E99CAA       F3B1B8A7
E8C5     E8C5 E8C5 E8C5 E8C5 6696 F0C7 F0C7 F0C7 F0C7 7047 75C7 F0C7 9730      E99CB0       E99CB0       F3B1B8A8
E8C6     E8C6 E8C6 E8C6 E8C6 6698 F0C8 F0C8 F0C8 F0C8 7048 75C8 F0C8 9739      E99CB9       E99CB9       F3B1B8A9
E8C7     E8C7 E8C7 E8C7 E8C7 669A F0C9 F0C9 F0C9 F0C9 7049 75C9 F0C9 973D      E99CBD       E99CBD       F3B1B8AA
E8C8     E8C8 E8C8 E8C8 E8C8 669B F0CA F0CA F0CA F0CA 704A 75D1 F0CA 973E      E99CBE       E99CBE       F3B1B8AB
E8C9     E8C9 E8C9 E8C9 E8C9 669D F0CB F0CB F0CB F0CB 704B 75D2 F0CB 9744      E99D84       E99D84       F3B1B8AC
E8CA     E8CA E8CA E8CA E8CA 669C F0CC F0CC F0CC F0CC 704C 75D3 F0CC 9746      E99D86       E99D86       F3B1B8AD
E8CB     E8CB E8CB E8CB E8CB 669F F0CD F0CD F0CD F0CD 704D 75D4 F0CD 9748      E99D88       E99D88       F3B1B8AE
E8CC     E8CC E8CC E8CC E8CC 66A0 F0CE F0CE F0CE F0CE 704E 75D5 F0CE 9742      E99D82       E99D82       F3B1B8AF
E8CD     E8CD E8CD E8CD E8CD 66A1 F0CF F0CF F0CF F0CF 704F 75D6 F0CF 9749      E99D89       E99D89       F3B1B8B0
E8CE     E8CE E8CE E8CE E8CE 66A6 F0D0 F0D0 F0D0 F0D0 7050 75D7 F0D0 975C      E99D9C       E99D9C       F3B1B8B1
E8CF     E8CF E8CF E8CF E8CF 66A7 F0D1 F0D1 F0D1 F0D1 7051 75D8 F0D1 9760      E99DA0       E99DA0       F3B1B8B2
E8D0     E8D0 E8D0 E8D0 E8D0 66A8 F0D2 F0D2 F0D2 F0D2 7052 75D9 F0D2 9764      E99DA4       E99DA4       F3B1B8B3
E8D1     E8D1 E8D1 E8D1 E8D1 66A9 F0D3 F0D3 F0D3 F0D3 7053 75E2 F0D3 9766      E99DA6       E99DA6       F3B1B8B4
E8D2     E8D2 E8D2 E8D2 E8D2 66AA F0D4 F0D4 F0D4 F0D4 7054 75E3 F0D4 9768      E99DA8       E99DA8       F3B1B8B5
E8D3     E8D3 E8D3 E8D3 E8D3 53D6 F0D5 F0D5 F0D5 F0D5 7055 75E4 F0D5 52D2      E58B92       E58B92       F3B1B8B6
E8D4     E8D4 E8D4 E8D4 E8D4 53B5 F0D6 F0D6 F0D6 F0D6 7056 75E5 F0D6 976B      E99DAB       E99DAB       F3B1B8B7
E8D5     E8D5 E8D5 E8D5 E8D5 52CE BFD9 F0D7 F0D7 BFD9 3F59 6FE8 F0D7 9771      E99DB1       E99DB1       F3B1B8B8
E8D6     E8D6 E8D6 E8D6 E8D6 66AC F0D8 F0D8 F0D8 F0D8 7058 75E7 F0D8 9779      E99DB9       E99DB9       F3B1B8B9
E8D7     E8D7 E8D7 E8D7 E8D7 66AD F0D9 F0D9 F0D9 F0D9 7059 75E8 F0D9 9785      E99E85       E99E85       F3B1B8BA
E8D8     E8D8 E8D8 E8D8 E8D8 66AE F0DA F0DA F0DA F0DA 705A 75E9 F0DA 977C      E99DBC       E99DBC       F3B1B8BB
E8D9     E8D9 E8D9 E8D9 E8D9 66AF F0DB F0DB F0DB F0DB 705B 754A F0DB 9781      E99E81       E99E81       F3B1B8BC
E8DA     E8DA E8DA E8DA E8DA 66B0 F0DC F0DC F0DC F0DC 705C 755B F0DC 977A      E99DBA       E99DBA       F3B1B8BD
E8DB     E8DB E8DB E8DB E8DB 5277 F0DD F0DD F0DD F0DD 705D 755A F0DD 9786      E99E86       E99E86       F3B1B8BE
E8DC     E8DC E8DC E8DC E8DC 66B1 F0DE F0DE F0DE F0DE 705E 755F F0DE 978B      E99E8B       E99E8B       F3B1B8BF
E8DD     E8DD E8DD E8DD E8DD 66B2 F0DF F0DF F0DF F0DF 705F 756D F0DF 978F      E99E8F       E99E8F       F3B1B980
E8DE     E8DE E8DE E8DE E8DE 66B3 F0E0 F0E0 F0E0 F0E0 7060 7579 F0E0 9790      E99E90       E99E90       F3B1B981
E8DF     E8DF E8DF E8DF E8DF 66B4 F0E1 F0E1 F0E1 F0E1 7061 7557 F0E1 979C      E99E9C       E99E9C       F3B1B982
E8E0     E8E0 E8E0 E8E0 E8E0 66B5 F0E2 F0E2 F0E2 F0E2 7062 7559 F0E2 97A8      E99EA8       E99EA8       F3B1B983
E8E1     E8E1 E8E1 E8E1 E8E1 66B6 F0E3 F0E3 F0E3 F0E3 7063 7562 F0E3 97A6      E99EA6       E99EA6       F3B1B984
E8E2     E8E2 E8E2 E8E2 E8E2 66B7 F0E4 F0E4 F0E4 F0E4 7064 7563 F0E4 97A3      E99EA3       E99EA3       F3B1B985
E8E3     E8E3 E8E3 E8E3 E8E3 66B8 F0E5 F0E5 F0E5 F0E5 7065 7564 F0E5 97B3      E99EB3       E99EB3       F3B1B986
E8E4     E8E4 E8E4 E8E4 E8E4 66B9 F0E6 F0E6 F0E6 F0E6 7066 7565 F0E6 97B4      E99EB4       E99EB4       F3B1B987
E8E5     E8E5 E8E5 E8E5 E8E5 66BA F0E7 F0E7 F0E7 F0E7 7067 7566 F0E7 97C3      E99F83       E99F83       F3B1B988
E8E6     E8E6 E8E6 E8E6 E8E6 66BB F0E8 F0E8 F0E8 F0E8 7068 7567 F0E8 97C6      E99F86       E99F86       F3B1B989
E8E7     E8E7 E8E7 E8E7 E8E7 66BC F0E9 F0E9 F0E9 F0E9 7069 7568 F0E9 97C8      E99F88       E99F88       F3B1B98A
E8E8     E8E8 E8E8 E8E8 E8E8 66BD F0EA F0EA F0EA F0EA 706A 7569 F0EA 97CB      E99F8B       E99F8B       F3B1B98B
E8E9     E8E9 E8E9 E8E9 E8E9 66BE F0EB F0EB F0EB F0EB 706B 7570 F0EB 97DC      E99F9C       E99F9C       F3B1B98C
E8EA     E8EA E8EA E8EA E8EA 66BF F0EC F0EC F0EC F0EC 706C 7571 F0EC 97ED      E99FAD       E99FAD       F3B1B98D
E8EB     E8EB E8EB E8EB E8EB 686F F0ED F0ED F0ED F0ED 706D 7572 F0ED 9F4F      E9BD8F       E9BD8F       F3B1B98E
E8EC     E8EC E8EC E8EC E8EC 66C0 F0EE F0EE F0EE F0EE 706E 7573 F0EE 97F2      E99FB2       E99FB2       F3B1B98F
E8ED     E8ED E8ED E8ED E8ED 66C1 F0EF F0EF F0EF F0EF 706F 7574 F0EF 7ADF      E7AB9F       E7AB9F       F3B1B990
E8EE     E8EE E8EE E8EE E8EE 52FC F0F0 F0F0 F0F0 F0F0 7070 7575 F0F0 97F6      E99FB6       E99FB6       F3B1B991
E8EF     E8EF E8EF E8EF E8EF 66C2 F0F1 F0F1 F0F1 F0F1 7071 7576 F0F1 97F5      E99FB5       E99FB5       F3B1B992
E8F0     E8F0 E8F0 E8F0 E8F0 66C3 F0F2 F0F2 F0F2 F0F2 7072 7577 F0F2 980F      E9A08F       E9A08F       F3B1B993
E8F1     E8F1 E8F1 E8F1 E8F1 5456 F0F3 F0F3 F0F3 F0F3 7073 7578 F0F3 980C      E9A08C       E9A08C       F3B1B994
E8F2     E8F2 E8F2 E8F2 E8F2 50FB B7DB F0F4 F0F4 B7DB 375B F74A F0F4 9838      E9A0B8       E9A0B8       F3B1B995
E8F3     E8F3 E8F3 E8F3 E8F3 66C5 F0F5 F0F5 F0F5 F0F5 7075 758B F0F5 9824      E9A0A4       E9A0A4       F3B1B996
E8F4     E8F4 E8F4 E8F4 E8F4 66C6 F0F6 F0F6 F0F6 F0F6 7076 759B F0F6 9821      E9A0A1       E9A0A1       F3B1B997
E8F5     E8F5 E8F5 E8F5 E8F5 66C7 F0F7 F0F7 F0F7 F0F7 7077 759C F0F7 9837      E9A0B7       E9A0B7       F3B1B998
E8F6     E8F6 E8F6 E8F6 E8F6 66C8 F0F8 F0F8 F0F8 F0F8 7078 75A0 F0F8 983D      E9A0BD       E9A0BD       F3B1B999
E8F7     E8F7 E8F7 E8F7 E8F7 51DC F0F9 F0F9 F0F9 F0F9 7079 75AB F0F9 9846      E9A186       E9A186       F3B1B99A
E8F8     E8F8 E8F8 E8F8 E8F8 66C9 F0FA F0FA F0FA F0FA 707A 75B0 F0FA 984F      E9A18F       E9A18F       F3B1B99B
E8F9     E8F9 E8F9 E8F9 E8F9 66CA F0FB F0FB F0FB F0FB 707B 75C0 F0FB 984B      E9A18B       E9A18B       F3B1B99C
E8FA     E8FA E8FA E8FA E8FA 66CD F0FC F0FC F0FC F0FC 707C 756A F0FC 986B      E9A1AB       E9A1AB       F3B1B99D
E8FB     E8FB E8FB E8FB E8FB 66CE F0FD F0FD F0FD F0FD 707D 75D0 F0FD 986F      E9A1AF       E9A1AF       F3B1B99E
E8FC     E8FC E8FC E8FC E8FC 66CF F0FE F0FE F0FE F0FE 707E 75A1 F0FE 9870      E9A1B0       E9A1B0       F3B1B99F
E940     E940 E940 E940 E940 66D0 F1A1 F1A1 F1A1 F1A1 7121 764F F1A1 9871      E9A1B1       E9A1B1       F3B1B9A0
E941     E941 E941 E941 E941 66D1 F1A2 F1A2 F1A2 F1A2 7122 767F F1A2 9874      E9A1B4       E9A1B4       F3B1B9A1
E942     E942 E942 E942 E942 66D2 F1A3 F1A3 F1A3 F1A3 7123 767B F1A3 9873      E9A1B3       E9A1B3       F3B1B9A2
E943     E943 E943 E943 E943 66D3 F1A4 F1A4 F1A4 F1A4 7124 76E0 F1A4 98AA      E9A2AA       E9A2AA       F3B1B9A3
E944     E944 E944 E944 E944 66D4 F1A5 F1A5 F1A5 F1A5 7125 766C F1A5 98AF      E9A2AF       E9A2AF       F3B1B9A4
E945     E945 E945 E945 E945 66D5 F1A6 F1A6 F1A6 F1A6 7126 7650 F1A6 98B1      E9A2B1       E9A2B1       F3B1B9A5
E946     E946 E946 E946 E946 66D6 F1A7 F1A7 F1A7 F1A7 7127 767D F1A7 98B6      E9A2B6       E9A2B6       F3B1B9A6
E947     E947 E947 E947 E947 66D7 F1A8 F1A8 F1A8 F1A8 7128 764D F1A8 98C4      E9A384       E9A384       F3B1B9A7
E948     E948 E948 E948 E948 66D8 F1A9 F1A9 F1A9 F1A9 7129 765D F1A9 98C3      E9A383       E9A383       F3B1B9A8
E949     E949 E949 E949 E949 66D9 F1AA F1AA F1AA F1AA 712A 765C F1AA 98C6      E9A386       E9A386       F3B1B9A9
E94A     E94A E94A E94A E94A 66DC F1AB F1AB F1AB F1AB 712B 764E F1AB 98E9      E9A3A9       E9A3A9       F3B1B9AA
E94B     E94B E94B E94B E94B 66DE F1AC F1AC F1AC F1AC 712C 766B F1AC 98EB      E9A3AB       E9A3AB       F3B1B9AB
E94C     E94C E94C E94C E94C 66E0 F1AD F1AD F1AD F1AD 712D 7660 F1AD 9903      E9A483       E9A483       F3B1B9AC
E94D     E94D E94D E94D E94D 54BC F1AE F1AE F1AE F1AE 712E 764B F1AE 9909      E9A489       E9A489       F3B1B9AD
E94E     E94E E94E E94E E94E 66E2 F1AF F1AF F1AF F1AF 712F 7661 F1AF 9912      E9A492       E9A492       F3B1B9AE
E94F     E94F E94F E94F E94F 66E3 F1B0 F1B0 F1B0 F1B0 7130 76F0 F1B0 9914      E9A494       E9A494       F3B1B9AF
E950     E950 E950 E950 E950 66E4 F1B1 F1B1 F1B1 F1B1 7131 76F1 F1B1 9918      E9A498       E9A498       F3B1B9B0
E951     E951 E951 E951 E951 66E7 F1B2 F1B2 F1B2 F1B2 7132 76F2 F1B2 9921      E9A4A1       E9A4A1       F3B1B9B1
E952     E952 E952 E952 E952 66E1 F1B3 F1B3 F1B3 F1B3 7133 76F3 F1B3 991D      E9A49D       E9A49D       F3B1B9B2
E953     E953 E953 E953 E953 66E8 F1B4 F1B4 F1B4 F1B4 7134 76F4 F1B4 991E      E9A49E       E9A49E       F3B1B9B3
E954     E954 E954 E954 E954 66E9 F1B5 F1B5 F1B5 F1B5 7135 76F5 F1B5 9924      E9A4A4       E9A4A4       F3B1B9B4
E955     E955 E955 E955 E955 66EA F1B6 F1B6 F1B6 F1B6 7136 76F6 F1B6 9920      E9A4A0       E9A4A0       F3B1B9B5
E956     E956 E956 E956 E956 66EB F1B7 F1B7 F1B7 F1B7 7137 76F7 F1B7 992C      E9A4AC       E9A4AC       F3B1B9B6
E957     E957 E957 E957 E957 66EC F1B8 F1B8 F1B8 F1B8 7138 76F8 F1B8 992E      E9A4AE       E9A4AE       F3B1B9B7
E958     E958 E958 E958 E958 66ED F1B9 F1B9 F1B9 F1B9 7139 76F9 F1B9 993D      E9A4BD       E9A4BD       F3B1B9B8
E959     E959 E959 E959 E959 66EE F1BA F1BA F1BA F1BA 713A 767A F1BA 993E      E9A4BE       E9A4BE       F3B1B9B9
E95A     E95A E95A E95A E95A 66EF F1BB F1BB F1BB F1BB 713B 765E F1BB 9942      E9A582       E9A582       F3B1B9BA
E95B     E95B E95B E95B E95B 66F0 F1BC F1BC F1BC F1BC 713C 764C F1BC 9949      E9A589       E9A589       F3B1B9BB
E95C     E95C E95C E95C E95C 66F1 F1BD F1BD F1BD F1BD 713D 767E F1BD 9945      E9A585       E9A585       F3B1B9BC
E95D     E95D E95D E95D E95D 66F2 F1BE F1BE F1BE F1BE 713E 766E F1BE 9950      E9A590       E9A590       F3B1B9BD
E95E     E95E E95E E95E E95E 66F3 F1BF F1BF F1BF F1BF 713F 766F F1BF 994B      E9A58B       E9A58B       F3B1B9BE
E95F     E95F E95F E95F E95F 66F4 F1C0 F1C0 F1C0 F1C0 7140 767C F1C0 9951      E9A591       E9A591       F3B1B9BF
E960     E960 E960 E960 E960 52E8 F1C1 F1C1 F1C1 F1C1 7141 76C1 F1C1 9952      E9A592       E9A592       F3B1BA80
E961     E961 E961 E961 E961 66F5 F1C2 F1C2 F1C2 F1C2 7142 76C2 F1C2 994C      E9A58C       E9A58C       F3B1BA81
E962     E962 E962 E962 E962 66F6 F1C3 F1C3 F1C3 F1C3 7143 76C3 F1C3 9955      E9A595       E9A595       F3B1BA82
E963     E963 E963 E963 E963 66F7 F1C4 F1C4 F1C4 F1C4 7144 76C4 F1C4 9997      E9A697       E9A697       F3B1BA83
E964     E964 E964 E964 E964 66F8 F1C5 F1C5 F1C5 F1C5 7145 76C5 F1C5 9998      E9A698       E9A698       F3B1BA84
E965     E965 E965 E965 E965 66FA F1C6 F1C6 F1C6 F1C6 7146 76C6 F1C6 99A5      E9A6A5       E9A6A5       F3B1BA85
E966     E966 E966 E966 E966 66FB F1C7 F1C7 F1C7 F1C7 7147 76C7 F1C7 99AD      E9A6AD       E9A6AD       F3B1BA86
E967     E967 E967 E967 E967 66FC F1C8 F1C8 F1C8 F1C8 7148 76C8 F1C8 99AE      E9A6AE       E9A6AE       F3B1BA87
E968     E968 E968 E968 E968 66FD F1C9 F1C9 F1C9 F1C9 7149 76C9 F1C9 99BC      E9A6BC       E9A6BC       F3B1BA88
E969     E969 E969 E969 E969 66FE F1CA F1CA F1CA F1CA 714A 76D1 F1CA 99DF      E9A79F       E9A79F       F3B1BA89
E96A     E96A E96A E96A E96A 6741 F1CB F1CB F1CB F1CB 714B 76D2 F1CB 99DB      E9A79B       E9A79B       F3B1BA8A
E96B     E96B E96B E96B E96B 6742 F1CC F1CC F1CC F1CC 714C 76D3 F1CC 99DD      E9A79D       E9A79D       F3B1BA8B
E96C     E96C E96C E96C E96C 6743 F1CD F1CD F1CD F1CD 714D 76D4 F1CD 99D8      E9A798       E9A798       F3B1BA8C
E96D     E96D E96D E96D E96D 6744 F1CE F1CE F1CE F1CE 714E 76D5 F1CE 99D1      E9A791       E9A791       F3B1BA8D
E96E     E96E E96E E96E E96E 6745 F1CF F1CF F1CF F1CF 714F 76D6 F1CF 99ED      E9A7AD       E9A7AD       F3B1BA8E
E96F     E96F E96F E96F E96F 6746 F1D0 F1D0 F1D0 F1D0 7150 76D7 F1D0 99EE      E9A7AE       E9A7AE       F3B1BA8F
E970     E970 E970 E970 E970 6748 F1D1 F1D1 F1D1 F1D1 7151 76D8 F1D1 99F1      E9A7B1       E9A7B1       F3B1BA90
E971     E971 E971 E971 E971 6749 F1D2 F1D2 F1D2 F1D2 7152 76D9 F1D2 99F2      E9A7B2       E9A7B2       F3B1BA91
E972     E972 E972 E972 E972 674A F1D3 F1D3 F1D3 F1D3 7153 76E2 F1D3 99FB      E9A7BB       E9A7BB       F3B1BA92
E973     E973 E973 E973 E973 674B F1D4 F1D4 F1D4 F1D4 7154 76E3 F1D4 99F8      E9A7B8       E9A7B8       F3B1BA93
E974     E974 E974 E974 E974 674C F1D5 F1D5 F1D5 F1D5 7155 76E4 F1D5 9A01      E9A881       E9A881       F3B1BA94
E975     E975 E975 E975 E975 674D F1D6 F1D6 F1D6 F1D6 7156 76E5 F1D6 9A0F      E9A88F       E9A88F       F3B1BA95
E976     E976 E976 E976 E976 674E F1D7 F1D7 F1D7 F1D7 7157 76E6 F1D7 9A05      E9A885       E9A885       F3B1BA96
E977     E977 E977 E977 E977 6747 F1D8 F1D8 F1D8 F1D8 7158 76E7 F1D8 99E2      E9A7A2       E9A7A2       F3B1BA97
E978     E978 E978 E978 E978 674F F1D9 F1D9 F1D9 F1D9 7159 76E8 F1D9 9A19      E9A899       E9A899       F3B1BA98
E979     E979 E979 E979 E979 6750 F1DA F1DA F1DA F1DA 715A 76E9 F1DA 9A2B      E9A8AB       E9A8AB       F3B1BA99
E97A     E97A E97A E97A E97A 6751 F1DB F1DB F1DB F1DB 715B 764A F1DB 9A37      E9A8B7       E9A8B7       F3B1BA9A
E97B     E97B E97B E97B E97B 6753 F1DC F1DC F1DC F1DC 715C 765B F1DC 9A45      E9A985       E9A985       F3B1BA9B
E97C     E97C E97C E97C E97C 6754 F1DD F1DD F1DD F1DD 715D 765A F1DD 9A42      E9A982       E9A982       F3B1BA9C
E97D     E97D E97D E97D E97D 6752 F1DE F1DE F1DE F1DE 715E 765F F1DE 9A40      E9A980       E9A980       F3B1BA9D
E97E     E97E E97E E97E E97E 6755 F1DF F1DF F1DF F1DF 715F 766D F1DF 9A43      E9A983       E9A983       F3B1BA9E
E980     E980 E980 E980 E980 6756 F1E0 F1E0 F1E0 F1E0 7160 7679 F1E0 9A3E      E9A8BE       E9A8BE       F3B1BA9F
E981     E981 E981 E981 E981 6757 F1E1 F1E1 F1E1 F1E1 7161 7657 F1E1 9A55      E9A995       E9A995       F3B1BAA0
E982     E982 E982 E982 E982 6758 F1E2 F1E2 F1E2 F1E2 7162 7659 F1E2 9A4D      E9A98D       E9A98D       F3B1BAA1
E983     E983 E983 E983 E983 675A F1E3 F1E3 F1E3 F1E3 7163 7662 F1E3 9A5B      E9A99B       E9A99B       F3B1BAA2
E984     E984 E984 E984 E984 675B F1E4 F1E4 F1E4 F1E4 7164 7663 F1E4 9A57      E9A997       E9A997       F3B1BAA3
E985     E985 E985 E985 E985 675C F1E5 F1E5 F1E5 F1E5 7165 7664 F1E5 9A5F      E9A99F       E9A99F       F3B1BAA4
E986     E986 E986 E986 E986 675D F1E6 F1E6 F1E6 F1E6 7166 7665 F1E6 9A62      E9A9A2       E9A9A2       F3B1BAA5
E987     E987 E987 E987 E987 675F F1E7 F1E7 F1E7 F1E7 7167 7666 F1E7 9A65      E9A9A5       E9A9A5       F3B1BAA6
E988     E988 E988 E988 E988 6760 F1E8 F1E8 F1E8 F1E8 7168 7667 F1E8 9A64      E9A9A4       E9A9A4       F3B1BAA7
E989     E989 E989 E989 E989 675E F1E9 F1E9 F1E9 F1E9 7169 7668 F1E9 9A69      E9A9A9       E9A9A9       F3B1BAA8
E98A     E98A E98A E98A E98A 6762 F1EA F1EA F1EA F1EA 716A 7669 F1EA 9A6B      E9A9AB       E9A9AB       F3B1BAA9
E98B     E98B E98B E98B E98B 6761 F1EB F1EB F1EB F1EB 716B 7670 F1EB 9A6A      E9A9AA       E9A9AA       F3B1BAAA
E98C     E98C E98C E98C E98C 6763 F1EC F1EC F1EC F1EC 716C 7671 F1EC 9AAD      E9AAAD       E9AAAD       F3B1BAAB
E98D     E98D E98D E98D E98D 6764 F1ED F1ED F1ED F1ED 716D 7672 F1ED 9AB0      E9AAB0       E9AAB0       F3B1BAAC
E98E     E98E E98E E98E E98E 6765 F1EE F1EE F1EE F1EE 716E 7673 F1EE 9ABC      E9AABC       E9AABC       F3B1BAAD
E98F     E98F E98F E98F E98F 6766 F1EF F1EF F1EF F1EF 716F 7674 F1EF 9AC0      E9AB80       E9AB80       F3B1BAAE
E990     E990 E990 E990 E990 6767 F1F0 F1F0 F1F0 F1F0 7170 7675 F1F0 9ACF      E9AB8F       E9AB8F       F3B1BAAF
E991     E991 E991 E991 E991 676A F1F1 F1F1 F1F1 F1F1 7171 7676 F1F1 9AD1      E9AB91       E9AB91       F3B1BAB0
E992     E992 E992 E992 E992 6768 F1F2 F1F2 F1F2 F1F2 7172 7677 F1F2 9AD3      E9AB93       E9AB93       F3B1BAB1
E993     E993 E993 E993 E993 6769 F1F3 F1F3 F1F3 F1F3 7173 7678 F1F3 9AD4      E9AB94       E9AB94       F3B1BAB2
E994     E994 E994 E994 E994 676D F1F4 F1F4 F1F4 F1F4 7174 7680 F1F4 9ADE      E9AB9E       E9AB9E       F3B1BAB3
E995     E995 E995 E995 E995 676E F1F5 F1F5 F1F5 F1F5 7175 768B F1F5 9ADF      E9AB9F       E9AB9F       F3B1BAB4
E996     E996 E996 E996 E996 676F F1F6 F1F6 F1F6 F1F6 7176 769B F1F6 9AE2      E9ABA2       E9ABA2       F3B1BAB5
E997     E997 E997 E997 E997 6770 F1F7 F1F7 F1F7 F1F7 7177 769C F1F7 9AE3      E9ABA3       E9ABA3       F3B1BAB6
E998     E998 E998 E998 E998 6771 F1F8 F1F8 F1F8 F1F8 7178 76A0 F1F8 9AE6      E9ABA6       E9ABA6       F3B1BAB7
E999     E999 E999 E999 E999 6772 F1F9 F1F9 F1F9 F1F9 7179 76AB F1F9 9AEF      E9ABAF       E9ABAF       F3B1BAB8
E99A     E99A E99A E99A E99A 6773 F1FA F1FA F1FA F1FA 717A 76B0 F1FA 9AEB      E9ABAB       E9ABAB       F3B1BAB9
E99B     E99B E99B E99B E99B 6774 F1FB F1FB F1FB F1FB 717B 76C0 F1FB 9AEE      E9ABAE       E9ABAE       F3B1BABA
E99C     E99C E99C E99C E99C 6775 F1FC F1FC F1FC F1FC 717C 766A F1FC 9AF4      E9ABB4       E9ABB4       F3B1BABB
E99D     E99D E99D E99D E99D 6776 F1FD F1FD F1FD F1FD 717D 76D0 F1FD 9AF1      E9ABB1       E9ABB1       F3B1BABC
E99E     E99E E99E E99E E99E 6777 F1FE F1FE F1FE F1FE 717E 76A1 F1FE 9AF7      E9ABB7       E9ABB7       F3B1BABD
E99F     E99F E99F E99F E99F 6778 F2A1 F2A1 F2A1 F2A1 7221 774F F2A1 9AFB      E9ABBB       E9ABBB       F3B1BABE
E9A0     E9A0 E9A0 E9A0 E9A0 6779 F2A2 F2A2 F2A2 F2A2 7222 777F F2A2 9B06      E9AC86       E9AC86       F3B1BABF
E9A1     E9A1 E9A1 E9A1 E9A1 677A F2A3 F2A3 F2A3 F2A3 7223 777B F2A3 9B18      E9AC98       E9AC98       F3B1BB80
E9A2     E9A2 E9A2 E9A2 E9A2 677B F2A4 F2A4 F2A4 F2A4 7224 77E0 F2A4 9B1A      E9AC9A       E9AC9A       F3B1BB81
E9A3     E9A3 E9A3 E9A3 E9A3 677C F2A5 F2A5 F2A5 F2A5 7225 776C F2A5 9B1F      E9AC9F       E9AC9F       F3B1BB82
E9A4     E9A4 E9A4 E9A4 E9A4 677D F2A6 F2A6 F2A6 F2A6 7226 7750 F2A6 9B22      E9ACA2       E9ACA2       F3B1BB83
E9A5     E9A5 E9A5 E9A5 E9A5 677E F2A7 F2A7 F2A7 F2A7 7227 777D F2A7 9B23      E9ACA3       E9ACA3       F3B1BB84
E9A6     E9A6 E9A6 E9A6 E9A6 677F F2A8 F2A8 F2A8 F2A8 7228 774D F2A8 9B25      E9ACA5       E9ACA5       F3B1BB85
E9A7     E9A7 E9A7 E9A7 E9A7 6780 F2A9 F2A9 F2A9 F2A9 7229 775D F2A9 9B27      E9ACA7       E9ACA7       F3B1BB86
E9A8     E9A8 E9A8 E9A8 E9A8 6781 F2AA F2AA F2AA F2AA 722A 775C F2AA 9B28      E9ACA8       E9ACA8       F3B1BB87
E9A9     E9A9 E9A9 E9A9 E9A9 6782 F2AB F2AB F2AB F2AB 722B 774E F2AB 9B29      E9ACA9       E9ACA9       F3B1BB88
E9AA     E9AA E9AA E9AA E9AA 6783 F2AC F2AC F2AC F2AC 722C 776B F2AC 9B2A      E9ACAA       E9ACAA       F3B1BB89
E9AB     E9AB E9AB E9AB E9AB 6784 F2AD F2AD F2AD F2AD 722D 7760 F2AD 9B2E      E9ACAE       E9ACAE       F3B1BB8A
E9AC     E9AC E9AC E9AC E9AC 6785 F2AE F2AE F2AE F2AE 722E 774B F2AE 9B2F      E9ACAF       E9ACAF       F3B1BB8B
E9AD     E9AD E9AD E9AD E9AD 6787 F2AF F2AF F2AF F2AF 722F 7761 F2AF 9B32      E9ACB2       E9ACB2       F3B1BB8C
E9AE     E9AE E9AE E9AE E9AE 6789 F2B0 F2B0 F2B0 F2B0 7230 77F0 F2B0 9B44      E9AD84       E9AD84       F3B1BB8D
E9AF     E9AF E9AF E9AF E9AF 678A F2B1 F2B1 F2B1 F2B1 7231 77F1 F2B1 9B43      E9AD83       E9AD83       F3B1BB8E
E9B0     E9B0 E9B0 E9B0 E9B0 5389 F2B2 F2B2 F2B2 F2B2 7232 77F2 F2B2 9B4F      E9AD8F       E9AD8F       F3B1BB8F
E9B1     E9B1 E9B1 E9B1 E9B1 678B F2B3 F2B3 F2B3 F2B3 7233 77F3 F2B3 9B4D      E9AD8D       E9AD8D       F3B1BB90
E9B2     E9B2 E9B2 E9B2 E9B2 678C F2B4 F2B4 F2B4 F2B4 7234 77F4 F2B4 9B4E      E9AD8E       E9AD8E       F3B1BB91
E9B3     E9B3 E9B3 E9B3 E9B3 678D F2B5 F2B5 F2B5 F2B5 7235 77F5 F2B5 9B51      E9AD91       E9AD91       F3B1BB92
E9B4     E9B4 E9B4 E9B4 E9B4 678E F2B6 F2B6 F2B6 F2B6 7236 77F6 F2B6 9B58      E9AD98       E9AD98       F3B1BB93
E9B5     E9B5 E9B5 E9B5 E9B5 6790 F2B7 F2B7 F2B7 F2B7 7237 77F7 F2B7 9B74      E9ADB4       E9ADB4       F3B1BB94
E9B6     E9B6 E9B6 E9B6 E9B6 6792 F2B8 F2B8 F2B8 F2B8 7238 77F8 F2B8 9B93      E9AE93       E9AE93       F3B1BB95
E9B7     E9B7 E9B7 E9B7 E9B7 6794 F2B9 F2B9 F2B9 F2B9 7239 77F9 F2B9 9B83      E9AE83       E9AE83       F3B1BB96
E9B8     E9B8 E9B8 E9B8 E9B8 6795 F2BA F2BA F2BA F2BA 723A 777A F2BA 9B91      E9AE91       E9AE91       F3B1BB97
E9B9     E9B9 E9B9 E9B9 E9B9 6796 F2BB F2BB F2BB F2BB 723B 775E F2BB 9B96      E9AE96       E9AE96       F3B1BB98
E9BA     E9BA E9BA E9BA E9BA 6797 F2BC F2BC F2BC F2BC 723C 774C F2BC 9B97      E9AE97       E9AE97       F3B1BB99
E9BB     E9BB E9BB E9BB E9BB 6798 F2BD F2BD F2BD F2BD 723D 777E F2BD 9B9F      E9AE9F       E9AE9F       F3B1BB9A
E9BC     E9BC E9BC E9BC E9BC 6799 F2BE F2BE F2BE F2BE 723E 776E F2BE 9BA0      E9AEA0       E9AEA0       F3B1BB9B
E9BD     E9BD E9BD E9BD E9BD 679A F2BF F2BF F2BF F2BF 723F 776F F2BF 9BA8      E9AEA8       E9AEA8       F3B1BB9C
E9BE     E9BE E9BE E9BE E9BE 679C F2C0 F2C0 F2C0 F2C0 7240 777C F2C0 9BB4      E9AEB4       E9AEB4       F3B1BB9D
E9BF     E9BF E9BF E9BF E9BF 679D F2C1 F2C1 F2C1 F2C1 7241 77C1 F2C1 9BC0      E9AF80       E9AF80       F3B1BB9E
E9C0     E9C0 E9C0 E9C0 E9C0 679E F2C2 F2C2 F2C2 F2C2 7242 77C2 F2C2 9BCA      E9AF8A       E9AF8A       F3B1BB9F
E9C1     E9C1 E9C1 E9C1 E9C1 67A0 F2C3 F2C3 F2C3 F2C3 7243 77C3 F2C3 9BB9      E9AEB9       E9AEB9       F3B1BBA0
E9C2     E9C2 E9C2 E9C2 E9C2 67A1 F2C4 F2C4 F2C4 F2C4 7244 77C4 F2C4 9BC6      E9AF86       E9AF86       F3B1BBA1
E9C3     E9C3 E9C3 E9C3 E9C3 67A2 F2C5 F2C5 F2C5 F2C5 7245 77C5 F2C5 9BCF      E9AF8F       E9AF8F       F3B1BBA2
E9C4     E9C4 E9C4 E9C4 E9C4 67A3 F2C6 F2C6 F2C6 F2C6 7246 77C6 F2C6 9BD1      E9AF91       E9AF91       F3B1BBA3
E9C5     E9C5 E9C5 E9C5 E9C5 67A4 F2C7 F2C7 F2C7 F2C7 7247 77C7 F2C7 9BD2      E9AF92       E9AF92       F3B1BBA4
E9C6     E9C6 E9C6 E9C6 E9C6 67A5 F2C8 F2C8 F2C8 F2C8 7248 77C8 F2C8 9BE3      E9AFA3       E9AFA3       F3B1BBA5
E9C7     E9C7 E9C7 E9C7 E9C7 67A6 F2C9 F2C9 F2C9 F2C9 7249 77C9 F2C9 9BE2      E9AFA2       E9AFA2       F3B1BBA6
E9C8     E9C8 E9C8 E9C8 E9C8 67A7 F2CA F2CA F2CA F2CA 724A 77D1 F2CA 9BE4      E9AFA4       E9AFA4       F3B1BBA7
E9C9     E9C9 E9C9 E9C9 E9C9 67A8 F2CB F2CB F2CB F2CB 724B 77D2 F2CB 9BD4      E9AF94       E9AF94       F3B1BBA8
E9CA     E9CA E9CA E9CA E9CA 67A9 F2CC F2CC F2CC F2CC 724C 77D3 F2CC 9BE1      E9AFA1       E9AFA1       F3B1BBA9
E9CB     E9CB E9CB E9CB E9CB 51F0 B0B3 F2CD F2CD B0B3 3033 F0F3 F2CD 9C3A      E9B0BA       E9B0BA       F3B1BBAA
E9CC     E9CC E9CC E9CC E9CC 67AC F2CE F2CE F2CE F2CE 724E 77D5 F2CE 9BF2      E9AFB2       E9AFB2       F3B1BBAB
E9CD     E9CD E9CD E9CD E9CD 67AB F2CF F2CF F2CF F2CF 724F 77D6 F2CF 9BF1      E9AFB1       E9AFB1       F3B1BBAC
E9CE     E9CE E9CE E9CE E9CE 5551 F2D0 F2D0 F2D0 F2D0 7250 77D7 F2D0 9BF0      E9AFB0       E9AFB0       F3B1BBAD
E9CF     E9CF E9CF E9CF E9CF 67AF F2D1 F2D1 F2D1 F2D1 7251 77D8 F2D1 9C15      E9B095       E9B095       F3B1BBAE
E9D0     E9D0 E9D0 E9D0 E9D0 67B0 F2D2 F2D2 F2D2 F2D2 7252 77D9 F2D2 9C14      E9B094       E9B094       F3B1BBAF
E9D1     E9D1 E9D1 E9D1 E9D1 67B2 F2D3 F2D3 F2D3 F2D3 7253 77E2 F2D3 9C09      E9B089       E9B089       F3B1BBB0
E9D2     E9D2 E9D2 E9D2 E9D2 67B3 F2D4 F2D4 F2D4 F2D4 7254 77E3 F2D4 9C13      E9B093       E9B093       F3B1BBB1
E9D3     E9D3 E9D3 E9D3 E9D3 67B4 F2D5 F2D5 F2D5 F2D5 7255 77E4 F2D5 9C0C      E9B08C       E9B08C       F3B1BBB2
E9D4     E9D4 E9D4 E9D4 E9D4 67B5 F2D6 F2D6 F2D6 F2D6 7256 77E5 F2D6 9C06      E9B086       E9B086       F3B1BBB3
E9D5     E9D5 E9D5 E9D5 E9D5 67B6 F2D7 F2D7 F2D7 F2D7 7257 77E6 F2D7 9C08      E9B088       E9B088       F3B1BBB4
E9D6     E9D6 E9D6 E9D6 E9D6 67B7 F2D8 F2D8 F2D8 F2D8 7258 77E7 F2D8 9C12      E9B092       E9B092       F3B1BBB5
E9D7     E9D7 E9D7 E9D7 E9D7 67B8 F2D9 F2D9 F2D9 F2D9 7259 77E8 F2D9 9C0A      E9B08A       E9B08A       F3B1BBB6
E9D8     E9D8 E9D8 E9D8 E9D8 67AD F2DA F2DA F2DA F2DA 725A 77E9 F2DA 9C04      E9B084       E9B084       F3B1BBB7
E9D9     E9D9 E9D9 E9D9 E9D9 67B9 F2DB F2DB F2DB F2DB 725B 774A F2DB 9C2E      E9B0AE       E9B0AE       F3B1BBB8
E9DA     E9DA E9DA E9DA E9DA 67AE F2DC F2DC F2DC F2DC 725C 775B F2DC 9C1B      E9B09B       E9B09B       F3B1BBB9
E9DB     E9DB E9DB E9DB E9DB 67BA F2DD F2DD F2DD F2DD 725D 775A F2DD 9C25      E9B0A5       E9B0A5       F3B1BBBA
E9DC     E9DC E9DC E9DC E9DC 67BB F2DE F2DE F2DE F2DE 725E 775F F2DE 9C24      E9B0A4       E9B0A4       F3B1BBBB
E9DD     E9DD E9DD E9DD E9DD 67BC F2DF F2DF F2DF F2DF 725F 776D F2DF 9C21      E9B0A1       E9B0A1       F3B1BBBC
E9DE     E9DE E9DE E9DE E9DE 67BD F2E0 F2E0 F2E0 F2E0 7260 7779 F2E0 9C30      E9B0B0       E9B0B0       F3B1BBBD
E9DF     E9DF E9DF E9DF E9DF 67BE F2E1 F2E1 F2E1 F2E1 7261 7757 F2E1 9C47      E9B187       E9B187       F3B1BBBE
E9E0     E9E0 E9E0 E9E0 E9E0 67BF F2E2 F2E2 F2E2 F2E2 7262 7759 F2E2 9C32      E9B0B2       E9B0B2       F3B1BBBF
E9E1     E9E1 E9E1 E9E1 E9E1 67C0 F2E3 F2E3 F2E3 F2E3 7263 7762 F2E3 9C46      E9B186       E9B186       F3B1BC80
E9E2     E9E2 E9E2 E9E2 E9E2 67C1 F2E4 F2E4 F2E4 F2E4 7264 7763 F2E4 9C3E      E9B0BE       E9B0BE       F3B1BC81
E9E3     E9E3 E9E3 E9E3 E9E3 67C2 F2E5 F2E5 F2E5 F2E5 7265 7764 F2E5 9C5A      E9B19A       E9B19A       F3B1BC82
E9E4     E9E4 E9E4 E9E4 E9E4 67C3 F2E6 F2E6 F2E6 F2E6 7266 7765 F2E6 9C60      E9B1A0       E9B1A0       F3B1BC83
E9E5     E9E5 E9E5 E9E5 E9E5 67C4 F2E7 F2E7 F2E7 F2E7 7267 7766 F2E7 9C67      E9B1A7       E9B1A7       F3B1BC84
E9E6     E9E6 E9E6 E9E6 E9E6 67C5 F2E8 F2E8 F2E8 F2E8 7268 7767 F2E8 9C76      E9B1B6       E9B1B6       F3B1BC85
E9E7     E9E7 E9E7 E9E7 E9E7 67C6 F2E9 F2E9 F2E9 F2E9 7269 7768 F2E9 9C78      E9B1B8       E9B1B8       F3B1BC86
E9E8     E9E8 E9E8 E9E8 E9E8 67C8 F2EA F2EA F2EA F2EA 726A 7769 F2EA 9CE7      E9B3A7       E9B3A7       F3B1BC87
E9E9     E9E9 E9E9 E9E9 E9E9 67C9 F2EB F2EB F2EB F2EB 726B 7770 F2EB 9CEC      E9B3AC       E9B3AC       F3B1BC88
E9EA     E9EA E9EA E9EA E9EA 67CA F2EC F2EC F2EC F2EC 726C 7771 F2EC 9CF0      E9B3B0       E9B3B0       F3B1BC89
E9EB     E9EB E9EB E9EB E9EB 67CB F2ED F2ED F2ED F2ED 726D 7772 F2ED 9D09      E9B489       E9B489       F3B1BC8A
E9EC     E9EC E9EC E9EC E9EC 54FC F2EE F2EE F2EE F2EE 726E 7773 F2EE 9D08      E9B488       E9B488       F3B1BC8B
E9ED     E9ED E9ED E9ED E9ED 67C7 F2EF F2EF F2EF F2EF 726F 7774 F2EF 9CEB      E9B3AB       E9B3AB       F3B1BC8C
E9EE     E9EE E9EE E9EE E9EE 67CC F2F0 F2F0 F2F0 F2F0 7270 7775 F2F0 9D03      E9B483       E9B483       F3B1BC8D
E9EF     E9EF E9EF E9EF E9EF 67CD F2F1 F2F1 F2F1 F2F1 7271 7776 F2F1 9D06      E9B486       E9B486       F3B1BC8E
E9F0     E9F0 E9F0 E9F0 E9F0 67CE F2F2 F2F2 F2F2 F2F2 7272 7777 F2F2 9D2A      E9B4AA       E9B4AA       F3B1BC8F
E9F1     E9F1 E9F1 E9F1 E9F1 67CF F2F3 F2F3 F2F3 F2F3 7273 7778 F2F3 9D26      E9B4A6       E9B4A6       F3B1BC90
E9F2     E9F2 E9F2 E9F2 E9F2 50E5 B2A9 F2F4 F2F4 B2A9 3229 F25D F2F4 9DAF      E9B6AF       E9B6AF       F3B1BC91
E9F3     E9F3 E9F3 E9F3 E9F3 67D1 F2F5 F2F5 F2F5 F2F5 7275 778B F2F5 9D23      E9B4A3       E9B4A3       F3B1BC92
E9F4     E9F4 E9F4 E9F4 E9F4 67D2 F2F6 F2F6 F2F6 F2F6 7276 779B F2F6 9D1F      E9B49F       E9B49F       F3B1BC93
E9F5     E9F5 E9F5 E9F5 E9F5 67D7 F2F7 F2F7 F2F7 F2F7 7277 779C F2F7 9D44      E9B584       E9B584       F3B1BC94
E9F6     E9F6 E9F6 E9F6 E9F6 67D3 F2F8 F2F8 F2F8 F2F8 7278 77A0 F2F8 9D15      E9B495       E9B495       F3B1BC95
E9F7     E9F7 E9F7 E9F7 E9F7 67D4 F2F9 F2F9 F2F9 F2F9 7279 77AB F2F9 9D12      E9B492       E9B492       F3B1BC96
E9F8     E9F8 E9F8 E9F8 E9F8 67D5 F2FA F2FA F2FA F2FA 727A 77B0 F2FA 9D41      E9B581       E9B581       F3B1BC97
E9F9     E9F9 E9F9 E9F9 E9F9 67D6 F2FB F2FB F2FB F2FB 727B 77C0 F2FB 9D3F      E9B4BF       E9B4BF       F3B1BC98
E9FA     E9FA E9FA E9FA E9FA 67D8 F2FC F2FC F2FC F2FC 727C 776A F2FC 9D3E      E9B4BE       E9B4BE       F3B1BC99
E9FB     E9FB E9FB E9FB E9FB 67D9 F2FD F2FD F2FD F2FD 727D 77D0 F2FD 9D46      E9B586       E9B586       F3B1BC9A
E9FC     E9FC E9FC E9FC E9FC 67DA F2FE F2FE F2FE F2FE 727E 77A1 F2FE 9D48      E9B588       E9B588       F3B1BC9B
EA40     EA40 EA40 EA40 EA40 67DB F3A1 F3A1 F3A1 F3A1 7321 784F F3A1 9D5D      E9B59D       E9B59D       F3B1BC9C
EA41     EA41 EA41 EA41 EA41 67DC F3A2 F3A2 F3A2 F3A2 7322 787F F3A2 9D5E      E9B59E       E9B59E       F3B1BC9D
EA42     EA42 EA42 EA42 EA42 67E0 F3A3 F3A3 F3A3 F3A3 7323 787B F3A3 9D64      E9B5A4       E9B5A4       F3B1BC9E
EA43     EA43 EA43 EA43 EA43 67DE F3A4 F3A4 F3A4 F3A4 7324 78E0 F3A4 9D51      E9B591       E9B591       F3B1BC9F
EA44     EA44 EA44 EA44 EA44 67DF F3A5 F3A5 F3A5 F3A5 7325 786C F3A5 9D50      E9B590       E9B590       F3B1BCA0
EA45     EA45 EA45 EA45 EA45 67DD F3A6 F3A6 F3A6 F3A6 7326 7850 F3A6 9D59      E9B599       E9B599       F3B1BCA1
EA46     EA46 EA46 EA46 EA46 67E1 F3A7 F3A7 F3A7 F3A7 7327 787D F3A7 9D72      E9B5B2       E9B5B2       F3B1BCA2
EA47     EA47 EA47 EA47 EA47 5342 F3A8 F3A8 F3A8 F3A8 7328 784D F3A8 9D89      E9B689       E9B689       F3B1BCA3
EA48     EA48 EA48 EA48 EA48 67E3 F3A9 F3A9 F3A9 F3A9 7329 785D F3A9 9D87      E9B687       E9B687       F3B1BCA4
EA49     EA49 EA49 EA49 EA49 67EA F3AA F3AA F3AA F3AA 732A 785C F3AA 9DAB      E9B6AB       E9B6AB       F3B1BCA5
EA4A     EA4A EA4A EA4A EA4A 67E5 F3AB F3AB F3AB F3AB 732B 784E F3AB 9D6F      E9B5AF       E9B5AF       F3B1BCA6
EA4B     EA4B EA4B EA4B EA4B 67E6 F3AC F3AC F3AC F3AC 732C 786B F3AC 9D7A      E9B5BA       E9B5BA       F3B1BCA7
EA4C     EA4C EA4C EA4C EA4C 67E7 F3AD F3AD F3AD F3AD 732D 7860 F3AD 9D9A      E9B69A       E9B69A       F3B1BCA8
EA4D     EA4D EA4D EA4D EA4D 67E8 F3AE F3AE F3AE F3AE 732E 784B F3AE 9DA4      E9B6A4       E9B6A4       F3B1BCA9
EA4E     EA4E EA4E EA4E EA4E 67E9 F3AF F3AF F3AF F3AF 732F 7861 F3AF 9DA9      E9B6A9       E9B6A9       F3B1BCAA
EA4F     EA4F EA4F EA4F EA4F 67EB F3B0 F3B0 F3B0 F3B0 7330 78F0 F3B0 9DB2      E9B6B2       E9B6B2       F3B1BCAB
EA50     EA50 EA50 EA50 EA50 67EC F3B1 F3B1 F3B1 F3B1 7331 78F1 F3B1 9DC4      E9B784       E9B784       F3B1BCAC
EA51     EA51 EA51 EA51 EA51 67ED F3B2 F3B2 F3B2 F3B2 7332 78F2 F3B2 9DC1      E9B781       E9B781       F3B1BCAD
EA52     EA52 EA52 EA52 EA52 67EE F3B3 F3B3 F3B3 F3B3 7333 78F3 F3B3 9DBB      E9B6BB       E9B6BB       F3B1BCAE
EA53     EA53 EA53 EA53 EA53 67EF F3B4 F3B4 F3B4 F3B4 7334 78F4 F3B4 9DB8      E9B6B8       E9B6B8       F3B1BCAF
EA54     EA54 EA54 EA54 EA54 67F0 F3B5 F3B5 F3B5 F3B5 7335 78F5 F3B5 9DBA      E9B6BA       E9B6BA       F3B1BCB0
EA55     EA55 EA55 EA55 EA55 67F1 F3B6 F3B6 F3B6 F3B6 7336 78F6 F3B6 9DC6      E9B786       E9B786       F3B1BCB1
EA56     EA56 EA56 EA56 EA56 67F2 F3B7 F3B7 F3B7 F3B7 7337 78F7 F3B7 9DCF      E9B78F       E9B78F       F3B1BCB2
EA57     EA57 EA57 EA57 EA57 67F3 F3B8 F3B8 F3B8 F3B8 7338 78F8 F3B8 9DC2      E9B782       E9B782       F3B1BCB3
EA58     EA58 EA58 EA58 EA58 67F5 F3B9 F3B9 F3B9 F3B9 7339 78F9 F3B9 9DD9      E9B799       E9B799       F3B1BCB4
EA59     EA59 EA59 EA59 EA59 67F6 F3BA F3BA F3BA F3BA 733A 787A F3BA 9DD3      E9B793       E9B793       F3B1BCB5
EA5A     EA5A EA5A EA5A EA5A 67F7 F3BB F3BB F3BB F3BB 733B 785E F3BB 9DF8      E9B7B8       E9B7B8       F3B1BCB6
EA5B     EA5B EA5B EA5B EA5B 67F8 F3BC F3BC F3BC F3BC 733C 784C F3BC 9DE6      E9B7A6       E9B7A6       F3B1BCB7
EA5C     EA5C EA5C EA5C EA5C 67F9 F3BD F3BD F3BD F3BD 733D 787E F3BD 9DED      E9B7AD       E9B7AD       F3B1BCB8
EA5D     EA5D EA5D EA5D EA5D 67FA F3BE F3BE F3BE F3BE 733E 786E F3BE 9DEF      E9B7AF       E9B7AF       F3B1BCB9
EA5E     EA5E EA5E EA5E EA5E 67FB F3BF F3BF F3BF F3BF 733F 786F F3BF 9DFD      E9B7BD       E9B7BD       F3B1BCBA
EA5F     EA5F EA5F EA5F EA5F 67FC F3C0 F3C0 F3C0 F3C0 7340 787C F3C0 9E1A      E9B89A       E9B89A       F3B1BCBB
EA60     EA60 EA60 EA60 EA60 67FD F3C1 F3C1 F3C1 F3C1 7341 78C1 F3C1 9E1B      E9B89B       E9B89B       F3B1BCBC
EA61     EA61 EA61 EA61 EA61 6841 F3C2 F3C2 F3C2 F3C2 7342 78C2 F3C2 9E1E      E9B89E       E9B89E       F3B1BCBD
EA62     EA62 EA62 EA62 EA62 6842 F3C3 F3C3 F3C3 F3C3 7343 78C3 F3C3 9E75      E9B9B5       E9B9B5       F3B1BCBE
EA63     EA63 EA63 EA63 EA63 6843 F3C4 F3C4 F3C4 F3C4 7344 78C4 F3C4 9E79      E9B9B9       E9B9B9       F3B1BCBF
EA64     EA64 EA64 EA64 EA64 6844 F3C5 F3C5 F3C5 F3C5 7345 78C5 F3C5 9E7D      E9B9BD       E9B9BD       F3B1BD80
EA65     EA65 EA65 EA65 EA65 6845 F3C6 F3C6 F3C6 F3C6 7346 78C6 F3C6 9E81      E9BA81       E9BA81       F3B1BD81
EA66     EA66 EA66 EA66 EA66 6846 F3C7 F3C7 F3C7 F3C7 7347 78C7 F3C7 9E88      E9BA88       E9BA88       F3B1BD82
EA67     EA67 EA67 EA67 EA67 6847 F3C8 F3C8 F3C8 F3C8 7348 78C8 F3C8 9E8B      E9BA8B       E9BA8B       F3B1BD83
EA68     EA68 EA68 EA68 EA68 6848 F3C9 F3C9 F3C9 F3C9 7349 78C9 F3C9 9E8C      E9BA8C       E9BA8C       F3B1BD84
EA69     EA69 EA69 EA69 EA69 51F9 F3CA F3CA F3CA F3CA 734A 78D1 F3CA 9E92      E9BA92       E9BA92       F3B1BD85
EA6A     EA6A EA6A EA6A EA6A 6849 F3CB F3CB F3CB F3CB 734B 78D2 F3CB 9E95      E9BA95       E9BA95       F3B1BD86
EA6B     EA6B EA6B EA6B EA6B 684A F3CC F3CC F3CC F3CC 734C 78D3 F3CC 9E91      E9BA91       E9BA91       F3B1BD87
EA6C     EA6C EA6C EA6C EA6C 684B F3CD F3CD F3CD F3CD 734D 78D4 F3CD 9E9D      E9BA9D       E9BA9D       F3B1BD88
EA6D     EA6D EA6D EA6D EA6D 684C F3CE F3CE F3CE F3CE 734E 78D5 F3CE 9EA5      E9BAA5       E9BAA5       F3B1BD89
EA6E     EA6E EA6E EA6E EA6E 53B4 F3CF F3CF F3CF F3CF 734F 78D6 F3CF 9EA9      E9BAA9       E9BAA9       F3B1BD8A
EA6F     EA6F EA6F EA6F EA6F 684D F3D0 F3D0 F3D0 F3D0 7350 78D7 F3D0 9EB8      E9BAB8       E9BAB8       F3B1BD8B
EA70     EA70 EA70 EA70 EA70 684E F3D1 F3D1 F3D1 F3D1 7351 78D8 F3D1 9EAA      E9BAAA       E9BAAA       F3B1BD8C
EA71     EA71 EA71 EA71 EA71 684F F3D2 F3D2 F3D2 F3D2 7352 78D9 F3D2 9EAD      E9BAAD       E9BAAD       F3B1BD8D
EA72     EA72 EA72 EA72 EA72 6852 F3D3 F3D3 F3D3 F3D3 7353 78E2 F3D3 9761      E99DA1       E99DA1       F3B1BD8E
EA73     EA73 EA73 EA73 EA73 6853 F3D4 F3D4 F3D4 F3D4 7354 78E3 F3D4 9ECC      E9BB8C       E9BB8C       F3B1BD8F
EA74     EA74 EA74 EA74 EA74 51A7 F3D5 F3D5 F3D5 F3D5 7355 78E4 F3D5 9ECE      E9BB8E       E9BB8E       F3B1BD90
EA75     EA75 EA75 EA75 EA75 6854 F3D6 F3D6 F3D6 F3D6 7356 78E5 F3D6 9ECF      E9BB8F       E9BB8F       F3B1BD91
EA76     EA76 EA76 EA76 EA76 6855 F3D7 F3D7 F3D7 F3D7 7357 78E6 F3D7 9ED0      E9BB90       E9BB90       F3B1BD92
EA77     EA77 EA77 EA77 EA77 6857 F3D8 F3D8 F3D8 F3D8 7358 78E7 F3D8 9ED4      E9BB94       E9BB94       F3B1BD93
EA78     EA78 EA78 EA78 EA78 6858 F3D9 F3D9 F3D9 F3D9 7359 78E8 F3D9 9EDC      E9BB9C       E9BB9C       F3B1BD94
EA79     EA79 EA79 EA79 EA79 6859 F3DA F3DA F3DA F3DA 735A 78E9 F3DA 9EDE      E9BB9E       E9BB9E       F3B1BD95
EA7A     EA7A EA7A EA7A EA7A 685A F3DB F3DB F3DB F3DB 735B 784A F3DB 9EDD      E9BB9D       E9BB9D       F3B1BD96
EA7B     EA7B EA7B EA7B EA7B 685B F3DC F3DC F3DC F3DC 735C 785B F3DC 9EE0      E9BBA0       E9BBA0       F3B1BD97
EA7C     EA7C EA7C EA7C EA7C 685C F3DD F3DD F3DD F3DD 735D 785A F3DD 9EE5      E9BBA5       E9BBA5       F3B1BD98
EA7D     EA7D EA7D EA7D EA7D 685D F3DE F3DE F3DE F3DE 735E 785F F3DE 9EE8      E9BBA8       E9BBA8       F3B1BD99
EA7E     EA7E EA7E EA7E EA7E 685E F3DF F3DF F3DF F3DF 735F 786D F3DF 9EEF      E9BBAF       E9BBAF       F3B1BD9A
EA80     EA80 EA80 EA80 EA80 685F F3E0 F3E0 F3E0 F3E0 7360 7879 F3E0 9EF4      E9BBB4       E9BBB4       F3B1BD9B
EA81     EA81 EA81 EA81 EA81 6860 F3E1 F3E1 F3E1 F3E1 7361 7857 F3E1 9EF6      E9BBB6       E9BBB6       F3B1BD9C
EA82     EA82 EA82 EA82 EA82 6861 F3E2 F3E2 F3E2 F3E2 7362 7859 F3E2 9EF7      E9BBB7       E9BBB7       F3B1BD9D
EA83     EA83 EA83 EA83 EA83 6862 F3E3 F3E3 F3E3 F3E3 7363 7862 F3E3 9EF9      E9BBB9       E9BBB9       F3B1BD9E
EA84     EA84 EA84 EA84 EA84 6863 F3E4 F3E4 F3E4 F3E4 7364 7863 F3E4 9EFB      E9BBBB       E9BBBB       F3B1BD9F
EA85     EA85 EA85 EA85 EA85 6864 F3E5 F3E5 F3E5 F3E5 7365 7864 F3E5 9EFC      E9BBBC       E9BBBC       F3B1BDA0
EA86     EA86 EA86 EA86 EA86 6865 F3E6 F3E6 F3E6 F3E6 7366 7865 F3E6 9EFD      E9BBBD       E9BBBD       F3B1BDA1
EA87     EA87 EA87 EA87 EA87 6866 F3E7 F3E7 F3E7 F3E7 7367 7866 F3E7 9F07      E9BC87       E9BC87       F3B1BDA2
EA88     EA88 EA88 EA88 EA88 6867 F3E8 F3E8 F3E8 F3E8 7368 7867 F3E8 9F08      E9BC88       E9BC88       F3B1BDA3
EA89     EA89 EA89 EA89 EA89 6868 F3E9 F3E9 F3E9 F3E9 7369 7868 F3E9 76B7      E79AB7       E79AB7       F3B1BDA4
EA8A     EA8A EA8A EA8A EA8A 6869 F3EA F3EA F3EA F3EA 736A 7869 F3EA 9F15      E9BC95       E9BC95       F3B1BDA5
EA8B     EA8B EA8B EA8B EA8B 686A F3EB F3EB F3EB F3EB 736B 7870 F3EB 9F21      E9BCA1       E9BCA1       F3B1BDA6
EA8C     EA8C EA8C EA8C EA8C 686B F3EC F3EC F3EC F3EC 736C 7871 F3EC 9F2C      E9BCAC       E9BCAC       F3B1BDA7
EA8D     EA8D EA8D EA8D EA8D 686C F3ED F3ED F3ED F3ED 736D 7872 F3ED 9F3E      E9BCBE       E9BCBE       F3B1BDA8
EA8E     EA8E EA8E EA8E EA8E 686D F3EE F3EE F3EE F3EE 736E 7873 F3EE 9F4A      E9BD8A       E9BD8A       F3B1BDA9
EA8F     EA8F EA8F EA8F EA8F 6870 F3EF F3EF F3EF F3EF 736F 7874 F3EF 9F52      E9BD92       E9BD92       F3B1BDAA
EA90     EA90 EA90 EA90 EA90 6871 F3F0 F3F0 F3F0 F3F0 7370 7875 F3F0 9F54      E9BD94       E9BD94       F3B1BDAB
EA91     EA91 EA91 EA91 EA91 6872 F3F1 F3F1 F3F1 F3F1 7371 7876 F3F1 9F63      E9BDA3       E9BDA3       F3B1BDAC
EA92     EA92 EA92 EA92 EA92 6873 F3F2 F3F2 F3F2 F3F2 7372 7877 F3F2 9F5F      E9BD9F       E9BD9F       F3B1BDAD
EA93     EA93 EA93 EA93 EA93 6874 F3F3 F3F3 F3F3 F3F3 7373 7878 F3F3 9F60      E9BDA0       E9BDA0       F3B1BDAE
EA94     EA94 EA94 EA94 EA94 6875 F3F4 F3F4 F3F4 F3F4 7374 7880 F3F4 9F61      E9BDA1       E9BDA1       F3B1BDAF
EA95     EA95 EA95 EA95 EA95 6876 F3F5 F3F5 F3F5 F3F5 7375 788B F3F5 9F66      E9BDA6       E9BDA6       F3B1BDB0
EA96     EA96 EA96 EA96 EA96 6877 F3F6 F3F6 F3F6 F3F6 7376 789B F3F6 9F67      E9BDA7       E9BDA7       F3B1BDB1
EA97     EA97 EA97 EA97 EA97 6878 F3F7 F3F7 F3F7 F3F7 7377 789C F3F7 9F6C      E9BDAC       E9BDAC       F3B1BDB2
EA98     EA98 EA98 EA98 EA98 6879 F3F8 F3F8 F3F8 F3F8 7378 78A0 F3F8 9F6A      E9BDAA       E9BDAA       F3B1BDB3
EA99     EA99 EA99 EA99 EA99 687A F3F9 F3F9 F3F9 F3F9 7379 78AB F3F9 9F77      E9BDB7       E9BDB7       F3B1BDB4
EA9A     EA9A EA9A EA9A EA9A 687B F3FA F3FA F3FA F3FA 737A 78B0 F3FA 9F72      E9BDB2       E9BDB2       F3B1BDB5
EA9B     EA9B EA9B EA9B EA9B 687C F3FB F3FB F3FB F3FB 737B 78C0 F3FB 9F76      E9BDB6       E9BDB6       F3B1BDB6
EA9C     EA9C EA9C EA9C EA9C 687D F3FC F3FC F3FC F3FC 737C 786A F3FC 9F95      E9BE95       E9BE95       F3B1BDB7
EA9D     EA9D EA9D EA9D EA9D 687E F3FD F3FD F3FD F3FD 737D 78D0 F3FD 9F9C      E9BE9C       E9BE9C       F3B1BDB8
EA9E     EA9E EA9E EA9E EA9E 687F F3FE F3FE F3FE F3FE 737E 78A1 F3FE 9FA0      E9BEA0       E9BEA0       F3B1BDB9
# Category 2 JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
EA9F     EA9F EA9F EA9F EA9F 4E53 5CC3 F4A1 F4A1 B6C6 3646 F6C6 F4A1 582F      E5A0AF       E5A0AF       F3B1BDBA
EAA0     EAA0 EAA0 EAA0 EAA0 4DB9 61FC F4A2 F4A2 CBEA 4B6A D269 F4A2 69C7      E6A787       E6A787       F3B1BDBB
EAA1     EAA1 EAA1 EAA1 EAA1 52F3 6BA2 F4A3 F4A3 CDDA 4D5A D4E9 F4A3 9059      E98199       E98199       F3B1BDBC
EAA2     EAA2 EAA2 EAA2 EAA2 5EBB 65A7 F4A4 F4A4 E0F6 6076 799B F4A4 7464      E791A4       E791A4       F3B1BDBD
# End of Category 2 JIS C 6226-1978 Versus JIS X 0208-1983, CJKV Information Processing by Ken Lunde 1999
# JIS X 0208-1983 Versus JIS X 0208-1990, CJKV Information Processing by Ken Lunde 1999
EAA3     EAA3 EAA3 EAA3 EAA3 6884  --   --  F4A5 44A4 B4A8 8448 F4A5 51DC      E5879C       E5879C       F3B1BDBE
EAA4     EAA4 EAA4 EAA4 EAA4 4F68 64B8 64B8 F4A6 58A8 C3BA 948A F4A6 7199      E78699       E78699       F3B1BDBF
# End of JIS X 0208-1983 Versus JIS X 0208-1990, CJKV Information Processing by Ken Lunde 1999
# IBM Selected Kanji and Non-kanji, Appendix C, CJKV Information Processing by Ken Lunde 1999
FA40     FA40 FA40 EEEF 86B3 41B1 74C1 74C1 74C1 77EB A2BD 428E  --  2170      E285B0       E285B0       F3B2AB9C
FA41     FA41 FA41 EEF0 86B4 41B2 74C2 74C2 74C2 77EC A2BE 428F  --  2171      E285B1       E285B1       F3B2AB9D
FA42     FA42 FA42 EEF1 86B5 41B3 74C3 74C3 74C3 77ED A2BF 4290  --  2172      E285B2       E285B2       F3B2AB9E
FA43     FA43 FA43 EEF2 86B6 41B4 74C4 74C4 74C4 77EE A2C0 4291  --  2173      E285B3       E285B3       F3B2AB9F
FA44     FA44 FA44 EEF3 86B7 41B5 74C5 74C5 74C5 77EF A2C1 4292  --  2174      E285B4       E285B4       F3B2ABA0
FA45     FA45 FA45 EEF4 86B8 41B6 74C6 74C6 74C6 77F0 A2C2 4293  --  2175      E285B5       E285B5       F3B2ABA1
FA46     FA46 FA46 EEF5 86B9 41B7 74C7 74C7 74C7 77F1 A2C3 4294  --  2176      E285B6       E285B6       F3B2ABA2
FA47     FA47 FA47 EEF6 86BA 41B8 74C8 74C8 74C8 77F2 A2C4 4295  --  2177      E285B7       E285B7       F3B2ABA3
FA48     FA48 FA48 EEF7 86BB 41B9 74C9 74C9 74C9 77F3 A2C5 4296  --  2178      E285B8       E285B8       F3B2ABA4
FA49     FA49 FA49 EEF8 86BC 41BA 74CA 74CA 74CA 77F4 A2C6 4297  --  2179      E285B9       E285B9       F3B2ABA5
FA55     FA55 FA55 EEFA  --  E9F5  --   --   --  76A1 A5FB 45FB  --  FFE4      EFBFA4       EFBFA4       F3B2ABB1
FA56     FA56 FA56 EEFB 81AD 427D  --   --   --  76A2 A3F9 43EF  --  FF07      EFBC87       EFBC87       F3B2ABB2
FA57     FA57 FA57 EEFC 81AE 427F  --   --   --  76A3 A3FA 43FA  --  FF02      EFBC82       EFBC82       F3B2ABB3
FA5C     FA5C FA5C ED40 EDB5 51BF 67FE 67FE 67FE 5FDC C9D8 9AAE D463 7E8A      E7BA8A       E7BA8A       F3B2ABB8
FA5D     FA5D FA5D ED41 EE8E 51DA 69E8 69E8 69E8 64B5 CEA9 A349 DC5F 891C      E8A49C       E8A49C       F3B2ABB9
FA5E     FA5E FA5E ED42 EF58 51F7 6CC7 6CC7 6CC7 6AD9 D2B9 A789 E469 9348      E98D88       E98D88       F3B2ABBA
FA5F     FA5F FA5F ED43 EF4D 524B 6CA3 6CA3 6CA3 69FE D1ED A6CF E378 9288      E98A88       E98A88       F3B2ABBB
FA60     FA60 FA60 ED44 F8D3 527B 69AE 69AE 69AE 62C4 CCCA 9F9D D921 84DC      E8939C       E8939C       F3B2ABBC
FA61     FA61 FA61 ED45 87B7 5294 59E4 59E4 59E4 42C7 B2F6 82EC B13B 4FC9      E4BF89       E4BF89       F3B2ABBD
FA62     FA62 FA62 ED46 F580 529E 63FE 63FE 63FE 70E1 C2FE 93FE  --  70BB      E782BB       E782BB       F3B2ABBE
FA63     FA63 FA63 ED47 EB54 52EB 60F8 60F8 60F8 51A3 BDBF 8E90 C22D 6631      E698B1       E698B1       F3B2ABBF
FA64     FA64 FA64 ED48 EB89 5386 61DC 61DC 61DC 70CD BFB7 9087 C37C 68C8      E6A388       E6A388       F3B2AC80
FA65     FA65 FA65 ED49 FB42 5394 6CBB 6CBB 6CBB 6AC9 D2B6 A786 E450 92F9      E98BB9       E98BB9       F3B2AC81
FA66     FA66 FA66 ED4A EB56 5397 60D3 60D3 60D3 51FC BDD0 8EA5 C23F 66FB      E69BBB       E69BBB       F3B2AC82
FA67     FA67 FA67 ED4B EAB8 53C7 5ED8 5ED8 5ED8 4CF6 BAC7 8A98 BC74 5F45      E5BD85       E5BD85       F3B2AC83
FA68     FA68 FA68 ED4C 87A2 5644 59AC 59AC 59AC 41AC B2AB 8252 B029 4E28      E4B8A8       E4B8A8       F3B2AC84
FA69     FA69 FA69 ED4D 87A6 565D 59C1 59C1 59C1 41E8 B2C2 8293 B048 4EE1      E4BBA1       E4BBA1       F3B2AC85
FA6A     FA6A FA6A ED4E  --  5662 59C8 59C8 59C8 43CF B2D3 82A8  --  4EFC      E4BBBC       E4BBBC       F3B2AC86
FA6B     FA6B FA6B ED4F F05E 5665 59C7 59C7 59C7 41EF B2CD 82A2 B052 4F00      E4BC80       E4BC80       F3B2AC87
FA6C     FA6C FA6C ED50 87A9 5666 9FA1 9FA1 9FA1 41F0 B2CE 82A3 B054 4F03      E4BC83       E4BC83       F3B2AC88
FA6D     FA6D FA6D ED51  --  566A 59CE 59CE 59CE 41FB B2DB 82BB B063 4F39      E4BCB9       E4BCB9       F3B2AC89
FA6E     FA6E FA6E ED52 87AE 566E 59CB 59CB 59CB 42A8 B2D6 82AC B06E 4F56      E4BD96       E4BD96       F3B2AC8A
FA6F     FA6F FA6F ED53 F06C 5670 59DB 59DB 59DB 42B5 B2EA 82CC B127 4F92      E4BE92       E4BE92       F3B2AC8B
FA70     FA70 FA70 ED54 87B3 5671 59E0 59E0 59E0 42B3 B2E7 82B9 B123 4F8A      E4BE8A       E4BE8A       F3B2AC8C
FA71     FA71 FA71 ED55 F06D 5674 59DA 59DA 59DA 42B9 BADA 8ABA B12C 4F9A      E4BE9A       E4BE9A       F3B2AC8D
FA72     FA72 FA72 ED56 87B4 567B 59D9 59D9 59D9 42B7 B2E1 82B3 B129 4F94      E4BE94       E4BE94       F3B2AC8E
FA73     FA73 FA73 ED57  --  5686 59E6 59E6 59E6 42CA B2F5 82EB B13E 4FCD      E4BF8D       E4BF8D       F3B2AC8F
FA74     FA74 FA74 ED58 87BF 568A 5AA1 5AA1 5AA1 42E7 B3B2 8382 B15F 5040      E58180       E58180       F3B2AC90
FA75     FA75 FA75 ED59 F07C 5691 59F1 59F1 59F1 42DF B3A1 8341 B158 5022      E580A2       E580A2       F3B2AC91
FA76     FA76 FA76 ED5A  --  5694 59FA 59FA 59FA 43D0 B3B3 8383 B148 4FFF      E4BFBF       E4BFBF       F3B2AC92
FA77     FA77 FA77 ED5B F07B 5698 59F3 59F3 59F3 42DE B3A6 8346 B157 501E      E5809E       E5809E       F3B2AC93
FA78     FA78 FA78 ED5C F080 56A2 5AA7 5AA7 5AA7 42E9 B3C0 8391 B163 5046      E58186       E58186       F3B2AC94
FA79     FA79 FA79 ED5D F087 56A3 5AAC 5AAC 5AAC 42F5 B3C3 8394 B174 5070      E581B0       E581B0       F3B2AC95
FA7A     FA7A FA7A ED5E F07E 56A4 5AA6 5AA6 5AA6 42E8 B3C4 8395 B161 5042      E58182       E58182       F3B2AC96
FA7B     FA7B FA7B ED5F 87C2 56A8 5AB0 5AB0 5AB0 42FC B3C6 8397 B223 5094      E58294       E58294       F3B2AC97
FA7C     FA7C FA7C ED60  --  56B2 5ABE 5ABE 5ABE 43B9 B3D2 83A7  --  50F4      E583B4       E583B4       F3B2AC98
FA7D     FA7D FA7D ED61  --  56B5 5AC1 5AC1 5AC1 43B2 B3D5 83AA B23B 50D8      E58398       E58398       F3B2AC99
FA7E     FA7E FA7E ED62 87CD 56C9 5ACB 5ACB 5ACB 43D3 B3E8 83CA B266 514A      E5858A       E5858A       F3B2AC9A
FA80     FA80 FA80 ED63  --  56CE 5AD3 5AD3 5AD3 43D9 B3F0 83DC B26D 5164      E585A4       E585A4       F3B2AC9B
FA81     FA81 FA81 ED64 87D2 56DC 5AE1 5AE1 5AE1 43F2 B3F7 83ED B275 519D      E5869D       E5869D       F3B2AC9C
FA82     FA82 FA82 ED65  --  56E8 5AE4 5AE4 5AE4 43F7 B4A2 8442 B27C 51BE      E586BE       E586BE       F3B2AC9D
FA83     FA83 FA83 ED66  --  56EF 6DC3 6DC3 6DC3 44AC D4A8 A948  --  51EC      E587AC       E587AC       F3B2AC9E
FA84     FA84 FA84 ED67 87DB 56F7 9FA3 9FA3 9FA3 44B4 B4B4 8484 B335 5215      E58895       E58895       F3B2AC9F
FA85     FA85 FA85 ED68  --  574F 5AF0 5AF0 5AF0 44F7 B4C1 8492 B358 529C      E58A9C       E58A9C       F3B2ACA0
FA86     FA86 FA86 ED69 F162 5750 5AF1 5AF1 5AF1 44D2 B4C3 8494 B35B 52A6      E58AA6       E58AA6       F3B2ACA1
FA87     FA87 FA87 ED6A  --  5756 5AF7 5AF7 5AF7 44FA B4D0 84A5 B365 52C0      E58B80       E58B80       F3B2ACA2
FA88     FA88 FA88 ED6B 87E5 5758 5BA1 5BA1 5BA1 44ED B4D8 84AE B36E 52DB      E58B9B       E58B9B       F3B2ACA3
FA89     FA89 FA89 ED6C 87E9 5761 5BA9 5BA9 5BA9 45A3 B4E8 84CA B37B 5300      E58C80       E58C80       F3B2ACA4
FA8A     FA8A FA8A ED6D 87EA 5763 5BAB 5BAB 5BAB 45A8 B4EC 84CE  --  5307      E58C87       E58C87       F3B2ACA5
FA8B     FA8B FA8B ED6E F16D 576B 5BAF 5BAF 5BAF 45B3 B4EF 84DB  --  5324      E58CA4       E58CA4       F3B2ACA6
FA8C     FA8C FA8C ED6F  --  577C 5BBC 5BBC 5BBC 45D2 B5A7 8547 B440 5372      E58DB2       E58DB2       F3B2ACA7
FA8D     FA8D FA8D ED70 87F0 5780 5BC4 5BC4 5BC4 45DB B5AD 8554 B447 5393      E58E93       E58E93       F3B2ACA8
FA8E     FA8E FA8E ED71 87F2 5787 5BC9 5BC9 5BC9 45E2 B5B1 8581 B450 53B2      E58EB2       E58EB2       F3B2ACA9
FA8F     FA8F FA8F ED72  --  578B 5BCE 5BCE 5BCE 45F3 B5BC 858D B45E 53DD      E58F9D       E58F9D       F3B2ACAA
FA90     FA90 FA90 ED73  --  578E 5BD3 5BD3 5BD3 45F2 B5BE 858F  --  FA0E      EFA88E       EFA88E       F3B2ACAB
FA91     FA91 FA91 ED74 87F6 57A9 5BDB 5BDB 5BDB 46C4 B5D3 85A8 B52A 549C      E5929C       E5929C       F3B2ACAC
FA92     FA92 FA92 ED75 F18C 57AE 5BDA 5BDA 5BDA 46C1 B5CF 85A4  --  548A      E5928A       E5928A       F3B2ACAD
FA93     FA93 FA93 ED76 8840 57BD 5BE1 5BE1 5BE1 46C9 B5DC 85BC B52F 54A9      E592A9       E592A9       F3B2ACAE
FA94     FA94 FA94 ED77 8842 57C0 5BE3 5BE3 5BE3 45FA B5E0 85B2 B544 54FF      E593BF       E593BF       F3B2ACAF
FA95     FA95 FA95 ED78 8849 57DE 5BED 5BED 5BED 45F6 B5EE 85DA B568 5586      E59686       E59686       F3B2ACB0
FA96     FA96 FA96 ED79  --  5860 5CB2 5CB2 5CB2 48C9 B6D6 86AC  --  5759      E59D99       E59D99       F3B2ACB1
FA97     FA97 FA97 ED7A  --  5863 5CB6 5CB6 5CB6 48CA B6DE 86BE B742 5765      E59DA5       E59DA5       F3B2ACB2
FA98     FA98 FA98 ED7B F1EA 586C 5CB9 5CB9 5CB9 48CB B6E3 86B5 B764 57AC      E59EAC       E59EAC       F3B2ACB3
FA99     FA99 FA99 ED7C 886E 5870 5CBD 5CBD 5CBD 47F7 B6E5 86B7 B768 57C8      E59F88       E59F88       F3B2ACB4
FA9A     FA9A FA9A ED7D 886D 5873 9FA6 9FA6 9FA6 47F6 B6E6 86B8 B767 57C7      E59F87       E59F87       F3B2ACB5
FA9B     FA9B FA9B ED7E 886A 5877 5CBB 5CBB 5CBB 48CC B6E7 86B9  --  FA0F      EFA88F       EFA88F       F3B2ACB6
#
# U+FA10
# https://glyphwiki.org/wiki/ufa10
#
FA9C     FA9C FA9C ED80 8876 5882 C4CD 5CC7 5CC7 C4CD 444D C4D4  --  FA10      EFA890       EFA890       F3B2ACB7
FA9D     FA9D FA9D ED81 887C 588A 5CD1 5CD1 5CD1 48BB B7A8 8748  --  589E      E5A29E       E5A29E       F3B2ACB8
FA9E     FA9E FA9E ED82  --  588C 5CD3 5CD3 5CD3 48CD B7AE 8755 B84E 58B2      E5A2B2       E5A2B2       F3B2ACB9
FA9F     FA9F FA9F ED83 F259 58A1 5BCD 5BCD 5BCD 48D0 B7B8 8788 B861 590B      E5A48B       E5A48B       F3B2ACBA
FAA0     FAA0 FAA0 ED84  --  58AD 5CE4 5CE4 5CE4 48E3 B7CB 879E B875 5953      E5A593       E5A593       F3B2ACBB
FAA1     FAA1 FAA1 ED85 F264 58B0 5CE6 5CE6 5CE6 48E4 B7CF 87A4 B877 595B      E5A59B       E5A59B       F3B2ACBC
FAA2     FAA2 FAA2 ED86 8889 58B1 5CE8 5CE8 5CE8 48E5 B7CE 87A3 B878 595D      E5A59D       E5A59D       F3B2ACBD
FAA3     FAA3 FAA3 ED87  --  58B2 5CE9 5CE9 5CE9 48EB B7D3 87A8 B87C 5963      E5A5A3       E5A5A3       F3B2ACBE
FAA4     FAA4 FAA4 ED88 888D 58BE 5CF4 5CF4 5CF4 49A3 B7DB 87BB B92F 59A4      E5A6A4       E5A6A4       F3B2ACBF
FAA5     FAA5 FAA5 ED89  --  58C0 5CF7 5CF7 5CF7 49F0 B7E4 87B6 B937 59BA      E5A6BA       E5A6BA       F3B2AD80
FAA6     FAA6 FAA6 ED8A 9875 58EC 5DB3 5DB3 5DB3 49F2 B8B4 8884 BA3E 5B56      E5AD96       E5AD96       F3B2AD81
FAA7     FAA7 FAA7 ED8B F0A2 58FC 5DC2 5DC2 5DC2 4AC9 B8CA 889D BA5B 5BC0      E5AF80       E5AF80       F3B2AD82
FAA8     FAA8 FAA8 ED8C ECC7 58FE 5DC5 5DC5 5DC5 4ABD B8CC 889F CD2A 752F      E794AF       E794AF       F3B2AD83
FAA9     FAA9 FAA9 ED8D 9878 5942 5DC6 5DC6 5DC6 4AC0 B8CD 88A2 BA61 5BD8      E5AF98       E5AF98       F3B2AD84
FAAA     FAAA FAAA ED8E 9879 5944 5DC7 5DC7 5DC7 4AC6 B8D4 88A9  --  5BEC      E5AFAC       E5AFAC       F3B2AD85
FAAB     FAAB FAAB ED8F 987B 5951 5DD2 5DD2 5DD2 4AD4 B8E3 88B5 BA6B 5C1E      E5B09E       E5B09E       F3B2AD86
FAAC     FAAC FAAC ED90 F0C1 5967 5DDE 5DDE 5DDE 4AF1 B8FB 88FB BB33 5CA6      E5B2A6       E5B2A6       F3B2AD87
FAAD     FAAD FAAD ED91 9889 5968 5DDD 5DDD 5DDD 4AF3 B8F9 88EF BB38 5CBA      E5B2BA       E5B2BA       F3B2AD88
FAAE     FAAE FAAE ED92  --  5970 5DE6 5DE6 5DE6 4BE4 B9A9 8949  --  5CF5      E5B3B5       E5B3B5       F3B2AD89
FAAF     FAAF FAAF ED93 9891 5978 5DE8 5DE8 5DE8 4AFB B9B1 8981 BB4A 5D27      E5B4A7       E5B4A7       F3B2AD8A
FAB0     FAB0 FAB0 ED94 9895 597F 5DF0 5DF0 5DF0 4BA3 C6B7 9787  --  5D53      E5B593       E5B593       F3B2AD8B
FAB1     FAB1 FAB1 ED95 9892 5980 5DEC 5DEC 5DEC 4BD0 B9BA 898A  --  FA11      EFA891       EFA891       F3B2AD8C
FAB2     FAB2 FAB2 ED96 F0D3 5984 5DEF 5DEF 5DEF 4BA2 B9BD 898E BB50 5D42      E5B582       E5B582       F3B2AD8D
FAB3     FAB3 FAB3 ED97 9897 5987 9FA9 9FA9 9FA9 4BD3 B9C0 8991 BB5E 5D6D      E5B5AD       E5B5AD       F3B2AD8E
FAB4     FAB4 FAB4 ED98 989C 5993 5EA3 5EA3 5EA3 4BDF B9D3 89A8 BB74 5DB8      E5B6B8       E5B6B8       F3B2AD8F
FAB5     FAB5 FAB5 ED99 F0E5 5995 5EA5 5EA5 5EA5 4BE0 B9D5 89AA BB75 5DB9      E5B6B9       E5B6B9       F3B2AD90
FAB6     FAB6 FAB6 ED9A  --  5997 5EA7 5EA7 5EA7 4BE5 B9DA 89BA BB79 5DD0      E5B790       E5B790       F3B2AD91
FAB7     FAB7 FAB7 ED9B  --  59C7 5ED2 5ED2 5ED2 4CEB BABF 8A90 BC64 5F21      E5BCA1       E5BCA1       F3B2AD92
FAB8     FAB8 FAB8 ED9C EAB4 59CB 5ED3 5ED3 5ED3 4CEE BAC1 8A92 BC6D 5F34      E5BCB4       E5BCB4       F3B2AD93
FAB9     FAB9 FAB9 ED9D EABC 59D8 5FCA 5FCA 5FCA 4DA2 BAD1 8AA6 BC7E 5F67      E5BDA7       E5BDA7       F3B2AD94
FABA     FABA FABA ED9E EAC3 59E8 5EE6 5EE6 5EE6 4DC0 BAEC 8ACE  --  5FB7      E5BEB7       E5BEB7       F3B2AD95
FABB     FABB FABB ED9F EAC5 59F1 5EEE 5EEE 5EEE 4DC7 BAFE 8AFE BD42 5FDE      E5BF9E       E5BF9E       F3B2AD96
FABC     FABC FABC EDA0  --  5A44 5EF8 5EF8 5EF8 4DCC BBA9 8C49 BD67 605D      E6819D       E6819D       F3B2AD97
FABD     FABD FABD EDA1  --  5A51 5EF9 5EF9 5EF9 4EA8 BBB1 8C81  --  6085      E68285       E68285       F3B2AD98
FABE     FABE FABE EDA2 F2CB 5A57 5EF6 5EF6 5EF6 4DD0 B5E5 85B7 BD70 608A      E6828A       E6828A       F3B2AD99
FABF     FABF FABF EDA3  --  5A5F 5FA5 5FA5 5FA5 4EC1 BBC4 8C95 BE30 60DE      E6839E       E6839E       F3B2AD9A
FAC0     FAC0 FAC0 EDA4 EAD3 5A65 5FA3 5FA3 5FA3 4EBD BBBD 8C8E BE2C 60D5      E68395       E68395       F3B2AD9B
FAC1     FAC1 FAC1 EDA5  --  5A67 5FAA 5FAA 5FAA 4EED BBC9 8C9A  --  6120      E684A0       E684A0       F3B2AD9C
FAC2     FAC2 FAC2 EDA6 F2D6 5A68 5FAD 5FAD 5FAD 4EC2 BBCE 8CA3 BE33 60F2      E683B2       E683B2       F3B2AD9D
FAC3     FAC3 FAC3 EDA7  --  5A76 5FAF 5FAF 5FAF 4EEE BBD4 8CA9 BE3D 6111      E68491       E68491       F3B2AD9E
FAC4     FAC4 FAC4 EDA8 EAD9 5A79 5FB1 5FB1 5FB1 4ED0 BBD6 8CAC BE4D 6137      E684B7       E684B7       F3B2AD9F
FAC5     FAC5 FAC5 EDA9 EAD8 5A7E 5FB3 5FB3 5FB3 4ECE BBD8 8CAE BE49 6130      E684B0       E684B0       F3B2ADA0
FAC6     FAC6 FAC6 EDAA F2E7 5A94 5FBE 5FBE 5FBE 4EDB B6A8 8648 BE64 6198      E68698       E68698       F3B2ADA1
FAC7     FAC7 FAC7 EDAB  --  5AB5 5BD7 5BD7 5BD7 4EF7 BBF9 8CEF BF28 6213      E68893       E68893       F3B2ADA2
FAC8     FAC8 FAC8 EDAC F341 5ADA 5FD6 5FD6 5FD6 4FB5 BCB1 8D81 BF49 62A6      E68AA6       E68AA6       F3B2ADA3
FAC9     FAC9 FAC9 EDAD EAF2 5AFB 5FE2 5FE2 5FE2 4FED BCCC 8D9F C044 63F5      E68FB5       E68FB5       F3B2ADA4
FACA     FACA FACA EDAE EAF8 5B50 5FEB 5FEB 5FEB 4FFA BCD9 8DAF C064 6460      E691A0       E691A0       F3B2ADA5
FACB     FACB FACB EDAF F36F 5B53 5FEE 5FEE 5FEE 50A2 BCDB 8DBB C074 649D      E6929D       E6929D       F3B2ADA6
FACC     FACC FACC EDB0 EAFA 5B54 5FF2 5FF2 5FF2 4FA7 BCDC 8DBC C126 64CE      E6938E       E6938E       F3B2ADA7
FACD     FACD FACD EDB1  --  5B7E 5FF8 5FF8 5FF8 50C9 BCF0 8DDC  --  654E      E6958E       E6958E       F3B2ADA8
FACE     FACE FACE EDB2 EB4B 5B97 60B3 60B3 60B3 51BD BDBD 8E8E C175 6600      E69880       E69880       F3B2ADA9
FACF     FACF FACF EDB3 EB4D 5B98 60B1 60B1 60B1 51C4 BDB5 8E85 C17C 6615      E69895       E69895       F3B2ADAA
# IBM Selected Kanji and Non-Kanji, Appendix Q, CJKV Information Processing by Ken Lunde 1999
#
# U+663B
# https://glyphwiki.org/wiki/u663b
#
FAD0     FAD0 FAD0 EDB4  --  4F9F B9B7 60B6 60B6 B9B7 3937 F9F7  --  663B      E698BB       E698BB       F3B2ADAB
# End of IBM Selected Kanji and Non-Kanji, Appendix Q, CJKV Information Processing by Ken Lunde 1999
FAD1     FAD1 FAD1 EDB5 EB4C 5B9D 60AF 60AF 60AF 51C0 BDB7 8E87 C178 6609      E69889       E69889       F3B2ADAC
FAD2     FAD2 FAD2 EDB6  --  5BA0 60BE 60BE 60BE 51A1 BDC2 8E93 C22B 662E      E698AE       E698AE       F3B2ADAD
FAD3     FAD3 FAD3 EDB7 EB4E 5BA1 60C4 60C4 60C4 51C7 BDC4 8E95 C221 661E      E6989E       E6989E       F3B2ADAE
FAD4     FAD4 FAD4 EDB8 EB51 5BA4 60B9 60B9 60B9 51CA BDC7 8E98 C225 6624      E698A4       E698A4       F3B2ADAF
FAD5     FAD5 FAD5 EDB9 EB5D 5BA7 60D1 60D1 60D1 51D9 BDD6 8EAC  --  6665      E699A5       E699A5       F3B2ADB0
FAD6     FAD6 FAD6 EDBA F39E 5BA8 60D6 60D6 60D6 52A1 BDE0 8EB2 C238 6657      E69997       E69997       F3B2ADB1
FAD7     FAD7 FAD7 EDBB EB5A 5BAB 60D7 60D7 60D7 51D4 BDDE 8EBE C23A 6659      E69999       E69999       F3B2ADB2
FAD8     FAD8 FAD8 EDBC  --  5BAE 60DD 60DD 60DD 51DE BDE4 8EB6  --  FA12      EFA892       EFA892       F3B2ADB3
FAD9     FAD9 FAD9 EDBD EB5E 5BAF 60E1 60E1 60E1 51B3 BDE6 8EB8 C244 6673      E699B3       E699B3       F3B2ADB4
FADA     FADA FADA EDBE F3AB 5BB4 60E6 60E6 60E6 51E5 BDF7 8EED C252 6699      E69A99       E69A99       F3B2ADB5
FADB     FADB FADB EDBF EB63 5BB6 60EA 60EA 60EA 51AB C5D8 96AE C257 66A0      E69AA0       E69AA0       F3B2ADB6
FADC     FADC FADC EDC0 EB64 5BB8 60EE 60EE 60EE 51EB BEA5 8F45 C25B 66B2      E69AB2       E69AB2       F3B2ADB7
FADD     FADD FADD EDC1 F3B1 5BBB 60EF 60EF 60EF 51F1 BEAC 8F53 C25E 66BF      E69ABF       E69ABF       F3B2ADB8
FADE     FADE FADE EDC2 EB6B 5BC7 60C7 60C7 60C7 52A2 BDD1 8EA6 C26D 66FA      E69BBA       E69BBA       F3B2ADB9
FADF     FADF FADF EDC3  --  5BC8 60FC 60FC 60FC 52BC BEBC 8F8D C270 670E      E69C8E       E69C8E       F3B2ADBA
FAE0     FAE0 FAE0 EDC4 EB6D 5BC9 61A7 61A7 61A7 52AD BEBE 8F8F  --  F929      EFA4A9       EFA4A9       F3B2ADBB
FAE1     FAE1 FAE1 EDC5 EB6F 5BD8 61B2 61B2 61B2 70CE BEC8 8F99 C321 6766      E69DA6       E69DA6       F3B2ADBC
FAE2     FAE2 FAE2 EDC6 EB77 5BE3 61BB 61BB 61BB 53B2 BED9 8FAF C335 67BB      E69EBB       E69EBB       F3B2ADBD
FAE3     FAE3 FAE3 EDC7 EB7D 5BEC 61BE 61BE 61BE 52EE BEE5 8FB7 C349 6852      E6A192       E6A192       F3B2ADBE
FAE4     FAE4 FAE4 EDC8 EB79 5BF0 9FAC 9FAC 9FAC 53B4 BEE8 8FCA C339 67C0      E69F80       E69F80       F3B2ADBF
FAE5     FAE5 FAE5 EDC9 EB7C 5BF3 61CC 61CC 61CC 53C0 BEE2 8FB4  --  6801      E6A081       E6A081       F3B2AE80
FAE6     FAE6 FAE6 EDCA F3DD 5BF9 61C6 61C6 61C6 53C7 BEF0 8FDC C358 6844      E6A184       E6A184       F3B2AE81
FAE7     FAE7 FAE7 EDCB F3FB 5C5F 61DF 61DF 61DF 70CF BFBA 908A C37E 68CF      E6A38F       E6A38F       F3B2AE82
FAE8     FAE8 FAE8 EDCC F3F7 5C66 61E0 61E0 61E0 70D0 BFBC 908D  --  FA13      EFA893       EFA893       F3B2AE83
FAE9     FAE9 FAE9 EDCD EB98 5C6E 61E9 61E9 61E9 53ED BFC0 9091 C44C 6968      E6A5A8       E6A5A8       F3B2AE84
FAEA     FAEA FAEA EDCE EB9A 5C78 61F8 61F8 61F8 70D1 BFD7 90AD  --  FA14      EFA894       EFA894       F3B2AE85
FAEB     FAEB FAEB EDCF F455 5C79 65F4 65F4 65F4 52F4 C6AA 9751 C459 6998      E6A698       E6A698       F3B2AE86
FAEC     FAEC FAEC EDD0 EBA0 5C90 61EF 61EF 61EF 70D2 BFE4 90B6 C46A 69E2      E6A7A2       E6A7A2       F3B2AE87
FAED     FAED FAED EDD1 F460 5C9A 62A3 62A3 62A3 70D3 BFE5 90B7 C47D 6A30      E6A8B0       E6A8B0       F3B2AE88
FAEE     FAEE FAEE EDD2 EBAE 5C9B 62A9 62A9 62A9 54AD BFE6 90B8  --  6A6B      E6A9AB       E6A9AB       F3B2AE89
FAEF     FAEF FAEF EDD3 F472 5CA6 62AE 62AE 62AE 52E8 BFEC 90CE C527 6A46      E6A986       E6A986       F3B2AE8A
FAF0     FAF0 FAF0 EDD4 EBAF 5CA9 62B2 62B2 62B2 54AF BFF0 90DC C535 6A73      E6A9B3       E6A9B3       F3B2AE8B
FAF1     FAF1 FAF1 EDD5 F479 5CAF 62B3 62B3 62B3 54B0 BFF8 90EE C536 6A7E      E6A9BE       E6A9BE       F3B2AE8C
FAF2     FAF2 FAF2 EDD6  --  5CB9 62B9 62B9 62B9 54BC C0A8 9148  --  6AE2      E6ABA2       E6ABA2       F3B2AE8D
FAF3     FAF3 FAF3 EDD7 EBB6 5CBC 62B8 62B8 62B8 54BD C0A5 9145 C555 6AE4      E6ABA4       E6ABA4       F3B2AE8E
FAF4     FAF4 FAF4 EDD8 EBC9 5CEA 62D0 62D0 62D0 54F8 C0CF 91A4 C638 6BD6      E6AF96       E6AF96       F3B2AE8F
FAF5     FAF5 FAF5 EDD9 F4B3 5CF6 62D5 62D5 62D5 55B5 C0D9 91AF C657 6C3F      E6B0BF       E6B0BF       F3B2AE90
FAF6     FAF6 FAF6 EDDA EBD0 5CF9 62D6 62D6 62D6 55BD C0DE 91BE C660 6C5C      E6B19C       E6B19C       F3B2AE91
FAF7     FAF7 FAF7 EDDB EBD5 5CFD 62D8 62D8 62D8 55C3 C0E5 91B7 C66A 6C86      E6B286       E6B286       F3B2AE92
FAF8     FAF8 FAF8 EDDC  --  5CFE 62D9 62D9 62D9 55BF C0EB 91CD C663 6C6F      E6B1AF       E6B1AF       F3B2AE93
FAF9     FAF9 FAF9 EDDD F4C2 5D4E 62E2 62E2 62E2 55D3 C0F4 91EA C721 6CDA      E6B39A       E6B39A       F3B2AE94
FAFA     FAFA FAFA EDDE EBDF 5D58 62E4 62E4 62E4 55DE C1A7 9247 C72B 6D04      E6B484       E6B484       F3B2AE95
FAFB     FAFB FAFB EDDF EBE9 5D60 62EB 62EB 62EB 56A3 C1B6 9286 C747 6D87      E6B687       E6B687       F3B2AE96
FAFC     FAFC FAFC EDE0  --  5D62 62F2 62F2 62F2 70DE C1C7 9298 C743 6D6F      E6B5AF       E6B5AF       F3B2AE97
FB40     FB40 FB40 EDE1  --  5D6B 62EE 62EE 62EE 56A7 C1BF 9290 C74B 6D96      E6B696       E6B696       F3B2AE98
FB41     FB41 FB41 EDE2 EBED 5D72 62FC 62FC 62FC 56AD C1CA 929D C74F 6DAC      E6B6AC       E6B6AC       F3B2AE99
FB42     FB42 FB42 EDE3 F4E6 5D74 63A4 63A4 63A4 56B2 C1D9 92AF C759 6DCF      E6B78F       E6B78F       F3B2AE9A
FB43     FB43 FB43 EDE4  --  5D7A 62FD 62FD 62FD 56B8 C1CD 92A2  --  6DF8      E6B7B8       E6B7B8       F3B2AE9B
FB44     FB44 FB44 EDE5  --  5D7F 63A6 63A6 63A6 70DF C1DF 92BF  --  6DF2      E6B7B2       E6B7B2       F3B2AE9C
FB45     FB45 FB45 EDE6 EBF4 5D80 63BD 63BD 63BD 55B1 C1D1 92A6 C766 6DFC      E6B7BC       E6B7BC       F3B2AE9D
FB46     FB46 FB46 EDE7  --  5D8A 63B7 63B7 63B7 56C2 C1FB 92FB C76E 6E39      E6B8B9       E6B8B9       F3B2AE9E
FB47     FB47 FB47 EDE8 EBF8 5D8E 63B4 63B4 63B4 56CF C1F3 92DF C77C 6E5C      E6B99C       E6B99C       F3B2AE9F
FB48     FB48 FB48 EDE9 F4EE 5D93 63AE 63AE 63AE 56BE C1F4 92EA C76B 6E27      E6B8A7       E6B8A7       F3B2AEA0
FB49     FB49 FB49 EDEA F4F1 5D95 63B8 63B8 63B8 56C3 C1F0 92DC C770 6E3C      E6B8BC       E6B8BC       F3B2AEA1
FB4A     FB4A FB4A EDEB F543 5DA6 9FB6 9FB6 9FB6 56DF C2A8 9348 C831 6EBF      E6BABF       E6BABF       F3B2AEA2
FB4B     FB4B FB4B EDEC EC4E 5DC1 63D8 63D8 63D8 56FE C2CE 93A3 C865 6F88      E6BE88       E6BE88       F3B2AEA3
FB4C     FB4C FB4C EDED F560 5DCA 63DF 63DF 63DF 57AE C2DF 93BF C878 6FB5      E6BEB5       E6BEB5       F3B2AEA4
FB4D     FB4D FB4D EDEE EC5A 5DD5 63E4 63E4 63E4 57B8 C2E4 93B6 C926 6FF5      E6BFB5       E6BFB5       F3B2AEA5
FB4E     FB4E FB4E EDEF EC5B 5DD7 63EB 63EB 63EB 57B9 C2EF 93DB C92B 7005      E78085       E78085       F3B2AEA6
FB4F     FB4F FB4F EDF0 F569 5DD8 63E8 63E8 63E8 57BA C2EE 93DA C92D 7007      E78087       E78087       F3B2AEA7
FB50     FB50 FB50 EDF1 EC5D 5DE3 63F1 63F1 63F1 57BC C2F1 93DD  --  7028      E780A8       E780A8       F3B2AEA8
FB51     FB51 FB51 EDF2 EC64 5DEA 63F8 63F8 63F8 57D1 C2FB 93FB C94A 7085      E78285       E78285       F3B2AEA9
FB52     FB52 FB52 EDF3 EC66 5DED 63FD 63FD 63FD 57DE C2FC 93FC C953 70AB      E782AB       E782AB       F3B2AEAA
FB53     FB53 FB53 EDF4 EC6B 5DF5 9FB9 9FB9 9FB9 58A6 C3A8 9448 C969 710F      E7848F       E7848F       F3B2AEAB
FB54     FB54 FB54 EDF5 F58B 5DF6 64A5 64A5 64A5 58A5 C3A6 9446 C963 7104      E78484       E78484       F3B2AEAC
FB55     FB55 FB55 EDF6 EC76 5DFB 64AD 64AD 64AD 57F1 C3B8 9488 C97C 715C      E7859C       E7859C       F3B2AEAD
FB56     FB56 FB56 EDF7 EC71 5DFC 64AE 64AE 64AE 57EB C3B9 9489 C974 7146      E78586       E78586       F3B2AEAE
FB57     FB57 FB57 EDF8 EC72 5DFD 64AF 64AF 64AF 57EC C3BC 948D C975 7147      E78587       E78587       F3B2AEAF
FB58     FB58 FB58 EDF9 EC79 5E49 64B9 64B9 64B9 70E3 C3CE 94A3  --  FA15      EFA895       EFA895       F3B2AEB0
FB59     FB59 FB59 EDFA EC7D 5E4B 64B4 64B4 64B4 57F6 C3D0 94A5 CA33 71C1      E78781       E78781       F3B2AEB1
FB5A     FB5A FB5A EDFB EC81 5E5A 64C2 64C2 64C2 58A9 C3D5 94AA CA3D 71FE      E787BE       E787BE       F3B2AEB2
FB5B     FB5B FB5B EDFC F5BB 5E73 64CC 64CC 64CC 58FA C3EB 94CD CA6F 72B1      E78AB1       E78AB1       F3B2AEB3
FB5C     FB5C FB5C EE40 EC89 5E75 9FBA 9FBA 9FBA 58D6 C3EE 94DA CA71 72BE      E78ABE       E78ABE       F3B2AEB4
FB5D     FB5D FB5D EE41  --  5E86 64D0 64D0 64D0 58ED C3FB 94FB CB2E 7324      E78CA4       E78CA4       F3B2AEB5
FB5E     FB5E FB5E EE42 EC8F 5E8A 64CE 64CE 64CE 58EF C3F8 94EE  --  FA16      EFA896       EFA896       F3B2AEB6
FB5F     FB5F FB5F EE43 F5D5 5E94 64D2 64D2 64D2 58F6 C4A5 9545 CB4A 7377      E78DB7       E78DB7       F3B2AEB7
FB60     FB60 FB60 EE44  --  5E99 64DF 64DF 64DF 59BA C4BA 958A CB66 73BD      E78EBD       E78EBD       F3B2AEB8
FB61     FB61 FB61 EE45 EC99 5E9E 64DB 64DB 64DB 59BC C4B8 9588 CB6A 73C9      E78F89       E78F89       F3B2AEB9
FB62     FB62 FB62 EE46 EC9B 5E9F 64E4 64E4 64E4 59C0 C4C1 9592 CB70 73D6      E78F96       E78F96       F3B2AEBA
FB63     FB63 FB63 EE47 EC9D 5EA1 64E5 64E5 64E5 59C3 C4C2 9593 CB74 73E3      E78FA3       E78FA3       F3B2AEBB
FB64     FB64 FB64 EE48  --  5EA2 64E8 64E8 64E8 5AAC C4C6 9597 CB6E 73D2      E78F92       E78F92       F3B2AEBC
FB65     FB65 FB65 EE49 EC9F 5EA7 64EB 64EB 64EB 59D0 C4C7 9598 CC25 7407      E79087       E79087       F3B2AEBD
FB66     FB66 FB66 EE4A  --  5EA8 64EC 64EC 64EC 59C7 C4CB 959E CB79 73F5      E78FB5       E78FB5       F3B2AEBE
FB67     FB67 FB67 EE4B ECA4 5EAA 64F5 64F5 64F5 59D7 C4D2 95A7 CC2B 7426      E790A6       E790A6       F3B2AEBF
FB68     FB68 FB68 EE4C ECA6 5EAB 64F6 64F6 64F6 59DA C4D3 95A8 CC2E 742A      E790AA       E790AA       F3B2AF80
FB69     FB69 FB69 EE4D  --  5EAD 64FC 64FC 64FC 59D9 C4DB 95BB CC2D 7429      E790A9       E790A9       F3B2AF81
FB6A     FB6A FB6A EE4E ECA9 5EAE 64F8 64F8 64F8 59DD C4D7 95AD CC32 742E      E790AE       E790AE       F3B2AF82
FB6B     FB6B FB6B EE4F ECB1 5EBC 65A8 65A8 65A8 5AAD C4ED 95CF CC42 7462      E791A2       E791A2       F3B2AF83
FB6C     FB6C FB6C EE50 ECB6 5EBD 65AE 65AE 65AE 59F4 C4F3 95DF CC50 7489      E79289       E79289       F3B2AF84
FB6D     FB6D FB6D EE51 ECB9 5EC0 65B2 65B2 65B2 59FB C4FB 95FB CC59 749F      E7929F       E7929F       F3B2AF85
FB6E     FB6E FB6E EE52 ECC5 5ED1 65B9 65B9 65B9 5AB8 C5AC 9653  --  7501      E79481       E79481       F3B2AF86
FB6F     FB6F FB6F EE53 ECC8 5EE7 65C3 65C3 65C3 5AD0 C5B4 9684 CD3B 756F      E795AF       E795AF       F3B2AF87
FB70     FB70 FB70 EE54  --  5F69 65CA 65CA 65CA 5AFE C5D3 96A8  --  7682      E79A82       E79A82       F3B2AF88
FB71     FB71 FB71 EE55 F68F 5F70 65CE 65CE 65CE 5BA4 C5D9 96AF CE3B 769C      E79A9C       E79A9C       F3B2AF89
FB72     FB72 FB72 EE56 ECE1 5F71 65CF 65CF 65CF 5BA5 C5DA 96BA  --  769E      E79A9E       E79A9E       F3B2AF8A
FB73     FB73 FB73 EE57 ECDF 5F72 65D1 65D1 65D1 5AFD C5DB 96BB CE3A 769B      E79A9B       E79A9B       F3B2AF8B
FB74     FB74 FB74 EE58 ECE2 5F73 65D2 65D2 65D2 5BA9 C5DF 96BF CE43 76A6      E79AA6       E79AA6       F3B2AF8C
FB75     FB75 FB75 EE59  --  5F7A 65D8 65D8 65D8 5BB7 C5E5 96B7  --  FA17      EFA897       EFA897       F3B2AF8D
FB76     FB76 FB76 EE5A ECF0 5F8E 65E6 65E6 65E6 5BCC C5F2 96DE CE72 7746      E79D86       E79D86       F3B2AF8E
FB77     FB77 FB77 EE5B F163 5FAA 65F6 65F6 65F6 44DC B4C7 8498 B35D 52AF      E58AAF       E58AAF       F3B2AF8F
FB78     FB78 FB78 EE5C F6C1 5FAE 65F7 65F7 65F7 5BFC C6B1 9781 CF55 7821      E7A0A1       E7A0A1       F3B2AF90
FB79     FB79 FB79 EE5D ED41 5FB2 65FA 65FA 65FA 5CA7 C6B5 9785 CF62 784E      E7A18E       E7A18E       F3B2AF91
FB7A     FB7A FB7A EE5E F6C4 5FB3 66A1 66A1 66A1 5CAE C6BD 978E CF69 7864      E7A1A4       E7A1A4       F3B2AF92
FB7B     FB7B FB7B EE5F  --  5FB6 66A5 66A5 66A5 5CB2 C6C4 9795 CF6D 787A      E7A1BA       E7A1BA       F3B2AF93
FB7C     FB7C FB7C EE60 F6DC 5FD1 66AE 66AE 66AE 5CCE C6DA 97BA  --  7930      E7A4B0       E7A4B0       F3B2AF94
FB7D     FB7D FB7D EE61  --  5FD2 66B0 66B0 66B0 70E5 C6DD 97BD  --  FA18      EFA898       EFA898       F3B2AF95
FB7E     FB7E FB7E EE62 ED5B 5FD5 66BB 66BB 66BB 5CDA C6EB 97CD  --  FA19      EFA899       EFA899       F3B2AF96
FB80     FB80 FB80 EE63 ED5C 5FD9 66C2 66C2 66C2 5CDC C6F7 97ED  --  FA1A      EFA89A       EFA89A       F3B2AF97
FB81     FB81 FB81 EE64 F6E5 5FDC 66D0 66D0 66D0 5CE6 C7A3 9843 D065 7994      E7A694       E7A694       F3B2AF98
FB82     FB82 FB82 EE65 ED60 5FDD 66D3 66D3 66D3 5CE4 C7A6 9846  --  FA1B      EFA89B       EFA89B       F3B2AF99
FB83     FB83 FB83 EE66 F6E8 5FDF 66D4 66D4 66D4 5CE8 C7AA 9851 D069 799B      E7A69B       E7A69B       F3B2AF9A
FB84     FB84 FB84 EE67 F758 604F 66FE 66FE 66FE 5DDD C7F2 98DE D168 7AD1      E7AB91       E7AB91       F3B2AF9B
FB85     FB85 FB85 EE68 F759 6058 67A2 67A2 67A2 5DE1 C7F5 98EB  --  7AE7      E7ABA7       E7ABA7       F3B2AF9C
FB86     FB86 FB86 EE69  --  6059 67A6 67A6 67A6 5DE5 C7F7 98ED  --  FA1C      EFA89C       EFA89C       F3B2AF9D
FB87     FB87 FB87 EE6A ED77 605A 67A4 67A4 67A4 5DE4 C7F8 98EE D16C 7AEB      E7ABAB       E7ABAB       F3B2AF9E
FB88     FB88 FB88 EE6B ED83 6082 67B7 67B7 67B7 5EA3 C8BB 998C D23B 7B9E      E7AE9E       E7AE9E       F3B2AF9F
FB89     FB89 FB89 EE6C  --  60AD 67CC 67CC 67CC 5ED3 C8E8 99CA  --  FA1D      EFA89D       EFA89D       F3B2AFA0
FB8A     FB8A FB8A EE6D EDA0 60C4 67DD 67DD 67DD 5EFD C9A7 9A47 D361 7D48      E7B588       E7B588       F3B2AFA1
FB8B     FB8B FB8B EE6E EDA2 60C6 67E1 67E1 67E1 5EE0 C9AC 9A53 D368 7D5C      E7B59C       E7B59C       F3B2AFA2
FB8C     FB8C FB8C EE6F F7C5 60D4 67EC 67EC 67EC 5FE4 C9B9 9A89 D427 7DB7      E7B6B7       E7B6B7       F3B2AFA3
FB8D     FB8D FB8D EE70 EDA6 60DA 67EB 67EB 67EB 5FB6 C9B6 9A86  --  7DA0      E7B6A0       E7B6A0       F3B2AFA4
FB8E     FB8E FB8E EE71 EDAA 60E1 67F1 67F1 67F1 5FBE C9BD 9A8E  --  7DD6      E7B796       E7B796       F3B2AFA5
FB8F     FB8F FB8F EE72 EDB3 60FD 67FB 67FB 67FB 5FD6 C9D5 9AAA D454 7E52      E7B992       E7B992       F3B2AFA6
FB90     FB90 FB90 EE73 EDB7 6157 68A5 68A5 68A5 5FE8 BFFB 90FB D472 7F47      E7BD